# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: suit_utils.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:31 $
# 

 
# ---------------------------------------------------
# This module contains procs that have SUIT dependencies.


# ---------------------------------------------------
# Object to manage host mapping data.  No dependancy on UI, but uses
#   Suit's data objects.  Was part of Quick Setup, hence the Qsu_ prefix.
#   Should be fixed at a less precarious moment.

Class Qsu_Hosts


# dataTbl is the name of a Suit Table object.

Qsu_Hosts instproc init {dataTbl} {
    $self instvar _dataTbl

    set _dataTbl   $dataTbl
}


# If the operation is "add", "mod" or "del", do it.
#   Return "" if OK, or error msg if not.

Qsu_Hosts instproc commit {recPtr id oper} {
    $self instvar _dataTbl
    upvar $recPtr   rec

    # @@@ puts "committing  id $id   oper $oper"
    # puts "rec [array get rec]|"

    # Ignore "noop" or ""
    if { ! ([cequal $oper "add"] || \
	    [cequal $oper "mod"] || \
	    [cequal $oper "del"])} {
	return ""
    }

    $_dataTbl buffer $id
    $_dataTbl setVal systemName      $rec(systemName)
    $_dataTbl setVal networkAddress  $rec(networkAddress)
    $_dataTbl setVal aliases         $rec(aliases)
    $_dataTbl setVal comment         $rec(comment)
    switch $oper {
	add {
	    set stat [$_dataTbl add newID]
	}
	mod {
	    set stat [$_dataTbl modify]
	}
	del {
	    set stat [$_dataTbl delete $id]
	}
    }
    return $stat
}


# Look for host record for specified host and IP address.
# If both IP and name blank, set oper to "noop".
# If IP and name found, set oper to "noop".
# If IP found but name different, set oper to "mod".
# If name found but IP different, set oper to "mod".
# If still not found and name fully qualified, look for unqualified name.
# If still not found, set oper to "add".
# Return ID if found, and "default" if not.

Qsu_Hosts instproc find {recPtr} {
    $self instvar _dataTbl
    upvar $recPtr   rec

    # if both name and IP not specified, do nothing
    if {[cequal $rec(systemName) ""] && [cequal $rec(networkAddress) ""]} {
	set id "default"
    } else {

	# look for IP and name first
	set id [$_dataTbl findRec {networkAddress systemName} rec]

	# if not found, just check for IP
	if {$id <= -1} {
	    set id [$_dataTbl findRec {networkAddress} rec]
	}

	# if still not found, just check for name
	if {$id <= -1} {
	    set id [$_dataTbl findRec {systemName} rec]
	}

	# if still not found, check for unqualified name
	if {$id <= -1} {
	    set target(systemName) [lindex [split $rec(systemName) .] 0]
	    if { ! [cequal $rec(systemName) $target(systemName)]} {
		set id [$_dataTbl findRec {systemName} target]
	    }
	}

	# if not found, use default record as base
	if {$id <= -1} {
	    set id   "default"
	}
    }

    # Read data from found record.
    #  If not found, keep name and address sent in.
    #  If found and name and/or address not specified for search, read them
    #  from found record.
    $_dataTbl buffer $id
    set rec(aliases)            [$_dataTbl getVal aliases]
    set rec(comment)            [$_dataTbl getVal comment]
    if { ! [cequal $id "default"]} {
	if {[cequal $rec(systemName) ""]} {
	    set rec(systemName)     [$_dataTbl getVal systemName]
	}
	if {[cequal $rec(networkAddress) ""]} {
	    set rec(networkAddress) [$_dataTbl getVal networkAddress]
	}
    }

    return $id
}


# Check the specifed data against the record given by the ID, to see if
#   record should be added, modified or nothing needs to be done.
# If both IP and name blank, return "noop".
# If ID is "default", return "add".
# If IP or name changed, return "mod".
# If IP and name the same, return "noop".

Qsu_Hosts instproc getOper {recPtr id} {
    $self instvar _dataTbl
    upvar $recPtr   rec

    # if both name and IP not specified, do nothing
    if {[cequal $rec(systemName) ""] && [cequal $rec(networkAddress) ""]} {
	return "noop"
    }

    # If record new, add it.
    if {[cequal $id "default"]} {
	return "add"
    }

    $_dataTbl buffer $id
    if {! [cequal [$_dataTbl getVal systemName]     $rec(systemName)]  ||
	! [cequal [$_dataTbl getVal networkAddress] $rec(networkAddress)]} {
        return "mod"
    }

    return "noop"
}


# Validate the specified host and IP address for the
#   specified host record and operation.  Return "" if OK, or error msg
#   if not.  If the host is fully qualified, make sure there is an alias
#   for the unqualified name.  While DNS can resolve the unresolved
#   names, it requires an unnecessary trip across the network.

Qsu_Hosts instproc validate {recPtr id oper} {
    $self instvar _dataTbl
    upvar $recPtr   rec

    # @@@ puts "validating  id $id   oper $oper"
    # puts "rec [array get rec]|"

    # Ignore "noop" or ""
    if { ! ([cequal $oper "add"] || \
	    [cequal $oper "mod"] || \
	    [cequal $oper "del"])} {
	return ""
    }

    $_dataTbl buffer $id
    set origName [$_dataTbl getVal systemName]
    $_dataTbl setVal systemName      $rec(systemName)
    $_dataTbl setVal networkAddress  $rec(networkAddress)
    $_dataTbl setVal comment         $rec(comment)

    # If adding or changing the system name, "fix" aliases.
    if {! [cequal $oper "del"] && \
	    ! [cequal $origName $rec(systemName)]} {
	set origAliases  $rec(aliases)

	# If prev host name qualified and unqualified name in aliases, remove.
	set unqual [Qsu_extractHostNameOnly $origName]
	if { ! [cequal $origName $unqual]} {
	    set alIdx [lsearch -exact $rec(aliases) $unqual]
	    if {$alIdx >= 0} {
		set rec(aliases) [lreplace $rec(aliases) $alIdx $alIdx]
	    }
	}

	# If host name qualified and unqualified name not in aliases, add it.
	set unqual [Qsu_extractHostNameOnly $rec(systemName)]
	if { ! [cequal $rec(systemName) $unqual] && 
	 [lsearch -exact $rec(aliases) $unqual] <= -1} {
	    lappend rec(aliases)  $unqual
	}

    }
    $_dataTbl setVal aliases         $rec(aliases)

    # If doesn't validate, revert aliases.
    set stat [$_dataTbl validate $oper]
    if { ! [cequal $stat ""]} {
	set rec(aliases)             $origAliases
    }
	
    return $stat
}

# ---------------------------------------------------
# Return just the host name from the fully qualified host name.
# For example, given zeus.unx.dec.com, return zeus

proc Qsu_extractHostNameOnly {fullHostName} {
    set full [split $fullHostName .]
    return [lindex $full 0]
}


# ---------------------------------------------------
# Return just the network domain from the fully qualified host name.
# For example, given zeus.unx.dec.com, return unx.dec.com

proc Qsu_extractNetDomain {fullHostName} {
    set full [split $fullHostName .]
    return [join [lrange $full 1 end] .]
}
