/* Prototypes for low-level cache routines.
 * Each cache entry is identified by NUL-terminated character string name
 * qualified by an integer type code.
 */
#define TCACHE_MAX_RECSIZE 16384
#define TCACHE_MAX_NAME 256
#define TCACHE_TYPE_CACHE -1
#define TCACHE_SUCCESS 1
#define TCACHE_ACCESS_SUCCESS 1
#define TCACHE_ACCESS_ACCESSED 1
#define TCACHE_ACCESS_CREATED 2
#define TCACHE_ACCESS_BLOCKED 3
#define TCACHE_ACCESS_FAILURE 4
#define TCACHE_FAILURE 4
#define TCACHE_OVERRUN 6
#define TCACHE_NULL_CTX ((tcache_ctxptr) 0)

#ifndef tcache_item_ptr
#define tcache_item_ptr void *
#endif
#ifndef tcache_rec_ptr
#define tcache_rec_ptr void *
#endif

struct tcache_context {
    tcache_item_ptr cache_item;		/* opaque pointer to cache entry */
    tcache_rec_ptr next_rec;
};
typedef struct tcache_context *tcache_ctxptr;

int tcache_set_max_recsize ( int rec_size );
    /*  Called before, tcache_init, tcache_set_max_recsize overrides
     *  the maximum recordsize supported by the cache (by default set to
     *  TCACHE_MAX_RECSIZE).  Rec_size is rounded up to next longword and
     *  must be at least 4096.  Return value is calculated rec_size.
     */

int tcache_init ( int size, int data_limit );
    /* tcache_init initializes the internal data structures used by the
     * data cache module and must be called only once.  The size parameter
     * is the number of bytes of virtual memory to reserve for cached data
     * (includes overhead for management structures).  Data_limit is the
     * maximum number of bytes a single cache entry may contain.
     *
     * Return values:
     *   1	Success.
     *   0      Error, could not reserve size bytes of virtual memory.
     */


int tcache_access ( int type, char *name, 
	tcache_ctxptr ctx, void **last_rec, int *length );
    /* tcache_access looks up name/type entry in cache and if found initializes
     * tcache_context structure, last_rec, and length parameters.  If not
     * found, a new entry is created and appropriate status is returned:
     *    ACCESSED  -	Entry found, available for access.
     *    CREATED -	New entry created, records must be added by caller.
     *    BLOCKED -	Entry found, but creator still accessing.
     *    FAILURE -	Entry corrupt or insufficient resources.
     */
int tcache_deaccess ( tcache_ctxptr ctx, int delete_flag );
    /* tcache_deacess runs down the context set upt by tcache_access.
     * If delete_flag non-zero, item is marked for deletion and will be
     * deleted when object has no more accessors (entry becomes invisible
     * to new accesses).
     */

int tcache_get ( tcache_ctxptr ctx, void **next_rec, int *length );
    /* Retreive next record stored for cache entry, only valid if tcache_access
     * returned ACCESSED */

int tcache_put ( tcache_ctxptr ctx, void *rec, int length );
    /* Save record.  Only valid if tcache_access return CREATED. REturn 
     * values:
     *    SUCCESS - data saved.
     *    OVERRUN - Data not saved, data_limit would be exceeded.
     *    FAILURE - Global allocation failure.
     */

int tcache_clear ( tcache_ctxptr ctx );
    /* Delete all records from cache entry, only valid if tcache_access
     * which initialized ctx returned CREATED */

int tcache_delete_all();
    /* Mark all cache entries for delete */

int tcache_delete_writelocked();
   /* Mark all cache entries open for write for delete */

int tcache_scan_entries ( tcache_ctxptr ctx, int *type, char **name,
	int *state, int *ref_count, void **last_rec, int *lr_length );
    /* This routine locks the cache database and returns cache entry data
     * for all entries over successive calls.  Init cache_item pointer in
     * ctx to null for first call.
     */
void tcache_end_scan ( tcache_ctxptr ctx );
     /* Unlocks cache structures, must be called */
