/*
 * Define thread-friendly wrappers for basic TCP/IP client operations.
 * Note that the time limit only affect tc_read and tc_write operations.
 *
 * The attributes argument in tc_open_tcp_socket is an array of character
 * pointers where is character string specifies an attribute modifier
 * for the created socket.
 *
 *    "local_port=nnn"		Use local port of nnn rather than 'ANY'
 *    "keepalive"		Set keepalive attribute.
 */
typedef void  *tc_socket;

int tc_open_tcp_socket ( 
    char *target_host, 			/* Remote host to connect to */
    int target_port, 			/* Report port, (local port 'ANY') */
    int attr_count,			/* Optional attributes */
    char **attributes, 			/* Attribute specifcations, see below */
    tc_socket *cnx, 			/* Handle to connection */
    char err_msg[256] );		/* Description of error code, if any */

int tc_close ( tc_socket cnx );		/* Synchronous close */
int tc_set_time_limit ( tc_socket cnx,  int seconds );
int tc_write ( tc_socket, void *buffer, int bufsize, int *written );
int tc_read ( tc_socket, void *buffer, int bufsize, int *written );
/*
 * The thread that opens a socket controls that socket and thread
 * exit will cause all sockets it owns to rundown.  You can transfer
 * ownership of a socket by having the current owner detach followed
 * the new owner attaching.
 */
int tc_detach ( tc_socket cnx );	/* allow shareing of connection */
int tc_attach ( tc_socket cnx );	/* allow shareing of connection */
/*
 * The following routines define an aynchronous interface for full duplex
 * operation.  The start_io function initiates a read or write operation as
 * indicated by the direction argument (0-write, 1-read).  Initiating a read
 * while a read is pending or a write while a write is pending is an error,
 * however a read and write may be pending at the same time.
 *
 * The io_stall function blocks the thread until a pending I/O operation
 * completes or another thread calls tc_attention.  The flags argument
 * returns status bits:
 *	0 - if set, write completed with bytes_written bytes transferred.
 *      1 - if set, read completed with bytes_read bytes transferred.
 *      2 - Unexpected return status of system service operation.
 *      3 - If set, attention routine has been called.
 */
int tc_start_io ( tc_socket cnx, int direction, void *buffer, int bufsize );
int tc_io_stall ( tc_socket cnx, int *flags, int *bytes_read, int *bytes_written );
int tc_attention ( tc_socket cnx );
/*
 * The listener object creates a context for receiving connection requests,
 * tc_accept waits for a connection request completes the connection by
 * creating an open socket.
 *
 * Note that the listener object is not associated with a particular thread.
 * and must be explicitly destroyed.
 *
 * Attributes options:
 *    "local_host=ip-address"
 *    "reuseaddr"
 */
typedef void *tc_listener;
int tc_create_tcp_listener ( int port, int backlog, int attr_count,
		char **attributes, tc_listener *ctx, char errmsg[256] );
int tc_destroy_listener ( tc_listener ctx );
int tc_accept ( tc_listener, tc_socket *, int *, unsigned char *, int, char * );
