/* Albums & songs editing subsystem */

#include <Xm/Text.h>
#include <X11/Xresource.h>
#include <Xm/SelectioB.h>

#include descrip
#include ctype

#include "cdrom-data.h"

#define max_tracks 255
#define big_text_buffer 4096
#define medium_text_buffer 255
#define small_text_buffer 15
#define title_field_width 60

typedef struct dsc$descriptor_s string;
#define string_dynamic {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}
#define PPTR(x) ((x)->dsc$a_pointer)
#define PLEN(x) ((x)->dsc$w_length)
#define SPTR(x) ((x).dsc$a_pointer)
#define SLEN(x) ((x).dsc$w_length)
#define NULL (void *)(0)

#ifndef MAX
#define MAX(a,b) (((b) > (a)) ? (b) : (a))
#endif
#ifndef MIN
#define MIN(a,b) (((b) < (a)) ? (b) : (a))
#endif

typedef struct
{
    int unique_key;
    char *key;
} cdKeyStruct;

typedef struct
{
    int pos;
    cdKeyStruct key;
} cdListPointer;

typedef struct
{
    int unique_key;
    int song;
} cdSongKey;

typedef struct
{
    cdSongKey key;
    Widget label_widget;
    Widget title_widget;
} cdSongWindow;

static string null_string = {1, 0, 0, "\0"};
static Widget widget_EditorBulletinBoard;
static Widget widget_EditorList;
static Widget widget_EditorPanedWindow;
static Widget widget_AlbumTitle;

static unsigned char *album_default_font;

static int highest_song;
static cdSongWindow cdsong[max_tracks];

void change_album_title(),
     change_song_title(),
     album_start_watch(),
     album_stop_watch();


/* Utility routine to convert text to normal form */
void normalise_asciz_string (char *buffer)
{
    static unsigned char ctrl_replace = 183; /*  */
    int indx;

    /* replace control characters with something else */
    for (indx = 0; indx < strlen (buffer); indx++)
    {
	if (iscntrl(buffer[indx])) buffer[indx] = ctrl_replace;
    }
}

/* Called when input focus moves to a text box */
void edit_title_begin (Widget text_box)
{
    char *name_string;
    int col;

    name_string = XmTextGetString(text_box);
    col = strlen(name_string);
    if (col)
    {
	set_value(text_box, XmNcursorPosition, col+1);
    }
    XtFree (name_string);
}

/* Called to add a song to the on-screen list.
 * One widget for label (song length), one for the title.
 * The title widget is returned so that it can be the top
 * widget for the next song.                               */
Widget add_song_to_dialog(Widget top, int song_num,
			  cdListPointer *listPtr,
			  cdSongKey key,
			  char *label, char *title)
{
    int argc;
    Arg args[10];
    XmString string;
    
    if (cdsong[song_num].label_widget == NULL)
    {
	argc = 0;
	XtSetArg(args[argc], XmNtopAttachment, XmATTACH_WIDGET); argc++;
	XtSetArg(args[argc], XmNtopWidget, top); argc++;
	XtSetArg(args[argc], XmNleftAttachment, XmATTACH_FORM); argc++;

	cdsong[song_num].label_widget = XmCreateLabel
	(
	    widget_EditorBulletinBoard,
	    "songWidgetNLabel",
	    args,
	    argc
	);

    }

    string = XmStringCreateLtoR(label, album_default_font);
    set_value (cdsong[song_num].label_widget, XmNlabelString, string);
    XmStringFree(string);
    set_value(cdsong[song_num].label_widget, XmNuserData, &cdsong[song_num]);

    if (cdsong[song_num].title_widget == NULL)
    {
	argc = 0;
	XtSetArg(args[argc], XmNresizeWidth, 1); argc++;
	XtSetArg(args[argc], XmNeditMode, XmSINGLE_LINE_EDIT); argc++;
        XtSetArg(args[argc], XmNeditable, 1); argc++;
	XtSetArg(args[argc], XmNtopAttachment, XmATTACH_WIDGET); argc++;
	XtSetArg(args[argc], XmNtopWidget, top); argc++;
	XtSetArg(args[argc], XmNleftAttachment, XmATTACH_WIDGET); argc++;
	XtSetArg(args[argc], XmNleftWidget, cdsong[song_num].label_widget); argc++;
	XtSetArg(args[argc], XmNrightAttachment, XmATTACH_FORM); argc++;
	XtSetArg(args[argc], XmNcolumns, title_field_width); argc++;

	cdsong[song_num].title_widget = XmCreateText
	(
	    widget_EditorBulletinBoard,
	    "songWidgetN",
	    args,
	    argc
	);

	XtAddCallback (cdsong[song_num].title_widget,
		       XmNlosingFocusCallback, change_song_title, 0);
	XtAddCallback (cdsong[song_num].title_widget,
		       XmNfocusCallback, edit_title_begin, 0);

    }

    XmTextSetString (cdsong[song_num].title_widget, title);
    set_value(cdsong[song_num].title_widget, XmNcursorPosition, 0);
    set_value(cdsong[song_num].title_widget, XmNuserData, &cdsong[song_num]);

    /* .song is used as a flag for whether widgets are managed */
    if (cdsong[song_num].key.song == 0)
    {
	cdsong[song_num].key = key;
	XtManageChild(cdsong[song_num].label_widget);
	XtManageChild(cdsong[song_num].title_widget);
    }

    return cdsong[song_num].title_widget;
}


/* Create an entry for each song */
void create_song_entries (string album_record, cdListPointer *listPtr)
{
    static string song_title = string_dynamic;
    scsi_read_toc_data toc;
    int songs, i, status, skip;
    Widget above = widget_AlbumTitle;
    cdSongKey song_key;
    char label[small_text_buffer];
    int s, e, l;

    cvt_key_to_toc(SPTR(album_record), &toc);
    songs = MIN(toc.header.last_track,max_tracks);

    /* Track n is toc entry n-1 */
    /* There is a dummy final toc entry for the end */
    s = toc.track[0].absolute_address.msf.m * 60 * 75 +
	toc.track[0].absolute_address.msf.s * 75 +
	toc.track[0].absolute_address.msf.f;

    for (i = 1; i <= songs; ++i)
    {
	e = toc.track[i].absolute_address.msf.m * 60 * 75 +
	    toc.track[i].absolute_address.msf.s * 75 +
	    toc.track[i].absolute_address.msf.f;
	l = e - s;

	song_key.unique_key = listPtr->key.unique_key;
	song_key.song = i;

	sprintf(label,"%2d (%2d:%02d.%02d)",
		i, l / 75 / 60, (l / 75) % 60, l % 75);

	status = db_lookup_song_title_via_album (song_key.unique_key, i, &song_title);
	if (status & 1)
	{
	    str$append (&song_title, &null_string);
	    normalise_asciz_string (SPTR(song_title));    
	    above = add_song_to_dialog(above, i, listPtr, song_key, label, SPTR(song_title));
	}
	else
	{
	    above = add_song_to_dialog(above, i, listPtr, song_key, label, "");
	}
	s = e;
    }

    /* Now clear out any songs from previous album */
    if (songs < highest_song)
    {
	for (i = highest_song; i > songs; i--)
	{
	    /* .song is used as a flag for whether widgets are managed */
	    if (cdsong[i].key.song)
	    {
		XtUnmanageChild (cdsong[i].label_widget);
		XtUnmanageChild (cdsong[i].title_widget);
		cdsong[i].key.song = 0;
	    }
	}
    }
    highest_song = songs;
}


/* Callback for when an album is selected */
void albumListSelectionMade
    (Widget widget, char *tag, XmListCallbackStruct *reason)
{
    static string record = string_dynamic;
    cdListPointer *listPtr;
    Arg args[10];
    cdKeyStruct *key_array;
    int argc;
    int pos;

    album_start_watch();

    /* The address of the array of album db keys is stored in the userData attribute */
    pos = reason->item_position;
    get_value(widget, XmNuserData, &key_array);

    db_lookup_album_via_short_key(key_array[pos].unique_key, &record);

    if (widget_AlbumTitle == NULL)
    {
        argc = 0;
        XtSetArg(args[argc], XmNresizeWidth, 1); argc++;
        XtSetArg(args[argc], XmNeditable, 1); argc++;
        XtSetArg(args[argc], XmNeditMode, XmSINGLE_LINE_EDIT); argc++;
        XtSetArg(args[argc], XmNtopAttachment, XmATTACH_FORM); argc++;
        XtSetArg(args[argc], XmNleftAttachment, XmATTACH_FORM); argc++;
        XtSetArg(args[argc], XmNrightAttachment, XmATTACH_FORM); argc++;
	XtSetArg(args[argc], XmNcolumns, title_field_width); argc++;

        widget_AlbumTitle = XmCreateText
        (
            widget_EditorBulletinBoard,
            "albumEditorAlbumTitle",
            args,
            argc
        );

        XtAddCallback (widget_AlbumTitle,
		       XmNlosingFocusCallback, change_album_title, 0);
	XtAddCallback (widget_AlbumTitle,
		       XmNfocusCallback, edit_title_begin, 0);
        XtManageChild(widget_AlbumTitle);
    }

    set_value(widget_AlbumTitle, XmNeditable, 1);

    /* Store this album's number and db key in the userData attribute */
    get_value(widget_AlbumTitle, XmNuserData, &listPtr);
    if (listPtr == NULL)
    {
	listPtr = malloc (sizeof(cdListPointer));
    }
    listPtr->pos = pos;
    listPtr->key = key_array[pos];
    set_value(widget_AlbumTitle, XmNuserData, listPtr);

    str$append(&record, &null_string);
    XmTextSetString(widget_AlbumTitle, SPTR(record) + 255);

    create_song_entries (record, listPtr);

    album_stop_watch();
}


void change_album_title(Widget widget, char *tag, XmAnyCallbackStruct *reason)
{
    XmString item;
    char *value;
    cdListPointer *info;

    album_start_watch();

    get_value(widget, XmNuserData, &info);

    if (info)
    {
        value = XmTextGetString(widget);

	normalise_asciz_string (value);    

        item = XmStringCreateLtoR(value, album_default_font);
        XmListReplaceItemsPos
        (
            widget_EditorList,
            &item,
            1,
            info->pos
        );
        XmStringFree(item);
    
        db_update_album_via_short_key(info->key.unique_key, value);

        XtFree(value);
    }

    album_stop_watch();
}


void change_song_title(Widget widget, char *tag, XmAnyCallbackStruct *reason)
{
    char *value;
    cdSongWindow *info;

    album_start_watch();

    get_value(widget, XmNuserData, &info);

    if (info)
    {
	value = XmTextGetString(widget);

	normalise_asciz_string (value);    

	db_update_song_via_album_key (info->key.unique_key, info->key.song, value);

	XtFree(value);
    }

    album_stop_watch();
}

/* Updates the (scrolled) list of albums */
update_list_box (Widget listBox)
{
    static string result = string_dynamic;
    XmString item;
    int status, i, position, short_key, num_albums, array_size;
    cdKeyStruct *key_array, *current;
    XmString *string_array, *tmp_string;

    /* The address of the array of album db keys is stored in the userData attribute */
    /* Discard any previous list */
    get_value(listBox, XmNuserData, &current);
    if (current)
    {
        for (i = 1; i <= current[0].unique_key; ++i)
        {
            if (current[i].key) free(current[i].key);
        }
        free(current);
        set_value(listBox, XmNuserData, 0);
        XmListDeleteAllItems(listBox);
    }

    /* get max # of albums from the first <null toc> record.. */
    status = db_first_album(&result);
    if (status & 1)
    {
        int *key;
	key = (int)(SPTR(result) + 251);

        num_albums = *key;
	array_size = num_albums+1; /* Allow for entry 0 */

	/* Create & store array of album info */
        key_array = calloc(1, (array_size * sizeof (cdKeyStruct)));
        set_value(listBox, XmNuserData, key_array);

        /* pop through the albums ... */
        position = 1;
        status = db_next_album(&result);
        while (status & 1)
        {
            int indx;
            char *buffer;

            if (SLEN(result) > 255)
            {
                buffer = malloc(SLEN(result) - 255 + 1);
                sprintf(buffer, "%.*s", SLEN(result) - 255, SPTR(result) + 255);
		normalise_asciz_string (buffer);
            }
            else
            {
                buffer = 0;
            }

            key = (int)(SPTR(result) + 251);
            key_array[position].unique_key = *key;
            key_array[position].key = buffer;
            ++position;

            status = db_next_album(&result);
        }

        /* first array element contains number of albums */
        key_array[0].unique_key = num_albums;
	key_array[0].key = 0;

        /* sort the albums - good old fashioned bubble sort? lazy me. */
        for (i = 1; i < num_albums; ++i)
        {
            int j;

            if (key_array[i].key)   /* if there's a key to compare.. */
            {
                for (j = i + 1; j <= num_albums; ++j)
                {
                    if (key_array[j].key == 0)  /* if there isn't a key... */
                    {   /* nothing must be lower than everything else eh? */
                        cdKeyStruct tmp;

                        tmp = key_array[j];
                        key_array[j] = key_array[i];
                        key_array[i] = tmp;
                        j = num_albums; /* skip the rest */
                    }
                    else
                    {
                        if (strcmp(key_array[j].key, key_array[i].key) < 0)
                        { /* it's lower? swap em.. */
                            cdKeyStruct tmp;
    
                            tmp = key_array[j];
                            key_array[j] = key_array[i];
                            key_array[i] = tmp;
                        }
                    }
                }
            }
        }


        string_array = malloc(1, (num_albums * sizeof (XmString)));
        tmp_string = string_array;

        /* loop through again and add to list in selection box */
        for (i = 1; i <= num_albums; ++i)
        {
            if (key_array[i].key)
            {
                *tmp_string = XmStringCreateLtoR(key_array[i].key, album_default_font);
            }
            else
            {
                *tmp_string = XmStringCreateLtoR("<no title>", album_default_font);
            }
            ++tmp_string;
        }

        XmListAddItems(listBox, string_array, num_albums);

        for (i = 0; i < num_albums; ++i)
        {
            XmStringFree(string_array[i]);
        }
        free(string_array);

	/* Now mark the current album as 'selected' */
        short_key = db_current_album_unique_key();
        if (short_key)
        {
            for (i = 1; i <= num_albums; ++i)
            {
                if (short_key == key_array[i].unique_key)
                {
                    XmListSelectPos(listBox, i, 1);
                    XmListSetPos(listBox, i);
                }
            }
        }
    }
}


/* Setup the album editor subsystem */
void album_edit_setup (Widget w1, Widget w2, Widget w3, unsigned char *font)
{
    widget_EditorBulletinBoard = w1;
    widget_EditorList = w2;
    widget_EditorPanedWindow = w3;
    album_default_font = font;

    XtManageChild(widget_EditorPanedWindow);

    album_start_watch();
    update_list_box (widget_EditorList);
    album_stop_watch();
}


/* Set cursor to be the watch symbol */
void album_start_watch()   
{
    static Cursor watch = NULL;

    start_watch (widget_EditorPanedWindow, &watch);
}

/* Set cursor back to normal */
void album_stop_watch()
{
    stop_watch (widget_EditorPanedWindow);
}

/* DEC/CMS REPLACEMENT HISTORY, Element CD-ALBUM.C*/
/* *10   12-JUN-1994 17:16:40 SYSTIMK "Fixed title editing"*/
/* *9    12-JUN-1994 12:31:37 SYSTIMK "Added cursor=watch support"*/
/* *8    30-MAY-1994 18:09:54 SYSTIMK "made #includes consistent"*/
/* *7    30-MAY-1994 12:50:27 SYSTIMK "Minor bugs fixed/improvements"*/
/* *6    30-MAY-1994 12:13:58 SYSTIMK "Misc. tidy-up"*/
/* *5    29-MAY-1994 18:25:45 SYSTIMK "Made text fields single-line, as for main & tracks"*/
/* *4    29-MAY-1994 17:38:06 SYSTIMK "Minor improvements to name entry"*/
/* *3    29-MAY-1994 17:10:08 SYSTIMK "Added consistent filtering out of control chars."*/
/* *2    29-MAY-1994 12:35:05 SYSTIMK "Now working"*/
/* *1    29-MAY-1994 10:04:13 SYSTIMK "UserData resource not working"*/
/* DEC/CMS REPLACEMENT HISTORY, Element CD-ALBUM.C*/
