/*

The following description describes my intent for the near future. Currently
the stuff is implemented differently, but this is what I'd rather wish I did,
and what I'd like to see it be. Let me know what you think!
Unimplemented bits are marked: &&&

Album DB Layout: Contains album information

[rtype][long-key][unique-key][data]

    rtype = unsigned byte
    long key = 250 bytes (TOC)
    [unique-key]  = unsigned longword
    data = variable length

    Primary key = rtype + long-key 
        The layout of the key is to aid in collecting all those annoying
        NULLs together at the end of the key!
    Secondary key = rtype+unique-key (for reverse lookups from related DB's)
&&& Tertiary key = rtype+start-of-title

    if rtype = 0 and long-key = NULLS and unique-key = 0 then

        data = [next-unique-key][DB creation time]

            next-unique-key = unsigned longword
&&&         DB creation time = quadword VMS time.

            next-unique-key is a short key to be used as an index to SONG
                DB, as well as other future DB's. This is the key which
                you can use to "join" multiple DB's. When the album DB is
                created, this is set to 1, and incremented for every new record,
		so it's also the number of albums in the db.
&&&         DB creation time is a magic number used to verify that a song DB
&&&             (and other related DB's) are really part of the same group of
&&&             related DB's.

    if rtype = 0 then
        long-key = incorrectly compressed TOC of CD
            note that toc is compressed to 20 bits per track, up to 100 tracks
            is legal. 2000 bits is 250 bytes.
        unique-key = index for joining to Song DB
        data = album title

    if rtype = 1 then
        long-key = compressed TOC of CD
            note that toc is compressed to 20 bits per track, up to 100 tracks
            is legal. 2000 bits is 250 bytes.
        unique-key = index for joining to Song DB
        data = album title

Song DB layout: Contains song information

[album-key][track][rtype][flags][data]

    Primary key = album-key + track + rtype (note that the order was chosen
        to make a sequential trek through the file appear reasonable).
&&& Secondary key = rtype+start-of-title

    album-key = unsigned longword
    track = unsigned byte
    rtype = unsigned byte
    flags = unsigned byte
	    bit 0: track is in playlist (i.e. "preferred")
    data = variable length

    if rtype = 1 then
        data = song title (for "track")

&&& if rtype = 0 then
&&&     data = [final-track][group-title]
&&&            the tracks between "track" and "final-track" are to be considered
&&&            as a single "group" i.e. a symphony.

*/

#include descrip
#include rms
#include ssdef

#include "cdrom-data.h"

typedef struct dsc$descriptor_s string;
#define string_dynamic {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}

#define PPTR(x) ((x)->dsc$a_pointer)
#define PLEN(x) ((x)->dsc$w_length)
#define SPTR(x) ((x).dsc$a_pointer)
#define SLEN(x) ((x).dsc$w_length)

#ifndef MAX
#define MAX(a,b) (((b) > (a)) ? (b) : (a))
#endif
#ifndef MIN
#define MIN(a,b) (((b) < (a)) ? (b) : (a))
#endif

/* Use RMS multibuffering for efficiency */
/* (Also saves having to bother with caching here!) */
#define K_RMS_BLOCKS 10
#define K_RMS_BUFFS 2
#define K_RMS_OPTS RAB$M_RAH+RAB$M_WBH

#define album_k_rtype_entry 1
#define album_k_rtype_oldtoc 0

#define read_write 0
#define read_only 1

static struct RAB *ctx_album, *ctx_songs;

/* the key size is 250 bytes for the TOC, plus 1 byte for a record type */
#define LONG_KEY_SIZE 251

#define ALBUM_DB_NAME       "album"
#define ALBUM_DB_DEFAULT    "sys$login:album.cd"

#define ALBUM_KEY0_LENGTH 1+250
#define ALBUM_KEY1_LENGTH 1+4
#define ALBUM_KEY2_LENGTH 1+50  /* reasonable length? */

struct AlbumRecord
{
    unsigned char rtype;
    unsigned char toc[250];
    unsigned long unique_key;
    unsigned char data[];   /* unknown length, must use struct via pointer */
};

#define SONG_DB_NAME       "songs"
#define SONG_DB_DEFAULT    "sys$login:songs.cd"

#define SONG_KEY0_LENGTH 4+1+1
#define SONG_KEY1_LENGTH 1+50   /* reasonable length? */

/* short key size includes: unique album id, plus track #, plus record type */
#define SHORT_KEY_SIZE SONG_KEY0_LENGTH 

#define song_k_rtype_entry 1

struct SongRecord
{
    unsigned long album_key;
    unsigned char track;
    unsigned char rtype;
    unsigned char flags;
    unsigned char data[];   /* unknown length, must use struct via pointer */
};

static char null_key[LONG_KEY_SIZE]; /* null record, contains short key.. */

cvt_toc_to_key(scsi_read_toc_data *toc, unsigned char *result)
{
    int i, nibble, index, status;
    unsigned long int l;

    /* compress the toc */
    index = 0;
    result[index] = album_k_rtype_entry; /* set the rtype byte */

    nibble = 0;
    for (i = 0; i <= toc->header.last_track; ++i)
    {

        l = (toc->track[i].absolute_address.msf.m * 60 * 75) +
            (toc->track[i].absolute_address.msf.s * 75) +
            toc->track[i].absolute_address.msf.f;
        if (l > 0xfffff) /* 20 bits total (only 19 needed actually!) */
        {
            printf("Internal coding error, cvt_toc_to_key\n");
            return 0;
        }
        if (nibble)
        {
            result[index] |= (l & 0x0f) << 4;
            l = l >> 4;
            result[++index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0xff;
        }
        else
        {
            result[++index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0x0f;
        }
        nibble = !nibble;
    }
}

cvt_toc_to_bad_key(scsi_read_toc_data *toc, unsigned char *result)
{
    int i, nibble, index, status;
    unsigned long int l;

    index = 1;  /* skip the rtype byte */

    /* compress the toc */
    index = 0;
    result[index++] = album_k_rtype_oldtoc; /* set the rtype byte */

    nibble = 0;
    for (i = 0; i <= toc->header.last_track; ++i)
    {

        l = (toc->track[i].absolute_address.msf.m * 60 * 75) +
            (toc->track[i].absolute_address.msf.s * 75) +
            toc->track[i].absolute_address.msf.f;
        if (l > 0xfffff) /* 20 bits total (only 19 needed actually!) */
        {
            printf("Internal coding error, cvt_toc_to_bad_key\n");
            return 0;
        }
        if (nibble)
        {
            result[index] |= (l & 0x0f) << 4;
            l = l >> 4;
            result[++index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0xff;
        }
        else
        {
            result[index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0xff;
            l = l >> 8;
            result[++index] = l & 0x0f;
        }
        nibble = !nibble;
    }
}

cvt_key_to_toc(unsigned char *result, scsi_read_toc_data *toc)
{
    int i, nibble, index;
    unsigned long int l;
    int old;

    old = (result[0] == 0);

    /* uncompress the toc */
    index = 1;
    toc->header.first_track = 1;
    nibble = 0;
    for (i = 0; i < 100; i++)
    {
        l = 0;
        if (!nibble)
        {
            unsigned int a,b,c;

            a = result[index++];
            b = result[index++] << 8;
            c = (result[index] & 0x0f) << 16;
            l = a | b | c;
        }
        else
        {
            unsigned int a,b,c;

            a = (result[index++] & 0xf0) >> 4;
            b = result[index++] << 4;
            c = result[index] << 12;

            if (! old) ++index;

            l = a | b | c;
        }

        if (l == 0) 
        {
            toc->header.last_track = i - 1;
            i = 100;
        }
        else
        {
            if (old)
            {   /* well, the toc info is useless, since I screwed up .. */
                toc->track[i].absolute_address.msf.f = 0;
                toc->track[i].absolute_address.msf.s = 0;
                toc->track[i].absolute_address.msf.m = 0;
            }
            else
            {   /* phew, a converted one that's correct! */
                toc->track[i].absolute_address.msf.f =  l % 75;
                l = l / 75;
                toc->track[i].absolute_address.msf.s = l % 60;
                toc->track[i].absolute_address.msf.m = l / 60;
            }
        }

        nibble = !nibble;
    }
    return 1;
}

string *db__cvt_current_toc_to_key()
{
    static unsigned char b[1+15+250];   /* +15 to make coding lazier */
    static string dummy = {LONG_KEY_SIZE, 0, 0, b};
    scsi_read_toc_data toc;
    int i, status;

    status = cd_get_toc(&toc);
    if (!(status & 1))
    {
        return 0;
    }

    b[0] = 1;
    for (i = 1; i < sizeof b; ++i) b[i] = 0;
    cvt_toc_to_key(&toc, b);

    return &dummy;
}

string *db__cvt_current_toc_to_bad_key()
{
    static unsigned char b[1+15+250];   /* +15 to make coding lazier */
    static string dummy = {LONG_KEY_SIZE, 0, 0, b};
    scsi_read_toc_data toc;
    int i, status;

    status = cd_get_toc(&toc);
    if (!(status & 1))
    {
        return 0;
    }

    for (i = 0; i < sizeof b; ++i) b[i] = 0;
    cvt_toc_to_bad_key(&toc, b);

    return &dummy;
}

int db_modify_album_title(char *name)
{
    string *long_key = db__cvt_current_toc_to_key();
    static string record = string_dynamic;
    string key = {0, 0, 0, 0};
    string fixed_part = {LONG_KEY_SIZE + 4, 0, 0, 0};
    string name_desc = {strlen(name), 0, 0, name};
    char *buffer;
    int status;

    if (!long_key)
    {   /* No TOC at the moment */
	return 0;
    }
    status = db__lookup_record (ctx_album, long_key, &record, read_write);
    if (status & 1)
    {
        SPTR(fixed_part) = SPTR(record);
        str$concat(&record, &fixed_part, &name_desc);
        SLEN(key) = LONG_KEY_SIZE;
        SPTR(key) = SPTR(record);
        status = db__update_record( ctx_album, &key,  &record, 0);
    }

    return status;
}

int db_get_album_title(string *record)
{
    int status;
    status = db_lookup_album(record);
    if (status & 1)
    {
        int skip = LONG_KEY_SIZE + 4 + 1;   /* + 1 for str$right... */
        status = str$right(record, record, &skip);
    }
    return status;
}

int db_current_album_unique_key()
{
    static string record = string_dynamic;
    int key = 0, status;

    status = db_lookup_album(&record);
    if (status & 1)
    {
        key = *(int *)(SPTR(record) + 251);
    }

    return key;
}

int db_lookup_album(string *record)
{
    string *long_key = db__cvt_current_toc_to_key();
    int status;

    if (!long_key)
    {   /* No TOC at the moment */
	return 0;
    }
    status = db__lookup_record (ctx_album, long_key, record, read_only);
    if (!(status & 1)) 
    {
        long_key = db__cvt_current_toc_to_bad_key();
        status = db__lookup_record( ctx_album, long_key, record );
        if (status & 1)
        {   /* need to convert from wrong format to correct format */
            static string new = string_dynamic;
            string dummy = {PLEN(record) - 251, 0, 0, PPTR(record) + 251};

            /* delete the errant record, and replace it with the new one */
            db__delete_record(ctx_album, long_key, 0);

            long_key = db__cvt_current_toc_to_key();

            str$copy_dx(&new, long_key);
            str$append(&new, &dummy);

            db__add_record(ctx_album, long_key, &new);
        }
        else
        {
            return status;
        }
    }

    return db__unlock(ctx_album);
}

int db_add_album(string *record)
{
    static string tmp_record = string_dynamic;
    string index_key = {sizeof null_key, 0, 0, null_key};
    string *long_key = db__cvt_current_toc_to_key();
    int *unique_key, i, status;
    string short_key = {4, 0, 0, 0};

    if (!long_key)
    {   /* No TOC at the moment */
	return 0;
    }

    /* we need to create a short_key and return it.. */
    /* perform read of dummy record, short key contains next short key to use */
    /* increment it, update the record, then use the short key... */
    status = db__lookup_record (ctx_album, &index_key, &tmp_record, read_write);
    if (status == RMS$_RNF)
    {   /* oooh, first time apparently? */
        /* append longword '0' to end of record... */
        int zero = 0;
        string first_short_key = {4, 0, 0, &zero};
        str$copy_dx(&tmp_record, &index_key);
        str$append(&tmp_record, &first_short_key);
        status = db__add_record( ctx_album, &index_key,  &tmp_record);
        if (!(status & 1)) return status;
        /* 'get' the record, and lock it... */
        status = db__lookup_record( ctx_album, &index_key, &tmp_record, read_write);
    }
    if (!(status & 1)) return status;

    unique_key = (int *) ( (int)SPTR(tmp_record) + LONG_KEY_SIZE );
    ++(*unique_key);
    status = db__update_record( ctx_album, &index_key,  &tmp_record, 0);
    if (!(status & 1)) return status;

    str$copy_dx(&tmp_record, long_key);
    SPTR(short_key) = unique_key;
    str$append(&tmp_record, &short_key);
    status = db__add_record( ctx_album, long_key, &tmp_record );
    if (!(status & 1)) return status;

    return db__unlock(ctx_album);
}

int db__song_get_album_key (int track_num, string *key)
{
/* Get the key for the current album */

    static string album_rec = string_dynamic;
    int status, start_pos;
    string track_desc = {1, 0, 0, &track_num};
    int rtype = song_k_rtype_entry;
    string rtype_desc = {1, 0, 0, &rtype};

    status = db_lookup_album (&album_rec);
    if (status & 1)
    {
        /* Get unique key representing the album */
	start_pos = 1+LONG_KEY_SIZE;
	str$len_extr (key, &album_rec, &start_pos, &4);

	/* Append track number */
        str$append (key, &track_desc);

	/* Append rtype */
        str$append (key, &rtype_desc);
    }

    return status;
}

int db_song_get_flags (int track_num, int *flags)
{
    static string record = string_dynamic;
    static string key = string_dynamic;
    string track_desc = {1, 0, 0, &track_num};
    string flags_desc = {1, 0, 0, flags};
    int status;

    status = db__song_get_album_key (track_num, &key);
    if (status & 1)
    {
	status = db__lookup_record (ctx_songs, &key, &record, read_only);
    }

    if (status & 1)
    {
	/* Song record found */
        int skip = SONG_KEY0_LENGTH+1;
        status = str$len_extr (&flags_desc, &record, &skip, &1);
    }

    return status;
}

int db_song_set_flags (int track_num, int *flags)
{
    static string track_name = string_dynamic;
    static string new_record = string_dynamic;
    static string record = string_dynamic;
    static string key = string_dynamic;
    string flags_desc = {1, 0, 0, flags};
    string fixed_part = {SONG_KEY0_LENGTH, 0, 0, 0}; /* minus flags byte */
    int status, skip;

    status = db__song_get_album_key (track_num, &key);
    if (status & 1)
    {
	status = db__lookup_record (ctx_songs, &key, &record, read_write);
    }

    if (status & 1)
    {
	/* Song record found */
        SPTR(fixed_part) = SPTR(record);
        str$concat (&new_record, &fixed_part, &flags_desc);
        skip = SONG_KEY0_LENGTH+1+1;
        str$right (&track_name, &record, &skip);
	str$append (&new_record, &track_name);
        status = db__update_record (ctx_songs, &key, &new_record, 0);
    }

    return status;
}

int db_modify_song_title (char *name, int track_num)
{
    static string record = string_dynamic;
    static string key = string_dynamic;
    string name_desc = {strlen(name), 0, 0, name};
    string flags_desc = {1, 0, 0, "\0"};
    string fixed_part = {SONG_KEY0_LENGTH+1, 0, 0, 0}; /* incl. flags byte */
    int status;

    status = db__song_get_album_key (track_num, &key);
    if (status & 1)
    {
	status = db__lookup_record (ctx_songs, &key, &record, read_write);
    }

    if (status == RMS$_RNF)
    {   
	/* New record needed */
        str$concat (&record, &key, &flags_desc, &name_desc);
        status = db__add_record (ctx_songs, &key, &record);
    }
    else if (status & 1)
    {
	/* Song record found */
        SPTR(fixed_part) = SPTR(record);
        str$concat (&record, &fixed_part, &name_desc);
        status = db__update_record (ctx_songs, &key, &record, 0);
    }

    return status;
}

int db_get_song_title (string *name, int track_num, int *flags)
{
    static string record = string_dynamic;
    static string key = string_dynamic;
    string flags_desc = {1, 0, 0, flags};
    int status;

    status = db__song_get_album_key (track_num, &key);
    if (status & 1)
    {
	status = db__lookup_record (ctx_songs, &key, &record, read_only);
    }

    if (status & 1)
    {
	/* Song record found */
        int skip = SONG_KEY0_LENGTH+1;
        str$len_extr (&flags_desc, &record, &skip, &1);

        skip = skip+1;
        status = str$right (name, &record, &skip);
    }

    return status;
}

db_open_cd_databases()
{
    int status;

    status = db__open_album_db(&ctx_album);
    if (!(status & 1)) return status;

    status = db__open_songs_db(&ctx_songs);
    if (!(status & 1)) return status;

    return 1;
}

db__open_album_db(struct RAB **context)
{
    struct FAB *fab;
    struct RAB *rab;
    struct XABKEY *xab1, *xab2, *xab3;
    int status;

    rab = malloc(sizeof(struct RAB));
    if (rab == 0)
    {
        return SS$_INSFMEM;
    }
    memcpy(rab, &cc$rms_rab, sizeof(struct RAB));
    *context = rab;

    fab = malloc(sizeof(struct FAB));
    if (fab == 0)
    {
        db__free_rms(rab);
        return SS$_INSFMEM;
    }
    memcpy(fab,&cc$rms_fab,sizeof(struct FAB));
    rab->rab$l_fab = fab;
    rab->rab$b_mbc = K_RMS_BLOCKS;
    rab->rab$b_mbf = K_RMS_BUFFS;

    fab->fab$l_dna = ALBUM_DB_DEFAULT;
    fab->fab$b_dns = strlen(ALBUM_DB_DEFAULT);
    fab->fab$l_fna = ALBUM_DB_NAME;
    fab->fab$b_fns = strlen(ALBUM_DB_NAME);
    fab->fab$b_fac = FAB$M_GET | FAB$M_PUT | FAB$M_UPD | FAB$M_DEL;
    fab->fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD | FAB$M_SHRDEL;

    xab1 = malloc(sizeof(struct XABKEY));
    if (xab1 == 0)
    {
        db__free_rms(rab);
        return SS$_INSFMEM;
    }
    memcpy(xab1, &cc$rms_xabkey, sizeof(struct XABKEY));
    fab->fab$l_xab = xab1;

    xab2 = malloc(sizeof(struct XABKEY));
    if (xab2 == 0)
    {
        db__free_rms(rab);
        return SS$_INSFMEM;
    }
    memcpy(xab2, &cc$rms_xabkey, sizeof(struct XABKEY));
    xab1->xab$l_nxt = xab2;

    status = sys$open(fab);
    if (status == RMS$_PRV)
    {   /* if we can't modify the DB, we oughtta at least try allow reading */
        fab->fab$l_dna = ALBUM_DB_DEFAULT;
        fab->fab$b_dns = strlen(ALBUM_DB_DEFAULT);
        fab->fab$l_fna = ALBUM_DB_NAME;
        fab->fab$b_fns = strlen(ALBUM_DB_NAME);
        fab->fab$b_fac = FAB$M_GET;
        fab->fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD | FAB$M_SHRDEL;
        status = sys$open(fab);
    }
    if (!(status & 1))
    {
        fab->fab$b_org = FAB$C_IDX;
        fab->fab$b_rat = FAB$M_CR;
        fab->fab$b_rfm = FAB$C_VAR;

        xab1->xab$b_dtp = XAB$C_STG;
        xab1->xab$b_siz0 = 251;
        xab1->xab$w_pos0 = 0;
        xab1->xab$l_knm = "record type + table of contents "; /* 32 bytes */

        xab2->xab$b_ref = 1;
        xab2->xab$b_dtp = XAB$C_STG;
        xab2->xab$b_siz0 = 4;
        xab2->xab$w_pos0 = 251;
        xab2->xab$b_siz1 = 1;
        xab2->xab$w_pos1 = 0;
        xab2->xab$b_flg = XAB$M_CHG | XAB$M_DUP;
        xab2->xab$l_knm = "short key + record type         "; /* 32 bytes */

        status = sys$create(fab);
    }

    if (!(status & 1))
    {
        db__free_rms(rab);
        *context = 0;
        return status;
    }

    status = sys$connect(rab);
    if (! (status & 1))
    {
        sys$close(fab);
        db__free_rms(rab);
        *context = 0;
        return status;
    }

    return 1;
}

db__open_songs_db(struct RAB **context)
{
    struct FAB *fab;
    struct RAB *rab;
    struct XABKEY *xab1, *xab2, *xab3;
    int status;

    rab = malloc(sizeof(struct RAB));
    if (rab == 0)
    {
        return SS$_INSFMEM;
    }
    memcpy(rab, &cc$rms_rab, sizeof(struct RAB));
    *context = rab;

    fab = malloc(sizeof(struct FAB));
    if (fab == 0)
    {
        db__free_rms(rab);
        return SS$_INSFMEM;
    }
    memcpy(fab,&cc$rms_fab,sizeof(struct FAB));
    rab->rab$l_fab = fab;
    rab->rab$b_mbc = K_RMS_BLOCKS;
    rab->rab$b_mbf = K_RMS_BUFFS;

    fab->fab$l_dna = SONG_DB_DEFAULT;
    fab->fab$b_dns = strlen(SONG_DB_DEFAULT);
    fab->fab$l_fna = SONG_DB_NAME;
    fab->fab$b_fns = strlen(SONG_DB_NAME);
    fab->fab$b_fac = FAB$M_GET | FAB$M_PUT | FAB$M_UPD | FAB$M_DEL;
    fab->fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD | FAB$M_SHRDEL;

    xab1 = malloc(sizeof(struct XABKEY));
    if (xab1 == 0)
    {
        db__free_rms(rab);
        return SS$_INSFMEM;
    }
    memcpy(xab1, &cc$rms_xabkey, sizeof(struct XABKEY));
    fab->fab$l_xab = xab1;

    status = sys$open(fab);
    if (status == RMS$_PRV)
    {   /* if we can't modify the DB, we oughtta at least try allow reading */
        fab->fab$l_dna = SONG_DB_DEFAULT;
        fab->fab$b_dns = strlen(SONG_DB_DEFAULT);
        fab->fab$l_fna = SONG_DB_NAME;
        fab->fab$b_fns = strlen(SONG_DB_NAME);
        fab->fab$b_fac = FAB$M_GET;
        fab->fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD | FAB$M_SHRDEL;
        status = sys$open(fab);
    }
    if (!(status & 1))
    {
        fab->fab$b_org = FAB$C_IDX;
        fab->fab$b_rat = FAB$M_CR;
        fab->fab$b_rfm = FAB$C_VAR;

        xab1->xab$b_dtp = XAB$C_STG;
        xab1->xab$b_siz0 = SONG_KEY0_LENGTH;
        xab1->xab$w_pos0 = 0;
        xab1->xab$l_knm = "album key + track + record type "; /* 32 bytes */

        status = sys$create(fab);
    }

    if (!(status & 1))
    {
        db__free_rms(rab);
        *context = 0;
        return status;
    }

    status = sys$connect(rab);
    if (! (status & 1))
    {
        sys$close(fab);
        db__free_rms(rab);
        *context = 0;
        return status;
    }

    return 1;
}

db_first_album(string *result)
{
    int status;

    /* rewind using "record type + short key" (would prefer data as key) */
    status = db__rewind(ctx_album, 0);
    if (!(status & 1)) return status;

    return db__next_record(ctx_album, result, 0);
}

db_next_album(string *result)
{
    return db__next_record(ctx_album, result, 0);
}

db_lookup_album_via_short_key(key, record)
    int key;    
    string *record;
{
    struct
    {
        int unique_key;
        char rtype;
    } keyval = {key, album_k_rtype_entry};
    string dummy = {sizeof keyval, 0, 0, &keyval};
    int status;

    status = db__lookup_record_via_index(ctx_album, 1, &dummy, record, read_only);
    if (!(status & 1))
    {
        keyval.rtype = album_k_rtype_oldtoc; /* old type of record? */
        status = db__lookup_record_via_index(ctx_album, 1, &dummy, record, read_only);
    }
}

db_lookup_song_title_via_album (key, song, title)
    int key;    
    int song;
    string *title;
{
    struct
    {
        int unique_key;
        unsigned char track;
        unsigned char rtype;
    } keyval = {key, song, song_k_rtype_entry};
    string dummy = {sizeof keyval, 0, 0, &keyval};
    static string record = string_dynamic;
    int status;

    status = db__lookup_record(ctx_songs, &dummy, &record, read_only);
    if (status & 1)
    {
	/* Song record found */
        int skip = SONG_KEY0_LENGTH+1+1;
        status = str$right (title, &record, &skip);
    }

    return status;
}

db_update_album_via_short_key(key, new_name)
    int key;    
    char *new_name;
{
    struct
    {
        int unique_key;
        char rtype;
    } keyval = {key, album_k_rtype_entry};
    string dummy = {sizeof keyval, 0, 0, &keyval};
    string replace = {strlen(new_name), 0, 0, new_name};
    static string record = string_dynamic;
    int length;
    int status;

    status = db__lookup_record_via_index(ctx_album, 1, &dummy, &record, read_write);
    if (!(status & 1))
    {
        keyval.rtype = album_k_rtype_entry;   /* old type of record? */
        status = db__lookup_record_via_index(ctx_album, 1, &dummy, &record, read_write);
    }

    length = 255;
    str$left(&record, &record, &length);
    str$append(&record, &replace);

    return db__update_record(ctx_album, &dummy, &record, 1);
}

db_update_song_via_album_key(key, song, new_name)
    int key;    
    int song;
    char *new_name;
{
    struct
    {
        int unique_key;
        unsigned char track;
        unsigned char rtype;
    } keyval = {key, song, song_k_rtype_entry};
    string dummy = {sizeof keyval, 0, 0, &keyval};
    int zero = 0;
    string flags_desc = {1, 0, 0, &zero};
    string replace = {strlen(new_name), 0, 0, new_name};
    static string record = string_dynamic;
    int status, length;

    status = db__lookup_record(ctx_songs, &dummy, &record, read_write);
    if (status & 1)
    {
        length = SONG_KEY0_LENGTH+1;
        str$left(&record, &record, &length);
        str$append(&record, &replace);
        status = db__update_record(ctx_songs, &dummy, &record, 1);
    }
    else
    {
        str$copy_dx(&record, &dummy);
        str$append(&record, &flags_desc);
        str$append(&record, &replace);
        status = db__add_record(ctx_songs, &dummy, &record);
    }
    return status;
}

/* Lookup record by primary key */
db__lookup_record (rab, key, record, read_flag)
    struct RAB *rab;
    string *key, *record;
    int read_flag;
{
    char buffer[4096]; /* this oughta be big enough eh? */
    string s1 = {0, 0, 0, buffer};
    int status;

    /* set up the key and buffer address/lengths */
    rab->rab$b_krf = 0;
    rab->rab$l_kbf = PPTR(key);
    rab->rab$b_ksz = PLEN(key);
    rab->rab$b_rac = RAB$C_KEY;
    if (read_flag)
    {
        rab->rab$l_rop = RAB$M_NLK+K_RMS_OPTS; /* no record lock */
    }
    else
    {
        rab->rab$l_rop = RAB$M_RLK+K_RMS_OPTS; /* lock the record */
    }
    rab->rab$l_ubf = buffer;
    rab->rab$w_usz = sizeof buffer;

    status = sys$get(rab);
    if (!(status & 1)) return status;

    s1.dsc$w_length = MIN(rab->rab$w_usz, rab->rab$w_rsz);
    return str$copy_dx(record, &s1);
}

/* Lookup record by given key number */
db__lookup_record_via_index(rab, index, key, record, read_flag)
    struct RAB *rab;
    int index;
    string *key, *record;
    int read_flag;
{
    char buffer[4096]; /* this oughta be big enough eh? */
    string s1 = {0, 0, 0, buffer};
    int status;

    /* set up the key and buffer address/lengths */
    rab->rab$b_krf = index;
    rab->rab$l_kbf = PPTR(key);
    rab->rab$b_ksz = PLEN(key);
    rab->rab$b_rac = RAB$C_KEY;
    if (read_flag)
    {
        rab->rab$l_rop = RAB$M_NLK+K_RMS_OPTS; /* no record lock */
    }
    else
    {
        rab->rab$l_rop = RAB$M_RLK+K_RMS_OPTS; /* lock the record */
    }
    rab->rab$l_ubf = buffer;
    rab->rab$w_usz = sizeof buffer;

    status = sys$get(rab);
    if (!(status & 1)) return status;

    s1.dsc$w_length = MIN(rab->rab$w_usz, rab->rab$w_rsz);
/*
    printf("_via_index: record @%x: {%x,%x}\n",record,record->dsc$w_length,
						      record->dsc$a_pointer);
    printf("_via_index: string @%x: {%x,%x}\n",&s1,s1.dsc$w_length,
						   s1.dsc$a_pointer);
*/
    return str$copy_dx(record, &s1);
}

db__rewind(rab, index)
    struct RAB *rab;
    int index;
{
    rab->rab$b_krf = index;
    rab->rab$b_rac = RAB$C_SEQ;

    return sys$rewind(rab);
}

/* Return the next record in the given index */
db__next_record(rab, record, index)
    struct RAB *rab;
    string *record;
    int index;
{
    char buffer[4096]; /* this oughta be big enough eh? */
    string s1 = {0, 0, 0, buffer};
    int status;

    /* set up the key and buffer address/lengths */
    rab->rab$b_krf = index;
    rab->rab$b_rac = RAB$C_SEQ;
    rab->rab$l_rop = RAB$M_NLK+K_RMS_OPTS; /* no record lock */
    rab->rab$l_ubf = buffer;
    rab->rab$w_usz = sizeof buffer;

    status = sys$get(rab);
    if (!(status & 1)) return status;

    s1.dsc$w_length = rab->rab$w_rsz;
    return str$copy_dx(record, &s1);
}

db__delete_record(rab, key, index)
    struct RAB *rab;
    string *key;
    int index;
{
    int status;

    /* set up the key and buffer address/lengths */
    rab->rab$l_kbf = PPTR(key);
    rab->rab$b_ksz = PLEN(key);
    rab->rab$b_rac = RAB$C_KEY;
    rab->rab$l_rop = RAB$M_RLK+K_RMS_OPTS; /* lock the record */
    rab->rab$b_krf = index;

    status = sys$get(rab);
    if (!(status & 1)) return status;

    return sys$delete(rab);
}

db__add_record(rab, key, val)
    struct RAB *rab;
    string *key, *val;
{
    /* set up the key and buffer address/lengths */
    rab->rab$b_krf = 0;
    rab->rab$l_kbf = PPTR(key);
    rab->rab$b_ksz = PLEN(key);
    rab->rab$b_rac = RAB$C_KEY;
    rab->rab$l_rbf = PPTR(val);
    rab->rab$w_rsz = PLEN(val);
    rab->rab$l_rop = RAB$M_RLK+K_RMS_OPTS; /* lock the record */

    return sys$put(rab);
}

db__update_record(rab, key, val, index)
    struct RAB *rab;
    string *key, *val;
    int index;
{
    /* set up the key and buffer address/lengths */
    rab->rab$l_kbf = PPTR(key);
    rab->rab$b_ksz = PLEN(key);
    rab->rab$l_rbf = PPTR(val);
    rab->rab$w_rsz = PLEN(val);
    rab->rab$l_rop = RAB$M_NLK+K_RMS_OPTS; /* don't lock the record */
    rab->rab$b_krf = index;

    return sys$update(rab);
}

db__unlock(rab)
    struct RAB *rab;
{
    int status;

    status = sys$free(rab);
    if (status == RMS$_RNL) status = 1;
    return status;
}

db__free_rms(struct RAB *rab)
{
    if (rab)
    {
        struct FAB *fab = rab->rab$l_fab;
        if (fab)
        {
            struct NAM
                *nam = fab->fab$l_nam;
            struct XAB
                *xab = fab->fab$l_xab,
                *nxt;
            if (nam) free(nam);
            while (xab)
            {
                struct XAB *nxt = xab->xab$l_nxt;
                free(xab);
                xab = nxt;
            }
            free(fab);
        }
        free(rab);
    }
}
/* DEC/CMS REPLACEMENT HISTORY, Element CD-DB.C*/
/* *9    30-MAY-1994 18:09:58 SYSTIMK "made #includes consistent"*/
/* *8    30-MAY-1994 12:14:03 SYSTIMK "Misc. tidy-up"*/
/* *7    29-MAY-1994 12:34:47 SYSTIMK "Changed cd-album's song lookup to be just for title"*/
/* *6    29-MAY-1994 12:00:42 SYSTIMK "Added MIN(RSZ,USZ)"*/
/* *5    29-MAY-1994 10:02:27 SYSTIMK "Merged with 1a1; bugs fixed; routine names improved"*/
/*  1A1  25-MAY-1994 09:54:43 SYSTIMK "joe's version of 25-May-1994"*/
/* *4    24-MAY-1994 12:43:44 SYSTIMK "Added rms multibuffering"*/
/* *3    24-MAY-1994 10:11:27 SYSTIMK "Added playlist save/restore"*/
/* *2    23-MAY-1994 12:53:07 SYSTIMK "Added songs db support"*/
/* *1    22-MAY-1994 09:12:43 SYSTIMK "DB utilities"*/
/* DEC/CMS REPLACEMENT HISTORY, Element CD-DB.C*/
