/************************************************************
*					
*         WHO v 1.5    JUN-1994  (A unix-like who for VAX/VMS)
*         ====================================================
*         Author    :  Andreas W. Wylach 
*         Copyright :  Public Domain, See GNU License.
*         File      :  WHO.C
*         Language  :  C
*         Compiler  :  VAX/VMS C v 3.2 
*         VMS Ver.  :  5.5
*
***************************************************************
*
*         Change History
*         ==============
*         Date        Change                                        Initials
*         ------------------------------------------------------------------
*         DEC 1991    Initial implementation                 V1.0   LH
*         APR 1992    Added "-p" flag to look for process id V1.1   LH
*         SEP 1992    Modified "Uptime" to show date and     
*		      time of boot.			     V1.2   LH
*         JUL 1994    Modified source code            
*                     Added broadcast availability
*                     Modified header line (NODENAME)
*                     Added help lib file                    V1.3   AW 
*         JUL 1994    Release of who. beta module compl.     V1.3.1 AW
*         SEP 1994    New idletime measurment coded.         V1.3.2 AW
*                     Source modifications.
*	  OCT 1994    Changed output format of -b
*                     Added option n # for quick who
*		      Source code modifications		     V1.4   AW
*	  OCT 1994    Added the error processing MSG_*	     V1.5   AW
* To be continued...
**************************************************************/

#include types
#include ctype
#include descrip
#include iodef
#include ssdef
#include stdio
#include syidef
#include sordef
#include strdef
#include dvidef
#include jpidef
#include unixlib
#include time
#include stat
#include "getopt"

/************************** D E F I N E S *******************************/
#define TRUE  1					/* TRUE flag 		*/
#define FALSE 0					/* FALSE flag		*/	
#define VERSION	"Version 1.5 BETA (VMS)"	/* Version string	*/
#define SIZE_USERNAME 12			/* length username	*/
#define ASCENDING  1				/* ascending sort type  */
#define DESCENDING 0				/* descending sort type */
#ifndef JPI$_TT_ACCPORNAM			/* define , if not	*/
#define JPI$_TT_ACCPORNAM	813
#endif

/********************** G L O B A L  VARIABLES ****************************/
unsigned short return_len;			/* itmelist return address */
int show_headers = FALSE;			/* show options headerline */
int who_am_i = FALSE;				/* show my term		   */		
int include_mesg = FALSE;			/* show boradcast setting  */
int list_all = FALSE;				/* Show almost all	   */
int list_boot_time = FALSE;			/* Show the boot time	   */
int number_given = FALSE;			/* ShoW n users on line -q */
						/* Exit status messages    */
globalvalue 	MSG_IVLOGNAM, MSG_NONEXPR, 
		MSG_NOPRIV, MSG_SUSPENDED, 
		MSG_NOMATCH;
int include_idle = FALSE;			/* show idle time	   */ 
int short_list = FALSE;				/* show quick who	   */
int normal_list = TRUE;				/* almost always true	   */
static int show_version;			/* Show version string     */
static int show_help;				/* Show who usage text	   */

/************************** G L O B A L S *******************************/
typedef struct 					/* sort struct		   */
{
  unsigned short s_keys;
  unsigned short s_type;
  unsigned short s_order;
  unsigned short s_offset;
  unsigned short s_length;
} sort_block;
  
struct item_list 				/* itemlist struct	   */
{
  unsigned short buffer_length;
  unsigned short item_code;
  int buffer_address;
  int ret_len_address;
};

static struct option const longopts[] = {	/* argument longoptions    */
      {"count",	no_argument, NULL, 'q'},
      {"number", 1, NULL, 'n'},
      {"idle",	no_argument, NULL, 't'},
      {"heading",no_argument, NULL, 'H'},
      {"help",	no_argument, &show_help, 1},
      {"version", no_argument, &show_version, 1},
      {"file", 1, NULL, 'f'},
      {"image", 1, NULL, 'I'},
      {"pid", 1, NULL, 'p'},
      {"user", 1, NULL, 'u'},
      {"uptime", no_argument, NULL, 'b'},
      {"mesg", no_argument, NULL, 'w'},
      {"broadcast", no_argument, NULL, 'w'},
      {NULL, 0, NULL, 0}
};

/********************** P R O T O T Y P E S *******************************/
void scan_processes(char *username,char* imagename,char *procid, int *user_count);
void strupcase(char *outstring, char *instring);
char *vms_message(unsigned int code);
void check_for_error(int status);
void cleanup(int status);
void usage(int status);
void print_heading();
void del_space_at_front(char *str);

/****************************************************************************/
/* Function : scan_process_list()					    */
/* Descrip  : Scan the processlist for given items			    */
/****************************************************************************/
void 
scan_processes(username,imagename,procid,user_count)
char *username;				/* given username  (-u)		*/
char *imagename;			/* given imagename (-i)		*/
char *procid;				/* given process id (-p)	*/
int *user_count;
{
  long int status;			/* status return		*/
  int  sys_time[2];			/* system time			*/
  int  diff_time[2];			/* time difference		*/
  int  boot_time[2];			/* temp. buffer boot time	*/
					/* job modes			*/
  int  total_jobs, batch_jobs, interactive_jobs, detached_jobs, network_jobs;
  unsigned long pid = -1;		/* Process id			*/
  char *pointer;				
  char time_buffer[23];			/* temp. buffer string time 	*/
  char new_image[255];			/* temp. buffer string new_image*/
  char image[80];			/* temp. buffer string image	*/
  char idle_string[12];			/* temp. buffer string idletime */			
  char brd_buf;				/* temp. buffer string broadcst	*/
  int day,hrs,min,sec;			/* time vars for idletime 	*/
  int status_term;			/* status function for idle 	*/
  int flag_found = 0;			/* search string matched?	*/
  int line_count = 0;
  $DESCRIPTOR(time_desc, time_buffer);	/* time in descriptor format	*/	
					/* process modes		*/
  char *modes[] = {"O", "N", "B", "I" };   	
					/* process types		*/
  char *types[] = {"D", "N", "B", "L", "L", "R" };	

  unsigned short sort_reclen;		/* sort length of items		*/
					/* sort mode is ascending	*/
  unsigned char  sort_process = ASCENDING;

  int  username_len  = 0;
  int  imagename_len = 0;

  struct  
  {
    char user[SIZE_USERNAME+1];
    unsigned int pid;
    unsigned int mpid;
    int  type;
    int  mode;
    int  time[2];
    char term[12];
    char image[255];
    char remote[255];
  } job;

  $DESCRIPTOR(termnam,job.term);	/* terminalname in descriptor format */
  sort_block key_buffer = { 1, DSC$K_DTYPE_T, 0, 0, SIZE_USERNAME};

  struct dsc$descriptor_s job_desc =
    { sizeof(job), DSC$K_DTYPE_T, DSC$K_CLASS_S, &job };
  
  struct TermInfo {
    long int duetim;
    long int pid;
    short int rtt_link;
  } terminfo;

  /* Itemlist of jpi's */
  struct item_list jpi_itemlist[] =
  {
    { sizeof(job.pid),    JPI$_PID,          &job.pid,    &return_len },
    { sizeof(job.mpid),   JPI$_MASTER_PID,   &job.mpid,   &return_len },
    { sizeof(job.user),   JPI$_USERNAME,     &job.user,   &return_len },
    { sizeof(job.image),  JPI$_IMAGNAME,     &job.image,  &return_len },
    { sizeof(job.remote), JPI$_TT_ACCPORNAM, &job.remote, &return_len },
    { sizeof(job.type),   JPI$_JOBTYPE,      &job.type,   &return_len },
    { sizeof(job.mode),   JPI$_MODE,         &job.mode,   &return_len },
    { sizeof(job.term),   JPI$_TERMINAL,     &job.term,   &return_len },
    { sizeof(job.time),   JPI$_LOGINTIM,     &job.time,   &return_len },
    { 0, 0, 0, 0},
  };

  total_jobs = batch_jobs = interactive_jobs = detached_jobs = network_jobs = 0;

  sort_reclen = sizeof(job);
  status = sor$begin_sort(&key_buffer, &sort_reclen, 
                          0, 0, 0, 0, &sort_process, 0, 0);

  username_len  = strlen(username);
  imagename_len = strlen(imagename);

  /* get system time */
  lib$date_time(&time_desc);
  lib$convert_date_string(&time_desc, &sys_time, 0, 0, 0, 0);

  if (*imagename) 
    sprintf(image, "]%s", imagename);
  if (*procid) 
    sscanf(procid, "%x", &pid);

  /* while everything is ok, start it over*/
  while (status == SS$_NORMAL)  {
    memset(&job, '\0', sizeof(job));
    status = sys$getjpiw(0, &pid, 0, jpi_itemlist, 0, 0, 0);
    check_for_error(status);

    switch (status)  {

      case SS$_NORMAL:
        if (*job.image == '\0') 
	  strcpy(job.image, "]DCL.EXE");
        if (username_len && imagename_len)  {
          if ((strncmp(job.user, username, username_len) == 0) &&
              strstr(job.image, image))
            status = sor$release_rec(&job_desc, 0);
        }
        else if (username_len)  {
          if (strncmp(job.user, username, username_len) == 0)  {
	    flag_found++;
            status = sor$release_rec(&job_desc, 0); 
	  } 
        }
        else if (imagename_len)  {
          if (strstr(job.image, image))  {
	    flag_found++;
            status = sor$release_rec(&job_desc, 0);
	  }
        }
        else if (*procid)  {
	  flag_found++;
          status = sor$release_rec(&job_desc, 0);
          status = SS$_ENDOFFILE;
        }
        else if (*job.user)  {
	  flag_found++;
          status = sor$release_rec(&job_desc, 0);
	}
	break;

      case SS$_NOMOREPROC:
        break;

      case SS$_SUSPENDED:
	status = SS$_NORMAL;
        break;
      
      default:
        fprintf(stderr, "sys$getjpi: %s\n", vms_message(status));
        break; 
    }
  }
  if (flag_found == 0 && list_all)
    cleanup(MSG_NOMATCH);			/* No such user */
  if (show_headers)
    print_heading();				/* print header if desired */

  if (list_boot_time)  { 
    struct item_list sys_itemlist[] =  {
      { sizeof(boot_time), SYI$_BOOTTIME, boot_time, &return_len },
      { 0, 0, 0, 0},
    };
    char days[6];				/* temp. buffer days	*/
    char hours[12];				/* temp. buffer hours	*/

    status = sys$getsyi(0, 0, 0, sys_itemlist, 0, 0, 0);
    lib$sub_times(sys_time, boot_time, diff_time);
    lib$sys_asctim(&return_len, &time_desc, &diff_time, 0);
    time_buffer[4] = '-';
    time_buffer[return_len] = '\0';
    for (pointer=time_buffer+strlen(time_buffer)-1;
      pointer > time_buffer; pointer--)  {
      if (*pointer == ':')  {
        *pointer = '\0';
        break;
      }
    }
    sscanf(time_buffer, "%s %s", days, hours);
    printf("    .            SYSBOOT %s ", days);                            
    /* uptime in complete date-string */
    lib$sys_asctim(&return_len, &time_desc, &boot_time, 0);
    time_buffer[11] = '\0';
    del_space_at_front(time_buffer);
    printf("                           ");
    printf(" [date=%-.*s]\n", (int) sizeof (time_buffer), time_buffer);
  }

  status = sor$sort_merge(0);
  while (status == SS$_NORMAL)  {
    status = sor$return_rec(&job_desc, 0, 0);
    switch (status)  {

      case SS$_NORMAL:
        for (pointer=job.image+strlen(job.image)-1; 
             (*pointer != ']') && (pointer > job.image); 
             pointer--) if (*pointer == '.') *pointer = '\0';
        strcpy(new_image, pointer+1);
        new_image[8] = '\0';

	/* Oh no, formatting again, sorry for that... */
        lib$sub_times(sys_time, job.time, diff_time);
        lib$sys_asctim(&return_len, &time_desc, &diff_time, 0);
        time_buffer[4] = '-';
        time_buffer[return_len] = '\0';
        for (pointer=time_buffer+strlen(time_buffer)-1;
             pointer > time_buffer; pointer--)  {
          if (*pointer == ':')  {
            *pointer = '\0';
            break;
          }
        }
	job.term[12] = '\0';
        if (list_all || include_mesg)  {
          lib$getdvi(&DVI$_TT_NOBRDCST,0,&termnam,&return_len);
          if (job.mode == JPI$K_INTERACTIVE)
            brd_buf = (!return_len) ? '+' : '-';
	  else
	    brd_buf = ' ';
	}

     	if (include_idle || list_all)  {
     	  status_term = term_info(&termnam,&terminfo);
     	  if ((status_term & 1) && (terminfo.duetim > 60) )  {
     	    min = terminfo.duetim / 60;
            hrs = min / 60;
            day = hrs / 24;
            sec = terminfo.duetim - (min * 60);
            min = min - (hrs * 60);
            hrs = hrs - (day * 24);
            if (terminfo.duetim > (24 * 60 * 60))  	/* one day idle	*/	
              sprintf(idle_string," old");		/* term is idle	*/	
            else
  	      					/* there's a idle time */
  	      sprintf(idle_string,"%02d:%02d", hrs, min);
          }
          else  {
            if (status_term & 1)  
              sprintf(idle_string,"  .");		/* term not idle*/
            else  
              idle_string[0] = '\0';
          }
 	}
        for (pointer=job.term+strlen(job.term)-1;
	     pointer > job.term; pointer--)
          if (*pointer == ':')  { 
            *pointer = '\0';
	  break;
        }

	/* here it comes, the formatted output */
	if (short_list && job.mode == JPI$K_INTERACTIVE)  {
          char trimmed_name[sizeof(job.user) + 1];
          int i;
          strncpy(trimmed_name, job.user, sizeof(job.user));
      	  trimmed_name[sizeof(job.user)] = ' ';
   	  for (i = 0; i <= sizeof(job.user); i++ )  {
	    if (trimmed_name[i] == ' ')
	      break;
	  }
	  if (number_given && (user_count == line_count)) {	     
	    printf(" \n");
	    line_count = 0;
	  }
	  trimmed_name[i] = '\0';
	  line_count++;
          printf("%-8s ",trimmed_name);
	}

	if ((!short_list) && normal_list && 
		job.mode == JPI$K_INTERACTIVE)  {
	  printf("%-10.*s", (int) sizeof (job.user), job.user);
          if (include_mesg || list_all)
            printf("%c    ", brd_buf);
	  printf("%-5.*s", (int) sizeof (job.term), job.term);
          printf("%-7.*s", (int) sizeof (time_buffer), time_buffer);

   	  if (include_idle || list_all)   
	    printf(" %-5.*s", (int) sizeof (idle_string), idle_string);

	  if (list_all)  {
	    printf(" %-.*s", (int) sizeof (types[job.type]), types[job.type]);
	    printf("/%-.*s", (int) sizeof (modes[job.mode]), modes[job.mode]);
	    printf(" %-4.*X", (int) sizeof (job.pid), job.pid);
	  }
	  if (list_all)
	    printf(" %-8.*s", (int) sizeof (new_image), new_image);
	  if (job.remote[0] == '\0')
	    printf("\0");
	  else
            printf(" (%-.*s)", (int) sizeof (job.remote), job.remote);
	  if (list_all && (job.term[0] == '\0'))  	/*is spawned prc! */
	    printf("       [mpid=%-.*X]", (int) sizeof (job.mpid), job.mpid);
	  printf("\n");
	}
        total_jobs++;

	/* check the job modes for all jobs running an count'em */
        switch (job.mode)  {

          case JPI$K_OTHER:
            if (job.mode == JPI$K_DETACHED) detached_jobs++;
            break;
          case JPI$K_NETWORK:
            network_jobs++;
            break;
          case JPI$K_BATCH:
            batch_jobs++; 
            break;
          case JPI$K_INTERACTIVE:
            interactive_jobs++; 
            break;
        };
        break;

      case SS$_ENDOFFILE:
        break;

      default:
        fprintf(stderr, "sor$return_rec error: %d\n", status);
	status = SS$_NORMAL;
        break; 
    }
  }

  status = sor$end_sort(0);
  /* if show_headers true, write the bottom line with job amounts 
     this line is print at option -a (all) */
  if (show_headers && list_all && (!include_mesg) && (!who_am_i) )  {
    fprintf(stdout,
      "\n# Jobs=%d, Interact=%d, Batch=%d, Detach=%d, Net=%d", 
      total_jobs, interactive_jobs, batch_jobs, detached_jobs, network_jobs);
  }
  if (short_list) 
    printf("\n# users=%d",interactive_jobs);
}

void
print_heading()
{
  printf("%-12s","USER");
  if (include_mesg || list_all || list_boot_time)
    printf("MESG ");
  printf("%-5s","LINE");
  printf("   AGE     ");
  if (include_idle || list_all || list_boot_time)
    printf("IDLE  ");
  if (list_all || list_boot_time)  {
    printf("TYP PID  ");
    printf("IMAGE    ");
  }
  printf("FROM");
  if (list_all || list_boot_time)
    printf("  COMMENTS");
  printf("\n");
}

void
del_space_at_front(str)
char *str;
{
int i = 0, j = 0;

  while (isspace(str[i++]));
    i--;
  while (str[j++] = str[i++]);
}

/****************************************************************************/
/* Function : strupcase()						    */
/* Descrip  : upcase the string, services need that!			    */
/****************************************************************************/
void 
strupcase(outstring,instring)
char *outstring;
char *instring;
{
  while (*instring) 
    *outstring++ = toupper(*instring++);
  *outstring = '\0';
}

/****************************************************************************/
/* Function : vms_message()						    */
/* Descrip  : put out a status message, if need	(only for sort)		    */
/****************************************************************************/
char 
*vms_message(code)
unsigned int code;
{
  static char  message[256];			/* temp. message buffer	*/
  unsigned short length = 0;			/* length of message	*/
  struct dsc$descriptor message_dx;		/* message in descriptor*/

  memset(message,0,sizeof(message));
  message_dx.dsc$w_length  = sizeof(message);
  message_dx.dsc$a_pointer = &message;
  sys$getmsg(code,&length,&message_dx,0xf,0);
  message[length] = '\0';
  return(&message);
}

/****************************************************************************/
/* Function : check_for_error()						    */
/* Descrip  : check for system messages and call cleanup		    */
/* This function is not used fully, will be in future versions		    */
/****************************************************************************/
void
check_for_error(status)
int status;
{
  if (status == SS$_IVLOGNAM)  {
    cleanup(MSG_IVLOGNAM);
  }
  if (status == SS$_NONEXPR)  {
    cleanup(MSG_NONEXPR);
  }
  if (status == SS$_NOPRIV)  {
    cleanup(MSG_NOPRIV);
  }
}


/****************************************************************************/
/* Function : cleanup()							    */
/* Descrip  : exit with status message					    */
/****************************************************************************/
void 
cleanup(status)
int status;
{
  exit(status);
}

/****************************************************************************/
/* Function : usage()							    */
/* Descrip  : print a short usage information				    */
/****************************************************************************/
void
usage(status)
int status;
{
  if (status != 0)  
    fprintf(stderr, 
	"Try 'WHO --help' for quick help, or 'HELP WHO' for more information.\n");
  else  {
    printf("Usage: WHO [-OPTION]... | ARG1 ARG2 \n");
    printf("\
\n\
   -a                    all options (options btwui)\n\
   -b, --uptime          print boot information of VMS\n\
   -h, --heading         print line of column headings and machine uptime info\n\
                         (if used, must be first list element)\n\
   -t, --idle            add users idle time as HOURS:MINUTES, . or old\n\
   -n # 		 specify number of users per line for -q\n\
   -w, --mesg            output users with boradcast info\n\
   -q, --count        	 quick who\n\
   -m                    only user associated with stdin\n\
   -q, --pid PID         output with given PID\n\
   -u, --user USERNAME   output with given username\n\
   -f, --file FILENAME   write output into specified file (for options apiuw)\n\
   -i, --image IMAGE     who is using IMAGE\n\
   -s                    (ignored)\n\
       --version         output version information and exit\n\
       --help            display this help and exit\n\
       --broadcast       same as -w\n\
\n\
  If ARG1 ARG2 given, -m presumed: 'am i' or 'mom likes' are usual.\n\
  You can get more information with the DCL command HELP WHO.\n\
  ");  }
  exit(status);
}


/****************************************************************************/
/* Function : main()							    */
/* Descrip  : definetly the main					    */
/****************************************************************************/
main(argc,argv)
int argc;
char **argv[];
{
  char lookup_user[SIZE_USERNAME+1] = "\0";	/* given username	   */
  char lookup_image[80] = "\0";			/* given image name	   */
  char lookup_pid[80] = "\0";			/* given pid		   */
  int user_count = 0;
  char buf[3];					/* temp. buffer 	   */
  int optc, longind; 				/* declarations for getopt */
  int my_line_only = 0;				/* one of many switches	   */

  while ((optc = getopt_long(argc,argv,"f:i:u:p:n:qbsmawht",longopts,&longind))
	!= EOF)
    {   
    switch (optc)  {

        case 0:
	  break;

        /********************************/
	/* S, ignored			*/
        /********************************/
        case 's':
	  break;

        /********************************/
	/* B, get boot time information */
        /********************************/
	case 'b':
	  normal_list = FALSE;
	  list_boot_time = TRUE;
          break;

        /********************************/
	/* A, output is all		*/
	/********************************/
	case 'a':
	  list_all = TRUE;
	  list_boot_time = TRUE;
	  break;
	/********************************/
	/* F, the input filename	*/
	/********************************/
        case 'f': 
	  list_all = TRUE;
          fclose(stdout);
          stdout = fopen(optarg, "w");
          break;

	/********************************/
	/* I, option is Image		*/
	/********************************/
        case 'i':
	  list_all = TRUE;
          strcpy(lookup_image, optarg);
          strupcase(lookup_image, lookup_image);
          break;

	/********************************/
	/* U, option is username	*/
	/********************************/
        case 'u':
	  list_all = TRUE;
          strcpy(lookup_user, optarg);
          strupcase(lookup_user, lookup_user);
          break;

	/********************************/
	/* P, option is PID		*/
	/********************************/
        case 'p':
	  list_all = TRUE;
          strcpy(lookup_pid, optarg);
          strupcase(lookup_pid, lookup_pid);
          break;

	/********************************/
	/* H, option is header line	*/
	/********************************/
        case 'h':
          show_headers = TRUE;
          break;

	/********************************/
	/* M, who am i			*/
	/********************************/
        case 'm':
          who_am_i = TRUE;
	  my_line_only = 1;
	  sprintf(buf,"%X",getpid());
	  strcpy(lookup_pid,buf); 
	  strupcase(lookup_pid, lookup_pid);
          break;

	/********************************/
	/* Q, quick who			*/
	/********************************/
	case 'q':
	  short_list = TRUE;
	  break;

	
	/**********************************************/
	/* N, specify number of users/line (option -q */
	/**********************************************/
	case 'n':
	  number_given = TRUE;
	  user_count = atoi(optarg);
	  break;

	/********************************/
	/* T, output users with idletime*/
	/********************************/
	case 't':
	  include_idle = TRUE;
	  break;

	/********************************/
	/* W, mesg, output brdcst string*/
	/********************************/
	case 'w':
	  include_mesg = 1; 
	  break;

	/********************************/
	/* error, write error message	*/
	/********************************/
        default:
	  usage(1);
          break;
      }
    }
    if (show_version)  {
      printf("WHO - %s\n", VERSION);
      exit(0);
    }
    if (show_help)
      usage(0);

    switch (argc - optind)  {
      
      case 0:				/* who	*/
    	scan_processes(lookup_user, lookup_image, lookup_pid, user_count);
	break;

      case 2:				/* who <blubber> <bla>	*/
        who_am_i = TRUE;
	sprintf(buf,"%X",getpid());
	strcpy(lookup_pid,buf); 
	strupcase(lookup_pid, lookup_pid);
    	scan_processes(lookup_user, lookup_image, lookup_pid, user_count);
	break;
      default:
        usage(1);
    }    
    exit(0);
}

/* EOF */
