/***************************************************************************
                          qwtplotwidget.cpp  -  description
                             -------------------
    begin                : Mon Mai 26 2003
    copyright            : (C) 2003 by Rainer Lehrig
                         : Angel Maza
    email                : lehrig@t-online.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qwtplotwidget.h"
#include "tcputil.h"

QwtPlotWidget::QwtPlotWidget(int *sock, int ident, QWidget *parent, int numberCurves, int numberMarker) : QwtPlot(parent)
{
    line = NULL;
    nCurves = numberCurves;
    nMarker = numberMarker;
    marker = new long[nMarker];
    curves = new long[nCurves];
    s = sock;
    id = ident;

    connect( this,  SIGNAL(plotMouseMoved(const QMouseEvent&)), SLOT(slotMouseMoved( const QMouseEvent&)));
    connect( this,  SIGNAL(plotMousePressed(const QMouseEvent&)), SLOT(slotMousePressed( const QMouseEvent&)));
    connect( this,  SIGNAL(plotMouseReleased(const QMouseEvent&)), SLOT(slotMouseReleased( const QMouseEvent&)));

/****************************************************************
    setTitle("Frequency Response of a Second-Order System");

    setCanvasBackground(darkBlue);

    // outline
    enableOutline(TRUE);
    setOutlinePen(green);

    // legend
    setAutoLegend(TRUE);
    enableLegend(TRUE);
    setLegendPos(Qwt::Bottom);
    setLegendFrameStyle(QFrame::Box|QFrame::Sunken);

    // grid
    enableGridXMin();
    setGridMajPen(QPen(white, 0, DotLine));
    setGridMinPen(QPen(gray, 0 , DotLine));

    // axes
    enableAxis(QwtPlot::yRight);
    setAxisTitle(QwtPlot::xBottom, "Normalized Frequency");
    setAxisTitle(QwtPlot::yLeft, "Amplitude [dB]");
    setAxisTitle(QwtPlot::yRight, "Phase [deg]");

    setAxisOptions(QwtPlot::xBottom, QwtAutoScale::Logarithmic);
    setAxisMaxMajor(QwtPlot::xBottom, 6);
    setAxisMaxMinor(QwtPlot::xBottom, 10);

    // curves
    curves[0] = insertCurve("Amplitude");
    setCurvePen(curves[0], QPen(yellow));
    setCurveYAxis(curves[0], QwtPlot::yLeft);

    curves[1] = insertCurve( "Phase");
    setCurvePen(curves[1], QPen(cyan));
    setCurveYAxis(curves[1], QwtPlot::yRight);

    // marker
    marker[0] = insertMarker();
    setMarkerLineStyle(marker[0], QwtMarker::VLine);
    setMarkerPos(marker[0], 0.0,0.0);
    setMarkerLabelAlign(marker[0], AlignRight|AlignBottom);
    setMarkerPen(marker[0], QPen(green, 0, DashDotLine));
    setMarkerFont(marker[0], QFont("Helvetica", 10, QFont::Bold));

    marker[1] = insertLineMarker("", QwtPlot::yLeft);
    setMarkerLabelAlign(marker[1], AlignRight|AlignBottom);
    setMarkerPen(marker[1], QPen(QColor(200,150,0), 0, DashDotLine));
    setMarkerFont(marker[1], QFont("Helvetica", 10, QFont::Bold));
    setMarkerSymbol(marker[1],
        QwtSymbol(QwtSymbol::Diamond, yellow, green, QSize(7,7)));

    // setDamp(0.0);
**********************************************************************/
}

QwtPlotWidget::~QwtPlotWidget()
{
  delete [] curves;
  delete [] marker;
}

int QwtPlotWidget::getText(const char *c, char *t, int maxlength)
{
  int i;
  *t = '\0';
  while(*c != '\0' && *c != '"') c++; // search for first "
  if(*c == '\0')
  {
    return -1;
  }
  c++;
  i = 0;
  while(*c != '\0') // until terminating " is seen
  {
    if(i >= maxlength)
    {
      *t = '\0';
      return 1;
    }
    i++;
    if(*c == '\\')
    {
      c++;
      if     (*c == '"')  *t++ = '"';
      else if(*c == '\n') *t++ = '\n';
      else if(*c == '\t') *t++ = '\t';
      else if(*c == '\0') break;
      c++;
    }
    else if(*c == '"')
    {
      *t = '\0';
      break;
    }
    else
    {
      *t++ = *c++;
    }
  }
  *t = '\0';
  return 0;
}

int QwtPlotWidget::isCommand(const char *command)
{
  int i = 0;
  if(line == NULL) return 0;
  while(command[i] != '\0')
  {
    if(command[i] != line[i]) return 0;
    i++;
  }
  return 1;
}

int QwtPlotWidget::interpret(const char *command, double *x, double *y)
{
  if(command == NULL) return -1;
  line = command;

  if     (isCommand("setCurveData("))
  {
    if(x == NULL) return -1;
    if(y == NULL) return -1;
    int c,count;
    sscanf(command,"setCurveData(%d,%d",&c,&count);
    if(c<0 || c>=nCurves) return -1;
    setCurveData(curves[c],x,y,count);
  }
  else if(isCommand("replot("))
  {
    replot();
  }
  else if(isCommand("setTitle("))
  {
    char text[1024];
    if(getText(command,text,sizeof(text)-1) != 0) return -1;
    setTitle(text);
  }
  else if(isCommand("setCanvasBackground("))
  {
    int r,g,b;
    sscanf(command,"setCanvasBackground(%d,%d,%d",&r,&g,&b);
    setCanvasBackground(QColor(r,g,b));
  }
  else if(isCommand("enableOutline("))
  {
    int val;
    sscanf(command,"enableOutline(%d",&val);
    if( val == 0 || val == 1 )
    {
      enableOutline(val);
      setOutlineStyle(Qwt::Rect );
    }
    else
    {
      enableOutline( 1 );
      setOutlineStyle(Qwt::Cross );
    }
  }
  else if(isCommand("setOutlinePen("))
  {
    int r,g,b;
    sscanf(command,"setOutlinePen(%d,%d,%d",&r,&g,&b);
    setOutlinePen(QColor(r,g,b));
  }
  else if(isCommand("setAutoLegend("))
  {
    int val;
    sscanf(command,"setAutoLegend(%d",&val);
    setAutoLegend(val);
  }
  else if(isCommand("enableLegend("))
  {
    int val;
    sscanf(command,"enableLegend(%d",&val);
    enableLegend(val);
  }
  else if(isCommand("setLegendPos("))
  {
    int val;
    sscanf(command,"setLegendPos(%d",&val);
    setLegendPos(val);
  }
  else if(isCommand("setLegendFrameStyle("))
  {
    int val;
    sscanf(command,"setLegendFrameStyle(%d",&val);
    setLegendFrameStyle(val);
  }
  else if(isCommand("enableGridXMin("))
  {
    enableGridXMin();
  }
  else if(isCommand("setGridMajPen("))
  {
    int r,g,b,style;
    sscanf(command,"setGridMajPen(%d,%d,%d,%d",&r,&g,&b,&style);
    setGridMajPen(QPen(QColor(r,g,b),0,(Qt::PenStyle) style));
  }
  else if(isCommand("setGridMinPen("))
  {
    int r,g,b,style;
    sscanf(command,"setGridMinPen(%d,%d,%d,%d",&r,&g,&b,&style);
    setGridMinPen(QPen(QColor(r,g,b),0,(Qt::PenStyle) style));
  }
  else if(isCommand("enableAxis("))
  {
    int pos;
    sscanf(command,"enableAxis(%d",&pos);
    enableAxis(pos);
  }
  else if(isCommand("setAxisTitle("))
  {
    int pos;
    char text[1024];
    sscanf(command,"setAxisTitle(%d",&pos);
    if(getText(command,text,sizeof(text)-1) != 0) return -1;
    setAxisTitle(pos,text);
  }
  else if(isCommand("setAxisOptions("))
  {
    int pos,val;
    sscanf(command,"setAxisOptions(%d,%d",&pos,&val);
    setAxisOptions(pos,val);
  }
  else if(isCommand("setAxisMaxMajor("))
  {
    int pos,val;
    sscanf(command,"setAxisMaxMajor(%d,%d",&pos,&val);
    setAxisMaxMajor(pos,val);
  }
  else if(isCommand("setAxisMaxMinor("))
  {
    int pos,val;
    sscanf(command,"setAxisMaxMinor(%d,%d",&pos,&val);
    setAxisMaxMinor(pos,val);
  }
  else if(isCommand("insertCurve("))
  {
    int pos;
    char text[1024];
    sscanf(command,"insertCurve(%d",&pos);
    if(getText(command,text,sizeof(text)-1) != 0) return -1;
    if(pos<0 || pos>=nCurves) return -1;
    removeCurve( curves[pos] );
    curves[pos] = insertCurve(text);
  }
  else if(isCommand("removeCurve("))
  {
    int pos;
    sscanf(command,"removeCurve(%d",&pos);
    if(pos<0 || pos>=nCurves) return -1;
    removeCurve( curves[pos] );
  }
  else if(isCommand("setCurvePen("))
  {
    int pos,r,g,b,width,style;
    sscanf(command,"setCurvePen(%d,%d,%d,%d,%d,%d",&pos,&r,&g,&b,&width,&style);
    if(pos<0 || pos>=nCurves) return -1;
    setCurvePen(curves[pos],QPen(QColor(r,g,b),width,(QPen::PenStyle) style));
  }
  else if(isCommand("setCurveSymbol("))
  {
    int pos,symbol,r1,g1,b1,r2,g2,b2,w,h;
    sscanf(command,"setCurveSymbol(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d",&pos,&symbol,
                    &r1,&g1,&b1,&r2,&g2,&b2,&w,&h);
    if(pos<0 || pos>=nCurves) return -1;
    setCurveSymbol(curves[pos], QwtSymbol((QwtSymbol::Style) symbol, QColor(r1,g1,b1), QColor(r2,g2,b2), QSize(w,h)));
  }
  else if(isCommand("setCurveYAxis("))
  {
    int pos,pos2;
    sscanf(command,"setCurveYAxis(%d,%d",&pos,&pos2);
    if(pos<0 || pos>=nCurves) return -1;
    setCurveYAxis(curves[pos],pos2);
  }
  else if(isCommand("insertMarker("))
  {
    int pos;
    sscanf(command,"insertMarker(%d",&pos);
    if(pos<0 || pos>=nMarker) return -1;
    marker[pos] = insertMarker();
  }
  else if(isCommand("setMarkerLineStyle("))
  {
    int pos,style;
    sscanf(command,"setMarkerLineStyle(%d,%d",&pos,&style);
    if(pos<0 || pos>=nMarker) return -1;
    setMarkerLineStyle(marker[pos],(QwtMarker::LineStyle) style);
  }
  else if(isCommand("setMarkerPos("))
  {
    int pos;
    float x,y;
    sscanf(command,"setMarkerPos(%d,%f,%f",&pos,&x,&y);
    if(pos<0 || pos>=nMarker) return -1;
    setMarkerPos(marker[pos],x,y);
  }
  else if(isCommand("setMarkerLabelAlign("))
  {
    int pos,align;
    sscanf(command,"setMarkerLabelAlign(%d,%d",&pos,&align);
    if(pos<0 || pos>=nMarker) return -1;
    setMarkerLabelAlign(marker[pos],align);
  }
  else if(isCommand("setMarkerLabel("))
  {
    int pos;
    char text[1024];
    sscanf( command, "setMarkerLabel(%d",&pos );
    if(getText(command, text, sizeof(text)-1) != 0 ) return -1;
    setMarkerLabel(marker[pos], text );
  }
  else if(isCommand("setMarkerPen("))
  {
    int pos,r,g,b,style;
    sscanf(command,"setMarkerPen(%d,%d,%d,%d,%d",&pos,&r,&g,&b,&style);
    if(pos<0 || pos>=nMarker) return -1;
    setMarkerPen(marker[pos],QPen(QColor(r,g,b),0,(Qt::PenStyle) style));
  }
  else if(isCommand("setMarkerFont("))
  {
    int pos,size,style;
    char family[1024];
    sscanf(command,"setMarkerFont(%d,%d,%d",&pos,&size,&style);
    if(pos<0 || pos>=nMarker) return -1;
    if(getText(command,family,sizeof(family)-1) != 0) return -1;
    setMarkerFont(marker[pos],QFont(family,size,style));
  }
  else if(isCommand("setMarkerSymbol("))
  {
    int pos,symbol,r1,g1,b1,r2,g2,b2,w,h;
    sscanf(command,"setMarkerSymbol(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d",
                  &pos,&symbol,&r1,&g1,&b1,&r2,&g2,&b2,&w,&h);
    if(pos<0 || pos>=nMarker) return -1;
    setMarkerSymbol(marker[pos], QwtSymbol((QwtSymbol::Style) symbol, QColor(r1,g1,b1), QColor(r2,g2,b2), QSize(w,h)));
  }
  else if(isCommand("insertLineMarker("))
  {
    int pos,pos2;
    char text[1024];
    sscanf(command,"insertLineMarker(%d,%d",&pos,&pos2);
    if(getText(command,text,sizeof(text)-1) != 0) return -1;
    if(pos<0 || pos>=nMarker) return -1;
    marker[pos] = insertLineMarker(text, pos2);
  }
  else if( isCommand("setAxisScaleDraw("))
  {
    int pos;
    char text[1024];
    sscanf( command, "setAxisScaleDraw(%d",&pos );
    if( getText(command, text, sizeof(text)-1 ) != 0 ) return -1;
    setAxisScaleDraw( pos, new UserScaleDraw( text ));
  }
  else
  {
    return -1;
  }

  return 0;
}

void QwtPlotWidget::slotMouseMoved( const QMouseEvent &e)
{
  double x,y;
  char buf[100];

  x = this->invTransform(QwtPlot::xBottom, e.pos().x());
  y = this->invTransform(QwtPlot::yLeft, e.pos().y());
  sprintf( buf, "QPlotMouseMoved(%d,%f,%f)\n",id, x, y );
  tcp_send(s,buf,strlen(buf));
}

void QwtPlotWidget::slotMousePressed( const QMouseEvent &e)
{
  double x,y;
  char buf[100];

  x = this->invTransform(QwtPlot::xBottom, e.pos().x());
  y = this->invTransform(QwtPlot::yLeft, e.pos().y());
  sprintf( buf, "QPlotMousePressed(%d,%f,%f)\n",id, x, y );
  tcp_send(s,buf,strlen(buf));
}

void QwtPlotWidget::slotMouseReleased( const QMouseEvent &e)
{
  double x,y;
  char buf[100];

  x = this->invTransform(QwtPlot::xBottom, e.pos().x());
  y = this->invTransform(QwtPlot::yLeft, e.pos().y());
  sprintf( buf, "QPlotMouseReleased(%d,%f,%f)\n",id, x, y );
  tcp_send(s,buf,strlen(buf));
}
