/***************************************************************************
 *   Copyright (C) 2005 by Rainer Lehrig                                   *
 *   lehrig@t-online.de                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "pvbuilder.h"
#include "dlginsertfunction.h"
#include "dlgdaemon.h"
#include "logo1.xpm"

#ifdef _WIN32
#include <process.h>
#include <stdlib.h>
#endif

#include <qdir.h>
#include <qurl.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtoolbox.h>
#include <qlistbox.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtextedit.h>
#include <qtextstream.h> 
#include <qfile.h>
#include <qfiledialog.h>
#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qprinter.h>
#include <qapplication.h>
#include <qaccel.h>
#include <qtextstream.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qwhatsthis.h>
#include <qspinbox.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qcheckbox.h>
#include <qvaluelist.h>
#include <qstringlist.h>
#include <qinputdialog.h>
#include <qcombobox.h>

#include "filesave.xpm"
#include "fileopen.xpm"
#include "fileprint.xpm"

#include "view.h"

#ifdef _WIN32
#include <windows.h>
#include <winreg.h>
static int get_iexplore(char *buf)
{
  HKEY applications,iexplore,shell,open,command;
  long ret;
  unsigned long size,type;
  char *cptr;

  buf[0] = '\0';

  ret = RegOpenKey(
    HKEY_CLASSES_ROOT,        // handle to open key
    "Applications",           // address of name of subkey to open
    &applications             // address of handle to open key
  );
  if(ret != ERROR_SUCCESS)
  {
    return -1;
  }

  ret = RegOpenKey(
    applications,             // handle to open key
    "iexplore.exe",           // address of name of subkey to open
    &iexplore                 // address of handle to open key
  );
  if(ret != ERROR_SUCCESS)
  {
    RegCloseKey(applications);
    return -1;
  }

  ret = RegOpenKey(
    iexplore,                 // handle to open key
    "shell",                  // address of name of subkey to open
    &shell                    // address of handle to open key
  );
  if(ret != ERROR_SUCCESS)
  {
    RegCloseKey(applications);
    RegCloseKey(iexplore);
    return -1;
  }

  ret = RegOpenKey(
    shell,                    // handle to open key
    "open",                   // address of name of subkey to open
    &open                     // address of handle to open key
  );
  if(ret != ERROR_SUCCESS)
  {
    RegCloseKey(applications);
    RegCloseKey(iexplore);
    RegCloseKey(shell);
    return -1;
  }

  ret = RegOpenKey(
    open,                     // handle to open key
    "command",                // address of name of subkey to open
    &command                  // address of handle to open key
  );
  if(ret != ERROR_SUCCESS)
  {
    RegCloseKey(applications);
    RegCloseKey(iexplore);
    RegCloseKey(shell);
    RegCloseKey(open);
    return -1;
  }

  ret = RegQueryValueEx(
    command,               // handle to key to query
    "",                    // address of name of value to query
    NULL,                  // reserved
    &type,                 // address of buffer for value type
    (unsigned char *) buf, // address of data buffer
    &size                  // address of data buffer size
  );

  RegCloseKey(applications);
  RegCloseKey(iexplore);
  RegCloseKey(shell);
  RegCloseKey(open);
  RegCloseKey(command);

  if(ret != ERROR_SUCCESS) return -1;

  cptr = strstr(buf," %1");
  if(cptr != NULL) *cptr = '\0';

  return 0;
}

static int winsystem(const char *command)
{
  int ret;
  STARTUPINFO         si = { sizeof(si)};
  PROCESS_INFORMATION pi;
  char cmd[4096];

  strcpy(cmd,command);
  ret = (int) CreateProcess( NULL, cmd
                           , NULL, NULL
                           , FALSE, CREATE_NO_WINDOW
                           , NULL, NULL
                           , &si, &pi);
  return ret;
}
#endif

static int mysystem(const char *command)
{
#ifdef _WIN32
  return winsystem(command);
#else
  //printf("system(%s)\n",command);
  return system(command);
#endif
}

dlgNewPrj::dlgNewPrj(QWidget* parent, const char* name)
          :newprj(parent,name,TRUE)
{
  QObject::connect(pushButtonCancel,SIGNAL(clicked()),SLOT(reject()));
  QObject::connect(pushButtonOk,    SIGNAL(clicked()),SLOT(accept())); 
  QObject::connect(pushButtonDir,   SIGNAL(clicked()),SLOT(slotDir()));
}

dlgNewPrj::~dlgNewPrj()
{
}

QString dlgNewPrj::run()
{
  lineEditDirectory->setText(QDir::current().path());
  lineEditName->setText("pvs");
  int ret = exec();
  if(ret == QDialog::Accepted)
  {
    if(lineEditDirectory->text() != "") QDir::setCurrent(lineEditDirectory->text());
    return lineEditName->text();
  }
  else
  {
    return "";
  }
}

void dlgNewPrj::slotDir()
{
  QString dir = QFileDialog::getExistingDirectory();
  if(dir == "") return;
  lineEditDirectory->setText(dir);
}
//##################################################################################
dlgPasteWidget::dlgPasteWidget(const char *_widgetname, QWidget* parent, const char* name)
               :pasteWidget(parent,name,TRUE)
{
  widgetname = _widgetname;
  QObject::connect(pushButtonCancel,SIGNAL(clicked()),SLOT(reject()));
  QObject::connect(pushButtonInsert,SIGNAL(clicked()),SLOT(accept())); 
}

dlgPasteWidget::~dlgPasteWidget()
{
}

QString dlgPasteWidget::run()
{
  QString text;
  int ret = exec();
  if(ret == QDialog::Accepted)
  {
    text = "";
    if(radioButtonPrintf->isChecked())         { text = "    pvPrintf(p,"; text += widgetname; text += ",\"%d\",val);"; }
    if(radioButtonTablePrintf->isChecked())    { text = "    pvTablePrintf(p,"; text += widgetname; text += ",x,y,\"%d\",val);"; }
    if(radioButtonListViewPrintf->isChecked()) { text = "    pvListViewPrintf(p,"; text += widgetname; text += ",path,column,\"%d\",val);"; }
    if(radioButtonButtonEvent->isChecked())    { text = "  if(id == "; text += widgetname; text += ")\n  {\n    return 1; // call mask 1\n  }";}
    return text;
  }
  else
  {
    return "";
  }
}
//##################################################################################

pvbuilder::pvbuilder()
    : QMainWindow( 0, "pvbuilder", WDestructiveClose )
{
    para = index = 0;

    printer = new QPrinter;
    QPixmap openIcon, saveIcon, printIcon;

    setIcon(logo1);

    QToolBar * fileTools = new QToolBar( this, "file operations" );
    fileTools->setLabel( tr("File Operations") );

    openIcon = QPixmap( fileopen );
    QToolButton * fileOpen
  = new QToolButton( openIcon, tr("Open File"), QString::null,
         this, SLOT(choose()), fileTools, "open file" );

    saveIcon = QPixmap( filesave );
    QToolButton * fileSave
  = new QToolButton( saveIcon, tr("Save File"), QString::null,
         this, SLOT(save()), fileTools, "save file" );

    printIcon = QPixmap( fileprint );
    QToolButton * filePrint
  = new QToolButton( printIcon, tr("Print File"), QString::null,
         this, SLOT(print()), fileTools, "print file" );


    (void)QWhatsThis::whatsThisButton( fileTools );

    QString fileOpenText = tr("<p><img source=\"fileopen\"> "
           "Click this button to open a <em>new file</em>. <br>"
                 "You can also select the <b>Open</b> command "
                 "from the <b>File</b> menu.</p>");

    QWhatsThis::add( fileOpen, fileOpenText );

    QMimeSourceFactory::defaultFactory()->setPixmap( "fileopen", openIcon );

    QString fileSaveText = tr("<p>Click this button to save the file you "
                 "are editing. You will be prompted for a file name.\n"
                 "You can also select the <b>Save</b> command "
                 "from the <b>File</b> menu.</p>");

    QWhatsThis::add( fileSave, fileSaveText );

    QString filePrintText = tr("Click this button to print the file you "
                 "are editing.\n You can also select the Print "
                 "command from the File menu.");

    QWhatsThis::add( filePrint, filePrintText );

    QPopupMenu * file = new QPopupMenu( this );
    menuBar()->insertItem( tr("&File"), file );
    file->insertItem( tr("&new visualization"), this, SLOT(newDoc()), CTRL+Key_N );
    int id;
    id = file->insertItem( openIcon, tr("&Open..."),
         this, SLOT(choose()), CTRL+Key_O );
    file->setWhatsThis( id, fileOpenText );
    id = file->insertItem( saveIcon, tr("&Save"),
         this, SLOT(save()), CTRL+Key_S );
    file->setWhatsThis( id, fileSaveText );
    id = file->insertItem( printIcon, tr("&Print"),
         this, SLOT(print()), CTRL+Key_P );
    file->setWhatsThis( id, filePrintText );
    file->insertSeparator();
    file->insertItem( tr("&Quit"), qApp, SLOT( closeAllWindows() ), CTRL+Key_Q );


    QPopupMenu *edit = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Edit"), edit );
    edit->insertItem( tr("&Find"),       this, SLOT(slotFind())    ,CTRL+Key_F );
    edit->insertItem( tr("Find &Next"),  this, SLOT(slotFindNext()),Key_F3 );
    edit->insertItem( tr("&Goto Line"),  this, SLOT(slotGotoLine()),CTRL+Key_G );

    menuBar()->insertSeparator();
    QPopupMenu *action = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Action"), action );
    action->insertItem( tr("&designer"),     this, SLOT(slotQtDesigner())    ,CTRL+Key_D );
    action->insertItem( tr("&insert mask"),  this, SLOT(slotInsertMask())    ,CTRL+Key_I );
    action->insertItem( tr("&make"),         this, SLOT(slotMake())          ,CTRL+Key_M );
    action->insertItem( tr("start &server"), this, SLOT(slotStartServer())   ,CTRL+Key_S );
    action->insertItem( tr("start &browser"),this, SLOT(slotStartPvbrowser()),CTRL+Key_B );

    menuBar()->insertSeparator();
    QPopupMenu *rllib = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Rllib"), rllib );
    rllib->insertItem( tr("&uncomment rllib"),       this, SLOT(slotUncommentRllib()) );
    rllib->insertItem( tr("&uncomment modbus"),      this, SLOT(slotUncommentRllibModbus()) );
    rllib->insertItem( tr("&uncomment siemens tcp"), this, SLOT(slotUncommentRllibSiemensTcp()) );
    rllib->insertItem( tr("&uncomment ppi"),         this, SLOT(slotUncommentRllibPpi()) );

    menuBar()->insertSeparator();
    QPopupMenu *mkdaemon = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Daemon"),mkdaemon);
    mkdaemon->insertItem(tr("&Modbus"),    this, SLOT(slotModbus()));
    mkdaemon->insertItem(tr("&SiemensTCP"),this, SLOT(slotSiemensTCP()));
    mkdaemon->insertItem(tr("&PPI"),       this, SLOT(slotPPI()));

#ifdef unix
    menuBar()->insertSeparator();
    QPopupMenu *linuxmenu = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Linux"),linuxmenu);
    linuxmenu->insertItem(tr("&WriteStartscript"),this, SLOT(slotWriteStartscript()));
#endif

    menuBar()->insertSeparator();
    QPopupMenu * help = new QPopupMenu( this );
    menuBar()->insertItem( tr("&Help"), help );

    help->insertItem( tr("&Create new project"), this, SLOT(newproject()));
    help->insertItem( tr("&Run ProcessViewServer or daemon in the background"), this, SLOT(background()));
    help->insertItem( tr("&Modbus visualization"), this, SLOT(modbus()));
    help->insertItem( tr("Siemens&TCP visualization"), this, SLOT(siemensTCP()));
    help->insertItem( tr("Siemens&PPI visualization"), this, SLOT(siemensPPI()));
    help->insertSeparator();
    help->insertItem( tr("M&anual"), this, SLOT(reference()));
    help->insertItem( tr("&Solutions"), this, SLOT(solutions()));
    help->insertSeparator();
    help->insertItem( tr("&About"), this, SLOT(about()), Key_F1 );
    help->insertItem( tr("About &Qt"), this, SLOT(aboutQt()) );
    help->insertSeparator();
    help->insertItem( tr("What's &This"), this, SLOT(whatsThis()), SHIFT+Key_F1 );
    v = new view( this, "editor" );
    e = v->e;
    e->setFont(QFont("Courier", 12));
    highlighter = new syntax(e);
    v->setFocus();
#ifndef unix
    v->checkBoxSuServer->hide();
#endif
    setCentralWidget( v );
    statusBar()->message( tr("Ready"), 2000 );

    QObject::connect(v->widgetname,      SIGNAL(clicked(QListBoxItem *)), SLOT(slotWidgetname(QListBoxItem *)));

    QObject::connect(v->radioProject,    SIGNAL(clicked()), SLOT(slotProject()));
    QObject::connect(v->radioMain,       SIGNAL(clicked()), SLOT(slotMain()));
    QObject::connect(v->radioHeader,     SIGNAL(clicked()), SLOT(slotHeader()));
    QObject::connect(v->radioMask,       SIGNAL(clicked()), SLOT(slotMask()));
    QObject::connect(v->radioSlots,      SIGNAL(clicked()), SLOT(slotSlots()));

    QObject::connect(v->spinBoxMask,     SIGNAL(valueChanged(int)), SLOT(slotMaskChanged(int)));    

    QObject::connect(v->radioTop,        SIGNAL(clicked()), SLOT(slotTop()));
    QObject::connect(v->radioDefineMask, SIGNAL(clicked()), SLOT(slotDefineMask()));
    QObject::connect(v->radioShowData,   SIGNAL(clicked()), SLOT(slotShowData()));
    QObject::connect(v->radioEvents,     SIGNAL(clicked()), SLOT(slotEvent()));

    QObject::connect(v->comboBoxEvent,   SIGNAL(activated(int)), SLOT(slotComboEvents(int)));

    QObject::connect(v->pushButtonInsertFunction,SIGNAL(clicked()), SLOT(slotInsertFunction()));

    v->comboBoxEvent->insertItem("//###");
    v->comboBoxEvent->insertItem("typedef");
    v->comboBoxEvent->insertItem("Init");
    v->comboBoxEvent->insertItem("NullEvent");
    v->comboBoxEvent->insertItem("ButtonEvent");
    v->comboBoxEvent->insertItem("ButtonPressedEvent");
    v->comboBoxEvent->insertItem("ButtonReleasedEvent");
    v->comboBoxEvent->insertItem("TextEvent");
    v->comboBoxEvent->insertItem("SliderEvent");
    v->comboBoxEvent->insertItem("CheckboxEvent");
    v->comboBoxEvent->insertItem("RadioButtonEvent");
    v->comboBoxEvent->insertItem("GlInitializeEvent");
    v->comboBoxEvent->insertItem("GlPaintEvent");
    v->comboBoxEvent->insertItem("GlResizeEvent");
    v->comboBoxEvent->insertItem("GlIdleEvent");
    v->comboBoxEvent->insertItem("TabEvent");
    v->comboBoxEvent->insertItem("TableTextEvent");
    v->comboBoxEvent->insertItem("TableClickedEvent");
    v->comboBoxEvent->insertItem("SelectionEvent");
    v->comboBoxEvent->insertItem("ClipboardEvent");
    v->comboBoxEvent->insertItem("RightMouseEvent");
    v->comboBoxEvent->insertItem("KeyboardEvent");
    v->comboBoxEvent->insertItem("MouseMovedEvent");
    v->comboBoxEvent->insertItem("MousePressedEvent");
    v->comboBoxEvent->insertItem("MouseReleasedEvent");
    v->comboBoxEvent->insertItem("UserEvent");

    resize(800,600);
    showMaximized();
}


pvbuilder::~pvbuilder()
{
    delete printer;
    delete highlighter;
}



void pvbuilder::newDoc()
{
    //pvbuilder *ed = new pvbuilder;
    //ed->setCaption(tr("Qt Example - Application"));
    //ed->show();
    dlgNewPrj dlg;
    name = dlg.run();
    if(name != "") 
    {
      generate(name);
#ifdef unix
      mysystem("designer mask1.ui");
      mysystem("ui2pvc mask1.ui mask1.cpp");
      mysystem("pvsmake " + name + ".pro -o Makefile;make");
#else
      mysystem("pvb_designer.bat mask1 " + name);
#endif
      //slotMake();
      load(name + ".pro");
      v->radioProject->setChecked(TRUE);
    }
}

void pvbuilder::choose()
{
    QString fn = QFileDialog::getOpenFileName( QString::null, "*.pvproject",this);
    if( !fn.isEmpty() )
    {
      loadProject(fn);
    }
    else
    {
      statusBar()->message( tr("Loading aborted"), 2000 );
    }
}

void pvbuilder::loadProject(const char *prjfile)
{
    QUrl url(prjfile);
    if(url.dirPath() != "") QDir::setCurrent(url.dirPath());
    v->spinBoxMask->setValue(1);
    name = url.fileName();
    name = name.section('.',0,0);
    name = getProject("target=");
    load(name + ".pro");
    v->radioProject->setChecked(TRUE);
}

void pvbuilder::load( const QString &fileName )
{
    if(e->isModified())
    {
      if(QMessageBox::question(this,tr("The file was modified:"),tr("Do you want to save ?"),
         QMessageBox::No, QMessageBox::Yes) == (QMessageBox::Yes || QMessageBox::Ok))
      {
        save();
      }
    }

    filename = fileName;
    QFile f(fileName);
    if(!f.open(IO_ReadOnly)) 
    {
      if(fileName.find("_slots") == -1)
      {
        v->spinBoxMask->setValue(1);
        f.setName("mask1.cpp");
        filename = "mask1.cpp";
        if(!f.open(IO_ReadOnly)) return;
      }
      else if(fileName.startsWith("mask")) 
      {
        v->spinBoxMask->setValue(1);
        f.setName("mask1_slots.h");
        filename = "mask1_slots.h";
        if(!f.open(IO_ReadOnly)) return;
      }
      else  
      {
        return;
      }
    }
    QTextStream ts( &f );
    ts.setEncoding(QTextStream::UnicodeUTF8);
    e->setText( ts.read() );
    e->setModified( FALSE );
#ifdef unix
    setCaption( "pvbuilder: " + QDir::currentDirPath() + "/" + filename );
#else
    setCaption( "pvbuilder: " + QDir::currentDirPath() + "\\" + filename );
#endif
    f.close();
    getWidgetNames(fileName);
    if     (fileName.startsWith("mask") && fileName.find("_slots") != -1)
    {
      slotComboEvents(v->comboBoxEvent->currentItem());
    }
    else if(fileName.startsWith("mask"))
    {
      if(v->radioTop->isChecked())        slotTop();
      if(v->radioDefineMask->isChecked()) slotDefineMask();
      if(v->radioShowData->isChecked())   slotShowData();
      if(v->radioEvents->isChecked())     slotEvent();
    }
    statusBar()->message( tr("Loaded document %1").arg(fileName), 2000 );
}


void pvbuilder::save()
{
    if(notLoaded()) return;
    if( filename.isEmpty() )
    {
      saveAs();
      return;
    }

    QString text = e->text();
    QFile f( filename );
    if( !f.open( IO_WriteOnly ) )
    {
      statusBar()->message( tr("Could not write to %1").arg(filename), 2000);
      return;
    }

    QTextStream t( &f );
    t.setEncoding(QTextStream::UnicodeUTF8);
    t << text;
    f.close();

    e->setModified( FALSE );

    setCaption("pvbuilder: " + filename );

    statusBar()->message( tr( "File %1 saved" ).arg( filename ), 2000 );
}


void pvbuilder::saveAs()
{
    if(notLoaded()) return;
    QString fn = QFileDialog::getSaveFileName( QString::null, QString::null, this);
    if ( !fn.isEmpty() ) 
    {
      filename = fn;
      save();
    }
    else 
    {
      statusBar()->message( tr("Saving aborted"), 2000 );
    }
}


void pvbuilder::print()
{
    if(notLoaded()) return;
    // ###### Rewrite to use QSimpleRichText to print here as well
    const int Margin = 10;
    int pageNo = 1;

    if ( printer->setup(this) ) 
    {   // printer dialog
      statusBar()->message( tr("Printing...") );
      QPainter p;
      if( !p.begin( printer ) )               // paint on printer
        return;

      p.setFont( e->font() );
      int yPos  = 0;      // y-position for each line
      QFontMetrics fm = p.fontMetrics();
      QPaintDeviceMetrics metrics( printer ); // need width/height
            // of printer surface
      for( int i = 0 ; i < e->lines() ; i++ ) 
      {
        if ( Margin + yPos > metrics.height() - Margin ) {
        QString msg( "Printing (page " );
        msg += QString::number( ++pageNo );
        msg += ")...";
        statusBar()->message( msg );
        printer->newPage();   // no more room on this page
        yPos = 0;     // back to top of page
      }
      p.drawText( Margin, Margin + yPos,
      metrics.width(), fm.lineSpacing(),
      ExpandTabs | DontClip,
      e->text( i ) );
      yPos = yPos + fm.lineSpacing();
    }
    p.end();        // send job to printer
    statusBar()->message( tr("Printing completed"), 2000 );
  }
  else 
  {
    statusBar()->message( tr("Printing aborted"), 2000 );
  }
}

void pvbuilder::closeEvent( QCloseEvent* ce )
{
    if ( !e->isModified() ) 
    {
      ce->accept();
      return;
    }

    switch( QMessageBox::information( this, tr("pvbuilder"),
              tr("Do you want to save the changes"
              " to the document?"),
              tr("Yes"), tr("No"), tr("Cancel"),
              0, 1 ) ) 
    {
      case 0:
        save();
        ce->accept();
        break;
      case 1:
        ce->accept();
        break;
      case 2:
        default: // just for sanity
        ce->ignore();
        break;
    }
}

void pvbuilder::slotInsertMask()
{
  QStringList::Iterator it;
  QStringList list;
  QString line,buf;
  QTextStream in,out;
  int found,found2;
  QFile f;
  if(notLoaded()) return;

  // determine imask
  int imask = 1;
  while(1)
  {
    buf.sprintf("mask%d.cpp",imask);
    f.setName(buf);
    if(!f.open(IO_ReadOnly)) break; 
    f.close();
    imask++;
  }

  // generate .pro file
  found = found2 = 0;
  f.setName(name + ".pro");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    out << str << "\n";
    if(found == 0 && str.find("SOURCES += main.cpp") != -1)
    {
      buf.sprintf("           mask%d.cpp \\\n",imask);
      out << buf;
      found = 1;
    }
    if(found2 == 0 && str.find("HEADERS += pvapp.h") != -1)
    {
      buf.sprintf("           mask%d_slots.h \\\n",imask);
      out << buf;
      found2 = 1;
    }
  }
  f.close();

  // generate pvapp.h file
  found = 0;
  f.setName("pvapp.h");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    out << str << "\n";
    if(found == 0 && str.find("show_mask") != -1)
    {
      buf.sprintf("int show_mask%d(PARAM *p);\n",imask);
      out << buf;
      found = 1;
    }
  }
  f.close();

  // generate main.cpp file
  found = 0;
  f.setName("main.cpp");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    out << str << "\n";
    if(found == 1)
    {
      buf.sprintf("      case %d:\n",imask);
      out << buf;
      buf.sprintf("        pvStatusMessage(p,-1,-1,-1,\"mask%d\");\n",imask);
      out << buf;
      buf.sprintf("        ret = show_mask%d(p);\n",imask);
      out << buf;
      buf.sprintf("        break;\n");
      out << buf;
      found = 2;
    }
    if(found == 0 && str.find("switch(ret)") != -1)
    {
      found = 1;
    }
  }
  f.close();

  doAction(DO_INSERT_DESIGN_AND_MAKE);
}

void pvbuilder::slotUncommentRllib()
{
  slotUncommentRllib(0);
}

void pvbuilder::slotUncommentRllibModbus()
{
  slotUncommentRllib(DATA_AQUISITION_MODBUS);
}

void pvbuilder::slotUncommentRllibSiemensTcp()
{
  slotUncommentRllib(DATA_AQUISITION_SIEMENS_TCP);
}

void pvbuilder::slotUncommentRllibPpi()
{
  slotUncommentRllib(DATA_AQUISITION_PPI);
}

void pvbuilder::slotUncommentRllib(int data_aquisiton)
{
  QStringList::Iterator it;
  QStringList list;
  QString line,buf;
  QTextStream in,out;
  int i;
  QFile f;
  QString pattern;
  int debug = 0;

  if(e->isModified()) save();
  switch(data_aquisiton)
  {
    case 0:
      break;
    default:
      slotUncommentRllib();
      break;
  }
  switch(data_aquisiton)
  {
    case DATA_AQUISITION_MODBUS:
      pattern = "modbus";
      break;
    case DATA_AQUISITION_SIEMENS_TCP:
      pattern = "siemens";
      break;
    case DATA_AQUISITION_PPI:
      pattern = "ppi";
      break;
    default:
      pattern = "rllib";
      break;
  }

  if(notLoaded()) return;

  // determine imask
  int imask = 1;
  while(1)
  {
    buf.sprintf("mask%d.cpp",imask);
    f.setName(buf);
    if(!f.open(IO_ReadOnly)) break; 
    f.close();
    imask++;
  }

  // generate .pro file
  f.setName(name + ".pro");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    if(str.find(pattern,0,FALSE) != -1 && str.startsWith("#"))
    {
      out << str.mid(1) << "\n"; // uncomment '#'
      if(debug) printf("uncomment pro: %s\n", (const char *) str);
    }
    else
    {
      out << str << "\n";
      if(debug) printf("pro: %s\n", (const char *) str);
    }
  }
  f.close();

  // generate pvapp.h file
  f.setName("pvapp.h");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    if(str.find(pattern,0,FALSE) != -1 && str.startsWith("//#include"))
    {
      out << str.mid(2) << "\n"; // uncomment '//'
      if(debug) printf("uncomment pvapp: %s\n", (const char *) str);
    }
    else
    {
      out << str << "\n";
      if(debug) printf("pvapp: %s\n", (const char *) str);
    }
  }
  f.close();

  // generate main.cpp file
  f.setName("main.cpp");
  if(!f.open(IO_ReadOnly)) return; 
  in.setDevice(&f);

  list.clear();
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    list.append(line);
  }
  f.close();

  if(!f.open(IO_WriteOnly)) return; 
  out.setDevice(&f);
  for(it = list.begin(); it != list.end(); ++it)
  {
    QString str(*it);
    if(str.find(pattern,0,FALSE) != -1 && str.startsWith("//rl"))
    {
      out << str.mid(2) << "\n"; // uncomment '//'
      if(debug) printf("uncomment main: %s\n", (const char *) str);
    }
    else
    {
      out << str << "\n";
      if(debug) printf("main: %s\n", (const char *) str);
    }
  }
  f.close();

  for(i=1; i<imask; i++)
  {
    buf.sprintf("mask%d_slots.h",i);
    f.setName(buf);
    if(!f.open(IO_ReadOnly)) return; 
    in.setDevice(&f);

    list.clear();
    while(1)
    {
      line = in.readLine();
      if(line.isNull()) break;
      list.append(line);
    }
    f.close();

    if(!f.open(IO_WriteOnly)) return; 
    out.setDevice(&f);
    for(it = list.begin(); it != list.end(); ++it)
    {
      QString str(*it);
      if(str.find(pattern,0,FALSE) != -1 && str.startsWith("//extern"))
      {
        out << str.mid(2) << "\n"; // uncomment '//'
        if(debug) printf("uncomment slots: %s\n", (const char *) str);
      }
      else
      {
        out << str << "\n";
        if(debug) printf("slots: %s\n", (const char *) str);
      }
    }
    f.close();
  }
  if(debug) printf("load(%s)\n",(const char *) name);
  load(name + ".pro");
  v->radioProject->setChecked(TRUE);
}

int pvbuilder::notLoaded()
{
  if(name != "") return 0;
  QMessageBox::information(this,tr("no project loaded"),tr("Pease load project"),QMessageBox::Ok);
  return 1;
}

void pvbuilder::slotFind()
{
  bool ok;

  QString text = QInputDialog::getText(tr("pvbuilder"), tr("Find string"), QLineEdit::Normal, pattern, &ok, this);
  if(ok && !text.isEmpty()) 
  {
    // user entered something and pressed OK
    pattern = text;
    if(e->find(pattern,FALSE,FALSE,TRUE,&para,&index))
    {
      e->scrollToBottom();
      e->setCursorPosition(para,0);
      e->ensureCursorVisible();
    }
  }
  else
  {
    QMessageBox::information(this, tr("pvbuilder"), tr("no more found"));
  }
}

void pvbuilder::slotFindNext()
{
  e->removeSelection();
  e->moveCursor(QTextEdit::MoveWordForward,0);
  if(e->find(pattern,FALSE,FALSE,TRUE))
  {
    e->scrollToBottom();
    e->ensureCursorVisible();
  }
  else
  {
    QMessageBox::information(this, tr("pvbuilder"), tr("no more found"));
  }
}

void pvbuilder::slotGotoLine()
{
  int line;

  line = QInputDialog::getInteger(tr("pvbuilder"), tr("Goto Line"));
  if(line > 0)
  {
    para = line - 1;
    e->setSelection(para,0,para,1024);
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::slotQtDesigner()
{
  if(notLoaded()) return;
  doAction(DO_DESIGN_AND_MAKE);
}

void pvbuilder::slotMake()
{
  if(notLoaded()) return;
  doAction(DO_MAKE);
}

void pvbuilder::slotStartServer()
{
  if(notLoaded()) return;
  doAction(DO_MAKE_AND_RUN);
}

void pvbuilder::slotStartPvbrowser()
{
  if(notLoaded()) return;
  doAction(DO_START_BROWSER);
}

void pvbuilder::doAction(int what)
{
  int i;
  QString buf;
  if(e->isModified())
  {
    if(what == DO_MAKE || what == DO_MAKE_AND_RUN)
    {
      save();
    }
    else if(QMessageBox::question(this,tr("The file was modified:"),tr("Do you want to save ?"),
       QMessageBox::No,
       QMessageBox::Yes)
       ==
       (QMessageBox::Yes || QMessageBox::Ok))
    {
      save();
    }
  }
  switch(what)
  {
    case DO_MAKE:
#ifdef unix
      mysystem("xterm -e \"make;echo ready;read\"&");
#else
      system("pvb_make.bat " + name);
#endif  
      break;
    case DO_MAKE_AND_RUN:
#ifdef unix
      if(v->checkBoxSuServer->isChecked())
      {
        mysystem("xterm -e \"make;su -c ./" + name + "\"&");
      }
      else
      {
        mysystem("xterm -e \"make;./" + name + "\"&");
      }
#else
      system("start pvb_server.bat " + name);
#endif
      break;
    case DO_DESIGN_AND_MAKE:
#ifdef unix
      buf.sprintf("designer mask%d.ui;make",v->spinBoxMask->value());
      mysystem(buf);
#else
      buf.sprintf("pvb_designer.bat mask%d %s",v->spinBoxMask->value(),(const char *) getProject("target="));
      system(buf);
#endif
      break;
    case DO_INSERT_DESIGN_AND_MAKE:
      i = 1;
      while(1)
      {
        buf.sprintf("mask%d.cpp",i);
        QFile f(buf);
        if(!f.open(IO_ReadOnly)) break; 
        f.close();
        i++;
      }
#ifdef unix
      buf.sprintf("designer mask%d.ui;ui2pvc mask%d.ui mask%d.cpp;pvsmake " + name + ".pro -o Makefile;make",i,i,i);
      mysystem(buf);
#else
      buf.sprintf("pvb_designer.bat mask%d %s",i,(const char *) getProject("target="));
      system(buf);
#endif
      break;
    case DO_START_SERVER:
#ifdef unix
      mysystem("xterm -e su -c \"./" + name + ";exit\" &");
#else
      system("start pvb_server.bat " + name);
#endif
      break;
    case DO_START_BROWSER:
#ifdef unix
      mysystem("processview &");
#else
      system("start processview");
#endif
      break;
    default:
      break;
  }
  load(name + ".pro");
  v->radioProject->setChecked(TRUE);
}

void pvbuilder::slotPPI()
{
  if(notLoaded()) return;
  dlgDaemon dlg(PPI_DAEMON);
  dlg.run();
}

void pvbuilder::slotSiemensTCP()
{
  if(notLoaded()) return;
  dlgDaemon dlg(SIEMENSTCP_DAEMON);
  dlg.run();
}

void pvbuilder::slotModbus()
{
  if(notLoaded()) return;
  dlgDaemon dlg(MODBUS_DAEMON);
  dlg.run();
}

void pvbuilder::about()
{
    QMessageBox::about( this, tr("Build a pvbrowser application"),
      tr("You can create your pvserver\n"
      "nearly without programming.\n"
      "Just a few clicks.\n"
      "But you are not limited to that"));
}


void pvbuilder::aboutQt()
{
    QMessageBox::aboutQt( this, tr("pvbuilder") );
}

void pvbuilder::slotWidgetname(QListBoxItem *item)
{
  QString text;
  if(item == NULL) return;
  dlgPasteWidget dlg(item->text());
  text = dlg.run();
  if(text != "") e->insert(text+"\n");
}

void pvbuilder::slotProject()
{
  load(name + ".pro");
  v->radioProject->setChecked(TRUE);
}

void pvbuilder::slotMain()
{
  load("main.cpp");
  v->radioMain->setChecked(TRUE);
}

void pvbuilder::slotHeader()
{
  load("pvapp.h");
  v->radioHeader->setChecked(TRUE);
}

void pvbuilder::slotMask()
{
  QString mask = "mask" + v->spinBoxMask->text() + ".cpp";
  load(mask);
  v->radioMask->setChecked(TRUE);
}

void pvbuilder::slotSlots()
{
  QString mask = "mask" + v->spinBoxMask->text() + "_slots.h";
  load(mask);
  v->radioSlots->setChecked(TRUE);
}

void pvbuilder::slotMaskChanged(int val)
{
  QString buf;
  buf.sprintf("%d",val);
  QString mask = "mask" + buf + "_slots.h";
  load(mask);
  v->radioSlots->setChecked(TRUE);
}

void pvbuilder::slotTop()
{
  int para  = 0;
  int index = 0;  
  if(e->find("#include \"pvapp.h\"",FALSE,FALSE,TRUE,&para,&index))
  {
    e->removeSelection();
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::slotDefineMask()
{
  int para  = 0;
  int index = 0;  
  if(e->find("static int defineMask(PARAM *p)",FALSE,FALSE,TRUE,&para,&index))
  {
    e->removeSelection();
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::slotShowData()
{
  int para  = 0;
  int index = 0;
  if(e->find("static int showData(PARAM *p, DATA *d)",FALSE,FALSE,TRUE,&para,&index))
  {
    e->removeSelection();
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::slotEvent()
{
  int para  = 0;
  int index = 0;  
  if(e->find("case BUTTON_EVENT:",FALSE,FALSE,TRUE,&para,&index))
  {
    e->removeSelection();
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::slotComboEvents(int i)
{
  int para  = 0;
  int index = 0;  
  if(e->find(v->comboBoxEvent->text(i),FALSE,FALSE,TRUE,&para,&index))
  {
    e->scrollToBottom();
    e->setCursorPosition(para,0);
    e->ensureCursorVisible();
  }
}

void pvbuilder::generate(QString name)
{
  QTextStream out;
  QFile project(name+".pvproject");
  QFile pro(name+".pro"); 
  QFile main("main.cpp");
  QFile header("pvapp.h");
  if(!project.open(IO_WriteOnly)) return;

  out.setDevice(&project);
  out << "target=" << name << "\n";
  project.close();

  if(!pro.open(IO_WriteOnly))     return;
  out.setDevice(&pro);
  out << "######################################################################\n";
  out << "# generated by pvbuilder at: " << QDateTime::currentDateTime().toString() << "\n";
  out << "######################################################################\n";
  out << "\n";
  out << "TEMPLATE = app\n";
  out << "CONFIG   = warn_on release\n";
  out << "\n";
  out << "# Input\n";
  out << "HEADERS += pvapp.h      \\\n";
  out << "           mask1_slots.h\n";
  out << "SOURCES += main.cpp     \\\n";
  out << "           mask1.cpp\n";
  out << "\n";
  out << "unix:LIBS          += /usr/lib/libpvsmt.so /usr/lib/libpthread.so\n";
  out << "#unix:LIBS         += /usr/lib/libpvsid.so\n";
  out << "#unix:LIBS         += /usr/lib/librllib.so\n";
  out << "#unix:INCLUDEPATH  += /usr/local/include/rllib\n";
  out << "\n";
  out << "win32:LIBS         += $(PVBDIR)\\bin\\serverlib.lib wsock32.lib\n";
  out << "win32:INCLUDEPATH  += $(PVBDIR)\\inc\n";
  out << "#win32:LIBS        += $(RLLIBDIR)\\rllib.lib\n";
  out << "#win32:INCLUDEPATH += $(RLLIBDIR)\n";
  out << "\n";
  out << "#DEFINES += USE_INETD\n";
  out << "TARGET = " << name << "\n";
  pro.close();

  if(!main.open(IO_WriteOnly))    return;
  out.setDevice(&main);
  out << "/***************************************************************************\n";
  out << "                          main.cpp  -  description\n";
  out << "                             -------------------\n";
  out << "    begin                : " << QDateTime::currentDateTime().toString() << "\n";
  out << "    generated by         : pvbuilder (C) 2000-2005 by Rainer Lehrig\n";
  out << "    email                : lehrig@t-online.de\n";
  out << " ***************************************************************************/\n";
  out << "#include \"pvapp.h\"\n";
  out << "// todo: comment me out. you can insert these objects as extern in your masks.\n";
  out << "//rlModbusClient     modbus(modbusdaemon_MAILBOX,modbusdaemon_SHARED_MEMORY,modbusdaemon_SHARED_MEMORY_SIZE);\n";
  out << "//rlSiemensTCPClient siemensTCP(siemensdaemon_MAILBOX,siemensdaemon_SHARED_MEMORY,siemensdaemon_SHARED_MEMORY_SIZE);\n";
  out << "//rlPPIClient        ppi(ppidaemon_MAILBOX,ppidaemon_SHARED_MEMORY,ppidaemon_SHARED_MEMORY_SIZE);\n";
  out << "\n";
  out << "int pvMain(PARAM *p)\n";
  out << "{\n";
  out << "int ret;\n";
  out << "\n";
  out << "  pvSetCaption(p,\"" << name << "\");\n";
  out << "  pvResize(p,0,1280,1024);\n";
  out << "  //pvScreenHint(p,1024,768); // this may be used to automatically set the zoomfactor\n";
  out << "  ret = 1;\n";
  out << "  pvGetInitialMask(p);\n";
  out << "  if(strcmp(p->initial_mask,\"mask1\") == 0) ret = 1;\n";
  out << "\n";
  out << "  while(1)\n";
  out << "  {\n";
  out << "    switch(ret)\n";
  out << "    {\n";
  out << "      case 1:\n";
  out << "        pvStatusMessage(p,-1,-1,-1,\"mask1\");\n";
  out << "        ret = show_mask1(p);\n";
  out << "        break;\n";
  out << "      default:\n";
  out << "        return 0;\n";
  out << "    }\n";
  out << "  }\n";
  out << "}\n";
  out << "\n";
  out << "#ifdef USE_INETD\n";
  out << "int main(int ac, char **av)\n";
  out << "{\n";
  out << "PARAM p;\n";
  out << "\n";
  out << "  pvInit(ac,av,&p);\n";
  out << "  /* here you may interpret ac,av and set p->user to your data */\n";
  out << "  pvMain(&p);\n";
  out << "  return 0;\n";
  out << "}\n";
  out << "#else  // multi threaded server\n";
  out << "int main(int ac, char **av)\n";
  out << "{\n";
  out << "PARAM p;\n";
  out << "int   s,i;\n";
  out << "\n";
  out << "  pvInit(ac,av,&p);\n";
  out << "  /* here you may interpret ac,av and set p->user to your data */\n";
  out << "  for(i=1; i<ac; i++)\n";
  out << "  {\n";
  out << "    if(strcmp(av[i],\"-1\") == 0)\n";
  out << "    {\n";
  out << "      s = pvAccept(&p);\n";
  out << "      if(s > 0)\n";
  out << "      {\n";
  out << "        p.s = s;\n";
  out << "        p.free = 0;\n";
  out << "        pvMain(&p);\n";
  out << "      }\n";
  out << "      else\n";
  out << "      {\n";
  out << "        printf(\"pvAccept error\\n\");\n";
  out << "      }\n";
  out << "      return 0;\n";
  out << "    }\n";
  out << "  }\n";
  out << "  while(1)\n";
  out << "  {\n";
  out << "    s = pvAccept(&p);\n";
  out << "    if(s != -1) pvCreateThread(&p,s);\n";
  out << "    else        break;\n";
  out << "  }\n";
  out << "  return 0;\n";
  out << "}\n";
  out << "#endif\n";
  main.close();

  if(!header.open(IO_WriteOnly))  return;
  out.setDevice(&header);
  out << "/***************************************************************************\n";
  out << "                          pvapp.h  -  description\n";
  out << "                             -------------------\n";
  out << "    begin                : " << QDateTime::currentDateTime().toString() << "\n";
  out << "    generated by         : pvbuilder (C) 2000-2005 by Rainer Lehrig\n";
  out << "    email                : lehrig@t-online.de\n";
  out << " ***************************************************************************/\n";
  out << "#ifndef _PVAPP_H_\n";
  out << "#define _PVAPP_H_\n";
  out << "\n";
  out << "#include \"processviewserver.h\"\n";
  out << "// todo: comment me out\n";
  out << "//#include \"rlmodbusclient.h\"\n";
  out << "//#include \"rlsiemenstcpclient.h\"\n";
  out << "//#include \"rlppiclient.h\"\n";
  out << "//#include \"modbusdaemon.h\"             // this is generated\n";
  out << "//#include \"siemensdaemon.h\"            // this is generated\n";
  out << "//#include \"ppidaemon.h\"                // this is generated\n";
  out << "\n";
  out << "int show_mask1(PARAM *p);\n";
  out << "\n";
  out << "#endif\n";
  header.close();
}

QString pvbuilder::getProject(QString tag)
{
  QTextStream in;
  QFile project(name + ".pvproject");
  if(!project.open(IO_ReadOnly)) return "";

  in.setDevice(&project);
  while(1)
  {
    QString line = in.readLine();
    if(line.isNull()) break;
    if(line.startsWith(tag))
    {
      project.close();
      return line.section('=',1,1);
    }
  }
  project.close();
  return "";
}

void pvbuilder::getWidgetNames(const QString &filename)
{
  QString line;
  v->widgetname->clear();
  if(!filename.startsWith("mask")) return;
  QTextStream in;
  QFile mask("mask" + v->spinBoxMask->text() + ".cpp");
  if(!mask.open(IO_ReadOnly)) return; 

  in.setDevice(&mask);
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    if(line.find("ID_MAIN_WIDGET") != -1) break;
  }
  while(1)
  {
    line = in.readLine();
    if(line.isNull()) break;
    if(line.find("ID_END_OF_WIDGETS") != -1) break;
    else
    {
      line.remove(' ');
      line.remove(',');
      v->widgetname->insertItem(line);
    }
  }
  mask.close();
}

void pvbuilder::slotInsertFunction()
{
  static dlgInsertFunction dlg;
  QString text = dlg.run();
  if(text != "") e->insert(text+"\n");
}

void pvbuilder::slotWriteStartscript()
{
  QTextStream out;
  QFile script("startscript");
  if(!script.open(IO_WriteOnly)) return;

  out.setDevice(&script);
  out << "#!/bin/sh\n";
  out << "# generated by pvbuilder. Please adjust DAEMON_PATH and DAEMON to your needs.\n";
  out << "# copy this file to /etc/init.d and link it to runlevel 5 .\n";
  out << "DAEMON_PATH=" << QDir::current().path() << "\n";
  out << "DAEMON=" << name << "\n";
  out << ". /etc/rc.status\n";
  out << "rc_reset\n";
  out << "case \"$1\" in\n";
  out << "    start)\n";
  out << "      echo -n \"Starting $DAEMON\"\n";
  out << "      startproc $DAEMON_PATH/$DAEMON -sleep=100 -cd=$DAEMON_PATH > /dev/null\n";
  out << "      rc_status -v\n";
  out << "      ;;\n";
  out << "    stop)\n";
  out << "      echo -n \"Shutting down $DAEMON\"\n";
  out << "      killproc -TERM $DAEMON_PATH/$DAEMON\n";
  out << "      rc_status -v\n";
  out << "      ;;\n";
  out << "    try-restart)\n";
  out << "      $0 status >/dev/null &&  $0 restart\n";
  out << "      rc_status\n";
  out << "      ;;\n";
  out << "    restart)\n";
  out << "      $0 stop\n";
  out << "      $0 start\n";
  out << "      rc_status\n";
  out << "      ;;\n";
  out << "    force-reload)\n";
  out << "      echo -n \"Reload service $DAEMON\"\n";
  out << "      killproc -HUP $DAEMON_PATH/$DAEMON\n";
  out << "      rc_status -v\n";
  out << "      ;;\n";
  out << "    reload)\n";
  out << "      echo -n \"Reload service $DAEMON\"\n";
  out << "      killproc -HUP $DAEMON_PATH/$DAEMON\n";
  out << "      rc_status -v\n";
  out << "      ;;\n";
  out << "    status)\n";
  out << "      echo -n \"Checking for service $DAEMON\"\n";
  out << "      checkproc $DAEMON_PATH/$DAEMON\n";
  out << "      rc_status -v\n";
  out << "      ;;\n";
  out << "    *)\n";
  out << "    echo \"Usage: $0 {start|stop|status|try-restart|restart|force-reload|reload}\"\n";
  out << "    exit 1\n";
  out << "    ;;\n";
  out << "esac\n";
  out << "rc_exit\n";
  script.close();

  mysystem("chmod ugoa+x startscript");
}

void pvbuilder::newproject()
{
  dlghtml.setCaption("New Project");
  dlghtml.setSource("newproject1.html");
  dlghtml.show();
  dlghtml.raise();
}

void pvbuilder::background()
{
  dlghtml.setCaption("Running in Background");
  dlghtml.setSource("background1.html");
  dlghtml.show();
  dlghtml.raise();
}

void pvbuilder::modbus()
{
  dlghtml.setCaption("Modbus");
  dlghtml.setSource("modbus1.html");
  dlghtml.show();
  dlghtml.raise();
}

void pvbuilder::siemensTCP()
{
  dlghtml.setCaption("SiemensTCP");
  dlghtml.setSource("siemensTCP1.html");
  dlghtml.show();
  dlghtml.raise();
}

void pvbuilder::siemensPPI()
{
  dlghtml.setCaption("SiemensPPI");
  dlghtml.setSource("siemensPPI1.html");
  dlghtml.show();
  dlghtml.raise();
}

void pvbuilder::reference()
{
#ifdef unix
  browse("/usr/share/doc/packages/ProcessView/en/html/index.html");
#endif
#ifdef _WIN32
  QString file;
  file  = getenv("PVBDIR");
  file += "\\..\\doc\\html\\index.html";
  browse(file);
#endif
}

void pvbuilder::solutions()
{
#ifdef unix
  browse("/usr/share/doc/packages/ProcessView/en/solutions/index.html");
#endif
#ifdef _WIN32
  QString file;
  file  = getenv("PVBDIR");
  file += "\\..\\doc\\en\\solutions\\index.html";
  browse(file);
#endif
}

void pvbuilder::browse(const char *html)
{
#ifdef unix
  QString c;
  c  = "konqueror ";
  c += html;
  c += " &";
  mysystem(c);
#endif
#ifdef _WIN32
  char buf[4096];
  if(get_iexplore(buf) < 0) return;
  QString c;
  c += buf;
  c += " \"";
  c += html;
  c += "\"";
  //QMessageBox::information(0, "Application name",c);
  mysystem(c);
#endif
}

