#include <qapplication.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qpainter.h>
#include "cpupiemarker.h"
#include "cpuaveragecurve.h"
#include "cpuplot.h"

class TimeScaleDraw: public QwtScaleDraw
{
public:
	TimeScaleDraw(const QTime &base):
		baseTime(base)
	{
		setLabelRotation(-50.0);
		setLabelAlignment(Qt::AlignLeft | Qt::AlignBottom);
	}
	virtual QString label(double v) const
	{
		QTime upTime = baseTime.addSecs((int)v);
		return upTime.toString();
	}
private:
	QTime baseTime;
};

CpuPlot::CpuPlot(QWidget *parent):
	QwtPlot(parent),
	dataCount(0)
{
	setCanvasBackground(Qt::darkGray);

	setAutoLegend(TRUE);
	setLegendPos(Qwt::Right);

    setAxisTitle(QwtPlot::xBottom, " System Uptime [h:m:s]");
	setAxisScaleDraw(QwtPlot::xBottom, 
		new TimeScaleDraw(cpuStat.upTime()));
	setAxisScale(QwtPlot::xBottom, 0, HISTORY);

    setAxisTitle(QwtPlot::yLeft, "Cpu Usage [%]");
	setAxisScale(QwtPlot::yLeft, 0, 100);

	data[Total].curve = insertCurve("Total");
	setCurvePen(data[Total].curve, QPen(black));
	setCurveBrush(data[Total].curve, 
		QBrush(QColor(), QBrush::SolidPattern));

	data[Idle].curve = insertCurve("Idle");
	setCurvePen(data[Idle].curve, QPen(lightGray));
	setCurveBrush(data[Idle].curve, 
		QBrush(QColor(), QBrush::Dense4Pattern));

	data[User].curve = insertCurve("User");
	setCurvePen(data[User].curve, QPen(blue));
	setCurveBrush(data[User].curve, 
		QBrush(QColor(), QBrush::Dense2Pattern));

	data[System].curve = insertCurve("System");
	setCurvePen(data[System].curve, QPen(red));
	setCurveBrush(data[System].curve, 
		QBrush(QColor(), QBrush::Dense2Pattern));

	CpuAverageCurve *curve = new CpuAverageCurve(this);
	data[Average].curve = insertCurve(curve);
	setCurveTitle(data[Average].curve, "Average");
	setCurvePen(data[Average].curve, QPen(NoPen));
	setCurveBrush(data[Average].curve, 
		QBrush(QColor(Qt::black), QBrush::BDiagPattern));

	CpuPieMarker *pie = new CpuPieMarker(this);
	(void)insertMarker(pie);
	
	toggleCurve(data[Total].curve);
	toggleCurve(data[Idle].curve);
	toggleCurve(data[Average].curve);

	for ( int i = 0; i < HISTORY; i++ )
		timeData[HISTORY - 1 - i] = i;

	(void)startTimer(1000); // 1 second
	replot();

	connect(this, SIGNAL(legendClicked(long)),
		SLOT(toggleCurve(long)));
}

void CpuPlot::drawCanvas(QPainter *p)
{
	QwtDiMap yMap = canvasMap(QwtPlot::yLeft);

	QRect r = p->window();
	r.setHeight(yMap.transform(80.0) - r.top());
	p->fillRect(r, QColor(Qt::white));

	r.setTop(yMap.transform(80.0));
	r.setHeight(yMap.transform(40.0) - yMap.transform(80.0));
	p->fillRect(r, QColor(Qt::gray));

	QwtPlot::drawCanvas(p);
}

void CpuPlot::timerEvent(QTimerEvent *)
{
	for ( int i = dataCount; i > 0; i-- )
	{
		for ( int c = 0; c < NCpuData; c++ )
		{
			if ( i < HISTORY )
				data[c].data[i] = data[c].data[i-1];
		}
	}

	cpuStat.statistic(data[User].data[0], data[System].data[0]);

	data[Total].data[0] = data[User].data[0] + 
		data[System].data[0];
	data[Idle].data[0] = 100.0 - data[Total].data[0];
	data[Average].data[0] = cpuStat.average();

	if ( dataCount < HISTORY )
		dataCount++;

	for ( int j = 0; j < HISTORY; j++ )
		timeData[j]++;

	setAxisScale(QwtPlot::xBottom, 
		timeData[HISTORY - 1], timeData[0]);

	for ( int c = 0; c < NCpuData; c++ )
	{
		setCurveRawData(data[c].curve, 
			timeData, data[c].data, dataCount);
	}

	replot();
}

void CpuPlot::toggleCurve(long curveId)
{
	QwtPlotCurve *c = curve(curveId);
	if ( c )
	{
		c->setEnabled(!c->enabled());
		replot();
	}
}

int main(int argc, char **argv)
{
    QApplication a(argc, argv); 
    
	QVBox vBox;
	vBox.setCaption("Cpu Plot");

    CpuPlot *plot = new CpuPlot(&vBox);
	plot->setTitle("History");
	plot->setMargin(5);

	QString info("Press the legend to en/disable a curve");

	(void)new QLabel(info, &vBox);
    a.setMainWidget(&vBox);

    vBox.resize(500,300);
    vBox.show();

    return a.exec();  
}   

