/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

// vim: expandtab

#ifndef QWT_LEGEND_H
#define QWT_LEGEND_H

#include <qpen.h>
#include <qscrollview.h>
#include <qpushbutton.h>
#include <qintdict.h>
#include "qwt_global.h"
#include "qwt_symbol.h"

class QPainter;

#if defined(QWT_TEMPLATEDLL)
// MOC_SKIP_BEGIN
template class QWT_EXPORT QIntDict<QWidget>;
// MOC_SKIP_END
#endif

/*!
  \brief A legend button

  QwtLegendButton represents a curve on a legend. 
  It displays an curve identifier with an explaining text. 
  The identifier might be a combination of curve symbol and line.
  
  \sa QwtLegend, QwtCurve
*/

class QWT_EXPORT QwtLegendButton: public QPushButton
{
    Q_OBJECT

public:

    enum IdentifierMode
    {
        NoIdentifier = 0,
        ShowLine = 1,
        ShowSymbol = 2,
        ShowText = 4
    };

    QwtLegendButton(QWidget *parent = 0, const char *name = 0);
    QwtLegendButton(const QwtSymbol &, const QPen &,
        const QString &, QWidget *parent = 0, const char *name = 0);

    /*! 
        Or'd values of IdentifierMode.
    The default is QwtLegendButton::ShowLine | QwtLegendButton::ShowText.
        \sa QwtLegendButton::setIdentifierMode()
    */
    int identifierMode() const { return d_identifierMode; }

    /*!     
        \return The curve symbol.
        \sa QwtLegendButton::setSymbol()
    */
    const QwtSymbol& symbol() const { return d_symbol; }

    /*!     
        \return The curve pen.
        \sa QwtLegendButton::setCurvePen()
    */
    const QPen& curvePen() const { return d_curvePen; }

    void setIdentifierMode(int);
    void setSymbol(const QwtSymbol &);
    void setCurvePen(const QPen &);
    
    virtual void drawContents(QPainter *p, const QRect &) const; 

    virtual QSize sizeHint() const;
    virtual QSize minimumSizeHint() const;

protected:
    virtual void updateIconset();
    virtual void drawButtonLabel(QPainter *);
    virtual void drawIdentifier(QPainter *, const QRect &) const;

private:
    int d_identifierMode;
    QwtSymbol d_symbol;
    QPen d_curvePen;
};

/*!
  \brief The legend widget

  The QwtLegend widget is a tabular arrangement of legend items. Legend
  items might be any type of widget, but in general they will be
  a QwtLegendButton.

  \sa QwtLegendButton, QwtPlot
*/

class QWT_EXPORT QwtLegend : public QScrollView
{
    Q_OBJECT

public:

    enum LegendDisplayPolicy
    {
        None = 0,
        Fixed = 1,
        Auto = 2
    };

    QwtLegend(QWidget *parent = 0, const char *name = 0);
    
    /*!
      Set the legend display policy to:
      - LegendDisplayPolicy::None\n
      the client code is responsible how to display of each legend item.
      The Qwt library will not interfere.

      - LegendDisplayPolicy::Fixed\n
      all legend items are displayed with the QwtLegendButton::IdentifierMode
      to be passed in 'mode'.

      - LegendDisplayPolicy::Auto\n
      each legend item is displayed with a mode that is a bitwise or of
      QwtLegendButton::ShowLine (if its curve is drawn with a line) and
      QwtLegendButton::ShowSymbol (if its curve is drawn with symbols) and
      QwtLegendButton::ShowText (if the has a title).
    */
    void setDisplayPolicy(int policy, int mode = -1);

    /*! Return the legend display policy.
        Default is LegendDisplayPolicy::Auto.
     */
    int displayPolicy() const { return d_displayPolicy; }

    /*! Return the IdentifierMode to be used in combination with
        LegendDisplayPolicy::Fixed.
    Default is ShowLine | ShowSymbol | ShowText.
     */
    int identifierMode() const { return d_identifierMode; }

    /*! The contents widget is the only child of the viewport() and
        the parent widget of all legend items.
     */
    QWidget *contentsWidget() { return d_contentsWidget; }

    /*! The contents widget is the only child of the viewport() and
        the parent widget of all legend items.
     */
    const QWidget *contentsWidget() const { return d_contentsWidget; }

    void insertItem(QWidget *, long key);

    //! Find the item for a given key.
    QWidget *findItem(long key) { return d_items.find(key); }

    //! Find the item for a given key.
    const QWidget *findItem(long key) const { return d_items.find(key); }

    //! Find the item for a given key and remove it from the item list.
    QWidget *takeItem(long key) { return d_items.take(key); }

    virtual QIntDictIterator<QWidget> itemIterator() const;

    long key(const QWidget *) const;

    void clear();
    
    //! \return TRUE, if there are no legend items.
    bool isEmpty() const { return d_items.count() == 0; }

    //! \return Number of legend items.
    uint itemCount() const { return d_items.count(); }

    virtual bool eventFilter(QObject *, QEvent *);

    virtual QSize sizeHint() const;
    virtual int heightForWidth(int w) const;

protected:
    virtual void viewportResizeEvent(QResizeEvent *);
    virtual void layoutContents();

private:
    int d_displayPolicy;
    int d_identifierMode;
    QIntDict<QWidget> d_items;
    QWidget *d_contentsWidget;
};

#endif // QWT_LEGEND_H
