/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#ifndef QWT_PAINTER_H
#define QWT_PAINTER_H

#include <qpoint.h>
#include <qpointarray.h>
#include <qrect.h>
#include "qwt_global.h"

class QPainter;
class QPaintDevice;
class QWMatrix;

#if defined(Q_WS_X11)
// Warning: Don't care about wrong QCOORD_MIN, QCOORD_MAX on X11.
#define QWT_COORD_MAX 16384
#define QWT_COORD_MIN (-QWT_COORD_MAX - 1)
#else
#define QWT_COORD_MAX QCOORD_MAX
#define QWT_COORD_MIN QCOORD_MIN
#endif

/*!
  \brief A collection of QPainter workarounds

  1) Clipping to coordinate system limits

  On X11 pixel coordinates are stored in shorts. Qt 
  produces overruns when mapping QCOORDS to shorts. 

  2) Scaling to device metrics

  QPainter scales fonts, line and fill patterns to the metrics
  of the paint device. Other values like the geometries of rects, points
  remain device independend. To enable a device independent widget 
  implementation, QwtPainter adds scaling of these geometries.
  (Unfortunately QPainter::scale scales both types of paintings,
   so the objects of the first type would be scaled twice).

  3) Hide some Qt2/3 incompatibilities
*/

class QWT_EXPORT QwtPainter
{
public:
    static void setScaleMetrics(const QPaintDevice *, const QPaintDevice *);
    static void resetScaleMetrics();
    static double scaleMetricsX();
    static double scaleMetricsY();

    static void setDeviceClipping(bool);
    static bool deviceClipping();

    static QRect invScale(const QRect &, const QPainter * = NULL);
    static QPoint invScale(const QPoint &, const QPainter * = NULL);
    static QRect scale(const QRect &, const QPainter * = NULL);
    static QPoint scale(const QPoint &, const QPainter * = NULL);
    static QPointArray scale(const QPointArray &, const QPainter * = NULL);
    
    static void setClipRect(QPainter *, const QRect &);

    static void drawText(QPainter *, int x, int y, 
        const QString &, int len = -1);
    static void drawText(QPainter *, const QPoint &, 
        const QString &, int len = -1);
    static void drawText(QPainter *, int x, int y, int w, int h, 
        int flags, const QString &, int len = -1);
    static void drawText(QPainter *, const QRect &, 
        int flags, const QString &, int len = -1);

    static void drawRect(QPainter *, int x, int y, int w, int h);
    static void drawRect(QPainter *, const QRect &rect);
    static void fillRect(QPainter *, const QRect &, const QBrush &); 

    static void drawEllipse(QPainter *, const QRect &);

    static void drawLine(QPainter *, int x1, int y1, int x2, int y2);
    static void drawLine(QPainter *, const QPoint &p1, const QPoint &p2);
    static void drawPolygon(QPainter *, const QPointArray &pa);
    static void drawPolyline(QPainter *, const QPointArray &pa);
    static void drawPoint(QPainter *, int x, int y);

	static QRect map(const QWMatrix &, const QRect &);
	static QPointArray map(const QWMatrix &, const QPointArray &);

	static void drawRoundFrame(QPainter *, const QRect &,
    	int width, bool sunken);

private:
	static const QRect &deviceClipRect();

    static bool insideEdge( const QPoint &p, const QRect &r, int edge );
    static QPoint intersectEdge(const QPoint &p1, const QPoint &p2,
        const QRect &r, int edge );
    static QPointArray clipPolyline(const QRect &r,
        const QPointArray &pa, int &index, int &npoints );
    static QPointArray clip(const QPointArray &);

    static bool d_deviceClipping;
    static double d_scaleMetricsX;
    static double d_scaleMetricsY;
};

#endif
