/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <iostream>
#include <math.h>
#include <qpainter.h>
#include <qpixmap.h>
#include "qwt_math.h"
#include "qwt_paint_buffer.h"
#include "qwt_painter.h"
#include "qwt_dial_needle.h"
#include "qwt_dial.h"

QwtDialScaleDraw::QwtDialScaleDraw(QwtDial *parent):
	d_parent(parent)
{
}

QString QwtDialScaleDraw::label(double value) const
{
	if ( d_parent == NULL )
		return QwtScaleDraw::label(value);

	return d_parent->scaleLabel(value);
}

/*!
  \brief Constructor
  \param parent Parent widget
  \param name Widget name

  Create a dial with no scale and no needle. 
  The default origin is 90.0 with no valid value. It accepts
  mouse and keyboard inputs and has no step size. The default mode
  is QwtDial::RotateNeedle.
*/  

QwtDial::QwtDial(QWidget* parent, const char* name):
    	QwtSliderBase(Qt::Horizontal, parent, name, 
			WRepaintNoErase|WResizeNoErase),
	d_visibleBackground(FALSE),
	d_frameShadow(Sunken),
	d_lineWidth(0),
    d_mode(RotateNeedle),
    d_wrapping(FALSE),
	d_origin(90.0),
	d_minScaleArc(0.0),
	d_maxScaleArc(0.0),
	d_scaleDraw(0),
    d_maxMajIntv(36),
    d_maxMinIntv(10),
    d_scaleStep(0.0),
	d_needle(0)
{
	setFocusPolicy(QWidget::TabFocus);

    QPalette p = palette();
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
		const QPalette::ColorGroup cg = (QPalette::ColorGroup)i;

		// Base: background color of the circle inside the frame.
		// Foreground: background color of the circle inside the scale

        p.setColor(cg, QColorGroup::Foreground, 
			p.color(cg, QColorGroup::Base));
    }
    setPalette(p);

	d_scaleDraw = new QwtDialScaleDraw(this);
    d_scaleDraw->setGeometry(0, 0, 0, QwtScaleDraw::Round);

	setScaleArc(0.0, 360.0); // scale as a full circle
	setRange(0.0, 360.0, 1.0); // degrees as deafult
}

//!  Destructor
QwtDial::~QwtDial() 
{
	delete d_scaleDraw;
	delete d_needle;
}

/*!
  Show/Hide the area outside of the compass frame
  \param show Show if TRUE, hide if FALSE

  \sa QwtDial::hasVisibleBackground, QWidget::setMask
  \warning When QwtDial is a toplevel widget the window
           border might disappear too.
*/

void QwtDial::showBackground(bool show)
{
	if ( d_visibleBackground != show )
	{
		d_visibleBackground = show;

		if ( d_visibleBackground )
			clearMask();
		else
		{
        	const QRect r = boundingRect();

        	QRegion region(QRect(r.x() - 1, r.y() - 1,
            	r.width() + 2, r.height() + 2), QRegion::Ellipse);

        	setMask(region);
		}
		update();
	}
}

/*!
  TRUE when the area outside of the compass frame is visible

  \sa QwtDial::showBackground, QWidget::setMask
*/

bool QwtDial::hasVisibleBackground() const 
{ 
	return d_visibleBackground; 
}

/*!
  Sets the frame shadow value from the frame style.
  \param shadow Frame shadow
  \sa QwtDial::setLineWidth, QFrame::setFrameShadow
*/

void QwtDial::setFrameShadow(Shadow shadow)
{
	if ( shadow != d_frameShadow )
	{
		d_frameShadow = shadow;
		if ( lineWidth() > 0 )
			update();
	}
}

/*!
  \return Frame shadow
  /sa QwtDial::setFrameShadow, QwtDial::lineWidth, 
      QFrame::frameShadow
*/

QwtDial::Shadow QwtDial::frameShadow() const 
{ 
	return d_frameShadow; 
}

/*!
  Sets the line width
  \param isVisible Show if TRUE, hide if FALSE
  \sa QwtDial::setFrameShadow
*/

void QwtDial::setLineWidth(int lineWidth)
{
	if ( lineWidth < 0 )
		lineWidth = 0;

	if ( d_lineWidth != lineWidth )
	{
		d_lineWidth = lineWidth;
		update();
	}
}

/*!
  \return Line width of the frame
  \sa QwtDial::setLineWidth, QwtDial::frameShadow, 
      QFrame::lineWidth
*/
int QwtDial::lineWidth() const 
{ 
	return d_lineWidth; 
}

/*!
  \return bounding rect of the compass circle without the frame
  \sa QwtDial::setLineWidth, QwtDial::scaleContentsRect, QwtDial::boundingRect
*/

QRect QwtDial::contentsRect() const
{
	const int lw = lineWidth();

	QRect r = boundingRect();
	if ( lw > 0 )
	{
		r.setRect(r.x() + lw, r.y() + lw, 
			r.width() - 2 * lw, r.height() - 2 * lw);
	}
	return r;
}

/*!
  \return bounding rect of the compass circle including the frame
  \sa QwtDial::setLineWidth, QwtDial::scaleContentsRect, QwtDial::contentsRect
*/

QRect QwtDial::boundingRect() const
{
	const int radius = QMIN(width(), height()) / 2;
	const QPoint center = rect().center();

	int x = center.x() + width() % 2 - radius;
	if ( width() % 2 == 0 )
		x++;

	int y = center.y() - radius;
	if ( height() % 2 == 0 )
		y++;

	return QRect(x, y, 2 * radius, 2 * radius);
}

/*!
  \return rect inside the scale
  \sa setLineWidth, QwtDial::boundingRect, QwtDial::contentsRect
*/

QRect QwtDial::scaleContentsRect() const
{
    const QPen scalePen(colorGroup().text(), 0, NoPen);

	int scaleDist = 0;
	if ( d_scaleDraw )
	{
		scaleDist = QMAX(
			d_scaleDraw->maxWidth(scalePen, fontMetrics()),
        	d_scaleDraw->maxHeight(scalePen, fontMetrics()));
	}

	const QRect rect = contentsRect();

	return QRect(rect.x() + scaleDist, rect.y() + scaleDist,
		rect.width() - 2 * scaleDist, rect.height() - 2 * scaleDist);
}

/*!
  \brief Change the mode of the meter.
  \param mode New mode 
    
  The value of the meter is indicated by the difference
  between north of the scale and the direction of the needle.
  In case of QwtCompass::RotateNeedle north is pointing
  to the origin() and the needle is rotating, in case of
  QwtCompass::RotateScale, the needle points to origin()
  and the scale is rotating.
    
  \sa QwtCompass::mode, QwtDial::setValue, QwtDial::setOrigin
*/  
    
void QwtDial::setMode(Mode mode)
{   
    if ( mode != d_mode )
    {
        d_mode = mode;
        update(); 
    }
}       

/*! 
  \return mode of the compass.
    
  The value of the compass is indicated by the difference
  between north of the scale and the direction of the needle.
  In case of QwtDial::RotateNeedle north is pointing
  to the origin() and the needle is rotating, in case of
  QwtDial::RotateScale, the needle points to origin()
  and the scale is rotating.
 
  \sa QwtCompass::setMode, QwtDial::origin, QwtDial::value
*/

QwtDial::Mode QwtDial::mode() const
{
    return d_mode;
}

void QwtDial::setWrapping(bool wrapping)
{
	d_wrapping = wrapping;
} 

bool QwtDial::wrapping() const
{
	return d_wrapping;
}

//! Resize the compass
void QwtDial::resizeEvent(QResizeEvent *e)
{
	QWidget::resizeEvent(e);

	if ( !hasVisibleBackground() )
	{
		const QRect r = boundingRect();
	
		QRegion region(QRect(r.x() - 1, r.y() - 1, 
			r.width() + 2, r.height() + 2), QRegion::Ellipse);

		setMask(region);
	}
}

//! Repaint the compass
void QwtDial::paintEvent(QPaintEvent *e)
{
    const QRect &ur = e->rect();
    if ( ur.isValid() )
    {
        QwtPaintBuffer paintBuffer(this, ur);

		QPainter *painter = paintBuffer.painter();

		drawFrame(painter);
		drawContents(painter);

#if 0
	    if ( hasFocus() )
		{
			QRect focusRect = contentsRect();

			const int margin = 2;
			focusRect.setRect(focusRect.x() + margin, focusRect.y() + margin,
				focusRect.width() - 2 * margin, focusRect.height() - margin);

			painter->setBrush(Qt::NoBrush);
			painter->setPen(QPen(colorGroup().mid(), 0, Qt::DotLine));
			painter->drawEllipse(focusRect);
		}
#endif
    }
}

/*!
  Draw the compass frame
  \param painter Painter
  \sa QwtDial::lineWidth, QwtDial::frameShadow
*/

void QwtDial::drawFrame(QPainter *painter)
{
	const int lw = lineWidth();

	QRect r = boundingRect();
	r.setRect(r.x() + lw / 2, r.y() + lw / 2,
		r.width() - lw, r.height() - lw);

	if ( lw > 0 )
	{
		switch(d_frameShadow)
		{
			case QFrame::Raised:
				QwtPainter::drawRoundFrame(painter, r, lw, FALSE);
				break;
			case QFrame::Sunken:
				QwtPainter::drawRoundFrame(painter, r, lw, TRUE);
				break;
			default: // Plain
			{
				painter->setPen(QPen(Qt::black, lw));
				painter->drawEllipse(r);
			}
		}
	}
}

/*!
  \brief Draw the contents inside of the frame
 
  QColorGroup::Backgound is the background color outside of the frame.
  QColorGroup::Base is the background color inside the frame.
  QColorGroup::Foreground is the background color inside the scale.

  \param painter Painter
  \sa QwtDial::boundingRect, QwtDial::contentsRect,
	QwtDial::scaleContentsRect, QWidget::setPalette
*/

void QwtDial::drawContents(QPainter *painter) const
{
	if ( colorGroup().base() != colorGroup().background() )
	{
		painter->setPen(NoPen);
		painter->setBrush(colorGroup().base());
		painter->drawEllipse(contentsRect());
	}

    QPalette::ColorGroup cg;
    if ( isEnabled() )
        cg = hasFocus() ? QPalette::Active : QPalette::Inactive;
    else
        cg = QPalette::Disabled;

	const QRect insideScaleRect = scaleContentsRect();
	if ( colorGroup().base() != colorGroup().background() )
	{
		painter->setPen(NoPen);
		painter->setBrush(colorGroup().base());
		painter->drawEllipse(insideScaleRect);
	}

	const QPoint center = insideScaleRect.center();
	const int radius = insideScaleRect.width() / 2;

#if 0
	const int margin = 4;
#endif
	
    if ( colorGroup().foreground() != colorGroup().base() )
    {
        painter->setPen(NoPen);
        painter->setBrush(colorGroup().foreground());
        painter->drawEllipse(insideScaleRect);
    }

	painter->save();
	drawScaleContents(painter, center, radius);
	painter->restore();

	double direction = d_origin;

	if (isValid())
	{
		direction = d_origin + d_minScaleArc;
		if ( maxValue() > minValue() && d_maxScaleArc > d_minScaleArc )
		{
			const double ratio = 
				(value() - minValue()) / (maxValue() - minValue());
			direction += ratio * (d_maxScaleArc - d_minScaleArc);
		}

		if ( direction >= 360.0 )
			direction -= 360.0;
	}

	double origin = d_origin;
	if ( mode() == RotateScale )
	{
		origin -= direction - d_origin;
		direction = d_origin;
	}

	painter->save();
	drawScale(painter, center, radius, origin, d_minScaleArc, d_maxScaleArc);
	painter->restore();

	if ( isValid() )
	{
		painter->save();
		drawNeedle(painter, center, radius, direction, cg);
		painter->restore();
	}
}

/*!
  Draw the needle
  \param painter Painter
  \param center Center of the dial
  \param radius Length for the needle
  \param direction Direction of the needle in degrees, counter clockwise
  \param cg ColorGroup
*/

void QwtDial::drawNeedle(QPainter *painter, const QPoint &center, 
	int radius, double direction, QPalette::ColorGroup cg) const
{
	if ( d_needle )
	{
		direction = 360.0 - direction; // counter clockwise
		d_needle->draw(painter, center, radius, direction, cg);
	}
}

void QwtDial::drawScale(QPainter *painter, const QPoint &center,
	int radius, double origin, double minArc, double maxArc) const
{
	if ( d_scaleDraw == NULL )
		return;

	origin -= 270.0; // hardcoded origin of QwtScaleDraw

	double angle = maxArc - minArc;
    if ( angle > 360.0 )
        angle = fmod(angle, 360.0);

	minArc += origin;
	if ( minArc < -360.0 )
		minArc = fmod(minArc, 360.0);
	
	maxArc = minArc + angle;
	if ( maxArc > 360.0 )
	{
		// QwtScaleDraw::setAngleRange accepts only values
		// in the range [-360.0..360.0]
		minArc -= 360.0;
		maxArc -= 360.0;
	}
	
	d_scaleDraw->setAngleRange(minArc, maxArc);
    d_scaleDraw->setGeometry(center.x() - radius, center.y() - radius,
        2 * radius, QwtScaleDraw::Round);

	painter->setFont(font());
	painter->setPen(colorGroup().text());
    d_scaleDraw->draw(painter);
}

/*!
  Draw the contents inside the scale

  \param painter Painter
  \param center Center of the contents circle
  \param radius Radius of the contents circle
*/

void QwtDial::drawScaleContents(QPainter *, 
	const QPoint &, int) const
{
	// empty default implementation
}

/*!
  Set a needle for the dial
  \param needle Needle
  \warning The needle will be deleted, when a different needle is
    set or in ~QwtDial
*/

void QwtDial::setNeedle(QwtDialNeedle *needle)
{
	if ( needle != d_needle )
	{
		if ( d_needle )
			delete d_needle;

		d_needle = needle;
		update();
	}
}

/*! 
  \return needle
  \sa QwtDial::setNeedle
*/

const QwtDialNeedle *QwtDial::needle() const 
{ 
	return d_needle; 
}

/*! 
  \return needle
  \sa QwtDial::setNeedle
*/

QwtDialNeedle *QwtDial::needle() 
{ 
	return d_needle; 
}

void QwtDial::rangeChange()
{
	updateScale();
}

void QwtDial::updateScale()
{
	if ( d_scaleDraw )
	{
		d_scaleDraw->setScale(minValue(), maxValue(),
			d_maxMajIntv, d_maxMinIntv, d_scaleStep);
	}
}

void QwtDial::setScaleDraw(QwtDialScaleDraw *scaleDraw)
{
    if ( scaleDraw != d_scaleDraw )
    {
        if ( d_scaleDraw )
            delete d_scaleDraw;
    
        d_scaleDraw = scaleDraw;
		updateScale();
        update();
    }
}

void QwtDial::setScale(int maxMajIntv, int maxMinIntv, double step)
{
	d_maxMajIntv = maxMajIntv;
	d_maxMinIntv = maxMinIntv;
	d_scaleStep = step;

	updateScale();
}

QString QwtDial::scaleLabel(double value) const
{
#if 1
    if ( value == -0 )
        value = 0;
#endif

    QString text;
    text.sprintf("%g", value);

    return text;
}

void QwtDial::setOrigin(double origin)
{
	d_origin = origin;
	update();
}

double QwtDial::origin() const
{
	return d_origin;
}

void QwtDial::setScaleArc(double minArc, double maxArc)
{
	// ueberdenken

	if ( minArc != 360.0 && minArc != -360.0 )
		minArc = fmod(minArc, 360.0);
	if ( maxArc != 360.0 && maxArc != -360.0 )
		maxArc = fmod(maxArc, 360.0);

	d_minScaleArc = QMIN(minArc, maxArc);
	d_maxScaleArc = QMAX(minArc, maxArc);
	if ( d_maxScaleArc - d_minScaleArc > 360.0 )
		d_maxScaleArc = d_minScaleArc + 360.0;
	
	update();
}

void QwtDial::valueChange()
{
	update();
	QwtSliderBase::valueChange();
}

/*!
  \return QwtDial::sizeHint()
*/
QSize QwtDial::sizeHint() const
{
	int sh = 0;
	if ( d_scaleDraw )
    	sh = d_scaleDraw->minHeight( QPen(), fontMetrics() );

    const int d = 6 * sh + 2 * lineWidth();
    
    return QSize( d, d );
}

/*! 
  \brief Return a minimum size hint
  \warning The return value of QwtDial::minimumSizeHint() depends on the
           font and the scale.
*/  
QSize QwtDial::minimumSizeHint() const
{   
	int sh = 0;
	if ( d_scaleDraw )
    	sh = d_scaleDraw->minHeight( QPen(), fontMetrics() );

    const int d = 3 * sh + 2 * lineWidth();
    
    return QSize( d, d );
}

static double line2Radians(const QPoint &p1, const QPoint &p2)
{
    const QPoint p = p2 - p1;

    double angle;
    if ( p.x() == 0 )
        angle = ( p.y() <= 0 ) ? M_PI_2 : 3 * M_PI_2;
    else
    {
        angle = atan(double(-p.y()) / double(p.x()));
        if ( p.x() < 0 )
            angle += M_PI;
        if ( angle < 0.0 )
            angle += 2 * M_PI;
    }
    return 360.0 - angle * 180.0 / M_PI;
}

double QwtDial::getValue(const QPoint &p)
{
	if ( d_maxScaleArc == d_minScaleArc || maxValue() == minValue() )
		return minValue();

	double dir = line2Radians(rect().center(), p) - d_origin;
	if ( dir < 0.0 )
		dir += 360.0;

	double value;

    if ( mode() == RotateScale )
	{
		value = 360.0 - dir;
	}
	else
	{
		const double ratio = 
			(dir - d_minScaleArc) / (d_maxScaleArc - d_minScaleArc);
		value = minValue() + ratio * (maxValue() - minValue());
	}

	if ( d_scrollMode == ScrMouse )
	{
		if ( value - d_mouseOffset < 0.0 )
			value += 360.0;
		if ( value - d_mouseOffset >= 360.0 )
			value -= 360.0;
	}

	return value;
}

void QwtDial::getScrollMode(const QPoint &p, int &scrollMode, int &direction)
{
	direction = 0;

    const QRegion region(contentsRect(), QRegion::Ellipse);
    if ( !region.contains(p) || p == rect().center() )
		scrollMode = ScrNone;
	else
		scrollMode = ScrMouse;
}

