/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <math.h>
#include <qpainter.h>
#include "qwt_math.h"
#include "qwt_painter.h"
#include "qwt_dial_needle.h"

QwtDialNeedle::QwtDialNeedle()
{
    const QColor dark(16,16,16);
    const QColor light(64,64,64);

	QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Dark, dark);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Light, light);
    }
	setPalette(palette);
}

QwtDialNeedle::~QwtDialNeedle() 
{
}

void QwtDialNeedle::setPalette(const QPalette &p) 
{ 
	d_palette = p; 
}

const QPalette &QwtDialNeedle::palette() const 
{ 
	return d_palette; 
}

const QColorGroup &QwtDialNeedle::colorGroup(QPalette::ColorGroup cg) const
{
	switch(cg)
	{
		case QPalette::Disabled:
			return d_palette.disabled();
		case QPalette::Inactive:
			return d_palette.inactive();
		default:
			return d_palette.active();
	}
}

QwtCompassNeedle3::QwtCompassNeedle3(const QColor &base, const QColor &mid)
{
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Dark, base);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Mid, mid);
    }

    setPalette(palette);
}

void QwtCompassNeedle3::draw(QPainter *painter, const QPoint &center,
	int length, double direction, QPalette::ColorGroup cg) const
{
	drawNeedle(painter, center, length, direction, 
		colorGroup(cg).dark(), colorGroup(cg).mid());
}

void QwtCompassNeedle3::drawNeedle(QPainter *painter, const QPoint &center,
	int length, double direction, const QColor &c,
	const QColor &arrowColor) 
{
	direction *= M_PI / 180.0;

	painter->save();
	painter->setPen(Qt::NoPen);

	int width = (int)QMAX(length * 0.03, 9);
	if ( width % 2 == 0 )
		width++;

	const double peak = width;

	const QPoint p1 = qwtPolar2Pos(center, 0.2 * length, M_PI + direction);
	const QPoint p2 = qwtPolar2Pos(center, length - peak, direction);
	const QPoint p3 = qwtPolar2Pos(center, length, direction);

	QPointArray pa(7);
	pa.setPoint(0, qwtPolar2Pos(p1, width / 2, direction + M_PI_2));
	pa.setPoint(1, qwtPolar2Pos(p1, width / 2, direction - M_PI_2));
	pa.setPoint(2, qwtPolar2Pos(p2, width / 2, direction - M_PI_2));
	pa.setPoint(3, qwtPolar2Pos(p2, width, direction - M_PI_2));
	pa.setPoint(4, p3);
	pa.setPoint(5, qwtPolar2Pos(p2, width, direction + M_PI_2));
	pa.setPoint(6, qwtPolar2Pos(p2, width / 2, direction + M_PI_2));

	painter->setBrush(arrowColor);
	painter->drawPolygon(pa);

	QRect r(0, 0, 2 * width, 2 * width);
	r.moveCenter(center);

	painter->setBrush(c);
	painter->drawEllipse(r);

	painter->restore();
}

QwtCompassNeedle4::QwtCompassNeedle4(
	const QColor &light, const QColor &dark, const QColor &mid)
{   
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Light, light);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Dark, dark);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Mid, mid);
    }

    setPalette(palette); 
}


void QwtCompassNeedle4::draw(QPainter *painter, const QPoint &center,
    int length, double direction, QPalette::ColorGroup cg) const
{
    drawNeedle(painter, center, length, direction, 
		colorGroup(cg).light(), colorGroup(cg).dark(), colorGroup(cg).mid());
}

void QwtCompassNeedle4::drawNeedle(QPainter *painter, const QPoint &center, 
    int length, double direction, 
    const QColor &light, const QColor &dark, const QColor &mid)
{
	painter->save();

    int colorOffset =  10;
    if ( direction < 135.0 || direction > 135.0 + 180.0 )
        colorOffset = -colorOffset;

	const int width = QMAX(qRound(length / 6.0), 3);

	drawPointer(painter, dark, colorOffset, 
		center, length, width, direction);
	drawPointer(painter, light, -colorOffset, 
		center, length, width, direction + 180.0);
	
	QRect knobRect(0, 0, width, width);
	knobRect.moveCenter(center);

	painter->setBrush(QBrush(mid));
	painter->drawEllipse(knobRect);

	QwtPainter::drawRoundFrame(painter, knobRect, 3, FALSE);

	painter->restore();
}

void QwtCompassNeedle4::drawPointer(QPainter *painter, const QColor &color,
	int colorOffset, const QPoint &center, int length, 
	int width, double direction)
{
	painter->save();

	const int peak = QMAX(qRound(length / 10.0), 5);

    const int knobWidth = width + 8;
	QRect knobRect(0, 0, knobWidth, knobWidth);
	knobRect.moveCenter(center);

    QPointArray pa(5);

    pa.setPoint(0, qwtDegree2Pos(center, width / 2, direction + 90.0));
    pa.setPoint(1, center);
    pa.setPoint(2, qwtDegree2Pos(pa.point(1), length - peak, direction));
    pa.setPoint(3, qwtDegree2Pos(center, length, direction));
    pa.setPoint(4, qwtDegree2Pos(pa.point(0), length - peak, direction));

	painter->setPen(Qt::NoPen);
    painter->setBrush(color.dark(100 + colorOffset));
    painter->drawPolygon(pa);
	painter->drawPie(knobRect, qRound(direction * 16), 90 * 16);

    pa.setPoint(0, qwtDegree2Pos(center, width / 2, direction - 90.0));
    pa.setPoint(4, qwtDegree2Pos(pa.point(0), length - peak, direction));

    painter->setBrush(color.dark(100 - colorOffset));
    painter->drawPolygon(pa);
	painter->drawPie(knobRect, qRound(direction * 16), -90 * 16);

	painter->restore();
}

QwtCompassMagnetNeedle::QwtCompassMagnetNeedle(
	const QColor &light, const QColor &dark)
{   
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Light, light);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Dark, dark);
    }

    setPalette(palette); 
}


void QwtCompassMagnetNeedle::draw(QPainter *painter, const QPoint &center,
    int length, double direction, QPalette::ColorGroup cg) const
{
    drawNeedle(painter, center, length, direction, 
		colorGroup(cg).light(), colorGroup(cg).dark());
}

void QwtCompassMagnetNeedle::drawNeedle(QPainter *painter, 
	const QPoint &center, int length, double direction, 
    const QColor &light, const QColor &dark)
{
	const int width = qRound(length / 3.0);

	int colorOffset =  10;
	if ( direction < 135.0 || direction > 135.0 + 180.0 )
		colorOffset = -colorOffset;

	painter->save();

	QPointArray pa(3);
    pa.setPoint(0, center);
    pa.setPoint(1, qwtDegree2Pos(center, length, direction));

    pa.setPoint(2, qwtDegree2Pos(center, width / 2, direction + 90.0));
    painter->setBrush(dark.dark(100 + colorOffset));
    painter->drawPolygon(pa);

    pa.setPoint(2, qwtDegree2Pos(center, width / 2, direction - 90.0));
    painter->setBrush(dark.dark(100 - colorOffset));
    painter->drawPolygon(pa);

	// --

    pa.setPoint(1, qwtDegree2Pos(center, length, direction + 180.0));

    pa.setPoint(2, qwtDegree2Pos(center, width / 2, direction + 90.0));
    painter->setBrush(light.dark(100 + colorOffset));
    painter->drawPolygon(pa);

    pa.setPoint(2, qwtDegree2Pos(center, width / 2, direction - 90.0));
    painter->setBrush(light.dark(100 - colorOffset));
    painter->drawPolygon(pa);

	painter->restore();
}


QwtCompassNeedle1::QwtCompassNeedle1(const QColor &mid)
{
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Mid, mid);
    }

    setPalette(palette);
}

void QwtCompassNeedle1::draw(QPainter *painter, const QPoint &center,
	int length, double direction, QPalette::ColorGroup cg) const
{
	drawNeedle(painter, center, length, direction, colorGroup(cg).mid());
}

void QwtCompassNeedle1::drawNeedle(QPainter *painter, const QPoint &center,
	int length, double direction, const QColor &color) 
{
	const double AR1[] = {0, 0.4, 0.3, 1, 0.8, 1, 0.3, 0.4};
	const double AW1[] = {0, -45, -20, -15, 0, 15, 20, 45};

	QPointArray pa(8);
	pa.setPoint(0, center);
	for (int i=1; i<8; i++) 
	{
		const QPoint p = qwtDegree2Pos(center, 
			AR1[i] * length, direction + AW1[i]);
		pa.setPoint(i, p);
	}

	painter->save();
	painter->setPen(Qt::NoPen);
	painter->setBrush(color);
	painter->drawPolygon(pa);
	painter->restore();
}

QwtCompassNeedle2::QwtCompassNeedle2(const QColor &light, const QColor &mid)
{
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Light, light);
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Mid, mid);
    }

    setPalette(palette);
}

void QwtCompassNeedle2::draw(QPainter *painter, const QPoint &center,
	int length, double direction, QPalette::ColorGroup cg) const
{
	drawNeedle(painter, center, length, direction, 
		colorGroup(cg).light(), colorGroup(cg).mid());
}

void QwtCompassNeedle2::drawNeedle(QPainter *painter, const QPoint &center,
	int length, double direction, const QColor &light, const QColor &mid)
{
	painter->save();
	painter->setPen(Qt::NoPen);

	const double angle = 12.0;
	const double ratio = 0.7;

	QPointArray pa(3);

	pa.setPoint(0, center);
	pa.setPoint(2, qwtDegree2Pos(center, ratio * length, direction));

	pa.setPoint(1, qwtDegree2Pos(center, length, direction + angle));
	painter->setBrush(mid);
	painter->drawPolygon(pa);

	pa.setPoint(1, qwtDegree2Pos(center, length, direction - angle));
	painter->setBrush(light);
	painter->drawPolygon(pa);

	painter->restore();
}

QwtCompassLineNeedle::QwtCompassLineNeedle(const QColor &foreground)
{
    QPalette palette;
    for ( int i = 0; i < QPalette::NColorGroups; i++ )
    {
        palette.setColor((QPalette::ColorGroup)i,
            QColorGroup::Foreground, foreground);
    }

    setPalette(palette);
}

void QwtCompassLineNeedle::draw(QPainter *painter, const QPoint &center,
	int length, double direction, QPalette::ColorGroup cg) const
{
	drawNeedle(painter, center, length, direction, colorGroup(cg).foreground());
}

void QwtCompassLineNeedle::drawNeedle(QPainter *painter, const QPoint &center,
	int length, double direction, const QColor &color)
{
	painter->save();
	painter->setPen(QPen(color, 3));
	painter->drawLine(center, qwtDegree2Pos(center, length, direction));
	painter->restore();
}

