/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

// vim: expandtab

#include <qapplication.h> 
#include <qpainter.h>
#include <qpicture.h>
#include <qbitmap.h>
#include <qstyle.h>
#include "qwt_legend.h"
#include "qwt_painter.h"
#include "qwt_dyngrid_layout.h"

static const int IdentifierWidth = 8;
/*!
  \param symbol Curve symbol
  \param pen Curve pen
  \param text Button text
  \param parent Parent widget
  \param name Widget name
*/
QwtLegendButton::QwtLegendButton(
        const QwtSymbol &symbol, const QPen &pen, const QString &text,
        QWidget *parent, const char *name): 
    QPushButton(text, parent, name),
    d_identifierMode(QwtLegendButton::ShowLine
             | QwtLegendButton::ShowText), 
    d_symbol(symbol), 
    d_curvePen(pen)
{
    setFlat(TRUE);
    updateIconset();
}

/*!
  \param parent Parent widget
  \param name Widget name
*/
QwtLegendButton::QwtLegendButton(QWidget *parent, const char *name): 
    QPushButton(parent, name),
    d_identifierMode(QwtLegendButton::ShowLine
             | QwtLegendButton::ShowText), 
    d_curvePen(Qt::NoPen)
{
    setFlat(TRUE);
    updateIconset();
}

/*! 
  Set identifier mode.
  Default is QwtLegendButton::ShowLine | QwtLegendButton::ShowText.
  \param mode Or'd values of IdentifierMode

  \sa QwtLegendButton::identifierMode()
*/
void QwtLegendButton::setIdentifierMode(int mode) 
{
    if ( mode != d_identifierMode )
    {
        d_identifierMode = mode;
        updateIconset();
    }
}

/*! 
  Set curve symbol.
  \param symbol Symbol

  \sa QwtLegendButton::symbol()
*/
void QwtLegendButton::setSymbol(const QwtSymbol &symbol) 
{
    if ( symbol != d_symbol )
    {
        d_symbol = symbol;
        updateIconset();
    }
}

/*! 
  Set curve pen.
  \param pen Curve pen

  \sa QwtLegendButton::curvePen()
*/
void QwtLegendButton::setCurvePen(const QPen &pen) 
{
    if ( pen != d_curvePen )
    {
        d_curvePen = pen;
        updateIconset();
    }
}

/*! 
  Update the iconset according to the current identifier properties
*/
void QwtLegendButton::updateIconset()
{
    const QFontMetrics fm(font());

    QPixmap pm(IdentifierWidth, fm.height());
    pm.fill(this, 0, 0);

    QPainter p(&pm);
    drawIdentifier(&p, QRect(0, 0, pm.width(), pm.height()) );
    p.end();

    pm.setMask(pm.createHeuristicMask());

    setIconSet(QIconSet(pm));
}

/*! 
  Paint the identifier to a given rect.
  \param painter Painter
  \param rect Rect where to paint
*/
void QwtLegendButton::drawIdentifier(
    QPainter *painter, const QRect &rect) const
{
    if ( rect.isEmpty() )
        return;

    if ( (d_identifierMode & ShowLine ) && (d_curvePen.style() != Qt::NoPen) )
    {
        QRect scaledRect = QwtPainter::scale(rect, painter);

        painter->save();
        painter->setPen(d_curvePen);
        painter->drawLine(scaledRect.left(), scaledRect.center().y(), 
            scaledRect.right(), scaledRect.center().y());
        painter->restore();
    }

    if ( (d_identifierMode & ShowSymbol) 
        && (d_symbol.style() != QwtSymbol::None) )
    {
        QRect symbolRect;
        symbolRect.setSize(d_symbol.size().boundedTo(rect.size()));
        symbolRect.moveCenter(rect.center());

        painter->save();
        d_symbol.draw(painter, symbolRect);
        painter->restore();
    }
}

/*!
  Draw the legend button contents.
  \param painter Painter
  \param rect Rect where to paint the button
  \sa drawButtonLabel
  \note drawContents is intended for printing
*/

void QwtLegendButton::drawContents(QPainter *painter, 
    const QRect &rect) const
{
    const QFontMetrics fm(font());

    QPixmap pm(IdentifierWidth, fm.height());

    QRect identifierRect(rect.x() + 2, rect.y(), 
        IdentifierWidth, fm.height());
    identifierRect.moveCenter(QPoint(identifierRect.center().x(), 
        rect.center().y()));

    drawIdentifier(painter, identifierRect);

    QRect textRect = rect;
    textRect.setX(identifierRect.right() + 4);
    QwtPainter::drawText(painter, textRect, 
        AlignLeft | AlignVCenter, text());
}

class PaintFilter: public QPicture
{
    // Filter drawButtonLabel paint commands, and change
    // them to align the text 4 pixels left to the icon.

public:
    PaintFilter():
        d_textStart(0) 
    {
    }

    virtual bool cmd(int c, QPainter *p, QPDevCmdParam *param)
    {
        switch(c)
        {
            case PdcDrawPixmap:
            {
                // we save the position of the pixmap.
#if QT_VERSION < 300
                d_textStart = param[0].point->x() 
                    + param[1].pixmap->width();
#else
                d_textStart = param[0].rect->right();
#endif
                d_textStart += 4;
                break;
            }
            case PdcDrawTextFormatted:
            case PdcDrawText2Formatted:
            {
                if ( d_textStart > 0 )
                {
                    // we align the label 4 pixels left of
                    // the pixmap

                    QRect r = *(param[0].rect);
                    r.moveBy(d_textStart - r.x(), 0);

                    QPDevCmdParam changed[3];
                    changed[0].rect = &r;
                    changed[1].ival = param[1].ival;
                    changed[1].ival &= ~Qt::AlignHCenter;
                    changed[1].ival |= Qt::AlignLeft;

                    changed[2].str = param[2].str;

                    return QPicture::cmd(c, p, changed);
                }
                break;
            }
        }
        return QPicture::cmd(c, p, param);
    }
private:
    int d_textStart;
};

/*!
  Draws the button label. Does some dirty tricks to draw
  only 90% style conform.

  \param painter Painter
  \sa QwtLegendButton::draw()
*/

void QwtLegendButton::drawButtonLabel(QPainter *painter)
{
    // Most (maybe all ?) styles align the button text
    // centered. We need to have it left to the
    // icon. Unfortunately there is no easy way to do it,
    // so we record all paint commands to a QPicture,
    // filter the drawText command and replace the
    // alignment. 

    // In opposite to a regular QPushButton, QwtLegendButton
    // has more a character of an info label than a control element. 
    // So even in disabled state, we like to see the active palette.

    PaintFilter filter;
    
    bool disabled = (bool)testWState(WState_Disabled);
    clearWState(WState_Disabled);

    QPainter picPainter(&filter);
    QPushButton::drawButtonLabel(&picPainter);
    picPainter.end();

    if ( disabled )
        setWState(WState_Disabled); // reset state

    filter.play(painter); // paint the filtered paint commands
}

/*! 
  \return Minimum size hint
  \sa minimumSizeHint()
*/
QSize QwtLegendButton::sizeHint() const
{
    return minimumSizeHint();
}

/*! 
  Minimum size hint
  \sa sizeHint()
*/
QSize QwtLegendButton::minimumSizeHint() const
{
    const QFontMetrics fm(font());

    int h = QPushButton::sizeHint().height();

    int w = iconSet()->pixmap(
        QIconSet::Small, QIconSet::Normal).width();
    w += 4;
    w += fm.size(ShowPrefix, text()).width();

#if QT_VERSION < 300
    w += 2 * (style().buttonMargin() - 1);
    if ( isDefault() || autoDefault() ) 
        w += 2 * style().buttonDefaultIndicatorWidth();
#else
    w = style().sizeFromContents(QStyle::CT_PushButton, 
        this, QSize(w, fm.height() + 2)).width();
#endif

    return QSize(w, h).expandedTo(QApplication::globalStrut());
}

/*!
  \param parent Parent widget
  \param name Widget name
*/
QwtLegend::QwtLegend(QWidget *parent, const char *name): 
    QScrollView(parent, name),
    d_displayPolicy(QwtLegend::Auto),
    d_identifierMode(QwtLegendButton::ShowLine
             | QwtLegendButton::ShowSymbol
             | QwtLegendButton::ShowText)
{
    setFrameStyle(NoFrame);
    setResizePolicy(Manual);

    viewport()->setBackgroundMode(QWidget::NoBackground); // Avoid flicker

    d_contentsWidget = new QWidget(viewport());
    d_contentsWidget->installEventFilter(this);

    QwtDynGridLayout *layout = new QwtDynGridLayout(d_contentsWidget);
    layout->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
    layout->setAutoAdd(TRUE);

    addChild(d_contentsWidget);
}

/*!
  \param int policy
  \param int mode
*/
void QwtLegend::setDisplayPolicy(int policy, int mode)
{
    d_displayPolicy = policy;
    if (-1 != mode)
       d_identifierMode = mode;
}

/*!
  Insert a new item for a specific key.
  \param item New legend item
  \param key Unique key. Key must be >= 0.
  \note The parent of item will be changed to QwtLegend::contentsWidget()
  \note In case of key < 0, nothing will be inserted.
*/
void QwtLegend::insertItem(QWidget *item, long key)
{
    if ( item == NULL || key < 0 )
        return;

    if ( item->parent() != d_contentsWidget )
        item->reparent(d_contentsWidget, QPoint(0, 0));

    item->show();

    if ( d_items.count() > d_items.size() - 5 )
        d_items.resize(d_items.count() + 5);

    d_items.insert(key, item);

    layoutContents();

    QWidget *w = 0;

    if ( d_contentsWidget->layout() )
    {
        // set tab focus chain

        QLayoutIterator layoutIterator = 
            d_contentsWidget->layout()->iterator();

        for ( QLayoutItem *item = layoutIterator.current();
            item != 0; item = ++layoutIterator)
        {
            if ( w && item->widget() )
                QWidget::setTabOrder(w, item->widget());

            w = item->widget();
        }
    }
}

/*!
  Return the key of an legend item.
  \param item Legend item
  \return key of the item, or -1 if the item can't be found.
*/
long QwtLegend::key(const QWidget *item) const
{
    QIntDictIterator<QWidget> it(d_items);
    for ( const QWidget *w = it.toFirst(); w != 0; w = ++it)
    {
        if ( w == item )
            return it.currentKey();
    }
    return -1;
}

//! Remove all items.
void QwtLegend::clear()
{
    // We can't delete the items while we are running
    // through the iterator. So we collect them in
    // a list first.

    QValueList<QWidget *> clearList;
    
    QIntDictIterator<QWidget> it(d_items);
    for ( QWidget *item = it.toFirst(); item != 0; item = ++it)
        clearList += item;

    for ( uint i = 0; i < clearList.count(); i++ )
        delete clearList[i];

#if QT_VERSION < 232
    // In Qt 2.3.0 the ChildRemoved events are not sent, before the
    // first show of the legend. Thus the deleted items are not cleared
    // from the list in QwtLegend::eventFilter. In most cases
    // the following clear is useless, but is is safe to do so.
    
    d_items.clear();
#endif
}

//! Return an item iterator.
QIntDictIterator<QWidget> QwtLegend::itemIterator() const
{
    return QIntDictIterator<QWidget>(d_items);
}

//! Return a size hint.
QSize QwtLegend::sizeHint() const
{
    QSize hint = d_contentsWidget->sizeHint();
    hint += QSize(2 * frameWidth(), 2 * frameWidth());
    
    return hint;
}

/*!
  \return The preferred height, for the width w.
*/
int QwtLegend::heightForWidth(int w) const
{
    w -= 2 * frameWidth();

    int h = d_contentsWidget->heightForWidth(w);
    if ( h <= 0 ) // not implemented, we try the layout
    {
        QLayout *l = d_contentsWidget->layout();
        if ( l && l->hasHeightForWidth() )
        {
            h = l->heightForWidth(w);
            h += 2 * frameWidth();
        }
    }

    return h;
}

/*!
  Adjust contents widget and item layout to the size of the viewport().
*/
void QwtLegend::layoutContents()
{
    const QSize visibleSize = viewport()->size();

    const QLayout *l = d_contentsWidget->layout();
    if ( l && l->inherits("QwtDynGridLayout") )
    {
        const QwtDynGridLayout *tl = (const QwtDynGridLayout *)l;

        const int minW = int(tl->maxItemWidth()) + 2 * tl->margin();

        int w = QMAX(visibleSize.width(), minW);
        int h = QMAX(tl->heightForWidth(w), visibleSize.height());

        const int vpWidth = viewportSize(w, h).width();

        if ( w > vpWidth )
        {
            w = QMAX(vpWidth, minW);
            h = QMAX(tl->heightForWidth(w), visibleSize.height());
        }

        d_contentsWidget->resize(w, h);
        resizeContents(w, h);
    }
}

/*
  Filter QEvent::ChildRemoved, and QEvent::LayoutHint for
  QwtLegend::contentsWidget().
*/

bool QwtLegend::eventFilter(QObject *o, QEvent *e)
{
    if ( o == d_contentsWidget )
    {
        switch(e->type())
        {
            case QEvent::ChildRemoved:
            {   
                const QChildEvent *ce = (const QChildEvent *)e;
                if ( ce->child()->isWidgetType() )
                    (void)takeItem(key((QWidget *)ce->child()));
                break;
            }
            case QEvent::LayoutHint:
            {
                layoutContents();
                break;
            }
            default:
                break;
        }
    }
    
    return QScrollView::eventFilter(o, e);
}

/*
  Resize the viewport() and post a QEvent::LayoutHint to
  QwtLegend::contentsWidget() to update the layout.
*/
void QwtLegend::viewportResizeEvent(QResizeEvent *e)
{
    QScrollView::viewportResizeEvent(e);

    // It's not safe to update the layout now, because
    // we are in an internal update of the scrollview framework.
    // So we delay the update by posting a LayoutHint.

    QApplication::postEvent(d_contentsWidget, new QEvent(QEvent::LayoutHint));
}
