/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <qwindowdefs.h>
#include <qrect.h>
#include <qpainter.h>
#include <qpaintdevice.h>
#include <qpaintdevicemetrics.h>
#include "qwt_painter.h"
#include "qwt_math.h"

#if defined(Q_WS_X11)
bool QwtPainter::d_deviceClipping = TRUE;
#else
bool QwtPainter::d_deviceClipping = FALSE;
#endif

double QwtPainter::d_scaleMetricsX = 1.0;
double QwtPainter::d_scaleMetricsY = 1.0;

/*!
  En/Disable device clipping. On X11 the default
  for device clipping is enabled, otherwise it is disabled.
  \sa QwtPainter::deviceClipping()
*/
void QwtPainter::setDeviceClipping(bool enable)
{
    d_deviceClipping = enable;
}

/*!
  Returns whether device clipping is enabled. On X11 the default
  is enabled, otherwise it is disabled.
  \sa QwtPainter::setDeviceClipping()
*/

bool QwtPainter::deviceClipping()
{
    return d_deviceClipping;
}

/*!
  Returns rect for device clipping
  \sa QwtPainter::setDeviceClipping()
*/
const QRect &QwtPainter::deviceClipRect()
{
    static QRect clip;

    if ( !clip.isValid() )
    {
        clip.setCoords(QWT_COORD_MIN, QWT_COORD_MIN,
            QWT_COORD_MAX, QWT_COORD_MAX);
    }
    return clip;
}

/*!
  Scale all QwtPainter drawing operations using the ratio
  QwtPaintMetrics(from).logicalDpiX() / QwtPaintMetrics(to).logicalDpiX()
  and QwtPaintMetrics(from).logicalDpiY() / QwtPaintMetrics(to).logicalDpiY()

  \sa QwtPainter::resetScaleMetrics(), QwtPainter::scaleMetricsX,
        QwtPainter::scaleMetricsY()
*/
void QwtPainter::setScaleMetrics(const QPaintDevice *from, 
    const QPaintDevice *to)
{
    if ( from == 0 || to == 0 )
    {
        resetScaleMetrics();
        return;
    }
    QPaintDeviceMetrics m1(from);
    QPaintDeviceMetrics m2(to);

    d_scaleMetricsX = double(m2.logicalDpiX()) / double(m1.logicalDpiX());
    d_scaleMetricsY = double(m2.logicalDpiY()) / double(m1.logicalDpiY());
}

/*!
  Disable scaling
  \sa QwtPainter::setScaleMetrics
*/
void QwtPainter::resetScaleMetrics()
{
    d_scaleMetricsX = d_scaleMetricsY = 1.0;
}

/*!
  Return the x-scaling factor
  \sa QwtPainter::setScaleMetrics(), QwtPainter::scaleMetricsY()
*/
double QwtPainter::scaleMetricsX()
{
    return d_scaleMetricsX;
}

/*!
  Return the y-scaling factor
  \sa QwtPainter::setScaleMetrics(), QwtPainter::scaleMetricsX()
*/
double QwtPainter::scaleMetricsY()
{
    return d_scaleMetricsY;
}

/*!
  Wrapper for QWMatrix::mapRect. Hides Qt2/3 incompatibilities.
*/

QRect QwtPainter::map(const QWMatrix &m, const QRect &rect)
{
#if QT_VERSION < 300
    return m.map(rect.normalize());
#else
    return m.mapRect(rect);
#endif
}

/*!
  QPointArray QWMatrix::operator*(const QPointArray &) const.
  Hides Qt2/3 incompatibilities.
*/

QPointArray QwtPainter::map(const QWMatrix &m, const QPointArray &pa)
{
#if QT_VERSION < 300
    return m.map(pa);
#else
    return m * pa;
#endif
}

/*!
  Scale a rectangle.
  \param rect Rect to scale
  \param painter Respect translations, if painter != 0
  \return scaled rect
  \sa QwtPainter::setScaleMetrics()
*/
QRect QwtPainter::scale(const QRect &rect, const QPainter *painter)
{
    if ( d_scaleMetricsX == 1.0 && d_scaleMetricsY == 1.0 )
        return rect;
    
    QRect scaledRect(rect);
    if ( painter )
    {
        scaledRect = QwtPainter::map(
            painter->worldMatrix(), scaledRect);
    }

    scaledRect = QRect(
        qwtInt(scaledRect.x() * d_scaleMetricsX),
        qwtInt(scaledRect.y() * d_scaleMetricsY),
        qwtInt(scaledRect.width() * d_scaleMetricsX),
        qwtInt(scaledRect.height() * d_scaleMetricsY)
    );

    if ( painter )
    {
        scaledRect = QwtPainter::map(
            painter->worldMatrix().invert(), scaledRect);
    }

    return scaledRect;
}

/*!
  Scale a rectangle.
  \param pos Point to scale
  \param painter Respect translations, if painter != 0
  \return Scaled point
  \sa QwtPainter::setScaleMetrics()
*/

QPoint QwtPainter::scale(const QPoint &pos, const QPainter *painter)
{
    if ( d_scaleMetricsX == 1.0 && d_scaleMetricsY == 1.0 )
        return pos;

    QPoint scaledPos(pos);

    if ( painter )
        scaledPos = painter->worldMatrix().map(scaledPos);

    scaledPos = QPoint(qwtInt(scaledPos.x() * d_scaleMetricsX), 
        qwtInt(scaledPos.y() * d_scaleMetricsY));

    if ( painter )
        scaledPos = painter->worldMatrix().invert().map(scaledPos);

    return scaledPos;
}

/*!
  Scale a point array.
  \param pa Point array to scale
  \param painter Respect translations, if painter != 0
  \return Scaled point array
  \sa QwtPainter::setScaleMetrics()
*/

QPointArray QwtPainter::scale(const QPointArray &pa, 
    const QPainter *painter)
{
    if ( d_scaleMetricsX == 1.0 && d_scaleMetricsY == 1.0 )
        return pa;

    QPointArray scaledPa(pa);

    if ( painter )
        scaledPa = QwtPainter::map(painter->worldMatrix(), scaledPa);

    QWMatrix m;
    m.scale(d_scaleMetricsX, d_scaleMetricsY);
    scaledPa = QwtPainter::map(m, scaledPa);

    if ( painter )
    {
        scaledPa = QwtPainter::map(
            painter->worldMatrix().invert(), scaledPa);
    }

    return scaledPa;
}

/*!
  Scale a rect, with inverted scale metrics
  \param rect Rectange to be scaled
  \param painter Respect translations, if painter != 0
  \return Scaled point array
  \sa QwtPainter::setScaleMetrics()
*/
QRect QwtPainter::invScale(const QRect &rect, const QPainter *painter)
{
    if ( d_scaleMetricsX == 1.0 && d_scaleMetricsY == 1.0 )
        return rect;
    
    QRect scaledRect(rect);
    if ( painter )
    {
        scaledRect = QwtPainter::map(
            painter->worldMatrix(), scaledRect);
    }

    scaledRect = QRect(
        qwtInt(scaledRect.x() / d_scaleMetricsX),
        qwtInt(scaledRect.y() / d_scaleMetricsY),
        qwtInt(scaledRect.width() / d_scaleMetricsX),
        qwtInt(scaledRect.height() / d_scaleMetricsY)
    );

    if ( painter )
    {
        scaledRect = QwtPainter::map(
            painter->worldMatrix().invert(), scaledRect);
    }

    return scaledRect;
}

/*!
  Scale a point, with inverted scale metrics
  \param pos Point to scale
  \param painter Respect translations, if painter != 0
  \return Scaled point
  \sa QwtPainter::setScaleMetrics()
*/
QPoint QwtPainter::invScale(const QPoint &pos, const QPainter *painter)
{
    if ( d_scaleMetricsX == 1.0 && d_scaleMetricsY == 1.0 )
        return pos;

    QPoint scaledPos(pos);

    if ( painter )
        scaledPos = painter->worldMatrix().map(scaledPos);

    scaledPos = QPoint(qwtInt(scaledPos.x() / d_scaleMetricsX), 
        qwtInt(scaledPos.y() / d_scaleMetricsY));

    if ( painter )
        scaledPos = painter->worldMatrix().invert().map(scaledPos);

    return scaledPos;
}

/*!
    Wrapper for QPainter::setClipRect()
*/
void QwtPainter::setClipRect(QPainter *painter, const QRect &rect)
{
    painter->setClipRect(scale(rect, painter));
}

/*!
    Wrapper for QPainter::drawRect()
*/
void QwtPainter::drawRect(QPainter *painter, int x, int y, int w, int h) 
{
    QwtPainter::drawRect(painter, QRect(x, y, w, h));
}

/*!
    Wrapper for QPainter::drawRect()
*/
void QwtPainter::drawRect(QPainter *painter, const QRect &rect) 
{
    painter->drawRect(scale(rect, painter));
}

/*!
    Wrapper for QPainter::fillRect()
*/
void QwtPainter::fillRect(QPainter *painter, 
    const QRect &rect, const QBrush &brush)
{
    painter->fillRect(scale(rect, painter), brush);
}

/*!
    Wrapper for QPainter::drawEllipse()
*/
void QwtPainter::drawEllipse(QPainter *painter, const QRect &rect)
{
    painter->drawEllipse(scale(rect, painter));
}

/*!
    Wrapper for QPainter::drawText()
*/
void QwtPainter::drawText(QPainter *painter, int x, int y, 
        const QString &text, int len)
{
    drawText(painter, QPoint(x, y), text, len);
}

/*!
    Wrapper for QPainter::drawText()
*/
void QwtPainter::drawText(QPainter *painter, const QPoint &pos, 
        const QString &text, int len)
{
    painter->drawText(scale(pos, painter), text, len);
}

/*!
    Wrapper for QPainter::drawText()
*/
void QwtPainter::drawText(QPainter *painter, int x, int y, int w, int h, 
        int flags, const QString &text, int len)
{
    drawText(painter, QRect(x, y, w, h), flags, text, len);
}

/*!
    Wrapper for QPainter::drawText()
*/
void QwtPainter::drawText(QPainter *painter, const QRect &rect, 
        int flags, const QString &text, int len)
{
    painter->drawText(scale(rect, painter), 
        flags, text, len);
}

/*!
  Wrapper for QPainter::drawLine()
*/
void QwtPainter::drawLine(QPainter *painter, 
    const QPoint &p1, const QPoint &p2) 
{
    if ( d_deviceClipping && 
        !(deviceClipRect().contains(p1) && deviceClipRect().contains(p2)) )
    {
        QPointArray pa(2);
        pa.setPoint(0, p1);
        pa.setPoint(1, p2);
        drawPolyline(painter, pa);
    }
    else
    {
        painter->drawLine(scale(p1), scale(p2));
    }
}

/*!
  Wrapper for QPainter::drawLine()
*/

void QwtPainter::drawLine(QPainter *painter, int x1, int y1, int x2, int y2)
{
    drawLine(painter, QPoint(x1, y1), QPoint(x2, y2));
}

/*!
  Wrapper for QPainter::drawPolygon()
*/

void QwtPainter::drawPolygon(QPainter *painter, const QPointArray &pa)
{
    QPointArray cpa = scale(pa);
    if ( d_deviceClipping )
        cpa = clip(cpa);
    painter->drawPolygon(cpa);
}

/*!
    Wrapper for QPainter::drawPolyline()
*/
void QwtPainter::drawPolyline(QPainter *painter, const QPointArray &pa)
{
    QPointArray cpa = scale(pa);
    if ( d_deviceClipping )
        cpa = clip(cpa);
    painter->drawPolyline(cpa);
}

/*!
    Wrapper for QPainter::drawPoint()
*/

void QwtPainter::drawPoint(QPainter *painter, int x, int y)
{
    if ( d_deviceClipping && !deviceClipRect().contains(x, y) )
        return;

    painter->drawPoint(scale(QPoint(x,y)));
}

QPointArray QwtPainter::clip(const QPointArray &pa)
{
    int idx = 0;
    int size = pa.size();
    
    return clipPolyline(deviceClipRect(), pa, idx, size);
}

bool QwtPainter::insideEdge( const QPoint &p, const QRect &r, int edge )
{
    switch ( edge ) 
    {
        case 0:
            return p.x() > r.left();
        case 1:
            return p.y() > r.top();
        case 2:
            return p.x() < r.right();
        case 3:
            return p.y() < r.bottom();
    }

    return FALSE;
}

QPoint QwtPainter::intersectEdge(const QPoint &p1, const QPoint &p2,
    const QRect &r, int edge )
{
    int x=0, y=0;
    double m = 0;
    double dy = p2.y() - p1.y();
    double dx = p2.x() - p1.x();

    switch ( edge ) 
    {
        case 0:
            x = r.left();
            m = double(QABS(p1.x() - x)) / QABS(dx);
            y = p1.y() + int(dy * m);
            break;
        case 1:
            y = r.top();
            m = double(QABS(p1.y() - y)) / QABS(dy);
            x = p1.x() + int(dx * m);
            break;
        case 2:
            x = r.right();
            m = double(QABS(p1.x() - x)) / QABS(dx);
            y = p1.y() + int(dy * m);
            break;
        case 3:
            y = r.bottom();
            m = double(QABS(p1.y() - y)) / QABS(dy);
            x = p1.x() + int(dx * m);
            break;
    }

    return QPoint(x,y);
}

/*!
    Sutherland-Hodgman polygon clipping
*/

QPointArray QwtPainter::clipPolyline(const QRect &r,
    const QPointArray &pa, int &index, int &npoints )
{
    if ( r.contains( pa.boundingRect() ) )
        return pa;

    QPointArray rpa = pa;
    QPointArray cpa( pa.size() );
    unsigned int count = 0;

#define ADD_POINT(p) if ( cpa.size() == count ) cpa.resize( count+5 ); \
             cpa.setPoint( count++, p );

    for ( int e = 0; e < 4; e++ ) 
    {
        count = 0;
        QPoint s = rpa.point( index );
        for ( int j = 0; j < npoints; j++ ) 
        {
            QPoint p = rpa.point( index+j );
            if ( insideEdge( p, r, e ) ) 
            {
                if ( insideEdge( s, r, e ) ) 
                {
                    ADD_POINT(p);
                } 
                else 
                {
                    QPoint i = intersectEdge( s, p, r, e );
                    ADD_POINT(i);
                    ADD_POINT(p);
                }
            } 
            else 
            {
                if ( insideEdge( s, r, e ) ) 
                {
                    QPoint i = intersectEdge( s, p, r, e );
                    ADD_POINT(i);
                }
            }
            s = p;
        }
        index = 0;
        npoints = count;
        rpa = cpa.copy();
    }

#undef ADD_POINT

    cpa.resize( npoints );

    return cpa;
}

void QwtPainter::drawRoundFrame(QPainter *painter, const QRect &rect,
    int width, bool sunken)
{
    const int colorDelta = 5;
    const int offset = 4;

    int color = 255;
    for ( int angle = 0; angle < 360; angle += offset )
    {
        int c = color;
        if ( !sunken )
            c = 100 + 255 - c;

        QPen pen(QColor(c, c, c), width);
        painter->setPen(pen);
        painter->drawArc(rect, angle * 16, offset * 16);

        if ( ( angle > 145 ) && ( angle < 360 ) ) 
            color += colorDelta;         // brighter
        else 
            color -= colorDelta;         // darker

        if ( color < 100 )
            color = 100;

        if ( color > 255 )
            color = 255;
    }
}

