/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <qlabel.h>
#include "qwt_rect.h"
#include "qwt_plot_canvas.h"
#include "qwt_scale.h"
#include "qwt_legend.h"
#include "qwt_plot_layout.h"

class QwtPlotLayoutData
{
    friend class QwtPlotLayout;

protected:
    QwtPlotLayoutData();

    void init(const QwtPlot *, const QRect &rect);

    struct t_legendData
    {
        int frameWidth;
        int vScrollBarWidth;
        int hScrollBarHeight;
        QSize hint;
    } legend;
    
    struct t_titleData
    {
        QFont font;
        QString text;
        int align;
        int frameWidth;
    } title;

    struct t_scaleData
    {
        bool isEnabled;
        QString title;
        int titleAlign;
        QFont titleFont;
        QFont scaleFont;
        int start;
        int end;
        int baseLineOffset;
        int tickOffset; 
        int dimWithoutTitle;
    } scale[QwtPlot::axisCnt];

    struct t_canvasData
    {
        int frameWidth;
    } canvas;
};

/*!
  \brief Constructor
 */

QwtPlotLayout::QwtPlotLayout():
    d_margin(0),
    d_spacing(5),
    d_canvasMargin(4)
{
    setLegendPos(Qwt::Bottom);
    d_layoutData = new QwtPlotLayoutData;

    invalidate();
}

//! Destructor
QwtPlotLayout::~QwtPlotLayout()
{
    delete d_layoutData;
}

/*!
  Change the margin of the plot. The margin is the space
  around all components.
 
  \param margin new margin
  \sa QwtPlotLayout::margin(), QwtPlotLayout::setSpacing(),
      QwtPlot::setMargin()
*/

void QwtPlotLayout::setMargin(int margin)
{
    if ( margin < 0 )
        margin = 0;
    d_margin = margin;
}

/*!
    \return margin
    \sa QwtPlotLayout::setMargin(), QwtPlotLayout::spacing(), 
        QwtPlot::margin()
*/

int QwtPlotLayout::margin() const
{
    return d_margin;
}

/*!
  Change the spacing of the plot. The spacing is the distance
  between the plot components.
 
  \param spacing new spacing
  \sa QwtPlotLayout::setMargin(), QwtPlotLayout::spacing() 
*/

void QwtPlotLayout::setSpacing(int spacing)
{
    d_spacing = QMAX(0, spacing);
}

/*!
  \return spacing
  \sa QwtPlotLayout::margin(), QwtPlotLayout::setSpacing() 
*/
int QwtPlotLayout::spacing() const
{
    return d_spacing;
}

/*!
  \brief Specify the position of the legend
  \param pos The legend's position. Valid values are \c Qwt::Left,
           \c Qwt::Right, \c Qwt::Top, \c QwtBottom.
  \param ratio Ratio between legend and the bounding rect 
               of title, canvas and axes. The legend will be shrinked
               if it would need more space than the given ratio.
               The ratio is limited to ]0.0 .. 1.0]. In case of <= 0.0
               it will be reset to the default ratio.
               The default vertical/horizontal ratio is 0.33/0.5. 
               
  \sa QwtPlot::setLegendPos()
*/

void QwtPlotLayout::setLegendPos(int pos, double ratio)
{
    if ( ratio > 1.0 )
        ratio = 1.0;

    switch(pos)
    {
        case Qwt::Top:
        case Qwt::Bottom:
            if ( ratio <= 0.0 )
                ratio = 0.33;
            d_legendRatio = ratio;
            d_legendPos = pos;
            break;
        case Qwt::Left:
        case Qwt::Right:
            if ( ratio <= 0.0 )
                ratio = 0.5;
            d_legendRatio = ratio;
            d_legendPos = pos;
            break;
        default:
            break;
    }
}

/*!
  \return Position of the legend
  \sa QwtPlotLayout::setLegendPos(), QwtPlot::setLegendPos(),
      QwtPlot::legendPos()
*/

int QwtPlotLayout::legendPos() const
{
    return d_legendPos;
}

/*!
  \return Position of the legend
  \sa QwtPlotLayout::setLegendPos(), QwtPlot::setLegendPos()
*/

double QwtPlotLayout::legendRatio() const
{
    return d_legendRatio;
}

/*!
  \return Geometry for the title
  \sa QwtPlotLayout::activate(), QwtPlotLayout::invalidate()
*/

const QRect &QwtPlotLayout::titleRect() const
{
    return d_titleRect;
}

/*!
  \return Geometry for the legend
  \sa QwtPlotLayout::activate(), QwtPlotLayout::invalidate()
*/

const QRect &QwtPlotLayout::legendRect() const
{
    return d_legendRect;
}

/*!
  \param axis Axis index
  \return Geometry for the scale
  \sa QwtPlotLayout::activate(), QwtPlotLayout::invalidate()
*/

const QRect &QwtPlotLayout::scaleRect(int axis) const
{
    if ( axis < 0 || axis >= QwtPlot::axisCnt )
    {
        static QRect dummyRect;
        return dummyRect;
    }
    return d_scaleRect[axis];
}

/*!
  \return Geometry for the canvas
  \sa QwtPlotLayout::activate(), QwtPlotLayout::invalidate()
*/

const QRect &QwtPlotLayout::canvasRect() const
{
    return d_canvasRect;
}

/*!
  Invalidate the geometry of all components. 
  \sa QwtPlotLayout::activate()
*/
void QwtPlotLayout::invalidate()
{
    d_titleRect = d_legendRect = d_canvasRect = QRect();
    for (int axis = 0; axis < QwtPlot::axisCnt; axis++ )
        d_scaleRect[axis] = QRect();
}

/*!  
  \brief Return a minimum size hint
  \sa QwtPlot::minimumSizeHint()
*/

QSize QwtPlotLayout::minimumSizeHint(const QwtPlot *plot) const
{
    class ScaleData
    {
    public:
        ScaleData()
        {
            w = h = minLeft = minRight = tickOffset = 0;
        }

        int w;
        int h;
        int minLeft;
        int minRight;
        int tickOffset;
    } scaleData[QwtPlot::axisCnt];

    int axis;
    for ( axis = 0; axis < QwtPlot::axisCnt; axis++ )
    {
        const QwtScale *scl = plot->axis(axis);
        if ( scl )
        {
            ScaleData &sd = scaleData[axis];

            const QSize hint = scl->minimumSizeHint();
            sd.w = hint.width(); 
            sd.h = hint.height(); 
            scl->minBorderDist(sd.minLeft, sd.minRight);
            sd.tickOffset = scl->baseLineDist() +
                scl->scaleDraw()->majTickLength();
        }
    }

    const int canvasBorder = plot->canvas()->frameWidth() 
        + d_canvasMargin + 1;
    for ( axis = 0; axis < QwtPlot::axisCnt; axis++ )
    {
        ScaleData &sd = scaleData[axis];
        if ( sd.w && (axis == QwtPlot::xBottom || axis == QwtPlot::xTop) )
        {
            if ( (sd.minLeft > canvasBorder) 
                && scaleData[QwtPlot::yLeft].w )
            {
                int shiftLeft = sd.minLeft - canvasBorder;
                if ( shiftLeft > scaleData[QwtPlot::yLeft].w )
                    shiftLeft = scaleData[QwtPlot::yLeft].w;

                sd.w -= shiftLeft;
            }
            if ( (sd.minRight > canvasBorder) 
                && scaleData[QwtPlot::yRight].w )
            {
                int shiftRight = sd.minRight - canvasBorder;
                if ( shiftRight > scaleData[QwtPlot::yRight].w )
                    shiftRight = scaleData[QwtPlot::yRight].w;

                sd.w -= shiftRight;
            }
        }

        if ( sd.h && (axis == QwtPlot::yLeft || axis == QwtPlot::yRight) )
        {
            if ( (sd.minLeft > canvasBorder) &&
                scaleData[QwtPlot::xBottom].h )
            {
                int shiftBottom = sd.minLeft - canvasBorder;
                if ( shiftBottom > scaleData[QwtPlot::xBottom].tickOffset )
                    shiftBottom = scaleData[QwtPlot::xBottom].tickOffset;

                sd.h -= shiftBottom;
            }
            if ( (sd.minLeft > canvasBorder) &&
                scaleData[QwtPlot::xTop].h )
            {
                int shiftTop = sd.minRight - canvasBorder;
                if ( shiftTop > scaleData[QwtPlot::xTop].tickOffset )
                    shiftTop = scaleData[QwtPlot::xTop].tickOffset;

                sd.h -= shiftTop;
            }
        }
    }

    const QwtPlotCanvas *canvas = plot->canvas();

    int w = scaleData[QwtPlot::yLeft].w + scaleData[QwtPlot::yRight].w
        + QMAX(scaleData[QwtPlot::xBottom].w, scaleData[QwtPlot::xTop].w)
        + 2 * (canvas->frameWidth() + 1);
    int h = scaleData[QwtPlot::xBottom].h + scaleData[QwtPlot::xTop].h 
        + QMAX(scaleData[QwtPlot::yLeft].h, scaleData[QwtPlot::yRight].h)
        + 2 * (canvas->frameWidth() + 1);

    const QLabel *title = plot->titleLabel();
    if (title && !title->text().isEmpty())
    {
        // If only QwtPlot::yLeft or QwtPlot::yRight is showing, 
        // we center on the plot canvas.
        const bool centerOnCanvas = plot->axis(QwtPlot::yLeft) == 0 
            || plot->axis(QwtPlot::yRight) == 0;

        int titleW = w;
        if ( centerOnCanvas )
        {
            titleW -= scaleData[QwtPlot::yLeft].w 
                + scaleData[QwtPlot::yRight].w;
        }

        int titleH = title->heightForWidth(titleW);
        if ( titleH > titleW ) // Compensate for a long title
        {
            w = titleW = titleH;
            if ( centerOnCanvas )
            {
                w += scaleData[QwtPlot::yLeft].w
                    + scaleData[QwtPlot::yRight].w;
            }

            titleH = title->heightForWidth(titleW);
        }
        h += titleH + d_spacing;
    }

    // Compute the legend contribution

    const QwtLegend *legend = plot->legend();
    if ( legend && !legend->isEmpty() )
    {
        if ( d_legendPos == Qwt::Left || d_legendPos == Qwt::Right )
        {
            int legendW = legend->sizeHint().width();
            int legendH = legend->heightForWidth(legendW); 

            if ( legend->frameWidth() > 0 )
                w += d_spacing;

            if ( legendH > h )
                legendW += legend->verticalScrollBar()->sizeHint().height();

            if ( d_legendRatio < 1.0 )
                legendW = QMIN(legendW, int(w / (1.0 - d_legendRatio)));

            w += legendW;
        }
        else // Qwt::Top, Qwt::Bottom
        {
            int legendW = QMIN(legend->sizeHint().width(), w);
            int legendH = legend->heightForWidth(legendW); 

            if ( legend->frameWidth() > 0 )
                h += d_spacing;

            if ( d_legendRatio < 1.0 )
                legendH = QMIN(legendH, int(h / (1.0 - d_legendRatio)));
            
            h += legendH;
        }
    }

    w += 2 * d_margin;
    h += 2 * d_margin;

    return QSize( w, h );
}

/*!
  Find the geometry for the legend
  \param options Options how to layout the legend
  \param rect Rectangle where to place the legend
  \return Geometry for the legend
*/

QRect QwtPlotLayout::layoutLegend(int options, 
    const QRect &rect) const
{
    const QSize hint(d_layoutData->legend.hint);

    int dim;
    if ( d_legendPos == Qwt::Left || d_legendPos == Qwt::Right )
    {
        // We don't allow vertical legends to take more than
        // half of the available space.

        dim = QMIN(hint.width(), int(rect.width() * d_legendRatio));

        if ( !(options & IgnoreScrollbars) )
        {
            if ( hint.height() > rect.height() )
            {
                // The legend will need additional
                // space for the vertical scrollbar. 

                dim += d_layoutData->legend.vScrollBarWidth;
            }
        }
    }
    else
    {
        dim = QMIN(hint.height(), int(rect.height() * d_legendRatio));
        dim = QMAX(dim, d_layoutData->legend.hScrollBarHeight);
    }

    QRect legendRect = rect;
    switch(d_legendPos)
    {
        case Qwt::Left:
            legendRect.setWidth(dim);
            break;
        case Qwt::Right:
            legendRect.setX(rect.right() - dim + 1);
            legendRect.setWidth(dim);
            break;
        case Qwt::Top:
            legendRect.setHeight(dim);
            break;
        case Qwt::Bottom:
            legendRect.setY(rect.bottom() - dim + 1);
            legendRect.setHeight(dim);
            break;
    }

    return legendRect;
}

/*!
  Align the legend to the canvas
  \param canvasRect Geometry of the canvas
  \param legendRect Maximum geometry for the legend
  \return Geometry for the aligned legend
*/
QRect QwtPlotLayout::alignLegend(const QRect &canvasRect, 
    const QRect &legendRect) const
{
    QRect alignedRect = legendRect;

    if ( d_legendPos == Qwt::Bottom || d_legendPos == Qwt::Top )
    {
        if ( d_layoutData->legend.hint.width() < canvasRect.width() )
        {
            alignedRect.setX(canvasRect.x());
            alignedRect.setWidth(canvasRect.width());
        }
    }
    else
    {
        if ( d_layoutData->legend.hint.height() < canvasRect.height() )
        {
            alignedRect.setY(canvasRect.y());
            alignedRect.setHeight(canvasRect.height());
        }
    }

    return alignedRect;
}

/*!
  Expand all line breaks in text labels, and calculate the height
  of their widgets in orientation of the text.

  \param options Options how to layout the legend
  \param rect Bounding rect for title, axes and canvas.
  \param dimTitle Expanded height of the title widget
  \param dimAxis Expanded heights of the axis in axis orientation.
*/
void QwtPlotLayout::expandLineBreaks(int options, const QRect &rect, 
    int &dimTitle, int dimAxis[QwtPlot::axisCnt]) const
{
    dimTitle = 0;
    for ( int i = 0; i < QwtPlot::axisCnt; i++ )
        dimAxis[i] = 0;

    bool done = FALSE;
    while (!done)
    {
        done = TRUE;

        // the size for the 4 axis depend on each other. Expanding
        // the height of a horizontal axis will shrink the height
        // for the vertical axis, shrinking the height of a vertical
        // axis will result in a line break what will expand the
        // width and results in shrinking the width of a horizontal
        // axis what might result in a line break of a horizontal
        // axis ... . So we loop as long until no size changes.

        if ( !d_layoutData->title.text.isEmpty())
        {
            int w = rect.width();

            if ( d_layoutData->scale[QwtPlot::yLeft].isEnabled
                != d_layoutData->scale[QwtPlot::yRight].isEnabled )
            {
                // center to the canvas
                w -= dimAxis[QwtPlot::yLeft] + dimAxis[QwtPlot::yRight]; 
            }

            const QFontMetrics fm(d_layoutData->title.font);
            int d = fm.boundingRect(0, 0, w, QCOORD_MAX,
                d_layoutData->title.align, 
                d_layoutData->title.text).height();

            if ( !(options & IgnoreFrames) )
                d += 2 * d_layoutData->title.frameWidth;

            if ( d > dimTitle )
            {
                dimTitle = d;
                done = FALSE;
            }
        }

        for ( int axis = 0; axis < QwtPlot::axisCnt; axis++ )
        {
            const struct QwtPlotLayoutData::t_scaleData &scaleData = 
                d_layoutData->scale[axis];

            if (scaleData.isEnabled)
            {
                int length;
                if ( axis == QwtPlot::xTop || axis == QwtPlot::xBottom )
                {
                    length = rect.width() - dimAxis[QwtPlot::yLeft] 
                        - dimAxis[QwtPlot::yRight];
                    length += QMIN(dimAxis[QwtPlot::yLeft], 
                        scaleData.start);
                    length += QMIN(dimAxis[QwtPlot::yRight], 
                        scaleData.end);
                }
                else // QwtPlot::yLeft, QwtPlot::yRight
                {
                    length = rect.height() - dimAxis[QwtPlot::xTop] 
                        - dimAxis[QwtPlot::xBottom];

                    if ( dimAxis[QwtPlot::xBottom] > 0 )
                    {
                        length += QMIN(
                            d_layoutData->scale[QwtPlot::xBottom].tickOffset, 
                            scaleData.start);
                    }
                    if ( dimAxis[QwtPlot::xTop] > 0 )
                    {
                        length += QMIN(
                            d_layoutData->scale[QwtPlot::xTop].tickOffset, 
                            scaleData.end);
                    }

                    if ( dimTitle > 0 )
                        length -= dimTitle + d_spacing;
                }

                int d = scaleData.dimWithoutTitle;
                if ( !scaleData.title.isEmpty() )
                {
                    const QFontMetrics fm(scaleData.titleFont);
                    d += fm.boundingRect(
                        0, 0, length, QCOORD_MAX, 
                        scaleData.titleAlign, scaleData.title).height();
                }

                if ( options & AlignScales )
                    d -= scaleData.baseLineOffset;

                if ( d > dimAxis[axis] )
                {
                    dimAxis[axis] = d;
                    done = FALSE;
                }
            }
        }
    }
}

/*!
  Align the ticks of the axis to the canvas borders using
  the empty corners.
*/

void QwtPlotLayout::alignScales(int options,
    QRect scaleRect[QwtPlot::axisCnt]) const
{
    int backboneOffset = d_canvasMargin;
    if ( !(options & IgnoreFrames) )
        backboneOffset += d_layoutData->canvas.frameWidth;

    for (int axis = 0; axis < QwtPlot::axisCnt; axis++ )
    {
        if ( !scaleRect[axis].isValid() )
            continue;

        const int startDist = d_layoutData->scale[axis].start;
        const int endDist = d_layoutData->scale[axis].end;

        QRect &axisRect = scaleRect[axis];

        if ( axis == QwtPlot::xTop || axis == QwtPlot::xBottom )
        {
            int minLeft = axisRect.left();
            if ( scaleRect[QwtPlot::yLeft].isValid() )
                minLeft = scaleRect[QwtPlot::yLeft].left();

            int left = axisRect.left() + backboneOffset - startDist;
            axisRect.setLeft(QMAX(left, minLeft));

            int maxRight = axisRect.right();
            if ( scaleRect[QwtPlot::yRight].isValid() )
                maxRight = scaleRect[QwtPlot::yRight].right();

            int right = axisRect.right() - backboneOffset + endDist;
            axisRect.setRight(QMIN(right, maxRight));
        }
        else // QwtPlot::yLeft, QwtPlot::yRight
        {
            int maxBottom = axisRect.bottom();
            if ( scaleRect[QwtPlot::xBottom].isValid() )
            {
                maxBottom = scaleRect[QwtPlot::xBottom].top() + 
                    d_layoutData->scale[QwtPlot::xBottom].tickOffset;
            }

            int bottom = axisRect.bottom() - backboneOffset + startDist;
            axisRect.setBottom(QMIN(bottom, maxBottom));
        
            int minTop = axisRect.top();
            if ( scaleRect[QwtPlot::xTop].isValid() )
            {
                minTop = scaleRect[QwtPlot::xTop].bottom() -
                    d_layoutData->scale[QwtPlot::xTop].tickOffset;
            }

            int top = axisRect.top() + backboneOffset - endDist;
            axisRect.setTop(QMAX(top, minTop));
        }
    }
}

/*!
  \brief Recalculate the geometry of all components. 

  \param plot Plot to be layout
  \param plotRect Rect where to place the components
  \param options Options

  \sa QwtPlotLayout::invalidate(), QwtPlotLayout::titleRect(),
      QwtPlotLayout::legendRect(), QwtPlotLayout::scaleRect(), 
      QwtPlotLayout::canvasRect()
*/

void QwtPlotLayout::activate(const QwtPlot *plot,
    const QRect &plotRect, int options) 
{
    invalidate();

    QRect rect(plotRect);  // undistributed rest of the plot rect

    if ( !(options & IgnoreMargin) )
    {
        // subtract the margin

        rect.setRect(
            rect.x() + d_margin, 
            rect.y() + d_margin,
            rect.width() - 2 * d_margin, 
            rect.height() - 2 * d_margin
        );
    }

    // We extract all layout relevant data from the widgets,
    // filter them through pfilter and save them to d_layoutData.

    d_layoutData->init(plot, rect);

    if (!(options & IgnoreLegend)
        && plot->legend() && !plot->legend()->isEmpty() )
    {
        d_legendRect = layoutLegend(options, rect);

        // subtract d_legendRect from rect

        const QRegion region(rect);
        rect = region.subtract(d_legendRect).boundingRect(); 

        if ( d_layoutData->legend.frameWidth && 
            !(options & IgnoreFrames ) )
        {
            // In case of a frame we have to insert a spacing.
            // Otherwise the leading of the font separates
            // legend and scale/canvas

            switch(d_legendPos)
            {
                case Qwt::Left:
                    rect.setLeft(rect.left() + d_spacing);
                    break;
                case Qwt::Right:
                    rect.setRight(rect.right() - d_spacing);
                    break;
                case Qwt::Top:
                    rect.setTop(rect.top() + d_spacing);
                    break;
                case Qwt::Bottom:
                    rect.setBottom(rect.bottom() - d_spacing);
                    break;
            }
        }
    }

    /*
     +---+-----------+---+
     |       Title       |
     +---+-----------+---+
     |   |   Axis    |   |
     +---+-----------+---+
     | A |           | A |
     | x |  Canvas   | x |
     | i |           | i |
     | s |           | s |
     +---+-----------+---+
     |   |   Axis    |   |
     +---+-----------+---+
    */


    // axes and title include text labels. The height of each
    // label depends on its line breaks, that depend on the width
    // for the label. A line break in a horizontal text will reduce
    // the available width for vertical texts and vice versa. 
    // expandLineBreaks finds the height/width for title and axes
    // including all line breaks.

    int dimTitle, dimAxes[QwtPlot::axisCnt];
    expandLineBreaks(options, rect, dimTitle, dimAxes);

    if (dimTitle > 0 )
    {
        d_titleRect = QRect(rect.x(), rect.y(),
            rect.width(), dimTitle);

        if ( d_layoutData->scale[QwtPlot::yLeft].isEnabled !=
            d_layoutData->scale[QwtPlot::yRight].isEnabled )
        {
            // if only one of the y axes is missing we align
            // the title centered to the canvas

            d_titleRect.setX(rect.x() + dimAxes[QwtPlot::yLeft]);
            d_titleRect.setWidth(rect.width() 
                - dimAxes[QwtPlot::yLeft] - dimAxes[QwtPlot::yRight]);
        }

        // subtract title 
        rect.setTop(rect.top() + dimTitle + d_spacing);
    }

    d_canvasRect.setRect(
        rect.x() + dimAxes[QwtPlot::yLeft],
        rect.y() + dimAxes[QwtPlot::xTop],
        rect.width() - dimAxes[QwtPlot::yRight] - dimAxes[QwtPlot::yLeft],
        rect.height() - dimAxes[QwtPlot::xBottom] - dimAxes[QwtPlot::xTop]);

    for ( int axis = 0; axis < QwtPlot::axisCnt; axis++ )
    {
        // set the rects for the axes

        if ( dimAxes[axis] )
        {
            int dim = dimAxes[axis];
            QRect &scaleRect = d_scaleRect[axis];

            scaleRect = d_canvasRect;
            switch(axis)
            {
                case QwtPlot::yLeft:
                    scaleRect.setX(d_canvasRect.left() - dim);
                    scaleRect.setWidth(dim);
                    break;
                case QwtPlot::yRight:
                    scaleRect.setX(d_canvasRect.right() + 1);
                    scaleRect.setWidth(dim);
                    break;
                case QwtPlot::xBottom:
                    scaleRect.setY(d_canvasRect.bottom() + 1);
                    scaleRect.setHeight(dim);
                    break;
                case QwtPlot::xTop:
                    scaleRect.setY(d_canvasRect.top() - dim);
                    scaleRect.setHeight(dim);
                    break;
            }
            scaleRect = scaleRect.normalize();
        }
    }

    // +---+-----------+---+
    // |  <-   Axis   ->   |
    // +-^-+-----------+-^-+
    // | | |           | | |
    // |   |           |   |
    // | A |           | A |
    // | x |  Canvas   | x |
    // | i |           | i |
    // | s |           | s |
    // |   |           |   |
    // | | |           | | |
    // +-V-+-----------+-V-+
    // |   <-  Axis   ->   |
    // +---+-----------+---+

    // The ticks of the axes - not the labels above - should
    // be aligned to the canvas. So we try to use the empty
    // corners to extend the axes, so that the label texts
    // left/right of the min/max ticks are moved into them.
 
    alignScales(options, d_scaleRect);

    if (!d_legendRect.isEmpty() )
    {
        // We prefer to align the legend to the canvas - not to
        // the complete plot - if possible.

        d_legendRect = alignLegend(d_canvasRect, d_legendRect);
    }
}

QwtPlotLayoutData::QwtPlotLayoutData()
{
}

/*
  Extract all layout relevant data from the plot components
*/

void QwtPlotLayoutData::init(const QwtPlot *plot, const QRect &rect)
{
    // legend

    legend.frameWidth = plot->legend()->frameWidth();
    legend.vScrollBarWidth = 
        plot->legend()->verticalScrollBar()->sizeHint().width();
    legend.hScrollBarHeight = 
        plot->legend()->horizontalScrollBar()->sizeHint().width();

    const QSize hint = plot->legend()->sizeHint();

    int w = QMIN(hint.width(), rect.width());
    int h = plot->legend()->heightForWidth(w);
    if ( h == 0 )
        h = hint.height();

    if ( h > rect.height() )
        w += legend.vScrollBarWidth;

    legend.hint = QSize(w, h);

    // title 

    title.frameWidth = 0;
    title.text = QString::null;

    if (plot->titleLabel() && !plot->titleLabel()->text().isEmpty())
    {
        title.align = plot->titleLabel()->alignment();
        title.font = plot->titleLabel()->font();
        title.text = plot->titleLabel()->text();
        title.frameWidth = plot->titleLabel()->frameWidth();
    }

    // scales 

    for (int axis = 0; axis < QwtPlot::axisCnt; axis++ )
    {
        const QwtScale *sd = plot->axis(axis);
        if ( sd )
        {
            scale[axis].isEnabled = TRUE;

            scale[axis].title = sd->title();
            scale[axis].titleAlign = sd->titleAlignment();
            scale[axis].titleFont = sd->titleFont();

            scale[axis].scaleFont = sd->font();

            scale[axis].start = sd->startBorderDist();
            scale[axis].end = sd->endBorderDist();

            scale[axis].baseLineOffset = sd->baseLineDist();
            scale[axis].tickOffset = sd->baseLineDist() + 
                (int)sd->scaleDraw()->majTickLength();

            scale[axis].dimWithoutTitle = sd->dimForLength(QCOORD_MAX,
                scale[axis].titleFont, scale[axis].scaleFont);

            if ( !sd->title().isEmpty() )
            {
                const QFontMetrics fm(scale[axis].titleFont);
                scale[axis].dimWithoutTitle -= fm.boundingRect(
                    0, 0, QCOORD_MAX, QCOORD_MAX, sd->titleAlignment(),
                    sd->title()).height();
            }
        }
        else
        {
            scale[axis].title = QString::null;
            scale[axis].isEnabled = FALSE;
            scale[axis].start = 0;
            scale[axis].end = 0;
            scale[axis].baseLineOffset = 0;
            scale[axis].tickOffset = 0;
            scale[axis].dimWithoutTitle = 0;
        }
    }

    // canvas 

    canvas.frameWidth = plot->canvas()->frameWidth();
}
