/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qapplication.h>
#include "qwt_painter.h"
#include "qwt_symbol.h"

/*!
  Default Constructor

  The symbol is constructed with gray interior,
  black outline with zero width, no size and style 'None'.
*/
QwtSymbol::QwtSymbol(): 
    d_brush(Qt::gray), 
    d_pen(Qt::black), 
    d_size(0,0),
    d_style(QwtSymbol::None)
{
}

/*!
  \brief Constructor
  \param style Symbol Style
  \param brush brush to fill the interior
  \param pen outline pen 
  \param size size
*/
QwtSymbol::QwtSymbol(QwtSymbol::Style style, const QBrush &brush, 
        const QPen &pen, const QSize &size): 
    d_brush(brush), 
    d_pen(pen), 
    d_size(size),
    d_style(style)
{
}

//! Destructor
QwtSymbol::~QwtSymbol()
{
}

/*!
  \brief Specify the symbol's size

  If the 'h' parameter is left out or less than 0,
  and the 'w' parameter is greater than or equal to 0,
  the symbol size will be set to (w,w).
  \param w width
  \param h height (defaults to -1)
*/
void QwtSymbol::setSize(int w, int h)
{
    if ((w >= 0) && (h < 0)) 
        h = w;
    d_size = QSize(w,h);
}

//! Set the symbol's size
void QwtSymbol::setSize(const QSize &s)
{
    if (s.isValid()) 
        d_size = s;
}

/*!
  \brief Assign a brush

  The brush is used to draw the interior of the symbol.
  \param br brush
*/
void QwtSymbol::setBrush(const QBrush &br)
{
    d_brush = br;
}

/*!
  \brief Assign a pen

  The pen is used to draw the symbol's outline.

  \param pn pen
*/
void QwtSymbol::setPen(const QPen &pn)
{
    d_pen = pn;
}

/*!
  \brief Draw the symbol at a point (x,y).
  \note The symbol size will be scaled by the device metrics
        of the paint device.
*/
void QwtSymbol::draw(QPainter *p, int x, int y) const
{
    // Implicit scaling to the device resolution of the paint device
    // Qt scales texts, fill patterns, implicitely, others like
    // rects not. We like to behave symbols like texts.

    QPaintDeviceMetrics m1(QApplication::desktop());
    QPaintDeviceMetrics m2(p->device());

    const double scaleMetricsX = 
        double(m2.logicalDpiX()) / double(m1.logicalDpiX());
    const double scaleMetricsY = 
        double(m2.logicalDpiY()) / double(m1.logicalDpiY());

    int w = (int)(d_size.width() * scaleMetricsX);
    int h = (int)(d_size.height() * scaleMetricsY);

    draw(p, QRect(x - w/2 , y - h/2, w, h));
}


/*!
  \brief Draw the symbol into a bounding rectangle.

  This function overrides the symbol's size settings,
  but it doesn't modify them.
  \param p Painter
  \param r Bounding rectangle

  \note The symbol size won't be scaled by the device metrics
*/
void QwtSymbol::draw(QPainter *p, const QRect& r) const
{
    const int w2 = r.width() / 2;
    const int h2 = r.height() / 2;

    p->setBrush(d_brush);
    p->setPen(d_pen);
    
    switch(d_style)
    {
        case QwtSymbol::Ellipse:
            QwtPainter::drawEllipse(p, r);
            break;
        case QwtSymbol::Rect:
            QwtPainter::drawRect(p, r);
            break;
        case QwtSymbol::Diamond:
        {
            QPointArray pa(4);
            pa.setPoint(0, r.x() + w2, r.y());
            pa.setPoint(1, r.right(), r.y() + h2);
            pa.setPoint(2, r.x() + w2, r.bottom());
            pa.setPoint(3, r.x(), r.y() + h2);
            QwtPainter::drawPolygon(p, pa);
            break;
        }
        case QwtSymbol::Cross:
            QwtPainter::drawLine(p, r.x() + w2, r.y(), r.x() + w2, r.bottom());
            QwtPainter::drawLine(p, r.x(), r.y() + h2, r.right(), r.y() + h2);
            break;
        case QwtSymbol::XCross:
            QwtPainter::drawLine(p, r.x(), r.y(), r.right(), r.bottom());
            QwtPainter::drawLine(p, r.x(), r.bottom(), r.right(), r.top());
            break;
        case QwtSymbol::Triangle:
        case QwtSymbol::UTriangle:
        {
            QPointArray pa(3);
            pa.setPoint(0, r.x() + w2, r.y());
            pa.setPoint(1, r.right(), r.bottom());
            pa.setPoint(2, r.x(), r.bottom());
            QwtPainter::drawPolygon(p, pa);
            break;
        }
        case QwtSymbol::DTriangle:
        {
            QPointArray pa(3);
            pa.setPoint(0, r.x(), r.y());
            pa.setPoint(1, r.right(), r.y());
            pa.setPoint(2, r.x() +  w2, r.bottom());
            QwtPainter::drawPolygon(p, pa);
            break;
        }
        case QwtSymbol::LTriangle:
        {
            QPointArray pa(3);
            pa.setPoint(0, r.x(), r.y());
            pa.setPoint(1, r.right(), r.y() + h2);
            pa.setPoint(2, r.x(), r.bottom());
            QwtPainter::drawPolygon(p, pa);
            break;
        }
        case QwtSymbol::RTriangle:
        {
            QPointArray pa(3);
            pa.setPoint(0, r.right(), r.y());
            pa.setPoint(1, r.x(), r.y() + h2);
            pa.setPoint(2, r.right(), r.bottom());
            QwtPainter::drawPolygon(p, pa);
            break;
        }
        default:;
    }
}

/*!
  \brief Draw the symbol at a specified point

  \param p painter
  \param pt point

  \note The symbol size will be scaled by the device metrics
        of the paint device.
*/
void QwtSymbol::draw(QPainter *p, const QPoint &pt) const
{
    draw(p, pt.x(), pt.y());
}

/*!
  \brief Specify the symbol style

  The following styles are defined:<dl>
  <dt>QwtSymbol::None<dd>No Style. The symbol cannot be drawn.
  <dt>QwtSymbol::Ellipse<dd>Ellipse or circle
  <dt>QwtSymbol::Rect<dd>Rectangle
  <dt>QwtSymbol::Diamond<dd>Diamond
  <dt>QwtSymbol::Triangle<dd>Triangle pointing upwards
  <dt>QwtSymbol::DTriangle<dd>Triangle pointing downwards
  <dt>QwtSymbol::UTriangle<dd>Triangle pointing upwards
  <dt>QwtSymbol::LTriangle<dd>Triangle pointing left
  <dt>QwtSymbol::RTriangle<dd>Triangle pointing right
  <dt>QwtSymbol::Cross<dd>Cross
  <dt>QwtSymbol::XCross<dd>Diagonal cross</dl>
  \param s style
*/
void QwtSymbol::setStyle(QwtSymbol::Style s)
{
    d_style = s;
}

//! == operator
bool QwtSymbol::operator==(const QwtSymbol &other) const
{
    return brush() == other.brush() && pen() == other.pen()
            && style() == other.style() && size() == other.size();
}

//! != operator
bool QwtSymbol::operator!=(const QwtSymbol &other) const
{
    return !(*this == other);
}
