/* -*- mode: C++ ; c-file-style: "stroustrup" -*- *****************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 * Copyright (C) 2002   Uwe Rathmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the Qwt License, Version 1.0
 *****************************************************************************/

#include <qpainter.h>
#include <qstyle.h>
#include <qpixmap.h>
#include <qdrawutil.h>
#include "qwt_math.h"
#include "qwt_paint_buffer.h"
#include "qwt_thermo.h"

//! Constructor
QwtThermo::QwtThermo(QWidget *parent, const char *name): 
    QWidget(parent, name, WRepaintNoErase|WResizeNoErase)
{
    init();
}

//! Destructor
QwtThermo::~QwtThermo()
{
}

//! initialize data members
void QwtThermo::init()
{
    // initialize data members
    d_orient = Qt::Vertical;
    d_scalePos = Left;
    d_scaleDist = 3;
    d_thermoWidth = 10;
    d_borderWidth = 2;
    d_maxValue = 1.0;
    d_minValue = 0.0;
    d_value = 0.0;
    d_alarmLevel = 0.0;
    d_alarmEnabled = 0;

    // initialize colors;
    d_fillColor = black;
    d_alarmColor = white;

    // initialize scales
    d_map.setDblRange(d_minValue, d_maxValue);
    d_scaleDraw->setScale(d_minValue, d_maxValue, d_maxMajor, d_maxMinor);
}


//! Set the current value
void QwtThermo::setValue(double v)
{
    if (d_value != v)
    {
        d_value = v;
        update();
    }
}

//! Qt paint event
void QwtThermo::paintEvent(QPaintEvent *e)
{
    // Use double-buffering
    const QRect &ur = e->rect();
    if ( ur.isValid() )
    {
		QwtPaintBuffer paintBuffer(this, ur);
        draw(paintBuffer.painter(), ur);
    }
}

//! redraw the thermo
void QwtThermo::draw(QPainter *p, const QRect& ur)
{
    if ( !d_thermoRect.contains(ur) )
    {
        if (d_scalePos != None)
            d_scaleDraw->draw(p);

        qDrawShadePanel(p,
            d_thermoRect.x() - d_borderWidth,
            d_thermoRect.y() - d_borderWidth,
            d_thermoRect.width() + 2*d_borderWidth,
            d_thermoRect.height() + 2*d_borderWidth,
            colorGroup(), TRUE, d_borderWidth,0);
    }
    drawThermo(p);
}

//! Qt resize event handler
void QwtThermo::resizeEvent(QResizeEvent *)
{
    layoutThermo( FALSE );
}

//! Recalculate the thermo's geometry and layout based on
//  the current rect and fonts.
//  \param update_geometry notify the layout system and call update
//         to redraw the scale
void QwtThermo::layoutThermo( bool update_geometry )
{
    QRect r = this->rect();
    int mbd = 0;
    if ( d_scalePos != None )
	{
		int d1, d2;
        d_scaleDraw->minBorderDist(fontMetrics(), d1, d2);
        mbd = QMAX(d1, d2);
	}

    if ( d_orient == Qt::Horizontal )
    {
        switch ( d_scalePos )
        {
            case None:
                d_thermoRect.setRect(r.x() + d_borderWidth,
                    r.y() + d_borderWidth,
                    r.width() - 2*d_borderWidth,
                    r.height() - 2*d_borderWidth);
                break;

            case Bottom:
                d_thermoRect.setRect(r.x() + mbd + d_borderWidth,
                    r.y() + d_borderWidth,
                    r.width() - 2*(d_borderWidth + mbd),
                    d_thermoWidth);
                d_scaleDraw->setGeometry(d_thermoRect.x(),
                    d_thermoRect.y() + d_thermoRect.height()
                    + d_borderWidth + d_scaleDist,
                    d_thermoRect.width(),
                    QwtScaleDraw::Bottom);
                break;

            case Top:
            default:
                d_thermoRect.setRect(r.x() + mbd + d_borderWidth,
                    r.y() + r.height()
                    - d_thermoWidth - 2*d_borderWidth,
                    r.width() - 2*(d_borderWidth + mbd),
                    d_thermoWidth);
                d_scaleDraw->setGeometry(d_thermoRect.x(),
                    d_thermoRect.y() - d_borderWidth - d_scaleDist,
                    d_thermoRect.width(),
                    QwtScaleDraw::Top);
                break;
        }
        d_map.setIntRange(d_thermoRect.x(),
        d_thermoRect.x() + d_thermoRect.width() - 1);
    }
    else // Qt::Vertical
    {
        switch ( d_scalePos )
        {
            case None:
                d_thermoRect.setRect(r.x() + d_borderWidth,
                    r.y() + d_borderWidth,
                    r.width() - 2*d_borderWidth,
                    r.height() - 2*d_borderWidth);
                break;

            case Left:
                d_thermoRect.setRect(r.x() +  r.width()
                    - 2*d_borderWidth - d_thermoWidth,
                    r.y() + mbd + d_borderWidth,
                    d_thermoWidth,
                    r.height() - 2*(d_borderWidth + mbd));
                d_scaleDraw->setGeometry(d_thermoRect.x() - 
                    d_scaleDist - d_borderWidth,
                    d_thermoRect.y(),
                    d_thermoRect.height(),
                    QwtScaleDraw::Left);
                break;

            case Right:
            default:
                d_thermoRect.setRect(r.x() + d_borderWidth,
                    r.y() + mbd + d_borderWidth,
                    d_thermoWidth,
                    r.height() - 2*(d_borderWidth + mbd));
                d_scaleDraw->setGeometry(d_thermoRect.x() + d_thermoRect.width()
                    + d_borderWidth + d_scaleDist,
                    d_thermoRect.y(),
                    d_thermoRect.height(),
                    QwtScaleDraw::Right);
                break;
        }
        d_map.setIntRange( d_thermoRect.y() + d_thermoRect.height() - 1,
            d_thermoRect.y());
    }
    if ( update_geometry )
    {
        updateGeometry();
        update();
    }
}

/*!
  \brief Change the thermometer's orientation

  The scale position None disables the scale.
  \param o orientation. Possible values are Qt::Horizontal and Qt::Vertical.
         The default value is Qt::Vertical.
  \param s Position of the scale. For a horizontal
         orientation, the scale position can
     be Top, Bottom or None. A vertical
     thermometer may have the scale positions
     Left, Right or None. The default is None.
*/
void QwtThermo::setOrientation(Qt::Orientation o, ScalePos s)
{
    switch(o)
    {
        case Qt::Horizontal:
            d_orient = Qt::Horizontal;
            if ((s == None) || (s == Bottom) || (s == Top))
                d_scalePos = s;
            else
                d_scalePos = None;
            break;

        case Qt::Vertical:
            d_orient = Qt::Vertical;
            if ((s == None) || (s == Left) || (s == Right))
                d_scalePos = s;
            else
                d_scalePos = None;
            break;
    }
    layoutThermo();
}

//! notify changed font
void QwtThermo::fontChange(const QFont &f)
{
    QWidget::fontChange( f );
    layoutThermo();
}

//!notify changed scale
void QwtThermo::scaleChange()
{
    update();
    layoutThermo();
}

//! Redraw the thermometer
void QwtThermo::drawThermo(QPainter *p)
{
    int alarm  = 0, taval = 0;

    QRect fRect;
    QRect aRect;
    QRect bRect;

    int inverted = ( d_maxValue < d_minValue );

    //
    //  Determine if value exceeds alarm threshold.
    //  Note: The alarm value is allowed to lie
    //        outside the interval (minValue, maxValue).
    //
    if (d_alarmEnabled)
    {
        if (inverted)
        {
            alarm = ((d_alarmLevel >= d_maxValue)
                 && (d_alarmLevel <= d_minValue)
                 && (d_value >= d_alarmLevel));
        
        }
        else
        {
            alarm = (( d_alarmLevel >= d_minValue)
                 && (d_alarmLevel <= d_maxValue)
                 && (d_value >= d_alarmLevel));
        }
    }

    //
    //  transform values
    //
    int tval = d_map.limTransform(d_value);

    if (alarm)
       taval = d_map.limTransform(d_alarmLevel);

    //
    //  calculate recangles
    //
    if ( d_orient == Qt::Horizontal )
    {
        if (inverted)
        {
            bRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
                  tval - d_thermoRect.x(),
                  d_thermoRect.height());
        
            if (alarm)
            {
                aRect.setRect(tval, d_thermoRect.y(),
                      taval - tval + 1,
                      d_thermoRect.height());
                fRect.setRect(taval + 1, d_thermoRect.y(),
                      d_thermoRect.x() + d_thermoRect.width() - (taval + 1),
                      d_thermoRect.height());
            }
            else
            {
                fRect.setRect(tval, d_thermoRect.y(),
                      d_thermoRect.x() + d_thermoRect.width() - tval,
                      d_thermoRect.height());
            }
        }
        else
        {
            bRect.setRect(tval + 1, d_thermoRect.y(),
                  d_thermoRect.width() - (tval + 1 - d_thermoRect.x()),
                  d_thermoRect.height());
        
            if (alarm)
            {
                aRect.setRect(taval, d_thermoRect.y(),
                      tval - taval + 1,
                      d_thermoRect.height());
                fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
                      taval - d_thermoRect.x(),
                      d_thermoRect.height());
            }
            else
            {
                fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
                      tval - d_thermoRect.x() + 1,
                      d_thermoRect.height());
            }
        
        }
    }
    else // Qt::Vertical
    {
        if (tval < d_thermoRect.y())
            tval = d_thermoRect.y();
        else 
        {
            if (tval > d_thermoRect.y() + d_thermoRect.height())
                tval = d_thermoRect.y() + d_thermoRect.height();
        }

        if (inverted)
        {
            bRect.setRect(d_thermoRect.x(), tval + 1,
            d_thermoRect.width(),
            d_thermoRect.height() - (tval + 1 - d_thermoRect.y()));

            if (alarm)
            {
                aRect.setRect(d_thermoRect.x(), taval,
                    d_thermoRect.width(),
                    tval - taval + 1);
                fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
                    d_thermoRect.width(),
                taval - d_thermoRect.y());
            }
            else
            {
                fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
                    d_thermoRect.width(),
                    tval - d_thermoRect.y() + 1);
            }
        }
        else
        {
            bRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
            d_thermoRect.width(),
            tval - d_thermoRect.y());
            if (alarm)
            {
                aRect.setRect(d_thermoRect.x(),tval,
                    d_thermoRect.width(),
                    taval - tval + 1);
                fRect.setRect(d_thermoRect.x(),taval + 1,
                    d_thermoRect.width(),
                    d_thermoRect.y() + d_thermoRect.height() - (taval + 1));
            }
            else
            {
                fRect.setRect(d_thermoRect.x(),tval,
                    d_thermoRect.width(),
                d_thermoRect.y() + d_thermoRect.height() - tval);
            }
        }
    }

    //
    // paint thermometer
    //
    p->fillRect(bRect, colorGroup().color(QColorGroup::Background));

    if (alarm)
       p->fillRect(aRect, d_alarmColor);

    p->fillRect(fRect, d_fillColor);
}

//! Set the border width of the pipe.
void QwtThermo::setBorderWidth(int w)
{
    if ((w >= 0) && (w < (qwtMin(d_thermoRect.width(), 
        d_thermoRect.height()) + d_borderWidth) / 2  - 1))
    {
        d_borderWidth = w;
        layoutThermo();
    }
}

/*!
  \brief Set the range
  \param vmin Value corresponding lower or left end of the thermometer
  \param vmax Value corresponding to the upper or right end of the thermometer
*/
void QwtThermo::setRange(double vmin, double vmax)
{
    d_minValue = vmin;
    d_maxValue = vmax;

    d_map.setDblRange(d_minValue, d_maxValue);
    if (!hasUserScale())
    {
        QwtScaleDiv oldscl(d_scaleDraw->scaleDiv());

        d_scaleDraw->setScale(d_minValue, d_maxValue, d_maxMajor, d_maxMinor);
        if (oldscl != d_scaleDraw->scaleDiv())
            scaleChange();
    }
    layoutThermo();
}

/*!
  \brief Change the color of the liquid.
  \param c New color. The default color is black.
*/
void QwtThermo::setFillColor(const QColor &c)
{
    d_fillColor = c;
    update();
}

/*!
  \brief Specify liquid color above the alarm threshold
  \param c New color. The default is white.
*/
void QwtThermo::setAlarmColor(const QColor &c)
{
    d_alarmColor = c;
    update();
}

//! Specify the alarm threshold
void QwtThermo::setAlarmLevel(double v)
{
    d_alarmLevel = v;
    d_alarmEnabled = 1;
    update();
}

//! Change the width of the pipe
void QwtThermo::setPipeWidth(int w)
{
    if (w > 0)
    {
        d_thermoWidth = w;
        layoutThermo();
    }
}


/*!
  \brief Specify the distance between the pipe's endpoints
         and the widget's border

  The margin is used to leave some space for the scale
  labels. If a large font is used, it is advisable to
  adjust the margins.
  \param m New Margin. The default values are 10 for
           horizontal orientation and 20 for vertical
           orientation.
  \warning The margin has no effect if the scale is disabled.
  \warning This function is a NOOP because margins are determined
           automatically.
*/
void QwtThermo::setMargin(int)
{
}


/*!
  \brief Enable or disable alarm threshold
  \param tf \c ZERO (disabled) or non-zero (enabled)
*/
void QwtThermo::setAlarmEnabled(int tf)
{
    d_alarmEnabled = tf;
    update();
}

/*!
  \return Fixed/MinimumExpanding for vertical, 
          MinimumExpanding/Fixed for horizontal thermos.
*/ 

QSizePolicy QwtThermo::sizePolicy() const
{
    QSizePolicy sp;
    if ( d_scaleDraw->orientation() == QwtScaleDraw::Left ||
        d_scaleDraw->orientation() == QwtScaleDraw::Right )
    {
        sp.setHorData( QSizePolicy::Fixed );
        sp.setVerData( QSizePolicy::MinimumExpanding );
    }
    else
    {
        sp.setHorData( QSizePolicy::MinimumExpanding );
        sp.setVerData( QSizePolicy::Fixed );
    }
    return sp;
}

/*!
  \return the minimum size hint
  \sa QwtThermo::minimumSizeHint
*/
QSize QwtThermo::sizeHint() const
{
    return minimumSizeHint();
}

/*!
  \brief Return a minimum size hint
  \warning The return value depends on the font and the scale.
  \sa QwtThermo::sizeHint
*/
QSize QwtThermo::minimumSizeHint() const
{
    int w = 0, h = 0;

    if ( d_scalePos != None )
    {
        int smw = d_scaleDraw->minWidth( QPen(), fontMetrics() );
        int smh = d_scaleDraw->minHeight( QPen(), fontMetrics() );

        if ( d_orient == Qt::Vertical )
        {
            w = d_thermoWidth + smw + 3 * d_borderWidth + d_scaleDist;
            h = smh + 2 * d_borderWidth;
        }
        else
        {
            w = smw + 2 * d_borderWidth;
            h = d_thermoWidth + smh + 3 * d_borderWidth + d_scaleDist;
        }

    }
    else // no scale
    {
        if ( d_orient == Qt::Vertical )
        {
            w = d_thermoWidth + 2 * d_borderWidth;
            h = 200 + 2 * d_borderWidth;
        }
        else
        {
            w = 200 + 2 * d_borderWidth;
            h = d_thermoWidth + 2 * d_borderWidth;
        }
    }
    return QSize( w, h );
}
