/***************************************************************************
                          QDrawWidget.cpp  -  description
                             -------------------
    begin                : Fri Nov 17 2000
    copyright            : (C) 2000 by Rainer Lehrig
    email                : lehrig@t-online.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "opt.h"
#include "qdrawwidget.h"
#include "tcputil.h"
#include <QSvgRenderer>

extern OPT opt;

QDrawWidget::QDrawWidget( QWidget *parent, const char *name, int wFlags, int *sock, int ident)
            : QWidget( parent, name, (Qt::WFlags) wFlags )
{
  fp   = NULL;
  flog = NULL;
  s  = sock;
  id = ident;
  zoomx = zoomy = 1.0f;
  linestyle = LINESTYLE_NONE;
  boxx = boxy = boxw = boxh = 0;
  xmin = 0.0f;
  xmax = 100.0f;
  ymin = 0.0f;
  ymax = 100.0f;
  dx = dy = 1.0f;
  br = 255;
  bg = 255;
  bb = 255;
  sr = 3;
  xold = yold = 0;
  setBackgroundMode(Qt::NoBackground);
  strcpy(floatFormat,"%f");
}

QDrawWidget::~QDrawWidget()
{
  if(fp != NULL) fclose(fp);
}

void QDrawWidget::resize(int w, int h)
{
  QWidget::resize(w,h);
  buffer.resize(w,h);
  buffer.fill(QColor(br,bg,bb));
}

void QDrawWidget::setGeometry(int x, int y, int w, int h)
{
  QWidget::setGeometry(x,y,w,h);
}

void QDrawWidget::setGeometry(const QRect &r)
{
  setGeometry(r.left(),r.top(),r.width(),r.height());
}

void QDrawWidget::setBackgroundColor(int r, int g, int b)
{
  br = r;
  bg = g;
  bb = b;
  buffer.fill(QColor(r,g,b));
}

void QDrawWidget::beginDraw()
{
  p.begin(&buffer);
  buffer.fill(QColor(br,bg,bb));
  fontsize = p.fontInfo().pointSize();
  fontsize = (zx(fontsize)+zy(fontsize)) / 2;
}

void QDrawWidget::endDraw()
{
  if(fp != NULL)
  {
    p.end();
    fclose(fp);
    fp = NULL;
    return;
  }
  p.end();
  repaint();
}

void QDrawWidget::showFromSocket(int *sock)
{
  s = sock;
  beginDraw();
  while(interpretall() != -1);
  //*s = -1;
}

void QDrawWidget::showFromFile(const char *filename)
{
  fp = fopen(filename,"r");
  if(fp == NULL) return;
  beginDraw();
  while(interpretall() != -1);
}

void QDrawWidget::print()
{
  QPrinter printer;
  if(printer.setup(this))
  {
    QPainter painter;
    painter.begin(&printer);
    painter.drawPixmap(0,0,buffer);
    painter.end();
  };
}

void QDrawWidget::logBmpToFile(const char *filename)
{
  buffer.save(filename,"BMP");
}

void QDrawWidget::logToFile(const char *filename)
{
  if(flog != NULL) return;
  flog = fopen(filename,"w");
  if(flog != NULL)
  {
    fprintf(flog,"pvMetafile(%d,%d)\n",width(),height());
  }
}

void QDrawWidget::getDimensions(const char *filename, int *width, int *height)
{
  FILE *f;
  char line[1024],*cptr;

  *width = *height = 0;
  f = fopen(filename,"r");
  if(f == NULL) return;
  cptr = fgets(line,sizeof(line)-1,f);
  if(cptr != NULL && strncmp(line,"pvMetafile",10) == 0)
  {
    sscanf(line,"pvMetafile(%d,%d)",width,height);
  }
  fclose(f);
}

void QDrawWidget::copyToClipboard()
{
  QApplication::clipboard()->setPixmap(buffer);
}

void QDrawWidget::moveTo(int x, int y)
{
  xold = zx(x);
  yold = zy(y);
  //p.moveTo(zx(x),zy(y));
}

void QDrawWidget::lineTo(int x, int y)
{
  int xnew, ynew;
  xnew = zx(x);
  ynew = zy(y);
  p.drawLine(xold,yold,xnew,ynew);
  xold = xnew;
  yold = ynew;
  //p.lineTo(zx(x),zy(y));
}

void QDrawWidget::textInAxis(float x, float y, int alignment, const char *txt)
{
  float dy;
/*
  ymax-ymin        dy
  -----------   =  ----------
  boxh             fontsize/2
*/
  if(alignment == ALIGN_CENTER)
  {
    dy = ((ymax-ymin) * (fontsize/2) ) / (boxh);
    text(tx(x), ty(y+dy), alignment, txt);
  }
  else
  {
    text(tx(x), ty(y), alignment, txt);
  }
}
  
void QDrawWidget::text(int x, int y, int alignment, const char *txt)
{
int len;
        
  len = strlen(txt);
  switch(alignment)
  {
    case ALIGN_LEFT:     //  |example
      p.drawText(zx(x),zy(y-fontsize/2),zx(len*fontsize),(zx(fontsize)+zy(fontsize)),Qt::AlignLeft,txt);
      break;
    case ALIGN_CENTER:   //  exa|mple
      p.drawText(zx(x-(len*fontsize)/2),zy(y-fontsize/2),zx(len*fontsize),(zx(fontsize)+zy(fontsize)),Qt::AlignCenter,txt);
      break;
    case ALIGN_RIGHT:    //  example|
      p.drawText(zx(x-len*fontsize),zy(y-fontsize/2),zx(len*fontsize),(zx(fontsize)+zy(fontsize)),Qt::AlignRight,txt);
      break;
    case ALIGN_VERT_CENTER:
      {
//#ifndef unix
        QMatrix m;
        QMatrix mident;
        m.translate(zx(x),zy(y));
        m.rotate(-90.0F);

        p.setWorldMatrix(m);
        p.drawText(-(len*fontsize*29)/100,-fontsize/2,zx(len*fontsize),(zx(fontsize)+zy(fontsize)),Qt::AlignLeft,txt);
        p.setWorldMatrix(mident);
/*
#else
        int i,dy;
        dy = y-(len*((12*fontsize)/10))/2;
        for(i=0; i<len; i++)
        {
          p.drawText(zx(x),zy(dy),zx(fontsize*2),zy(fontsize*2),AlignLeft,&txt[i],1);
          dy += (12*fontsize)/10;
        }
#endif
*/
      }
      break;
    default:
      break;
  }
}

void QDrawWidget::box(int x, int y, int w, int h)
{
  boxx = x;
  boxy = y;
  boxw = w;
  boxh = h;
  moveTo(x  ,y  );
  lineTo(x+w,y  );
  lineTo(x+w,y+h);
  lineTo(x  ,y+h);
  lineTo(x  ,y  );
}

void QDrawWidget::rect(int x, int y, int w, int h)
{
  p.drawRect(zx(x),zy(y),zx(w),zy(h));
}

static int beautyval(char *text)
{
  int i,val;

  i = strlen(text);
  if(i>0)
  {
    if(text[i-1] == '9' && i>2 && text[i-2] == '9' && strchr(text,'.') != NULL ) // round up
    {
      i--;

      while(i>0)
      {
        if     (text[i] == '9') text[i] = '\0';
        else if(text[i] == '.')
        {
          sscanf(text,"%d",&val);
          val += 1;
          sprintf(&text[i-1],"%d",val);
          return 0;
        }
        else
        {
          sscanf(&text[i],"%d",&val);
          val += 1;
          sprintf(&text[i],"%d",val);



          return 0;
        }
      }
      return 0;
    }
    else if(text[i-1] == '1' && i>2 && text[i-2] == '0' && strchr(text,'.') != NULL ) // round down
    {
      i--;
      text[i] = '\0';
      i--;
      while(i>0)
      {
        if     (text[i] == '0') text[i] = '\0';
        else if(text[i] == '.')
        {
          text[i] = '\0';
          return 0;
        }
        else
        {
          return 0;
        }
      }
      return 0;
    }
    
    while(i>0)
    {
      i--;
      if(text[i] == '0') text[i] = '\0';
      else if(text[i] == '.')
      {
        text[i] = '\0';
        return 0;
      }
      else return 0;
    }
  }
  return 0;
}

void QDrawWidget::xGrid()
{
  for(float x=(xmin+dx); x<xmax; x+=dx)
  {
    xold = zx(tx(x));
    yold = zy(boxy+boxh);
    p.drawLine(zx(tx(x)),zy(boxy),xold,yold);
    //p.moveTo(zx(tx(x)),zy(boxy));
    //p.lineTo(zx(tx(x)),zy(boxy+boxh));
  }
}

void QDrawWidget::yGrid()
{
  for(float y=(ymin+dy); y<=(ymax-dy); y+=dy)
  {
    xold = zx(boxx+boxw);
    yold = zy(ty(y));
    p.drawLine(zx(boxx),zy(ty(y)),xold,yold);
    //p.moveTo(zx(boxx),     zy(ty(y)));
    //p.lineTo(zx(boxx+boxw),zy(ty(y)));
  }
}

void QDrawWidget::xAxis(float start, float delta, float end, int draw)
{
  float x;
  int   y;
  char  txt[80];

  xmin = start;
  dx   = delta;
  xmax = end;
  if(draw==0) return;

  x = start;
  y = boxy + boxh;

  while(x <= end)
  {
    moveTo(tx(x),y);
    lineTo(tx(x),y+4);
    sprintf(txt,floatFormat,x);
    if(floatFormat[1] == 'f') beautyval(txt);
    text(tx(x),y+8,ALIGN_CENTER,txt);
    x += delta;
  }

  if(start < 0.0f && end > 0.0f)
  {
    moveTo(tx(0.0f),boxy);
    lineTo(tx(0.0f),boxy+boxh);
  }
}

void QDrawWidget::yAxis(float start, float delta, float end, int draw)
{

float y;
int   x;
char  txt[80];

  ymin = start;
  dy   = delta;
  ymax = end;
  if(draw==0) return;
  x = boxx;
  y = start;

  while(y <= end)
  {
    moveTo(x,ty(y));
    lineTo(x-4,ty(y));
    sprintf(txt,floatFormat,y);
    if(floatFormat[1] == 'f') beautyval(txt);
    text(x-8,ty(y),ALIGN_RIGHT,txt);
    y += delta;
  }


  if(start < 0.0f && end > 0.0f)
  {
    moveTo(boxx,ty(0.0f));
    lineTo(boxx+boxw,ty(0.0f));
  }
}

void QDrawWidget::rightYAxis(float start, float delta, float end, int draw)
{
float y;
int   x;
char  txt[80];

  ymin = start;
  dy   = delta;
  ymax = end;
  if(draw==0) return;
  x = boxx+boxw;
  y = start;

  while(y <= end)
  {
    xold = x+4;
    yold = ty(y);
    p.drawLine(x,ty(y),xold,yold);
    //p.moveTo(x,ty(y));
    //p.lineTo(x+4,ty(y));
    sprintf(txt,floatFormat,y);
    if(floatFormat[1] == 'f') beautyval(txt);
    text(x+8,ty(y),ALIGN_LEFT,txt);
    y += delta;
  }

  if(start < 0.0f && end > 0.0f)
  {
    int xnew, ynew;
    xold = boxx+boxw;
    yold = ty(0.0f);
    xnew = boxx+boxw;
    ynew = ty(0.0f);
    p.drawLine(xold,yold,xnew,ynew);
    xold = xnew;
    yold = ynew;
    //p.moveTo(boxx,ty(0.0f));
    //p.lineTo(boxx+boxw,ty(0.0f));
  }
}

void QDrawWidget::drawSymbol(int x, int y, int type)
{
  int xnew, ynew;
  switch(type)
  {
    case LINESTYLE_CIRCLE:
      p.drawArc(zx(x-sr),zy(y-sr),zx(2*sr),zy(2*sr),0,360);
      break;

    case LINESTYLE_CROSS:
      xold = zx(x+sr);
      yold = zy(y-sr);
      p.drawLine(zx(x-sr),zy(y+sr),xold,yold);
      xold = zx(x+sr);
      yold = zy(y+sr);
      p.drawLine(zx(x-sr),zy(y-sr),xold,yold);
      //p.moveTo(zx(x-sr),zy(y+sr));
      //p.lineTo(zx(x+sr),zy(y-sr));
      //p.moveTo(zx(x-sr),zy(y-sr));
      //p.lineTo(zx(x+sr),zy(y+sr));
      break;
    case LINESTYLE_RECT:
      xold = zx(x+sr);
      yold = zy(y-sr);
      p.drawLine(zx(x-sr),zy(y-sr),xold,yold);
      xnew = zx(x+sr);
      ynew = zy(y+sr);
      p.drawLine(xold,yold,xnew,ynew);
      xold = xnew;
      yold = ynew;
      xnew = zx(x-sr);
      ynew = zy(y+sr);
      p.drawLine(xold,yold,xnew,ynew);
      xold = xnew;
      yold = ynew;
      xnew = zx(x-sr);
      ynew = zy(y-sr);
      p.drawLine(xold,yold,xnew,ynew);
      xold = xnew;
      yold = ynew;
      //p.moveTo(zx(x-sr),zy(y-sr));
      //p.lineTo(zx(x+sr),zy(y-sr));
      //p.lineTo(zx(x+sr),zy(y+sr));
      //p.lineTo(zx(x-sr),zy(y+sr));
      //p.lineTo(zx(x-sr),zy(y-sr));
      break;
    default:
      break;
  }
}

void QDrawWidget::line(float *x, float *y, int n)
{
  moveTo(tx(x[0]),ty(y[0]));
  for(int i=1; i<n; i++) lineTo(tx(x[i]),ty(y[i]));
  if(linestyle != LINESTYLE_NONE)
  {
    for(int i=1; i<n; i++) drawSymbol(tx(x[i]),ty(y[i]),linestyle);
  }
}

void QDrawWidget::setColor(int r, int g, int b)
{
  p.setBrush(QColor(r,g,b));
  p.setPen(QColor(r,g,b));
}

void QDrawWidget::setFont(const char *family, int size, int weight, int italic)
{
  size = (int) (((size * zoomx) + (size * zoomy)) / 2);
  fontsize = size;
  p.setFont(QFont(family,size,weight,italic));
}

void QDrawWidget::setLinestyle(int style)
{
  linestyle = style;
}

void QDrawWidget::paintEvent( QPaintEvent *e)
{
  if(e != NULL)
    bitBlt(this,0,0,&buffer);
}

void QDrawWidget::setZoomX(float zx)
{
  zoomx = zx;
}

void QDrawWidget::setZoomY(float zy)
{
  zoomy = zy;
}

int QDrawWidget::tx(float x)
{
  return (int)(boxx + (((x-xmin)*boxw)/(xmax-xmin)));    // x          i
}                                                        // xmax-xmin  boxw

int QDrawWidget::ty(float y)
{
  int i0;
  i0 = (int)((ymin * boxh)/(ymax-ymin));
  return i0 + (int)(boxy+boxh-((y*boxh)/(ymax-ymin))); // y          i        ymin       i0
}                                                      // ymax-ymin  boxh     ymax-ymin  boxh

int QDrawWidget::zx(int x)
{

  return (int)(zoomx * ((float) x) );
}

int QDrawWidget::zy(int y)
{
  return (int)(zoomy * ((float) y) );
}

void QDrawWidget::playSVG(const char *filename)
{
  char buf[1024+1],*cptr;
  QByteArray stream;
  QSvgRenderer renderer(this);
  FILE *fin;
  int foundw,foundh,foundclose;

  if(opt.arg_debug) printf("Qt4 playSVG(%s)\n",filename);
  fin = fopen(filename,"r");
  if(fin == NULL) return;
  foundw = foundh = foundclose = 0;
  while(fgets(buf,sizeof(buf)-1,fin) != NULL)
  {
    if(opt.arg_debug) printf("PlaySVG::buf=%s",buf);
    if(foundw == 0 && (cptr = strstr(buf," width=")) != NULL)
    {
      if(strstr(buf,">") != NULL) foundclose = 1;
      sprintf(buf,"width=\"%dpx\"\n",width());
      if(foundclose) strcat(buf,">\n");
      foundw = 1;
    }
    if(foundh == 0 && (cptr = strstr(buf," height=")) != NULL)
    {
      if(strstr(buf,">") != NULL) foundclose = 1;
      sprintf(buf,"height=\"%dpx\"\n",height());
      if(foundclose) strcat(buf,">\n");
      foundh = 1;
    }
    stream.append(buf);
    if(opt.arg_debug > 2) printf("svgbuf=%s",buf);
  }
  fclose(fin);

  //renderer.setViewBox( QRect(0,0,width(),height()) );
  renderer.load(stream);
  p.scale(zoomx,zoomy);
  renderer.render(&p);
  p.scale(1.0,1.0);
}

void QDrawWidget::socketPlaySVG()
{
  char buf[1024+1];
  int ret,foundw,foundh;
  QByteArray stream;
  QSvgRenderer renderer(this);

  if(opt.arg_debug) printf("Qt4 socketPlaySVG\n");
  if(s == NULL) return;
  foundw = foundh = 0;
  while(1)
  {
    ret = tcp_rec(s,buf,sizeof(buf)-1);
    if(ret < 0) return;
    if(foundw == 0 && strncmp(buf,"width",5) == 0)
    {
      sprintf(buf,"width=\"%dpx\"\n",width());
      foundw = 1;
    }
    if(foundh == 0 && strncmp(buf,"height",6) == 0)
    {
      sprintf(buf,"height=\"%dpx\"\n",height());
      foundh = 1;
    }
    if(strstr(buf,"<svgend></svgend>") != NULL) break; 
    stream.append(buf);
    if(opt.arg_debug > 2) printf("svgbuf=%s",buf);
  }
  renderer.load(stream);
  p.scale(zoomx,zoomy);
  renderer.render(&p);
  p.scale(1.0,1.0);
}

int QDrawWidget::getText(const char *line, char *text)
{
  *text = '\0';
  while(*line != '"' && *line >= ' ') line++;
  if(*line < ' ') return 0;
  line++;
  while(*line != '"' && *line >= ' ') *text++ = *line++;
  *text = '\0';
  return 0;
}

int QDrawWidget::gets(char *line, int len)
{
  if(s != NULL)
  {
    if(tcp_rec(s,line,len) == -1) return -1;
    if(flog != NULL)
    {
      fputs(line,flog);
      if(strncmp(line,"gendDraw",8) == 0)
      {
        fclose(flog);
        flog = NULL;
      }
    }
    return 0;
  }
  if(fp != NULL)
  {
    if(fgets(line,len,fp) == NULL) return -1;
    return 0;
  }
  return -1;
}

int QDrawWidget::interpretall()
{
char linebuf[1024];

  if(gets(linebuf,sizeof(linebuf)-1) == -1) { endDraw(); return -1; }
  return interpret(linebuf);

}

int QDrawWidget::interpret(const char *linebuf)
{
int x,y,w,h,r,g,b,n,i;

  if(opt.arg_debug > 1) printf("QDrawWidget=%s",linebuf);
  switch(linebuf[1])
  {
    case 'b':
      if(strncmp(linebuf,"gbeginDraw",10) == 0)
      {
        strcpy(floatFormat,"%f");
        beginDraw();
      }
      else if(strncmp(linebuf,"gbox",4) == 0)
      {
        sscanf(linebuf,"gbox(%d,%d,%d,%d)",&x,&y,&w,&h);
        box(x,y,w,h);
      }
      break;
    case 'd':
      if(strncmp(linebuf,"gdrawArc",8) == 0)
      {
        int astart,alen;
        sscanf(linebuf,"gdrawArc(%d,%d,%d,%d,%d,%d)",&x,&y,&w,&h,&astart,&alen);
        p.drawArc(zx(x),zy(y),zx(w),zy(h),astart*16,alen*16);
      }
      else if(strncmp(linebuf,"gdrawPie",8) == 0)
      {
        int astart,alen;
        sscanf(linebuf,"gdrawPie(%d,%d,%d,%d,%d,%d)",&x,&y,&w,&h,&astart,&alen);
        p.drawPie(zx(x),zy(y),zx(w),zy(h),astart*16,alen*16);
      }
      else if(strncmp(linebuf,"gdrawPolygon",12) == 0)
      {
        char buf[80];
        int cnt;
        sscanf(linebuf,"gdrawPolygon(%d)",&cnt);
        QPolygon points(cnt);
        i=0;
        while(1)
        {
          gets(buf,sizeof(buf)-1);
          if(buf[0] == '}') break;
          sscanf(buf,"%d,%d",&x,&y);
          points.setPoint(i++,zx(x),zy(y));
        }
        p.drawPolygon(points);
      }
      break;
    case 'e':
      if(strncmp(linebuf,"gendDraw",8) == 0)
      {
        strcpy(floatFormat,"%f");
        endDraw();
        return -1;
      }
      break;
    case 'l':
      if(strncmp(linebuf,"glineTo",7) == 0)
      {
        sscanf(linebuf,"glineTo(%d,%d)",&x,&y);
        lineTo(x,y);
      }
      else if(strncmp(linebuf,"gline(",6) == 0)
      {
        float xa[MAXARRAY],ya[MAXARRAY];
        char buf[80];
        for(n=0; 1; n++)
        {
          if(gets(buf,sizeof(buf)-1) == -1) break;
          if(buf[0] == '}') break;
          if(n >= MAXARRAY) break;
          sscanf(buf,"%f,%f",&xa[n],&ya[n]);
        }
        if(n>0) line(xa,ya,n);
      }
      break;
    case 'm':
      if(strncmp(linebuf,"gmoveTo",7) == 0)
      {
        sscanf(linebuf,"gmoveTo(%d,%d)",&x,&y);
        moveTo(x,y);
      }
      break;
    case 'p':
      if(strncmp(linebuf,"gplaySVGsocket",14) == 0)
      {
        socketPlaySVG();
      }
      else if(strncmp(linebuf,"gplaySVG",8) == 0)
      {
        char filename[1024];
        getText(linebuf,filename);
        playSVG(filename);
      }
      break;
    case 'r':
      if(strncmp(linebuf,"grect",5) == 0)
      {
        sscanf(linebuf,"grect(%d,%d,%d,%d)",&x,&y,&w,&h);
        rect(x,y,w,h);
      }
      else if(strncmp(linebuf,"grightYAxis",11) == 0)
      {
        float start,delta,end;
        int draw;
        sscanf(linebuf,"grightYAxis(%f,%f,%f,%d)",&start,&delta,&end,&draw);
        rightYAxis(start,delta,end,draw);
      }
      else if(strncmp(linebuf,"grotate",7) == 0)
      {
        float rx;
        sscanf(linebuf,"grotate(%f)",&rx);
        p.rotate(rx);
      }
      break;
    case 's':
      if(strncmp(linebuf,"gsetBackgroundColor",19) == 0)
      {
        sscanf(linebuf,"gsetBackgroundColor(%d,%d,%d)",&r,&g,&b);
        setBackgroundColor(r,g,b);
      }
      else if(strncmp(linebuf,"gsetColor",9) == 0)
      {
        sscanf(linebuf,"gsetColor(%d,%d,%d)",&r,&g,&b);
        setColor(r,g,b);
      }
      else if(strncmp(linebuf,"gsetWidth",9) == 0)
      {
        QPen pen;
        sscanf(linebuf,"gsetWidth(%d)",&w);
        pen = p.pen();
        pen.setWidth(w);
        p.setPen(pen);
      }
      else if(strncmp(linebuf,"gsetStyle",9) == 0)
      {
        QPen pen;
        sscanf(linebuf,"gsetStyle(%d)",&i);
        pen = p.pen();
        if(i==0) pen.setStyle(Qt::SolidLine);
        if(i==1) pen.setStyle(Qt::DotLine);
        p.setPen(pen);
      }
      else if(strncmp(linebuf,"gsetFont",8) == 0)
      {
        char family[80],*cptr;
        int  size,weight,italic;
        getText(linebuf,family);
        cptr = strstr(linebuf,"\",");
        if(cptr != NULL)
        {
          cptr += 2;
          sscanf(cptr,"%d,%d,%d",&size,&weight,&italic);
          setFont(family,size,weight,italic);
        }
      }
      else if(strncmp(linebuf,"gsetLinestyle",13) == 0)
      {
        sscanf(linebuf,"gsetLinestyle(%d)",&i);
        setLinestyle(i);
      }
      else if(strncmp(linebuf,"gsetFloatFormat",15) == 0)
      {
        char txt[1024];
        getText(linebuf,txt);
        strcpy(floatFormat,txt);
      }
      else if(strncmp(linebuf,"gscale",6) == 0)
      {
        float rx,ry;
        sscanf(linebuf,"gscale(%f,%f)",&rx,&ry);
        p.scale(rx,ry);
      }
      break;
    case 't':
      if(strncmp(linebuf,"gtext(",6) == 0)
      {
        char txt[1024];
        int alignment;
        sscanf(linebuf,"gtext(%d,%d,%d,",&x,&y,&alignment);
        getText(linebuf,txt);
        text(x,y,alignment,txt);
      }
      else if(strncmp(linebuf,"gtextInAxis(",12) == 0)
      {
        char txt[1024];
        int alignment;
        float fx,fy;
        sscanf(linebuf,"gtextInAxis(%f,%f,%d,",&fx,&fy,&alignment);
        getText(linebuf,txt);
        textInAxis(fx,fy,alignment,txt);
      }
      else if(strncmp(linebuf,"gtranslate",10) == 0)
      {
        float rx,ry;
        sscanf(linebuf,"gtranslate(%f,%f)",&rx,&ry);
        p.translate(rx,ry);
      }
      break;
    case 'x':
      if(strncmp(linebuf,"gxAxis",6) == 0)
      {
        float start,delta,end;
        int draw;
        sscanf(linebuf,"gxAxis(%f,%f,%f,%d)",&start,&delta,&end,&draw);
        xAxis(start,delta,end,draw);
      }
      else if(strncmp(linebuf,"gxGrid",6) == 0)
      {
        xGrid();
      }
      break;
    case 'y':
      if(strncmp(linebuf,"gyAxis",6) == 0)
      {
        float start,delta,end;
        int draw;
        sscanf(linebuf,"gyAxis(%f,%f,%f,%d)",&start,&delta,&end,&draw);
        yAxis(start,delta,end,draw);
      }
      else if(strncmp(linebuf,"gyGrid",6) == 0)
      {
        yGrid();
      }
      break;
  }
  return 0;
}

