/****************************************************************************
**
** Copyright (C) 2000-2006 Lehrig Software Engineering.
**
** This file is part of the pvbrowser project.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include <stdlib.h>

#include "opt.h"
#include "interpreter.h"
#include "mainwindow.h"
#include "pvserver.h"

#include <qmovie.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qevent.h>
#include <qsizepolicy.h> 
#include <qlayout.h>
#include <qsound.h>
#include <QKeyEvent>
#include <QLabel>
#include <Q3PopupMenu>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>
#include <QPixmap>
#include <QDesktopWidget>
#ifdef _WIN32
#include <io.h>
#include <direct.h>
#else
#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#endif
#include <sys/stat.h>
#include <fcntl.h>
#include "tcputil.h"
#include <qlineedit.h>

#ifdef unix
#include <utime.h>
#endif

#ifdef _WIN32
#include <sys/utime.h>
#endif

#ifndef _WIN32
#define closesocket close
#endif

#define LEFT_PRINT_MARGIN  10
#define UPPER_PRINT_MARGIN 70

extern OPT opt;
extern QString l_print_header;

Interpreter::Interpreter()
{
  mainWindow = NULL;
  mainLayout = NULL;
  hasLayout = 0;
  timet = time(NULL);
  zoom = zoom_original = 100;
  allModal = allBase = NULL;
  allBase = allModal = 0;
  modalDialog = NULL;

#ifdef USE_OPEN_GL
  //QGLWidget gl;
  //gl.makeCurrent();
#endif

  s    = NULL;
  all  = NULL;
  nmax = 0;
  setPath(qApp->argv()[0]);
}

Interpreter::~Interpreter()
{
  if(all != NULL) free(all);
  if(s != NULL && *s != -1) tcp_close(s);
  if(s != NULL) *s = -1;
  all = NULL;
}

void Interpreter::registerMainWindow(MainWindow *mainwin)
{
  zoom = zoom_original = opt.zoom;
  mainWindow = mainwin;
  s = & mainWindow->s;
}

void Interpreter::slotModalTerminate()
{
  char buf[80];
  if(all != NULL)
  {
    free(all);
  }
  if(allBase != NULL) all = allBase;
  allBase = allModal = NULL;
  nmax = nmaxBase;
  sprintf(buf,"QPushButton(0)\n");
  tcp_send(s,buf,strlen(buf));
  modalDialog = NULL;
}

QImage *Interpreter::readBmpFromSocket(int w, int h, char **d)
{
  char *data,text[MAX_PRINTF_LENGTH];
  int y,ret,r,g,b,ii,icol;
  QImage *temp;
  QRgb rgb[4*256];

  if(w == -1 && h == -1) // load bmp from file (e.g. PHP server)
  {
    data = NULL;
    temp = new QImage();
    temp->load("temp.bmp");
    QRgb col;
    for(icol=0; icol < temp->numColors(); icol++)
    {
      col = temp->color(icol);
      if(qRed(col) == 1 && qGreen(col) == 1 && qBlue(col) == 1)
      {
        col = qRgba(1,1,1,0); // transparent color
        temp->setColor(icol, col);
      }
    }
    temp->setAlphaBuffer(true);
    return temp;
  }

  data = new char[w*h];

  if(data==NULL) qFatal("out of memory -> exit");
  for(y=0; y<h; y++) // get image
  {
    ret = tcp_rec_binary(s, data+(y*w), w);
    if(ret == -1) { delete data; return NULL; }
  }

  for(icol=0; 1; icol++) // get color table
  {
#ifdef _WIN32
retry:
#endif
    ret = tcp_rec(s, text, sizeof(text)-1);
    if(ret == -1) { delete data; return NULL; };
#ifdef _WIN32
    if(ret == -2) goto retry;
#endif
    if(strncmp(text,"end",3) == 0) break;
    sscanf(text,"qRgb(%d,%d,%d,%d)",&ii,&r,&g,&b);
    if(r==1 && g==1 && b==1) rgb[icol] = qRgba(r,g,b,0);
    else                     rgb[icol] = qRgba(r,g,b,255);
  }

  temp = new QImage((uchar *) data,w,h,8,rgb,icol,QImage::IgnoreEndian);
  *d = data;
  temp->setAlphaBuffer(true);
  return temp;
}

void Interpreter::metaFileLog()
{
QString dir;

  QFileDialog dlg;
  dir = dlg.getExistingDirectory();
  if(!dir.isEmpty())
  {
    for(int i=1; i<nmax; i++)
    {
      if(all[i]->type == TQDraw)
      {
        char filename[MAX_PRINTF_LENGTH],buf[80];
        sprintf(buf,"pvMetaFile%d.pvm",i);
        strcpy(filename,(const char *) dir); strcat(filename,buf);
        QDrawWidget *ptr = (QDrawWidget *) all[i]->w;
        if(opt.arg_debug) printf("logpvm(%s)\n",filename);
        ptr->logToFile(filename);
      }
    }
  }
}

void Interpreter::metaFileBmp()
{
QString dir;

  QFileDialog dlg;
  dir = dlg.getExistingDirectory();
  if(!dir.isEmpty())
  {
    for(int i=1; i<nmax; i++)
    {
      if(all[i]->type == TQDraw)
      {
        char filename[MAX_PRINTF_LENGTH],buf[80];
        sprintf(buf,"pvMetaFile%d.bmp",i);
        strcpy(filename,(const char *) dir); strcat(filename,buf);
        QDrawWidget *ptr = (QDrawWidget *) all[i]->w;
        if(opt.arg_debug) printf("logbmp(%s)\n",filename);
        ptr->logBmpToFile(filename);
      }
    }
  }
}

void Interpreter::setPath(const char *p)
{
  char *cptr;

  strcpy(path,p);
#ifdef unix
  cptr = strrchr(path,'/');
#endif
#ifdef __VMS
  cptr = strrchr(path,']');
#endif
#ifdef _WIN32
  cptr = strrchr(path,'\\');
#endif
  if(cptr != NULL)
  {
    cptr++;
    while(*cptr != '\0') *cptr++ = '\0';
  }
  else
  {
    cptr = &path[0];
    while(*cptr != '\0') *cptr++ = '\0';
  }
}

void Interpreter::downloadFile(const char *file)
{
  struct utimbuf utb;
  int   ret;
  short len;
  char  buf[4096];
  QString filename = temp + file;

  unlink(filename);
#ifdef unix
  int fhdl = ::open(filename, O_CREAT | O_WRONLY, S_IRUSR | S_IWUSR | S_IRGRP);
#endif
#ifdef _WIN32
  int fhdl = ::open(filename, _O_CREAT | _O_WRONLY | _O_BINARY, _S_IREAD | _S_IWRITE);
#endif
#ifdef __VMS
  int fhdl = ::open(filename, O_CREAT | O_WRONLY, 0);
#endif
  if(fhdl <= 0)
  {
    QMessageBox::warning(mainWindow,
    "pvbrowser","Could not open: " + filename);
    return;
  }

  QApplication::setOverrideCursor( Qt::WaitCursor );
  while(1)
  {
    ret = tcp_rec_binary(s, (char *) &len, 2);
    if(ret == -1) break;
    len = ntohs(len);
    if(len <= 0)  break;
    if(len > (short) sizeof(buf))
    {
      QMessageBox::warning(mainWindow,
      "pvbrowser","buffer too small in downloadFile: " + filename);
      break;
    }
    ret = tcp_rec_binary(s, buf, len);
    if(ret == -1) break;
    ret = ::write(fhdl,buf,ret);
    if(ret == -1) break;
  }
#ifndef _WIN32
  ::fsync(fhdl);
#endif
  ::close(fhdl);
  //time_t timet; // remote file time for caching
  utb.actime  = timet;
  utb.modtime = timet;
  utime(file,&utb); // set file modification time
  QApplication::restoreOverrideCursor();
}


int Interpreter::get_text(const char *c, QString &txt)
{
  char mytext[MAX_PRINTF_LENGTH+16],*t;
  int i;

  t = &mytext[0];
  *t = '\0';
  while(*c != '\0' && *c != '"') c++; // search for first "
  if(*c == '\0')
  {
    char *cptr;
    if(tcp_rec(s,t,MAX_PRINTF_LENGTH-1) < 0) return -1;
    cptr = strchr(t,'\n');
    if(cptr != NULL) *cptr = '\0';
    txt = QString::fromUtf8(mytext);
    return 0;
  }
  c++;
  i = 0;
  while(*c != '\0') // until terminating " is seen
  {
    if(i >= MAX_PRINTF_LENGTH)
    {
      *t = '\0';
      return 1;
    }
    i++;
    if(*c == '\\')
    {
      c++;
      if     (*c == '"')  *t++ = '"';
      else if(*c == '\n') *t++ = '\n';
      else if(*c == '\t') *t++ = '\t';
      else if(*c == '\\') *t++ = '\\';
      else if(*c == '\0') break;
      c++;
    }
    else if(*c == '"')
    {
      *t = '\0';
      break;
    }
    else
    {
      *t++ = *c++;
    }
  }
  *t = '\0';

  txt = QString::fromUtf8(mytext);
  return 0;
}

void Interpreter::popupMenu(int id_return, const char *menu)
{
  char buf[800];
  int  i,ifirst,ret;
  Q3PopupMenu popupMenu;

  ifirst = 0;
  for(i=0;; i++)
  {
    if(menu[i] == ',' || menu[i] == '\0')
    {
      if(i == ifirst && i == 0)
      {
        popupMenu.insertSeparator();
      }
      else if(menu[i] != '\0' && menu[i+1] == ',')
      {
        if(i > 0 && menu[i-1] != ',')
        {
          strncpy(buf,&menu[ifirst],i-ifirst);
          buf[i-ifirst] = '\0';
          popupMenu.insertItem(buf);
        }
        popupMenu.insertSeparator();
      }
      else
      {
        if(i > 0 && menu[i-1] != ',')
        {
          strncpy(buf,&menu[ifirst],i-ifirst);
          buf[i-ifirst] = '\0';
          popupMenu.insertItem(buf);
        }
      }
      ifirst = i+1;
    }
    if(menu[i] == '\0') break;
  }
  
  ret = popupMenu.exec(QCursor::pos());
  if(ret != -1)
  {
    sprintf(buf,"text(%d,\"%s\")\n", id_return, (const char *) popupMenu.text(ret).toUtf8());
  }
  else
  {
    sprintf(buf,"text(%d,\"\")\n", id_return);
  }
  tcp_send(s,buf,strlen(buf));
}

void Interpreter::interpret(const char *command)
{
  switch(command[0])
  {
    case 'a':
      interpreta(command);
      break;
    case 'b':
      interpretb(command);
      break;
    case 'c':
      interpretc(command);
      break;
    case 'd':
      interpretd(command);
      break;
    case 'e':
      interprete(command);
      break;
    case 'f':
      interpretf(command);
      break;
    case 'g':
      interpretg(command);
      break;
    case 'h':
      interpreth(command);
      break;
    case 'i':
      interpreti(command);
      break;
    case 'm':
      interpretm(command);
      break;
    case 'p':
      interpretp(command);
      break;
    case 'r':
      interpretr(command);
      break;
    case 's':
      interprets(command);
      break;
    case 't':
      interprett(command);
      break;
    case 'v':
      interpretv(command);
      break;
    case 'q':
      interpretq(command);
      break;
    case 'Q':
      interpretQ(command);
      break;
    default:
      break;
  }
}

void Interpreter::interpreta(const char *command)
{
  if(strncmp(command,"addTab(",7) == 0)
  {
    int child;
    sscanf(command,"addTab(%d,%d,",&i,&child);
    get_text(command,text);
    if(all[i]->type == TQTabWidget)
    {
      MyQTabWidget *ptr = (MyQTabWidget *) all[i]->w;
      ptr->addTab(all[child]->w, text);
    }
    else if(all[i]->type == TQToolBox)
    {
      MyQToolBox *ptr = (MyQToolBox *) all[i]->w;
      ptr->addItem(all[child]->w, text);
    }
  }
  else if(strncmp(command,"addColumn(",10) == 0)
  {
    int size;
    sscanf(command,"addColumn(%d,%d,",&i,&size);
    get_text(command,text);
    if(all[i]->type == TQListView)
    {
      MyListView *ptr = (MyListView *) all[i]->w;
      ptr->addColumn(text, size);
    }
  }
  else if(strncmp(command,"addStretch(",11) == 0)
  {
    int param;
    sscanf(command,"addStretch(%d,%d,",&i,&param);
    if(i == 0)
    {
      if(mainLayoutType == TQVbox) ((QVBoxLayout *) mainLayout)->addStretch(param);
      if(mainLayoutType == TQHbox) ((QHBoxLayout *) mainLayout)->addStretch(param);
      //if(mainLayoutType == TQGrid) ((QGridLayout *) mainLayout)->addStretch(param);
    }
    else
    {
      if(all[i]->type == TQVbox) ((QVBoxLayout *) all[i]->w)->addStretch(param);
      if(all[i]->type == TQHbox) ((QHBoxLayout *) all[i]->w)->addStretch(param);
      //if(all[i]->type == TQGrid) ((QGridLayout *) all[i]->w)->addStretch(param);
    }
  }
  else if(strncmp(command,"addWidgetOrLayout(",18) == 0)
  {
    int item,type,row,col;
    QWidget *w;
    sscanf(command,"addWidgetOrLayout(%d,%d,%d,%d",&i,&item,&row,&col);
    if(i == 0) w = mainLayout;
    else       w = all[i]->w;
    if(i == 0) type = mainLayoutType;
    else       type = all[i]->type;
    if(type == TQVbox)
    {
      QVBoxLayout *ptr = (QVBoxLayout *) w;
      if     (all[item]->type == TQVbox) ptr->addLayout((QVBoxLayout *) all[item]->w);
      else if(all[item]->type == TQHbox) ptr->addLayout((QHBoxLayout *) all[item]->w);
      else if(all[item]->type == TQGrid) ptr->addLayout((QGridLayout *) all[item]->w);
      else                               ptr->addWidget(all[item]->w);
    }
    if(type == TQHbox)
    {
      QHBoxLayout *ptr = (QHBoxLayout *) w;
      if     (all[item]->type == TQVbox) ptr->addLayout((QVBoxLayout *) all[item]->w);
      else if(all[item]->type == TQHbox) ptr->addLayout((QHBoxLayout *) all[item]->w);
      else if(all[item]->type == TQGrid) ptr->addLayout((QGridLayout *) all[item]->w);
      else                               ptr->addWidget(all[item]->w);
    }
    if(type == TQGrid)
    {
      QGridLayout *ptr = (QGridLayout *) w;
      if     (all[item]->type == TQVbox) ptr->addLayout((QVBoxLayout *) all[item]->w,row,col);
      else if(all[item]->type == TQHbox) ptr->addLayout((QHBoxLayout *) all[item]->w,row,col);
      else if(all[item]->type == TQGrid) ptr->addLayout((QGridLayout *) all[item]->w,row,col);
      else                               ptr->addWidget(all[item]->w,row,col);
    }
  }
}

void Interpreter::interpretb(const char *command)
{
  if(strncmp(command,"beep(",5) == 0)
  {
    qApp->beep();
  }
}

void Interpreter::interpretc(const char *command)
{
  if(strncmp(command,"clear(",6) == 0)
  {
    sscanf(command,"clear(%d",&i);
    if(all[i]->type == TQListBox)
    {
      MyListBox *ptr = (MyListBox *) all[i]->w;
      ptr->clear();
    }
    else if(all[i]->type == TQMultiLineEdit)
    {
      MyMultiLineEdit *ptr = (MyMultiLineEdit *) all[i]->w;
      ptr->clear();
    }
    else if(all[i]->type == TQListView)
    {
      MyListView *ptr = (MyListView *) all[i]->w;
      ptr->clear();
    }
    else if(all[i]->type == TQIconView)
    {
      MyIconView *ptr = (MyIconView *) all[i]->w;
      ptr->clear();
    }
    else if(all[i]->type == TQComboBox)
    {
      MyComboBox *ptr = (MyComboBox *) all[i]->w;
      ptr->clear();
    }
  }
  else if(strncmp(command,"changeItem(",11) == 0)
  {
    int index,pixmap;
    char *data,text2[80];
    sscanf(command,"changeItem(%d,%d,%d,",&i,&index,&pixmap);
    get_text(command,text);
    if(all[i]->type == TQListBox)
    {
      MyListBox *ptr = (MyListBox *) all[i]->w;
      QPixmap pm;
      if(pixmap==0) ptr->changeItem(pm,text,index);
      else
      {
        tcp_rec(s,text2,sizeof(text2));
        sscanf(text2,"(%d,%d)",&w,&h);
        QImage *temp = readBmpFromSocket(w,h,&data);
        if(temp != NULL)
        {
          pm = *temp;
          pm.setMask(pm.createHeuristicMask());
          ptr->changeItem(pm,text,index);
          if(data != NULL) delete data;
          if(temp != NULL) delete temp;
        }
      }
    }
    else
    {
      if(pixmap==0) ;
      else
      {
        tcp_rec(s,text2,sizeof(text2));
        sscanf(text2,"(%d,%d)",&w,&h);
        QImage *temp = readBmpFromSocket(w,h,&data);
        if(temp != NULL) delete temp;
      }
    }
  }
}

void Interpreter::interpretd(const char *command)
{
  if(strncmp(command,"displayNum(",11) == 0)
  {
    sscanf(command,"displayNum(%d,%d)",&i,&n);
    if(all[i]->type == TQLCDNumber)
    {
      QLCDNumber *ptr = (QLCDNumber *) all[i]->w;
      ptr->display(n);
    }
  }
  else if(strncmp(command,"displayFloat(",13) == 0)
  {
    float val;
    sscanf(command,"displayFloat(%d,%f)",&i,&val);

    if(all[i]->type == TQLCDNumber)
    {
      QLCDNumber *ptr = (QLCDNumber *) all[i]->w;
      ptr->display((double) val);
    }
  }
  else if(strncmp(command,"displayStr(",11) == 0)
  {
    sscanf(command,"displayStr(%d,",&i);
    get_text(command,text);

    if(all[i]->type == TQLCDNumber)
    {
      QLCDNumber *ptr = (QLCDNumber *) all[i]->w;
      ptr->display(text);
    }
  }
  else if(strncmp(command,"downloadFile(",13) == 0)
  {
    get_text(command,text);
    downloadFile(text);
  }
}

void Interpreter::interprete(const char *command)
{
  if(strncmp(command,"endDefinition",13) == 0) // endDefinition of new MainWidget
  {
    if(allBase == NULL)
    {
      if(opt.arg_debug) printf("endDefinition allBase == NULL\n");
      w = mainWindow->view->width();
      h = mainWindow->view->height();
      if(opt.arg_debug) printf("endDefinition w=%d h=%d\n",w,h);
      if(mainWindow->scroll != NULL)
      {
        if(opt.arg_debug) printf("endDefinition scroll != NULL\n");
        mainWindow->scroll->hide();
        delete mainWindow->scroll;
        mainWindow->scroll = NULL;
      }
      else if(mainWindow->view != NULL)
      {
        if(opt.arg_debug) printf("endDefinition view != NULL\n");
        mainWindow->view->hide();
        delete mainWindow->view;
        mainWindow->view = NULL;
      }
      mainWindow->view = v;
      if(opt.arg_debug) printf("endDefinition2\n");
      if(hasLayout == 0)
      {
        if(opt.arg_debug) printf("endDefinition hasLayout == 0\n");
        mainWindow->scroll = new QScrollArea;
        mainWindow->scroll->setWidget(mainWindow->view);
        if(opt.scrollbars == 0)
        {
          //scroll->setResizePolicy(Q3ScrollView::AutoOneFit);
          //scroll->setVScrollBarMode(Q3ScrollView::AlwaysOff);
          //scroll->setHScrollBarMode(Q3ScrollView::AlwaysOff);
        }
        if(opt.arg_debug) printf("endDefinition setCentralWidget\n");
        mainWindow->setCentralWidget(mainWindow->scroll);
        if(opt.arg_debug) printf("endDefinition resize\n");
        mainWindow->view->resize(w,h);
        //mainWindow->scroll->resizeContents(w,h);
//#ifdef _WIN32
//        mainWindow->scroll->resize(w,h);
//#endif
        if(opt.arg_debug) printf("endDefinition show\n");
        mainWindow->scroll->show();
      }
      else
      {
        if(opt.arg_debug) printf("endDefinition hasLayout == 1\n");
        mainWindow->scroll = NULL;
        mainWindow->view->resize(w,h);
        if(opt.arg_debug) printf("endDefinition set central widget\n");
        mainWindow->setCentralWidget(mainWindow->view);
        if(opt.arg_debug) printf("endDefinition show\n");
        mainWindow->view->show();
      }
    }
    else
    {
      QString capt;
      if(opt.arg_debug) printf("endDefinition allBase != NULL modalDialog\n");
      capt.sprintf("%s - pvbrowser",(const char *) mainWindow->curFile.toAscii());
      modalDialog->setWindowTitle(capt);
      modalDialog->show();
    }
    if(hasLayout)
    {
      for(int i=1; i<nmax; i++)
      {
        if(all[i]->type == TQGroupBox    ||
           all[i]->type == TQButtonGroup ||
           all[i]->type == TQFrame        )
        {
          QRect rect = all[i]->w->childrenRect();
          all[i]->w->setMinimumSize(rect.width(),2*rect.height());
        }
      }
    }
    if(opt.arg_debug) printf("endDefinition end\n");
  }
  else if(strncmp(command,"ensureCellVisible(",18) == 0)
  {
    int row,col;
    sscanf(command,"ensureCellVisible(%d,%d,%d)",&i,&row,&col);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTable)
    {
      MyTable *p = (MyTable *) all[i]->w;
      p->ensureCellVisible(row,col);
    }
  }
  else if(strncmp(command,"ensureListVisible(",18) == 0)
  {
    sscanf(command,"ensureListVisible(%d",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQListView)
    {
      get_text(command,text);
      MyListView *l = (MyListView *) all[i]->w;
      l->ensureVisible(text, (MyListViewItem *) l->firstChild());
    }
  }
}

void Interpreter::interpretf(const char *command)
{
  if(strncmp(command,"fileCache(",10) == 0) // cache downloaded files
  {
    struct stat statbuf;
    struct tm remote;
    int ret;
    char buf[80];

    memset(&remote,sizeof(remote),0);
    sscanf(command,"fileCache(%d,%d,%d,%d,%d,%d,%d,%d,%d"
      ,&remote.tm_sec
      ,&remote.tm_min
      ,&remote.tm_hour
      ,&remote.tm_mon
      ,&remote.tm_year
      ,&remote.tm_mday
      ,&remote.tm_wday
      ,&remote.tm_yday
      ,&remote.tm_isdst);
    get_text(command,text);
    timet = mktime(&remote);
    ret = stat(text,&statbuf);
    if(ret < 0) // file does not exist
    {
      strcpy(buf,"cache(0)\n");   // file was not cached
      tcp_send(s,buf,strlen(buf));
    }
    else if(memcmp(&statbuf.st_mtime, &timet,sizeof(time_t)) == 0)
    {
      strcpy(buf,"cache(1)\n"); // file was cached
      tcp_send(s,buf,strlen(buf));
    }
    else
    {
      strcpy(buf,"cache(0)\n"); // file time is different (not cached)
      tcp_send(s,buf,strlen(buf));
    }
  }
  else if(strncmp(command,"fileDialog(",11) == 0) // Open, Save, FindDirectory
  {
    QString result;
    int id_return,type;
    sscanf(command,"fileDialog(%d,%d",&id_return,&type);
    if     (type==0) result = QFileDialog::getOpenFileName();
    else if(type==1) result = QFileDialog::getSaveFileName();
    else if(type==2) result = QFileDialog::getExistingDirectory();
    text.sprintf("text(%d,\"%s\")\n", id_return, (const char *) result);
    tcp_send(s,text,strlen(text));
  }
}

void Interpreter::interpretg(const char *command)
{
  if(strncmp(command,"gbeginDraw(",11) == 0) // start drawing
  {
    sscanf(command,"gbeginDraw(%d)",&i);
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->showFromSocket(s);
    }
  }
  else if(strncmp(command,"gCopyToClipboard(",17) == 0)
  {
    sscanf(command,"gCopyToClipboard(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->copyToClipboard();
    }
    else
    {
      QWidget *w = all[i]->w;
      QPixmap pm(QPixmap::grabWindow(w->winId(),0,0,w->width(),w->height()));
      QApplication::clipboard()->setPixmap(pm);
    }
  }
  else if(strncmp(command,"gSetZoomX(",10) == 0)
  {
    float zx;
    sscanf(command,"gSetZoomX(%d,%f)",&i,&zx);
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->setZoomX(zx);
    }
  }
  else if(strncmp(command,"gSetZoomY(",10) == 0)
  {
    float zy;
    sscanf(command,"gSetZoomY(%d,%f)",&i,&zy);
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->setZoomY(zy);
    }
  }
  else if(strncmp(command,"gSave(",6) == 0)
  {
    sscanf(command,"gSave(%d,",&i);
    get_text(command,text);
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->logToFile(text);
    }
  }
  else if(strncmp(command,"gSaveAsBmp(",11) == 0)
  {
    sscanf(command,"gSaveAsBmp(%d,",&i);
    get_text(command,text);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *p = (QDrawWidget *) all[i]->w;
      p->logBmpToFile(text);
    }
    else
    {
      QWidget *w = (QDrawWidget *) all[i]->w;
      QPixmap pm(QPixmap::grabWindow(w->winId(),0,0,w->width(),w->height()));
      pm.save(text,"BMP");
    }
  }
}

void Interpreter::interpreth(const char *command)
{
  if(strncmp(command,"hide(",5) == 0) // hide the widget
  {
    sscanf(command,"hide(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    all[i]->w->hide();
  }
  else if(strncmp(command,"hyperlink(",10) == 0) // follow a hyperlink to a different computer
  {
    get_text(command,text);
    mainWindow->slotUrl(text);
  }
}

void Interpreter::interpreti(const char *command)
{
  if(strncmp(command,"insertCols(",11) == 0)
  {
    int col = 0;
    int count = 0;
    sscanf(command,"insertCols(%d,%d,%d",&i,&col,&count);
    if(all[i]->type == TQTable)
    {
      MyTable *ptr = (MyTable *) all[i]->w;
      ptr->insertColumns(col,count);
    }
  }
  else if(strncmp(command,"insertItem(",11) == 0)
  {
    int index,pixmap;
    char *data,text2[80];
    sscanf(command,"insertItem(%d,%d,%d,",&i,&index,&pixmap);
    get_text(command,text);
    if(all[i]->type == TQListBox)
    {
      MyListBox *ptr = (MyListBox *) all[i]->w;
      QPixmap pm;
      if(pixmap==0) ptr->insertItem(pm,text,index);
      else
      {
        tcp_rec(s,text2,sizeof(text2));
        sscanf(text2,"(%d,%d)",&w,&h);
        QImage *temp = readBmpFromSocket(w,h,&data);
        if(temp != NULL)
        {
          pm = *temp;
          pm.setMask(pm.createHeuristicMask());
          ptr->insertItem(pm,text,index);
          if(data != NULL) delete data;
          if(temp != NULL) delete temp;
        }
      }
    }
    else if(all[i]->type == TQComboBox)
    {
      MyComboBox *ptr = (MyComboBox *) all[i]->w;
      QPixmap pm;
      if(pixmap==0) ptr->insertItem(text,index);
      else
      {
        tcp_rec(s,text2,sizeof(text2));
        sscanf(text2,"(%d,%d)",&w,&h);
        QImage *temp = readBmpFromSocket(w,h,&data);
        if(temp != NULL)
        {
          pm = *temp;
          pm.setMask(pm.createHeuristicMask());
          ptr->insertItem(pm,text,index);
          if(data != NULL) delete data;
          if(temp != NULL) delete temp;
        }
      }
    }
    else
    {
      if(pixmap==0) ;
      else
      {
        tcp_rec(s,text2,sizeof(text2));
        sscanf(text2,"(%d,%d)",&w,&h);
        QImage *temp = readBmpFromSocket(w,h,&data);
        if(temp != NULL) delete temp;
      }
    }
  }
  else if(strncmp(command,"insertRows(",11) == 0)
  {
    int row = 0;
    int count = 0;
    sscanf(command,"insertRows(%d,%d,%d",&i,&row,&count);
    if(all[i]->type == TQTable)
    {
      MyTable *ptr = (MyTable *) all[i]->w;
      ptr->insertRows(row,count);
    }
  }
  else if(strncmp(command,"inputDialog(",12) == 0)
  {
    char *cptr;
    char default_text[MAX_PRINTF_LENGTH] = "";
    bool ok;
    int  id_return;
    sscanf(command,"inputDialog(%d,",&id_return);
    get_text(command,text);
    for(int i=0; i<text.length(); i++)
    {
      if(text[i] == 12) text[i] = '\n'; // replace FF
    }
    tcp_rec(s,default_text,sizeof(default_text));
    cptr = strchr(default_text,'\n');
    if(cptr != NULL) *cptr = '\0';
    if(opt.arg_debug) printf("QInputDialog text=%s default_text=%s\n", (const char *) text.toAscii(), default_text);
    QString result = QInputDialog::getText("pvbrowser",text,QLineEdit::Normal,default_text,&ok,mainWindow);
    if(ok && result.length() > 0)
    {
      result.truncate(MAX_PRINTF_LENGTH-80);
      text.sprintf("text(%d,\"%s\")\n", id_return, (const char *) result);
      tcp_send(s,text,strlen(text));
    }
    else
    {
      text.sprintf("text(%d,\"\")\n", id_return);
      tcp_send(s,text,strlen(text));
    }
  }
}

void Interpreter::interpretm(const char *command)
{
  if(strncmp(command,"move(",5) == 0) // move the widget
  {
    sscanf(command,"move(%d,%d,%d)",&i,&x,&y);
    if(zoom != 100)
    {
      x = (x*zoom)/100;
      y = (y*zoom)/100;
    }
    if(i < 0) return;
    if(i >= nmax) return;
    all[i]->w->move(x,y);
  }
  else if(strncmp(command,"messageBox(",11) == 0) // open a messageBox()
  {
    char buf[80];
    int ret,type,b0,b1,b2,id_return;
    sscanf(command,"messageBox(%d,%d,%d,%d,%d,",&id_return,&type,&b0,&b1,&b2);
    get_text(command,text);
    for(int i=0; i<text.length(); i++)
    {
      if(text[i] == 12) text[i] = '\n'; // replace FF
    }
    ret = -1;
    if(opt.arg_debug) printf("QMessageBox type=%d text=%s\n",type,(const char *) text);
    if(type==0) ret = QMessageBox::information(mainWindow,"pvbrowser",text,b0,b1,b2);
    if(type==1) ret = QMessageBox::warning    (mainWindow,"pvbrowser",text,b0,b1,b2);
    if(type==2) ret = QMessageBox::critical   (mainWindow,"pvbrowser",text,b0,b1,b2);
    sprintf(buf,"slider(%d,%d)\n",id_return,ret);
    tcp_send(s,buf,strlen(buf));
  }
  else if(strncmp(command,"moveContent(",12) == 0) // move in QTextBrowser
  {
    int pos;
    sscanf(command,"moveContent(%d,%d",&i,&pos);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTextBrowser)
    {
      MyTextBrowser *ptr = (MyTextBrowser *) all[i]->w;
      ptr->moveContent(pos);
    }
  }
  else if(strncmp(command,"modalInit(",10) == 0) // init modalDialog
  {
    int width, height;
    sscanf(command,"modalInit(%d,%d",&width,&height);
    allBase = all;
    nmaxBase = nmax;
    modalDialog = new MyDialog(this,s,0,mainWindow);
    modalDialog->resize((width*zoom)/100,(height*zoom)/100);
    modalDialog->setFixedSize((width*zoom)/100,(height*zoom)/100);
  }
  else if(strncmp(command,"modalTerminate(",15) == 0) // termnate modalDialog
  {
    if(modalDialog != NULL)
    {
      printf("command=%s\n",command);
      delete modalDialog;
      modalDialog = NULL;
    }
  }
  else if(strncmp(command,"modalBase(",10) == 0) // switch all to baseDialog
  {
    if(allBase != NULL)
    {
      all = allBase;
      nmax = nmaxBase;
    }
  }
  else if(strncmp(command,"modalModal(",11) == 0) // switch all to modalDialog
  {
    if(allModal != NULL)
    {
      all = allModal;
      nmax = nmaxModal;
    }
  }
  else if(strncmp(command,"movie(",6) == 0) // control movie within QLabel
  {
    int func,param;
    sscanf(command,"movie(%d,%d,%d",&i,&func,&param);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQLabel)
    {
      QLabel *ptr = (QLabel *) all[i]->w;
      switch(func)
      {
        case 0: // pvSetMovie
          get_text(command,text);
          if(param == 0) ptr->setBackgroundMode(Qt::NoBackground);
          if(param == 1) ptr->setBackgroundMode(Qt::PaletteBackground);
          //ptr->setAutoResize(true);
          ptr->setScaledContents(true);
          ptr->setMovie(new QMovie(text));
          break;
        case 1: // pvMovieControl
          if(param == -2) ptr->movie()->start();
          if(param == -1) ptr->movie()->setPaused(false);
          if(param ==  0) ptr->movie()->setPaused(true);
          //if(param >   0) ptr->movie()->step(param);
          break;
        case 2: // pvMovieSpeed
          ptr->movie()->setSpeed(param);
          break;
        default:
          break;  
      }
    }
  }
}

void Interpreter::interpretp(const char *command)
{
  if(strncmp(command,"print(",6) == 0) // print the widget
  {
    sscanf(command,"print(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *ptr = (QDrawWidget *) all[i]->w;
      ptr->print();
    }
    else
    {
      QPrinter printer;
      printer.setOrientation(QPrinter::Landscape);
      printer.setColorMode(QPrinter::Color);
      QWidget *w = all[i]->w;
      QPixmap pm(QPixmap::grabWindow(w->winId(),0,0,w->width(),w->height()));
      if(printer.setup(mainWindow))
      {
        QPainter painter;
        painter.begin(&printer);
//#ifndef _WIN32
//        if(printer.orientation() == QPrinter::Landscape)
//        {
//          painter.rotate(-90.0f);
//          painter.translate(-painter.viewport().width(),0.0f);
//        }
//#endif
        if(strlen(l_print_header) > 0)
        {
          QString date_time;
          date_time  = l_print_header;
          date_time += QDateTime::currentDateTime().toString(Qt::TextDate);
          painter.drawText(LEFT_PRINT_MARGIN,UPPER_PRINT_MARGIN - 30,1280,100,Qt::TextExpandTabs | Qt::TextDontClip, date_time);
        }
        painter.drawPixmap(LEFT_PRINT_MARGIN,UPPER_PRINT_MARGIN,pm);
        painter.end();
      }
    }
  }
#ifdef USE_OPEN_GL
  else if(strncmp(command,"pvGlUpdate(",11) == 0) // update OpenGL widget
  {
    PvGLWidget *ptr;
    sscanf(command,"pvGlUpdate(%d)",&i);

    if(all[i]->type == TQGl)
    {
      ptr = (PvGLWidget *) all[i]->w;
      ptr->updateGL();
    }
  }
  else if(strncmp(command,"pvGlBegin(",10) == 0) // begin OpenGL commands
  {
    GLenum mode;
    PvGLWidget *ptr;

    sscanf(command,"pvGlBegin(%d,%d)",&i,(int *) &mode);
    if(all[i]->type == TQGl)
    {
      ptr = (PvGLWidget *) all[i]->w;
      ptr->makeCurrent();
      ptr->gl_interpret();
    }
  }
#endif
  else if(strncmp(command,"playSound(",10) == 0) // play a (WAV) sound
  {
    get_text(command,text);
    printf("playSound(\"%s\")\n",(const char *) text);
    if(QSound::isAvailable())
    {
      QSound::play(text);
    }
    else
    {
      printf("sound not available on this platform\n");
      printf("On Microsoft Windows the underlying multimedia system is used; only WAVE format sound files are supported.\n");
      printf("On X11 the Network Audio System is used if available, otherwise all operations work silently. NAS supports WAVE and AU files.\n");
      printf("On Macintosh, ironically, we use QT (QuickTime) for sound, this means all QuickTime formats are supported by Qt/Mac.\n");
      printf("On Qt/Embedded, a built-in mixing sound server is used, which accesses /dev/dsp directly. Only the WAVE format is supported.\n");
      qApp->beep();
    }
  }
  else if(strncmp(command,"popupMenu(",10) == 0) // open a popupMenu()
  {
    int id_return;
    sscanf(command,"popupMenu(%d",&id_return);
    get_text(command,text);
    popupMenu(id_return,text);
  }
}

void Interpreter::interpretr(const char *command)
{
  if(strncmp(command,"removeCol(",10) == 0)
  {
    int col = 0;
    sscanf(command,"removeCol(%d,%d",&i,&col);
    if(all[i]->type == TQTable)
    {
      MyTable *ptr = (MyTable *) all[i]->w;
      ptr->removeColumn(col);
    }
  }
  else if(strncmp(command,"removeRow(",10) == 0)
  {
    int row = 0;
    sscanf(command,"removeRow(%d,%d",&i,&row);
    if(all[i]->type == TQTable)
    {
      MyTable *ptr = (MyTable *) all[i]->w;
      ptr->removeRow(row);
    }
  }
  else if(strncmp(command,"resize(",7) == 0) // resize the widget
  {
    sscanf(command,"resize(%d,%d,%d)",&i,&w,&h);
    if(zoom != 100)
    {
      w = (w*zoom)/100;
      h = (h*zoom)/100;
      if(w < 1) w = 1;
      if(h < 1) h = 1;
    }
    if(i == 0)
    {
      mainWindow->view->resize(w,h);
      if(mainWindow->scroll != NULL)
      {
        mainWindow->scroll->resize(w,h);
        //mainWindow->scroll->resizeContents(w,h);
      }
    }
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQDraw)
    {
      QDrawWidget *iw = (QDrawWidget *) all[i]->w;
      iw->resize(w,h);
      if(zoom != 100)
      {
        float fzoom = ((float)zoom)/100.0f;
        iw->setZoomX(fzoom);
        iw->setZoomY(fzoom);
      }
    }
    else if(all[i]->type == TQLCDNumber)
    {
      QLCDNumber *iw = (QLCDNumber *) all[i]->w;
      iw->resize(w,h);
    }
    else
    {
      all[i]->w->resize(w,h);
    }
  }
  else if(strncmp(command,"removeItem(",11) == 0)
  {
    int index;
    sscanf(command,"removeItem(%d,%d",&i,&index);
    if(all[i]->type == TQListBox)
    {
      MyListBox *ptr = (MyListBox *) all[i]->w;
      ptr->removeItem(index);
    }
    else if(all[i]->type == TQComboBox)
    {
      MyComboBox *ptr = (MyComboBox *) all[i]->w;
      ptr->removeItem(index);
    }
  }
  else if(strncmp(command,"removeItemByName(",17) == 0)
  {
    sscanf(command,"removeItemByName(%d",&i);
    get_text(command,text);
    if(all[i]->type == TQListBox)
    {
      MyListBox *ptr = (MyListBox *) all[i]->w;
      ptr->removeItemByName(text);
    }
    else if(all[i]->type == TQComboBox)
    {
      MyComboBox *ptr = (MyComboBox *) all[i]->w;
      ptr->removeItemByName(text);
    }
  }
  else if(strncmp(command,"removeListViewItem(",19) == 0)
  {
    sscanf(command,"removeListViewItem(%d",&i);
    get_text(command,text);
    if(all[i]->type == TQListView)
    {
      MyListView *ptr = (MyListView *) all[i]->w;
      ptr->deleteListViewItem(text, (MyListViewItem *) ptr->firstChild());
    }
  }
  else if(strncmp(command,"removeIconViewItem(",19) == 0)
  {
    sscanf(command,"removeIconViewItem(%d",&i);
    get_text(command,text);
    if(all[i]->type == TQIconView)
    {
      MyIconView *ptr = (MyIconView *) all[i]->w;
      ptr->deleteIconViewItem(text);
    }
  }
}

void Interpreter::interprets(const char *command)
{
  if(strncmp(command,"set",3) == 0)
  {
    switch(command[3])
    {
      case 'A':
        if(strncmp(command,"setAlignment(",13) == 0) // set text alignment
        {
          int alignment,qtalignment;
          sscanf(command,"setAlignment(%d,%d",&i,&alignment);
          if(i < 0) return;
          if(i >= nmax) return;

          qtalignment = 0;

          if(alignment & PV::AlignLeft)    qtalignment |= Qt::AlignLeft;
          if(alignment & PV::AlignRight)   qtalignment |= Qt::AlignRight;
          if(alignment & PV::AlignHCenter) qtalignment |= Qt::AlignHCenter;
          if(alignment & PV::AlignJustify) qtalignment |= Qt::AlignJustify;

          if(alignment & PV::AlignTop)     qtalignment |= Qt::AlignTop;
          if(alignment & PV::AlignBottom)  qtalignment |= Qt::AlignBottom;
          if(alignment & PV::AlignVCenter) qtalignment |= Qt::AlignVCenter;

          if(all[i]->type == TQLabel)
          {
            QLabel *ptr = (QLabel *) all[i]->w;
            ptr->setAlignment(qtalignment);
          }
          else if(all[i]->type == TQLineEdit)
          {
            QLineEdit *ptr = (QLineEdit *) all[i]->w;
            ptr->setAlignment((Qt::Alignment) qtalignment);
          }
        }
        break;
      case 'B':
        if(strncmp(command,"setBackgroundColor(",19) == 0) // set background color
        {
          sscanf(command,"setBackgroundColor(%d,%d,%d,%d)",&i,&r,&g,&b);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQLabel)
          {
            if(r==-1 && g==-1 && b==-1) all[i]->w->unsetPalette();
            else                        all[i]->w->setBackgroundColor(QColor(r,g,b));
          }
          else if(all[i]->type == TQPushButton)
          {
            QPushButton *but = (QPushButton *) all[i]->w;
            QPixmap pm = QPixmap(but->width()/2,but->height());
            pm.fill(QColor(r,g,b));
            but->setPixmap(pm);
          }
          else if(all[i]->type == TQDraw)
          {
            QDrawWidget *ptr = (QDrawWidget *) all[i]->w;
            if(r==-1 && g==-1 && b==-1) all[i]->w->unsetPalette();
            else                        ptr->setBackgroundColor(r,g,b);
          }
          else if(all[i]->type == TQImage)
          {
            QImageWidget *ptr = (QImageWidget *) all[i]->w;
            if(r==-1 && g==-1 && b==-1) all[i]->w->unsetPalette();
            else                        ptr->setBackgroundColor(QColor(r,g,b));
          }
        }
        break;
      case 'C':
        if(strncmp(command,"setChecked(",11) == 0) // set button state
        {
          int state;
          sscanf(command,"setChecked(%d,%d)",&i,&state);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQRadio)
          {
            MyRadioButton *but = (MyRadioButton *) all[i]->w;
            if(state == 1) but->setChecked(true);
            else           but->setChecked(false);
          }
          else if(all[i]->type == TQCheck)
          {
            MyCheckBox *but = (MyCheckBox *) all[i]->w;
            if(state == 1) but->setChecked(true);
            else           but->setChecked(false);
          }
        }
        else if(strncmp(command,"setCurrentItem(",15) == 0) // set current item for comboBox
        {
          int index;
          sscanf(command,"setCurrentItem(%d,%d",&i,&index);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQComboBox)
          {
            MyComboBox *t = (MyComboBox *) all[i]->w;
            t->setCurrentItem(index);
          }
        }
        else if(strncmp(command,"setColumnWidth(",15) == 0) // set column with of the table
        {
          int column, width;
          sscanf(command,"setColumnWidth(%d,%d,%d",&i,&column,&width);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            if(column >= 0)
            {
              t->setColumnWidth(column,width);
            }
            else
            {
              t->setLeftMargin(width);
            }
          }
        }
        break;
      case 'D':
        if(strncmp(command,"setDate(",8) == 0) // set date
        {
          int year, month, day;
          sscanf(command,"setDate(%d,%d,%d,%d)",&i,&year,&month,&day);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQDateEdit)
          {
            Q3DateEdit *e = (Q3DateEdit *) all[i]->w;
            e->setDate(QDate(year,month,day));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->dateEdit()->setDate(QDate(year,month,day));
          }
        }
        else if(strncmp(command,"setDateOrder(",13) == 0) // set date order
        {
          int order;
          sscanf(command,"setDateOrder(%d,%d)",&i,&order);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQDateEdit)
          {
            Q3DateEdit *e = (Q3DateEdit *) all[i]->w;
            e->setOrder((Q3DateEdit::Order) order);
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->dateEdit()->setOrder((Q3DateEdit::Order) order);
          }
        }
        break;
      case 'E':
        if(strncmp(command,"setEchoMode(",12) == 0) // set echo mode in QLineEdit
        {
          int mode;
          sscanf(command,"setEchoMode(%d,%d",&i,&mode);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQLineEdit)
          {
            MyLineEdit *le = (MyLineEdit *) all[i]->w;
            if     (mode == 0) le->setEchoMode(QLineEdit::NoEcho);
            else if(mode == 2) le->setEchoMode(QLineEdit::Password);
            else               le->setEchoMode(QLineEdit::Normal);
          }
        }
        else if(strncmp(command,"setEditable(",12) == 0) // set editable in QLineEdit, QMultiLineEdit
        {
          int mode;
          sscanf(command,"setEditable(%d,%d",&i,&mode);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQLineEdit)
          {
            MyLineEdit *le = (MyLineEdit *) all[i]->w;
            if     (mode == 0) le->setReadOnly(true);
            else if(mode == 1) le->setReadOnly(false);
          }
          else if(all[i]->type == TQMultiLineEdit)
          {
            MyMultiLineEdit *le = (MyMultiLineEdit *) all[i]->w;
            if     (mode == 0) le->setReadOnly(true);
            else if(mode == 1) le->setReadOnly(false);
          }
          else if(all[i]->type == TQTable)
          {
            Q3Table *t = (Q3Table *) all[i]->w;
            if     (mode == 0) t->setReadOnly(true);
            else if(mode == 1) t->setReadOnly(false);
          }
        }
        else if(strncmp(command,"setEnabled(",11) == 0) // set enabled all widgets
        {
          int mode;
          sscanf(command,"setEnabled(%d,%d",&i,&mode);
          if(i < 0) return;
          if(i >= nmax) return;
          all[i]->w->setEnabled(mode);
        }
        break;
      case 'F':
        if(strncmp(command,"setFont(",8) == 0) // set font
        {
          int pointsize,bold,italic,underline,strikeout;
          sscanf(command,"setFont(%d,%d,%d,%d,%d,%d",&i,&pointsize,&bold,&italic,&underline,&strikeout);
          get_text(command,text); // text = family of font

          if(i < 0) return;
          if(i >= nmax) return;
          if(zoom != 100)
          {
            pointsize = (pointsize*zoom)/100;
          }

          QFont font(text, pointsize);
          if(bold      == 0) font.setBold(false);
          if(italic    == 0) font.setItalic(false);
          if(underline == 0) font.setUnderline(false);
          if(strikeout == 0) font.setStrikeOut(false);
          if(bold      == 1) font.setBold(true);
          if(italic    == 1) font.setItalic(true);
          if(underline == 1) font.setUnderline(true);
          if(strikeout == 1) font.setStrikeOut(true);
          if(all[i]->type == TQMultiLineEdit)
          {
            MyMultiLineEdit *me = (MyMultiLineEdit *) all[i]->w;
            me->setCurrentFont(font);
          }
          else
          {
            QWidget *w = (QWidget *) all[i]->w;
            w->setFont(font);
          }
        }
        else if(strncmp(command,"setFontColor(",13) == 0) // set font color
        {
          int r,g,b;
          sscanf(command,"setFontColor(%d,%d,%d,%d",&i,&r,&g,&b);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQMultiLineEdit)
          {
            MyMultiLineEdit *me = (MyMultiLineEdit *) all[i]->w;
            me->setColor(QColor(r,g,b));
          }
          else
          {
            all[i]->w->setForegroundRole(QPalette::WindowText);
            all[i]->w->setPaletteForegroundColor(QColor(r,g,b));
          }
        }
        break;
      case 'G':
        if(strncmp(command,"setGeometry(",12) == 0) // set widget geometry
        {
          sscanf(command,"setGeometry(%d,%d,%d,%d,%d)",&i,&x,&y,&w,&h);
          if(zoom != 100)
          {
            x = (x*zoom)/100;
            y = (y*zoom)/100;
            w = (w*zoom)/100;
            h = (h*zoom)/100;
            if(w < 1) w = 1;
            if(h < 1) h = 1;
          }
          if(i == 0)
          {
            printf("view->resize2 not implemented\n");
            //view->resize(w,h);
            //if(scroll != NULL)
            //{
            //  scroll->resize(w,h);
            //  scroll->resizeContents(w,h);
            //}
          }
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQImage)
          {
            QImageWidget *iw = (QImageWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQDraw)
          {
            QDrawWidget *iw = (QDrawWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
            iw->resize(w,h);
            if(zoom != 100)
            {
              float fzoom = ((float)zoom)/100.0f;
              iw->setZoomX(fzoom);
              iw->setZoomY(fzoom);
            }
          }
          else if(all[i]->type == TQLCDNumber)
          {
            QLCDNumber *iw = (QLCDNumber *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
#ifdef USE_OPEN_GL
          else if(all[i]->type == TQGl)
          {
            PvGLWidget *iw = (PvGLWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
#endif
#ifdef USE_VTK
          else if(all[i]->type == TQVtk)
          {
            pvVtkTclWidget *iw = (pvVtkTclWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
#endif
#ifndef NO_QWT
          else if(all[i]->type == TQwtPlotWidget)
          {
            QwtPlotWidget *iw = (QwtPlotWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtScale)
          {
            QwtScaleWidget *iw = (QwtScaleWidget *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtThermo)
          {
            QwtThermo *iw = (QwtThermo *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtKnob)
          {
            QwtKnob *iw = (QwtKnob *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtCounter)
          {
            QwtCounter *iw = (QwtCounter *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtWheel)
          {
            QwtWheel *iw = (QwtWheel *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtSlider)
          {
            QwtSlider *iw = (QwtSlider *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
          else if(all[i]->type == TQwtCompass)
          {
            QwtCompass *iw = (QwtCompass *) all[i]->w;
            iw->setGeometry(x,y,w,h);
          }
#endif
          else
          {
            if(all[i]->w != NULL)
            {
              all[i]->w->setGeometry(x,y,w,h);
            }
          }
        }
        break;
      case 'H':
        break;
      case 'I':
        if(strncmp(command,"setIconViewItem(",16) == 0) // set text and pixmap in QIconView
        {
          QPixmap pm;
          char *data;

          sscanf(command,"setIconViewItem(%d,%d,%d,",&i,&w,&h);
          get_text(command,text);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQIconView)
          {
            MyIconView *iv = (MyIconView*) all[i]->w;
            if(w==0 && h==0) iv->setIconViewItem(text,pm);
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL)
              {
                pm = *temp;
                pm.setMask(pm.createHeuristicMask());
              }
              iv->setIconViewItem(text,pm);
              if(data != NULL) delete data;
              if(temp != NULL) delete temp;
            }
          }
          else
          {
            if(w==0 && h==0) ;
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL) delete temp;
            }
          }
        }
        else if(strncmp(command,"setImage(",9) == 0) // set new image in existing image
        {
          sscanf(command,"setImage(%d,",&i);
          get_text(command,text); // text = filename of image
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQImage)
          {
            QString filename;
            filename = temp + text;
            QImageWidget *iw = (QImageWidget *) all[i]->w;
            iw->setImage(filename);
          }
        }
        break;
      case 'J':
        break;
      case 'K':
        break;
      case 'L':
        if(strncmp(command,"setListViewPixmap(",18) == 0) // set pixmap in list view
        {
          QPixmap pm;
          char *data;
          int column;
          sscanf(command,"setListViewPixmap(%d,%d,%d,%d,",&i,&w,&h,&column);
          get_text(command,text);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            MyListView *lv = (MyListView*) all[i]->w;
            if(w==0 && h==0) lv->setListViewPixmap(text,column,pm);
            else
            {
              QImage *temp = NULL;
              temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL)
              {
                pm = *temp;
                pm.setMask(pm.createHeuristicMask());
              }
              lv->setListViewPixmap(text,column,pm);
              if(data != NULL) delete data;
              if(temp != NULL) delete temp;
            }
          }
          else
          {
            if(w==0 && h==0) ;
            else
            {
              QImage *temp = NULL;
              temp= readBmpFromSocket(w,h,&data);
              if(temp != NULL) delete temp;
            }
          }
        }
        else if(strncmp(command,"setListViewText(",16) == 0) // set text in QListView
        {
          int column;
          QString path;
          char text2[MAX_PRINTF_LENGTH];
          sscanf(command,"setListViewText(%d,%d",&i,&column);
          get_text(command,path);
          tcp_rec(s,text2,sizeof(text2));
          get_text(text2,text);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            MyListView *lv = (MyListView*) all[i]->w;
            lv->setListViewText(path,column,text);
          }
        }
        break;
      case 'M':
        if(strncmp(command,"setMinimumSize(",15) == 0) // set minimum w,h
        {
          sscanf(command,"setMinimumSize(%d,%d,%d)",&i,&w,&h);
          if(i < 0) return;
          if(i >= nmax) return;
          all[i]->w->setMinimumSize(w,h);
        }
        else if(strncmp(command,"setMaximumSize(",15) == 0) // set maximum w,h
        {
          sscanf(command,"setMaximumSize(%d,%d,%d)",&i,&w,&h);
          if(i < 0) return;
          if(i >= nmax) return;
          all[i]->w->setMaximumSize(w,h);
        }
        else if(strncmp(command,"setMultiSelection(",18) == 0) // set selection mode in QListView
        {
          int mode;
          sscanf(command,"setMultiSelection(%d,%d",&i,&mode);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            MyListView *lv = (MyListView *) all[i]->w;
            lv->setMultiSelection(mode);
          }
          else if(all[i]->type == TQListBox)
          {
            MyListBox *lb = (MyListBox *) all[i]->w;
            if      (mode == 1) lb->setSelectionMode(Q3ListBox::Multi);
            else if (mode == 2) lb->setSelectionMode(Q3ListBox::NoSelection);
            else                lb->setSelectionMode(Q3ListBox::Single);
          }
        }
        else if(strncmp(command,"setMinDate(",11) == 0) // set min date
        {
          int year, month, day;
          sscanf(command,"setMinDate(%d,%d,%d,%d)",&i,&year,&month,&day);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQDateEdit)
          {
            Q3DateEdit *e = (Q3DateEdit *) all[i]->w;
            e->setMinValue(QDate(year,month,day));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->dateEdit()->setMinValue(QDate(year,month,day));
          }
        }
        else if(strncmp(command,"setMaxDate(",11) == 0) // set max date
        {
          int year, month, day;
          sscanf(command,"setMaxDate(%d,%d,%d,%d)",&i,&year,&month,&day);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQDateEdit)
          {
            Q3DateEdit *e = (Q3DateEdit *) all[i]->w;
            e->setMaxValue(QDate(year,month,day));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->dateEdit()->setMaxValue(QDate(year,month,day));
          }
        }
        else if(strncmp(command,"setMinTime(",11) == 0) // set min time
        {
          int hour, minute, second, msec;
          sscanf(command,"setMinTime(%d,%d,%d,%d,%d)",&i,&hour,&minute,&second,&msec);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTimeEdit)
          {
            Q3TimeEdit *e = (Q3TimeEdit *) all[i]->w;
            e->setMinValue(QTime(hour,minute,second,msec));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->timeEdit()->setMinValue(QTime(hour,minute,second,msec));
          }
        }
        else if(strncmp(command,"setMaxTime(",11) == 0) // set max time
        {
          int hour, minute, second, msec;
          sscanf(command,"setMaxTime(%d,%d,%d,%d,%d)",&i,&hour,&minute,&second,&msec);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTimeEdit)
          {
            Q3TimeEdit *e = (Q3TimeEdit *) all[i]->w;
            e->setMaxValue(QTime(hour,minute,second,msec));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->timeEdit()->setMaxValue(QTime(hour,minute,second,msec));
          }
        }
        break;
      case 'N':
        if(strncmp(command,"setNumRows(",11) == 0) // set table num rows
        {
          int num;
          sscanf(command,"setNumRows(%d,%d",&i,&num);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            t->setNumRows(num);
          }
        }
        else if(strncmp(command,"setNumCols(",11) == 0) // set table num cols
        {
          int num;
          sscanf(command,"setNumCols(%d,%d",&i,&num);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            t->setNumCols(num);
          }
        }
        break;
      case 'O':
        if(strncmp(command,"setOpen(",8) == 0)
        {
          int open;
          sscanf(command,"setOpen(%d,%d",&i,&open);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            get_text(command,text);
            MyListView *l = (MyListView *) all[i]->w;
            l->setItemOpen(text,open, (MyListViewItem *) l->firstChild());
          }
        }
        break;
      case 'P':
        if(strncmp(command,"setPixmap(",10) == 0) // set pixmap
        {
          QPixmap pm;
          char *data;
          sscanf(command,"setPixmap(%d,%d,%d)",&i,&w,&h);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQPushButton)
          {
            QPushButton *b = (QPushButton*) all[i]->w;
            if(w==0 && h==0) b->setPixmap(pm);
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL)
              {
                pm = *temp;
                pm.setMask(pm.createHeuristicMask());
              }
              b->setPixmap(pm);
              if(data != NULL) delete data;
              if(temp != NULL) delete temp;
            }
          }
          else
          {
            if(w==0 && h==0) ;
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL) delete temp;
            }
          }
        }
        else if(strncmp(command,"setPaletteBackgroundColor(",26) == 0) // set background color
        {
          sscanf(command,"setPaletteBackgroundColor(%d,%d,%d,%d)",&i,&r,&g,&b);
          if(i < 0) return;
          if(i >= nmax) return;
          if(r==-1 && g==-1 && b==-1)
          {
            all[i]->w->unsetPalette();
          }
          else
          {
            if(all[i]->type == TQPushButton)
            {
              all[i]->w->setForegroundRole(QPalette::Button);
              all[i]->w->setPaletteForegroundColor(QColor(r,g,b));
            }
            else
            {
              all[i]->w->setPaletteBackgroundColor(QColor(r,g,b));
            }
          }
        }
        else if(strncmp(command,"setPaletteForegroundColor(",26) == 0) // set background color
        {
          sscanf(command,"setPaletteForegroundColor(%d,%d,%d,%d)",&i,&r,&g,&b);
          if(i < 0) return;
          if(i >= nmax) return;
          if(r==-1 && g==-1 && b==-1)
          {
            all[i]->w->unsetPalette();
          }
          else
          {
            all[i]->w->setPaletteForegroundColor(QColor(r,g,b));
          }
        }
        break;
      case 'Q':
        break;
      case 'R':
        break;
      case 'S':
        if(strncmp(command,"setSelected(",12) == 0) // set selection of ListView
        {
          int col, mode; // martin@bangieff "col not used ???"
          sscanf(command,"setSelected(%d,%d,%d",&i,&col,&mode);
          get_text(command,text); // text = path
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            MyListView *lv = (MyListView *) all[i]->w;
            lv->setSelected(mode,text);
          }
        }
        else if(strncmp(command,"setSource(",10) == 0) // set source of QTextBrowser
        {
          sscanf(command,"setSource(%d",&i);
          get_text(command,text); // text = html filename
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTextBrowser)
          {
            MyTextBrowser *t = (MyTextBrowser *) all[i]->w;
            t->setSource(temp + text);
          }
        }
        else if(strncmp(command,"setSorting(",11) == 0) // set sorting of ListView
        {
          int col,mode;
          sscanf(command,"setSorting(%d,%d,%d",&i,&col,&mode);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQListView)
          {
            MyListView *lv = (MyListView *) all[i]->w;
            lv->setSorting(col,mode);
          }
        }
        else if(strncmp(command,"setStyle(",9) == 0) // set style of QLabel QFrame
        {
          int shape,shadow,line_width,margin;
          sscanf(command,"setStyle(%d,%d,%d,%d,%d",&i,&shape,&shadow,&line_width,&margin);
          if(i < 0) return;
          if(i >= nmax) return;
          switch(shape)
          {
            case PV::NoFrame:
              shape = QFrame::NoFrame;
              break;
            case PV::Box:
              shape = QFrame::Box;
              break;
            case PV::Panel:
              shape = QFrame::Panel;
              break;
            case PV::WinPanel:
              shape = QFrame::WinPanel;
              break;
            case PV::HLine:
              shape = QFrame::HLine;
              break;
            case PV::VLine:
              shape = QFrame::VLine;
              break;
            case PV::StyledPanel:
              shape = QFrame::StyledPanel;
              break;
            case PV::PopupPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::MenuBarPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::ToolBarPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::LineEditPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::TabWidgetPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::GroupBoxPanel:
              shape = QFrame::NoFrame;
              break;
            case PV::MShape:
              shape = QFrame::NoFrame;
              break;
            default:
              shape = QFrame::NoFrame;
              break;
          }
          switch(shadow)
          {
            case PV::Plain:
              shadow = QFrame::Plain;
              break;
            case PV::Raised:
              shadow = QFrame::Raised;
              break;
            case PV::Sunken:
              shadow = QFrame::Sunken;
              break;
            case PV::MShadow:
              shadow = QFrame::Plain;
              break;
            default:
              shadow = QFrame::Plain;
              break;
          }
          if(all[i]->type == TQLabel)
          {
            QLabel *ptr = (QLabel *) all[i]->w;
            if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            if(line_width != -1) ptr->setLineWidth(line_width);
            if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQLineEdit)
          {
            //trollmurx MyLineEdit *ptr = (MyLineEdit *) all[i]->w;
            //trollmurx if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            //trollmurx if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            //trollmurx if(line_width != -1) ptr->setLineWidth(line_width);
            //trollmurx if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQFrame)
          {
            MyFrame *ptr = (MyFrame *) all[i]->w;
            if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            if(line_width != -1) ptr->setLineWidth(line_width);
            if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQGroupBox)
          {
            //trollmurx MyGroupBox *ptr = (MyGroupBox *) all[i]->w;
            //trollmurx if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            //trollmurx if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            //trollmurx if(line_width != -1) ptr->setLineWidth(line_width);
            //trollmurx if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQLCDNumber)
          {
            //trollmurx QLCDNumber *ptr = (QLCDNumber *) all[i]->w;
            //trollmurx if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            //trollmurx if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            //trollmurx if(line_width != -1) ptr->setLineWidth(line_width);
            //trollmurx if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQProgressBar)
          {
            //trollmurx MyProgressBar *ptr = (MyProgressBar *) all[i]->w;
            //trollmurx if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            //trollmurx if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            //trollmurx if(line_width != -1) //trollmurx ptr->setLineWidth(line_width);
            //trollmurx if(margin     != -1) ptr->setMargin(margin);
          }
          else if(all[i]->type == TQToolBox)
          {
            //trollmurx MyQToolBox *ptr = (MyQToolBox *) all[i]->w;
            //trollmurx if(shape      != -1) ptr->setFrameShape((Q3Frame::Shape) shape);
            //trollmurx if(shadow     != -1) ptr->setFrameShadow((Q3Frame::Shadow) shadow);
            //trollmurx if(line_width != -1) ptr->setLineWidth(line_width);
            //trollmurx if(margin     != -1) ptr->setMargin(margin);
          }
        }
        break;
      case 'T':
        if(strncmp(command,"setText(",8) == 0) // set widget text
        {
          sscanf(command,"setText(%d,",&i);
          if(i >= nmax && i > 0) return;
          get_text(command,text);
          if(strncmp(text,"alloc(",6) == 0) // allocate big buffer for big text
          {
            int len,ret;
            char *buf;
            sscanf(text,"alloc(%d,",&len);
            //printf("alloc(%d)\n",len);
            buf = new char[len+1];
            ret = tcp_rec_binary(s, buf, len);
            buf[len] = '\0';
            text = buf;
            buf[len] = '\0';
            delete [] buf;
          }
          if(i <= 0)
          {
            text = text.left(40);
            mainWindow->setCurrentFile(text);
          }
          else if(all[i]->type == TQPushButton)
          {
            QPushButton *b = (QPushButton*) all[i]->w;
            b->setText(text);
          }
          else if(all[i]->type == TQLabel)
          {
            QLabel *l = (QLabel*) all[i]->w;
            l->setText(text);
          }
          else if(all[i]->type == TQLineEdit)
          {
            MyLineEdit *e = (MyLineEdit*) all[i]->w;
            e->setText(text);
          }
          else if(all[i]->type == TQMultiLineEdit)
          {
            MyMultiLineEdit *e = (MyMultiLineEdit*) all[i]->w;
            e->setText(text);
          }
          else if(all[i]->type == TQComboBox)
          {
            MyComboBox *c = (MyComboBox*) all[i]->w;
            c->insertItem(text);
          }
          else if(all[i]->type == TQRadio)
          {
            MyRadioButton *c = (MyRadioButton*) all[i]->w;
            c->setText(text);
          }
          else if(all[i]->type == TQCheck)
          {
            MyCheckBox *c = (MyCheckBox *) all[i]->w;
            c->setText(text);
          }
          else if(all[i]->type == TQTextBrowser)
          {
            MyTextBrowser *c = (MyTextBrowser *) all[i]->w;
            // mur was here: make cursor not jumping
            //int x = c->contentsX();
            //int y = c->contentsY();
            QTextCursor cursor = c->textCursor();
            c->setText(text);
            c->setTextCursor(cursor);
            //c->setContentsPos(x,y);
          }
        }
        else if(strncmp(command,"setTableText(",13) == 0) // set text in table
        {
          sscanf(command,"setTableText(%d,%d,%d,",&i,&x,&y);
          get_text(command,text);
          if(i < 0) return;
          if(i >= nmax) return;
          else if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            if(x >= 0 && y >=0)
            {
              Q3TableItem *item;
              t->setText(x,y,text);
              if(t->wrap)
              {
                item = t->item(x,y);
                if(item != NULL)
                {
                  item->setWordWrap(true);
                  t->adjustRow(x);
                }
              }
            }
            else if(x<0 && y >= 0)
            {
              t->horizontalHeader()->setLabel(y,text);
            }
            else if(y<0 && x >= 0)
            {
              t->verticalHeader()->setLabel(x,text);
            }
          }
        }
        else if(strncmp(command,"setTablePixmap(",15) == 0) // set pixmap in table
        {
          char *data,text2[80];
          sscanf(command,"setTablePixmap(%d,%d,%d",&i,&x,&y);
          if(i < 0) return;
          if(i >= nmax) return;
          else if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            QPixmap pm;
            tcp_rec(s,text2,sizeof(text2));
            sscanf(text2,"(%d,%d)",&w,&h);
            if(w==0 && h==0) t->setPixmap(y,x,pm);
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL)
              {
                pm = *temp;
                pm.setMask(pm.createHeuristicMask());
              }
              t->setPixmap(y,x,pm);
              if(data != NULL) delete data;
              if(temp != NULL) delete temp;
            }
          }
          else
          {
            tcp_rec(s,text2,sizeof(text2));
            sscanf(text2,"(%d,%d)",&w,&h);
            if(w==0 && h==0) ;
            else
            {
              QImage *temp = readBmpFromSocket(w,h,&data);
              if(temp != NULL) delete temp;
            }
          }
        }
        else if(strncmp(command,"setTableCheckBox(",17) == 0) // set checkbox in table
        {
          int state;
          sscanf(command,"setTableCheckBox(%d,%d,%d,%d,",&i,&x,&y,&state);
          get_text(command,text);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            t->setTableCheckBox(x,y,state,text);
          }
        }
        else if(strncmp(command,"setTableComboBox(",17) == 0) // set combobox in table
        {
          int editable;
          sscanf(command,"setTableComboBox(%d,%d,%d,%d,",&i,&x,&y,&editable);
          get_text(command,text);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTable)
          {
            MyTable *t = (MyTable *) all[i]->w;
            t->setTableComboBox(x,y,editable,text);
          }
        }
        else if(strncmp(command,"setTabPosition(",15) == 0) // set tab position Top|Bottom
        {
          int pos;
          sscanf(command,"setTabPosition(%d,%d",&i,&pos);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTabWidget)
          {
            MyQTabWidget *tab = (MyQTabWidget *) all[i]->w;
            if     (pos == 1) tab->setTabPosition(QTabWidget::South);
            else if(pos == 2) tab->setTabPosition(QTabWidget::West);
            else if(pos == 3) tab->setTabPosition(QTabWidget::East);
            else              tab->setTabPosition(QTabWidget::North);
          }
        }
        else if(strncmp(command,"setTime(",8) == 0) // set time
        {
          int hour, minute, second, msec;
          sscanf(command,"setTime(%d,%d,%d,%d,%d)",&i,&hour,&minute,&second,&msec);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQTimeEdit)
          {
            Q3TimeEdit *e = (Q3TimeEdit *) all[i]->w;
            e->setTime(QTime(hour,minute,second,msec));
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->timeEdit()->setTime(QTime(hour,minute,second,msec));
          }
        }
        else if(strncmp(command,"setTimeEditDisplay(",19) == 0) // set time edit display
        {
          int hour, minute, second, ampm, mask;
          sscanf(command,"setTimeEditDisplay(%d,%d,%d,%d,%d)",&i,&hour,&minute,&second,&ampm);
          if(i < 0) return;
          if(i >= nmax) return;
          mask = 0;
          if(hour   == 1) mask |= Q3TimeEdit::Hours;
          if(minute == 1) mask |= Q3TimeEdit::Minutes;
          if(second == 1) mask |= Q3TimeEdit::Seconds;
          if(ampm   == 1) mask |= Q3TimeEdit::AMPM;
          if(all[i]->type == TQTimeEdit)
          {
            Q3TimeEdit *e = (Q3TimeEdit *) all[i]->w;
            e->setDisplay(mask);
          }
          else if(all[i]->type == TQDateTimeEdit)
          {
            Q3DateTimeEdit *e = (Q3DateTimeEdit *) all[i]->w;
            e->timeEdit()->setDisplay(mask);
          }
        }
        break;
      case 'U':
        break;
      case 'V':
        if(strncmp(command,"setValue(",9) == 0) // set value
        {
          int value;
          sscanf(command,"setValue(%d,%d",&i,&value);
          if(i < 0) return;
          if(i >= nmax) return;
          if(all[i]->type == TQSpinBox)
          {
            MySpinBox *b = (MySpinBox*) all[i]->w;
            b->setValue(value);
          }
          else if(all[i]->type == TQDial)
          {
            MyDial *b = (MyDial*) all[i]->w;
            b->setValue(value);
          }
          else if(all[i]->type == TQProgressBar)
          {
            MyProgressBar *b = (MyProgressBar*) all[i]->w;
            b->setValue(value);
          }
          else if(all[i]->type == TQSlider)
          {
            MySlider *b = (MySlider*) all[i]->w;
            b->setValue(value);
          }
          else if(all[i]->type == TQTabWidget)
          {
            MyQTabWidget *b = (MyQTabWidget*) all[i]->w;
            b->setCurrentPage(value);
          }
        }
        break;
      case 'W':
        break;
      case 'X':
        break;
      case 'Y':
        break;
      case 'Z':
        break;
      default:
        break;
    }
  }
  else if(strncmp(command,"selection(",10) == 0) // request selection in QListView
  {
    sscanf(command,"selection(%d",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQListView)
    {
      MyListView *lv = (MyListView *) all[i]->w;
      lv->slotSendSelected();
    }
    else if(all[i]->type == TQListBox)
    {
      MyListBox *lb = (MyListBox *) all[i]->w;
      lb->slotSendSelected();
    }
  }
  else if(strncmp(command,"show(",5) == 0) // show the widget
  {
    sscanf(command,"show(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    all[i]->w->show();
  }
  else if(strncmp(command,"startDefinition(",16) == 0) // startDefinition of new MainWidget
  {
    char *ptr;
    mainLayout = NULL;
    hasLayout = 0;
    if(opt.arg_debug) printf("startDefinition command=%s",command);
    sscanf(command,"startDefinition(%d)",&n);
    if(allBase == NULL)
    {
      if(opt.arg_debug) printf("startDefinition free(all)\n");
      if(all != NULL) free(all);
      v = new MyQWidget(s,0,mainWindow);
    }
    if(opt.arg_debug) printf("startDefinition malloc(all)\n");
    ptr = (char *) malloc(n*sizeof(ptr) + n*sizeof(ALL));
    if(ptr==NULL) qFatal("out of memory -> exit");
    all = (ALL **) ptr;
    ptr += n*sizeof(ptr);
    for(i=0; i<n; i++) all[i] = (ALL *) (ptr + i*sizeof(ALL));
    if(allBase == NULL)
    {
      if(opt.arg_debug) printf("startDefinition all[0]->w = v;\n");
      all[0]->w = v;
    }
    else
    {
      if(opt.arg_debug) printf("startDefinition all[0]->w = modalDialog;\n");
      all[0]->w = modalDialog;
    }

    nmax = n;
    if(opt.arg_debug) printf("startDefinition 2\n");
    for(i=1; i<n; i++)
    {
      all[i]->w = (QWidget *) NULL;
      all[i]->type = -1;
    }
    if(allBase != NULL)
    {
      if(opt.arg_debug) printf("startDefinition allBase != NULL\n");
      allModal = all;
      nmaxModal = nmax;
    }
    if(opt.arg_debug) printf("startDefinition end\n");
  }
  else if(strncmp(command,"save(",5) == 0) // save table as TAB separated text file
  {
    sscanf(command,"save(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTable)
    {
      MyTable *table = (MyTable *) all[i]->w;
      table->saveTextfile();
    }
  }
  else if(strncmp(command,"screenHint(",11) == 0) // adjust zoom factor
  {
    int w,h,dw;
    sscanf(command,"screenHint(%d,%d)",&w,&h);
    dw = QApplication::desktop()->width();
    // dh = QApplication::desktop()->height();
    // dw = (zoom / 100) w;
    // ( dw / w ) * 100 = zoom;
    if(w>0 && dw>0) zoom = (dw *100) / w;
  }
  else if(strncmp(command,"statusMessage(",14) == 0) // output status message
  {
    int r,g,b;
    sscanf(command,"statusMessage(%d,%d,%d",&r,&g,&b);
    get_text(command,text);
    if(r < 0) mainWindow->statusBar()->unsetPalette();
    else      mainWindow->statusBar()->setPaletteBackgroundColor(QColor(r,g,b));
    mainWindow->statusBar()->message(text);
  }
}

void Interpreter::interprett(const char *command)
{
  if(strncmp(command,"text(",5) == 0)
  {
    char buf[MAX_PRINTF_LENGTH];
    sscanf(command,"text(%d)",&i);
    buf[0] = '\0';
    if(all[i]->type == TQLabel)
    {
      QLabel *w = (QLabel *) all[i]->w;
      sprintf(buf,"text(%d,\"%s\")\n", i, (const char *) w->text().toUtf8());
      if(buf[0] != '\0') tcp_send(s,buf,strlen(buf));
    }
    else if(all[i]->type == TQPushButton)
    {
      MyQPushButton *w = (MyQPushButton *) all[i]->w;
      sprintf(buf,"text(%d,\"%s\")\n", i, (const char *) w->text().toUtf8());
      if(buf[0] != '\0') tcp_send(s,buf,strlen(buf));
    }
    else if(all[i]->type == TQLineEdit)
    {
      MyLineEdit *w = (MyLineEdit *) all[i]->w;
      sprintf(buf,"text(%d,\"%s\")\n", i, (const char *) w->text().toUtf8());
      if(buf[0] != '\0') tcp_send(s,buf,strlen(buf));
    }
    else if(all[i]->type == TQComboBox)
    {
      MyComboBox *w = (MyComboBox *) all[i]->w;
      sprintf(buf,"text(%d,\"%s\")\n", i, (const char *) w->currentText().toUtf8());
      //sprintf(buf,"text(%d,\"%s\")\n", i, (const char *) w->currentText());
      if(buf[0] != '\0') tcp_send(s,buf,strlen(buf));
    }
    else if(all[i]->type == TQMultiLineEdit)
    {
      MyMultiLineEdit *w = (MyMultiLineEdit *) all[i]->w;
      w->slotSendToClipboard();
    }
  }
  else if(strncmp(command,"tableSetEnabled(",16) == 0)
  {
    int x,y,enabled;
    sscanf(command,"tableSetEnabled(%d,%d,%d,%d)",&i,&x,&y,&enabled);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTable)
    {
      Q3Table *t;
      Q3TableItem *item;
      t = (Q3Table *) all[i]->w;
      item = t->item(y,x);
      if(item == 0) 
      {
        t->setText(y,x,"");
        item = t->item(y,x);
      }
      if(item != 0) item->setEnabled(enabled);
    }
  }
  else if(strncmp(command,"tableSetHeaderResizeEnabled(",28) == 0)
  {
    int horizontal,enabled,section;
    sscanf(command,"tableSetHeaderResizeEnabled(%d,%d,%d,%d)",&i,&horizontal,&enabled,&section);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTable)
    {
      Q3Table *t;
      t = (Q3Table *) all[i]->w;
      if(horizontal == 1)
      {
        t->horizontalHeader()->setResizeEnabled(enabled,section);
      }
      else
      {
        t->verticalHeader()->setResizeEnabled(enabled,section);
      }
    }
  }
  else if(strncmp(command,"tableWrap(",10) == 0)
  {
    int wrap;
    sscanf(command,"tableWrap(%d,%d",&i,&wrap);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQTable)
    {
      MyTable *t = (MyTable *) all[i]->w;
      t->wrap = wrap;
    }
  }
  else if(strncmp(command,"tabOrder(",9) == 0)
  {
    int id1,id2;
    sscanf(command,"tabOrder(%d,%d",&id1,&id2);
    if(id1 >= 0 && id2 >= 0)
    {
      if(id1 < nmax && id2 < nmax)
      {
        QWidget::setTabOrder(all[id1]->w, all[id2]->w);
      }
    }
  }
}

void Interpreter::interpretv(const char *command)
{
#ifdef USE_VTK
  if(strncmp(command,"vtkTcl(",7) == 0) // interpret a VTK Tcl command
  {
    char text[MAX_PRINTF_LENGTH];
    sscanf(command,"vtkTcl(%d)",&i);
    tcp_rec(s,text,sizeof(text));
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQVtk)
    {
      pvVtkTclWidget *w = (pvVtkTclWidget *) all[i]->w;
      w->interpret(text);
      if(strstr(text,"renderer") != NULL && strstr(text,"Render") != NULL ) w->updateGL();
    }
  }
  else if(strncmp(command,"vtkTclScript(",13) == 0) // interpret a VTK Tcl file
  {
    sscanf(command,"vtkTclScript(%d,",&i);
    get_text(command,text);
    if(i < 0) return;
    if(i >= nmax) return;
    if(all[i]->type == TQVtk)
    {
      pvVtkTclWidget *w = (pvVtkTclWidget *) all[i]->w;
      w->interpretFile(text);
    }
  }
#endif
  if(command == NULL) return;
}

void Interpreter::interpretq(const char *command)
{
  if(command == NULL) return; // damn compiler warnings
#ifndef NO_QWT
  if(strncmp(command,"qpw(",4) == 0)
  {
    int ret;
    char qwtcommand[1024];
    sscanf(command,"qpw(%d)",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    QwtPlotWidget *pw = (QwtPlotWidget *) all[i]->w;
    ret = tcp_rec(s, qwtcommand, sizeof(qwtcommand)-1);
    if(ret == -1) return;
    if(all[i]->type != TQwtPlotWidget) return;
    if(opt.arg_debug) printf("qwtplot=%s",qwtcommand);
    if(strncmp(qwtcommand,"setCurveData(",13) == 0)
    {
      int cnt,c,count;
      char param[256];
      sscanf(qwtcommand,"setCurveData(%d,%d",&c,&count);
      if(count <= 0) return;
      double *x = new double[count];
      double *y = new double[count];
      for(cnt=0; cnt<count; cnt++)
      {
        ret = tcp_rec(s, param, sizeof(param)-1);
        if(ret == -1)
        {
          delete [] x;
          delete [] y;
          return;
        }
        sscanf(param,"%lf %lf",&x[cnt],&y[cnt]);
      }
      pw->interpret(qwtcommand,x,y);
      delete [] x;
      delete [] y;
    }
    else
    {
      pw->interpret(qwtcommand);
    }
  }
  else if(strncmp(command,"qwt(",4) == 0) // Commands for the QwtWidgets
  {
    int ret;
    char qwtcommand[1024];
    sscanf(command,"qwt(%d)",&i);
    ret = tcp_rec(s, qwtcommand, sizeof(qwtcommand)-1);
    if(ret == -1) return;
    if(i < 0) return;
    if(i >= nmax) return;
    if(opt.arg_debug) printf("qwtwidget=%s",qwtcommand);
    if(all[i]->type == TQwtScale)
    {
      // QwtScale
      QwtScaleWidget *wi = (QwtScaleWidget *) all[i]->w;
      if(strncmp(qwtcommand,"setTitle(",9) == 0)
      {
        get_text(qwtcommand,text);
        wi->setTitle(text);
      }
      else if(strncmp(qwtcommand,"setTitleColor(",14) == 0)
      {
        int r,g,b;
        sscanf(qwtcommand,"setTitleColor(%d,%d,%d",&r,&g,&b);
        wi->title().setColor(QColor(r,g,b));
      }
      else if(strncmp(qwtcommand,"setTitleFont(",13) == 0)
      {
        int pointsize,bold,italic,underline,strikeout;
        sscanf(qwtcommand,"setTitleFont(%d,%d,%d,%d,%d",&pointsize,&bold,&italic,&underline,&strikeout);
        get_text(qwtcommand,text); // text = family of font
        QFont font(text, pointsize);
        if(bold      == 1) font.setBold(true);
        if(italic    == 1) font.setItalic(true);
        if(underline == 1) font.setUnderline(true);
        if(strikeout == 1) font.setStrikeOut(true);
        wi->title().setFont(font);
      }
      else if(strncmp(qwtcommand,"setTitleAlignment(",18) == 0)
      {
        int flags;
        sscanf(qwtcommand,"setTitleAlignment(%d",&flags);
        //murx wi->setTitleAlignment(flags);
      }
      else if(strncmp(qwtcommand,"setBorderDist(",14) == 0)
      {
        int start,end;
        sscanf(qwtcommand,"setBorderDist(%d,%d",&start,&end);
        wi->setBorderDist(start,end);
      }
      else if(strncmp(qwtcommand,"setBaselineDist(",16) == 0)
      {
        int bd;
        sscanf(qwtcommand,"setBaselineDist(%d",&bd);
        wi->setBaselineDist(bd);
      }
      else if(strncmp(qwtcommand,"setScaleDiv(",12) == 0)
      {
        double lBound, hBound, step;
        int maxMaj, maxMin, log, ascend;
        QwtScaleDiv sd;
        sscanf(qwtcommand,"setScaleDiv(%lf,%lf,%d,%d,%d,%lf,%d",
                           &lBound,&hBound,&maxMaj,&maxMin,&log,&step,&ascend);
        //murx sd.rebuild(lBound,hBound,maxMaj,maxMin,log,step,ascend);
        //murx wi->setScaleDiv(sd);
      }
      else if(strncmp(qwtcommand,"setScaleDraw(",13) == 0)
      {
        //setScaleDraw(QwtScaleDraw *);
      }
      else if(strncmp(qwtcommand,"setLabelFormat(",15) == 0)
      {
        int f,prec,fieldWidth;
        sscanf(qwtcommand,"setLabelFormat(%d,%d,%d",&f,&prec,&fieldWidth);
        //murx wi->setLabelFormat(f,prec,fieldWidth);
      }
      else if(strncmp(qwtcommand,"setLabelAlignment(",18) == 0)
      {
        int align;
        sscanf(qwtcommand,"setLabelAlignment(%d",&align);
        //murx wi->setLabelAlignment(align);
        wi->setLabelAlignment((Qt::Alignment) align);
      }
      else if(strncmp(qwtcommand,"setLabelRotation(",17) == 0)
      {
        double rotation;
        sscanf(qwtcommand,"setLabelRotation(%lf",&rotation);
        wi->setLabelRotation(rotation);
      }
      else if(strncmp(qwtcommand,"setPosition(",12) == 0)
      {
        int pos;
        sscanf(qwtcommand,"setPosition(%d",&pos);
        //murx wi->setPosition((QwtScale::Position) pos);
        if(pos == PV::Horizontal)
          wi->setAlignment(QwtScaleDraw::BottomScale);
        else
          wi->setAlignment(QwtScaleDraw::LeftScale);
      }
    }
    else if(all[i]->type == TQwtThermo)
    {
      // QwtThermo
      QwtThermo *wi = (QwtThermo *) all[i]->w;
      if(strncmp(qwtcommand,"setScale(",9) == 0)
      {
        double min,max,step;
        int logarithmic;
        sscanf(qwtcommand,"setScale(%lf,%lf,%lf,%d",&min,&max,&step,&logarithmic);
        //murx wi->setScale(min,max,step,logarithmic);
        wi->setScale(min,max,step);
      }
      else if(strncmp(qwtcommand,"setOrientation(",15) == 0)
      {
        int ori,pos;
        sscanf(qwtcommand,"setOrientation(%d,%d",&ori,&pos);
        wi->setOrientation((Qt::Orientation) ori, (QwtThermo::ScalePos) pos);
      }
      else if(strncmp(qwtcommand,"setBorderWidth(",15) == 0)
      {
        int w;
        sscanf(qwtcommand,"setBorderWidth(%d",&w);
        wi->setBorderWidth(w);
      }
      else if(strncmp(qwtcommand,"setFillColor(",13) == 0)
      {
        int r,g,b;
        sscanf(qwtcommand,"setFillColor(%d,%d,%d",&r,&g,&b);
        wi->setFillColor(QColor(r,g,b));
      }
      else if(strncmp(qwtcommand,"setAlarmColor(",14) == 0)
      {
        int r,g,b;
        sscanf(qwtcommand,"setAlarmColor(%d,%d,%d",&r,&g,&b);
        wi->setAlarmColor(QColor(r,g,b));
      }
      else if(strncmp(qwtcommand,"setAlarmLevel(",14) == 0)
      {
        double v;
        sscanf(qwtcommand,"setAlarmLevel(%lf",&v);
        wi->setAlarmLevel(v);
      }
      else if(strncmp(qwtcommand,"setAlarmEnabled(",16) == 0)
      {
        int tf;
        sscanf(qwtcommand,"setAlarmEnabled(%d",&tf);
        wi->setAlarmEnabled(tf);
      }
      else if(strncmp(qwtcommand,"setPipeWidth(",13) == 0)
      {
        int w;
        sscanf(qwtcommand,"setPipeWidth(%d",&w);
        wi->setPipeWidth(w);
      }
      else if(strncmp(qwtcommand,"setRange(",9) == 0)
      {
        double vmin,vmax;
        sscanf(qwtcommand,"setRange(%lf,%lf",&vmin,&vmax);
        wi->setRange(vmin,vmax);
      }
      else if(strncmp(qwtcommand,"setMargin(",10) == 0)
      {
        int m;
        sscanf(qwtcommand,"setMargin(%d",&m);
        wi->setMargin(m);
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
    else if(all[i]->type == TQwtKnob)
    {
      // QwtKnob
      QwtKnob *wi = (QwtKnob *) all[i]->w;
      if(strncmp(qwtcommand,"setScale(",9) == 0)
      {
        double min,max,step;
        int logarithmic;
        sscanf(qwtcommand,"setScale(%lf,%lf,%lf,%d",&min,&max,&step,&logarithmic);
        //murx wi->setScale(min,max,step,logarithmic);
        wi->setScale(min,max,step);
      }
      else if(strncmp(qwtcommand,"setMass(",8) == 0)
      {
        float mass;
        sscanf(qwtcommand,"setMass(%f",&mass);
        wi->setMass(w);
      }
      else if(strncmp(qwtcommand,"setOrientation(",15) == 0)
      {
        int o;
        sscanf(qwtcommand,"setOrientation(%d",&o);
        wi->setOrientation((Qt::Orientation) o);
      }
      else if(strncmp(qwtcommand,"setReadOnly(",12) == 0)
      {
        int w;
        sscanf(qwtcommand,"setReadOnly(%d",&w);
        wi->setReadOnly(w);
      }
      else if(strncmp(qwtcommand,"setKnobWidth(",13) == 0)
      {
        int w;
        sscanf(qwtcommand,"setKnobWidth(%d",&w);
        wi->setKnobWidth(w);
      }
      else if(strncmp(qwtcommand,"setTotalAngle(",14) == 0)
      {
        double angle;
        sscanf(qwtcommand,"setTotalAngle(%lf",&angle);
        wi->setTotalAngle(angle);
      }
      else if(strncmp(qwtcommand,"setBorderWidth(",15) == 0)
      {
        int bw;
        sscanf(qwtcommand,"setBorderWidth(%d",&bw);
        wi->setBorderWidth(bw);
      }
      else if(strncmp(qwtcommand,"setSymbol(",10) == 0)
      {
        int s;
        sscanf(qwtcommand,"setSymbol(%d",&s);
        wi->setSymbol((QwtKnob::Symbol) s);
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
    else if(all[i]->type == TQwtCounter)
    {
      // QwtCounter
      QwtCounter *wi = (QwtCounter *) all[i]->w;
      if(strncmp(qwtcommand,"setStep(",8) == 0)
      {
        double s;
        sscanf(qwtcommand,"setStep(%lf",&s);
        wi->setStep(s);
      }
      else if(strncmp(qwtcommand,"setMinValue(",12) == 0)
      {
        double m;
        sscanf(qwtcommand,"setMinValue(%lf",&m);
        wi->setMinValue(m);
      }
      else if(strncmp(qwtcommand,"setMaxValue(",12) == 0)
      {
        double m;
        sscanf(qwtcommand,"setMaxValue(%lf",&m);
        wi->setMaxValue(m);
      }
      else if(strncmp(qwtcommand,"setStepButton1(",15) == 0)
      {
        int n;
        sscanf(qwtcommand,"setStepButton1(%d",&n);
        wi->setStepButton1(n);
      }
      else if(strncmp(qwtcommand,"setStepButton2(",15) == 0)
      {
        int n;
        sscanf(qwtcommand,"setStepButton2(%d",&n);
        wi->setStepButton2(n);
      }
      else if(strncmp(qwtcommand,"setStepButton3(",15) == 0)
      {
        int n;
        sscanf(qwtcommand,"setStepButton3(%d",&n);
        wi->setStepButton3(n);
      }
      else if(strncmp(qwtcommand,"setNumButtons(",14) == 0)
      {
        int n;
        sscanf(qwtcommand,"setNumButtons(%d",&n);
        wi->setNumButtons(n);
      }
      else if(strncmp(qwtcommand,"setIncSteps(",12) == 0)
      {
        int btn,n;
        sscanf(qwtcommand,"setIncSteps(%d,%d",&btn,&n);
        wi->setIncSteps((QwtCounter::Button) btn, n);
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
    else if(all[i]->type == TQwtWheel)
    {
      // QwtWheel
      QwtWheel *wi = (QwtWheel *) all[i]->w;
      if(strncmp(qwtcommand,"setMass(",8) == 0)
      {
        float mass;
        sscanf(qwtcommand,"setMass(%f",&mass);
        wi->setMass(w);
      }
      else if(strncmp(qwtcommand,"setOrientation(",15) == 0)
      {
        int o;
        sscanf(qwtcommand,"setOrientation(%d",&o);
        if(o == PV::Horizontal)
          wi->setOrientation(Qt::Horizontal);
        else
          wi->setOrientation(Qt::Vertical);
        //wi->setOrientation((Qt::Orientation) o);
      }
      else if(strncmp(qwtcommand,"setReadOnly(",12) == 0)
      {
        int w;
        sscanf(qwtcommand,"setReadOnly(%d",&w);
        wi->setReadOnly(w);
      }
      else if(strncmp(qwtcommand,"setTotalAngle(",14) == 0)
      {
        double angle;
        sscanf(qwtcommand,"setTotalAngle(%lf",&angle);
        wi->setTotalAngle(angle);
      }
      else if(strncmp(qwtcommand,"setTickCnt(",11) == 0)
      {
        int cnt;
        sscanf(qwtcommand,"setTickCnt(%d",&cnt);
        wi->setTickCnt(cnt);
      }
      else if(strncmp(qwtcommand,"setViewAngle(",13) == 0)
      {
        double angle;
        sscanf(qwtcommand,"setViewAngle(%lf",&angle);
        wi->setViewAngle(angle);
      }
      else if(strncmp(qwtcommand,"setInternalBorder(",18) == 0)
      {
        int width;
        sscanf(qwtcommand,"setInternalBorder(%d",&width);
        wi->setInternalBorder(width);
      }
      else if(strncmp(qwtcommand,"setMass(",8) == 0)
      {
        double val;
        sscanf(qwtcommand,"setMass(%lf",&val);
        wi->setMass(val);
      }
      else if(strncmp(qwtcommand,"setWheelWidth(",14) == 0)
      {
        int width;
        sscanf(qwtcommand,"setWheelWidth(%d",&width);
        wi->setWheelWidth(width);
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
    else if(all[i]->type == TQwtSlider)
    {
      // QwtSlider
      QwtSlider *wi = (QwtSlider *) all[i]->w;
      if(strncmp(qwtcommand,"setScale(",9) == 0)
      {
        double min,max,step;
        int logarithmic;
        sscanf(qwtcommand,"setScale(%lf,%lf,%lf,%d",&min,&max,&step,&logarithmic);
        //murx wi->setScale(min,max,step,logarithmic);
        wi->setScale(min,max,step);
      }
      else if(strncmp(qwtcommand,"setMass(",8) == 0)
      {
        float mass;
        sscanf(qwtcommand,"setMass(%f",&mass);
        wi->setMass(w);
      }
      else if(strncmp(qwtcommand,"setOrientation(",15) == 0)
      {
        int o;
        sscanf(qwtcommand,"setOrientation(%d",&o);
        if(o == PV::Horizontal)
          wi->setOrientation(Qt::Horizontal);
        else
          wi->setOrientation(Qt::Vertical);
      }
      else if(strncmp(qwtcommand,"setReadOnly(",12) == 0)
      {
        int w;
        sscanf(qwtcommand,"setReadOnly(%d",&w);
        wi->setReadOnly(w);
      }
      else if(strncmp(qwtcommand,"setBgStyle(",11) == 0)
      {
        int st;
        sscanf(qwtcommand,"setBgStyle(%d",&st);
        wi->setBgStyle((QwtSlider::BGSTYLE) st);
      }
      else if(strncmp(qwtcommand,"setScalePos(",12) == 0)
      {
        int st;
        sscanf(qwtcommand,"setScalePos(%d",&st);
        switch(st)
        {
          case PV::SliderLeft:
            wi->setScalePosition(QwtSlider::Left);
            break;
          case PV::SliderRight:
            wi->setScalePosition(QwtSlider::Right);
            break;
          case PV::SliderTop:
            wi->setScalePosition(QwtSlider::Top);
            break;
          case PV::SliderBottom:
            wi->setScalePosition(QwtSlider::Bottom);
            break;
          default:
            wi->setScalePosition(QwtSlider::None);
            break;
        }
      }
      else if(strncmp(qwtcommand,"setThumbLength(",15) == 0)
      {
        int l;
        sscanf(qwtcommand,"setThumbLength(%d",&l);
        wi->setThumbLength(l);
      }
      else if(strncmp(qwtcommand,"setThumbWidth(",14) == 0)
      {
        int w;
        sscanf(qwtcommand,"setThumbWidth(%d",&w);
        wi->setThumbWidth(w);
      }
      else if(strncmp(qwtcommand,"setBorderWidth(",15) == 0)
      {
        int bw;
        sscanf(qwtcommand,"setBorderWidth(%d",&bw);
        wi->setBorderWidth(bw);
      }
      else if(strncmp(qwtcommand,"setMargins(",11) == 0)
      {
        int x,y;
        sscanf(qwtcommand,"setMargins(%d,%d",&x,&y);
        wi->setMargins(x,y);
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
    else if(all[i]->type == TQwtCompass)
    {
      // QwtCompass
      QwtCompass *wi = (QwtCompass *) all[i]->w;
      if(strncmp(qwtcommand,"setMass(",8) == 0)
      {
        float mass;
        sscanf(qwtcommand,"setMass(%f",&mass);
        wi->setMass(w);
      }
      else if(strncmp(qwtcommand,"setOrientation(",15) == 0)
      {
        int o;
        sscanf(qwtcommand,"setOrientation(%d",&o);
        wi->setOrientation((Qt::Orientation) o);
      }
      else if(strncmp(qwtcommand,"setReadOnly(",12) == 0)
      {
        int w;
        sscanf(qwtcommand,"setReadOnly(%d",&w);
        wi->setReadOnly(w);
      }
      else if(strncmp(qwtcommand,"setSimpleCompassRose(",21) == 0)
      {
        int numThorns,numThornLevels;
        double width;
        sscanf(qwtcommand,"setSimpleCompassRose(%d,%d,%lf",&numThorns,&numThornLevels,&width);
        QwtSimpleCompassRose *rose = new QwtSimpleCompassRose(numThorns, numThornLevels);
        rose->setWidth(width);
        wi->setRose(rose);
      }
      else if(strncmp(qwtcommand,"setLabelMap(",12) == 0)
      {
        //setLabelMap(const QMap<double, QString> &map);
      }
      else if(strncmp(qwtcommand,"setFrameShadow(",15) == 0)
      {
        int s;
        sscanf(qwtcommand,"setFrameShadow(%d",&s);
        wi->setFrameShadow((QwtDial::Shadow) s);
      }
      else if(strncmp(qwtcommand,"showBackground(",15) == 0)
      {
        int b;
        sscanf(qwtcommand,"showBackground(%d",&b);
        wi->showBackground(b);
      }
      else if(strncmp(qwtcommand,"setLineWidth(",13) == 0)
      {
        int w;
        sscanf(qwtcommand,"setLineWidth(%d",&w);
        wi->setLineWidth(w);
      }
      else if(strncmp(qwtcommand,"setMode(",8) == 0)
      {
        int m;
        sscanf(qwtcommand,"setMode(%d",&m);
        wi->setMode((QwtDial::Mode) m);
      }
      else if(strncmp(qwtcommand,"setWrapping(",12) == 0)
      {
        int w;
        sscanf(qwtcommand,"setWrapping(%d",&w);
        wi->setWrapping(w);
      }
      else if(strncmp(qwtcommand,"setScale(",9) == 0)
      {
        int maxMajIntv,maxMinIntv;
        double step;
        sscanf(qwtcommand,"setScale(%d,%d,%lf",&maxMajIntv,&maxMinIntv,&step);
        wi->setScale(maxMajIntv,maxMinIntv,step);
      }
      else if(strncmp(qwtcommand,"setScaleArc(",12) == 0)
      {
        double min,max;
        sscanf(qwtcommand,"setScaleArc(%lf,%lf",&min,&max);
        wi->setScaleArc(min,max);
      }
      else if(strncmp(qwtcommand,"setOrigin(",10) == 0)
      {
        double o;
        sscanf(qwtcommand,"setOrigin(%lf",&o);
        wi->setOrigin(o);
      }
      else if(strncmp(qwtcommand,"setNeedle(",10) == 0)
      {
        int which,r1,r2,r3,g1,g2,g3,b1,b2,b3;
        sscanf(qwtcommand,"setNeedle(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d",&which,
                                       &r1,&g1,&b1,
                                       &r2,&g2,&b2,
                                       &r3,&g3,&b3);
        //murx if(which==1) wi->setNeedle(new QwtCompassNeedle1(QColor(r1,g1,b1)));
        //murx if(which==2) wi->setNeedle(new QwtCompassNeedle2(QColor(r1,g1,b1),QColor(r2,g2,b2)));
        //murx if(which==3) wi->setNeedle(new QwtCompassNeedle3(QColor(r1,g1,b1),QColor(r2,g2,b2)));
        //murx if(which==4) wi->setNeedle(new QwtCompassNeedle4(QColor(r1,g1,b1),QColor(r2,g2,b2),QColor(r3,g3,b3)));
        //murx if(which==5) wi->setNeedle(new QwtCompassLineNeedle(QColor(r1,g1,b1)));
        if(which==1) wi->setNeedle(new QwtCompassWindArrow(QwtCompassWindArrow::Style2, QColor(r1,g1,b1), QColor(r2,g2,b2)));
        else if(which==2) wi->setNeedle(new QwtCompassMagnetNeedle(QwtCompassMagnetNeedle::ThinStyle, QColor(r1,g1,b1), QColor(r2,g2,b2)));
        else wi->setNeedle(new QwtDialSimpleNeedle(QwtDialSimpleNeedle::Ray,true, QColor(r1,g1,b1), QColor(r2,g2,b2)));
      }
      else if(strncmp(qwtcommand,"setValue(",9) == 0)
      {
        double val;
        sscanf(qwtcommand,"setValue(%lf",&val);
        wi->setValue(val);
      }
    }
  }
#endif //#ifndef NO_QWT
}

void Interpreter::interpretQ(const char *command)
{
  if(strncmp(command,"QWidget(",8) == 0) // create a new QWidget
  {
    sscanf(command,"QWidget(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    //all[i]->w = new MyQWidget(s,i,all[p]->w);
    all[i]->w = new MyQWidget(s,i,0);
    all[i]->type = TQWidget;
  }
  else if(strncmp(command,"QPushButton(",12) == 0) // create a new QPushButton
  {
    sscanf(command,"QPushButton(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQPushButton(s,i,all[p]->w);
    all[i]->type = TQPushButton;
  }
  else if(strncmp(command,"QLabel(",7) == 0) // create a new QLabel
  {
    sscanf(command,"QLabel(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new QLabel(all[p]->w);
    all[i]->type = TQLabel;
  }
  else if(strncmp(command,"QLineEdit(",10) == 0) // create a new QLineEdit
  {
    MyLineEdit *ptr;
    sscanf(command,"QLineEdit(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyLineEdit(s,i,all[p]->w);
    all[i]->type = TQLineEdit;
    ptr = (MyLineEdit *) all[i]->w;
  }
  else if(strncmp(command,"QComboBox(",10) == 0) // create a new QComboBox
  {
    int editable,policy;
    QComboBox *ptr;
    sscanf(command,"QComboBox(%d,%d,%d,%d)",&i,&p,&editable,&policy);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyComboBox(s,i,all[p]->w);
    all[i]->type = TQComboBox;

    ptr = (MyComboBox *) all[i]->w;
    if(editable == 1) ptr->setEditable(true);
    if     (policy == 0) ptr->setInsertionPolicy(QComboBox::NoInsert);
    else if(policy == 1) ptr->setInsertionPolicy(QComboBox::InsertAtTop);
    else if(policy == 2) ptr->setInsertionPolicy(QComboBox::InsertAtCurrent);
    else if(policy == 3) ptr->setInsertionPolicy(QComboBox::InsertAtBottom);
    else if(policy == 4) ptr->setInsertionPolicy(QComboBox::InsertAfterCurrent);
    else if(policy == 5) ptr->setInsertionPolicy(QComboBox::InsertBeforeCurrent);
  }
  else if(strncmp(command,"QLCDNumber(",11) == 0) // create a new QLCDNumber
  {
    uint numDigits,segmentStyle,mode;

    sscanf(command,"QLCDNumber(%d,%d,%d,%d,%d)",&i,&p,&numDigits,&segmentStyle,&mode);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new QLCDNumber(numDigits,all[p]->w);
    all[i]->type = TQLCDNumber;
    QLCDNumber *ptr = (QLCDNumber *) all[i]->w;
    switch(mode)
    {
      case 0:
        ptr->setMode(QLCDNumber::HEX);
        break;
      case 1:
        ptr->setMode(QLCDNumber::DEC);
        break;
      case 2:
        ptr->setMode(QLCDNumber::OCT);
        break;
      case 3:
        ptr->setMode(QLCDNumber::BIN);
        break;
      default:
        break;
    }
    switch(segmentStyle)
    {
      case 0:
        ptr->setSegmentStyle(QLCDNumber::Outline);
        break;
      case 1:
        ptr->setSegmentStyle(QLCDNumber::Filled);
        break;
      case 2:
        ptr->setSegmentStyle(QLCDNumber::Flat);
        break;
      default:
        break;
    }
  }
  else if(strncmp(command,"QDrawWidget(",12) == 0) // create a new QDrawWidget
  {
    sscanf(command,"QDrawWidget(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;

    all[i]->w = (QWidget *) new QDrawWidget(all[p]->w,0,0,s,i);
    all[i]->type = TQDraw;
  }
  else if(strncmp(command,"QImage(",7) == 0) // create a new QImage
  {
    int  depth;
    sscanf(command,"QImage(%d,%d,%d,%d,%d",&i,&p,&w,&h,&depth);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    if(depth == 0) // image format other than 8bpp bmp
    {
      QString filename;
      get_text(command,text); // text = filename of image
      filename = temp + text;
      QImageWidget *imagewidget = new QImageWidget(s, i, all[p]->w);
      all[i]->w = (QWidget *) imagewidget;
      all[i]->type = TQImage;
      imagewidget->setImage(filename);
      return;
    }
    else if(depth == 8)
    {
      char *data;
      QImage *temp = readBmpFromSocket(w,h,&data);
      if(temp != NULL)
      {
        QImageWidget *imagewidget = new QImageWidget(s, i, all[p]->w);
        all[i]->w = (QWidget *) imagewidget;
        all[i]->type = TQImage;
        temp->setAlphaBuffer(true);
        imagewidget->setImage(temp);
        if(data != NULL) delete data;
        if(temp != NULL) delete temp;
      }
    }
  }
#ifdef USE_OPEN_GL
  else if(strncmp(command,"QPvGlWidget(",12) == 0) // create a new PvGlWidget
  {
    sscanf(command,"QPvGlWidget(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new PvGLWidget(all[p]->w,i,s);
    all[i]->type = TQGl;
  }
#endif
#ifdef USE_VTK
  else if(strncmp(command,"QVtkTclWidget(",14) == 0) // create a new pvVtkTclWidget
  {
    sscanf(command,"QVtkTclWidget(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new pvVtkTclWidget(all[p]->w,"vtkWidget",i,s);
    all[i]->type = TQVtk;
  }
#endif
#ifndef NO_QWT
  else if(strncmp(command,"QwtPlotWidget(",14) == 0) // create a new QwtPlotWidget
  {
    int nCurves, nMarker;
    sscanf(command,"QwtPlotWidget(%d,%d,%d,%d",&i,&p,&nCurves,&nMarker);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new QwtPlotWidget(s,i,all[p]->w,nCurves,nMarker);
    all[i]->type = TQwtPlotWidget;
  }
  else if(strncmp(command,"QwtScale(",9) == 0) // create a new QwtScale
  {
    int pos;
    sscanf(command,"QwtScale(%d,%d,%d",&i,&p,&pos);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtScale(s,i,(QwtScaleDraw::Alignment) pos, all[p]->w);
    all[i]->type = TQwtScale;
  }
  else if(strncmp(command,"QwtThermo(",10) == 0) // create a new QwtThermo
  {
    sscanf(command,"QwtThermo(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtThermo(s,i,all[p]->w);
    all[i]->type = TQwtThermo;
  }
  else if(strncmp(command,"QwtKnob(",8) == 0) // create a new QwtKnob
  {
    sscanf(command,"QwtKnob(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtKnob(s,i,all[p]->w);
    all[i]->type = TQwtKnob;
  }
  else if(strncmp(command,"QwtCounter(",11) == 0) // create a new QwtCounter
  {
    sscanf(command,"QwtCounter(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtCounter(s,i,all[p]->w);
    all[i]->type = TQwtCounter;
  }
  else if(strncmp(command,"QwtWheel(",9) == 0) // create a new QwtWheel
  {
    sscanf(command,"QwtWheel(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtWheel(s,i,all[p]->w);
    all[i]->type = TQwtWheel;
  }
  else if(strncmp(command,"QwtSlider(",10) == 0) // create a new QwtSlider
  {
    sscanf(command,"QwtSlider(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtSlider(s,i,all[p]->w);
    all[i]->type = TQwtSlider;
  }
  else if(strncmp(command,"QwtCompass(",11) == 0) // create a new QwtCompass
  {
    sscanf(command,"QwtCompass(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQwtCompass(s,i,all[p]->w);
    all[i]->type = TQwtCompass;
  }
#endif //#ifndef NO_QWT
  else if(strncmp(command,"QSlider(",8) == 0) // create a new Slider
  {
    int minValue; int maxValue; int pageStep; int value; int orientation, qtori;
    sscanf(command,"QSlider(%d,%d,%d,%d,%d,%d,%d)",&i,&p,&minValue,&maxValue,&pageStep,&value,&orientation);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    if(orientation == PV::Horizontal) qtori = Qt::Horizontal;
    else                              qtori = Qt::Vertical;
    all[i]->w = (QWidget *) new MySlider(s,i,minValue,maxValue,pageStep,value,(Qt::Orientation) qtori,all[p]->w);
    all[i]->type = TQSlider;
  }
  else if(strncmp(command,"QRadio(",7) == 0) // create a new radio button
  {
    sscanf(command,"QRadio(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyRadioButton(s,i,all[p]->w);
    all[i]->type = TQRadio;
  }
  else if(strncmp(command,"QButtonGroup(",13) == 0) // create a new button group
  {
    int columns; int o;
    sscanf(command,"QButtonGroup(%d,%d,%d,%d,",&i,&p,&columns,&o);
    get_text(command,text);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyButtonGroup(s,i,columns,(Qt::Orientation) o,text,all[p]->w);
    all[i]->type = TQButtonGroup;
  }
  else if(strncmp(command,"QCheckBox(",10) == 0) // create a new check box
  {
    sscanf(command,"QCheckBox(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyCheckBox(s,i,all[p]->w);
    all[i]->type = TQCheck;
  }
  else if(strncmp(command,"QFrame(",7) == 0) // create a new frame
  {
    int shape; int shadow; int line_width; int margin;
    sscanf(command,"QFrame(%d,%d,%d,%d,%d,%d)",&i,&p,&shape,&shadow,&line_width,&margin);

    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyFrame(s,i,shape,shadow,line_width,margin,all[p]->w);
    all[i]->type = TQFrame;
  }
  else if(strncmp(command,"QTabWidget(",11) == 0) // create a new QTabWidget
  {
    sscanf(command,"QTabWidget(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QTabWidget *) new MyQTabWidget(s,i,all[p]->w);
    all[i]->type = TQTabWidget;
  }
  else if(strncmp(command,"QToolBox(",9) == 0) // create a new QToolBox
  {
    sscanf(command,"QToolBox(%d,%d)",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QToolBox *) new MyQToolBox(s,i,all[p]->w);
    all[i]->type = TQToolBox;
  }
  else if(strncmp(command,"QGroupBox(",10) == 0) // create a new QGroupBox
  {
    int columns; int o;

    sscanf(command,"QGroupBox(%d,%d,%d,%d,",&i,&p,&columns,&o);
    get_text(command,text);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyGroupBox(s,i,columns,(Qt::Orientation) o,text,all[p]->w);
    all[i]->type = TQGroupBox;
  }
  else if(strncmp(command,"QListBox(",9) == 0) // create a new QListBox
  {
    sscanf(command,"QListBox(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyListBox(s,i,all[p]->w);
    all[i]->type = TQListBox;
  }
  else if(strncmp(command,"QTable(",7) == 0) // create a new QTable
  {
    int row; int col;
    sscanf(command,"QTable(%d,%d,%d,%d",&i,&p,&row,&col);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyTable(s,i,row,col,all[p]->w);
    all[i]->type = TQTable;
  }
  else if(strncmp(command,"QSpinBox(",9) == 0) // create a new QSpinBox
  {
    int min; int max; int step;
    sscanf(command,"QSpinBox(%d,%d,%d,%d,%d",&i,&p,&min,&max,&step);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MySpinBox(s,i,min,max,step,all[p]->w);
    all[i]->type = TQSpinBox;
  }
  else if(strncmp(command,"QDial(",6) == 0) // create a new QDial
  {
    int min; int max; int pageStep; int value;

    sscanf(command,"QDial(%d,%d,%d,%d,%d,%d",&i,&p,&min,&max,&pageStep,&value);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyDial(s,i,min,max,pageStep,value,all[p]->w);
    all[i]->type = TQDial;
  }
  else if(strncmp(command,"QProgressBar(",13) == 0) // create a new QProgressBar
  {
    int totalSteps;
    sscanf(command,"QProgressBar(%d,%d,%d",&i,&p,&totalSteps);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyProgressBar(s,i,totalSteps,all[p]->w);
    all[i]->type = TQProgressBar;
  }
  else if(strncmp(command,"QMultiLineEdit(",15) == 0) // create a new QMultiLineEdit
  {
    int editable,maxLines;
    sscanf(command,"QMultiLineEdit(%d,%d,%d,%d",&i,&p,&editable,&maxLines);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyMultiLineEdit(s,i,editable,maxLines,all[p]->w);
    all[i]->type = TQMultiLineEdit;
  }
  else if(strncmp(command,"QTextBrowser(",13) == 0) // create a new QTextBrowser
  {
    sscanf(command,"QTextBrowser(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyTextBrowser(s,i,all[p]->w);
    all[i]->type = TQTextBrowser;
  }
  else if(strncmp(command,"QListView(",10) == 0) // create a new QListView
  {
    sscanf(command,"QListView(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyListView(s,i,all[p]->w);
    all[i]->type = TQListView;
  }
  else if(strncmp(command,"QIconView(",10) == 0) // create a new QIconView
  {
    sscanf(command,"QIconView(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyIconView(s,i,all[p]->w);
    all[i]->type = TQIconView;
  }
  else if(strncmp(command,"QDateEdit(",10) == 0) // create a new QDateEdit
  {
    sscanf(command,"QDateEdit(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQDateEdit(s,i,all[p]->w);
    all[i]->type = TQDateEdit;
  }
  else if(strncmp(command,"QTimeEdit(",10) == 0) // create a new QTimeEdit
  {
    sscanf(command,"QTimeEdit(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQTimeEdit(s,i,all[p]->w);
    all[i]->type = TQTimeEdit;
  }
  else if(strncmp(command,"QDateTimeEdit(",14) == 0) // create a new QDateTimeEdit
  {
    sscanf(command,"QDateTimeEdit(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    all[i]->w = (QWidget *) new MyQDateTimeEdit(s,i,all[p]->w);
    all[i]->type = TQDateTimeEdit;
  }
  else if(strncmp(command,"QToolTip(",9) == 0) // create a new QToolTip
  {
    sscanf(command,"QToolTip(%d",&i);
    if(i < 0) return;
    if(i >= nmax) return;
    get_text(command,text);
    QToolTip::add(all[i]->w,text);
  }
  else if(strncmp(command,"QLayoutVbox(",12) == 0) // vbox layout
  {
    sscanf(command,"QLayoutVbox(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    if(i == 0)
    {
      mainLayout = (QWidget *) new QVBoxLayout(all[0]->w);
      mainLayoutType = TQVbox;
      ((QVBoxLayout *) mainLayout)->setMargin(11);
      ((QVBoxLayout *) mainLayout)->setSpacing(6);
    }
    else
    {
      all[i]->w = (QWidget *) new QVBoxLayout;
      all[i]->type = TQVbox;
    }
    hasLayout = 1;
  }
  else if(strncmp(command,"QLayoutHbox(",12) == 0) // hbox layout
  {
    sscanf(command,"QLayoutHbox(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    if(i == 0)
    {
      mainLayout = (QWidget *) new QHBoxLayout(all[0]->w);
      mainLayoutType = TQHbox;
      ((QHBoxLayout *) mainLayout)->setMargin(11);
      ((QHBoxLayout *) mainLayout)->setSpacing(6);
    }
    else
    {
      all[i]->w = (QWidget *) new QHBoxLayout;
      all[i]->type = TQHbox;
    }
    hasLayout = 1;
  }
  else if(strncmp(command,"QLayoutGrid(",12) == 0) // grid layout
  {
    sscanf(command,"QLayoutGrid(%d,%d",&i,&p);
    if(i < 0) return;
    if(i >= nmax) return;
    if(p >= nmax) return;
    if(i == 0)
    {
      mainLayout = (QWidget *) new QGridLayout(all[0]->w);
      mainLayoutType = TQGrid;
      ((QGridLayout *) mainLayout)->setMargin(11);
      ((QGridLayout *) mainLayout)->setSpacing(6);
    }
    else
    {
      all[i]->w = (QWidget *) new QGridLayout;
      all[i]->type = TQGrid;
    }
    hasLayout = 1;
  }

  if(i>0 && i<nmax)
  {
    if(all[i]->type != TQVbox && all[i]->type != TQHbox && all[i]->type != TQGrid)
    {
      all[i]->w->setAutoFillBackground(true);
      // all[i]->w->setBackgroundMode(Qt::PaletteBackground);
      // all[i]->w->setForegroundRole(QPalette::WindowText);
      // all[i]->w->setForegroundRole(QPalette::ButtonText);
      // all[i]->w->setBackgroundRole(QPalette::ButtonText);
      // all[i]->w->setBackgroundRole(QPalette::Window);
      // all[i]->w->setForegroundRole(QPalette::WindowText);
    }
  }

  if(zoom != 100)
  {
    QFont font = qApp->font();
    font.setPointSize((font.pointSize()*zoom)/100);
    all[i]->w->setFont(font);
  }
}
