//******************************************************
//
// Filename: vtkQGLRenderWindow.cpp
//
// Author: Jan Ehrhardt
// Email : ehrhardt@medinf.mu-luebeck.de
//
// Description: von vtkRenderWindow abgeleitete Klasse 
// als Schnittstelle zwischen Qt und VTK
// mit AddRenderer koennen vtkRenderer Objekte in diesem
// fenster gerendert werden,
// derzeit wird aber nur ein Renderer pro Fenster unterstuetzt
//
// Teile des Programms entsprechenden Methoden von
// vtkOpenGLRenderWindow
//              
//******************************************************
// Copyright (C) 1998 Jan Ehrhardt
//
// You can use, copy and distribute this software for all non-commercial purposes,
// provided the copyright notices and this notice are included in all copies and
// distributions.
// You have permission to modify the software, provided that modifications are not 
// distributed outside your site and that existing copyright notices are included in all copies. 
// For commercial purposes and modifications to be included into the software please
// contact the author. 
//
// WE MAKE NO WARRANTY THAT THIS PACKAGE IS FREE OF CONCEPTUAL OR PROGRAMMING ERRORS.
// THE AUTHORS ARE NOT RESPONSIBLE FOR ANY DATA LOSS OR EQUIPMENT DAMAGE RESULTING FROM 
// THE USE OR MISUSE OF THIS SOFTWARE. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, 
// THERE IS NO KIND OF SUPPORT AND MAINTENANCE.
//
//
//******************************************************
// RCS - Info:
//
// $Id: vtkQGLRenderWindow.cpp,v 1.1.1.1 2003/07/24 08:16:26 lehrig Exp $
// $Log: vtkQGLRenderWindow.cpp,v $
// Revision 1.1.1.1  2003/07/24 08:16:26  lehrig
// Version 1.7
//
// Revision 1.1.1.1  2003/02/04 13:46:02  lehrig
// This is ProcessViewBrowser + ProcessViewServer Version 1.5
// Checked in on 04.feb.2003
//
// Revision 1.3  1999/06/16 07:29:54  ehrhardt
// fuer vtk 2.3 angepasst
//
// Revision 1.2  1999/01/25 08:48:57  ehrhardt
// *** empty log message ***
//
// Revision 1.1  1998/10/22 15:16:01  ehrhardt
// Initial revision
//
//
//

#include <stdlib.h>
#include <math.h>
#include <iostream.h>
#include "vtkQGLRenderWindow.h"
#include "vtkQGLWidget.h"
#include "vtkOpenGLRenderer.h"
#include "vtkOpenGLProperty.h"
#include "vtkOpenGLTexture.h"
#include "vtkOpenGLCamera.h"
#include "vtkOpenGLLight.h"
#include "vtkOpenGLActor.h"
#include "vtkOpenGLPolyDataMapper.h"
#include "GL/gl.h"
#include "GL/glu.h"

vtkQGLRenderWindow::vtkQGLRenderWindow(vtkQGLWidget *widget)
{
  //this->MultiSamples = 8;
  this->bInitialized = 0;

  this->m_qglWidget = widget;

  if ( this->WindowName ) 
    delete [] this->WindowName;
  this->WindowName = new char[strlen("Visualization Toolkit - QGL")+1];
    strcpy( this->WindowName, "Visualization Toolkit - QGL" );
}

// Description:
// free up memory & close the window
vtkQGLRenderWindow::~vtkQGLRenderWindow()
{
  vtkOpenGLRenderer *ren;
  // the Renderer calls ReleaseGraphicsResources() to tell
  // all added actors that the window is destroyed
  this->Renderers->InitTraversal();
  for ( ren = (vtkOpenGLRenderer *) this->Renderers->GetNextItemAsObject();
	ren != NULL;
	ren = (vtkOpenGLRenderer *) this->Renderers->GetNextItemAsObject() )
    {
      ren->SetRenderWindow(NULL);
    }

}

void vtkQGLRenderWindow::PrintSelf(ostream& os, vtkIndent indent)
{
  this->vtkRenderWindow::PrintSelf(os,indent);

  //os << indent << "ContextId: " << this->ContextId << "\n";
  //os << indent << "MultiSamples: " << this->MultiSamples << "\n";
}

// Description:
// Initialize the window for rendering.
void vtkQGLRenderWindow::Initialize (void)
{
  int x, y, width, height; // nItems;

  // make sure we havent already been initialized 
  if (this->IsInitialized())
    return;

  if(! m_qglWidget->isValid()) 
    {
      cerr<<"QGLWidget not valid !!\n"<<flush;
      vtkErrorMacro(<< "QGLWidget not valid !!\n");
      return;
    }
  
  x = m_qglWidget->x();
  y =  m_qglWidget->y();
  this->SetPosition(x,y);

  width = m_qglWidget->width();
  height = m_qglWidget->height();
  this->SetSize(width,height);

  this->MakeCurrent();
  
  vtkDebugMacro(<< " glMatrixMode ModelView\n");
  glMatrixMode( GL_MODELVIEW );

  vtkDebugMacro(<< " zbuffer enabled\n");
  glDepthFunc( GL_LEQUAL );
  glEnable( GL_DEPTH_TEST );

  vtkDebugMacro(" texture stuff\n");
  glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );

  // initialize blending for transparency
  vtkDebugMacro(<< " blend func stuff\n");
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
  glEnable(GL_BLEND);
  
  glEnable( GL_NORMALIZE );
  glAlphaFunc(GL_GREATER,0);
  
  this->bInitialized = 1;
  this->Mapped = 1;
}

// Begin the rendering process.
void vtkQGLRenderWindow::Start(void)
{
  // if the renderer has not been initialized, do so now
  if (!this->IsInitialized())
    {
    this->Initialize();
    }

  // set the current window 
  this->MakeCurrent();
}

// End the rendering process and display the image.
void vtkQGLRenderWindow::Frame(void)
{
  //cout<<"Frame()";
  glFlush();
  if (!this->AbortRender && this->DoubleBuffer&&this->SwapBuffers)
   {
     m_qglWidget->swapBuffers();
  // vtkDebugMacro(<< " glXSwapBuffers\n");
  }
}


// Update system if needed due to stereo rendering.
void vtkQGLRenderWindow::StereoUpdate(void)
{
  // if stereo is on and it wasn't before
  if (this->StereoRender && (!this->StereoStatus))
    {
    switch (this->StereoType) 
      {
      case VTK_STEREO_CRYSTAL_EYES:
	{
	}
	break;
      case VTK_STEREO_RED_BLUE:
	{
        this->StereoStatus = 1;
	}
      }
    }
  else if ((!this->StereoRender) && this->StereoStatus)
    {
    switch (this->StereoType) 
      {
      case VTK_STEREO_CRYSTAL_EYES:
	{
        this->StereoStatus = 0;
	}
	break;
      case VTK_STEREO_RED_BLUE:
	{
        this->StereoStatus = 0;
	}
      }
    }
}

// Specify the size of the rendering window.
void vtkQGLRenderWindow::SetSize(int x,int y)
{
  if ((this->Size[0] != x)||(this->Size[1] != y))
    {
    this->Modified();
    this->Size[0] = x;
    this->Size[1] = y;
    }
  
  // if we arent mappen then just set the ivars 
  if (!this->Mapped)
    {
    return;
    }
  // ??? make now problems but doesnt seems necessary
  m_qglWidget->resize(x,y);
}

// Change the window to fill the entire screen.
void vtkQGLRenderWindow::SetFullScreen(int arg)
{
  if(arg == 0) return;
  // fullscreen doesnt work
}

// Set the preferred window size to full screen.
void vtkQGLRenderWindow::PrefFullScreen()
{
  int size[2] = {640,480};

  //size = this->m_GetScreenSize();

  // use full screen 
  this->Position[0] = 0;
  this->Position[1] = 0;
  this->Size[0] = size[0];
  this->Size[1] = size[1];

  // don't show borders 
  this->Borders = 0;
}

void vtkQGLRenderWindow::MakeCurrent()
{
  // set the current window 
  m_qglWidget->makeCurrent();
}

// the following lines are from the vtkOpenGLRenderWindow
/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkQGLRenderWindow.cpp,v $
  Language:  C++
  Date:      $Date: 2003/07/24 08:16:26 $
  Version:   $Revision: 1.1.1.1 $


Copyright (c) 1993-1998 Ken Martin, Will Schroeder, Bill Lorensen.

This software is copyrighted by Ken Martin, Will Schroeder and Bill Lorensen.
The following terms apply to all files associated with the software unless
explicitly disclaimed in individual files. This copyright specifically does
not apply to the related textbook "The Visualization Toolkit" ISBN
013199837-4 published by Prentice Hall which is covered by its own copyright.

The authors hereby grant permission to use, copy, and distribute this
software and its documentation for any purpose, provided that existing
copyright notices are retained in all copies and that this notice is included
verbatim in any distributions. Additionally, the authors grant permission to
modify this software and its documentation for any purpose, provided that
such modifications are not distributed without the explicit consent of the
authors and that existing copyright notices are retained in all copies. Some
of the algorithms implemented by this software are patented, observe all
applicable patent law.

IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES THEREOF,
EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE IS PROVIDED ON AN
"AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.


=========================================================================*/
unsigned char *vtkQGLRenderWindow::GetPixelData(int x1, int y1, 
						   int x2, int y2, 
						   int front)
{
  int     y_low, y_hi;
  int     x_low, x_hi;
  unsigned char   *data = NULL;

    // set the current window 
  this->MakeCurrent();

  if (y1 < y2)
    {
    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }

  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }

  if (front)
    {
    glReadBuffer(GL_FRONT);
    }
  else
    {
    glReadBuffer(GL_BACK);
    }

  data = new unsigned char[(x_hi - x_low + 1)*(y_hi - y_low + 1)*3];

#ifdef sparc
  // We need to read the image data one row at a time and convert it
  // from RGBA to RGB to get around a bug in Sun OpenGL 1.1
  long    xloop, yloop;
  unsigned char *buffer;
  unsigned char *p_data = NULL;
  
  buffer = new unsigned char [4*(x_hi - x_low + 1)];
  p_data = data;
  for (yloop = y_low; yloop <= y_hi; yloop++)
    {
    // read in a row of pixels
    glReadPixels(x_low,yloop,(x_hi-x_low+1),1,
		 GL_RGBA, GL_UNSIGNED_BYTE, buffer);
    for (xloop = 0; xloop <= x_hi-x_low; xloop++)
      {
      *p_data = buffer[xloop*4]; p_data++;
      *p_data = buffer[xloop*4+1]; p_data++;
      *p_data = buffer[xloop*4+2]; p_data++;
      }
    }
  
  delete [] buffer;  
#else
  // If the Sun bug is ever fixed, then we could use the following
  // technique which provides a vast speed improvement on the SGI
  
  // Calling pack alignment ensures that we can grab the any size window
  glPixelStorei( GL_PACK_ALIGNMENT, 1 );
  glReadPixels(x_low, y_low, x_hi-x_low+1, y_hi-y_low+1, GL_RGB,
               GL_UNSIGNED_BYTE, data);
#endif
  
  return data;
}

#ifdef BookVTK40
void vtkQGLRenderWindow::SetPixelData(int x1, int y1, int x2, int y2,
                         unsigned char *data, int front)
#else
int  vtkQGLRenderWindow::SetPixelData(int x1, int y1, int x2, int y2,
                         unsigned char *data, int front)
#endif
{
  int     y_low, y_hi;
  int     x_low, x_hi;

    // set the current window 
  this->MakeCurrent();

  if (front)
    {
    glDrawBuffer(GL_FRONT);
    }
  else
    {
    glDrawBuffer(GL_BACK);
    }

  if (y1 < y2)
    {

    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }
  
  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }

#ifdef sparc
  // We need to read the image data one row at a time and convert it
  // from RGBA to RGB to get around a bug in Sun OpenGL 1.1
  long    xloop, yloop;
  unsigned char *buffer;
  unsigned char *p_data = NULL;
  
  buffer = new unsigned char [4*(x_hi - x_low + 1)];

  // now write the binary info one row at a time
  glDisable(GL_BLEND);
  p_data = data;
  for (yloop = y_low; yloop <= y_hi; yloop++)
    {
    for (xloop = 0; xloop <= x_hi - x_low; xloop++)
      {
      buffer[xloop*4] = *p_data; p_data++;
      buffer[xloop*4+1] = *p_data; p_data++;
      buffer[xloop*4+2] = *p_data; p_data++;
      buffer[xloop*4+3] = 0xff;
      }
    /* write out a row of pixels */
    glMatrixMode( GL_MODELVIEW );
    glPushMatrix();
    glLoadIdentity();
    glMatrixMode( GL_PROJECTION );
    glPushMatrix();
    glLoadIdentity();
    glRasterPos3f( (2.0 * (GLfloat)(x_low) / this->Size[0] - 1),
		   (2.0 * (GLfloat)(yloop) / this->Size[1] - 1),
		   -1.0 );
    glMatrixMode( GL_PROJECTION );
    glPopMatrix();
    glMatrixMode( GL_MODELVIEW );
    glPopMatrix();

    glDrawPixels((x_hi-x_low+1),1, GL_RGBA, GL_UNSIGNED_BYTE, buffer);
    }
  glEnable(GL_BLEND);
#else
  // If the Sun bug is ever fixed, then we could use the following
  // technique which provides a vast speed improvement on the SGI
  
  // now write the binary info
  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  glRasterPos3f( (2.0 * (GLfloat)(x_low) / this->Size[0] - 1), 
                 (2.0 * (GLfloat)(y_low) / this->Size[1] - 1),
                 -1.0 );
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();

  glPixelStorei( GL_UNPACK_ALIGNMENT, 1);
  glDisable(GL_BLEND);
  glDrawPixels((x_hi-x_low+1), (y_hi - y_low + 1),
               GL_RGB, GL_UNSIGNED_BYTE, data);
  glEnable(GL_BLEND);
#endif
#ifndef BookVTK40
  return 0;
#endif
}

float *vtkQGLRenderWindow::GetRGBAPixelData(int x1, int y1, int x2, int y2, int front)
{
  int     y_low, y_hi;
  int     x_low, x_hi;
  int     width, height;
  float   *data = NULL;

    // set the current window 
  this->MakeCurrent();

  if (y1 < y2)
    {
    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }

  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }

  if (front)
    {
    glReadBuffer(GL_FRONT);
    }
  else
    {
    glReadBuffer(GL_BACK);
    }

  width  = abs(x_hi - x_low) + 1;
  height = abs(y_hi - y_low) + 1;

  data = new float[ (width*height*4) ];

  glReadPixels( x_low, y_low, width, height, GL_RGBA, GL_FLOAT, data);

  return data;
}

#ifdef BookVTK40
void vtkQGLRenderWindow::SetRGBAPixelData(int x1, int y1, int x2, int y2,
                         float *data, int front, int blend)
#else
int  vtkQGLRenderWindow::SetRGBAPixelData(int x1, int y1, int x2, int y2,
                         float *data, int front, int blend)
#endif
{
  int     y_low, y_hi;
  int     x_low, x_hi;
  int     width, height;

    // set the current window 
  this->MakeCurrent();

  if (front)
    {
    glDrawBuffer(GL_FRONT);
    }
  else
    {
    glDrawBuffer(GL_BACK);
    }

  if (y1 < y2)
    {
    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }
  
  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }
  
  width  = abs(x_hi-x_low) + 1;
  height = abs(y_hi-y_low) + 1;

  /* write out a row of pixels */
  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  glRasterPos3f( (2.0 * (GLfloat)(x_low) / this->Size[0] - 1), 
                 (2.0 * (GLfloat)(y_low) / this->Size[1] - 1),
		 -1.0 );
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();

  if (!blend)
    {
    glDisable(GL_BLEND);
    glDrawPixels( width, height, GL_RGBA, GL_FLOAT, data);
    glEnable(GL_BLEND);
    }
  else
    {
    glDrawPixels( width, height, GL_RGBA, GL_FLOAT, data);
    }
#ifndef BookVTK40
  return 0;
#endif  
}

float *vtkQGLRenderWindow::GetZbufferData( int x1, int y1, int x2, int y2  )
{
  int             y_low, y_hi;
  int             x_low, x_hi;
  int             width, height;
  float           *z_data = NULL;

  // set the current window 
  this->MakeCurrent();

  if (y1 < y2)
    {
    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }

  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }

  width =  abs(x2 - x1)+1;
  height = abs(y2 - y1)+1;

  z_data = new float[width*height];

  glReadPixels( x_low, y_low, 
		width, height,
		GL_DEPTH_COMPONENT, GL_FLOAT,
		z_data );

  return z_data;
}

#ifdef BookVTK40
void vtkQGLRenderWindow::SetZbufferData( int x1, int y1, int x2, int y2,
                                         float *buffer )
#else
int  vtkQGLRenderWindow::SetZbufferData( int x1, int y1, int x2, int y2,
                                         float *buffer )
#endif
{
  int             y_low, y_hi;
  int             x_low, x_hi;
  int             width, height;

  // set the current window 
  this->MakeCurrent();

  if (y1 < y2)
    {
    y_low = y1; 
    y_hi  = y2;
    }
  else
    {
    y_low = y2; 
    y_hi  = y1;
    }

  if (x1 < x2)
    {
    x_low = x1; 
    x_hi  = x2;
    }
  else
    {
    x_low = x2; 
    x_hi  = x1;
    }

  width =  abs(x2 - x1)+1;
  height = abs(y2 - y1)+1;

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  glRasterPos2f( 2.0 * (GLfloat)(x_low) / this->Size[0] - 1, 
                 2.0 * (GLfloat)(y_low) / this->Size[1] - 1);
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();

  glDrawPixels( width, height, GL_DEPTH_COMPONENT, GL_FLOAT, buffer);
#ifndef BookVTK40
  return 0;
#endif
}


