/****************************************************************************
**
** Copyright (C) 2000-2006 Lehrig Software Engineering.
**
** This file is part of the pvbrowser project.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include <QtGui>

#include "mainwindow.h"
#include "opt.h"

extern OPT opt;

MainWindow::MainWindow()
{
    if(opt.arg_debug) printf("MainWindow start\n");

    createActions();
    createMenus();
    createToolBars();
    createStatusBar();
    readSettings();

    editor   = NULL;
    designer = NULL;
    //viewDesigner();
    viewEditor();

    setCurrentFile("");
    if(opt.arg_debug) printf("MainWindow end\n");
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (maybeSave()) {
        writeSettings();
        event->accept();
    } else {
        event->ignore();
    }
}

void MainWindow::newFile()
{
    if (maybeSave()) {
        editor->edit->clear();
        setCurrentFile("");
    }
}

void MainWindow::open()
{
  if (maybeSave())
  {
    QString fileName = QFileDialog::getOpenFileName(this);
    if(!fileName.isEmpty()) loadFile(fileName);
  }
}

bool MainWindow::save()
{
  if(curFile.isEmpty()) return saveAs();
  else                  return saveFile(curFile);
}

bool MainWindow::saveAs()
{
  QString fileName = QFileDialog::getSaveFileName(this);
  if(fileName.isEmpty()) return false;
  return saveFile(fileName);
}

void MainWindow::about()
{
  QMessageBox::about(this, tr("About Application"),
            tr("The <b>Application</b> example demonstrates how to "
               "write modern GUI applications using Qt, with a menu bar, "
               "toolbars, and a status bar."));
}

void MainWindow::documentWasModified()
{
  setWindowModified(editor->edit->document()->isModified());
}

void MainWindow::createActions()
{
  if(opt.arg_debug) printf("createActions\n");;

  newAct = new QAction(QIcon(":/images/new.png"), tr("&New"), this);
  newAct->setShortcut(tr("Ctrl+N"));
  newAct->setStatusTip(tr("Create a new file"));
  connect(newAct, SIGNAL(triggered()), this, SLOT(newFile()));

  openAct = new QAction(QIcon(":/images/open.png"), tr("&Open..."), this);
  openAct->setShortcut(tr("Ctrl+O"));
  openAct->setStatusTip(tr("Open an existing file"));
  connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

  saveAct = new QAction(QIcon(":/images/save.png"), tr("&Save"), this);
  saveAct->setShortcut(tr("Ctrl+S"));
  saveAct->setStatusTip(tr("Save the document to disk"));
  connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

  saveAsAct = new QAction(tr("Save &As..."), this);
  saveAsAct->setStatusTip(tr("Save the document under a new name"));
  connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

  exitAct = new QAction(tr("E&xit"), this);
  exitAct->setShortcut(tr("Ctrl+Q"));
  exitAct->setStatusTip(tr("Exit the application"));
  connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

  cutAct = new QAction(QIcon(":/images/cut.png"), tr("Cu&t"), this);
  cutAct->setShortcut(tr("Ctrl+X"));
  cutAct->setStatusTip(tr("Cut the current selection's contents to the "
                            "clipboard"));

  copyAct = new QAction(QIcon(":/images/copy.png"), tr("&Copy"), this);
  copyAct->setShortcut(tr("Ctrl+C"));
  copyAct->setStatusTip(tr("Copy the current selection's contents to the "
                             "clipboard"));

  pasteAct = new QAction(QIcon(":/images/paste.png"), tr("&Paste"), this);
  pasteAct->setShortcut(tr("Ctrl+V"));
  pasteAct->setStatusTip(tr("Paste the clipboard's contents into the current "
                              "selection"));

  viewEditorAct = new QAction(QIcon(":/images/editor.png"), tr("&Editor"), this);
  viewEditorAct->setShortcut(tr("Ctrl+E"));
  viewEditorAct->setStatusTip(tr("Show the Editor"));

  connect(viewEditorAct, SIGNAL(triggered()), this, SLOT(viewEditor()));

  viewDesignerAct = new QAction(QIcon(":/images/designer.png"), tr("&Designer"), this);
  viewDesignerAct->setShortcut(tr("Ctrl+D"));
  viewDesignerAct->setStatusTip(tr("Show the Designer"));

  connect(viewDesignerAct, SIGNAL(triggered()), this, SLOT(viewDesigner()));

  actionEditorAct = new QAction(QIcon(":/images/editor.png"), tr("&Editor"), this);
  actionEditorAct->setShortcut(tr("Ctrl+E"));
  actionEditorAct->setStatusTip(tr("Show the Editor"));

  connect(actionEditorAct, SIGNAL(triggered()), this, SLOT((slotEditor())));

  actionDesignerAct = new QAction(QIcon(":/images/designer.png"), tr("&Designer"), this);
  actionDesignerAct->setShortcut(tr("Ctrl+D"));
  actionDesignerAct->setStatusTip(tr("Show the Designer"));

  connect(actionDesignerAct, SIGNAL(triggered()), this, SLOT((slotDesigner())));

  actionMakeAct = new QAction(QIcon(":/images/make.png"), tr("&Make"), this);
  actionMakeAct->setShortcut(tr("Ctrl+M"));
  actionMakeAct->setStatusTip(tr("Make project"));

  connect(actionMakeAct, SIGNAL(triggered()), this, SLOT((slotMake())));

  actionStartServerAct = new QAction(QIcon(":/images/startserver.png"), tr("&StartServer"), this);
  actionStartServerAct->setShortcut(tr("Ctrl+S"));
  actionStartServerAct->setStatusTip(tr("Start the Server"));

  connect(actionStartServerAct, SIGNAL(triggered()), this, SLOT((slotStartServer())));

  actionStartPvbrowserAct = new QAction(QIcon(":/images/startpvbrowser.png"), tr("&pvbrowser"), this);
  actionStartPvbrowserAct->setShortcut(tr("Ctrl+P"));
  actionStartPvbrowserAct->setStatusTip(tr("Start pvbrowser"));

  connect(actionStartPvbrowserAct, SIGNAL(triggered()), this, SLOT((slotStartPvbrowser())));

  aboutAct = new QAction(tr("&About"), this);
  aboutAct->setStatusTip(tr("Show the application's About box"));
  connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

  aboutQtAct = new QAction(tr("About &Qt"), this);
  aboutQtAct->setStatusTip(tr("Show the Qt library's About box"));
  connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));

  cutAct->setEnabled(false);
  copyAct->setEnabled(false);
}

void MainWindow::connectActions()
{
  if(opt.arg_debug) printf("connectActions\n");
  if(editor == NULL) return;
  connect(cutAct, SIGNAL(triggered()), editor->edit, SLOT(cut()));
  connect(copyAct, SIGNAL(triggered()), editor->edit, SLOT(copy()));
  connect(pasteAct, SIGNAL(triggered()), editor->edit, SLOT(paste()));

  connect(editor->edit, SIGNAL(copyAvailable(bool)), cutAct, SLOT(setEnabled(bool)));
  connect(editor->edit, SIGNAL(copyAvailable(bool)), copyAct, SLOT(setEnabled(bool)));

  connect(editor->edit->document(), SIGNAL(contentsChanged()), this, SLOT(documentWasModified()));
}

void MainWindow::disconnectActions()
{
  if(opt.arg_debug) printf("disconnectActions\n");;
  disconnect(cutAct, SIGNAL(triggered()), editor->edit, SLOT(cut()));
  disconnect(copyAct, SIGNAL(triggered()), editor->edit, SLOT(copy()));
  disconnect(pasteAct, SIGNAL(triggered()), editor->edit, SLOT(paste()));
  disconnect(editor->edit, SIGNAL(copyAvailable(bool)), cutAct, SLOT(setEnabled(bool)));
  disconnect(editor->edit, SIGNAL(copyAvailable(bool)), copyAct, SLOT(setEnabled(bool)));

  disconnect(editor->edit->document(), SIGNAL(contentsChanged()), this, SLOT(documentWasModified()));
}

void MainWindow::createMenus()
{
  if(opt.arg_debug) printf("createMenus\n");;

  fileMenu = menuBar()->addMenu(tr("&File"));
  fileMenu->addAction(newAct);
  fileMenu->addAction(openAct);
  fileMenu->addAction(saveAct);
  fileMenu->addAction(saveAsAct);
  fileMenu->addSeparator();
  fileMenu->addAction(exitAct);

  editMenu = menuBar()->addMenu(tr("&Edit"));
  editMenu->addAction(cutAct);
  editMenu->addAction(copyAct);
  editMenu->addAction(pasteAct);

  actionMenu = menuBar()->addMenu(tr("&Action"));
  actionMenu->addAction(actionEditorAct);
  actionMenu->addAction(actionDesignerAct);
  actionMenu->addAction(actionMakeAct);
  actionMenu->addAction(actionStartServerAct);
  actionMenu->addAction(actionStartPvbrowserAct);

  menuBar()->addSeparator();

  helpMenu = menuBar()->addMenu(tr("&Help"));
  helpMenu->addAction(aboutAct);
  helpMenu->addAction(aboutQtAct);
}

void MainWindow::createToolBars()
{
  if(opt.arg_debug) printf("createToolBars\n");;

  fileToolBar = addToolBar(tr("File"));
  fileToolBar->addAction(newAct);
  fileToolBar->addAction(openAct);
  fileToolBar->addAction(saveAct);

  editToolBar = addToolBar(tr("Edit"));
  editToolBar->addAction(cutAct);
  editToolBar->addAction(copyAct);
  editToolBar->addAction(pasteAct);

  viewToolBar = addToolBar(tr("View"));
  viewToolBar->addAction(viewEditorAct);
  viewToolBar->addAction(viewDesignerAct);
}

void MainWindow::createStatusBar()
{
  if(opt.arg_debug) printf("createStatusBar\n");;

  statusBar()->showMessage(tr("Ready"));
}

void MainWindow::readSettings()
{
  if(opt.arg_debug) printf("readSettings\n");;
  QSettings settings("Trolltech", "Application Example");
  QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
  QSize size = settings.value("size", QSize(400, 400)).toSize();
  resize(size);
  move(pos);
}

void MainWindow::writeSettings()
{
  if(opt.arg_debug) printf("writeSettings\n");;
  QSettings settings("Trolltech", "Application Example");
  settings.setValue("pos", pos());
  settings.setValue("size", size());
}

bool MainWindow::maybeSave()
{
  if(editor->edit->document()->isModified())
  {
    int ret = QMessageBox::warning(this, tr("Application"),
                     tr("The document has been modified.\n"
                        "Do you want to save your changes?"),
                     QMessageBox::Yes | QMessageBox::Default,
                     QMessageBox::No,
                     QMessageBox::Cancel | QMessageBox::Escape);
    if(ret == QMessageBox::Yes)          return save();
    else if (ret == QMessageBox::Cancel) return false;
  }
  return true;
}

void MainWindow::loadFile(const QString &fileName)
{
  QFile file(fileName);
  if(!file.open(QFile::ReadOnly | QFile::Text))
  {
    QMessageBox::warning(this, tr("Application"),
                             tr("Cannot read file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
    return;
  }

  QTextStream in(&file);
  QApplication::setOverrideCursor(Qt::WaitCursor);
  editor->edit->setPlainText(in.readAll());
  QApplication::restoreOverrideCursor();

  setCurrentFile(fileName);
  statusBar()->showMessage(tr("File loaded"), 2000);
}

bool MainWindow::saveFile(const QString &fileName)
{
  QFile file(fileName);
  if(!file.open(QFile::WriteOnly | QFile::Text))
  {
    QMessageBox::warning(this, tr("Application"),
                             tr("Cannot write file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
    return false;
  }

  QTextStream out(&file);
  QApplication::setOverrideCursor(Qt::WaitCursor);
  out << editor->edit->toPlainText();
  QApplication::restoreOverrideCursor();

  setCurrentFile(fileName);
  statusBar()->showMessage(tr("File saved"), 2000);
  return true;
}

void MainWindow::setCurrentFile(const QString &fileName)
{
  if(editor == NULL) return;
  if(opt.arg_debug) printf("setCurrentFile=%s\n", (const char *) fileName.toAscii());

  curFile = fileName;
  editor->edit->document()->setModified(false);
  setWindowModified(false);

  QString shownName;
  if (curFile.isEmpty()) shownName = "no project loaded";
  else                   shownName = strippedName(curFile);

  setWindowTitle(tr("%1[*] - %2").arg(shownName).arg(tr("pvdevelop")));
}

QString MainWindow::strippedName(const QString &fullFileName)
{
  return QFileInfo(fullFileName).fileName();
}

void MainWindow::viewEditor()
{
  if(editor != NULL) return;
  if(opt.arg_debug) printf("viewEditor begin\n");;

  if(editor != NULL || designer != NULL) centralWidget()->hide();
  designer = NULL;
  editor = new Editor();
  hide();
  //setCentralWidget(editor->edit);
  setCentralWidget(editor->root);
  show();
  centralWidget()->show();
  connectActions();

  if(opt.arg_debug) printf("viewEditor end\n");;
}

void MainWindow::viewDesigner()
{
  if(designer != NULL) return;
  if(opt.arg_debug) printf("viewDesigner begin\n");;

  if(editor != NULL || designer != NULL) centralWidget()->hide();
  disconnectActions();
  editor = NULL;
  scroll = new QScrollArea;
  designer = new Designer();
  designer->root->resize(1280,1024);
  scroll->setWidget(designer->root);
  hide();
  setCentralWidget(scroll);
  show();
  centralWidget()->show();

  if(opt.arg_debug) printf("viewDesigner end\n");;
}

void MainWindow::slotEditor()
{
  action("editor");
}

void MainWindow::slotDesigner()
{
  action("designer");
}

void MainWindow::slotMake()
{
  action("make");
}

void MainWindow::slotStartServer()
{
  action("startserver");
}

void MainWindow::slotStartPvbrowser()
{
  action("pvbrowser");
}

int MainWindow::action(const char *command)
{
  if(opt.arg_debug) printf("action(%s)\n",command);
  if(strcmp(command,"editor") == 0)
  {
  }
  else if(strcmp(command,"designer") == 0)
  {
  }
  else if(strcmp(command,"make") == 0)
  {
  }
  else if(strcmp(command,"startserver") == 0)
  {
  }
  else if(strcmp(command,"pvbrowser") == 0)
  {
  }
  return -1;
}
