#include <qregexp.h>
#include <qapplication.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qstatusbar.h>
#include <qprinter.h>
#if QT_VERSION >= 0x040000
#include <qprintdialog.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#endif
#include <qwt_counter.h>
#include <qwt_plot_zoomer.h>
#include <qwt_text.h>
#include <qwt_math.h>
#include "pixmaps.h"
#include "bode_plot.h"
#include "bode.h"

//-----------------------------------------------------------------
//
//      bode.cpp -- A demo program featuring QwtPlot and QwtCounter
//
//      This example demonstrates the mapping of different curves
//      to different axes in a QwtPlot widget. It also shows how to
//      display the cursor position and how to implement zooming.
//
//-----------------------------------------------------------------

MainWin::MainWin(QWidget *parent): 
    QMainWindow(parent)
{
    d_plot = new BodePlot(this);
    d_plot->setMargin(5);

#if QT_VERSION >= 0x040000
    setContextMenuPolicy(Qt::NoContextMenu);
#endif

    d_zoomer[0] = new QwtPlotZoomer(QwtPlot::xBottom, QwtPlot::yLeft,
        QwtPicker::DragSelection, QwtPicker::AlwaysOff, d_plot->canvas());
    d_zoomer[0]->setRubberBandPen(QColor(Qt::green));

    d_zoomer[1] = new QwtPlotZoomer(QwtPlot::xTop, QwtPlot::yRight,
        QwtPicker::DragSelection, QwtPicker::AlwaysOff, d_plot->canvas());
    d_zoomer[1]->setRubberBand(QwtPicker::NoRubberBand);
    

    d_picker = new QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yLeft,
        QwtPicker::PointSelection | QwtPicker::DragSelection, 
        QwtPlotPicker::CrossRubberBand, QwtPicker::AlwaysOn, 
        d_plot->canvas());
    d_picker->setRubberBandPen(QColor(Qt::green));
    d_picker->setTrackerPen(QColor(Qt::white));

    setCentralWidget(d_plot);

    QToolBar *toolBar = new QToolBar(this);

    QToolButton *btnZoom = new QToolButton(toolBar);
#if QT_VERSION >= 0x040000
    btnZoom->setText("Zoom");
    btnZoom->setIcon(QIcon(zoom_xpm));
    btnZoom->setCheckable(true);
    btnZoom->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
#else
    btnZoom->setTextLabel("Zoom");
    btnZoom->setPixmap(zoom_xpm);
    btnZoom->setToggleButton(true);
    btnZoom->setUsesTextLabel(true);
#endif

    QToolButton *btnPrint = new QToolButton(toolBar);
#if QT_VERSION >= 0x040000
    btnPrint->setText("Print");
    btnPrint->setIcon(QIcon(print_xpm));
    btnPrint->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
#else
    btnPrint->setTextLabel("Print");
    btnPrint->setPixmap(print_xpm);
    btnPrint->setUsesTextLabel(true);
#endif

#if QT_VERSION >= 0x040100
    QToolButton *btnExport = new QToolButton(toolBar);
    btnExport->setText("Export PDF");
    btnExport->setIcon(QIcon(print_xpm));
    btnExport->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
#endif

#if QT_VERSION >= 0x040000
    toolBar->addWidget(btnZoom);
    toolBar->addWidget(btnPrint);
#if QT_VERSION >= 0x040100
    toolBar->addWidget(btnExport);
#endif
#endif
    toolBar->addSeparator();

    QWidget *hBox = new QWidget(toolBar);

    QHBoxLayout *layout = new QHBoxLayout(hBox);
    layout->setSpacing(0);
    layout->addWidget(new QWidget(hBox), 10); // spacer
    layout->addWidget(new QLabel("Damping Factor", hBox), 0);
    layout->addSpacing(10);

    QwtCounter *cntDamp = new QwtCounter(hBox);
    cntDamp->setRange(0.0, 5.0, 0.01);
    cntDamp->setValue(0.0);
    
    layout->addWidget(cntDamp, 0);

#if QT_VERSION >= 0x040000
    (void)toolBar->addWidget(hBox);
#else
    toolBar->setStretchableWidget(hBox);
#endif

    addToolBar(toolBar);
    (void)statusBar();

    zoom(false);
    showInfo();

    connect(cntDamp, SIGNAL(valueChanged(double)), 
        d_plot, SLOT(setDamp(double))); 

    connect(btnPrint, SIGNAL(clicked()), SLOT(print()));
#if QT_VERSION >= 0x040100
    connect(btnExport, SIGNAL(clicked()), SLOT(exportPDF()));
#endif
    connect(btnZoom, SIGNAL(toggled(bool)), SLOT(zoom(bool)));

    connect(d_picker, SIGNAL(moved(const QPoint &)),
            SLOT(moved(const QPoint &)));
    connect(d_picker, SIGNAL(selected(const QwtPicker::SelectedPoints &)),
            SLOT(selected(const QwtPicker::SelectedPoints &)));
}

void MainWin::print()
{
#if 1
    QPrinter printer;
#else
    QPrinter printer(QPrinter::HighResolution);
#if QT_VERSION < 0x040000
    printer.setOutputToFile(true);
#endif
    printer.setOutputFileName("/tmp/bode.ps");
#endif

    QString docName = d_plot->title().text();
    if ( docName.isEmpty() )
    {
        docName.replace (QRegExp (QString::fromLatin1 ("\n")), tr (" -- "));
        printer.setDocName (docName);
    }

    printer.setCreator("Bode example");
    printer.setOrientation(QPrinter::Landscape);

#if QT_VERSION >= 0x040000
    QPrintDialog dialog(&printer);
    if ( dialog.exec() )
    {
#if 0
        const QString fileName = printer.outputFileName();
        if ( !fileName.isEmpty() &&
            QFileInfo(fileName).suffix().toLower() == "pdf" )
        {
            printer.setOutputFormat(QPrinter::PdfFormat);
        }
#endif
#else
    if (printer.setup())
    {
#endif
        QwtPlotPrintFilter filter;
        if ( printer.colorMode() == QPrinter::GrayScale )
        {
            filter.setOptions(QwtPlotPrintFilter::PrintAll 
                & ~QwtPlotPrintFilter::PrintCanvasBackground);
        }
        d_plot->print(printer, filter);
    }
}

void MainWin::exportPDF()
{
#if QT_VERSION >= 0x040100
    const QString fileName = QFileDialog::getSaveFileName(
        this, "Export File Name", QString(),
        "PDF Documents (*.pdf)");
    if ( !fileName.isEmpty() )
    {
#if 0
        QPrinter printer(QPrinter::HighResolution);
#else
        QPrinter printer;
#endif
        printer.setOutputFormat(QPrinter::PdfFormat);
#if 0
        printer.setOrientation(QPrinter::Landscape);
#endif
        printer.setOutputFileName(fileName);

        QwtPlotPrintFilter filter;
#if 0
        filter.setOptions(QwtPlotPrintFilter::PrintAll 
            & ~QwtPlotPrintFilter::PrintCanvasBackground);
#endif

        printer.setCreator("Bode example");
        d_plot->print(printer, filter);
    }
#endif
}

void MainWin::zoom(bool on)
{
    d_zoomer[0]->setEnabled(on);
    d_zoomer[0]->zoom(0);

    d_zoomer[1]->setEnabled(on);
    d_zoomer[1]->zoom(0);

    d_picker->setRubberBand(
        on ? QwtPicker::NoRubberBand: QwtPicker::CrossRubberBand);

    showInfo();
}

void MainWin::showInfo(QString text)
{
    if ( text == QString::null )
    {
        if ( d_picker->rubberBand() )
            text = "Cursor Pos: Press left mouse button in plot region";
        else
            text = "Zoom: Press mouse button and drag";
    }

#if QT_VERSION >= 0x040000
    statusBar()->showMessage(text);
#else
    statusBar()->message(text);
#endif
}

void MainWin::moved(const QPoint &pos)
{
    QString info;
    info.sprintf("Freq=%g, Ampl=%g, Phase=%g",
        d_plot->invTransform(QwtPlot::xBottom, pos.x()),
        d_plot->invTransform(QwtPlot::yLeft, pos.y()),
        d_plot->invTransform(QwtPlot::yRight, pos.y())
    );
    showInfo(info);
}

void MainWin::selected(const QwtPicker::SelectedPoints &)
{
    showInfo();
}

int main (int argc, char **argv)
{
    QApplication a(argc, argv);

    MainWin w;
#if QT_VERSION < 0x040000
    a.setMainWidget(&w);
#endif
    w.resize(540,400);
    w.show();

    int rv = a.exec();
    return rv;
}
