#include <stdlib.h>

#include <qapplication.h>
#include <qwt_plot.h>
#include <qwt_plot_canvas.h>
#include <qwt_plot_marker.h>
#include <qwt_plot_curve.h>
#include <qwt_scale_widget.h>
#include <qwt_legend.h>
#include <qwt_scale_draw.h>
#include <qwt_math.h>


const int PLOT_SIZE = 201;      // 0 to 200

//-----------------------------------------------------------------
//      data_plot.cpp
//
//  This example shows how to display time-varying data
//  using QwtPlot.
//
//-----------------------------------------------------------------

class DataPlot : public QwtPlot
{
public:
    DataPlot();

protected:
    virtual void timerEvent(QTimerEvent *e);

private:
    void alignScales();

    double d_x[PLOT_SIZE]; 
    double d_y[PLOT_SIZE]; 
    double d_z[PLOT_SIZE];
};


//
//  Initialize main window
//
DataPlot::DataPlot() 
{
    alignScales();
    
    //  Initialize data
    for (int i = 0; i< PLOT_SIZE; i++)
    {
        d_x[i] = 0.5 * i;     // time axis
        d_y[i] = 0;
        d_z[i] = 0;
    }

    // Assign a title
    setTitle("Another Simple QwtPlot Demonstration");
    insertLegend(new QwtLegend(), QwtPlot::BottomLegend);

    // Insert new curves
    QwtPlotCurve *cRight = new QwtPlotCurve("Data Moving Right");
    cRight->attach(this);

    QwtPlotCurve *cLeft = new QwtPlotCurve("Data Moving Left");
    cLeft->attach(this);

    // Set curve styles
    cRight->setPen(QPen(Qt::red));
    cLeft->setPen(QPen(Qt::blue));

    // Attach (don't copy) data. Both curves use the same x array.
    cRight->setRawData(d_x, d_y, PLOT_SIZE);
    cLeft->setRawData(d_x, d_z, PLOT_SIZE);

    //  Insert zero line at y = 0
    QwtPlotMarker *mY = new QwtPlotMarker();
    mY->setLabelAlignment(Qt::AlignRight|Qt::AlignTop);
    mY->setLineStyle(QwtPlotMarker::HLine);
    mY->setYValue(0.0);
    mY->attach(this);

    // Set axis titles
    setAxisTitle(QwtPlot::xBottom, "Time/seconds");
    setAxisTitle(QwtPlot::yLeft, "Values");
    
    // Generate timer event every 50ms
    (void)startTimer(50);
}

//
//  Set a plain canvas frame and align the scales to it
//
void DataPlot::alignScales()
{
    // The code below shows how to align the scales to
    // the canvas frame, but is also a good example demonstrating
    // why the spreaded API needs polishing.

    canvas()->setFrameStyle(QFrame::Box | QFrame::Plain );
    canvas()->setLineWidth(1);

    for ( int i = 0; i < QwtPlot::axisCnt; i++ )
    {
        QwtScaleWidget *scaleWidget = (QwtScaleWidget *)axisWidget(i);
        if ( scaleWidget )
            scaleWidget->setBaselineDist(0);

        QwtScaleDraw *scaleDraw = (QwtScaleDraw *)axisScaleDraw(i);
        if ( scaleDraw )
            scaleDraw->enableComponent(QwtAbstractScaleDraw::Backbone, false);
    }
}

//
//  Generate new values every 500ms. 
//
void DataPlot::timerEvent(QTimerEvent *)
{
    static double phase = 0.0;

    if (phase > (M_PI - 0.0001)) 
        phase = 0.0;

    // y moves from left to right:
    // Shift y array right and assign new value to y[0].

    for ( int i = PLOT_SIZE - 1; i > 0; i-- )
        d_y[i] = d_y[i-1];
    d_y[0] = sin(phase) * (-1.0 + 2.0 * double(rand()) / double(RAND_MAX));

    for ( int j = 0; j < PLOT_SIZE - 1; j++ )
        d_z[j] = d_z[j+1];

    d_z[PLOT_SIZE - 1] = 0.8 - (2.0 * phase/M_PI) + 0.4 * 
        double(rand()) / double(RAND_MAX);

    // update the display
    replot();

    phase += M_PI * 0.02;
}

int main(int argc, char **argv)
{
    QApplication a(argc, argv);

    DataPlot plot;
#if QT_VERSION < 0x040000
    a.setMainWidget(&plot);
#endif

    plot.resize(500,300);
    plot.show();

    return a.exec(); 
}
