/*  Sapphire version 1 - an acoustic compiler
    Copyright (C) 1995 James C Finnis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char *rcsid="$Id: formats.c,v 15.0 1995/11/12 20:56:40 white Exp $";

/* File format support. Just a few in here. */

/* Add new file formats to here.

   Pointers to the routines should be added to the table later in the file
   (search for TABLE FOR FUNCTIONS to find this file).
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h> /* need to change these for DOS */
#include <unistd.h>	 /* they're for stat() */

#include "sapphire.h"

extern int debug;
extern LIST filepaths;
#define snark if(DTEST(DEBUG_SAMPLES))printf

extern void wavreadheader(struct filedata *hdr);
extern void wavwheader8(struct filedata *hdr);
extern void wavwheader16(struct filedata *hdr);
extern void wavwrite(struct filedata *hdr,float *data);
extern void wavend(struct filedata *hdr);
extern void wavread(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
					int channel,float *buffer);

/**********************************************************************
 * Dummy routines 
 */

static void dummywheader(struct filedata *data)
{
  snark("called dummywheader for %s\n",data->filename)
  ;
}

static void invalidwheader(struct filedata *data)
{
  yyraise("that format (%s) is for input only. Please specify number of bits.",
		  data->format);
}

static long getnumber(char *fn,char *p)
{
  char buf[80];
  printf("%s not known for %s. Please input : ",p,fn);fflush(stdout);
  gets(buf);
  return atol(buf);
}

static void dummyrheader(struct filedata *data)
{
  extern int samprate,channels;
  snark("called dummyrheader for %s\n",data->filename);

  /* dunno any of these */

  data->numsamples = getnumber(data->filename,"Number of samples");
  data->channels = getnumber(data->filename,"Channels");
  data->samprate = 0;
  data->bitspersample = getnumber(data->filename,"Bits per sample");
}

static void dummyend(struct filedata *data)
{
  snark("called dummyend for %s\n",data->filename);
  ;
}

struct stat *pathstat(char *name,struct stat *sbuf)
{
  NODE *p;
  char buf[256];

  /* look in the current directory and then each
	 directory in the path */

  if(!stat(name,sbuf))return sbuf;

  for(p=filepaths.head;p;p=p->next)
	{
	  char *pth = (char *)(p->data);
	  strcpy(buf,pth);
	  strcat(buf,name);
	  if(!stat(buf,sbuf))return sbuf;
	}
  return NULL;
}


/**********************************************************************
   Raw 8bit format
*/

static void raw8rheader(struct filedata *data)
{
  struct stat buf;

  extern int samprate,channels;
  snark("called raw8rheader for %s\n",data->filename);

  /* get the file data */
  
  if(!pathstat(data->filename,&buf))
	yyraise("unable to stat file %s\n",data->filename);

  if(!(data->channels))data->channels = getnumber(data->filename,"Channels");
  if(!(data->samprate))data->samprate = getnumber(data->filename,"Sampling frequency");
  data->bitspersample = 8;

  data->numsamples = buf.st_size/data->channels;

  printf("file %s is %ld samples, %d channels, 8 bits\n",data->filename,data->numsamples,
		 data->channels);
}

static void raw8write(struct filedata *hdr,float *data)
{
  unsigned long i;
  char c[100];

  snark("writing %d items of raw8 to %s\n",hdr->channels,hdr->filename);
  for(i=0;i<hdr->channels;i++)
	  c[i]=(data[i]*127.0)+128;
  fwrite(&c[0],1,i,hdr->file);
}

static void raw8read(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
					 int channel,float *buffer)
{
  unsigned long i;
  char c;

  snark("reading %ld samples of raw8/%d chans from %s to offset %ld\n",
		numsamps,hdr->channels,hdr->filename,offset);

  for(i=0;i<numsamps;i++)
	{
	  fseek(hdr->file,((i+offset)*hdr->channels)+channel,SEEK_SET);
	  fread(&c,1,1,hdr->file);
	  buffer[i]=((float)c)/127.0;
	}
}




/**********************************************************************
   Raw 16bit format
*/













static void raw16rheader(struct filedata *data)
{
  struct stat buf;

  extern int samprate,channels;
  snark("called raw16rheader for %s\n",data->filename);

  /* get the file data */
  
  if(!pathstat(data->filename,&buf))
	yyraise("unable to stat file %s\n",data->filename);

  if(!(data->channels))data->channels = getnumber(data->filename,"Channels");
  if(!(data->samprate))data->samprate = getnumber(data->filename,"Sampling frequency");
  data->bitspersample = 16;

  data->numsamples = buf.st_size/(2*data->channels);

  printf("file %s is %ld samples, %d channels, 16 bits\n",data->filename,data->numsamples,
		 data->channels);
}

static void raw16write(struct filedata *hdr,float *data)
{
  unsigned long i;
  short c[100];
  snark("writing %d items of raw16 to %s\n",hdr->channels,hdr->filename);
  for(i=0;i<hdr->channels;i++)
	  c[i]=(data[i]*32767.0);

  fwrite(&c[0],sizeof(short),i,hdr->file);
}

static void raw16read(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
					  int channel,float *buffer)
{
  unsigned long i,f,cs;
  short c;
  
  cs=hdr->channels;
  f=offset*cs;
  
  snark("reading %ld samples of raw8/%d chans from %s to offset %ld\n",
		numsamps,hdr->channels,hdr->filename,offset);

  for(i=0;i<numsamps;i++)
	{
	  fseek(hdr->file,sizeof(short)*(f+i*cs+channel),SEEK_SET);
	  fread(&c,sizeof(short),1,hdr->file);
	  buffer[i]=((float)c)/32767.0;
	}
}



/**********************************************************************

  TABLE FOR FUNCTIONS

**********************************************************************/

struct
{
  char *name;
  void (*writeheader)(struct filedata *data);
  void (*writedata)(struct filedata *hdr,float *data);
  void (*end)(struct filedata *hdr);

  void (*readheader)(struct filedata *data);
  int (*readdata)(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
				  int channel,float *buffer);
  char *desc;

} fileformats[]=
{
  {"raw8",dummywheader,raw8write,dummyend,raw8rheader,raw8read,"linear 8 bit raw"},
  {"raw16",dummywheader,raw16write,dummyend,raw16rheader,raw16read,"linear 16 bit raw"},
  {"wav",invalidwheader,wavwrite,wavend,wavreadheader,wavread,"generic WAV (use for input only)"},
  {"wav8",wavwheader8,wavwrite,wavend,wavreadheader,wavread,"8 bit WAV"},
  {"wav16",wavwheader16,wavwrite,wavend,wavreadheader,wavread,"16 bit WAV"},
  {NULL,dummywheader,raw16write,dummyend,dummyrheader,raw16read,"dummy"},
};



static int printformats(void)
{
  int i;
  printf("FileFormats------------------------------\n%-10s %s\n\n","Name","Description");
  for(i=0;;i++)
	{
	  if(!fileformats[i].name)break;
	  printf("%-10s %s\n",fileformats[i].name,fileformats[i].desc);
	}
  printf("-----------------------------------------\n");
}

/* external calls */
static int fnumber=0;
int setformat(char *name)
{
  int i,rv;
  for(i=0;;i++)	{
	  if(!fileformats[i].name)
		{
		  printformats();
		  yyraise("Format not known - %s",name);
		}
	  else if(!strcmp(name,fileformats[i].name)){
		rv=fnumber;
		fnumber=i;return rv;}
	}
}

/* there's a cache of open files, since we might have quite
   a few samples. */

#define MAXFILES 10

static unsigned long filetimestamp=0L;
static int openfiles=0;
static struct filedata *slots[MAXFILES];

void f_closefile(struct filedata *data)
{
  snark("closing file %s\n",data->filename);
  if(data->file)
	{
	  fclose(data->file);
	  data->file=NULL;
	  data->slot=-1;
	  slots[data->slot]=NULL;
	}
}

static int getfreefile(void)
{
  int i,j;
  unsigned long oldest=0L;

  snark("getting free slot...\n");
  /* look for oldest cache or first empty cache */

  for(i=0;i<MAXFILES;i++)
	{
	  if(!slots[i])return i;
	  if(slots[i]->ts<=oldest)
		{
		  oldest=slots[i]->ts;
		  j=i;
		}
	}
  /* close this file, but record the position */
  slots[j]->pos=ftell(slots[j]->file);
  f_closefile(slots[j]);
  snark("   free slot is %d\n",j);
  return j;
}

int f_openfile(struct filedata *data,char *mode)
{
  int slot=getfreefile();
	  
  /* if already open, just return */
  if(data->file)
	{
	  snark("opening file %s (%s) - already open\n",data->filename,mode);
	  return 0;
	}
  else
	{
	  snark("opening file %s (%s) \n",data->filename,mode);
	  data->file=pathfopen(data->filename,mode);
	  if(!(data->file))return 1;
	}
  slots[slot]=data;
  data->slot=slot;
  data->ts=filetimestamp++;

  return 0;
}

/* attempt to reopen a closed file */  
void f_reopenfile(struct filedata *data,char *mode)
{
  if(data->file)return; /* already open! */

  snark("REOPENING file %s (%s) to pos %ld\n",data->filename,mode,data->pos);

  /* open the file as normal */
  if(f_openfile(data,mode))
	yyraise("unable to reopen %s",data->filename);

  /* and seek to the former position */
  fseek(data->file,data->pos,SEEK_SET);
}

void f_writeheader(struct filedata *data)
{
  setformat(data->format);
  if(!data->file && data->slot<0)
	{
	  /* it's been closed - attempt to reopen */
	  f_reopenfile(data,"wb");
	}
  (*(fileformats[fnumber].writeheader))(data);
}

void f_writedata(struct filedata *a,float *data)
{
  setformat(a->format);
  if(!a->file && a->slot<0)
	f_reopenfile(a,"wb");
  (*(fileformats[fnumber].writedata))(a,data);
}

void f_end(struct filedata *a)
{
  setformat(a->format);
  if(!a->file && a->slot<0)
	f_reopenfile(a,"wb");
  (*(fileformats[fnumber].end))(a);
}

void f_readheader(struct filedata *data)
{
  setformat(data->format);
  if(!data->file && data->slot<0)
	f_reopenfile(data,"rb");
  (*(fileformats[fnumber].readheader))(data);
}

int f_rawreaddata(struct filedata *a,unsigned long numsamps,unsigned long offset,
			   int channel,float *buffer)
{
  setformat(a->format);
  if(!a->file && a->slot<0)
	f_reopenfile(a,"rb");
  (*(fileformats[fnumber].readdata))(a,numsamps,offset,channel,buffer);
}

/*********************************************************************************
 * Sample cache management 
 */

long cachesize=-1L,blocksize=-1L;

struct cacheslot
{
  struct filedata *hdr; /* file header */
  unsigned long blkno;	/* offset into file of block stored, /16384 */
  unsigned long ts;
  int channel;
  float *data;
};

static struct cacheslot *cache;
static unsigned long timestamp=1L;

void f_initcache(void)
{
  int i;

  snark("initialising cache\n");

  if(cachesize<0L)cachesize=CACHESIZE;
  if(blocksize<0L)blocksize=BLOCKSIZE;

  cache=emalloc(sizeof(struct cacheslot)*cachesize);

  for(i=0;i<cachesize;i++)
	{
	  cache[i].hdr=NULL;
	  cache[i].data=emalloc(blocksize*sizeof(float));
	  cache[i].ts=0L;
	}
  for(i=0;i<MAXFILES;i++)
	{
	  slots[i]=NULL;
	}

}

static int getfreecache(void)
{
  int i,j=-1;
  unsigned long oldest=0xffffffffL;

  /* look for oldest cache or first empty cache */

  snark("getting free cache\n");
  for(i=0;i<cachesize;i++)
	{
	  if(!cache[i].hdr)
		{
		  snark("  found empty - %d\n",i);
		  return i;
		}

	  if(cache[i].ts<=oldest)
		{
		  oldest=cache[i].ts;
		  j=i;
		}
	}
  if(j==-1)yyraise("cache error - no old slot found!");
  snark("returning oldest - %d (%s)\n",j);
  return j;
}

static int addtocache(struct filedata *hdr,unsigned long blkno,int channel,char *data)
{
  int i=getfreecache();

  snark("adding %s/%ld/%d to cache at %d\n",hdr->filename,blkno,channel,i);
  cache[i].blkno=blkno;
  cache[i].hdr=hdr;
  cache[i].channel=channel;
  if(data)memcpy(cache[i].data,data,blocksize*sizeof(float));
  cache[i].ts=timestamp++;

  return i;
}

/* look for a given block in the cache */

static int findincache(struct filedata *hdr,unsigned long blkno,int channel)
{
  static struct filedata *lastf=NULL;
  static unsigned long lastblk=0L;
  static int lastcache=0,lastchan=0;
  register int i;

  snark("looking for %s/%ld/%d\n",hdr->filename,blkno,channel);

  if(hdr==lastf && blkno==lastblk)
	{
	  cache[lastcache].ts=timestamp++;
	  snark("  found (was last used)\n");
	  return lastcache;
	}

  for(i=0;i<cachesize;i++)
	if(cache[i].blkno == blkno && cache[i].hdr == hdr && cache[i].channel == lastchan)
	  {
		lastblk=blkno;lastf=hdr;lastcache=i;
		lastchan=channel;cache[i].ts=timestamp++;
		snark("  found in cache at %d\n",i);
		return i;
	  }

  /* not in cache, we have to read it in and then return it */

  snark("  not in cache\n");

  i=addtocache(hdr,blkno,channel,NULL);
  f_rawreaddata(hdr,blocksize,blkno*blocksize,cache[i].channel,cache[i].data);
  return i;
}

  
int f_readdata(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
			   int channel,float *buffer)
{
  unsigned long i,endbk,startbk;
  int s;
  float *ptr=buffer;

  startbk=offset/blocksize;
  endbk=startbk+numsamps/blocksize;

  snark("***reading %ld samples from %s/%ld\n",numsamps,hdr->filename,offset);
  for(i=startbk;i<=endbk;i++)
	{
	  s=findincache(hdr,i,channel);
	  memcpy(ptr,cache[s].data,
			 ((i==endbk)?(numsamps%blocksize):blocksize)*sizeof(float));
	  ptr+=blocksize*sizeof(float);

	}
}


/* just get a single sample */
float f_getsample(struct filedata *hdr,unsigned long i)
{
  unsigned long offset;
  unsigned long bk;
  int s;
  
  if(i<0)yyraise("attempt to read -ve sample %s:%ld",hdr->filename,i);
  bk=i/blocksize;
  s=findincache(hdr,bk,0); /* always left channel */
  
  return cache[s].data[i-bk*blocksize];
}
