/*  Sapphire version 1 - an acoustic compiler
    Copyright (C) 1995 James C Finnis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	List of objects:

	Object       Author               Date

	copy         J Finnis             4/10/95
	add          J Finnis             4/10/95
	mul          J Finnis             4/10/95
	sub          J Finnis             4/10/95
	div          J Finnis             4/10/95
	switch       J Finnis             4/10/95
	osc          J Finnis             4/10/95
	adsr         J Finnis             4/10/95
	delay        J Finnis             4/10/95
	mix          J Finnis             4/10/95
	expodecay    J Finnis             4/10/95
	oneshot      J Finnis             4/10/95
	loopsamp     J Finnis             4/10/95
	stereomix    J Finnis             4/10/95
	samphold     J Finnis             4/10/95
	print        J Finnis             4/10/95
	random       J Finnis             4/10/95
	END_OF_OBJECT_LIST
*/

static char *rcsid="$Id: objects.c,v 15.0 1995/11/12 20:56:40 white Exp $";

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "sapphire.h"
#define	snark	if(DTEST(DEBUG_OBJECTS))printf


extern float *getfloat();

/*
	Object functions
*/

extern struct object *self;
extern int timesigdenom,timesignum,samprate,tempo;
extern unsigned long currentsample;

extern float **selfinpbase,**selfoutbase;

/* routines for I/O channel handling */

#define INPUT(n)	(*(selfinpbase[n]))
#define	OUTPUT(n,f)	(*(selfoutbase[n])=(f))


/* copies its single input to its output */
void f_copy(void)
{
	OUTPUT(0,INPUT(0));
}

void f_add(void)
{
	float q=0.0;
	int i;

	for(i=0;i<self->realins;i++)
	{
	  q += INPUT(i);
	}
	OUTPUT(0,q);
}

void f_mul(void)
{
	float q=1.0;
	int i;

	for(i=0;i<self->realins;i++)
	{
		q *= INPUT(i);
	}
	OUTPUT(0,q);
}


void f_sub(void)
{
	OUTPUT(0,INPUT(0)-INPUT(1));
}


void f_div(void)
{
	float f=INPUT(1);
	if(f==0.0)yyraise("runtime error: divide by zero");
	
	OUTPUT(0,INPUT(0)/f);
}

void f_switch(void)
{
	float f=INPUT(0);
	float g=INPUT(1);
	
	if(f<g)OUTPUT(0,INPUT(2));
	else OUTPUT(0,INPUT(3));
}

/************************************************************************

  OSC: waveform oscillator. Takes
    (wave)
	frequency
	amplitude

************************************************************************/

void i_osc(void)
{
	self->private=getfloat();
	*((float *)(self->private))=0;
}

void f_osc(void)
{
	float *index,size,incr;
	
	index= (float *)(self->private);
	size=(float)self->wavsize;
	
	/* output the current index value */
	OUTPUT(0,INPUT(1)*(self->wave[(long)*index]));

	/* increment the index */
	
	incr=INPUT(0)*size;
	if(incr<=0)incr=0;
	*index += incr/(float)samprate;

	while(*index>=size)*index -= size;
	while(*index<0)*index+=size;

}

/************************************************************************

	ADSR envelope shaper. Takes noteon,gate,a,d,s,r,signal

		noteon: triggers start of note and re-initialises
		gate: true when note is on
		a,d,r: TIMES in SECONDS
		s: level
		signal: signal to be shaped

************************************************************************/

struct adsr
{
	float rate,lval;
	unsigned long gateoff,x;
};
	
void i_adsr(void)
{
	struct adsr *a;
	self->private=a=emalloc(sizeof(struct adsr));
	a->gateoff=0;
	a->lval=0.0;
	a->rate=0.0;
	a->x=0;
}


void f_adsr(void)
{
	struct adsr *a;

	unsigned long attack,decay,release,x;
	float sustain;

	a=(struct adsr *)(self->private);

	attack=(unsigned long)(samprate*INPUT(2));
	if(attack<1)attack=1;
	decay=(unsigned long)(samprate*INPUT(3));
	sustain=INPUT(4);
	release=(unsigned long)(samprate*INPUT(5));
	if(release<1)release=1;

	if(INPUT(0))	/* noteon? */
	{
		a->x=0;
		a->rate=1.0/(float)attack;		/* reset the rate */
		a->gateoff=0;
	}
	x=a->x;
	
	if(INPUT(1))	/* gated? */
	{
		if( a->lval>=1.0)
		{
			a->rate=(sustain-1.0)/(float)(decay+1);
		}
		else if(a->rate<0 && a->lval<=sustain)
		{
			a->lval=sustain;a->rate=0.0;
		}
	}
	else
	{
		if(!(a->gateoff))
		{
			a->gateoff=1;
			a->rate= -sustain/(float)release;
		}	
	}
	a->lval += a->rate;
	if(a->lval<0.0)a->lval=0.0;
	a->x++;
	OUTPUT(0,INPUT(6) * (a->lval));
}
	
/************************************************************************

	INSTADDR:	Instance output adder. Internal use only!

************************************************************************/

void f_instaddr(void)
{
	float q=0.0;
	int i;

	for(i=0;i<self->realins;i++)
	{
		q += INPUT(i);
	}

	OUTPUT(0,q);
}



/************************************************************************

	DELAY:	takes
		signal to be delayed
		maximum size of delay line (only used in init) (seconds)
		current size of delay line (seconds)

************************************************************************/

struct delay
{
	long rdoff,wtoff;
	float *line;
	long maxsize,cursize;
};

void i_delay(void)
{
	unsigned long i;
	
	struct delay *d;
	self->private=d=emalloc(sizeof(struct delay));
	d->maxsize=(unsigned long)(samprate*INPUT(1));
	d->cursize=0;
	if(d->cursize>=d->maxsize)yyraise("delay too long");
	d->rdoff=0;
	d->wtoff=0;
	d->line=(float *)emalloc(sizeof(float)*d->maxsize);
	for(i=0;i<d->maxsize;i++)d->line[i]=0.0;
	
}

void f_delay(void)
{
	struct delay *d;
	float *line;
	unsigned long old;

	d=(struct delay *)(self->private);

	line=d->line;
	old=d->cursize;
	d->cursize=(unsigned long)(samprate*INPUT(2));
	if(old != d->cursize)	/* size change */
	{
		if(d->cursize>d->maxsize)
		 yyraise("delay too long - max %ld samples, current %ld",
			 d->maxsize,d->cursize);
		
		d->rdoff=d->wtoff-d->cursize;
		if(d->rdoff<0)d->rdoff = d->maxsize+d->rdoff;
	}
	
	if(d->wtoff>d->maxsize)d->wtoff -= d->maxsize;
	if(d->rdoff>d->maxsize)d->rdoff -= d->maxsize;

	line[d->wtoff++]=INPUT(0);
	OUTPUT(0,line[d->rdoff++]);
}

/************************************************************************

	MIX: takes some signals and averages them

************************************************************************/

void f_mix(void)
{
	float q=0.0;
	int i;

	for(i=0;i<self->realins;i++)
	{
		q += INPUT(i);
	}

	OUTPUT(0,q/(float)(self->realins));
}

/************************************************************************

	EXPODECAY:	takes trigger,startval,timeto16th,signal
		trigger (usually a $noteon)
		start value of signal (usually 1)
		time taken to decrease to 1/16th of original
		signal to be shaped

************************************************************************/

struct expodecay
{
  unsigned long countdown;
  float value;
  float lastval;
  float multiplier;
};

void i_expodecay(void)
{
  struct expodecay *e;
  double a,b,c;

  self->private=e=emalloc(sizeof(struct expodecay));

  e->countdown=(unsigned long)((float)samprate*INPUT(2));
  e->value = 0.0;

  /* multiplier = exp( log(1/16) * count ) */

  a=(float)(e->countdown);
  b=log(1.0/16.0);
  c=b/a;
  e->multiplier=(float)exp(c);
  e->lastval=0.0;
}
	
void f_expodecay(void)
{
  struct expodecay *e=self->private;
  float res,f;
  if(INPUT(0)) /* noteon? */
	{
	  e->value = INPUT(1); /* get new value */
	}
  else
	e->value *= e->multiplier;

  /* now we stick a very simple LPF (or smoothing function)
	 on the output to stop the orrible spikey bit */
  
  res=0.1*e->value + 0.9*e->lastval;
  e->lastval=res;

  f=INPUT(3)*res;

  OUTPUT(0,f*res);
}

/************************************************************************

	ONESHOT: One-shot wave/sample playback
	 noteon
	 pitch
	 amplitude

************************************************************************/

void i_oneshot(void)
{
	self->private=(void *)getfloat();
	*((float *)(self->private))=0;
}

void f_oneshot(void)
{
	float *index,size,incr;
	struct filedata *hdr;

	index= (float *)(self->private);
	size=(float)self->wavsize;

	/* running when index>=0.0 */

	if(INPUT(0)) *index=0.0; /* restart */

	if(*index<0.0)
	  {
		OUTPUT(0,0.0);
		return;
	  }
	
	/* get filedata structure */
	hdr=(struct filedata *)(self->wave);


	/* output the current index value */
	OUTPUT(0,INPUT(2)*(f_getsample(hdr,(long)*index)));

	/* increment the index. This is different to the way osc() works,
	   since we have to take the sampling frequency into account. */

	/* this is the ratio of the note we want to play and the frequency
	   of the sampled note */

	incr = INPUT(1)/hdr->pitch;

	/* take into account different sampling frequencies */

	incr *= hdr->samprate/((float)samprate);

	*index += incr;

	if(*index>=size)
	  {
		/* all done! Turn off sample */
		*index=-1.0;
	  }
}



/************************************************************************

	STEREOMIX: Stereo mix/pan
	 takes any number of signal, pan (-1 left to 1 right) and amplitude
	 triplets.
	 returns 2 outputs, left and right.
	 
************************************************************************/

void f_stereomix(void)
{
  float oleft=0.0,oright=0.0,rightamp;
  float thissig,thispan,thisamp;
  int numins=self->realins/3,inp,i;

  for(i=0,inp=0;i<numins;i++)
	{
	  thissig=INPUT(inp);
	  thispan=INPUT(inp+1);
	  thisamp=INPUT(inp+2);
	  inp+=3;

	  thissig *= thisamp;

	  rightamp=(thispan+1.0)/2.0;

	  oleft += rightamp*thissig;
	  oright += (1.0-rightamp)*thissig;
	}
  OUTPUT(0,oleft/(float)numins);
  OUTPUT(1,oright/(float)numins);
}

/************************************************************************

  EQ FILTER: signal, freq, gain, Q

************************************************************************/

struct filter
{
  /* inputs - quality, decibels, centre freq */
  float Q, dB, cf;                      /* the input parameters.*/
  float Qp,dBp,cfp;						/* and their prior values */
  float c1, c2, d1, d2;                 /* the coefficients. */
  float prevx1, prevx2, prevy1, prevy2; /* the previous 2 x,y's.*/
  float prevout;
};

void i_filter(void)
{
  struct filter *fl;

  self->private=emalloc(sizeof(struct filter));
  fl=(struct filter *)(self->private);

  fl->Q=1.0;
  fl->cf=1000.0;
  fl->dB=0.0;
  fl->Qp=1.0;
  fl->cfp=1000.0;
  fl->prevout=0.0;
  fl->dBp=1.0; /* deliberately different, to force recalc */
  fl->c1=0.0;fl->c2=0.0;fl->d1=0.0;fl->d2=0.0;
  fl->prevx1=0.0;fl->prevx2=0.0;fl->prevy1=0.0;fl->prevy2=0.0;
}

#ifndef PI
#define PI 3.1415927
#endif

void f_filter(void)
{
  float bwdth, phi, Rav, gain, Aav,i,y,a,b;
  float delr, rpole, rzero, apole, azero;
  float SF;

  struct filter *x;
  x=(struct filter *)(self->private);

  /* recalculate on change */

  x->cf=INPUT(1);
  x->dB=INPUT(2);
  x->Q=INPUT(3);
  
  if((x->cf != x->cfp) || (x->Q != x->Qp) || (x->dB != x->dBp))
	{
	  SF=(float)samprate;
	  bwdth = x->cf / x->Q;
	  /*if (bwdth > 20000) bwdth = 20000.0;*/
	  phi = sqrt(bwdth) * 100.0;
	  Rav = cos(phi * PI / SF);
	  gain = pow(10.0, x->dB / 20.0);
	  Aav = x->cf * 2.0 * PI / SF;
	  if(debug==2)printf("gain=%f, bwdth=%f, Rav=%f, Aav=%f\n", gain, bwdth, Rav, Aav);
  
	  delr = (1.0 - Rav) * (gain - 1.0) / (gain + 1.0);
	  rpole = Rav + delr;
	  rzero = Rav - delr;
	  if (rpole > 0.999999) rpole = 0.999999;
	  if (rzero > 0.999999) rzero = 0.999999;
	  if (rpole < 0.0) rpole = 0.0;
	  if (rzero < 0.0) rzero = 0.0;
	  apole = Aav;
	  azero = Aav;
	  if (apole > 3.14159) apole = 3.14159;
	  if (azero > 3.14159) azero = 3.14159;
	  if (apole < 0.0) apole = 0.0;
	  if (azero < 0.0) azero = 0.0;

	  if(debug==2)printf("rpole=%f, rzero=%f, apole=%f, azero=%f\n", rpole, rzero, apole, azero);
	
	  x->d1 = -2.0 * rzero * cos(azero);
	  x->d2 = rzero * rzero;
	  x->c1 = 2.0 * rpole * cos(apole);
	  x->c2 = -rpole * rpole;

	  if(debug==2)printf("c1=%f, c2=%f, d1=%f, d2=%f\n",x->c1,x->c2,x->d1,x->d2);
	  x->cfp=x->cf;x->Qp=x->Q;x->dBp=x->dB;
	}

  /* OK, now we have the parameters, do the actual calculations */

  i = INPUT(0); /* get the signal */
  a = i + x->d2*x->prevx2 + x->c1*x->prevy1;
  b = x->d1*x->prevx1 + x->c2*x->prevy2;
  y = a + b;

  y = y-x->prevout*0.0001;
  x->prevout=y;
  OUTPUT(0,y);

  x->prevx2 = x->prevx1;
  x->prevx1 = i;
  x->prevy2 = x->prevy1;
  x->prevy1 = y;
}
 
/************************************************************************

  SAMPHOLD: signal, interval

************************************************************************/

struct samphold
{
  unsigned long count;
  float value;
};

void i_samphold(void)
{
  struct samphold *s;

  s=emalloc(sizeof(struct samphold));
  self->private=(void *)s;

  s->count=0L;
  s->value=0.0;
}

void f_samphold(void)
{
  float sig;
  struct samphold *s;
  s=(struct samphold *)(self->private);

  if(!s->count--)
	{
	  s->count=(unsigned long)(samprate*INPUT(1));
	  s->value=INPUT(0);
	}
  OUTPUT(0,s->value);
}

/************************************************************************

  PRINT: output same as input, but prints to screen. Used for debugging.

************************************************************************/

void f_print(void)
{
  float sig;
  sig=INPUT(0);
  printf("%f\n",sig);
  OUTPUT(0,sig);
}


/************************************************************************

	LOOPSAMP: plays samples with loops in
	  takes
	  	(sample)
0		noteon
1		gate
2		pitch
3		amplitude
4		loopstart%
5		loopend%
6		looptransitiontime%
	  

************************************************************************/

void i_loopsamp(void)
{
	self->private=(void *)emalloc(sizeof(float));
	*((float *)(self->private))=0;
}

void f_loopsamp(void)
{
  float size,incr,loopstart,loopend,looptime,*index;
  float a,b,gate;
  struct loopsamp *l;
  struct filedata *hdr;

  l=(struct loopsamp *)(self->private);
  index= (float *)(self->private);

  if(*index<0.0)
	  {
		/* check for noteon */
		if(INPUT(0))
		  *index=0.0;
		else
		  {
			OUTPUT(0,0.0);
			return;
		  }
	  }

  /* get filedata structure */
  hdr=(struct filedata *)(self->wave);
  size=(float)(self->wavsize);

  /* work out the loop points */

  loopstart=INPUT(4)*size/100.0;
  loopend=INPUT(5)*size/100.0;
  looptime=loopend-INPUT(6)*(loopend-loopstart)/100.0;

  /* are we in a loop transition phase? Loops only work when gated. */

  gate=INPUT(1);

  if(!gate || *index < looptime)
	{
	  /* nope - easy case */
	  /* output the current index value */
	  OUTPUT(0,INPUT(3)*(f_getsample(hdr,(long)*index)));
	}
  else
	{
	  /* work out how far through the transition we are */
	  /* 'a' is the amount of the sample at the current index we use, 
		 and 'b' is the amount from the beginning of the loop. At
		 loopend-looptranstime, a=1 and b=0. At loopend, a=0 and b=1,
		 and we seamlessly move index to loopstart+looptranstime */

	  b=(*index-looptime) / (loopend-looptime);
	  a=1.0-b;

	  /* work out the signals */
	  a= a* f_getsample(hdr,(long)*index);
	  b= b* f_getsample(hdr,(long)(loopstart+(*index-looptime)));
	  OUTPUT(0,INPUT(3)*(a+b));
	}

  /* increment the index. This is different to the way osc() works,
	 since we have to take the sampling frequency into account. */

  /* this is the ratio of the note we want to play and the frequency
	 of the sampled note */

  incr = INPUT(2)/hdr->pitch;
  
  /* take into account different sampling frequencies */

  incr *= hdr->samprate/((float)samprate);

  *index += incr;

  if(gate && *index>=loopend)
	{
	  /* end of loop - wraparound time */
	  *index=loopstart+(loopend-looptime);
	}
  else if(*index>=(float)size)
	{
	  /* end of WHOLE THING! Turn off sample. */
	  *index=-1.0;
	}
}

/************************************************************************

  RANDOM: random number generator. Crap for whitenoise, but good for
  sample and hold stuff. No inputs.

************************************************************************/

#ifndef RAND_MAX
#define RAND_MAX (1L<<31L)
#endif
void f_random(void)
{
  float r,rng;

  rng=INPUT(0);

  r=(float)rand();
  r/=RAND_MAX/2;
  r-=1;

  OUTPUT(0,r*rng);
}
