%{

/*  Sapphire version 1 - an acoustic compiler
    Copyright (C) 1995 James C Finnis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char *rcsid="$Id: parser.y,v 15.0 1995/11/12 20:56:40 white Exp $";

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#define YYDEBUG 1
#ifndef UNIX
#define yyoverflow yyraise
#endif

#define	snark	if(DTEST(DEBUG_PARSE))printf
#include "sapphire.h"
#include "objects.h"

void yyerror(char *);
void yyraise(char *,...);
void warn(char *,...);
void raiseatline(int l,char *,...);

NODE *findeventbefore(unsigned long t);
unsigned long tmpid=0L;
static float zero_float_value=0.0;

struct paramstruct
{
	union
	{
		char var[KEYSIZE];
		float real;
	} d;
	int isvar;
};

NODE *join(NODE *a,NODE *b);
NODE *do_varparam(char *var);
NODE *do_realparam(float r);
NODE *do_instparam(char *id,int voices,char *instr,NODE *params);
NODE *do_objparam(char *name,char *wav,NODE *params);
NODE *do_insobjparam(char *name,char *wav,NODE *params);
struct object *do_object(char *str,char *wav,int no,
	NODE *p,char o[][KEYSIZE]);
void do_instance(char *id,int voices,char *instr,NODE *params,int no,
				 char o[][KEYSIZE]);
void do_insobject(char *str,char *wav,int no,
	NODE *p,char o[][KEYSIZE]);
void do_event(unsigned long t,int isend);
void do_ramp_event(char *varname,float startval,unsigned long starttime,
				   float endval,unsigned long endtime);
void do_instrument(char *str,float duradd);
void do_wave(char *str);
float *getfloat(float f);
void do_sample(char *id,char *format,char *filename,float pitch,float samprate,int chans);
void do_note(char *instance,unsigned long time,char *scalename,
	float amp,unsigned long dur);
struct scoreitem *make_item(void);
unsigned long get_time_real(long mins,long secs,long cents);
unsigned long get_time_bar(long bar,long beat,long frac);
unsigned long get_time_secs(float secs);
static void clearparamlist(NODE *p);

struct insobject
{
  char wave[KEYSIZE];
  int outputct;
  NODE *params;
  char outputs[NUMPARAMS][KEYSIZE];
};

struct instrument
{
	unsigned long duradd;
	NODE *objects;
};

char ioutputarray[NUMPARAMS][KEYSIZE];
int ioutputct=0;

char outputarray[NUMPARAMS][KEYSIZE];
int outputct=0;

float wavearr[MAXWAV];
int wavct=0;

char eventvars[NUMPARAMS][KEYSIZE];
float eventvals[NUMPARAMS];
int eventct=0;

extern int lines;
extern struct object *self;
extern struct filedata opfilehdr;


char scorename[KEYSIZE];
int transopt=DEFTRANSOPT,timesopt=1;
float speedopt=DEFSPEEDOPT,ampopt=DEFAMPOPT;

extern int timesigdenom,timesignum,samprate,tempo;
unsigned long endsample=0;

float amplitude,duration;
char currnotepitches[NUMPARAMS][KEYSIZE];
int npitch=0;
struct scale *thisscale;
LIST waves={NULL,NULL};
LIST samples={NULL,NULL};
LIST objects={NULL,NULL};
LIST events={NULL,NULL};
LIST vars={NULL,NULL};
LIST insobjects={NULL,NULL};
LIST instruments={NULL,NULL};
LIST currentscale={NULL,NULL};int numscaleitems=0;
LIST scales={NULL,NULL};
LIST currentscore={NULL,NULL};
LIST scores={NULL,NULL};
LIST instances={NULL,NULL};
LIST ramps={NULL,NULL};

%}

%start program

%union
{
  char str[64];
  long integer;
  float real;
  unsigned long tval;
  NODE *node;
}

%token <real>	REAL
%token <integer>	INTEGER
%token <real>	FRACTION
%token <str>	ID
%token <str>	PARAM
%token <str>	LOCAL
%token <str>	STRING

%token EVENT AT PLAYSCORE SCORE INSTRUMENT END SPEED AMP TRANS WAVE TIMES
%token INSTANCE OF SCALE NOTE DURADD SAMPLE FORMAT FILETOK PITCH SAMPRATE
%token FROM TO RAMP SIGNATURE TEMPO CHANS

%type	<real>	real
%type	<integer>	posint
%type	<tval>	time
%type	<str>	variable
%type   <node>  param
%type   <node>  paramlist
%type   <node>  insparam
%type   <node>  insparamlist
%%

program		:	itemlist
		;

itemlist	:	item
		|	itemlist item
 		;

item		:	object
		|	event
		|	wave
		|	sample
		|	instrument
		|	instance
		|	score
		|	playscore
		|	scale
		|	tempochange
		|	timesigchange
		|	note
		|	error
			{
				yyraise("expected a declaration");
			}
		;

tempochange :	TEMPO INTEGER ';'
			{
			  tempo=$2;
			}
			;

timesigchange : SIGNATURE INTEGER INTEGER ';'
			{
			  timesignum=$2;timesigdenom=$3;
			}
			;

note		:	NOTE time ID ID notelist time real';'
			{
				/* time, instance, scale, notes,
				duration, amp */
				do_note($3,$2,$4,$7,$6);
			}
		;
		
scale		:	SCALE ID '{' scalelist '}'
			{
				NODE *l;
				struct scale *sp;
				int i;

				if(!(find(&scales,$2)))
				  {
					
					sp=emalloc(sizeof(struct scale));
					sp->p=emalloc(sizeof(struct scaleitem)*
								  numscaleitems);

				
					for(i=0,l=currentscale.head;l;l=l->next)
					  {
						strcpy(sp->p[i].name,l->key);
						sp->p[i++].pitch=
						  *((float *)(l->data));
					  }
					sp->num=numscaleitems;
				
					insertatend(&scales,$2,sp);
					currentscale.head=NULL;
					currentscale.tail=NULL;
					numscaleitems=0;
				  }
			}
		;

scalelist	:	scaleitem
		|	scalelist ',' scaleitem
		;

scaleitem	:	ID ':' real
			{
				float *pf;
				if($3 < 0.0)
					yyraise("negative pitch in scale");

				if(find(&currentscale,$1))
					yyraise("note %s already defined",$1);
					
				pf=emalloc(sizeof(float));
				*pf=$3;
				insertatend(&currentscale,$1,pf);
				numscaleitems++;
			}
		;

instance	:	outputlist ':' INSTANCE ID posint OF ID paramlist';'
			{
				do_instance($4,$5,$7,$8,outputct,outputarray);
				outputct=0;
			}
		|	outputlist ':' INSTANCE ID posint OF ID ';'
			{
				do_instance($4,$5,$7,NULL,outputct,outputarray);
				outputct=0;
			}		
		;

object		:	outputlist ':' ID paramlist ';'
			{
				do_object($3,NULL,outputct,$4,outputarray);
				clearparamlist($4);
				outputct=0;
			}
		|	outputlist ':' ID '&' ID paramlist ';'
			{
				struct object *o;
				do_object($3,$5,outputct,$6,outputarray);
				clearparamlist($6);
				outputct=0;

			}
		;

event		:	EVENT AT time eventlist ';'
			{
				do_event($3,0);
				eventct=0;
			}
		|	END AT time ';'
			{
				do_event($3,1);
				eventct=0;
			}
		|	RAMP ID FROM real AT time TO real AT time ';'
			{
				do_ramp_event($2,$4,$6,$8,$10);
				eventct=0;
			}
		;

instrument	:	INSTRUMENT ID '{' objlist '}'
			{
				do_instrument($2,DEFAULT_DURADD);
				insobjects.head=NULL;
				insobjects.tail=NULL;
			}
			|	INSTRUMENT ID DURADD real '{' objlist '}'
			{
				/* nasty hack - duration multiplier, for long-release notes. */

				do_instrument($2,$4);
				insobjects.head=NULL;
				insobjects.tail=NULL;
			}
			;

score		:	scorehdr '{' scorelist '}'
			{
				struct score *lp;

				if(find(&scores,scorename))
					yyraise("score %s already defined",scorename);

				lp=emalloc(sizeof(struct score));
				lp->list=currentscore;
				lp->scale=thisscale;
				insertatend(&scores,scorename,lp);
				currentscore.head=currentscore.tail=NULL;
			}
		;

scorehdr	:	SCORE ID ID
			{
				NODE *n;
				
				strcpy(scorename,$2);
				if(!(n=find(&scales,$3)))
					yyraise("scale %s not found",$3);
				thisscale=(struct scale *)(n->data);
			}
		;

playscore	:	PLAYSCORE ID ID playoptlist AT time ';'
			{
				do_playscore($2,$3,$6,transopt,speedopt,ampopt,timesopt);
				transopt=DEFTRANSOPT;
				speedopt=DEFSPEEDOPT;
				ampopt=DEFAMPOPT;
				timesopt=1;
			}
		|	PLAYSCORE ID ID AT time ';'
			{
				do_playscore($2,$3,$5,transopt,speedopt,ampopt,timesopt);
			}
		;

sample		:	SAMPLE ID FORMAT ID FILETOK STRING PITCH real
			{
				do_sample($2,$4,$6,$8,-1.0,-1);
			}
			|	SAMPLE ID FORMAT ID FILETOK STRING PITCH real SAMPRATE real
			{
				do_sample($2,$4,$6,$8,$10,-1);
			}
			| 	SAMPLE ID FORMAT ID FILETOK STRING PITCH real SAMPRATE real CHANS INTEGER
			{
				do_sample($2,$4,$6,$8,$10,$12);
			}
		;

wave		:	WAVE ID '{' wavlist '}'
			{
				do_wave($2);
				wavct=0;
			}
		;

wavlist		:	wavitem
		|	wavlist wavitem
		;

wavitem		:	real
			{
				if(wavct==MAXWAV)yyraise("wave too big");
				wavearr[wavct++]=$1;
			}
		;
		
outputlist	:	outputitem
		|	outputlist ',' outputitem
		;

outputitem	:	variable
			{
				if(outputct==NUMPARAMS)
					yyraise("too many outputs");
				strcpy(outputarray[outputct++],$1);
			}
		;


paramlist	:	param
            |	paramlist param
			{
			  $$=join($1,$2);
			}
		;

insparamlist	:	insparam
            |	insparamlist insparam
			{
			  $$=join($1,$2);
			}
		;

param		:	variable
			{
			  $$=do_varparam($1);
			}
		|	real
			{
			  $$=do_realparam($1);
			}
		|	'{' ID paramlist '}'
			{
			  /* an object as a parameter */
			  $$=do_objparam($2,NULL,$3);
			}
		|	'{' ID '&' ID paramlist '}'
			{
			  /* an object with wave/sample as a parameter */
			  $$=do_objparam($2,$4,$5);
			}
		|	'{' INSTANCE ID posint OF ID paramlist '}'
			{
			  $$=do_instparam($3,$4,$6,$7);
			}
		|	'{' INSTANCE ID posint OF ID '}'
			{
			  $$=do_instparam($3,$4,$6,NULL);
			}
		;

insparam		:	variable
			{
			  $$=do_varparam($1);
			}
		|	real
			{
			  $$=do_realparam($1);
			}
		|	'{' ID insparamlist '}'
			{
			  $$=do_insobjparam($2,NULL,$3);
			}
		|	'{' ID '&' ID insparamlist '}'
			{
			  $$=do_insobjparam($2,$4,$5);
			}
		|	'{' INSTANCE ID posint OF ID insparamlist '}'
			{
			  yyraise("embedded instances not supported in instruments");
			}
		|	'{' INSTANCE ID posint OF ID '}'
			{
			  yyraise("embedded instances not supported in instruments");
			}
		;

eventlist	:	eventspec
		|	eventlist ',' eventspec
		;

eventspec	:	variable '=' real
			{
				if(eventct==NUMPARAMS)
					yyraise("too many events");
				eventvals[eventct]=$3;
				strcpy(eventvars[eventct++],$1);
			}
		;

objlist		:	insobject
		|	objlist insobject
		;

insobject	:	outputlist ':' ID insparamlist ';'
			{
				do_insobject($3,NULL,outputct,$4,outputarray);
				outputct=0;
			}					
		|	outputlist ':' ID '&' ID insparamlist ';'
			{
				do_insobject($3,$5,outputct,$6,outputarray);
				outputct=0;
			}
		;

scorelist	:	scoreitem
		|	scorelist scoreitem
		;

scoreitem	:	notelist ',' real ';'
			{
				duration=$3;
				amplitude=1.0;
				insertatend(&currentscore,"SCI",make_item());
			}
		|	notelist ',' real ',' real ';'
			{
				duration=$3;
				amplitude=$5;
				insertatend(&currentscore,"SCI",make_item());
			  }
		|	'.' ',' real ';'
	   		  {
				  duration=$3;
				  amplitude=0.0;
				  insertatend(&currentscore,"SCI",make_item());
			  }
		;

notelist	:	noteitem
		|	notelist '+' noteitem
		;

noteitem	:	ID
			{
				if(npitch==NUMPARAMS)
					yyraise("too many notes at once!");
				strcpy(currnotepitches[npitch++],$1);
			}
		|	real
			{
				if(npitch==NUMPARAMS)
					yyraise("too many notes at once!");
				sprintf(currnotepitches[npitch++],"%f",$1);
			}			
		;

playoptlist	:	playopt
		|	playoptlist playopt
		;

playopt		:	SPEED real
			{
				speedopt=$2;
				if(speedopt<=0.0)yyraise("invalid speed %f",speedopt);
			}
		|	TRANS INTEGER
			{
				transopt=$2;
			}
		|	AMP real
			{
				ampopt=$2;
			}
		|	TIMES INTEGER
			{
			  timesopt=$2;
			}
		;

real		:	INTEGER
			{
				$$=(float)$1;
			}
		|	REAL
			{
				$$=$1;
			}
		;

variable	:	PARAM
			{
				strcpy($$,$1);
			}
		|	LOCAL
			{
				strcpy($$,$1);
			}
		|	ID
			{
				strcpy($$,$1);
			}
		;
		
time		:	'(' posint ')'
			{
				$$=$2;
			}
		|	posint ':' posint ':' posint
			{
				$$=get_time_real($1,$3,$5);
			}
		|	real
			{
			  float f;
			  f=$1*(float)samprate;
			  $$=(unsigned long)f;
			}
		|	'/' posint ':' posint ':' posint '/'
			{
				unsigned long q;
				q=get_time_bar($2,$4,$6);
				$$=q;
			}
		|	'/' posint ':' posint '/'
			{
				unsigned long q;
				q=get_time_bar($2,$4,0);
				$$=q;
			}
		;

posint		:	INTEGER
			{
				if($1<0)yyraise("%d must be >=0",$1);
				$$=$1;
			}
		;
%%

int paramcount(NODE *params)
{
  int c=0;
  while(params)
	{
	  params=params->next;
	  c++;
	}
  return c;
}
NODE *join(NODE *a,NODE *b)
{
  register NODE *t;
  if(!a)return b;
  if(!b)return a;
  for(t=a;t->next;t=t->next);
  t->next=b;return a;
}
static void clearparamlist(NODE *p)
{
  NODE *q;
  for(;p;p=q)
  {
	q=p->next;
	free(p->data);
	free(p);
  }
}

/* Add an object to the internal object structure */

struct object *do_object(char *str,char *wav,int no,
	NODE *p,char o[][KEYSIZE])
{
	int i,j,t,np;
	float *last,*var;
	struct object *obj;
	NODE *n,*varnode;
	
	np=paramcount(p);

	for(i=0;i<NUMOBJS;i++)
		if(!strcmp(objtab[i].name,str))break;
	if(i==NUMOBJS)yyraise("unknown object type : %s",str);

	if(!wav)
	  {
		if(objtab[i].flags & OF_HASWAVE)
		  yyraise("object %s requires a wave table",str);
		if(objtab[i].flags & OF_HASSAMPL)
		  yyraise("object %s requires a sample",str);
	  }
		

	if(!(objtab[i].flags & OF_VARARGS))
	{
		if(np<objtab[i].numins)
			yyraise("not enough inputs for %s. %d got, %d required",str,np,
				objtab[i].numins);
		if(np>objtab[i].numins)
			warn("too many inputs for %s. %d required",str,np,
				objtab[i].numins);
	}

	if(no<objtab[i].numouts)
		warn("not enough outputs for %s. %d got,%d required",str,no,
			objtab[i].numins);

	obj=emalloc(sizeof(struct object));
	insertatend(&objects,str,obj);
	obj->func=objtab[i].func;
	obj->type=i;
	
	snark("object %s created, %d ins and %d outs\n",str,
		objtab[i].numins,objtab[i].numouts);
	
	/* got the object, now deal with input plumbing */

	obj->outsused=no;
	obj->numins=objtab[i].numins;
	obj->realins=np;

	t=obj->numins;

	if(np>t)t=np;
	j=0;
	for(n=p;j<t;j++)
	  {
		struct paramstruct *par;
		obj->invars[j]=NULL;
		if(j<np)
		  {
			par=(struct paramstruct *)(n->data);
			if(par->isvar)
			  {
				varnode=mkorfindvar(par->d.var);
				var=&(varnode->f);
				snark(" parameter %s is var at %lx.\n",
					  par->d.var,var);
				obj->inputs[j]=var;
				obj->invars[j]=varnode;
			  }
			else
			  {
				snark(" parameter %f is real.\n",
					  par->d.real);
				obj->inputs[j]=getfloat(par->d.real);
			  }
			n=n->next;
		  }
		else
		  {
			snark(" parameter omitted, setting to 0.\n");
			obj->inputs[j]= &zero_float_value;
		  }
	  }


	for(j=0;j<obj->numins;j++)
	{
		if(!(obj->inputs[j]))
			yyraise("object %s has null input at %d",str,j);
	}
	/* inputs done, now the outputs */

	obj->numouts=objtab[i].numouts;

	for(j=0;j<no;j++)
	{
	  varnode=mkorfindvar(o[j]);
	  var= &(varnode->f);
	  obj->outvars[j]=varnode;
	  if(j<objtab[j].numouts)
		{
		  last=obj->outputs[j]=var;
		  snark(" output %s OK.\n",o[j]);
		}
	  else
		{
		  snark(" extra output.\n");
		  obj->outputs[j]=var;
		}
	}
	obj->lastval=last;

	snark("lastval set\n");
	/* get the wave pointer */

	if(wav)
	  {
		if(objtab[i].flags & OF_HASWAVE)
		  {
			if(!(n=find(&waves,wav)))
			  yyraise("wave table %s not found",wav);
			obj->wave=(float *)(n->data);
			obj->wavsize=n->t;
		  }
		else if(objtab[i].flags & OF_HASSAMPL)
		  {
			struct filedata *hdr;
			if(!(n=find(&samples,wav)))
			  yyraise("sample %s not found",wav);

			hdr=(struct filedata *)(n->data);
			obj->wave=(float *)hdr;
			if(!(obj->wavsize=hdr->numsamples))
			  obj->wavsize=10000;
			
		  }
	  }
   
	snark("wave set\n");
	
	/* and run the initfunc */

	if(objtab[i].initfunc)
	  {
		extern float **selfinpbase,**selfoutbase;
		self=obj;
		selfinpbase=&(self->inputs[0]);
		selfoutbase=&(self->outputs[0]);
		(*(objtab[i].initfunc))();
	  }
	self=NULL;

	snark(" and object made OK.\n");
	return(obj);
}

static NODE *makeevent(unsigned long t)
{
  NODE *n;

  /* get the event list for this time
	 or make a new one */
  
  n=findeventbefore(t);

  if((n==NULL) || (n->t != t))
	{
	  snark("New event list : %ld\n",t);
	  if(n)
		{
		  snark("previous events at %ld\n",n->t);
		}
	  else
		{
		  snark("no event before %ld.\n",t);
		}
	  if(!n)
		{
		  insertatend(&events,"EVENT",NULL);
		  n=events.head;
		}
	  else
		  n=insertafter(&events,n,"EVENT",NULL);

	  n->t=t;
	}
  return n;
}
  
void do_event(unsigned long t,int isend)
{
	NODE *n;
	struct event *e;
	int i;
	
	if(isend)
	{
		endsample=t;
	}
	else
	{
snark("making event at %ld\n",t);
	  n=makeevent(t);
	  for(i=0;i<eventct;i++)
		{
		  e=emalloc(sizeof(struct event));
		  e->type=E_NORMAL;
		  e->next=(struct event *)(n->data);
		  (struct event *)(n->data)=e;
		  e->var=mkorfindvarf(eventvars[i]);
		  e->val=eventvals[i];
		}
	}
}

/* thing that builds events - used in other objects */
void add_event(unsigned long t,char *var,float f)
{
	extern int showevents;
	
	strcpy(eventvars[0],var);
	eventvals[0]=f;
	eventct=1;
	do_event(t,0);eventct=0;
}

void do_ramp_event(char *varname,float startval,unsigned long starttime,
				   float endval,unsigned long endtime)
{
  NODE *n;
  struct event *e;
  n=makeevent(starttime);
  e=emalloc(sizeof(struct event));
  e->next=(struct event *)(n->data);
  (struct event *)(n->data)=e;
  e->type=E_RAMP;
  e->var=mkorfindvarf(varname);
  e->val=startval;
  e->finalval=endval;
  e->endtime=endtime;
}

void *emalloc(unsigned long siz)
{
  static unsigned alloced=0L;
  static FILE *a=NULL;
  char *p;

  if(DTEST(DEBUG_MEM))
	{
	  if(!a)a=fopen("mem.log","w");
	  fprintf(a,"attempting to emalloc %ld - %ld so far\n",siz,alloced);
	}


  if(!(p=malloc(siz)))
	yyraise("memory allocation error for %ld bytes",siz);
  alloced+=siz;

  return(p);
}

void yyerror(char *s)
{
  void show_object(struct object *);
  if(lines>=0)
	printf("fatal error: %s at line %d\n",s,lines);
  else if(self)
	{
	  printf("fatal run-time error \"%s\" in object:\n ",s,self);show_object(self);
	}
  else
	printf("fatal error: %s\n",s);

  exit(10);
}

void yyraise(char *a,...)
{
	va_list argptr;
	char buf[80];

	va_start(argptr,a);
	vsprintf(buf,a,argptr);
	yyerror(buf);
}

void warn(char *a,...)
{
	va_list argptr;
	char buf[80];

	va_start(argptr,a);
	vsprintf(buf,a,argptr);
	printf("warning: %s at line %d\n",buf,lines);
}


/* get sample number from time */
unsigned long get_time_real(long mins,long secs,long hunds)
{
	unsigned long t;

	/* work out number of hundredth */

	t=hunds+secs*100+mins*6000;
	
	/* work out the sample number */

	t *= (samprate/100);
	return(t);
}

/* get time from number of seconds */
unsigned long get_time_secs(float secs)
{
	secs *= samprate;
	return((unsigned long)secs);
}

/* get sample number from bar/beat/64th */
unsigned long get_time_bar(long bar,long beat,long frac)
{
	float sampsperfrac;
	float fracs;
	float bard,beatd,fracd,sr,tmpo;
	unsigned long r;

	bard=(float)bar;
	beatd=(float)beat;
	fracd=(float)frac;
	tmpo=(float)tempo;
	sr=(float)samprate;
	
	/* how many samples in a frac */

	sampsperfrac=(sr*60.0)/tmpo;
	sampsperfrac /= 64.0;
	
	/* how many fracs are we talking about */

	fracs=fracd+(beatd+bard*(float)timesignum)*(float)64.0;

	r=(unsigned long)(fracs*sampsperfrac);

	return r;
}

NODE *insertafter(LIST *l,NODE *node,char *key,void *data)
{
	NODE *new=emalloc(sizeof(NODE));

	if(!node)
	{
		new->next=NULL;
	}
	else
	{
		new->next=node->next;
		node->next=new;
	}

	if(strlen(key)>KEYSIZE)yyraise("key %s too long",key);

	if(node==l->tail) l->tail=new;
	strcpy(new->key,key);
	new->data=data;
	return(new);
}

void clear_list(LIST *l)
{
  NODE *p,*q;

  for(p=l->head;p;p=q)
	{
	  q=p->next;
	  free(p);
	}
  l->head=l->tail=NULL;
}

void removefromlist(LIST *l,NODE *p)
{
  NODE *q;

  if(p==l->head)
	{
	  /* this is the head of the list */
	  if(p==l->tail)
		{
		  /* it's also the end of the list - empty the list */
		  free(p);
		  l->head=l->tail=NULL;
		}
	  else
		{
		  /* it's the head of a list with other nodes */
		  l->head=p->next;
		  free(p);
		}
	}
  else
	{
	  for(q=l->head;q;q=q->next)if(p==q->next)break;
	  if(q)
		{
		  q->next=p->next;
		  if(p==l->tail)l->tail=q;
		  free(p);
		}
	} 
}


void insertatend(LIST *l,char *key,void *data)
{
	insertafter(l,l->tail,key,data);
	if(!(l->head))l->head=l->tail;
}

void insertatbegin(LIST *l,char *key,void *data)
{
	NODE *new=insertafter(l,NULL,key,data);
	if(!(new->next=l->head))l->tail=new;
	l->head=new;
}
		
NODE *find(LIST *l,char *key)
{
	register NODE *p;
	for(p=l->head;p;p=p->next)
		if(!strcmp(p->key,key))return(p);
	return(NULL);
}

/* find or make a variable */
NODE *mkorfindvar(char *key)
{
	NODE *v;
	if(!(v=find(&vars,key)))
	{
		insertatend(&vars,key,NULL);
		v=vars.tail;
		vars.tail->f=0.0;
		snark("  variable %s created.\n",key);
	}
	return(v);
}
float *mkorfindvarf(char *key)
{
  NODE *v;
  v=mkorfindvar(key);
  return &(v->f);
}

/* find the event immediately before or at a given time. Can we
optimise? */
NODE *findeventbefore(unsigned long t)
{
	NODE *p,*q=NULL;
	if(!events.head)
	{
		snark("event list empty\n");
		return(NULL);
	}

	/* quick optimise - check the tail of the list */

	if(events.tail->t <= t)
	{
		return(events.tail);
	}

	/* oh well, do it the hard way. */

	for(p=events.head;p;p=p->next)
	{
		if(p->t > t) /* ok, got it */
		{
			return(q);
		}
		q=p;
	}
	return(events.tail);
}

/* these create parameter nodes. These are
   linked to each other with join(), and are not
   associated with a LIST, like normal nodes */
static NODE *mkparamnode(void)
{
  NODE *new=emalloc(sizeof(NODE));
  struct paramstruct *param=emalloc(sizeof(struct paramstruct));
  new->next=NULL;
  new->data=(void *)param;
  return new;
}

NODE *do_varparam(char *var)
{
  NODE *new;
  struct paramstruct *p;
  new=mkparamnode();
  p=(struct paramstruct *)(new->data);

  p->isvar=1;
  strcpy(p->d.var,var);
  return new;
}
NODE *do_realparam(float r)
{
  NODE *new;
  struct paramstruct *p;
  new=mkparamnode();
  p=(struct paramstruct *)(new->data);

  p->isvar=0;
  p->d.real=r;
  return new;
}

NODE *do_objparam(char *name,char *wav,NODE *params)
{
  char tmp[1][KEYSIZE];

  /* make a temporary variable name */

  sprintf(tmp[0],"_Go%lx",tmpid++);
  
  /* and construct an object to output into that variable */
  
  do_object(name,wav,1,params,&(tmp[0]));
  clearparamlist(params);

  /* return a variable param for the temporary variable */

  return do_varparam(tmp[0]);
}

NODE *do_insobjparam(char *name,char *wav,NODE *params)
{
  char tmp[1][KEYSIZE];

  /* make a temporary instance variable name */

  sprintf(tmp[0],"$_Gi%lx",tmpid++);

  /* and construct an instance object to output into that variable */
  
  do_insobject(name,wav,1,params,&(tmp[0]));
/*  clearparamlist(params); dargle */

  /* return a variable param for the temporary variable */

  return do_varparam(tmp[0]);
}

NODE *do_instparam(char *id,int voices,char *instr,NODE *params)
{
  char tmp[1][KEYSIZE];

  /* make a temporary variable name */

  sprintf(tmp[0],"_Gp%lx",tmpid++);
  
  /* and construct an instance to output into that variable */
  
  do_instance(id,voices,instr,params,1,&(tmp[0]));
  clearparamlist(params);

  /* return a variable param for the temporary variable */

  return do_varparam(tmp[0]);
}


void do_wave(char *str)
{
	float *d;
	
	if(!find(&waves,str))
	  {

		d=emalloc(sizeof(float)*wavct);
		memcpy(d,wavearr,sizeof(float)*wavct);

		insertatend(&waves,str,d);
		waves.tail->t=wavct;
	  }
}

void do_insobject(char *str,char *wav,int no,
				  NODE *params,char o[][KEYSIZE])
{
	int i,j,paramct;
	struct insobject *iobj;

	paramct=paramcount(params);

	for(i=0;i<NUMOBJS;i++)
		if(!strcmp(objtab[i].name,str))break;
	if(i==NUMOBJS)yyraise("unknown object type : %s",str);

	if(!wav)
	  {
		if(objtab[i].flags & OF_HASWAVE)
		  yyraise("object %s requires a wave table",str);
		if(objtab[i].flags & OF_HASSAMPL)
		  yyraise("object %s requires a sample",str);
	  }

	if(!(objtab[i].flags & OF_VARARGS))
	  {
		if(paramct<objtab[i].numins)
		  yyraise("not enough inputs for %s. %d required",str,
				  objtab[i].numins);
		if(paramct>objtab[i].numins)
		  warn("too many inputs for %s. %d required",str,
			   objtab[i].numins);
	  }

	if(no<objtab[i].numouts)
	  warn("not enough outputs for %s. %d required",str,
		   objtab[i].numins);

	iobj=emalloc(sizeof(struct insobject));
	iobj->outputct=no;
	iobj->params=params;

	for(i=0;i<no;i++)strcpy(iobj->outputs[i],o[i]);
	*(iobj->wave)='\0';
	if(wav)strcpy(iobj->wave,wav);

	insertatend(&insobjects,str,iobj);
}

void do_instrument(char *str,float duradd)
{
	extern float get_duration(float);
	
	struct instrument *inst;

	snark("declaring instrument %s\n",str);
	if(find(&instruments,str))
		yyraise("instrument already defined - %s",str);

	inst=emalloc(sizeof(struct instrument));
	inst->objects=insobjects.head;
	inst->duradd=get_duration(duradd);

	insertatend(&instruments,str,inst);
}

char *convertname(char *inst,int voice,char *s)
{ 
	static char buf[80];

	if(!strcmp("out",s))
	{
		sprintf(buf,"@%s.%d",inst,voice);
		return(buf);
	}
	
	switch(*s)
	{
		case '$':	/* this is a local parameter name */
			sprintf(buf,"_%s.%d.%s",inst,voice,s+1);break;
		case '%':	/* this is a local variable name */
			sprintf(buf,"__%s.%d.%s",inst,voice,s+1);break;
		default:strcpy(buf,s);
	}
	return(buf);
}

/* use the get an instance parameter by number in
   build_params */
static NODE *parambynumber(NODE *params,int i)
{
  while(params && i--)
	params=params->next;

  return params;
}

static NODE *copyparam(NODE *param)
{
  NODE *new;
  struct paramstruct *p,*q;

  new=mkparamnode();
  p=(struct paramstruct *)(new->data);
  q=(struct paramstruct *)(param->data);

  *p=*q;
  return new;
}


/* build the parameters for each instance object.
   is the instance id, voice is the voice number,
   io is the instance object, and params is the head
   of the parameter list of the instance. */
   
static NODE *build_params(char *id,int voice,struct insobject *io,NODE *params)
{
	char *s;
	int pn;
	NODE *p;
	NODE *objp=NULL;
	struct paramstruct *param;

	for(p=io->params;p;p=p->next)
	{
	  param=(struct paramstruct *)(p->data);
	  if(param->isvar)
		{
		  s=param->d.var;
		  snark(" parameter is %s is variable\n",s);

		  /* is this a instance parameter? If so, just copy */
		  if(*s=='$' && isdigit(s[1]))
			{
			  NODE *q;
			  pn=atoi(s+1);	/* get parameter number */
			  snark(" is an instance parameter (%d).\n",pn);

			  if(!(q=parambynumber(params,pn)))
				yyraise("instance parameter missing for instance %s",id);

			  objp=join(objp,copyparam(q));

			}
			else	/* no, just convert the name */
			{
			  objp=join(objp,do_varparam(convertname(id,voice,s)));
			  snark(" is internal variable %s\n",s);
			}
		}
		else	/* just a constant */
		{
		  objp=join(objp,do_realparam(param->d.real));
		  snark(" parameter is %f\n",param->d.real);		  
		}
	}
	return objp;
}

void build_outputs(char *id,int voice,struct insobject *io)
{
	int i;
	char *s;
	int pn;
	
	ioutputct=io->outputct;
	
	for(i=0;i<ioutputct;i++)
	{
		s=io->outputs[i];
		strcpy(ioutputarray[i],convertname(id,voice,s));
	}
}


void do_instance(char *id,int voices,char *instr,NODE *params,int no,
				 char o[][KEYSIZE])
{
	struct instrument *inst;
	struct insobject *io;
	struct instance  *ic;
	
	NODE *n,*objp;
	int i;

	if(find(&instances,id))yyraise("instance %s already exists.",id);
	
	if(!(n=find(&instruments,instr)))
		yyraise("instrument not found - %s",instr);

	inst=n->data;
	snark("attempting to create instance %s of %s/%d\n",id,instr,voices);

	/* allocate the instance structure */
	
	ic=emalloc(sizeof(struct instance));
	ic->voices=voices;
	ic->list=emalloc(sizeof(LIST)*ic->voices);
	ic->duradd=inst->duradd;
	
	for(i=0;i<voices;i++)
	{
		ic->list[i].head=NULL;
		ic->list[i].tail=NULL;
	}
	insertatend(&instances,id,ic);
	
	/* make the objects for each voice */
	for(i=0;i<voices;i++)
	{
		snark("creating voice %d for instrument instance %s\n",i,id);
		for(n=inst->objects;n;n=n->next)
		{
			io=(struct insobject *)n->data;

			/* OK, now decode the names and make the object */

			snark("making object for voice %s, %d outputs\n",
				n->key,io->outputct);
			objp=build_params(id,i,io,params);
			build_outputs(id,i,io);
			do_object(n->key,
				*(io->wave)?io->wave:NULL,ioutputct,objp,ioutputarray);
			clearparamlist(objp);
		}
	}

	/* now wire the voices together */

	objp=NULL;

	for(i=0;i<voices;i++)
	{
	  objp=join(objp,do_varparam(convertname(id,i,"out")));
	}
	do_object("instaddr",NULL,no,objp,o);
	clearparamlist(objp);
}

struct scoreitem *make_item(void)
{
	int i;
	struct scoreitem *n;

	n=emalloc(sizeof(struct scoreitem));
	n->duration=duration;
	n->amplitude=amplitude;
	
	for(i=0;i<npitch;i++)
	{
		strcpy(n->pitches[i],currnotepitches[i]);
	}
	n->npitches=npitch;
	npitch=0;
	return(n);
}

void do_note(char *instance,unsigned long time,char *scalename,
	float amp,unsigned long dur)
{
	struct scale *scale;
	NODE *n;
	int i;
	
	if(!(n=find(&scales,scalename)))yyraise("unknown scale %s",scalename);
	scale= (struct scale *)(n->data);
	
	for(i=0;i<npitch;i++)
		add_alloc_note(time,instance,scale,currnotepitches[i],
		amp,dur,0);
	npitch=0;
}

/* set up a sample rather than a wave */
void do_sample(char *id,char *format,char *filename,float pitch,float samprate,int chans)
{
  struct filedata *hdr;
  if(find(&samples,id))
	return;

  hdr=emalloc(sizeof(struct filedata));

  insertatend(&samples,id,hdr);

  strcpy(hdr->format,format);
  hdr->file=NULL;
  hdr->pitch=pitch;

  strcpy(hdr->filename,filename);
  /* attempt to open the file */

  if(f_openfile(hdr,"rb"))
	yyraise("unable to open file %s",filename);

  /* and revise the offsets and stuff */
  if(samprate!=-1.0)
	hdr->samprate=samprate;

  if(chans!=-1)
	hdr->channels = chans;

  /* and read the header */
  f_readheader(hdr);
}


/* this routine is used up there somewhere to allocate floats */

#define FLOATBLOCKSIZE	64

float *getfloat(float f)
{
  static float *buffer=NULL;
  static short floatindex=0;

  if(floatindex==64)floatindex=0;
  if(!floatindex)
	  buffer=(float *)emalloc(sizeof(float)*FLOATBLOCKSIZE);
  buffer[floatindex]=f;
  return buffer+(floatindex++);
}
  
void show_object(struct object *o)
{
  short i;

  for(i=0;i<o->outsused;i++)
	printf("%s ",o->outvars[i]->key);
  
  printf(" :   %s",objtab[o->type].name);
  
  for(i=0;i<o->realins;i++)
	if(o->invars[i])
	  printf(" %s",o->invars[i]->key);
	else
	  printf(" %f",*(o->inputs[i]));
  printf("\n");
}
