/*  Sapphire version 1 - an acoustic compiler
    Copyright (C) 1995 James C Finnis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char *rcsid="$Id: sapphire.c,v 15.0 1995/11/12 20:56:40 white Exp $";

/*
 * Main for the Sapphire compiler
 *
 */

#include <stdio.h>
#include <string.h>
#include "sapphire.h"

#define SHOWINTERVAL 500

char *verstring="Sapphire compiler version %s (C) J. Finnis 1993-1995\n";
char *version="2.1";

int debug=0;

extern FILE *yyin;

extern unsigned long endsample;
float *a_output0,*a_output1;
extern LIST vars;
LIST filepaths={NULL,NULL}; /* file search path */

NODE *currentevent;
unsigned long nexttime,currentsample;
struct object *self=NULL;
float **selfinpbase=NULL,**selfoutbase=NULL;


/* Default values live down 'ere */

int timesigdenom=4,timesignum=4,samprate=11025,tempo=120;
int channels=1;
int dolist=1;
char *formatname="wav16"; /* default format is 16-bit wavs */

void handle_events(void),handle_objects(void),handle_events_debug(void),
	handle_objects_debug(void),show_scores(void),handle_ramps(void);
void flush_buffer(void),output_values(void),show_objects(void);
struct filedata opfilehdr;

extern float maxo,mino;
extern int cachesize,blocksize,lines,dodurationadd;

void usage(void)
{
  printf("Usage:\n"
		 "-s<integer>      set sampling rate to <integer>/second\n"
		 "-I<directory>    add directory to include path\n"
		 "-c<integer>      set number of channels to <integer> (currently 1 or 2)\n"
		 "-f<name>         set format to <name>\n"
		 "-d<integer>      set debug flags to <integer>\n"
		 "-o<filename>     set output file to <filename>\n"
		 "-D               activate yacc debugging\n"
		 "-l               don't give running commentary\n"
		 "-C<integer>      set number of blocks in sample cache (default %d)\n"
		 "-B<integer>      set size of blocks in cache (default %d)\n"
		 "-a               turn off durationadd processing\n",
		 CACHESIZE,BLOCKSIZE);
  yyraise("bad command line argument");
}

void copyrt(int all)
{
  if(all)printf(
"\n\n    Sapphire version %s - an acoustic compiler\n"
"    Copyright (C) 1995 James C Finnis\n\n"
"    This program is free software; you can redistribute it and/or modify\n"
"    it under the terms of the GNU General Public License as published by\n"
"    the Free Software Foundation; either version 2 of the License, or\n"
"    (at your option) any later version.\n\n"
"    This program is distributed in the hope that it will be useful,\n"
"    but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"    GNU General Public License for more details.\n\n"
"    You should have received a copy of the GNU General Public License\n"
"    along with this program; if not, write to the Free Software\n"
"    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n",version);
else
  printf(
    "Sapphire comes with ABSOLUTELY NO WARRANTY.\n"
    "This is free software, and you are welcome to redistribute it\n"
    "under certain conditions. Use the -Q command line option for details.\n"
);
}

		 
main(int argc,char **argv)
{
  extern float limit;
  extern int yydebug;
  extern int yyparse();
  unsigned long startt,endt;

  register int debobj=0,debev=0;
  int i;
  NODE *p;
  char *fn=NULL,*outputfn=NULL;

  printf(verstring,version);
  copyrt(0);

	for(i=1;i<argc;i++)
	{
		if(*argv[i]=='-')
		{
			switch(argv[i][1])
			{
				case 's':samprate=atoi(argv[i]+2);break;
				case 'a':dodurationadd=0;break;
				case 'I':insertatend(&filepaths,"PATH",strdup(argv[i]+2));break;
				case 'c':channels=atoi(argv[i]+2);break;
				case 'f':formatname=argv[i]+2;break;
				case 'd':debug=atoi(argv[i]+2);break;
				case 'o':outputfn=argv[i]+2;break;
				case 'D':yydebug=1;break;
				case 'l':dolist=0;break;
				  /* options for controlling caching */
				case 'C':cachesize=atoi(argv[i]+2);break;
				case 'B':blocksize=atoi(argv[i]+2);break;
			case 'Q':copyrt(1);exit(0);
				default: usage();
			}
		}
		else if(fn)yyraise("file already specified");
		else fn=argv[i];
	}

	if(!strcmp(formatname,"?"))
	  setformat(formatname);

	if(fn)
	{
		if(!(yyin=fopen(fn,"r")))
		{
			yyraise("unable to open file %s for input",fn);
		}
	}
	else
		printf("No filename! Using stdin\n");


	/* add the default search paths */
	insertatend(&filepaths,"DEFPATH","insts/");
	insertatend(&filepaths,"DEFPATH","samples/");
	insertatend(&filepaths,"DEFPATH","scales/");

	/* initialise the sample cache */
	f_initcache();

	/* create initial events */
	
	add_event(0,"CH0",0);
	add_event(0,"CH1",0);

	/* create standard variables (the event manager must notice
	if these change */

	a_output0=mkorfindvarf("CH0");
	*a_output0=0.01;
	a_output1=mkorfindvarf("CH1");
	*a_output1=0.01;
	
	/* and parse the program, making the objects */

	yyparse();lines=-1;
	if(!endsample)yyraise("no 'end at' command given");
	printf("Parsed OK. Output format is %s, end sample is at %ld\n",formatname,
		   endsample);
	if(fn)fclose(yyin);

	/* generate and output the header */

	opfilehdr.numsamples=endsample;
	opfilehdr.channels=channels;
	opfilehdr.samprate=samprate;
	opfilehdr.file=NULL;

	if(outputfn)
	  {
		strcpy(opfilehdr.filename,outputfn);
		strcpy(opfilehdr.format,formatname);
		if(f_openfile(&opfilehdr,"wb"))yyraise("unable to open output file %s",outputfn);
		f_writeheader(&opfilehdr);
	  }


	if(DTEST(DEBUG_EVENTS))
	{
		struct event *l;
		for(p=events.head;p;p=p->next)
		{
			printf("event at %ld:\n",p->t);
			for(l=(struct event *)(p->data);l;l=l->next)
			{
				printf("   %lx=%f\n",l->var,l->val);
			}
		}
		for(p=vars.head;p;p=p->next)
		{
			printf("variable %s is at address %lx\n",p->key,&(p->f));
		}
	}
  
  if(DTEST(DEBUG_SCORES))show_scores();
  for(i=0,p=objects.head;p;p=p->next)i++;
  printf("Number of objects : %d\n",i);
  if(DTEST(DEBUG_SHOWOBJS))show_objects();

	/* Now the main loop! (gosh) */

	currentevent=events.head;
	nexttime=currentevent->t;

	time(&startt);

  debev = DTEST(DEBUG_EVENTS);
  debobj= DTEST(DEBUG_OBJECTS);

  for(currentsample=0;currentsample<endsample;currentsample++)
	{
	  if(!(currentsample%SHOWINTERVAL) && dolist)
		{
		  float perc;
		  perc = (float)currentsample * 100.0;
		  perc /= (float)endsample;
		  fprintf(stderr,"** %2.2f%% (max out=%4f [*%4f])**\n",perc,maxo,limit);
		}
	  if(debev)handle_events_debug(); else handle_events();
	  handle_ramps();
	  if(debobj)handle_objects_debug(); else handle_objects();
	  output_values();
	}

	f_end(&opfilehdr);
	f_closefile(&opfilehdr);

	time(&endt);

	printf("max : %f, min : %f\n",maxo,mino);
	{
	  float timepersample=(float)(endt-startt);
	  timepersample /= endsample;
	  printf("Time taken : %ld seconds for %ld samples = %f sec/sample\n",
		   endt-startt,endsample,timepersample);
	}
}


void handle_events(void)
{
  struct event *p;
  if(currentsample==nexttime)
	{
	  for(p=(struct event *)(currentevent->data);p;p=p->next)
		{
		  *(p->var)=p->val;
		  if(p->type==E_RAMP)
			{
			  struct ramp *r;
			  r=(struct ramp *)emalloc(sizeof(struct ramp));
			  r->var=p->var;
			  r->finalvalue=p->finalval;
			  r->samplesleft=p->endtime-currentsample;
			  r->add = (p->finalval - p->val)/(float)(r->samplesleft);
			  insertatend(&ramps,"RAMP",r);
			}
		}
	  currentevent=currentevent->next;
	  if(currentevent)nexttime=currentevent->t;
	}
}

void handle_events_debug(void)
{
	struct event *p;
	if(currentsample==nexttime)
	{
		printf("event found at time %ld\n",currentsample);
		for(p=(struct event *)(currentevent->data);p;p=p->next)
		{
			*(p->var)=p->val;
			if(p->type==E_RAMP)
			  {
				struct ramp *r;
				printf("event is ramp\n");
				r=(struct ramp *)emalloc(sizeof(struct ramp));
				r->var=p->var;
				r->finalvalue=p->finalval;
				r->samplesleft=p->endtime-currentsample;
				r->add = (p->finalval - p->val)/(float)(r->samplesleft);
				insertatend(&ramps,"RAMP",r);
			  }
			printf("   event successful.\n");
		}
		currentevent=currentevent->next;
		if(currentevent)nexttime=currentevent->t;
	}
}

void handle_objects(void)
{
	NODE *n;
	int i;
	
	for(n=objects.head;n;n=n->next)
	{
		self=(struct object *)(n->data);
		selfinpbase=&(self->inputs[0]);
		selfoutbase=&(self->outputs[0]);

		(*(self->func))();

		if(self->outsused > self->numouts)
		{
			for(i=self->numouts;i<self->outsused;i++)
				*(self->outputs[i])=*(self->lastval);
		}
	}
	self=NULL;
}

void handle_objects_debug(void)
{
	NODE *n;
	int i;
	
	for(n=objects.head;n;n=n->next)
	{
		printf("processing object %s\n",n->key);
		self=(struct object *)(n->data);
		selfinpbase=&(self->inputs[0]);
		selfoutbase=&(self->outputs[0]);
		printf("   calling function...\n");
		(*(self->func))();
		printf("   function called OK\n");

		if(self->outsused > self->numouts)
		{
			printf("    there are extra outputs.\n");


			for(i=self->numouts;i<self->outsused;i++)
				*(self->outputs[i])=*(self->lastval);
			printf("    extra outputs dealt with.\n");
		}
		printf("object completed OK\n");fflush(stdout);
	}
}

void handle_ramps(void)
{
  NODE *n,*m;
  struct ramp *r;
  for(n=ramps.head;n;n=m)
	{
	  r=(struct ramp *)(n->data);
	  m=n->next;

	  if(r->samplesleft--)
		  *(r->var) += r->add;
	  else
		{
		  *(r->var) = r->finalvalue;
		  removefromlist(&ramps,n);
		}
	}
}

void show_score(struct score *scr)
{
	LIST s;
	NODE *p;
	int i;
	struct scoreitem *sc;

	s=scr->list;
	for(p=s.head;p;p=p->next)
	{
		sc=(struct scoreitem *)(p->data);
		for(i=0;i<sc->npitches;i++)printf("%s=%f, ",sc->pitches[i],
			get_pitch(scr->scale,sc->pitches[i],0));
		printf("dur=%f amp=%f\n",sc->duration,sc->amplitude);
	}
}

void show_scores(void)
{
	NODE *p;

	printf("Score section----------------------------------------------\n");
	for(p=scores.head;p;p=p->next)
	{
		printf("--------SCORE %s---------\n",p->key);
		show_score((struct score *)p->data);
	}
	printf("-------------------------\n");
}

/* open a file, looking for it in a list of paths if we are to read it,
   but in the cwd if we are to write it */

FILE *pathfopen(char *name,char *mode)
{
  FILE *a;
  NODE *p;
  char buf[256];

  /* if writing, just call fopen */
 
  if(strchr(mode,'w'))
	return fopen(name,mode);


  /* otherwise, look in the current directory and then each
	 directory in the path */

  if(a=fopen(name,mode))return a;

  for(p=filepaths.head;p;p=p->next)
	{
	  char *pth = (char *)(p->data);
	  strcpy(buf,pth);
	  strcat(buf,name);
	  if(a=fopen(buf,mode))return a;
	}
  return NULL;
}

/* show all objects to stdout */
void show_objects(void)
{
  int i;
  NODE *n;
  struct object *o;
  extern void show_object(struct object *);

  for(n=objects.head;n;n=n->next)
	{
	  o=(struct object *)(n->data);
	  show_object(o);
	}
}  

int yywrap()
{
	return 1;
}
