/*  Sapphire version 1 - an acoustic compiler
    Copyright (C) 1995 James C Finnis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char *rcsid="$Id: wav.c,v 15.0 1995/11/12 20:56:40 white Exp $";

/*
   Functions for handling RIFF WAVE files (.wav)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sapphire.h"

extern int debug;
#define snark if(DTEST(DEBUG_SAMPLES))printf

#ifndef UNIX
#define MKID(a,b,c,d) ( ((unsigned long)(d)<<24L) + ((unsigned long)(c)<<16L)\
 + ((unsigned long)(b)<<8L) +(unsigned long)(a))
#else
#define MKID(a,b,c,d) ( ((unsigned long)(a)<<24L) + ((unsigned long)(b)<<16L)\
 + ((unsigned long)(c)<<8L) +(unsigned long)(d))
#endif

typedef short WORD;
typedef unsigned long DWORD;

struct wavformat
{
  WORD wFormatTag;
  WORD nChannels;
  DWORD nSamplesPerSec;
  DWORD nAvgBytesPerSec;
  WORD nBlockAlign;
  unsigned short nBitsPerSample;
};



static void readckhdr(FILE *a,unsigned long *ckID,unsigned long *datalen)
{
  fread(ckID,sizeof(unsigned long),1,a);
  fread(datalen,sizeof(unsigned long),1,a);
}

static void writeckhdr(FILE *a,unsigned long ckID,unsigned long datalen)
{
  fwrite(&ckID,sizeof(unsigned long),1,a);
  fwrite(&datalen,sizeof(unsigned long),1,a);
}

void wavreadheader(struct filedata *hdr)
{
  FILE *a;
  unsigned long id,siz,oldpos;
  struct wavformat formatdata;

  snark("WAV : reading header for %s\n",hdr->filename);

  if(!(a=hdr->file))yyraise("WAV file is not open!! Arghh!");
  oldpos=ftell(a);

  /* start the RIFF chunk */
  readckhdr(a,&id,&siz);
  if(id!=MKID('R','I','F','F'))
	yyraise("%s is not a RIFF/WAV file - header is %lx",hdr->filename,id);

  /* get the type */
  fread(&id,sizeof(unsigned long),1,a);
  if(id!=MKID('W','A','V','E'))
	yyraise("%s is not a WAV file - subchunk header is %lx",hdr->filename,id);

  /* read the format data */
  readckhdr(a,&id,&siz);
  if(id!=MKID('f','m','t',' '))
	yyraise("%s has no format chunk!",hdr->filename);
  fread(&formatdata,sizeof(struct wavformat),1,a);

printf("reading WAV file %s : category %d, chans %d, rate %ld, abps %ld,\n ba %d, bps %d\n",
 hdr->filename,
 formatdata.wFormatTag,formatdata.nChannels,formatdata.nSamplesPerSec,
 formatdata.nAvgBytesPerSec,formatdata.nBlockAlign,formatdata.nBitsPerSample);

  /* read the data header */
  readckhdr(a,&id,&siz);
  if(id!=MKID('d','a','t','a'))
	yyraise("%s does not have a data chunk - subchunk header is %lx",hdr->filename,id);

  hdr->dataoffset=ftell(a); /* we return the offset of the data */
  hdr->bitspersample=formatdata.nBitsPerSample;
  hdr->numsamples=siz/(formatdata.nBitsPerSample / 8);
  hdr->channels=formatdata.nChannels;
  hdr->samprate=formatdata.nSamplesPerSec;
printf("number of samples : %d\n",hdr->numsamples);
  fseek(a,oldpos,SEEK_SET);
}

static void wavwheader(struct filedata *hdr)
{
  FILE *a;
  unsigned long numbytes,oldpos,id;
  struct wavformat formatdata;

  snark("WAV : writing wav header for %s\n",hdr->filename);

  if(!(a=hdr->file))
	yyraise("unable to write WAV header - file not open! Aarggh!");

  /* assemble the format data */
  formatdata.wFormatTag=1;
  formatdata.nChannels=hdr->channels;
  formatdata.nSamplesPerSec=hdr->samprate;
  formatdata.nBitsPerSample=hdr->bitspersample;
  formatdata.nAvgBytesPerSec=((long)(hdr->channels) *
  	(long)(hdr->samprate) * (long)(hdr->bitspersample))/8L;
  formatdata.nBlockAlign=(hdr->channels * hdr->bitspersample)/8;

  numbytes=(long)(hdr->numsamples)*(long)(hdr->channels)*
	(long)(hdr->bitspersample)/8L;

printf("output WAV file %s : category %d, chans %d, rate %ld,\n abps %ld, ba %d, bps %d\n",
 hdr->filename,
 formatdata.wFormatTag,formatdata.nChannels,formatdata.nSamplesPerSec,
 formatdata.nAvgBytesPerSec,formatdata.nBlockAlign,formatdata.nBitsPerSample);

  /* write the RIFF */
  writeckhdr(a,MKID('R','I','F','F'),numbytes+8+sizeof(formatdata));

  /* write the WAVE */
  id=MKID('W','A','V','E');
  fwrite(&id,sizeof(unsigned long),1,a);

  /* write the format data */
  writeckhdr(a,MKID('f','m','t',' '),sizeof(formatdata));
  fwrite(&formatdata,sizeof(struct wavformat),1,a);

  /* write the data header */
  writeckhdr(a,MKID('d','a','t','a'),numbytes);
}

void wavwheader8(struct filedata *hdr)
{
  hdr->bitspersample=8;
  wavwheader(hdr);
}

void wavwheader16(struct filedata *hdr)
{
  hdr->bitspersample=16;
  wavwheader(hdr);
}
void wavwrite(struct filedata *hdr,float *data)
{
  unsigned long i;
  snark("WAV : writing data to %s\n",hdr->filename);

  if(hdr->bitspersample==8)
	{
	  unsigned char c[100];
	  short q;
	  for(i=0;i<hdr->channels;i++)
		{
		  q=(data[i]*127.0);
		  q+=128;
		  c[i]=(unsigned char)q;
		}
	  fwrite(&c[0],sizeof(char),i,hdr->file);
	}
  else if(hdr->bitspersample==16)
	{
	  short s[100];
	  for(i=0;i<hdr->channels;i++)
		  s[i]=(data[i]*32767.0);
	  fwrite(&s[0],sizeof(short),i,hdr->file);
	}
}

void wavend(struct filedata *hdr)
{
  ;
}

#define RBUFSIZ 1024

void wavread(struct filedata *hdr,unsigned long numsamps,unsigned long offset,
					int channel,float *buffer)
{
  unsigned long i,datastart,f,bct;
  short s;
  unsigned char c;
  char buf[RBUFSIZ];

  int chans,sampsize;

  snark("WAV : reading %ld samples of %d chans from %s to offset %ld\n",
		numsamps,hdr->channels,hdr->filename,offset);

  sampsize=hdr->bitspersample/8;

  chans=hdr->channels;
  f=offset*chans;
  datastart=hdr->dataoffset+sampsize*f;

  /* seek to the start of the first sample we want */
  /* and read some bytes */

  fseek(hdr->file,datastart,SEEK_SET);
  fread(&buf[0],RBUFSIZ,1,hdr->file);

  for(bct=i=0;i<numsamps;i++,bct++)
	{
	  /* are these bytes in our buffer? */

	  if(sampsize*bct*chans+channel >= RBUFSIZ)
		{
		  /* nope, get another buffer */
		  bct=0;
		  fseek(hdr->file,datastart+sampsize*i*chans+channel,SEEK_SET);
		  fread(&buf[0],RBUFSIZ,1,hdr->file);
		}

	  if(sampsize==1)
		{
		  fread(&c,sampsize,1,hdr->file);
		  c=buf[chans*bct+channel];
		  buffer[i]=((float)c-128.0)/128.0;
		}
	  else
		{
		  unsigned long idx;
		  idx=sizeof(short)*(chans*bct+channel);
		  s=buf[idx]+256*(short)(buf[idx+1]);
		  buffer[i]=((float)s)/32767.0;
		}
	}
}


