2 /* nodes.c -- How to get an Info file and node. */  L /* This file is part of GNU Info, a program for reading online documentation    stored in Info format.   4    Copyright (C) 1993 Free Software Foundation, Inc.  G    This program is free software; you can redistribute it and/or modify G    it under the terms of the GNU General Public License as published by F    the Free Software Foundation; either version 2, or (at your option)    any later version.   B    This program is distributed in the hope that it will be useful,A    but WITHOUT ANY WARRANTY; without even the implied warranty of @    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the/    GNU General Public License for more details.   D    You should have received a copy of the GNU General Public License>    along with this program; if not, write to the Free Software<    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  -    Written by Brian Fox (bfox@ai.mit.edu). */    #include <stdio.h> #include <ctype.h> #include <sys/types.h> #include <sys/file.h>  #include <sys/errno.h> #include <sys/stat.h>  #include "nodes.h" #include "search.h"  #include "filesys.h" #include "info-utils.h"    #if !defined (O_RDONLY)  #if defined (HAVE_SYS_FCNTL_H) #include <sys/fcntl.h> #else /* !HAVE_SYS_FCNTL_H */  #include <fcntl.h> #endif /* !HAVE_SYS_FCNTL_H */ #endif /* !O_RDONLY */   #if !defined (errno) extern int errno;  #endif /* !errno */   F /* **************************************************************** */ /*								    */. /*		     Functions Static to this File		    */ /*								    */F /* **************************************************************** */  7 static void forget_info_file (), remember_info_file (); 7 static void free_file_buffer_tags (), free_info_tag (); B static void get_nodes_of_tags_table (), get_nodes_of_info_file ();/ static void get_tags_of_indirect_tags_table (); 0 static void info_reload_file_buffer_contents ();! static char *adjust_nodestart (); ( static FILE_BUFFER *make_file_buffer ();L static FILE_BUFFER *info_load_file_internal (), *info_find_file_internal ();. static NODE *info_node_of_file_buffer_tags ();   static long get_node_length ();   K /* Magic number that RMS used to decide how much a tags table pointer could J    be off by.  I feel that it should be much smaller, like on the order of    4. */ #define DEFAULT_INFO_FUDGE 1000   D /* Passed to *_internal functions.  INFO_GET_TAGS says to do what isD    neccessary to fill in the nodes or tags arrays in FILE_BUFFER. */ #define INFO_NO_TAGS  0  #define INFO_GET_TAGS 1   F /* **************************************************************** */ /*								    */ /*			 Global Variables			    */  /*								    */F /* **************************************************************** */  G /* When non-zero, this is a string describing the recent file error. */ , char *info_recent_file_error = (char *)NULL;  ' /* The list of already loaded nodes. */ 7 FILE_BUFFER **info_loaded_files = (FILE_BUFFER **)NULL;   > /* The number of slots currently allocated to LOADED_FILES. */  int info_loaded_files_slots = 0;  F /* **************************************************************** */ /*								    */3 /*		 Public Functions for Node Manipulation		    */  /*								    */F /* **************************************************************** */  G /* Used to build "dir" menu from "localdir" files found in INFOPATH. */ $ extern void maybe_build_dir_node ();  M /* Return a pointer to a NODE structure for the Info node (FILENAME)NODENAME. O    FILENAME can be passed as NULL, in which case the filename of "dir" is used. O    NODENAME can be passed as NULL, in which case the nodename of "Top" is used. 9    If the node cannot be found, return a NULL pointer. */  NODE *" info_get_node (filename, nodename)      char *filename, *nodename;  {    FILE_BUFFER *file_buffer; 
   NODE *node;   $   file_buffer = (FILE_BUFFER *)NULL;(   info_recent_file_error = (char *)NULL;  1   info_parse_node (nodename, DONT_SKIP_NEWLINES);    nodename = (char *)NULL;     if (info_parsed_filename) $     filename = info_parsed_filename;     if (info_parsed_nodename) $     nodename = info_parsed_nodename;  ;   /* If FILENAME is not specified, it defaults to "dir". */    if (!filename)     filename = "dir";   I   /* If the file to be looked up is "dir", build the contents from all of +      the "localdir"'s found in INFOPATH. */ %   if (stricmp (filename, "dir") == 0) 0     maybe_build_dir_node (filename, "localdir");  #   /* Find the correct info file. */ *   file_buffer = info_find_file (filename);     if (!file_buffer)      {        if (filesys_error_number)  	info_recent_file_error = 9 	  filesys_error_string (filename, filesys_error_number);        return ((NODE *)NULL);     }   >   node = info_get_node_of_file_buffer (nodename, file_buffer);K   /* If the node looked for was "Top", try again looking for the node under "      a slightly different name. */D   if (!node && (nodename == NULL || stricmp (nodename, "Top") == 0))     { ?       node = info_get_node_of_file_buffer ("Top", file_buffer);        if (!node): 	node = info_get_node_of_file_buffer ("top", file_buffer);       if (!node): 	node = info_get_node_of_file_buffer ("TOP", file_buffer);     }    return (node); }   E /* Return a pointer to a NODE structure for the Info node NODENAME in B    FILE_BUFFER.  NODENAME can be passed as NULL, in which case theD    nodename of "Top" is used.  If the node cannot be found, return a    NULL pointer. */  NODE *4 info_get_node_of_file_buffer (nodename, file_buffer)      char *nodename;      FILE_BUFFER *file_buffer; {    NODE *node = (NODE *)NULL;  H   /* If we are unable to find the file, we have to give up.  There isn't       anything else we can do. */   if (!file_buffer)      return ((NODE *)NULL);  >   /* If the file buffer was gc'ed, reload the contents now. */   if (!file_buffer->contents) 3     info_reload_file_buffer_contents (file_buffer);   ;   /* If NODENAME is not specified, it defaults to "Top". */    if (!nodename)     nodename = "Top";   J   /* If the name of the node that we wish to find is exactly "*", then theJ      node body is the contents of the entire file.  Create and return such      a node. */ "   if (strcmp (nodename, "*") == 0)     { -       node = (NODE *)xmalloc (sizeof (NODE)); -       node->filename = file_buffer->fullpath; $       node->parent   = (char *)NULL;(       node->nodename = savestring ("*");-       node->contents = file_buffer->contents; ,       node->nodelen = file_buffer->filesize;       node->flags = 0;     }   L   /* If this is the "main" info file, it might contain a tags table.  SearchI      the tags table for an entry which matches the node that we want.  If L      there is a tags table, get the file which contains this node, but don't+      bother building a node list for it. */    else if (file_buffer->tags) A     node = info_node_of_file_buffer_tags (file_buffer, nodename);   .   /* Return the results of our node search. */   return (node); }   J /* Locate the file named by FILENAME, and return the information structureI    describing this file.  The file may appear in our list of loaded files I    already, or it may not.  If it does not already appear, find the file, H    and add it to the list of loaded files.  If the file cannot be found,"    return a NULL FILE_BUFFER *. */
 FILE_BUFFER *  info_find_file (filename)       char *filename; { =   return (info_find_file_internal (filename, INFO_GET_TAGS));  }   E /* Load the info file FILENAME, remembering information about it in a     file buffer. */
 FILE_BUFFER *  info_load_file (filename)       char *filename; { =   return (info_load_file_internal (filename, INFO_GET_TAGS));  }     F /* **************************************************************** */ /*								    */0 /*		    Private Functions Implementation		    */ /*								    */F /* **************************************************************** */  F /* The workhorse for info_find_file ().  Non-zero 2nd argument says toD    try to build a tags table (or otherwise glean the nodes) for thisG    file once found.  By default, we build the tags table, but when thisNF    function is called by info_get_node () when we already have a valid9    tags table describing the nodes, it is unnecessary. */n static FILE_BUFFER *, info_find_file_internal (filename, get_tags)      char *filename;      int get_tags; {G   register int i;s$   register FILE_BUFFER *file_buffer;  G   /* First try to find the file in our list of already loaded files. */    if (info_loaded_files)     {d:       for (i = 0; file_buffer = info_loaded_files[i]; i++)7 	if ((strcmp (filename, file_buffer->filename) == 0) ||A7 	    (strcmp (filename, file_buffer->fullpath) == 0) ||G 	    ((*filename != '/') &&  	     strcmp (filename,s= 		     filename_non_directory (file_buffer->fullpath)) == 0))  	  {% 	    struct stat new_info, *old_info;e  = 	    /* This file is loaded.  If the filename that we want is0B 	       specifically "dir", then simply return the file buffer. */A 	    if (stricmp (filename_non_directory (filename), "dir") == 0)e 	      return (file_buffer);  C 	    /* The file appears to be already loaded, and it is not "dir".hA 	       Check to see if it has changed since the last time it was( 	       loaded. */7 	    if (stat (file_buffer->fullpath, &new_info) == -1)! 	      { 		filesys_error_number = errno;e 		return ((FILE_BUFFER *)NULL);d 	      }  $ 	    old_info = &file_buffer->finfo;  3 	    if ((new_info.st_size != old_info->st_size) ||*, 		(new_info.st_mtime != old_info->st_mtime)) 	      {= 		/* The file has changed.  Forget that we ever had loaded it  		   in the first place. */* 		forget_info_file (filename); 		break; 	      }	 	    elset 	      {= 		/* The info file exists, and has not changed since the lastf> 		   time it was loaded.  If the caller requested a nodes list= 		   for this file, and there isn't one here, build the nodess? 		   for this file_buffer.  In any case, return the file_buffer_ 		   object. */t% 		if (get_tags && !file_buffer->tags)F' 		  build_tags_and_nodes (file_buffer);F   		return (file_buffer);i 	      } 	  }     }f  4   /* The file wasn't loaded.  Try to load it now. */=   file_buffer = info_load_file_internal (filename, get_tags);   K   /* If the file was loaded, remember the name under which it was found. */f   if (file_buffer)%     remember_info_file (file_buffer);      return (file_buffer);O }D  J /* The workhorse function for info_load_file ().  Non-zero second argumentF    says to build a list of tags (or nodes) for this file.  This is theD    default behaviour when info_load_file () is called, but it is notF    necessary when loading a subfile for which we already have tags. */ static FILE_BUFFER *, info_load_file_internal (filename, get_tags)      char *filename;      int get_tags; {*   char *fullpath, *contents;   long filesize;   struct stat finfo;   int retcode;1   FILE_BUFFER *file_buffer = (FILE_BUFFER *)NULL;   E   /* Get the full pathname of this file, as known by the info system.eE      That is to say, search along INFOPATH and expand tildes, etc. */d+   fullpath = info_find_fullpath (filename);l  &   /* Did we actually find the file? */$   retcode = stat (fullpath, &finfo);  K   /* If the file referenced by the name returned from info_find_fullpath ()*E      doesn't exist, then try again with the last part of the filename*      appearing in lowercase. */d   if (retcode < 0)     {        char *lowered_name;        char *basename;l  +       lowered_name = savestring (filename); 4       basename = (char *)rindex (lowered_name, '/');         if (basename)s 	basename++;
       else 	basename = lowered_name;          while (*basename)s 	{ 	  if (isupper (*basename)) % 	    *basename = tolower (*basename);o   	  basename++; 	}  3       fullpath = info_find_fullpath (lowered_name);n       free (lowered_name);  (       retcode = stat (fullpath, &finfo);     }E  D   /* If the file wasn't found, give up, returning a NULL pointer. */   if (retcode < 0)     {o#       filesys_error_number = errno;I#       return ((FILE_BUFFER *)NULL);      }n  (   /* Otherwise, try to load the file. */B   contents = filesys_read_info_file (fullpath, &filesize, &finfo);     if (!contents)!     return ((FILE_BUFFER *)NULL);i  H   /* The file was found, and can be read.  Allocate FILE_BUFFER and fill      in the various members. */r$   file_buffer = make_file_buffer ();0   file_buffer->filename = savestring (filename);0   file_buffer->fullpath = savestring (fullpath);   file_buffer->finfo = finfo;;#   file_buffer->filesize = filesize;*#   file_buffer->contents = contents;i:   if (file_buffer->filesize != file_buffer->finfo.st_size))     file_buffer->flags |= N_IsCompressed;   D   /* If requested, build the tags and nodes for this file buffer. */   if (get_tags) '     build_tags_and_nodes (file_buffer);b     return (file_buffer);r }   J /* Grovel FILE_BUFFER->contents finding tags and nodes, and filling in theK    various slots.  This can also be used to rebuild a tag or node table. */m void" build_tags_and_nodes (file_buffer)      FILE_BUFFER *file_buffer; {    SEARCH_BINDING binding;    long position;  &   free_file_buffer_tags (file_buffer);(   file_buffer->flags &= ~N_HasTagsTable;  7   /* See if there is a tags table in this info file. */ )   binding.buffer = file_buffer->contents;o(   binding.start = file_buffer->filesize;%   binding.end = binding.start - 1000;e   if (binding.end < 0)     binding.end = 0;   binding.flags = S_FoldCase;e  >   position = search_backward (TAGS_TABLE_END_LABEL, &binding);  F   /* If there is a tag table, find the start of it, and grovel over it#      extracting tag information. */*   if (position != -1) 
     while (1)u       {f' 	long tags_table_begin, tags_table_end;i   	binding.end = position;A 	binding.start = binding.end - 5 - strlen (TAGS_TABLE_END_LABEL);* 	if (binding.start < 0)c 	  binding.start = 0;s  + 	position = find_node_separator (&binding);n  B 	/* For this test, (and all others here) failure indicates a bogus$ 	   tags table.  Grovel the file. */ 	if (position == -1)	 	  break;e  * 	/* Remember the end of the tags table. */ 	binding.start = position;  	tags_table_end = binding.start; 	binding.end = 0;   * 	/* Locate the start of the tags table. */= 	position = search_backward (TAGS_TABLE_BEG_LABEL, &binding);e   	if (position == -1)	 	  break;    	binding.end = position;A 	binding.start = binding.end - 5 - strlen (TAGS_TABLE_BEG_LABEL);o+ 	position = find_node_separator (&binding);>   	if (position == -1)	 	  break;   A 	/* The file contains a valid tags table.  Fill the FILE_BUFFER's  	   tags member. */s& 	file_buffer->flags |= N_HasTagsTable; 	tags_table_begin = position;t  A 	/* If this isn't an indirect tags table, just remember the nodes @ 	   described locally in this tags table.  Note that binding.end5 	   is pointing to just after the beginning label. */  	binding.start = binding.end; % 	binding.end = file_buffer->filesize;a  : 	if (!looking_at (TAGS_TABLE_IS_INDIRECT_LABEL, &binding)) 	  {& 	    binding.start = tags_table_begin;" 	    binding.end = tags_table_end;5 	    get_nodes_of_tags_table (file_buffer, &binding);f 	    return; 	  } 	elsen 	  {> 	    /* This is an indirect tags table.  Build TAGS member. */ 	    SEARCH_BINDING indirect;   ' 	    indirect.start = tags_table_begin;. 	    indirect.end = 0;& 	    indirect.buffer = binding.buffer;! 	    indirect.flags = S_FoldCase;(  G 	    position = search_backward (INDIRECT_TAGS_TABLE_LABEL, &indirect);l   	    if (position == -1) 	      {) 		/* This file is malformed.  Give up. */m	 		return;  	      }   	    indirect.start = position; % 	    indirect.end = tags_table_begin;f& 	    binding.start = tags_table_begin;" 	    binding.end = tags_table_end;H 	    get_tags_of_indirect_tags_table (file_buffer, &indirect, &binding); 	    return; 	  }       }*  B   /* This file doesn't contain any kind of tags table.  Grovel the+      file and build node entries for it. */*'   get_nodes_of_info_file (file_buffer);f }i  C /* Search through FILE_BUFFER->contents building an array of TAG *,sB    one entry per each node present in the file.  Store the tags in:    FILE_BUFFER->tags, and the number of allocated slots in    FILE_BUFFER->tags_slots. */ static voidl$ get_nodes_of_info_file (file_buffer)      FILE_BUFFER *file_buffer; {.   long nodestart;U   int tags_index = 0;i   SEARCH_BINDING binding;)  )   binding.buffer = file_buffer->contents;G   binding.start = 0;&   binding.end = file_buffer->filesize;   binding.flags = S_FoldCase;   <   while ((nodestart = find_node_separator (&binding)) != -1)     {        int start, end;i       char *nodeline;_       TAG *entry;   0       /* Skip past the characters just found. */        binding.start = nodestart;L       binding.start += skip_node_separator (binding.buffer + binding.start);  <       /* Move to the start of the line defining the node. */0       nodeline = binding.buffer + binding.start;         /* Find "Node:" */9       start = string_in_line (INFO_NODE_LABEL, nodeline);"  :       /* If not there, this is not the start of a node. */       if (start == -1)
 	continue;  +       /* Find the start of the nodename. */ 2       start += skip_whitespace (nodeline + start);  )       /* Find the end of the nodename. */h       end = start +l= 	skip_node_characters (nodeline + start, DONT_SKIP_NEWLINES);r  D       /* Okay, we have isolated the node name, and we know where theA 	 node starts.  Remember this information in a NODE structure. */f,       entry = (TAG *)xmalloc (sizeof (TAG));<       entry->nodename = (char *)xmalloc (1 + (end - start));?       strncpy (entry->nodename, nodeline + start, end - start);e*       entry->nodename[end - start] = '\0';#       entry->nodestart = nodestart;;       {; 	SEARCH_BINDING node_body;  3 	node_body.buffer = binding.buffer + binding.start;n 	node_body.start = 0; - 	node_body.end = binding.end - binding.start;  	node_body.flags = S_FoldCase;/ 	entry->nodelen = get_node_length (&node_body);i       }f  .       entry->filename = file_buffer->fullpath;  L       /* Add this tag to the array of tag structures in this FILE_BUFFER. */A       add_pointer_to_array (entry, tags_index, file_buffer->tags, , 			    file_buffer->tags_slots, 100, TAG *);     }  }i  < /* Return the length of the node which starts at BINDING. */ static longe get_node_length (binding)a      SEARCH_BINDING *binding;f {    register int i; 
   char *body;_     /* From the Info-RFC file:A      [A node] ends with either a ^_, a ^L, or the end of file. */eI   for (i = binding->start, body = binding->buffer; i < binding->end; i++)f     { 7       if (body[i] == INFO_FF || body[i] == INFO_COOKIE)i 	break;c     }b%   return ((long) i - binding->start);  }u  L /* Build and save the array of nodes in FILE_BUFFER by searching through theN    contents of BUFFER_BINDING for a tags table, and groveling the contents. */ static void 5 get_nodes_of_tags_table (file_buffer, buffer_binding)n      FILE_BUFFER *file_buffer;$      SEARCH_BINDING *buffer_binding; {t   int offset, tags_index = 0;,   SEARCH_BINDING *search;m   long position;  )   search = copy_binding (buffer_binding);s  )   /* Find the start of the tags table. */f&   position = find_tags_table (search);      /* If none, we're all done. */   if (position == -1)I     return;f  C   /* Move to one character before the start of the actual table. */e   search->start = position; H   search->start += skip_node_separator (search->buffer + search->start);1   search->start += strlen (TAGS_TABLE_BEG_LABEL);h   search->start--;  I   /* The tag table consists of lines containing node names and positions.lH      Do each line until we find one that doesn't contain a node name. */:   while ((position = search_forward ("\n", search)) != -1)     {p       TAG *entry;        char *nodedef;  &       /* Prepare to skip this line. */       search->start = position;        search->start++;  ?       /* Skip past informative "(Indirect)" tags table line. */ K       if (!tags_index && looking_at (TAGS_TABLE_IS_INDIRECT_LABEL, search)) 
 	continue;  3       /* Find the label preceding the node name. */        offset =B 	string_in_line (INFO_NODE_LABEL, search->buffer + search->start);  D       /* If not there, not a defining line, so we must be out of the 	 tags table. */       if (offset == -1)R 	break;i  :       /* Point to the beginning of the node definition. */       search->start += offset;/       nodedef = search->buffer + search->start;r+       nodedef += skip_whitespace (nodedef);i  &       /* Move past the node's name. */       for (offset = 0;: 	   (nodedef[offset]) && (nodedef[offset] != INFO_TAGSEP); 	   offset++);  )       if (nodedef[offset] != INFO_TAGSEP)f
 	continue;  ,       entry = (TAG *)xmalloc (sizeof (TAG));5       entry->nodename = (char *)xmalloc (1 + offset);t1       strncpy (entry->nodename, nodedef, offset);a%       entry->nodename[offset] = '\0';f       offset++;n8       entry->nodestart = (long) atol (nodedef + offset);  6       /* We don't know the length of this node yet. */       entry->nodelen = -1;  H       /* The filename of this node is currently known as the same as the 	 name of this file. */f.       entry->filename = file_buffer->fullpath;  H       /* Add this node structure to the array of node structures in this 	 FILE_BUFFER. */ A       add_pointer_to_array (entry, tags_index, file_buffer->tags,u, 			    file_buffer->tags_slots, 100, TAG *);     }e   free (search); }   K /* A structure used only in get_tags_of_indirect_tags_table () to hold onto     an intermediate value. */ typedef struct {   char *filename;(   long first_byte;
 } SUBFILE;  N /* Remember in FILE_BUFFER the nodenames, subfilenames, and offsets within theM    subfiles of every node which appears in TAGS_BINDING.  The 2nd argument isf4    a binding surrounding the indirect files list. */ static voidoM get_tags_of_indirect_tags_table (file_buffer, indirect_binding, tags_binding)n      FILE_BUFFER *file_buffer;5      SEARCH_BINDING *indirect_binding, *tags_binding;n {    register int i;((   SUBFILE **subfiles = (SUBFILE **)NULL;-   int subfiles_index = 0, subfiles_slots = 0; 
   TAG *entry;)  E   /* First get the list of tags from the tags table.  Then lookup thetI      associated file in the indirect list for each tag, and update it. */o6   get_nodes_of_tags_table (file_buffer, tags_binding);  D   /* We have the list of tags in file_buffer->tags.  Get the list of)      subfiles from the indirect table. */i   {a     char *start, *end, *line;T     SUBFILE *subfile;   ?     start = indirect_binding->buffer + indirect_binding->start;);     end = indirect_binding->buffer + indirect_binding->end;l     line = start;        while (line < end)       {r 	int colon;H  $ 	colon = string_in_line (":", line);   	if (colon == -1)s	 	  break;r  1 	subfile = (SUBFILE *)xmalloc (sizeof (SUBFILE));e- 	subfile->filename = (char *)xmalloc (colon);g. 	strncpy (subfile->filename, line, colon - 1);% 	subfile->filename[colon - 1] = '\0';e2 	subfile->first_byte = (long) atol (line + colon);   	add_pointer_to_arrayIF 	  (subfile, subfiles_index, subfiles, subfiles_slots, 10, SUBFILE *);   	while (*line++ != '\n');a       }    }g  A   /* If we have successfully built the indirect files table, then 0      merge the information in the two tables. */   if (!subfiles)     {/*       free_file_buffer_tags (file_buffer);
       return;a     }e   else     {i       register int tags_index;       long header_length;        SEARCH_BINDING binding;(  J       /* Find the length of the header of the file containing the indirectB 	 tags table.  This header appears at the start of every file.  WeA 	 want the absolute position of each node within each subfile, sonB 	 we subtract the start of the containing subfile from the logicalD 	 position of the node, and then add the length of the header in. */-       binding.buffer = file_buffer->contents;n       binding.start = 0;*       binding.end = file_buffer->filesize;!       binding.flags = S_FoldCase;e  5       header_length = find_node_separator (&binding);e       if (header_length == -1) 	header_length = 0;g  5       /* Build the file buffer's list of subfiles. */o       {p 	char *containing_dir, *temp;e 	int len_containing_dir;  5 	containing_dir = savestring (file_buffer->fullpath); - 	temp = (char *)rindex (containing_dir, '/');_  
 	if (temp) 	  *temp = '\0';  . 	len_containing_dir = strlen (containing_dir);   	for (i = 0; subfiles[i]; i++);E  G 	file_buffer->subfiles = (char **) xmalloc ((1 + i) * sizeof (char *));d   	for (i = 0; subfiles[i]; i++) 	  { 	    char *fullpath;    	    fullpath = (char *) xmallocA 	      (2 + strlen (subfiles[i]->filename) + len_containing_dir);     	    sprintf (fullpath, "%s/%s",. 		     containing_dir, subfiles[i]->filename);  ) 	    file_buffer->subfiles[i] = fullpath;t 	  }) 	file_buffer->subfiles[i] = (char *)NULL;e 	free (containing_dir);r       }g  F       /* For each node in the file's tags table, remember the starting 	 position. */       for (tags_index = 0;* 	   entry = file_buffer->tags[tags_index]; 	   tags_index++)( 	{ 	  for (i = 0;C 	       subfiles[i] && entry->nodestart >= subfiles[i]->first_byte;/
 	       i++);r  < 	  /* If the Info file containing the indirect tags table is! 	     malformed, then give up. */e
 	  if (!i) 	    {= 	      /* The Info file containing the indirect tags table isr 		 malformed.  Give up. */$ 	      for (i = 0; subfiles[i]; i++) 		{ ! 		  free (subfiles[i]->filename);  		  free (subfiles[i]);e$ 		  free (file_buffer->subfiles[i]); 		}m- 	      file_buffer->subfiles = (char **)NULL;T+ 	      free_file_buffer_tags (file_buffer);d 	      return; 	    }  A 	  /* SUBFILES[i] is the index of the first subfile whose logicaltB 	     first byte is greater than the logical offset of this node's> 	     starting position.  This means that the subfile directly; 	     preceding this one is the one containing the node. */   2 	  entry->filename = file_buffer->subfiles[i - 1];2 	  entry->nodestart -= subfiles[i -1]->first_byte;% 	  entry->nodestart += header_length;b 	  entry->nodelen = -1;  	}  E       /* We have successfully built the tags table.  Remember that ity 	 was indirect. */+       file_buffer->flags |= N_TagsIndirect;a     }n  F   /* Free the structures assigned to SUBFILES.  Free the names as well>      as the structures themselves, then finally, the array. */   for (i = 0; subfiles[i]; i++)e     {o#       free (subfiles[i]->filename);D       free (subfiles[i]);t     }i   free (subfiles); }   G /* Return the node from FILE_BUFFER which matches NODENAME by searchinghI    the tags table in FILE_BUFFER.  If the node could not be found, returnb    a NULL pointer. */ 
 static NODE *o5 info_node_of_file_buffer_tags (file_buffer, nodename)       FILE_BUFFER *file_buffer;      char *nodename; {B   register int i;r   TAG *tag;i  .   for (i = 0; tag = file_buffer->tags[i]; i++).     if (strcmp (nodename, tag->nodename) == 0)       {o 	FILE_BUFFER *subfile;  A 	subfile = info_find_file_internal (tag->filename, INFO_NO_TAGS);U   	if (!subfile) 	  return ((NODE *)NULL);e   	if (!subfile->contents). 	  info_reload_file_buffer_contents (subfile);   	if (!subfile->contents) 	  return ((NODE *)NULL);d  > 	/* If we were able to find this file and load it, then return 	   the node within it. */ 	{ 	  NODE *node;  * 	  node = (NODE *)xmalloc (sizeof (NODE));( 	  node->filename = (subfile->fullpath);" 	  node->nodename = tag->nodename;7 	  node->contents = subfile->contents + tag->nodestart;o 	  node->flags	 = 0;  	  node->parent	 = (char *)NULL;  + 	  if (file_buffer->flags & N_HasTagsTable)e 	    {% 	      node->flags |= N_HasTagsTable;f  / 	      if (file_buffer->flags & N_TagsIndirect)D 		{h" 		  node->flags |= N_TagsIndirect;) 		  node->parent = file_buffer->fullpath;n 		}a 	    }  ' 	  if (subfile->flags & N_IsCompressed)A# 	    node->flags |= N_IsCompressed;   @ 	  /* If TAG->nodelen hasn't been calculated yet, then we aren'tA 	     in a position to trust the entry pointer.  Adjust things sorB 	     that ENTRY->nodestart gets the exact address of the start ofC 	     the node separator which starts this node, and NODE->contents D 	     gets the address of the line defining this node.  If we cannot- 	     do that, the node isn't really here. */; 	  if (tag->nodelen == -1) 	    { 	      int min, max; 	      char *node_sep;  	      SEARCH_BINDING node_body; 	      char *buff_end;  & 	      min = max = DEFAULT_INFO_FUDGE;  / 	      if (tag->nodestart < DEFAULT_INFO_FUDGE)e 		min = tag->nodestart;c   	      if (DEFAULT_INFO_FUDGE >h) 		  (subfile->filesize - tag->nodestart))t+ 		max = subfile->filesize - tag->nodestart;   B 	      /* NODE_SEP gets the address of the separator which defines7 		 this node, or (char *)NULL if the node wasn't found.u: 		 NODE->contents is side-effected to point to right after 		 the separator. */4 	      node_sep = adjust_nodestart (node, min, max);$ 	      if (node_sep == (char *)NULL) 		{m 		  free (node); 		  return ((NODE *)NULL); 		}n% 	      /* Readjust tag->nodestart. */t5 	      tag->nodestart = node_sep - subfile->contents;t  6 	      /* Calculate the length of the current node. */8 	      buff_end = subfile->contents + subfile->filesize;  ) 	      node_body.buffer = node->contents;i 	      node_body.start = 0; 3 	      node_body.end = buff_end - node_body.buffer;t 	      node_body.flags = 0;*3 	      tag->nodelen = get_node_length (&node_body);l 	    } 	  else  	    {@ 	      /* Since we know the length of this node, we have already9 		 adjusted tag->nodestart to point to the exact start ofa* 		 it.  Simply skip the node separator. */> 	      node->contents += skip_node_separator (node->contents); 	    }    	  node->nodelen = tag->nodelen; 	  return (node);l 	}       }f  D   /* There was a tag table for this file, and the node wasn't found.F      Return NULL, since this file doesn't contain the desired node. */   return ((NODE *)NULL); }s _F /* **************************************************************** */ /*								    */3 /*		Managing file_buffers, nodes, and tags.		    */s /*								    */F /* **************************************************************** */   static FILE_BUFFER * make_file_buffer ()t {)   FILE_BUFFER *file_buffer;f  >   file_buffer = (FILE_BUFFER *)xmalloc (sizeof (FILE_BUFFER));?   file_buffer->filename = file_buffer->fullpath = (char *)NULL;e'   file_buffer->contents = (char *)NULL;g#   file_buffer->tags = (TAG **)NULL;n(   file_buffer->subfiles = (char **)NULL;   file_buffer->tags_slots = 0;   file_buffer->flags = 0;r     return (file_buffer);L }s  ? /* Add FILE_BUFFER to our list of already loaded info files. */- static void   remember_info_file (file_buffer)      FILE_BUFFER *file_buffer; {s   int i;  =   for (i = 0; info_loaded_files && info_loaded_files[i]; i++)_     ;   :   add_pointer_to_array (file_buffer, i, info_loaded_files,/ 			info_loaded_files_slots, 10, FILE_BUFFER *);> }e  I /* Forget the contents, tags table, nodes list, and names of FILENAME. */1 static void> forget_info_file (filename)       char *filename; {o   register int i;l   FILE_BUFFER *file_buffer;I     if (!info_loaded_files),     return;b  6   for (i = 0; file_buffer = info_loaded_files[i]; i++):     if ((strcmp (filename, file_buffer->filename) == 0) ||1 	(strcmp (filename, file_buffer->fullpath) == 0))i       {  	free (file_buffer->filename); 	free (file_buffer->fullpath);   	if (file_buffer->contents)u  	  free (file_buffer->contents); 	e> 	/* Note that free_file_buffer_tags () also kills the subfiles? 	   list, since the subfiles list is only of use in conjunctionh 	   with tags. */h% 	free_file_buffer_tags (file_buffer);a  6 	while (info_loaded_files[i] = info_loaded_files[++i]) 	  ;   	break;h       }h }e  9 /* Free the tags (if any) associated with FILE_BUFFER. */u static voide# free_file_buffer_tags (file_buffer)       FILE_BUFFER *file_buffer; {n   register int i;.     if (file_buffer->tags)     {        register TAG *tag;  2       for (i = 0; tag = file_buffer->tags[i]; i++) 	free_info_tag (tag);          free (file_buffer->tags);e'       file_buffer->tags = (TAG **)NULL;n"       file_buffer->tags_slots = 0;     }*     if (file_buffer->subfiles)     {e0       for (i = 0; file_buffer->subfiles[i]; i++)! 	free (file_buffer->subfiles[i]);n  #       free (file_buffer->subfiles);p,       file_buffer->subfiles = (char **)NULL;     }_ }   ? /* Free the data associated with TAG, as well as TAG itself. */i static void  free_info_tag (tag)[      TAG *tag; {_   free (tag->nodename);*  F   /* We don't free tag->filename, because that filename is part of theE      subfiles list for the containing FILE_BUFFER.  free_info_tags () 6      will free the subfiles when it is appropriate. */  
   free (tag);n }(  G /* Load the contents of FILE_BUFFER->contents.  This function is calledfK    when a file buffer was loaded, and then in order to conserve memory, the L    file buffer's contents were freed and the pointer was zero'ed.  Note thatM    the file was already loaded at least once successfully, so the tags and/or /    nodes members are still correctly filled. */  static voidx% info_reload_file_buffer_contents (fb)l      FILE_BUFFER *fb;r {=   fb->flags &= ~N_IsCompressed;   2   /* Let the filesystem do all the work for us. */   fb->contents =I     filesys_read_info_file (fb->fullpath, &(fb->filesize), &(fb->finfo));o(   if (fb->filesize != fb->finfo.st_size)      fb->flags |= N_IsCompressed; }   E /* Return the actual starting memory location of NODE, side-effecting L    NODE->contents.  MIN and MAX are bounds for a search if one is necessary.K    Because of the way that tags are implemented, the physical nodestart mayfJ    not actually be where the tag says it is.  If that is the case, but theJ    node was found anyway, set N_UpdateTags in NODE->flags.  If the node isM    found, return non-zero.  NODE->contents is returned positioned right afteryH    the node separator that precedes this node, while the return value isK    position directly on the separator that precedes this node.  If the node 0    could not be found, return a NULL pointer. */
 static char *y! adjust_nodestart (node, min, max)       NODE *node;      int min, max; {R   long position;   SEARCH_BINDING node_body;e     /* Define the node body. */a$   node_body.buffer = node->contents;   node_body.start = 0;   node_body.end = max;   node_body.flags = 0;  G   /* Try the optimal case first.  Who knows?  This file may actually be %      formatted (mostly) correctly. */D4   if (node_body.buffer[0] != INFO_COOKIE && min > 2)     node_body.buffer -= 3;  .   position = find_node_separator (&node_body);  4   /* If we found a node start, then check it out. */   if (position != -1)b     {N       int sep_len;  5       sep_len = skip_node_separator (node->contents);   I       /* If we managed to skip a node separator, then check for this nodet 	 being the right one. */        if (sep_len != 0)b 	{ 	  char *nodedef, *nodestart;p 	  int offset;  5 	  nodestart = node_body.buffer + position + sep_len;b 	  nodedef = nodestart;e6 	  offset = string_in_line (INFO_NODE_LABEL, nodedef);   	  if (offset != -1) 	    { 	      nodedef += offset;i, 	      nodedef += skip_whitespace (nodedef);C 	      offset = skip_node_characters (nodedef, DONT_SKIP_NEWLINES);e1 	      if ((offset == strlen (node->nodename)) && 5 		  (strncmp (node->nodename, nodedef, offset) == 0))  		{D 		  node->contents = nodestart;n) 		  return (node_body.buffer + position);d 		}d 	    } 	}     }m  D   /* Oh well, I guess we have to try to find it in a larger area. */*   node_body.buffer = node->contents - min;   node_body.start = 0;   node_body.end = min + max;   node_body.flags = 0;  ?   position = find_node_in_binding (node->nodename, &node_body);t  3   /* If the node couldn't be found, we lose big. */>   if (position == -1)u     return ((char *)NULL);  J   /* Otherwise, the node was found, but the tags table could need updatingN      (if we used a tag to get here, that is).  Set the flag in NODE->flags. *//   node->contents = node_body.buffer + position;s9   node->contents += skip_node_separator (node->contents); #   if (node->flags & N_HasTagsTable)r      node->flags |= N_UpdateTags;'   return (node_body.buffer + position);  }e