" /* window.c -- Windows in Info. */  L /* This file is part of GNU Info, a program for reading online documentation    stored in Info format.   4    Copyright (C) 1993 Free Software Foundation, Inc.  G    This program is free software; you can redistribute it and/or modify G    it under the terms of the GNU General Public License as published by F    the Free Software Foundation; either version 2, or (at your option)    any later version.   B    This program is distributed in the hope that it will be useful,A    but WITHOUT ANY WARRANTY; without even the implied warranty of @    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the/    GNU General Public License for more details.   D    You should have received a copy of the GNU General Public License>    along with this program; if not, write to the Free Software<    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  -    Written by Brian Fox (bfox@ai.mit.edu). */    #include <stdio.h> #include <ctype.h> #include <sys/types.h> #include <sys/stat.h>  #include "nodes.h" #include "window.h"  #include "display.h" #include "info-utils.h"  #include "infomap.h"  , /* The window which describes the screen. */$ WINDOW *the_screen = (WINDOW *)NULL;  / /* The window which describes the echo area. */ ' WINDOW *the_echo_area = (WINDOW *)NULL;   " /* The list of windows in Info. */! WINDOW *windows = (WINDOW *)NULL;   2 /* Pointer to the active window in WINDOW_LIST. */' WINDOW *active_window = (WINDOW *)NULL;   L /* The size of the echo area in Info.  It never changes, irregardless of the    size of the screen. */  #define ECHO_AREA_HEIGHT 1  O /* Macro returns the amount of space that the echo area truly requires relative     to the entire screen. */ 6 #define echo_area_required (1 + the_echo_area->height)  H /* Initalize the window system by creating THE_SCREEN and THE_ECHO_AREA.     Create the first window ever.7    You pass the dimensions of the total screen size. */  void) window_initialize_windows (width, height)       int width, height;  { 3   the_screen = (WINDOW *)xmalloc (sizeof (WINDOW)); 6   the_echo_area = (WINDOW *)xmalloc (sizeof (WINDOW));0   windows = (WINDOW *)xmalloc (sizeof (WINDOW));   active_window = windows;  )   zero_mem (the_screen, sizeof (WINDOW)); ,   zero_mem (the_echo_area, sizeof (WINDOW));,   zero_mem (active_window, sizeof (WINDOW));  4   /* None of these windows has a goal column yet. */"   the_echo_area->goal_column = -1;"   active_window->goal_column = -1;   the_screen->goal_column = -1;   2   /* The active and echo_area windows are visible.       The echo_area is permanent.       The screen is permanent. */)   active_window->flags = W_WindowVisible; J   the_echo_area->flags = W_WindowIsPerm | W_InhibitMode | W_WindowVisible;(   the_screen->flags    = W_WindowIsPerm;  K   /* The height of the echo area never changes.  It is statically set right G      here, and it must be at least 1 line for display.  The size of the K      initial window cannot be the same size as the screen, since the screen J      includes the echo area.  So, we make the height of the initial windowJ      equal to the screen's displayable region minus the height of the echo
      area. */ +   the_echo_area->height = ECHO_AREA_HEIGHT; I   active_window->height = the_screen->height - 1 - the_echo_area->height; <   window_new_screen_size (width, height, (VFunction *)NULL);  G   /* The echo area uses a different keymap than normal info windows. */ +   the_echo_area->keymap = echo_area_keymap; &   active_window->keymap = info_keymap; }   D /* Given that the size of the screen has changed to WIDTH and HEIGHTF    from whatever it was before (found in the_screen->height, ->width),H    change the size (and possibly location) of each window in the screen.G    If a window would become too small, call the function DELETER on it, L    after deleting the window from our chain of windows.  If DELETER is NULL,K    nothing extra is done.  The last window can never be deleted, but it can     become invisible. */   L /* If non-null, a function to call with WINDOW as argument when the function3    window_new_screen_size () has deleted WINDOW. */ 8 VFunction *window_deletion_notifier = (VFunction *)NULL;   void& window_new_screen_size (width, height)      int width, height;  {    register WINDOW *win; /   int delta_height, delta_each, delta_leftover;    int numwins;  !   /* If no change, do nothing. */ A   if (width == the_screen->width && height == the_screen->height)      return;   ?   /* If the new window height is too small, make it be zero. */ 9   if (height < (WINDOW_MIN_SIZE + the_echo_area->height))      height = 0;    if (width < 0)     width = 0;  .   /* Find out how many windows will change. */D   for (numwins = 0, win = windows; win; win = win->next, numwins++);  F   /* See if some windows will need to be deleted.  This is the case ifF      the screen is getting smaller, and the available space divided byG      the number of windows is less than WINDOW_MIN_SIZE.  In that case, C      delete some windows and try again until there is either enough C      space to divy up among the windows, or until there is only one       window left. */D   while ((height - echo_area_required) / numwins <= WINDOW_MIN_SIZE)     { D       /* If only one window, make the size of it be zero, and return 	 immediately. */        if (!windows->next)  	{ 	  windows->height = 0; % 	  maybe_free (windows->line_starts); ( 	  windows->line_starts = (char **)NULL; 	  windows->line_count = 0; 	 	  break;  	}  B       /* If we have some temporary windows, delete one of them. *//       for (win = windows; win; win = win->next)  	if (win->flags & W_TempWindow) 	 	  break;   >       /* Otherwise, delete the first window, and try again. */       if (!win)  	win = windows;   #       if (window_deletion_notifier) # 	(*window_deletion_notifier) (win);   !       window_delete_window (win);        numwins--;     }   0   /* The screen has changed height and width. */E   delta_height = height - the_screen->height;	/* This is how much. */ >   the_screen->height = height;			/* This is the new height. */;   the_screen->width = width;			/* This is the new width. */   '   /* Set the start of the echo area. */ <   the_echo_area->first_row = height - the_echo_area->height;   the_echo_area->width = width;   B   /* Check to see if the screen can really be changed this way. */I   if ((!windows->next) && ((windows->height == 0) && (delta_height < 0)))      return;   @   /* Divide the change in height among the available windows. */&   delta_each = delta_height / numwins;9   delta_leftover = delta_height - (delta_each * numwins);   I   /* Change the height of each window in the chain by delta_each.  Change H      the height of the last window in the chain by delta_each and by theF      leftover amount of change.  Change the width of each window to be      WIDTH. */+   for (win = windows; win; win = win->next)      { G       if ((win->width != width) && ((win->flags & W_InhibitMode) == 0))  	{ 	  win->width = width; 	  maybe_free (win->modeline);/ 	  win->modeline = (char *)xmalloc (1 + width);  	}          win->height += delta_each;  H       /* If the previous height of this window was zero, it was the onlyA 	 window, and it was not visible.  Thus we need to compensate for  	 the echo_area. */ $       if (win->height == delta_each), 	win->height -= (1 + the_echo_area->height);  F       /* If this is not the first window in the chain, then change theC 	 first row of it.  We cannot just add delta_each to the first row, F 	 since this window's first row is the sum of the collective increasesF 	 that have gone before it.  So we just add one to the location of the  	 previous window's modeline. */       if (win->prev)A 	win->first_row = (win->prev->first_row + win->prev->height) + 1;   M       /* The last window in the chain gets the extra space (or shrinkage). */        if (!win->next)  	win->height += delta_leftover;          if (win->node) 	recalculate_line_starts (win);   #       win->flags |= W_UpdateWindow;      }   E   /* If the screen got smaller, check over the windows just shrunk to*J      keep them within bounds.  Some of the windows may have gotten smallerG      than WINDOW_MIN_HEIGHT in which case some of the other windows arepK      larger than the available display space in the screen.  Because of ourtI      intial test above, we know that there is enough space for all of theG      windows. */D   if ((delta_each < 0) && ((windows->height != 0) && windows->next))     {o       int avail;  E       avail = the_screen->height - (numwins + the_echo_area->height);a       win = windows;         while (win)R 	{+ 	  if ((win->height < WINDOW_MIN_HEIGHT) ||M 	      (win->height > avail))  	    { 	      WINDOW *lastwin;G  9 	      /* Split the space among the available windows. */ $ 	      delta_each = avail / numwins;7 	      delta_leftover = avail - (delta_each * numwins);t  0 	      for (win = windows; win; win = win->next) 		{, 		  lastwin = win; 		  if (win->prev) 		    win->first_row =7 		      (win->prev->first_row + win->prev->height) + 1;i 		  win->height = delta_each;< 		}t  A 	      /* Give the leftover space (if any) to the last window. */") 	      lastwin->height += delta_leftover;a
 	      break;w 	    } 	  elsei 	    win= win->next; 	}     }c }n  D /* Make a new window showing NODE, and return that window structure.F    If NODE is passed as NULL, then show the node showing in the activeG    window.  If the window could not be made return a NULL pointer.  Then"    active window is not changed.*/ WINDOW * window_make_window (node)e      NODE *node; {a   WINDOW *window;      if (!node)     node = active_window->node;e  F   /* If there isn't enough room to make another window, return now. */4   if ((active_window->height / 2) < WINDOW_MIN_SIZE)     return ((WINDOW *)NULL);  (   /* Make and initialize the new window.L      The fudging about with -1 and +1 is because the following window in theK      chain cannot start at window->height, since that is where the modelinecJ      for the previous window is displayed.  The inverse adjustment is made#      in window_delete_window (). */I/   window = (WINDOW *)xmalloc (sizeof (WINDOW));e$   window->width = the_screen->width;3   window->height = (active_window->height / 2) - 1; ! #if defined (SPLIT_BEFORE_ACTIVE)o/   window->first_row = active_window->first_row;( #elseo0   window->first_row = active_window->first_row +-     (active_window->height - window->height);o #endif   window->keymap = info_keymap;a   window->goal_column = -1;i9   window->modeline = (char *)xmalloc (1 + window->width);1&   window->line_starts = (char **)NULL;3   window->flags = W_UpdateWindow | W_WindowVisible; +   window_set_node_of_window (window, node);o  3   /* Adjust the height of the old active window. */W0   active_window->height -= (window->height + 1);! #if defined (SPLIT_BEFORE_ACTIVE)P3   active_window->first_row += (window->height + 1);s #endif)   active_window->flags |= W_UpdateWindow;u  J   /* Readjust the new and old windows so that their modelines and contents$      will be displayed correctly. */ #if defined (NOTDEF)K   /* We don't have to do this for WINDOW since window_set_node_of_window ()u      already did. */!   window_adjust_pagetop (window);     window_make_modeline (window); #endif /* NOTDEF */O  :   /* We do have to readjust the existing active window. */(   window_adjust_pagetop (active_window);'   window_make_modeline (active_window);)  ! #if defined (SPLIT_BEFORE_ACTIVE)iG   /* This window is just before the active one.  The active window getsc;      bumped down one.  The active window is not changed. */    window->next = active_window;c  %   window->prev = active_window->prev;f   active_window->prev = window;      if (window->prev)-      window->prev->next = window;   else     windows = window;o #elseeI   /* This window is just after the active one.  Which window is active iso      not changed. */   window->prev = active_window;w%   window->next = active_window->next;g   active_window->next = window;c   if (window->next)b      window->next->prev = window;! #endif /* !SPLIT_BEFORE_ACTIVE */    return (window); }u  ; /* These useful macros make it possible to read the code ind%    window_change_window_height (). */n0 #define grow_me_shrinking_next(me, next, diff) \   do { \     me->height += diff; \t     next->height -= diff; \t     next->first_row += diff; \#     window_adjust_pagetop (next); \ 
   } while (0)   0 #define grow_me_shrinking_prev(me, prev, diff) \   do { \     me->height += diff; \e     prev->height -= diff; \      me->first_row -=diff; \ #     window_adjust_pagetop (prev); \ 
   } while (0)I  0 #define shrink_me_growing_next(me, next, diff) \   do { \     me->height -= diff; \      next->height += diff; \w     next->first_row -= diff; \#     window_adjust_pagetop (next); \ 
   } while (0)i  0 #define shrink_me_growing_prev(me, prev, diff) \   do { \     me->height -= diff; \e     prev->height += diff; \n     me->first_row += diff; \#     window_adjust_pagetop (prev); \s
   } while (0)N  J /* Change the height of WINDOW by AMOUNT.  This also automagically adjustsI    the previous and next windows in the chain.  If there is only one usero)    window, then no change takes place. */e void, window_change_window_height (window, amount)      WINDOW *window;      int amount; {m%   register WINDOW *win, *prev, *next;   E   /* If there is only one window, or if the amount of change is zero,0      return immediately. */i$   if (!windows->next || amount == 0)     return;L  &   /* Find this window in our chain. */+   for (win = windows; win; win = win->next)w     if (win == window)       break;  H   /* If the window is isolated (i.e., doesn't appear in our window list,      then quit now. */   if (!win)e     return;t  E   /* Change the height of this window by AMOUNT, if that is possible.fE      It can be impossible if there isn't enough available room on the >      screen, or if the resultant window would be too small. */       prev = window->prev;     next = window->next;  "   /* WINDOW decreasing in size? */   if (amount < 0)c     { I       int abs_amount = -amount;	/* It is easier to deal with this way. */n  B       /* If the resultant window would be too small, stop here. */<       if ((window->height - abs_amount) < WINDOW_MIN_HEIGHT) 	return;  J       /* If we have two neighboring windows, choose the smaller one to get 	 larger. */       if (next && prev)  	{# 	  if (prev->height < next->height)h7 	    shrink_me_growing_prev (window, prev, abs_amount);v 	  elseh7 	    shrink_me_growing_next (window, next, abs_amount);a 	}       else if (next)3 	shrink_me_growing_next (window, next, abs_amount);u
       else3 	shrink_me_growing_prev (window, prev, abs_amount);      }a  "   /* WINDOW increasing in size? */   if (amount > 0)h     {d6       int total_avail, next_avail = 0, prev_avail = 0;         if (next)e- 	next_avail = next->height - WINDOW_MIN_SIZE;w         if (prev)w- 	prev_avail = prev->height - WINDOW_MIN_SIZE;d  ,       total_avail = next_avail + prev_avail;  O       /* If there isn't enough space available to grow this window, give up. */*       if (amount > total_avail)  	return;  L       /* If there aren't two neighboring windows, or if one of the neighborsD 	 is larger than the other one by at least AMOUNT, grow that one. */C       if ((next && !prev) || ((next_avail - amount) >= prev_avail))c/ 	grow_me_shrinking_next (window, next, amount); H       else if ((prev && !next) || ((prev_avail - amount) >= next_avail))/ 	grow_me_shrinking_prev (window, prev, amount); 
       else 	{ 	  int change;  D 	  /* This window has two neighbors.  They both must be shrunk in toD 	     make enough space for WINDOW to grow.  Make them both the same 	     size. */ 	  if (prev_avail > next_avail)n 	    {( 	      change = prev_avail - next_avail;5 	      grow_me_shrinking_prev (window, prev, change);e 	      amount -= change; 	    } 	  else  	    {( 	      change = next_avail - prev_avail;5 	      grow_me_shrinking_next (window, next, change);i 	      amount -= change; 	    }  @ 	  /* Both neighbors are the same size.  Split the difference in 	     AMOUNT between them. */k 	  while (amount)n 	    { 	      window->height++; 	      amount--;  3 	      /* Odd numbers grow next, even grow prev. */e 	      if (amount & 1) 		{h 		  prev->height--;l 		  window->first_row--; 		}e 	      else  		{a 		  next->height--;  		  next->first_row++; 		}r 	    }  	  window_adjust_pagetop (prev);  	  window_adjust_pagetop (next); 	}     }n   if (prev) "     prev->flags |= W_UpdateWindow;     if (next)n"     next->flags |= W_UpdateWindow;  "   window->flags |= W_UpdateWindow;!   window_adjust_pagetop (window);h }h  E /* Tile all of the windows currently displayed in the global variable I    WINDOWS.  If argument STYLE is TILE_INTERNALS, tile windows displaying.E    internal nodes as well, otherwise do not change the height of suchi    windows. */ void window_tile_windows (style)i      int style;  {-   WINDOW *win, *last_adjusted;/   int numwins, avail, per_win_height, leftover;    int do_internals;_     numwins = avail = 0;+   do_internals = (style == TILE_INTERNALS);   +   for (win = windows; win; win = win->next)h%     if (do_internals || !win->node ||g( 	(win->node->flags & N_IsInternal) == 0)       {e 	avail += win->height; 	numwins++;c       }*  *   if (numwins <= 1 || !the_screen->height)     return;u  J   /* Find the size for each window.  Divide the size of the usable portion/      of the screen by the number of windows. */u#   per_win_height = avail / numwins;i0   leftover = avail - (per_win_height * numwins);  !   last_adjusted = (WINDOW *)NULL;I+   for (win = windows; win; win = win->next)t     {d'       if (do_internals || !win->node ||r* 	  (win->node->flags & N_IsInternal) == 0) 	{ 	  last_adjusted = win;/  	  win->height = per_win_height; 	}     })     if (last_adjusted)&     last_adjusted->height += leftover;  <   /* Readjust the first_row of every window in the chain. */+   for (win = windows; win; win = win->next)s     {a       if (win->prev)? 	win->first_row = win->prev->first_row + win->prev->height + 1;t  "       window_adjust_pagetop (win);#       win->flags |= W_UpdateWindow;a     }i }f  K /* Toggle the state of line wrapping in WINDOW.  This can do a bit of fancyw    redisplay. */ void window_toggle_wrap (window)C      WINDOW *window; {o   if (window->flags & W_NoWrap)e     window->flags &= ~W_NoWrap;e   else     window->flags |= W_NoWrap;     if (window != the_echo_area)     {i       char **old_starts;!       int old_lines, old_pagetop;   '       old_starts = window->line_starts;d%       old_lines = window->line_count;c$       old_pagetop = window->pagetop;  %       calculate_line_starts (window);t  <       /* Make sure that point appears within this window. */%       window_adjust_pagetop (window);o  I       /* If the pagetop hasn't changed maybe we can do some scrolling nowiE 	 to speed up the display.  Many of the line starts will be the same,-2 	 so scrolling here is a very good optimization.*/)       if (old_pagetop == window->pagetop)  	display_scroll_line_startsy0 	  (window, old_pagetop, old_starts, old_lines);       maybe_free (old_starts);     }n"   window->flags |= W_UpdateWindow; }y  ! /* Set WINDOW to display NODE. */n void( window_set_node_of_window (window, node)      WINDOW *window;      NODE *node; {s   window->node = node;   window->pagetop = 0;   window->point = 0;#   recalculate_line_starts (window);w"   window->flags |= W_UpdateWindow;!   window_adjust_pagetop (window);t    window_make_modeline (window); }i  H /* Delete WINDOW from the list of known windows.  If this window was theI    active window, make the next window in the chain be the active window. J    If the active window is the next or previous window, choose that windowN    as the recipient of the extra space.  Otherwise, prefer the next window. */ void window_delete_window (window)       WINDOW *window; {g&   WINDOW *next, *prev, *window_to_fix;     next = window->next;   prev = window->prev;  @   /* You cannot delete the only window or a permanent window. */;   if ((!next && !prev) || (window->flags & W_WindowIsPerm))i     return;      if (next)      next->prev = prev;     if (!prev)     windows = next;e   else     prev->next = next;     if (window->line_starts)     free (window->line_starts);g     if (window->modeline)e     free (window->modeline);     if (window == active_window)     {eI       /* If there isn't a next window, then there must be a previous one,dE 	 since we cannot delete the last window.  If there is a next window, - 	 prefer to use that as the active window. */j       if (next)) 	active_window = next;
       else 	active_window = prev;     }f  $   if (next && active_window == next)     window_to_fix = next;f)   else if (prev && active_window == prev)i     window_to_fix = prev;    else if (next)     window_to_fix = next;n   else if (prev)     window_to_fix = prev;e   else     window_to_fix = windows;     3   if (window_to_fix->first_row > window->first_row)a     {p       int diff;l  E       /* Try to adjust the visible part of the node so that as littlel" 	 text as possible has to move. */:       diff = window_to_fix->first_row - window->first_row;3       window_to_fix->first_row = window->first_row;g  %       window_to_fix->pagetop -= diff;N%       if (window_to_fix->pagetop < 0)e 	window_to_fix->pagetop = 0;     }   K   /* The `+ 1' is to offset the difference between the first_row locations.t.      See the code in window_make_window (). */.   window_to_fix->height += window->height + 1;)   window_to_fix->flags |= W_UpdateWindow;i     free (window); }   G /* For every window in CHAIN, set the flags member to have FLAG set. */  void window_mark_chain (chain, flag)       WINDOW *chain;       int flag; {u   register WINDOW *win;e  )   for (win = chain; win; win = win->next)s     win->flags |= flag;p }i  @ /* For every window in CHAIN, clear the flags member of FLAG. */ void! window_unmark_chain (chain, flag)       WINDOW *chain;o      int flag; {=   register WINDOW *win;N  )   for (win = chain; win; win = win->next)c     win->flags &= ~flag; }t  G /* Return the number of characters it takes to display CHARACTER on thee    screen at HPOS. */  intm! character_width (character, hpos)o      int character, hpos;I {W   int printable_limit = 127;   int width = 1;     if (ISO_Latin_p)     printable_limit = 160;  "   if (character > printable_limit)     width = 3;   else if (iscntrl (character))      {n       switch (character) 	{ 	case '\r':) 	case '\n': $ 	  width = the_screen->width - hpos;	 	  break;o 	case '\t': & 	  width = ((hpos + 8) & 0xf8) - hpos;	 	  break;,	 	default:a
 	  width = 2;  	}     }n   else if (character == DEL)     width = 2;     return (width);O }n  K /* Return the number of characters it takes to display STRING on the screenv    at HPOS. */ int  string_width (string, hpos)_      char *string;      int hpos; {w)   register int i, width, this_char_width;e  (   for (width = 0, i = 0; string[i]; i++)     {t:       this_char_width = character_width (string[i], hpos);       width += this_char_width;i       hpos += this_char_width;     }a   return (width);  }   C /* Quickly guess the approximate number of lines to that NODE would A    take to display.  This really only counts carriage returns. *// int  window_physical_lines (node)      NODE *node; {p   register int i, lines;   char *contents;,     if (!node)     return (0);(     contents = node->contents;0   for (i = 0, lines = 1; i < node->nodelen; i++)     if (contents[i] == '\n')       lines++;     return (lines);h }w  N /* Calculate a list of line starts for the node belonging to WINDOW.  The lineA    starts are pointers to the actual text within WINDOW->NODE. */  void calculate_line_starts (window)      WINDOW *window; {-   register int i, hpos;_%   char **line_starts = (char **)NULL;)3   int line_starts_index = 0, line_starts_slots = 0;{   int bump_index;e
   NODE *node;_  &   window->line_starts = (char **)NULL;   window->line_count = 0;    node = window->node;     if (!node)     return;e  I   /* Grovel the node starting at the top, and for each line calculate thewI      width of the characters appearing in that line.  Add each line start       to our array. */x   i = 0;   hpos = 0;    bump_index = 0;      while (i < node->nodelen)      {d&       char *line = node->contents + i;       unsigned int cwidth, c;e  A       add_pointer_to_array (line, line_starts_index, line_starts,w' 			    line_starts_slots, 100, char *);        if (bump_index)> 	{ 	  i++;d 	  bump_index = 0; 	}         while (1)  	{ 	  c = node->contents[i];>& 	  cwidth = character_width (c, hpos);  G 	  /* If this character fits within this line, just do the next one. */i' 	  if ((hpos + cwidth) < window->width)r 	    { 	      i++;I 	      hpos += cwidth; 	      continue; 	    } 	  elsel 	    {E 	      /* If this character would position the cursor at the start ofl: 		 the next printed screen line, then do the next line. *// 	      if (c == '\n' || c == '\r' || c == '\t')t 		{t 		  i++;
 		  hpos = 0;s
 		  break; 		}  	      else  		{t> 		  /* This character passes the window width border.  Postion@ 		     the cursor after the printed character, but remember this? 		     line start as where this character is.  A bit tricky. */i  = 		  /* If this window doesn't wrap lines, proceed to the next- 		     physical line here. */F! 		  if (window->flags & W_NoWrap)d 		    {e 		      hpos = 0;i> 		      while (i < node->nodelen && node->contents[i] != '\n') 			i++;a  & 		      if (node->contents[i] == '\n') 			i++;g 		    }i 		  else 		    {s( 		      hpos = the_screen->width - hpos; 		      bump_index++;t 		    }d
 		  break; 		}t 	    } 	}     }e$   window->line_starts = line_starts;)   window->line_count = line_starts_index;  }n  I /* Given WINDOW, recalculate the line starts for the node it displays. */s void  recalculate_line_starts (window)      WINDOW *window; {w#   maybe_free (window->line_starts);w!   calculate_line_starts (window);  }   J /* Global variable control redisplay of scrolled windows.  If non-zero, itG    is the desired number of lines to scroll the window in order to makeaG    point visible.  A user might set this to 1 for smooth scrolling.  If K    set to zero, the line containing point is centered within the window. */d int window_scroll_step = 0;o  N /* Adjust the pagetop of WINDOW such that the cursor point will be visible. */ void window_adjust_pagetop (window)      WINDOW *window; {    register int line = 0;   char *contents;_     if (!window->node)     return;r  $   contents = window->node->contents;  G   /* Find the first printed line start which is after WINDOW->point. */a3   for (line = 0; line < window->line_count; line++)t     {n       char *line_start;d  -       line_start = window->line_starts[line];   2       if ((line_start - contents) > window->point) 	break;o     }o  G   /* The line index preceding the line start which is past point is the       one containing point. */o	   line--;t  F   /* If this line appears in the current displayable page, do nothing.H      Otherwise, adjust the top of the page to make this line visible. */!   if ((line < window->pagetop) ||l6       (line - window->pagetop > (window->height - 1)))     {nD       /* The user-settable variable "scroll-step" is used to attempt> 	 to make point visible, iff it is non-zero.  If that variable< 	 is zero, then the line containing point is centered within 	 the window. */.       if (window_scroll_step < window->height) 	{" 	  if ((line < window->pagetop) &&8 	      ((window->pagetop - window_scroll_step) <= line))+ 	    window->pagetop -= window_scroll_step; > 	  else if ((line - window->pagetop > (window->height - 1)) &&4 		   ((line - (window->pagetop + window_scroll_step) 		     < window->height)))+ 	    window->pagetop += window_scroll_step;r 	  else 9 	    window->pagetop = line - ((window->height - 1) / 2);O 	}
       else5 	window->pagetop = line - ((window->height - 1) / 2);o         if (window->pagetop < 0) 	window->pagetop = 0;e&       window->flags |= W_UpdateWindow;     }f }!  4 /* Return the index of the line containing point. */ intu window_line_of_point (window)>      WINDOW *window; {e   register int i, start = 0;  D   /* Try to optimize.  Check to see if point is past the pagetop forC      this window, and if so, start searching forward from there. */eG   if ((window->pagetop > -1 && window->pagetop < window->line_count) && E       (window->line_starts[window->pagetop] - window->node->contents)e       <= window->point)      start = window->pagetop;  .   for (i = start; i < window->line_count; i++)     {tL       if ((window->line_starts[i] - window->node->contents) > window->point) 	break;      }w     return (i - 1);i }w  5 /* Get and return the goal column for this window. */v int  window_get_goal_column (window)i      WINDOW *window; {i   if (!window->node)     return (-1);      if (window->goal_column != -1)!     return (window->goal_column);d  >   /* Okay, do the work.  Find the printed offset of the cursor      in this window. */u-   return (window_get_cursor_column (window));l }   L /* Get and return the printed column offset of the cursor in this window. */ inti! window_get_cursor_column (window)t      WINDOW *window; {g   int i, hpos, end;i
   char *line;i  $   i = window_line_of_point (window);     if (i < 0)     return (-1);      line = window->line_starts[i];8   end = window->point - (line - window->node->contents);  %   for (hpos = 0, i = 0; i < end; i++)/,     hpos += character_width (line[i], hpos);     return (hpos); }a  I /* Count the number of characters in LINE that precede the printed columnN    offset of GOAL. */t inte! window_chars_to_goal (line, goal)h      char *line;      int goal; {n   register int i, check, hpos;  -   for (hpos = 0, i = 0; line[i] != '\n'; i++)=     {e  5       check = hpos + character_width (line[i], hpos);          if (check > goal)e 	break;L         hpos = check;m     }i
   return (i);  }   E /* Create a modeline for WINDOW, and store it in window->modeline. */  void window_make_modeline (window)       WINDOW *window; {    register int i;    char *modeline;s   char location_indicator[4];o   int lines_remaining;  =   /* Only make modelines for those windows which have one. */t$   if (window->flags & W_InhibitMode)     return;   C   /* Find the number of lines actually displayed in this window. */h9   lines_remaining = window->line_count - window->pagetop;t     if (window->pagetop == 0)      {(,       if (lines_remaining <= window->height)$ 	strcpy (location_indicator, "All");
       else$ 	strcpy (location_indicator, "Top");     }h   else     {,,       if (lines_remaining <= window->height)$ 	strcpy (location_indicator, "Bot");
       else 	{ 	  float pt, lc; 	  int percentage;   	  pt = (float)window->pagetop;R" 	  lc = (float)window->line_count;    	  percentage = 100 * (pt / lc);  5 	  sprintf (location_indicator, "%2d%%", percentage);  	}     }_  K   /* Calculate the maximum size of the information to stick in MODELINE. */a   {t     int modeline_len = 0;[9     char *parent = (char *)NULL, *filename = "*no file*"; !     char *nodename = "*no node*";((     char *update_message = (char *)NULL;     NODE *node = window->node;  
     if (node)k       {l 	if (node->nodename) 	  nodename = node->nodename;n   	if (node->parent) 	  {4 	    parent = filename_non_directory (node->parent);D 	    modeline_len += strlen ("Subfile: ") + strlen (node->filename); 	  }   	if (node->filename)6 	  filename = filename_non_directory (node->filename);    	if (node->flags & N_UpdateTags)1 	  update_message = "--*** Tags out of Date ***";s       }n       if (update_message)h.       modeline_len += strlen (update_message);&     modeline_len += strlen (filename);&     modeline_len += strlen (nodename);:     modeline_len += 4;		/* strlen (location_indicator). */  G     /* 10 for the decimal representation of the number of lines in thisnJ        node, and the remainder of the text that can appear in the line. */@     modeline_len += 10 + strlen ("-----Info: (), lines ----, ");"     modeline_len += window->width;  2     modeline = (char *)xmalloc (1 + modeline_len);  4     /* Special internal windows have no filename. */     if (!parent && !*filename);       sprintf (modeline, "-%s---Info: %s, %d lines --%s--", / 	       (window->flags & W_NoWrap) ? "$" : "-",i: 	       nodename, window->line_count, location_indicator);     else?       sprintf (modeline, "-%s%s-Info: (%s)%s, %d lines --%s--",l/ 	       (window->flags & W_NoWrap) ? "$" : "-",i? 	       (node && (node->flags & N_IsCompressed)) ? "zz" : "--", # 	       parent ? parent : filename,i: 	       nodename, window->line_count, location_indicator);       if (parent)nG       sprintf (modeline + strlen (modeline), " Subfile: %s", filename);        if (update_message)hC       sprintf (modeline + strlen (modeline), "%s", update_message);        i = strlen (modeline);       if (i >= window->width) %       modeline[window->width] = '\0';o     else       {  	while (i < window->width) 	  modeline[i++] = '-';t 	modeline[i] = '\0';       }   (     strcpy (window->modeline, modeline);     free (modeline);   }o }i  E /* Make WINDOW start displaying at PERCENT percentage of its node. */r void( window_goto_percentage (window, percent)      WINDOW *window;      int percent;I {h   int desired_line;p     if (!percent)t     desired_line = 0;a   else     desired_line =C       (int) ((float)window->line_count * ((float)percent / 100.0));   !   window->pagetop = desired_line;i   window->point =aB     window->line_starts[window->pagetop] - window->node->contents;"   window->flags |= W_UpdateWindow;    window_make_modeline (window); }d  4 /* Get the state of WINDOW, and save it in STATE. */ void  window_get_state (window, state)      WINDOW *window;      WINDOW_STATE *state;* {v   state->node = window->node; #   state->pagetop = window->pagetop;    state->point = window->point;n })  1 /* Set the node, pagetop, and point of WINDOW. */e void  window_set_state (window, state)      WINDOW *window;      WINDOW_STATE *state;y { "   if (window->node != state->node)4     window_set_node_of_window (window, state->node);#   window->pagetop = state->pagetop;    window->point = state->point;m }h   lF /* **************************************************************** */ /*								    */, /*		   Manipulating Home-Made Nodes			    */ /*								    */F /* **************************************************************** */  + /* A place to buffer echo area messages. */t+ static NODE *echo_area_node = (NODE *)NULL;(  5 /* Make the node of the_echo_area be an empty one. */- static void  free_echo_area ()  {n   if (echo_area_node)      {I,       maybe_free (echo_area_node->contents);       free (echo_area_node);     }       echo_area_node = (NODE *)NULL;<   window_set_node_of_window (the_echo_area, echo_area_node); }t   E /* Clear the echo area, removing any message that is already present.h+    The echo area is cleared immediately. */  void window_clear_echo_area ()e {t   free_echo_area ();,   display_update_one_window (the_echo_area); }n  L /* Make a message appear in the echo area, built from FORMAT, ARG1 and ARG2.K    The arguments are treated similar to printf () arguments, but not all ofiM    printf () hair is present.  The message appears immediately.  If there wasoB    already a message appearing in the echo area, it is removed. */ void0 window_message_in_echo_area (format, arg1, arg2)      char *format;      void *arg1, *arg2;w {r   free_echo_area ();;   echo_area_node = build_message_node (format, arg1, arg2);-<   window_set_node_of_window (the_echo_area, echo_area_node);,   display_update_one_window (the_echo_area); }d  E /* Place a temporary message in the echo area built from FORMAT, ARG1wC    and ARG2.  The message appears immediately, but does not destroy+D    any existing message.  A future call to unmessage_in_echo_area ()     restores the old contents. */2 static NODE **old_echo_area_nodes = (NODE **)NULL;) static int old_echo_area_nodes_index = 0; ) static int old_echo_area_nodes_slots = 0;    void) message_in_echo_area (format, arg1, arg2)e      char *format;      void *arg1, *arg2;d {l   if (echo_area_node)      {NF       add_pointer_to_array (echo_area_node, old_echo_area_nodes_index,6 			    old_echo_area_nodes, old_echo_area_nodes_slots, 			    4, NODE *);     }s    echo_area_node = (NODE *)NULL;3   window_message_in_echo_area (format, arg1, arg2);o }l   void unmessage_in_echo_area ()_ {r   free_echo_area ();      if (old_echo_area_nodes_index)F     echo_area_node = old_echo_area_nodes[--old_echo_area_nodes_index];  <   window_set_node_of_window (the_echo_area, echo_area_node);,   display_update_one_window (the_echo_area); }e  ! /* A place to build a message. */w+ static char *message_buffer = (char *)NULL;s$ static int message_buffer_index = 0;# static int message_buffer_size = 0;   D /* Ensure that there is enough space to stuff LENGTH characters into    MESSAGE_BUFFER. */d static voidm message_buffer_resize (length)      int length; {s   if (!message_buffer)     {w'       message_buffer_size = length + 1;o=       message_buffer = (char *)xmalloc (message_buffer_size);f       message_buffer_index = 0;      }i  >   while (message_buffer_size <= message_buffer_index + length)     message_buffer = (char *)e       xrealloc (message_buffer,(- 		message_buffer_size += 100 + (2 * length));  }e  J /* Format MESSAGE_BUFFER with the results of printing FORMAT with ARG1 and    ARG2. */o static voidi) build_message_buffer (format, arg1, arg2)_      char *format;      void *arg1, *arg2;a {    register int i, len;   void *args[2];   int arg_index = 0;     args[0] = arg1;    args[1] = arg2;      len = strlen (format);     message_buffer_resize (len);     for (i = 0; format[i]; i++)c     {;       if (format[i] != '%')i 	{6 	  message_buffer[message_buffer_index++] = format[i];	 	  len--;i 	}
       else 	{
 	  char c;   	  c = format[++i];   
 	  switch (c)k 	    {, 	    case '%':		/* Insert a percent sign. */' 	      message_buffer_resize (len + 1);l4 	      message_buffer[message_buffer_index++] = '%';
 	      break;   9 	    case 's':		/* Insert the current arg as a string. */n 	      { 		char *string;e 		int string_len;l  % 		string = (char *)args[arg_index++];h 		string_len = strlen (string);   + 		message_buffer_resize (len + string_len); 	 		sprintfi: 		  (message_buffer + message_buffer_index, "%s", string);% 		message_buffer_index += string_len;p 	      }
 	      break;o  ; 	    case 'd':		/* Insert the current arg as an integer. */  	      { 		int integer;  # 		integer = (int)args[arg_index++];o  # 		message_buffer_resize (len + 32); 	 		sprintf ; 		  (message_buffer + message_buffer_index, "%d", integer);i1 		message_buffer_index = strlen (message_buffer);  	      }
 	      break;   < 	    case 'c':		/* Insert the current arg as a character. */ 	      { 		int character;  % 		character = (int)args[arg_index++];   " 		message_buffer_resize (len + 1);5 		message_buffer[message_buffer_index++] = character;t 	      }
 	      break;t  
 	    default:e 	      abort (); 	    } 	}     }*.   message_buffer[message_buffer_index] = '\0'; }e  F /* Build a new node which has FORMAT printed with ARG1 and ARG2 as the    contents. */  NODE *' build_message_node (format, arg1, arg2)       char *format;      void *arg1, *arg2;p {n
   NODE *node;p     message_buffer_index = 0;(,   build_message_buffer (format, arg1, arg2);  #   node = message_buffer_to_node ();    return (node); }l  ; /* Convert the contents of the message buffer to a node. */  NODE * message_buffer_to_node ()  {u
   NODE *node;   )   node = (NODE *)xmalloc (sizeof (NODE));     node->filename = (char *)NULL;   node->parent = (char *)NULL;    node->nodename = (char *)NULL;   node->flags = 0;  7   /* Make sure that this buffer ends with a newline. */s.   node->nodelen = 1 + strlen (message_buffer);7   node->contents = (char *)xmalloc (1 + node->nodelen);n*   strcpy (node->contents, message_buffer);+   node->contents[node->nodelen - 1] = '\n';e'   node->contents[node->nodelen] = '\0';s   return (node); }i  > /* Useful functions can be called from outside of window.c. */ void initialize_message_buffer () {    message_buffer_index = 0;  }   H /* Print FORMAT with ARG1,2 to the end of the current message buffer. */ void- printf_to_message_buffer (format, arg1, arg2)"      char *format;      void *arg1, *arg2;t {o,   build_message_buffer (format, arg1, arg2); }l  E /* Return the current horizontal position of the "cursor" on the mostN*    recently output message buffer line. */ intg" message_buffer_length_this_line () {    register int i;n     if (!message_buffer_index)     return (0);u  J   for (i = message_buffer_index; i && message_buffer[i - 1] != '\n'; i--);  0   return (string_width (message_buffer + i, 0)); }a  9 /* Pad STRING to COUNT characters by inserting blanks. */s intd pad_to (count, string)      int count;e      char *string; {o   register int i;o     i = strlen (string);     if (i >= count)      string[i++] = ' ';   else     {l       while (i < count)n 	string[i++] = ' ';      }    string[i] = '\0';i  
   return (i);  }r