/*****************************************************************************/
/*
                              CGIplusDemo.c

Simple program that can be used in both standard CGI and CGIplus environments.
The standard CGI section merely echoes the remote host name (lacking any
mechanism to list all variables ... apart perhaps from a SHOW SYMBOL WWW_*).
The CGIplus section uses two methods to access the CGI variables. One just
reads CGIPLUSIN checking each line for the required variable (and displaying
them as it goes). The other uses a more sophisticated function that processes
the CGIplus variables and the allows queries on variable names, etc. This
function is presented as a template in CGIplusSkel.c


21-JUN-98  MGD  v1.1.0, revision of CgiVar()
08-JUN-97  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CGIPLUSDEMO AXP-1.1.0";
#else
   char SoftwareID [] = "CGIPLUSDEMO VAX-1.1.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

char  Utility [] = "CGIPLUSDEMO";

int  UsageCounter,
     Debug,
     IsCgiPlus;

char  *CgiPlusEofPtr = NULL;

char  FirstUsedDateTime [32],
      NowDateTime [32];

unsigned long  UnixTime;
struct tm  *UnixTmPtr;

/* required prototypes */
char* CgiVar (char*);

/*****************************************************************************/
/*
*/

main ()

{
   /*********/
   /* begin */
   /*********/

   if (getenv ("CGIPLUSDEMO$DBUG") != NULL) Debug = 1;

   if (Debug) system ("show sym *");

   time (&UnixTime);
   UnixTmPtr = localtime (&UnixTime);
   if (!strftime (FirstUsedDateTime, sizeof(FirstUsedDateTime),
                  "%a, %d %b %Y %T", UnixTmPtr))
      strcpy (FirstUsedDateTime, "[error]");

   /* demonstrate the ease with which it may operate in both environments */
   IsCgiPlus = ((CgiPlusEofPtr = getenv("CGIPLUSEOF")) != NULL);

   do {

      time (&UnixTime);
      UnixTmPtr = localtime (&UnixTime);
      if (!strftime (NowDateTime, sizeof(NowDateTime),
                     "%a, %d %b %Y %T", UnixTmPtr))
         strcpy (NowDateTime, "[error]");

      if (IsCgiPlus)
         CgiPlusDemo ();
      else
         CgiStandardDemo ();

   } while (IsCgiPlus);

   exit (1);
}

/*****************************************************************************/
/*
Using standard CGI.  Get CGI variable data using getenv(), not using the
function CgiVar() illustrated below.
*/

CgiStandardDemo ()

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiStandardDemo()\n");

   fprintf (stdout,
"Content-Type: text/plain\n\
Expires: Thu, 01 Jan 1970 00:00:01 GMT\n\
\n\
Number of times used: %d (standard CGI)\n\
First used: %s\n\
Now: %s\n\
\n",
      ++UsageCounter, FirstUsedDateTime, NowDateTime);

   if ((cptr = getenv("WWW_REMOTE_ADDR")) != NULL)
      fprintf (stdout, "Remote-Addr: %s\n", cptr);
   else
      fprintf (stdout, "Remote-Addr: NOT FOUND!\n");
}

/*****************************************************************************/
/*
Alternate between the two methods of obtaining CGI variables in the CGIplus
environment.  First, using the function CgiVar(), which allows specific
variable names to be queried.  Second, using a simple read of <stdin>,
checking for the variable names required.
*/

CgiPlusDemo ()

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiPlusDemo()\n");

   if (UsageCounter & 0x01)
   {
      /* process the variables using a simple read loop */
      CgiPlusReader ();
   }
   else
   {
      /* wait for the next request */
      CgiVar ("");

   fprintf (stdout,
"Content-Type: text/plain\n\
Expires: Thu, 01 Jan 1970 00:00:01 GMT\n\
\n\
Number of times used: %d (CGIplus)\n\
First used: %s\n\
Now: %s\n\
Using: CgiVar()\n\
\n",
      ++UsageCounter, FirstUsedDateTime, NowDateTime);

      /* list all CGIplus variables names and values one-by-one */
      while (*(cptr = CgiVar ("*")))
      {
         if (!strncmp (cptr, "WWW_REMOTE_HOST=", 16))
            fprintf (stdout, "Remote-Host: %s\n", cptr+16);
         else
            fprintf (stdout, "|%s|\n", cptr);
      }

      /* query on the supplied variable name */
      fprintf (stdout, "\n");
      if (*(cptr = CgiVar ("WWW_REMOTE_ADDR")))
         fprintf (stdout, "Remote-Addr: %s\n", cptr);
      else
         fprintf (stdout, "Remote-Addr: NOT FOUND!\n");

      /* illustrate variable query when not found */
      if (*(cptr = CgiVar ("WWW_BOGUS_VARIABLE")))
         fprintf (stdout, "Bogus-Variable: %s\n", cptr);
      else
         fprintf (stdout, "Bogus-Variable: NOT FOUND!\n");
   }

   /* record-oriented <stdout>, no need to flush, write end-of-output line */
   fprintf (stdout, "%s\n", CgiPlusEofPtr);
}

/*****************************************************************************/
/*
CGIplus variable access method 1 (simple).

Use a simple loop to read lines from standard input.  Detect specific CGI
variables by checking the leading section of these lines for the required
variable name.  The value may then be extracted from the remaining section.
*/

CgiPlusReader ()

{
#  define CGIPLUS_LINE_SIZE 1024

   register char  *cptr;

   int  LineCount = 0;
   char  Line [CGIPLUS_LINE_SIZE];
   FILE  *CgiPlusIn;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiPlusReader()\n");

   if ((CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")) == NULL)
      exit (vaxc$errno);

   /* wait for request, discard initial record */
   if (fgets (Line, sizeof(Line), CgiPlusIn) == NULL) exit (vaxc$errno);

   fprintf (stdout,
"Content-Type: text/plain\n\
\n\
Number of times used: %d (CGIplus)\n\
First used: %s\n\
Now: %s\n\
Using: CgiPlusReader()\n\
\n",
      ++UsageCounter, FirstUsedDateTime, NowDateTime);

   for (;;)
   {
      if (fgets (Line, sizeof(Line), CgiPlusIn) == NULL) exit (vaxc$errno);
      /* first empty line signals the end of CGIplus variables */
      if (Line[0] == '\n') break;
      /* remove the trailing newline */
      for (cptr = Line; *cptr && *cptr != '\n'; cptr++);
      *cptr = '\0';

      /* select only the variable(s) we are interested in */
      if (!strncmp (Line, "WWW_REMOTE_HOST=", 16))
         fprintf (stdout, "Remote-Host: %s\n", Line+16);
      else
         fprintf (stdout, "|%s|\n", Line);
   }

   fclose (CgiPlusIn);
}

/*****************************************************************************/
/*
This function provides a simple, uniform mechanism for a C Language script to
access CGI variable values transparently in both the standard CGI and the
CGIplus environments.  Requires the global storage 'CgiPlusEofPtr' to be
non-NULL to indicate a CGIplus environment.

It is completely self-contained. Simply call the function with the CGI variable
name value to be accessed and if that CGI variable exists a pointer to the
value will be returned.  Non-existant variables return CGIPLUS_NONE (see
below).  The values pointed to should NOT be modified in any way (copy the
value if this is required). 

Supplying an empty variable name (i.e. "") will cause the function to block
until the arrival of a request, upon which it just returns a CGIPLUS_NONE
pointer, allowing start-of-request synchronization.  This will also release
allocated memory and reset in preparation for reading the next request, an
alternative to calling with a NULL parameter at the end of request processing
(see below).

For the CGIplus environment supplying a wilcard variable name (i.e. "*")
returns each CGIplus variable string (line from CGIPLUSIN) in turn, with no
more being indicated by a CGIPLUS_NONE pointer.

At the end of a CGIplus request call the function with a NULL parameter to
release allocated memory and reset ready for processing the next request's
variables.  It is not necessary to do this if calling with an empty parameter
to synchronize on the start of a request (see above).

The CGIplus functionality works by initially reading all lines from
CGIPLUSIN until the terminating blank line is encountered. These lines are
stored as a series of contiguous null-terminated strings, terminated by an
empty string, in allocated memory. These strings are searched for the required
variable name and a pointer to its value returned if found.

The value CGIPLUS_NONE shold be set according to the requirements of the
application the function is included within.  If calling functions can handle
NULL pointers being returned then set the value to NULL.  If non-NULL pointers
are prefered then set the value to an empty string.

21-JUN-98  MGD  version 2 of this function attempts to increase efficiency by
                reducing the number of realloc()s required and introducing the
                length of each variable improving search behaviour
01-JUL-97  MGD  version 1
*/

char* CgiVar (char *VarName)

{
#ifndef CGIPLUS_DEBUG
#   define CGIPLUS_DEBUG 0
#endif
/** #  define CGIPLUS_NONE NULL **/
#define CGIPLUS_NONE ""
#define CGIPLUS_LINE_SIZE 1024
#define CGIPLUS_CHUNK 2048

   static int  VarPtrSize = 0;
   static char  *NextPtr = NULL,
                *VarPtr = NULL;

   register char  *cptr, *sptr, *zptr;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

#if CGIPLUS_DEBUG
   fprintf (stdout, "CgiVar() |%s|\n", VarName);
#endif

   if (VarName == NULL || !VarName[0])
   {
      /* end of request, release memory and reinitialize */
      if (VarPtr != NULL)
      {
         free (VarPtr);
         NextPtr = VarPtr = NULL;
      }
      if (VarName == NULL) return (CGIPLUS_NONE);
   }

   if (CgiPlusEofPtr == NULL)
   {
      /* standard CGI environment, get variable from environment */
      if (!VarName[0]) return (CGIPLUS_NONE);
      if ((cptr = getenv(VarName)) == NULL)
         return (CGIPLUS_NONE);
      else
         return (cptr);
   }

   if (VarPtr == NULL)
   {
      /* read lines containing CGIplus variables from <CGIPLUSIN> */
      int  CurrentVarLength;
      char  Line [CGIPLUS_LINE_SIZE];
      char  *LengthPtr;
      FILE  *CgiPlusIn;
                                                 
      CurrentVarLength = VarPtrSize = 0;
      zptr = NULL;
      if ((CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")) == NULL)
         exit (vaxc$errno);
      while (fgets (Line, sizeof(Line), CgiPlusIn) != NULL)
      {
         /* ignore any non-alphabetic line */
         if (Line[0] != '\n' && !isalpha(Line[0])) continue;
         /* point to the next name=value storage in the variable buffer */
         LengthPtr = VarPtr + CurrentVarLength;
         sptr = LengthPtr + sizeof(unsigned short);
         for (cptr = Line; *cptr; *sptr++ = *cptr++)
         {
            if (sptr < zptr) continue;
            /* out of storage (or first variable) get more (or some) memory */
            if ((VarPtr = realloc (VarPtr, VarPtrSize+CGIPLUS_CHUNK)) == NULL)
               exit (vaxc$errno);
            VarPtrSize += CGIPLUS_CHUNK;
            /* recalculate pointers */
            LengthPtr = VarPtr + CurrentVarLength;
            sptr = LengthPtr + sizeof(unsigned short) + (cptr - Line);
            zptr = VarPtr + VarPtrSize;
         }
         /* remove the trailing newline */
         *--sptr = '\0';
         /* insert the length of this name=value pair immediately before it */
         *((unsigned short*)LengthPtr) = Length = cptr - Line;
         /* adjust the value of the current variable storage space used */
         CurrentVarLength += Length + sizeof(unsigned short);
         /* first empty line signals the end of CGIplus variables */
         if (Line[0] == '\n') break;
      }
      fclose (CgiPlusIn);
      if (VarPtr == NULL) return (CGIPLUS_NONE);

#if CGIPLUS_DEBUG
      fprintf (stdout, "VarPtr: %d VarPtrSize: %d CurrentVarLength: %d\n",
               VarPtr, VarPtrSize, CurrentVarLength);
      sptr = VarPtr;
      for (sptr = VarPtr;
           Length = *((unsigned short*)sptr);
           sptr += sizeof(unsigned short) + Length)
         fprintf (stdout, "%3d |%s|\n", Length, sptr+sizeof(unsigned short));
#endif
   }

   /* if just waiting for next request return now it's arrived */
   if (!VarName[0]) return (CGIPLUS_NONE);

   if (VarName[0] == '*')
   {
      /* return each stored line one at a time */
      if (NextPtr == NULL) NextPtr = VarPtr;
      if ((Length = *((unsigned short*)(sptr = NextPtr))) == 0)
      {
         NextPtr = NULL;
#if CGIPLUS_DEBUG
         fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
         return (CGIPLUS_NONE);
      }
      sptr += sizeof(unsigned short);
      NextPtr = sptr + Length;
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr);
#endif
      return (sptr);
   }

   /* search CGIplus variable strings for required variable */
   for (sptr = VarPtr;
        Length = *((unsigned short*)sptr);
        sptr = zptr + Length)
   {
      /* simple comparison between supplied and in-string variable names */
      zptr = sptr = sptr + sizeof(unsigned short);
      cptr = VarName;
      while (*cptr && *sptr && *sptr != '=')
      {
         if (toupper(*cptr) != toupper(*sptr)) break;
         cptr++;
         sptr++;
      }
      if (*cptr || *sptr != '=') continue;
      /* found, return a pointer to the value */
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr+1);
#endif
      return (sptr+1);
   }
   /* not found */
#if CGIPLUS_DEBUG
   fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
   return (CGIPLUS_NONE);
}

/*****************************************************************************/

