/*****************************************************************************/
/*
                              CGIplusSkel.c

A skeletal program serving as a template for CGI/CGIplus-capable scripts.

Uses a sophisticated single function for accessing CGI variables. The function
differentiates between standard and CGIplus environments and changes behaviour
accordingly.  The function also serves to block waiting for subsequent CGIplus
requests.


21-JUN-98  MGD  v1.1.0, revision of CgiVar()
08-JUN-97  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

int  IsCgiPlus = 0;
char  *CgiPlusEofPtr = NULL;

/* required prototypes */
char* CgiVar (char*);

/*****************************************************************************/
/*
*/

main ()

{
   int  Count = 0;
   char  *RemoteHostPtr;

   IsCgiPlus = ((CgiPlusEofPtr = getenv("CGIPLUSEOF")) != NULL);

   do {

      /* with CGIplus this call will block waiting for the next request */
      CgiVar ("");

      RemoteHostPtr = CgiVar ("WWW_REMOTE_HOST");

      fprintf (stdout,
"Content-Type: text/html\n\
Expires: Thu, 01 Jan 1970 00:00:01 GMT\n\
\n\
%d. <B>Hello world</B> ... well the \
<FONT SIZE=+1>&nbsp;<TT>%s</TT>&nbsp;</FONT> part of it anyway!\n",
               ++Count, RemoteHostPtr);

      /* record-oriented <stdout>, no need to flush, write end-of-output line */
      if (IsCgiPlus) fprintf (stdout, "%s\n", CgiPlusEofPtr);

   } while (IsCgiPlus);

   exit (1);
}

/*****************************************************************************/
/*
This function provides a simple, uniform mechanism for a C Language script to
access CGI variable values transparently in both the standard CGI and the
CGIplus environments.  Requires the global storage 'CgiPlusEofPtr' to be
non-NULL to indicate a CGIplus environment.

It is completely self-contained. Simply call the function with the CGI variable
name value to be accessed and if that CGI variable exists a pointer to the
value will be returned.  Non-existant variables return CGIPLUS_NONE (see
below).  The values pointed to should NOT be modified in any way (copy the
value if this is required). 

Supplying an empty variable name (i.e. "") will cause the function to block
until the arrival of a request, upon which it just returns a CGIPLUS_NONE
pointer, allowing start-of-request synchronization.  This will also release
allocated memory and reset in preparation for reading the next request, an
alternative to calling with a NULL parameter at the end of request processing
(see below).

For the CGIplus environment supplying a wilcard variable name (i.e. "*")
returns each CGIplus variable string (line from CGIPLUSIN) in turn, with no
more being indicated by a CGIPLUS_NONE pointer.

At the end of a CGIplus request call the function with a NULL parameter to
release allocated memory and reset ready for processing the next request's
variables.  It is not necessary to do this if calling with an empty parameter
to synchronize on the start of a request (see above).

The CGIplus functionality works by initially reading all lines from
CGIPLUSIN until the terminating blank line is encountered. These lines are
stored as a series of contiguous null-terminated strings, terminated by an
empty string, in allocated memory. These strings are searched for the required
variable name and a pointer to its value returned if found.

The value CGIPLUS_NONE shold be set according to the requirements of the
application the function is included within.  If calling functions can handle
NULL pointers being returned then set the value to NULL.  If non-NULL pointers
are prefered then set the value to an empty string.

21-JUN-98  MGD  version 2 of this function attempts to increase efficiency by
                reducing the number of realloc()s required and introducing the
                length of each variable improving search behaviour
01-JUL-97  MGD  version 1
*/

char* CgiVar (char *VarName)

{
#ifndef CGIPLUS_DEBUG
#   define CGIPLUS_DEBUG 0
#endif
/** #  define CGIPLUS_NONE NULL **/
#define CGIPLUS_NONE ""
#define CGIPLUS_LINE_SIZE 1024
#define CGIPLUS_CHUNK 2048

   static int  VarPtrSize = 0;
   static char  *NextPtr = NULL,
                *VarPtr = NULL;

   register char  *cptr, *sptr, *zptr;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

#if CGIPLUS_DEBUG
   fprintf (stdout, "CgiVar() |%s|\n", VarName);
#endif

   if (VarName == NULL || !VarName[0])
   {
      /* end of request, release memory and reinitialize */
      if (VarPtr != NULL)
      {
         free (VarPtr);
         NextPtr = VarPtr = NULL;
      }
      if (VarName == NULL) return (CGIPLUS_NONE);
   }

   if (CgiPlusEofPtr == NULL)
   {
      /* standard CGI environment, get variable from environment */
      if (!VarName[0]) return (CGIPLUS_NONE);
      if ((cptr = getenv(VarName)) == NULL)
         return (CGIPLUS_NONE);
      else
         return (cptr);
   }

   if (VarPtr == NULL)
   {
      /* read lines containing CGIplus variables from <CGIPLUSIN> */
      int  CurrentVarLength;
      char  Line [CGIPLUS_LINE_SIZE];
      char  *LengthPtr;
      FILE  *CgiPlusIn;
                                                 
      CurrentVarLength = VarPtrSize = 0;
      zptr = NULL;
      if ((CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")) == NULL)
         exit (vaxc$errno);
      while (fgets (Line, sizeof(Line), CgiPlusIn) != NULL)
      {
         /* ignore any non-alphabetic line */
         if (Line[0] != '\n' && !isalpha(Line[0])) continue;
         /* point to the next name=value storage in the variable buffer */
         LengthPtr = VarPtr + CurrentVarLength;
         sptr = LengthPtr + sizeof(unsigned short);
         for (cptr = Line; *cptr; *sptr++ = *cptr++)
         {
            if (sptr < zptr) continue;
            /* out of storage (or first variable) get more (or some) memory */
            if ((VarPtr = realloc (VarPtr, VarPtrSize+CGIPLUS_CHUNK)) == NULL)
               exit (vaxc$errno);
            VarPtrSize += CGIPLUS_CHUNK;
            /* recalculate pointers */
            LengthPtr = VarPtr + CurrentVarLength;
            sptr = LengthPtr + sizeof(unsigned short) + (cptr - Line);
            zptr = VarPtr + VarPtrSize;
         }
         /* remove the trailing newline */
         *--sptr = '\0';
         /* insert the length of this name=value pair immediately before it */
         *((unsigned short*)LengthPtr) = Length = cptr - Line;
         /* adjust the value of the current variable storage space used */
         CurrentVarLength += Length + sizeof(unsigned short);
         /* first empty line signals the end of CGIplus variables */
         if (Line[0] == '\n') break;
      }
      fclose (CgiPlusIn);
      if (VarPtr == NULL) return (CGIPLUS_NONE);

#if CGIPLUS_DEBUG
      fprintf (stdout, "VarPtr: %d VarPtrSize: %d CurrentVarLength: %d\n",
               VarPtr, VarPtrSize, CurrentVarLength);
      sptr = VarPtr;
      for (sptr = VarPtr;
           Length = *((unsigned short*)sptr);
           sptr += sizeof(unsigned short) + Length)
         fprintf (stdout, "%3d |%s|\n", Length, sptr+sizeof(unsigned short));
#endif
   }

   /* if just waiting for next request return now it's arrived */
   if (!VarName[0]) return (CGIPLUS_NONE);

   if (VarName[0] == '*')
   {
      /* return each stored line one at a time */
      if (NextPtr == NULL) NextPtr = VarPtr;
      if ((Length = *((unsigned short*)(sptr = NextPtr))) == 0)
      {
         NextPtr = NULL;
#if CGIPLUS_DEBUG
         fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
         return (CGIPLUS_NONE);
      }
      sptr += sizeof(unsigned short);
      NextPtr = sptr + Length;
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr);
#endif
      return (sptr);
   }

   /* search CGIplus variable strings for required variable */
   for (sptr = VarPtr;
        Length = *((unsigned short*)sptr);
        sptr = zptr + Length)
   {
      /* simple comparison between supplied and in-string variable names */
      zptr = sptr = sptr + sizeof(unsigned short);
      cptr = VarName;
      while (*cptr && *sptr && *sptr != '=')
      {
         if (toupper(*cptr) != toupper(*sptr)) break;
         cptr++;
         sptr++;
      }
      if (*cptr || *sptr != '=') continue;
      /* found, return a pointer to the value */
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr+1);
#endif
      return (sptr+1);
   }
   /* not found */
#if CGIPLUS_DEBUG
   fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
   return (CGIPLUS_NONE);
}

/*****************************************************************************/

