/*****************************************************************************/
/*
                                 Conan.c


CGI-compliant script providing access to VMS help and text libraries.
Can be used in the CGI-plus environment.

"Conan The Librarian"

Page layout and colouration may be specified via the appropriate command-line
qualifiers (or corresponding logical/symbol name). Defaults apply for any not
specified.  See "Qualifiers" section below, and also about the logical name or
symbol "CONAN$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"

Don't like explicitly setting a browser's colours?  A colour may be disabled by
setting it to empty.  The following example disables all colours.

  /PBGCOLOR/PBBGCOLOR/PHBGCOLOR/PHTEXT/PLINK/PTEXT/PVLINK

The script can format a page in either of two layouts.

  1. Tables are used to create a coloured header and button bar (DEFAULT).
     Default colours are white page with grey heading and button outlines.
  2. Textual header, horizontal rules and a textual button bar.
     No default colours.

Select other than the default using the following:

  /PLAYOUT=2

Local information may be included in the header. For layout 1 this should be
integrated with the <TABLE> formatted header and to the right of the header
information. Text, an image logo, just about anything could be included. This
is a example of providing a textual form of a local logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><FONT SIZE=-1 COLOR=#ffffff>"-
  "<B>WASD<HyperText Services</B></FONT></TD>"

This is an example of providing a local graphical logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><IMG SRC=/local/logo.gif></TD>"

Such local information with layout 2 is included immediately before the header
information at the top of the page.

Button labels are customizable (potentially to non-English language). They
comprise a label, equate symbol and URL-style path suitable for creating a
link. Multiple buttons are separated using the semicolon. Note that any such
button customization must provide escaped HTML-forbidden characters in the
button label and URI-forbidden characters in the path! The backslash
character, "\", escapes characters, including the button-delimitting "=" and
";". There are defaults, see DEFAULT_BUTTONS.

Here is an example of changing the button labels:

  /BUTTON="Close;Help=/conan/-/conanhelp.html"

Always have the equivalent of "Close" for the first button!  Additional buttons
may be created by adding "label=path;" elements to the button string.  In this
way an additional information page could be referenced as follows:

  /BUTTON="Close;Help=/conan/-/conanhelp.html;Other VMS=/vms/"

DIFFICULTY FITTING ALL THESE QUALIFIERS INTO A COMMAND LINE OR LOGICAL?
Use an existing, or create a new, DCL wrapper procedure for the script (same
name and directory) and build up a DCL symbol to provide the information. Up
to 1024 characters can be included in this way. For example:

  $ CONAN$PARAM = "/BUTTON=""Close;Help=/conan/-/conanhelp.html"""
  $ CONAN$PARAM = CONAN$PARAM + "/PBGCOLOR/PLINK/PVLINK"
  $ CONAN$PARAM = CONAN$PARAM + "/PHLOCAL=""<TD VALIGN=top>...</TD>"""
  $ RUN HT_EXE:CONAN

AND REMEMBER ... Conan can be used as a CGIplus script.  This may require the
script to be purged from the server before new startup parameters come into
effect.  Use a command like the following:

  HTTPD /DO=DCL=PURGE


HTTP SERVER CONFIGURATION
-------------------------
This script uses URL-style paths for accessing libraries. The HTTP server must
therefore provide a mapping for all directories that libraries will be
accessed. Something like:

  pass /sys$common/syshlp/* /sys$common/syshlp/*

If auto-scripting is available (as with WASD) these should also be a
configuration directive providing this (example from WASD):

  .HLB  application/x-script  /Conan  VMS help library


OSU ENVIRONMENT
---------------
Script responses are returned in OSU "raw" mode; the script taking care of the
full response header and correctly carriage-controlled data stream, text or
binary!!  The script standard output stream is reopened in binary mode (no
translation of '\n') with a maximum record size set to 4096 bytes (failure to
set this results in the OSU server reporting %SYSTEM-F-DATAOVERUN errors).  The
binary output is enclosed by suitably fflushed() "<DNETRAW>" and "</DNETRAW>"
control tags (the flush make them individual records as required by OSU).
If in debug the script output is placed into "text" mode.


HELP LIBRARIES
--------------

Help library modules have a reasonably complex internal structure described in 
the "Utility Routines Manual" and the "Command Definition, Librarian, and 
Message Utilities Manual".  The structure will not be described here. 

KeyWord[1]...KeyWord[10] represent the help keywords 1...10 that are used as 
keys into help information.


LOGICAL NAMES
-------------
CONAN$DBUG             turns on all "if (Debug)" statements
CONAN$PARAM            equivalent to (overrides) the command line
                       parameters/qualifiers (define as a system-wide logical)
HTTPD$GMT              timezone offset from Greenwich Mean Time
                       (e.g. "+09:30" and only needed if DTSS is not in use)


QUALIFIERS
----------
/BUTTONS=       string containing button labels/paths
/DBUG           turns on all "if (Debug)" statements
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PBBORDER=      width of button border
/PHBGCOLOR=     heading background color
/PHBORDER=      width of heading and button-bar border
/PHLOCAL=       local information to be included in header
/PHTEXT=        heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour


BUILD DETAILS
-------------
See BUILD_CONAN.COM procedure.


COPYRIGHT
---------
Copyright (c) 1996-1998 Mark G.Daniel
This program, comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under the conditions of the GNU GENERAL PUBLIC LICENSE, version 2.


VERSION HISTORY (update SoftwareID as well!)
---------------
08-AUG-98  MGD  v3.0.3, OSU output processing reworked,
                        bugfix; TimeSetTimezone() 'Seconds' unsigned to signed
06-AUG-98  MGD  v3.0.2, accomodation for OSU ... reduce HTTP response
                        header carriage control from <CR><LF> to <LF> only
                        (OSU/IE4 combination problematic)
01-AUG-98  MGD  v3.0.1, suppress table background colours if empty,
                        accomodations for OSU environment
29-APR-98  MGD  v3.0.0, remove need for WASD MapUrl() functions
                        (making it a GENERIC CGI script!!),
                        some tidying up and cosmetic changes
26-FEB-98  MGD  v2.9.2, added HTML table around title in help library,
                        TimeSetGmt() modified to be in line with HTTPd
19-AUG-97  MGD  v2.9.1, MapUrl() to MapUrl_Map() for conditional mapping
20-JUL-97  MGD  v2.9.0, added /BODY= qualifier (changed background colour)
20-JUN-97  MGD  v2.8.0, now compatible with standard and CGIplus environments,
                        "Pragma: no-cache" now overrides "If-Modified-Since:"
14-MAR-97  MGD  v2.7.1, bugfix; absorbtion of leading white-space
07-AUG-96  MGD  v2.7.0, allow help topic to be specified in path, for example
                        "/conan/run/process" or "/help/delete/file"
16-MAY-96  MGD  v2.6.2, changed XBMs to tranparaent GIFs
23-FEB-96  MGD  v2.6.1, bugfix, after modifying the HTTPD$GMT format for the
                        HTTPd server I had forgotten about some scripts
12-OCT-95  MGD  v2.6.0, added 'Referer:', 'Last-Modified:'/'If-Modified-Since:'
18-AUG-95  MGD  v2.5.0, added "search" form at appropriate places
20-JUN-95  MGD  v2.4.0, added "explode" feature, listing all help levels below
                        the key specified in a single page
24-MAY-95  MGD  v2.3.0, minor changes for AXP compatibility
31-MAR-95  MGD  v2.2.0, ongoing maintenance
02-FEB-95  MGD  v2.1.0, add "Conan the Librarian" icon
13-DEC-94  MGD  v2.0.0, made CGI compliant
09-JUL-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CONAN AXP-3.0.3";
#else
   char SoftwareID [] = "CONAN VAX-3.0.3";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <lbrdef.h>
#include <lhidef.h>
#include <lnmdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>
#include <syidef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define DEFAULT_LIBRARY "SYS$HELP:HELPLIB.HLB"

#define DEFAULT_BUTTONS "Close;Help=/conan/-/conanhelp.html"

/* this macro just plugs in some script-specific code into ButtonBar() */
#define SCRIPT_SPECIFIC_BUTTON_CODE \
   ButtonInternal[0] = CgiFormRefererPtr;

 #define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

char  *PageScheme [16];

char  Utility [] = "CONAN";

boolean  Debug,
         ErrorReported,
         ExplodeHelp,
         DoHeaderInformation,
         DoListLibraries,
         DoLibraryMenu,
         DoSearch,
         DoSearchStatistics,
         HttpHasBeenOutput,
         KeyNameWildcard,
         OsuEnvironment,
         TimeAheadOfGmt;
         
int  SearchHitCount,
     KeyWordCount,
     RecordsSearched,
     SearchStringLength;

unsigned long  LibraryIndex,
               IndexNumber;

unsigned long  IfModifiedSinceBinaryTime [2],
               TimeGmtDeltaBinary [2];

char  *ButtonPtr = DEFAULT_BUTTONS,
      *CgiHttpIfModifiedSincePtr,
      *CgiHttpPragmaPtr,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiRequestMethodPtr,
      *CgiScriptNamePtr,
      *CgiServerSoftwarePtr,
      *CgiFormDoPtr,
      *CgiFormExplodePtr,
      *CgiFormKeyPtr,
      *CgiFormRefererPtr,
      *CgiFormTitlePtr,
      *CgiFormSearchPtr;

char  GmDateTime [32],
      HtmlFormTitle [256],
      HtmlLibraryTitle [256],
      HtmlReferer [512],
      HtmlSearchString [512],
      LastModifiedGmDateTime [32],
      LibraryDirPath [256],
      LibraryName [256],
      LibraryPathInfo [256],
      LibrarySpec [256],
      LibraryTitle [256],
      TimeGmtString [32],
      TimeGmtVmsString [32],
      UnixDateTime [64],
      UriFormTitle [256],
      UriReferer [512];

/* element 0 is not used, keywords number from 1 to 10 */
char  KeyWord [11][256],
      KeyWordHtml [11][256],
      KeyName [11][256],
      KeyNameHtml [11][256];
      
struct lhidef  LibraryHeader;

struct {
   unsigned long  lo32;
   unsigned long  hi32;
} ModuleRFA;

/* CGIplus specific */
int  IsCgiPlus,
     CgiPlusUsageCount;
char  *CgiPlusEofPtr = NULL;
char  *CgiTypePtr;
char* CgiVar (char*);

/* required function prototypes */
char* SearchText (char*, char*, boolean);
int ListHelpModule (struct dsc$descriptor_s*, void*);
int ListTextModule (struct dsc$descriptor_s*, void*);
char* MungeUnderScores (char*);
char* FormatRequestKeys (boolean, int);
int OpenHelpModule (struct dsc$descriptor_s*, void*);
int OpenTextModule (struct dsc$descriptor_s*, void*);
int SearchHelpModule (struct dsc$descriptor_s*, void*);
int SearchTextModule (struct dsc$descriptor_s*, void*);
AtOsuExit ();

/*****************************************************************************/
/*
*/

main ()

{
   register int  idx;
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (getenv ("WWWEXEC_RUNDOWN_STRING") != NULL)
   {
      OsuEnvironment = true;
      atexit (&AtOsuExit);
   }

   if (getenv ("CONAN$DBUG") != NULL)
   {
      Debug = true;
      if (OsuEnvironment) fprintf (stdout, "<DNETTEXT>\n200 Success\n");
   }

   GetParameters ();

   if (Debug)
      system ("show sym *");
   else
   {
      if (OsuEnvironment)
      {
         /* reopen so that '\r' and '\n' are not filtered, maximum record size */
         if ((stdout =
             freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin", "mrs=4096"))
             == NULL)
            exit (vaxc$errno);

         fprintf (stdout, "<DNETRAW>");
         fflush (stdout);
      }
      else
      {
         /* reopen so that '\r' and '\n' are not filtered */
         if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
            exit (vaxc$errno);
      }
   }

   /*************/
   /* librarian */
   /*************/

   SetPageScheme ();

   if (OsuEnvironment)
      CgiTypePtr = "OSU";
   else
   if (IsCgiPlus = ((CgiPlusEofPtr = getenv("CGIPLUSEOF")) != NULL))
      CgiTypePtr = "plus";
   else
      CgiTypePtr = "standard";

   do {

      /* with CGIplus this call will block waiting for the next request */
      CgiVar ("");

      CgiPlusUsageCount++;

      ConanTheLibrarian ();

      if (IsCgiPlus)
      {
         /* binary output stream, flush before and after EOF record */
         fflush (stdout);
         fputs (CgiPlusEofPtr, stdout);
         fflush (stdout);
      }

   } while (IsCgiPlus);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Script executing in OSU environment exits.
*/

AtOsuExit ()

{
   if (Debug)
      fprintf (stdout, "</DNETTEXT>\n");
   else
   {
      fflush (stdout);
      fprintf (stdout, "</DNETRAW>");
      fflush (stdout);
   }
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration symbol or logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/
   if ((clptr = getenv ("CONAN$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

SetPageScheme ()

{
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Provides a divider for top and bottom of the content of the page. This can be
a coloured bar (using <TABLE>) or a horizontal rule depending on the page
layout. "Buttons" providing script-internal and/or additional user-specified
links ('ButtonPtr' string) can be placed with(in) this bar. All button labels
are derived from 'ButtonPtr', with script-internal buttons using link-paths
set up via 'ButtonInternal[]' array, or any user-specified path depending on
requirement. An empty path (i.e. PathPtr[0] == '\0') obviously does not have a
link created, it just displays the button label. For a button-bar at the top
of the document use 1, bottom of the document use 2, and for just a bar with
no buttons at all use 0.
*/

ButtonBar (int Top1Bottom2)

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ButtonBar() %d\n", Top1Bottom2);

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';
         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (Top1Bottom2)
   {
      /***********************************/
      /* set up script-specified buttons */
      /***********************************/

      SCRIPT_SPECIFIC_BUTTON_CODE
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      if (Top1Bottom2 == 2 || !Top1Bottom2)
      {
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
         if (!Top1Bottom2) return;
      }

      fprintf (stdout, "<FONT SIZE=-1><NOBR>\n");
      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (idx) fprintf (stdout, "&nbsp;");
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }
      fprintf (stdout, "</NOBR></FONT>\n");

      if (Top1Bottom2 == 1)
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR]);

      if (ButtonCount == 0 || !Top1Bottom2)
         fprintf (stdout, "&nbsp;");
      else
      {
         fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n",
            PageScheme[PS_BUTTONBORDER]);

         for (idx = 0; idx < ButtonCount; idx++)
         {
            if (ButtonInternal[idx] == NULL)
               PathPtr = ButtonPath[idx];
            else
               PathPtr = ButtonInternal[idx];
            if (PathPtr[0])
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
            else
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
         }

         fprintf (stdout, "</TR></TABLE>\n");
      }

      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n");
   }
}

/*****************************************************************************/
/*
*/

ConanTheLibrarian ()

{
   register int  idx;
   register char  *cptr, *sptr;
   unsigned long  UnixTime;
   struct tm  *UnixTmPtr;

   int  status;

   /*********/
   /* begin */
   /*********/

   time (&UnixTime);
   UnixTmPtr = localtime (&UnixTime);
   if (!strftime (UnixDateTime, sizeof(UnixDateTime),
                  "%a, %d %b %Y %T", UnixTmPtr))
      strcpy (UnixDateTime, "[error]");
   if (Debug) fprintf (stdout, "UnixDateTime |%s|\n", UnixDateTime);

   /* initialize anything specifically required for CGIplus */
   DoHeaderInformation = DoListLibraries = DoLibraryMenu = DoSearch =
      KeyNameWildcard = ExplodeHelp = HttpHasBeenOutput =
      ErrorReported = false;
   DoSearchStatistics = true;
   for (idx = 0; idx < 11; idx++)
      KeyWord[idx][0] = KeyWordHtml[idx][0] = 
         KeyName[idx][0] = KeyNameHtml[idx][0] = '\0';

   CgiServerSoftwarePtr = CgiVar ("WWW_SERVER_SOFTWARE");
   CgiRequestMethodPtr = CgiVar ("WWW_REQUEST_METHOD");
   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethodPtr != 0x00544547)
   {
      fprintf (stdout,
"HTTP/1.0 501 Not Implemented\r\n\
Server: %s\r\n\
\r\n",
         CgiServerSoftwarePtr);
      return;
   }

   CgiPathInfoPtr = CgiVar ("WWW_PATH_INFO");
   CgiPathTranslatedPtr = CgiVar ("WWW_PATH_TRANSLATED");
   CgiScriptNamePtr = CgiVar ("WWW_SCRIPT_NAME");
   CgiHttpPragmaPtr = CgiVar ("WWW_HTTP_PRAGMA");
   CgiHttpIfModifiedSincePtr = CgiVar ("WWW_HTTP_IF_MODIFIED_SINCE");

   CgiFormDoPtr = CgiVar ("WWW_FORM_DO");
   CgiFormExplodePtr = CgiVar ("WWW_FORM_EXPLODE");
   CgiFormKeyPtr = CgiVar ("WWW_FORM_KEY");
   CgiFormTitlePtr = CgiVar ("WWW_FORM_TITLE");

   CgiFormSearchPtr = CgiVar ("WWW_FORM_SEARCH");
   if (!CgiFormSearchPtr[0])
      CgiFormSearchPtr = CgiVar ("WWW_KEY_1");

   CgiFormRefererPtr = CgiVar ("WWW_FORM_REFERER");
   if (!CgiFormRefererPtr[0])
      CgiFormRefererPtr = CgiVar ("WWW_HTTP_REFERER");

   if (CgiFormRefererPtr[0])
   {
      /* re-escape the URL-escaped percentages */
      CopyIntoUri (UriReferer, CgiFormRefererPtr, -1);
      CopyIntoHtml (HtmlReferer, CgiFormRefererPtr, -1);
   }

   if (VMSnok (status = TimeSetGmt ()))
   {
      if (status != SS$_NOLOGNAM)
      {
         ErrorVmsStatus (status, CgiScriptNamePtr, "", __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   HttpGmTimeString (GmDateTime, 0);

   if (CgiHttpIfModifiedSincePtr[0])
   {
      if (VMSnok (HttpGmTime (CgiHttpIfModifiedSincePtr, 
                              &IfModifiedSinceBinaryTime)))
      {
         if (Debug) fprintf (stdout, "If-Modified-Since: NBG!\n");
         IfModifiedSinceBinaryTime[0] = IfModifiedSinceBinaryTime[0] = 0;
         CgiHttpIfModifiedSincePtr[0] = "";
      }
   }

   /**************/
   /* initialize */
   /**************/

   idx = 0;
   if (CgiFormKeyPtr[0])
   {
      /*****************************************/
      /* get keywords from "key=" query string */
      /*****************************************/

      cptr = CgiFormKeyPtr;
      while (*cptr)
      {
         if (idx <= 10) idx++;
         if (*cptr == '~') cptr++;
         sptr = KeyWord[idx];
         while (*cptr && *cptr != '~') *sptr++ = *cptr++;
         *sptr = '\0';
         CopyIntoHtml (KeyWordHtml[idx], KeyWord[idx], -1);
         if (Debug)
            fprintf (stdout, "KeyWord[idx] |%s|%s|\n",
                     KeyWord[idx], KeyWordHtml[idx]);
      }
   }
   else
   if (strstr (CgiPathTranslatedPtr, ".HLB") == NULL &&
       strstr (CgiPathTranslatedPtr, ".TLB") == NULL)
   {
      /*******************************/
      /* get keywords from path info */
      /*******************************/

      /* make sure the default help library is used */
      CgiPathTranslatedPtr = "";

      idx = 0;
      /* by comparing, scan past the script component of the path */
      sptr = CgiScriptNamePtr;
      cptr = CgiPathInfoPtr;
      while (toupper(*cptr) == toupper(*sptr)) { cptr++; sptr++; }
      while (*cptr)
      {
         if (idx <= 10) idx++;
         if (*cptr == '/') cptr++;
         sptr = KeyWord[idx];
         if (*cptr == '_' || *cptr == '/')
         {
            /*
               Some other HTML help systems use an underscore for qualifiers.
               This implmentation also allows two concurrent forward-slashes
               so that the qualifier may be entered literally!
            */
            *sptr++ = '/';
            cptr++;
         }
         while (*cptr && *cptr != '/') *sptr++ = *cptr++;
         *sptr = '\0';
         CopyIntoHtml (KeyWordHtml[idx], KeyWord[idx], -1);
         if (Debug)
            fprintf (stdout, "KeyWord[idx] |%s|%s|\n",
                     KeyWord[idx], KeyWordHtml[idx]);
      }
      /* finished with the path if we've used it for keywords! */
      CgiPathInfoPtr = "";
   }            
   KeyWordCount = idx;
   while (idx++ < 10) KeyWord[idx][0] = KeyWordHtml[idx][0] = '\0';

   strcpy (LibraryPathInfo, CgiPathInfoPtr);

   /* isolate the directory component of the path */
   strcpy (LibraryDirPath, CgiPathInfoPtr);
   for (cptr = LibraryDirPath; *cptr; cptr++);
   if (cptr > LibraryDirPath) cptr--;
   while (cptr > LibraryDirPath && *cptr != '/') cptr--;
   if (*cptr == '/') cptr++;
   *cptr = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", LibraryDirPath);

   strcpy (LibrarySpec, CgiPathTranslatedPtr);
   for (sptr = LibrarySpec; *sptr && *sptr != '*' && *sptr != '%'; sptr++);
   if (*sptr) DoListLibraries = true;

   switch (toupper(CgiFormDoPtr[0]))
   {
      case 'H' :  DoHeaderInformation = true;  break;
      case 'L' :  DoListLibraries = true;  break;
      case 'M' :  DoLibraryMenu = true;  break;
      case 'S' :  DoSearch = true;  break;
      default :
         for (cptr = CgiPathTranslatedPtr; *cptr; cptr++)
            if (*cptr == '*' || *cptr == '%') DoListLibraries = true;
   }

   if (CgiFormExplodePtr[0])
      ExplodeHelp = true;
   else
      ExplodeHelp = false;

   if (CgiFormSearchPtr[0])
   {
      DoSearch = true;
      SearchStringLength = strlen(CgiFormSearchPtr);
      CopyIntoHtml (HtmlSearchString, CgiFormSearchPtr, -1);
   }
   if (DoSearch && !CgiFormSearchPtr[0])
   {
      ErrorGeneral ("Search string not supplied.", __FILE__, __LINE__);
      return;
   }

   if (!KeyWord[1][0])
   {
      /* no module keyname supplied, wildcard to list all modules */
      strcpy (KeyWord[1], "*");
      KeyNameWildcard = true;
   }
   else
   {
      /* check for a wildcard in the keyname, list all modules if there is */
      for (sptr = KeyWord[1]; *sptr && *sptr != '*' && *sptr != '%'; sptr++);
      if (*sptr) KeyNameWildcard = true;
   }

   LibraryTitle[0] = '\0';
   if (!CgiPathTranslatedPtr[0]) CgiFormTitlePtr = "VMS Help";
   if (CgiFormTitlePtr[0]) strcpy (LibraryTitle, CgiFormTitlePtr);
   if (LibraryTitle[0])
      CopyIntoHtml (HtmlLibraryTitle, LibraryTitle, -1);
   else
   {
      sprintf (LibraryTitle, "Library %s", LibraryPathInfo);
      sprintf (HtmlLibraryTitle, "Library %s", LibraryPathInfo);
   }
   if (Debug) fprintf (stdout, "LibraryTitle |%s|\n", LibraryTitle);

   CopyIntoUri (UriFormTitle, LibraryTitle, -1);
   CopyIntoHtml (HtmlFormTitle, LibraryTitle, -1);

   /***********/
   /* process */
   /***********/

   if (DoListLibraries)
      ListLibraries ();
   else
   if (DoLibraryMenu)
      LibraryMenu ();
   else
      Librarian ();
 }

/****************************************************************************/
/*
Search for library files according to the the supplied specification (defaults 
to help libraries; SYS$HELP:*.HLB).  Display the name of each library in lower 
case as a list item.
*/

ListLibraries ()

{
   register int  len;
   register char  *cptr, *sptr;

   boolean  SupportedLibraryType;
   int  status,
        FileCount = 0;
   char  FileName [256],
         ExpandedFileSpec [256],
         String [1024];
   struct FAB  FileFab = cc$rms_fab;
   struct NAM  FileNam = cc$rms_nam;

   if (Debug) fprintf (stdout, "ListLibraries()\n");

   /* if no library specifcation provided then default to help libraries */
   if (!LibrarySpec[0]) strcpy (LibrarySpec, "SYS$HELP:*.HLB;0");

   /* initialize the file access block (FAB) */
   FileFab.fab$l_fna = LibrarySpec;
   FileFab.fab$b_fns = strlen(LibrarySpec);
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block (NAM) */
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
   {
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   BeginPage (LibraryPathInfo, LibraryPathInfo);

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (!FileCount++) fprintf (stdout, "<OL>\n");

      /* ignore directories if the file type includes them (numeric ".DIR") */
      if (*(unsigned long*)FileNam.nam$l_type == 0x5249442e) continue;

      /* for efficiency; numeric equivalent of ".HLB" and ".TLB" */
      if (*(unsigned long*)FileNam.nam$l_type == 0x424c482e ||
          *(unsigned long*)FileNam.nam$l_type == 0x424c542e)
         SupportedLibraryType = true;
      else
         SupportedLibraryType = false;

      sptr = String;
      strcpy (sptr, "<LI>");
      sptr += 4;
      if (SupportedLibraryType)
      {
         sptr += sprintf (sptr, "<A HREF=\"%s%s",
                          CgiScriptNamePtr, LibraryDirPath);
         *(char*)FileNam.nam$l_ver = '\0';
         for (cptr = FileNam.nam$l_name; *cptr; *sptr++ = tolower(*cptr++));
         *(char*)FileNam.nam$l_ver = ';';
         sptr += sprintf (sptr, "?do=menu&title=%s&referer=%s\">",
                          UriFormTitle, UriReferer);
      }

      /* if there was wildcard in the file type then include it */
      if (FileNam.nam$l_fnb & NAM$M_WILD_TYPE)
         *(char*)FileNam.nam$l_ver = '\0';
      else
         *(char*)FileNam.nam$l_type = '\0';
      cptr = (char*)FileNam.nam$l_name;
      while (*cptr) *sptr++ = tolower(*cptr++);
      *sptr = '\0';
      if (FileNam.nam$l_fnb & NAM$M_WILD_TYPE)
         *(char*)FileNam.nam$l_ver = ';';
      else
         *(char*)FileNam.nam$l_type = '.';

      if (SupportedLibraryType)
      {
         strcpy (sptr, "</A>\n");
         sptr += 5;
      }
      *sptr = '\0';

      fprintf (stdout, "%s", String);
   }

   if (FileCount)
      fprintf (stdout, "</OL>\n");
   else
      fprintf (stdout, "<P>\nNo files found.\n");

   if (status == RMS$_FNF || status == RMS$_NMF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (ErrorReported) return (status);
   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
   return (status);
}

/*****************************************************************************/
/*
For the specified library file provide a menu of four services; librarian, 
searching, library header and help.
*/ 
 
LibraryMenu ()
 
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LibraryMenu() |%s|\n", LibrarySpec);

   strcpy (LibraryTitle, LibraryPathInfo);
   CopyIntoHtml (HtmlLibraryTitle, LibraryPathInfo, -1);
   CopyIntoUri (UriFormTitle, LibraryTitle, -1);
   CopyIntoHtml (HtmlFormTitle, LibraryTitle, -1);

   BeginPage (LibraryTitle, HtmlLibraryTitle);

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<UL>\n\
<LI><A HREF=\"%s%s?title=%s&referer=%s\">Open Library</A>\n\
<LI><A HREF=\"%s%s?do=header&title=%s&referer=%s\">Library Header</A>\n\
</UL>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
   CgiScriptNamePtr, LibraryPathInfo, UriFormTitle, UriReferer,
   CgiScriptNamePtr, LibraryPathInfo, UriFormTitle, UriReferer,
   CgiScriptNamePtr, LibraryPathInfo,
   HtmlReferer, HtmlFormTitle);

   HttpHasBeenOutput = true;

   if (ErrorReported) return;
   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/*****************************************************************************/
/*
Initialise library control.  Open the library specified by LibrarySpec.  Get 
header information from the library so its internal format can be determined 
(help or text, other are not supported).  Call ProcessLibrary() to perform the 
requested access according to the library type.
*/ 
 
Librarian ()

{
   register char  *cptr;
   int  status;
   unsigned long  Function = LBR$C_READ,
                  Length;
   char  String [1024];
   $DESCRIPTOR (LibrarySpecDsc, "");
   static struct dsc$descriptor_s 
   LibraryDefDsc = { sizeof(DEFAULT_LIBRARY)-1, DSC$K_DTYPE_T,
                     DSC$K_CLASS_S, DEFAULT_LIBRARY },
   LibraryNameDsc = { sizeof(LibraryName)-1, DSC$K_DTYPE_T,
                      DSC$K_CLASS_S, LibraryName };

   /****************************************/
   /* initialize, open library, get header */
   /****************************************/

   if (Debug) fprintf (stdout, "Librarian()\n");

   if (VMSnok (status = lbr$ini_control (&LibraryIndex, &Function, 0, 0)))
   {
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }
 
   LibrarySpecDsc.dsc$w_length = strlen(LibrarySpec);
   LibrarySpecDsc.dsc$a_pointer = LibrarySpec;
 
   if (VMSnok (status = lbr$open (&LibraryIndex,
                                  &LibrarySpecDsc,
                                  0,
                                  &LibraryDefDsc,
                                  0,
                                  &LibraryNameDsc,
                                  &Length)))
   {
      if (Debug) fprintf (stdout, "lib$open() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }
 
   /* terminate, removing the trailing version number */
   LibraryName[Length] = '\0';
   for (cptr = LibraryName; *cptr && *cptr != ';'; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "LibraryName |%s|\n", LibraryName);

   if (VMSnok (status = lbr$get_header (&LibraryIndex, &LibraryHeader)))
   {
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (CgiHttpIfModifiedSincePtr[0])
   {
      if (VMSnok (ModifiedSince (&IfModifiedSinceBinaryTime,
                                 &LibraryHeader.lhi$l_updtim)))
      {
         /* library has not been modified since the date/time, don't send */
         status = lbr$close (&LibraryIndex);
         return (SS$_NORMAL);
      }
   }

   HttpGmTimeString (LastModifiedGmDateTime, &LibraryHeader.lhi$l_updtim);

   /*******************/
   /* process library */
   /*******************/

   if (DoHeaderInformation)
      HeaderInformation ();
   else
   if (LibraryHeader.lhi$l_type == LBR$C_TYP_TXT)
   {
      if (DoSearch)
         SearchTextLibraryModules ();
      else
      if (KeyNameWildcard)
         ListTextLibraryModules ();
      else
         OpenTextLibraryModule ();
   }
   else
   if (LibraryHeader.lhi$l_type == LBR$C_TYP_HLP)
   {
      if (DoSearch)
         SearchHelpLibraryModules ();
      else
      if (KeyNameWildcard)
         ListHelpLibraryModules ();
      else
         OpenHelpLibraryModule ();
   }
   else
   {
      sprintf (String,
      "Library type %d not supported. (Only HELP and TEXT libraries are valid)",
      LibraryHeader.lhi$l_type);
      ErrorGeneral (String, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }
 
   status = lbr$close (&LibraryIndex);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Provide selected library header information.
*/ 
 
int HeaderInformation ()
 
{
   static char  *LibraryTypes[] =
          { "Unknown", "VAX Object", "Macro", "Help", "Text",
            "VAX Sharable Image", "NCS",
            "Alpha Object", "Alpha Sharable Image", "?" };
   static  $DESCRIPTOR (CreatedFaoDsc,
           "<TR><TH>Created</TH><TD>!%D</TD></TR>\n");
   static  $DESCRIPTOR (RevisedFaoDsc,
           "<TR><TH>Revised</TH><TD>!%D</TD></TR>\n");

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   BeginPage (LibraryTitle, LibraryTitle);

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<TABLE BORDER=0 CELLPADDING=5 CELLSPACING=0>\n");

   fflush (stdout);
   HttpHasBeenOutput = true;

   if (LibraryHeader.lhi$l_type > 8) LibraryHeader.lhi$l_type = 9;
   fprintf (stdout, "<TR><TH ALIGN=right>Type:</TH><TD>%s</TD></TR>\n",
      LibraryTypes[LibraryHeader.lhi$l_type]);

   fprintf (stdout, "<TR><TH ALIGN=right>Creator:</TH><TD>%-*s</TD></TR>\n",
      *(char*)LibraryHeader.lhi$t_lbrver, (char*)LibraryHeader.lhi$t_lbrver+1);

   fprintf (stdout, "<TR><TH ALIGN=right>Format:</TH><TD>%d.%d</TD></TR>\n",
      LibraryHeader.lhi$l_majorid, LibraryHeader.lhi$l_minorid);

   sys$fao (&CreatedFaoDsc, &Length, &StringDsc, &LibraryHeader.lhi$l_credat);
   String[Length] = '\0';
   fprintf (stdout, "%s", String);

   sys$fao (&RevisedFaoDsc, &Length, &StringDsc, &LibraryHeader.lhi$l_updtim);
   String[Length] = '\0';
   fprintf (stdout, "%s", String);

   if (LibraryHeader.lhi$l_libstatus)
      fprintf (stdout,
"<TR><TH ALIGN=right>Status:</TH><TD>OK</TD></TR>\n");
   else
      fprintf (stdout,
"<TR><TH ALIGN=right>Status:</TH>\
<TD><FONT COLOR=\"#ff0000\">PROBLEM</FONT></TD></TR>\n");

   fprintf (stdout, "</TABLE>\n");

   if (ErrorReported) return;
   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/*****************************************************************************/
/*
*/
 
int ListHelpLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   if (Debug) fprintf (stdout, "ListHelpLibraryModules()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;

   BeginPage (LibraryTitle, HtmlLibraryTitle);

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n\
<P>\n",
   CgiScriptNamePtr, CgiPathInfoPtr,
   HtmlReferer, HtmlFormTitle);

   fflush (stdout);
   HttpHasBeenOutput = true;

   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &ListHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   fprintf (stdout, "\n");

   if (ErrorReported) return (status);
   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}
 
/*****************************************************************************/
/*
*/ 
 
int ListHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static int  PreviousCgiPlusUsageCount = 0;
   static char  PreviousAlphabetic = '\0';

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListHelpModule()\n");

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (!isalpha(KeyName[1][0]) ||
       toupper(KeyName[1][0]) == PreviousAlphabetic)
   {
      if (CgiPlusUsageCount == PreviousCgiPlusUsageCount)
         fprintf (stdout, ",\n");
   }
   else
   {
      if (CgiPlusUsageCount == PreviousCgiPlusUsageCount)
         fprintf (stdout, "\n<P>");
      PreviousAlphabetic = toupper(KeyName[1][0]);
   }

   fprintf (stdout, "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
            CgiScriptNamePtr, LibraryPathInfo,
            FormatRequestKeys(true,1),
            UriFormTitle, UriReferer,
            MungeUnderScores(KeyNameHtml[1]));

   PreviousCgiPlusUsageCount = CgiPlusUsageCount;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int OpenHelpLibraryModule ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   if (Debug) fprintf (stdout, "OpenHelpLibraryModule()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &OpenHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (!ErrorReported) ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}
 
/*****************************************************************************/
/*
This is a reasonably complex function, and reflects the complexities internal 
to help library modules!
*/ 
 
int OpenHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static int  MatchedCount = 0,
               PreviousCgiPlusUsageCount = 0;

   register int  Count;
   register char  *cptr, *sptr;

   boolean  AdditionalInformation = true,
            ExplodedHeading = false,
            OutputModuleTitle = true;
   int  status,
        AdditionalInformationCount = 0,
        BlankLineCount = 0,
        ExplodeLevel = 0,
        HelpLevel = 0,
        Length,
        MatchedLevel = 0,
        MatchesToLevel = 0,
        PreviousHelpLevel = 0,
        TextLineCount = 0;
   int  KeyCount [11];
   char  PreviousFirstBufferChar = '\0';
   char  Buffer [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "OpenHelpModule()\n");

   if (CgiPlusUsageCount != PreviousCgiPlusUsageCount)
   {
      MatchedCount = 0;
      PreviousCgiPlusUsageCount = CgiPlusUsageCount;
   }

   strncpy (KeyName[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyName[1][KeyNameDscPtr->dsc$w_length] = '\0';

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyName[1]);

   CopyIntoHtml (KeyWordHtml[1], KeyName[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyName[1], -1);
   if (!KeyWordCount)
   {
      strcpy (KeyWord[1], KeyName[1]);
      KeyWordCount = 1;
   }

   if (ExplodeHelp)
      for (Count = 0; Count < 11; KeyCount[Count++] = 0);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      /* terminate output buffer, removing any trailing white space */
      cptr = Buffer+OutBufferDsc.dsc$w_length-1;
      while (cptr >= Buffer && isspace(*cptr)) cptr--;
      *++cptr = '\0';
 
      /* comment record (line) in module */
      if (Buffer[0] == '!') continue;

      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      if ((isdigit(Buffer[0]) && Buffer[1] == ' ') || Buffer[0] == '/')
      {
         /***************************/
         /* help level or qualifier */
         /***************************/
 
         PreviousHelpLevel = HelpLevel;
         if (Buffer[0] == '/')
         {
            /* if no previous qualifier encountered at this help level */
            /* then the qualifier effectively creates a new help level */
            if (PreviousFirstBufferChar != '/') HelpLevel++;
            cptr = Buffer;
         }
         else
         {
            /* help level topic (line begins with a digit then a space) */
            HelpLevel = Buffer[0] - '0';
            for (cptr = Buffer + 1; isspace(*cptr); cptr++);
         }

         strcpy (KeyName[HelpLevel], cptr);
         CopyIntoHtml (KeyNameHtml[HelpLevel], cptr, -1);
         PreviousFirstBufferChar = Buffer[0];

         if (MatchesToLevel >= HelpLevel)
         {
            /* if the topic's been matched and output, finish up */
            if (MatchedLevel) break;

            MatchesToLevel = ExplodeLevel = 0;
            for (Count = 1; Count <= HelpLevel; Count++)
            {
               if (strsame (KeyWord[Count], KeyName[Count], -1))
                  MatchesToLevel = Count;
               else
                  break;
            }
         }
         else
         {
            for (Count = MatchesToLevel + 1; Count <= HelpLevel; Count++)
            {
               if (strsame (KeyWord[Count], KeyName[Count], -1))
                  MatchesToLevel = Count;
               else
                  break;
            }
         }

         if (MatchesToLevel == KeyWordCount && MatchesToLevel == HelpLevel)
         {
            /****************/
            /* module title */
            /****************/

            MatchedCount++;

            sptr = String;
            sptr += sprintf (sptr, "%s<BR>\n", LibraryTitle);
            for (Count = 1; Count <= HelpLevel; Count++)
            {
               if (Count == 1)
                  sptr += sprintf (sptr, "%s",
                             MungeUnderScores(KeyNameHtml[Count]));
               else
                  sptr += sprintf (sptr, ", %s",
                             MungeUnderScores(KeyNameHtml[Count]));
            }

            BeginPage (LibraryTitle, String);

            if (ExplodeHelp)
            {
               ExplodedHeading = true;
               ExplodeLevel = HelpLevel + 1;
            }

            MatchedLevel = HelpLevel;
         }
         else
         if (ExplodeLevel && HelpLevel >= ExplodeLevel)
         {
            /******************/
            /* exploding help */
            /******************/

            if (TextLineCount)
               if (PreviousHelpLevel == ExplodeLevel - 1)
                  fprintf (stdout, "\n</PRE>\n</BLOCKQUOTE>\n<P>\n");
               else
                  fprintf (stdout, "\n</PRE>\n</BLOCKQUOTE>\n");
            TextLineCount = 0;

            ExplodedHeading = true;
            for (Count = HelpLevel+1; Count < 11; KeyCount[Count++] = 0);
            KeyCount[HelpLevel]++;

            if (ExplodeLevel < HelpLevel ||
                KeyName[HelpLevel][0] == '/')
            {
               fprintf (stdout,
"<P>\n\
<FONT SIZE=+1><B>");
            }
            else
            if (PageScheme[PS_LAYOUT][0] == '2')
            {
               fprintf (stdout,
"<P>\n\
<FONT SIZE=+1><B><U>");
            }
            else
            {
               fprintf (stdout,
"<P>\n\
<TABLE BORDER=%s CELLPADDING=2 CELLSPACING=0 WIDTH=95%%>\n\
<TR><TD%s>&nbsp;\n\
<FONT COLOR=\"%s\" SIZE=+1><B>\n",
                  PageScheme[PS_HEADBORDER],
                  PageScheme[PS_HEADBGCOLOR],
                  PageScheme[PS_HEADTEXT]);
            }

            for (Count = ExplodeLevel; Count <= HelpLevel; Count++)
            {
               if (Count < HelpLevel)
                  fprintf (stdout, "%d.", KeyCount[Count]);
               else
                  fprintf (stdout, "%d", KeyCount[Count]);
            }

            if (ExplodeLevel < HelpLevel ||
                KeyName[HelpLevel][0] == '/')
            {
               fprintf (stdout, " - %s</B></FONT>\n",
                  MungeUnderScores(KeyNameHtml[HelpLevel]));
            }
            else
            if (PageScheme[PS_LAYOUT][0] == '2')
            {
               fprintf (stdout, " - %s</U></B></FONT>\n",
                  MungeUnderScores(KeyNameHtml[HelpLevel]));
            }
            else
            {
               fprintf (stdout,
" - %s\n\
</B></FONT>\n\
</TD></TR>\n\
</TABLE>\n",
                  MungeUnderScores(KeyNameHtml[HelpLevel]));
            }
         }
         else
         if (MatchesToLevel == KeyWordCount && HelpLevel == KeyWordCount+1)
         {
            /*************/
            /* subtopics */
            /*************/

            if (AdditionalInformation)
            {
               if (TextLineCount)
                  fprintf (stdout, "\n</PRE>\n</BLOCKQUOTE>\n");
               fprintf (stdout, "<BLOCKQUOTE>\n");
               if (HelpLevel == 2)
               {
                  fprintf (stdout,
"<P>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=hidden NAME=\"key\" VALUE=\"%s\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search &quot;%s&quot; for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
                  CgiScriptNamePtr, CgiPathInfoPtr,
                  HtmlReferer, HtmlFormTitle,
                  FormatRequestKeys(false,HelpLevel-1),
                  KeyWordHtml[1]);
               }
               fprintf (stdout,
"<P>\n\
<B>Additional Information \
<I>(<A HREF=\"%s%s?key=%s&explode=yes&title=%s&referer=%s\">explode</A>)</I> \
:</B>\n\
<P>\n\
<UL>\n",
                  CgiScriptNamePtr, LibraryPathInfo,
                  FormatRequestKeys(true,HelpLevel-1),
                  UriFormTitle, UriReferer);
               AdditionalInformation = false;
            }
            else
            {
               if (TextLineCount)
                  fprintf (stdout, "\n</PRE>\n</BLOCKQUOTE>\n");
            }

            fprintf (stdout,
               "<LI><A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>\n",
               CgiScriptNamePtr, LibraryPathInfo,
               FormatRequestKeys(true,HelpLevel),
               UriFormTitle, UriReferer,
               MungeUnderScores(KeyNameHtml[HelpLevel]));
            AdditionalInformationCount++;
            TextLineCount = 0;
         }
      }
      else
      {
         /*************/
         /* help text */
         /*************/
 
         if ((ExplodeLevel && HelpLevel >= ExplodeLevel) || 
             (MatchesToLevel == KeyWordCount && HelpLevel == KeyWordCount))
         {
            /* check to see if this module record has any non-space chars */
            for (cptr = Buffer;
                 *cptr && (*cptr == ' ' || *cptr == '\t');
                 cptr++);
            /* ignore special directive for DECwindows on-line help */
            if (*cptr && *cptr != '=')
            {
               if (!TextLineCount) fprintf (stdout, "<P>\n<BLOCKQUOTE>\n<PRE>");
               sptr = String;
               if (ExplodedHeading)
               {
                  ExplodedHeading = false;
                  BlankLineCount = 0;
               }
               if (BlankLineCount && TextLineCount)
               {
                  /** *sptr++ = '\r'; **/
                  *sptr++ = '\n';
               }
               if (BlankLineCount) BlankLineCount = 0;
               while (*cptr) *sptr++ = *cptr++;
               *sptr++ = '\n';
               *sptr = '\0';
               HtmlPrintString (String, true);
               TextLineCount++;
            }
            else
            if (!*cptr)
               BlankLineCount++;
         }
      }
   }
 
   if (AdditionalInformationCount)
      fprintf (stdout, "</UL>\n</BLOCKQUOTE>\n");
   else
   if (TextLineCount)
      fprintf (stdout, "\n</PRE>\n</BLOCKQUOTE>\n");
   else
   if (!MatchedCount)
   {
      /* fake it! */
      ErrorVmsStatus (0x00268838, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (ErrorReported) return (status);
   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
*/
 
int SearchHelpLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchHelpLibraryModules()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   sprintf (String, "Search &quot;%s&quot; for &quot;%s&quot;\n",
            HtmlLibraryTitle, HtmlSearchString);

   BeginPage (String, String);

   fprintf (stdout, "<BLOCKQUOTE>\n");

   RecordsSearched = SearchHitCount = 0;
   IndexNumber = 1;

   if (DoSearchStatistics) lib$init_timer (0);

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &SearchHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (!SearchHitCount) fprintf (stdout, "<B>Not found!</B>\n<P>\n");

   if (DoSearchStatistics) SearchStatistics ();

   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/*****************************************************************************/
/*
This is a reasonably complex function, and reflects the complexities internal 
to help library modules!
*/ 
 
int SearchHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static boolean  HitThisTopic = false;
   static char  PreviousFirstBufferChar = '\0';

   register int  Count;
   register char  *cptr, *hptr, *sptr;
   int  status,
        HelpLevel = 0,
        Length;
   char  Buffer [1024],
         String [8192];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchHelpModule()\n");

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoHtml (KeyWordHtml[1], KeyWord[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);
   if (!KeyWordCount) KeyWordCount = 1;

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      /* terminate output buffer, removing any trailing white space */
      cptr = Buffer+OutBufferDsc.dsc$w_length-1;
      while (cptr >= Buffer && isspace(*cptr)) cptr--;
      *++cptr = '\0';
 
      /* comment record (line) in module */
      if (Buffer[0] == '!') continue;

      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      if ((isdigit (Buffer[0]) && Buffer[1] == ' ') || Buffer[0] == '/')
      {
         /***************************/
         /* help level or qualifier */
         /***************************/
 
         if (Buffer[0] == '/')
         {
            /* if no previous qualifier encountered at this help level */
            /* then the qualifier effectively creates a new help level */
            if (PreviousFirstBufferChar != '/') HelpLevel++;
            cptr = Buffer;
         }
         else
         {
            /* help level topic (line begins with a digit then a space) */
            HelpLevel = Buffer[0] - '0';
            for (cptr = Buffer + 1; isspace(*cptr); cptr++);
         }

         PreviousFirstBufferChar = Buffer[0];
         strcpy (KeyName[HelpLevel], cptr);
         CopyIntoHtml (KeyNameHtml[HelpLevel], cptr, -1);

         /* if previous topic had a hit then end the list of the hit(s) */
         if (HitThisTopic) fprintf (stdout, "</UL>\n");
         HitThisTopic = false;
      }
      else
      {
         /*************/
         /* help text */
         /*************/
 
         /* check to see if this module record has any non-space chars */
         for (cptr = Buffer; *cptr && isspace(*cptr); cptr++);
         if (*cptr)
         {
            if (*(hptr = SearchText (cptr, CgiFormSearchPtr, false)))
            {
               /********/
               /* hit! */
               /********/

               if (Debug) fprintf (stdout, "Hit |%s|\n", hptr);
               SearchHitCount++;
               sptr = String;

               if (!HitThisTopic)
               {
                  strcpy (sptr, "<FONT SIZE=+1><B>");
                  sptr += 17;
                  for (Count = 1; Count <= HelpLevel; Count++)
                  {
                     if (Count > 1) { *sptr++ = ','; *sptr++ = ' '; }
                     sptr += sprintf (sptr,
                        "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
                        CgiScriptNamePtr, LibraryPathInfo,
                        FormatRequestKeys(true,Count),
                        UriFormTitle, UriReferer,
                        MungeUnderScores(KeyNameHtml[Count]));
                  }
                  strcpy (sptr, "</B></FONT>\n<UL>\n<LI>");
                  sptr += 21;
                  HitThisTopic = true;
               }
               else
               {
                  /* same topic, new hit */
                  strcpy (sptr, "<LI>");
                  sptr += 4;
               }
               sptr += CopyIntoHtml (sptr, cptr, hptr-cptr);
               strcpy (sptr, "<B>");
               sptr += 3;
               strncpy (sptr, hptr, SearchStringLength);
               sptr += SearchStringLength;
               strcpy (sptr, "</B>");
               sptr += 4;
               sptr += CopyIntoHtml (sptr, hptr+SearchStringLength, -1);
               *sptr++ = '\n';
               *sptr = '\0';

               fprintf (stdout, "%s", String);
            }
         }
      }
   }

   if (HitThisTopic)
   {
      fprintf (stdout, "</UL>\n");
      HitThisTopic = false;
   }

   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
String together the keys associated with this help item.
*/ 
 
char* FormatRequestKeys
(
boolean ForUri,
int ToLevel
)
{
   static char  RequestKeys [256];

   register int  idx;
   register char  *sptr;

   sptr = RequestKeys;
   for (idx = 1; idx <= ToLevel; idx++)
   {
      if (ForUri)
         sptr += CopyIntoUri (sptr, KeyName[idx], -1);
      else
         sptr += CopyIntoHtml (sptr, KeyName[idx], -1);
      *sptr++ = '~';
   }
   /* if needed, get rid of that last '~'! */
   if (idx > 1)
      sptr[-1] = '\0';
   else
      *sptr = '\0';

   return (RequestKeys);
}
 
/*****************************************************************************/
/*
*/
 
int ListTextLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListTextLibraryModules()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   BeginPage (LibraryTitle, HtmlLibraryTitle);

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n\
<P>\n",
      CgiScriptNamePtr, CgiPathInfoPtr,
      HtmlReferer, HtmlFormTitle);

   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &ListTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (ErrorReported) return (status);
   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}
 
/*****************************************************************************/
/*
*/ 
 
int ListTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static int  PreviousCgiPlusUsageCount = 0;
   static char  PreviousAlphabetic = '\0';

   char  String [1024],
         UriKeyName [256];
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListTextModule()\n");

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer,
            KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoUri (UriKeyName, KeyWord[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (!isalpha(KeyName[1][0]) ||
        KeyName[1][0] == PreviousAlphabetic)
   {
      if (CgiPlusUsageCount == PreviousCgiPlusUsageCount)
         fprintf (stdout, ",\n");
   }
   else
   {
      if (CgiPlusUsageCount == PreviousCgiPlusUsageCount)
         fprintf (stdout, "\n<P>");
      PreviousAlphabetic = KeyName[1][0];
   }
   fprintf (stdout, "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
            CgiScriptNamePtr, LibraryPathInfo, UriKeyName,
            UriFormTitle, UriReferer, KeyNameHtml[1]);

   PreviousCgiPlusUsageCount = CgiPlusUsageCount;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int OpenTextLibraryModule ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "OpenTextLibraryModule()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   fprintf (stdout,
"HTTP/1.0 200 Success\r\n\
Server: %s\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/plain\r\n\
\r\n",
      CgiServerSoftwarePtr, GmDateTime, LastModifiedGmDateTime);

   fflush (stdout);
   HttpHasBeenOutput = true;

   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &OpenTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   return (status);
}
 
/*****************************************************************************/
/*
*/ 
 
int OpenTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   register char  *sptr;

   int  status;
   char  Buffer [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "OpenTextModule()\n");

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      /* add HTTP required carriage-control */
      Buffer[OutBufferDsc.dsc$w_length] = '\n';
      Buffer[OutBufferDsc.dsc$w_length+1] = '\0';
      fputs (Buffer, stdout);
  }
 
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int SearchTextLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchTextLibraryModules()\n");
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   sprintf (String, "Search &quot;%s&quot; for &quot;%s&quot;\n",
            HtmlLibraryTitle, HtmlSearchString);

   BeginPage (String, String);

   fprintf (stdout, "<BLOCKQUOTE>\n");

   RecordsSearched = SearchHitCount = 0;
   IndexNumber = 1;

   if (DoSearchStatistics) lib$init_timer (0);

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &SearchTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   if (!SearchHitCount) fprintf (stdout, "<B>Not found!</B>\n<P>\n");

   if (DoSearchStatistics) SearchStatistics ();

   fprintf (stdout, "</BLOCKQUOTE>\n");
   ButtonBar (2);
   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/*****************************************************************************/
/*
*/ 
 
int SearchTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   register char  *cptr, *hptr, *sptr;

   boolean  HitThisModule = false;
   int  status;
   char  Buffer [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchTextModule()\n");

   strncpy (KeyName[1],
            KeyNameDscPtr->dsc$a_pointer,
            KeyNameDscPtr->dsc$w_length);
   KeyName[1][KeyNameDscPtr->dsc$w_length] = '\0';
   CopyIntoHtml (KeyNameHtml[1], KeyName[1], -1);
   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyName[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, HtmlLibraryTitle, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      Buffer[OutBufferDsc.dsc$w_length] = '\0';
      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      /* check to see if this module record has any non-space chars */
      for (cptr = Buffer; *cptr && isspace(*cptr); cptr++);
      if (*cptr)
      {
         if (*(hptr = SearchText (cptr, CgiFormSearchPtr, false)))
         {
            /********/
            /* hit! */
            /********/

            if (Debug) fprintf (stdout, "Hit |%s|\n", sptr);
            SearchHitCount++;
            sptr = String;
            if (!HitThisModule)
            {
               sptr += sprintf (sptr,
"<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">\
<FONT SIZE=+1><B>%s</B></FONT></A>\n<UL>\n",
                  CgiScriptNamePtr, LibraryPathInfo,
                  KeyName[1], UriFormTitle, UriReferer, KeyNameHtml[1]);
               HitThisModule = true;
            }
            strcpy (sptr, "<LI><NOBR><TT>");
            sptr += 14;
            sptr += CopyIntoHtml (sptr, cptr, hptr-cptr);
            strcpy (sptr, "<B>");
            sptr += 3;
            strncpy (sptr, hptr, SearchStringLength);
            sptr += SearchStringLength;
            strcpy (sptr, "</B>");
            sptr += 4;
            sptr += CopyIntoHtml (sptr, hptr+SearchStringLength, -1);
            strcpy (sptr, "</TT></NOBR>");
            sptr += 12;
            *sptr++ = '\n';
            *sptr = '\0';

            fprintf (stdout, "%s", String);
         }
      }
   }

   if (HitThisModule) fprintf (stdout, "</UL>\n");

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Case sensistive or case insensistive search of the string pointed to by 'sptr' 
for the string pointed to by 'SearchString'.
*/ 

char* SearchText
( 
register char *tptr,
register char *SearchString,
register boolean CaseSensitive
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchText()\n");

   RecordsSearched++;

   sptr = SearchString;
   if (CaseSensitive)
   {
      while (*tptr)
      {
         if (*tptr++ != *sptr) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (*cptr != *sptr) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   else
   {
      while (*tptr)
      {
         if (toupper(*tptr++) != toupper(*sptr)) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (toupper(*cptr) != toupper(*sptr)) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   return (tptr);
}

/*****************************************************************************/
/*
*/ 

SearchStatistics ()

{
   static long  LibElapsedTime = 1,
                LibCpuTime = 2;
   static $DESCRIPTOR (ElapsedFaoDsc, "!%T");

   static $DESCRIPTOR (SearchStats1FaoDsc,
"<TABLE BORDER=0 CELLPADDING=2 CELLSPACING=0!AZ>\n\
<TR><TD>\n\
<FONT SIZE=-1 COLOR=\"!AZ\">&nbsp;\n\
<B>Elapsed:</B> !AZ&nbsp; \
<B>CPU:</B> !2ZL:!2ZL.!2ZL&nbsp; \
<B>Records (lines):</B> !UL\n\
&nbsp;</FONT>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (SearchStats2FaoDsc,
"<FONT SIZE=-1>&nbsp;\n\
<U>\
<B>Elapsed:</B> !AZ&nbsp; \
<B>CPU:</B> !2ZL:!2ZL.!2ZL&nbsp; \
<B>Records (lines):</B> !UL\
</U>\n\
&nbsp;</FONT>\n\
</TD></TR>\n\
</FONT>\n");

   unsigned long  ElapsedTime [2];
   unsigned long  CpuTime;
   unsigned short  Length;
   char  Elapsed [32],
         String [512];
   $DESCRIPTOR (ElapsedDsc, Elapsed);
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchStatistics()\n");

   lib$stat_timer (&LibElapsedTime, &ElapsedTime, 0);
   lib$stat_timer (&LibCpuTime, &CpuTime, 0);

   sys$fao (&ElapsedFaoDsc, &Length, &ElapsedDsc, &ElapsedTime);
   Elapsed[Length] = '\0';
   if (PageScheme[PS_LAYOUT][0] == '2')
      sys$fao (&SearchStats2FaoDsc, &Length, &StringDsc,
               Elapsed+3,
               CpuTime/6000, CpuTime/100, CpuTime%100,
               RecordsSearched);
   else
      sys$fao (&SearchStats1FaoDsc, &Length, &StringDsc,
               PageScheme[PS_HEADBGCOLOR],
               PageScheme[PS_HEADTEXT],
               Elapsed+3,
               CpuTime/6000, CpuTime/100, CpuTime%100,
               RecordsSearched);
   String[Length] = '\0';
   fprintf (stdout, "%s", String);
}

/*****************************************************************************/
/*
Output the HTTP header and page header.
*/
 
int BeginPage
(
char *TitleString,
char *HeaderString
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "BeginPage()\n");
 
   fprintf (stdout,
"HTTP/1.0 200 Success\r\n\
Server: %s\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"date\" CONTENT=\"%s\">\n\
<META NAME=\"CGI\" CONTENT=\"%s\">\n\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      CgiServerSoftwarePtr, GmDateTime, LastModifiedGmDateTime,
      SoftwareID, UnixDateTime, CgiTypePtr,
      TitleString,
      PageScheme[PS_BODYTAG]);

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s<FONT SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Conan The Librarian</U>\n\
</FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=95%%>\n\
<P>\n",
         PageScheme[PS_HEADLOCAL],
         HeaderString);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Conan The Librarian</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADPADDING],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT],
         HeaderString,
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADLOCAL]);
   }

   fflush (stdout);
   HttpHasBeenOutput = true;
}
 
/*****************************************************************************/
/*
In a help key string turn underscores into spaces mixed case.
*/

char* MungeUnderScores (char *InString)

{
   static  OutString [128];

   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MungeUnderScores()\n");

   sptr = OutString;
   if (InString[0] != '/')
   {
      /* not a qualifier, see if it's all the one case! */
      if (isupper(InString[0]))
      {
         for (cptr = InString; *cptr; cptr++)
            if (isalpha(*cptr) && !isupper(*cptr)) break;
      }
      else
      {
         for (cptr = InString; *cptr; cptr++)
            if (isalpha(*cptr) && !islower(*cptr)) break;
      }
   }
   if (InString[0] != '/' && *cptr)
   {                            
      /* probably an underscore-joined topic string */
      for (cptr = InString; *cptr; cptr++)
      {
         if (*cptr == '_')
            *sptr++ = ' ';
         else
            *sptr++ = *cptr;
      }
   }
   else
   {
      /* all upper-case, probably some particular VMS parameter */
      for (cptr = InString; *cptr; cptr++)
         *sptr++ = *cptr;
   }
   *sptr = '\0';
   return (OutString);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   ErrorReported = true;

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber, Text);
   }
   else
   {
      fprintf (stdout,
"HTTP/1.0 404 Error\r\n\
Server: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      CgiServerSoftwarePtr, SoftwareID, cptr, SourceLineNumber, Text);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   char  *MoreInfoPtr;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   ErrorReported = true;

   if (StatusValue == RMS$_FNF)
      strcpy (Message, "Book not found");
   else
   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      strcpy (Message, "&quot;sys$getmsg() failed&quot;");

   if (StatusValue == RMS$_DEV)
      MoreInfoPtr = "<P>The required CD-ROM may not be mounted.\n";
   else
      MoreInfoPtr = "";

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
%s\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText, MoreInfoPtr);
   }
   else
   {
      fprintf (stdout,
"HTTP/1.0 404 Error\r\n\
Server: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
%s\
</BODY>\n\
</HTML>\n",
      CgiServerSoftwarePtr, SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText, MoreInfoPtr);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
If the object has been modified since the specified date and time then return 
a normal status indicating that the data transfer is to continue.  If not 
modified then send a "not modified" HTTP header and return an error status to 
indicate the object should not be sent.
*/ 
 
int ModifiedSince
(
unsigned long *SinceBinaryTimePtr,
unsigned long *BinaryTimePtr
)
{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };

   int  status;
   unsigned long  AdjustedBinTime [2],
                  ScratchBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ModifiedSince()\n");

   /* if request asks for a "reload" (not cached) then give it regardless */
   if (strsame (CgiHttpPragmaPtr, "no-cache", -1)) return (SS$_NORMAL);

   /*
      Add one second to the modified time.  Ensures a negative time
      for VMS where fractional seconds may result in inconclusive
      results when the target time is being specified in whole seconds.
   */ 
   if (VMSnok (status =
       lib$add_times (SinceBinaryTimePtr, &OneSecondDelta, &AdjustedBinTime)))
   {
      ErrorVmsStatus (status, "IfModifiedSince:", "", __FILE__, __LINE__);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$add_times() %%X%08.08X\n", status);

   /* if a positive time results the file has been modified */
   if (VMSok (status =
       lib$sub_times (BinaryTimePtr, &AdjustedBinTime, &ScratchBinTime)))
      return (status);

   if (Debug) fprintf (stdout, "sys$sub_times() %%X%08.08X\n", status);
   if (status != LIB$_NEGTIM)
   {
      ErrorVmsStatus (status, "IfModifiedSince:", "", __FILE__, __LINE__);
      return (status);
   }

   fprintf (stdout,
"HTTP/1.0 304 Not Modified\r\n\
Server: %s\r\n\
\r\n",
     CgiServerSoftwarePtr);

   return (LIB$_NEGTIM);
}

/*****************************************************************************/
/*
Create an HTTP format Greenwich Mean Time (UTC) time string in the storage 
pointed at by 'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123). 
This must be at least 30 characters capacity.  If 'BinTimePtr' is null the 
time string represents the current time.  If it points to a quadword, VMS time 
value the string represents that time.  'TimeString' must point to storage 
large enough for 31 characters.
*/

int HttpGmTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *DayNames [] =
      { "", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (HttpTimeFaoDsc, "!AZ, !2ZW !AZ !4ZW !2ZW:!2ZW:!2ZW GMT");
   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2],
                  GmTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&GmTime);
   else
   {
      GmTime[0] = BinTimePtr[0];
      GmTime[1] = BinTimePtr[1];
   }
   if (VMSnok (status = TimeAdjustGMT (true, &GmTime)))
      return (status);

   status = sys$numtim (&NumTime, &GmTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&GmTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 30;
   TimeStringDsc.dsc$a_pointer = TimeString;

   if (VMSnok (status =
       sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                DayNames[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                NumTime[0], NumTime[3], NumTime[4], NumTime[5])))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Given a string such as "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123), or "Friday,
25-Aug-1995 17:32:40 GMT" (RFC 1036), create an internal, local, binary time
with the current GMT offset.  See complementary function HttpGmTimeString().
*/ 

int HttpGmTime
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   register char  *tptr;

   int  status;
   unsigned short  Length;
   unsigned short  NumTime [7] = { 0,0,0,0,0,0,0 };
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTime() |%s|\n", TimeString);

   tptr = TimeString;
   /* hunt straight for the comma after the weekday name! */
   while (*tptr && *tptr != ',') tptr++;
   if (*tptr) tptr++;
   /* span white space between weekday name and date */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the date and then skip to month name */
   if (isdigit(*tptr)) NumTime[2] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the month number from the name and skip to the year */
   for (NumTime[1] = 1; NumTime[1] <= 12; NumTime[1]++)
      if (strsame (tptr, MonthName[NumTime[1]], 3)) break;
   if (NumTime[1] > 12) return (STS$K_ERROR);
   while (*tptr && isalpha(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the year and then skip to the hour */
   if (isdigit(*tptr))
   {
      NumTime[0] = atoi (tptr);
      if (NumTime[0] < 100) NumTime[0] += 1900;
   }
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the hour, minute and second */
   if (isdigit(*tptr)) NumTime[3] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[4] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[5] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (!*tptr) return (STS$K_ERROR);

   /* the only thing remaining should be the "GMT" */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!strsame (tptr, "GMT", 3)) return (STS$K_ERROR);

   /*******************************************/
   /* convert what looks like legitimate GMT! */
   /*******************************************/

   if (Debug)
      fprintf (stdout, "NumTime[] %d %d %d %d %d %d %d\n",
               NumTime[0], NumTime[1], NumTime[2], NumTime[3], 
               NumTime[4], NumTime[5], NumTime[6]); 
   status = lib$cvt_vectim (&NumTime, BinTimePtr);
   if (VMSnok (status)) return (status);
   if (Debug) fprintf (stdout, "lib$cvt_vectim() %%X%08.08X\n", status);

   return (TimeAdjustGMT (false, BinTimePtr));
}

/*****************************************************************************/
/*
Determine the offset from GMT (UTC) using either the HTTPD$GMT or
SYS$TIMEZONE_DIFFERENTIAL logicals. If HTTPD$GMT is not defined time should be
set from the timezone differential logical. Function RequestBegin() calls
this every hour to recheck GMT offset and detect daylight saving or other
timezone changes.
*/

int TimeSetGMT ()

{
   static boolean  UseTimezoneDifferential = false;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetGMT()\n");

   if (!UseTimezoneDifferential)
   {
      status = TimeSetHttpdGmt();
      /* return if error and that error was not that the name did not exist */
      if (VMSok (status) || status != SS$_NOLOGNAM) return (status);
   }

   UseTimezoneDifferential = true;
   return (TimeSetTimezone());
}

/*****************************************************************************/
/*
The SYS$TIMEZONE_DIFFERENTIAL logical contains the number of seconds offset
from GMT (UTC) as a positive (ahead) or negative (behind) number.  Set the
'TimeGmtString' global storage to a "+hh:mm" or "-hh:mm" string equivalent, and
the 'TimeAheadOfGmt' global boolean and 'TimeGmtVmsString' delta-time global
string.
*/

int TimeSetTimezone ()

{
   static unsigned short  Length;
   static $DESCRIPTOR (TimezoneLogicalNameDsc, "SYS$TIMEZONE_DIFFERENTIAL");
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (TimeGmtStringFaoDsc, "!AZ!2ZL:!2ZL");
   static $DESCRIPTOR (TimeGmtVmsStringFaoDsc, "0 !2ZL:!2ZL");
   static $DESCRIPTOR (TimeGmtStringDsc, TimeGmtString);
   static struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   int  status;
   long  Hours,
         Minutes,
         Seconds;
   char  *SignPtr;
   $DESCRIPTOR (TimeGmtVmsStringDsc, TimeGmtVmsString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetTimezone()\n");

   status = sys$trnlnm (0, &LnmSystemDsc, &TimezoneLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);
   Seconds = atol(TimeGmtString);
   if (Seconds < 0)
   {
      TimeAheadOfGmt = false;
      Seconds = -Seconds;
      SignPtr = "-";
   }
   else
   {
      TimeAheadOfGmt = true;
      SignPtr = "+";
   }
   Hours = Seconds / 3600;
   Minutes = (Seconds - Hours * 3600) / 60;
   if (Debug)
      fprintf (stdout, "%d %s%d:%d\n", Seconds, SignPtr, Hours, Minutes);
   sys$fao (&TimeGmtStringFaoDsc, &Length, &TimeGmtStringDsc,
            SignPtr, Hours, Minutes);
   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);

   sys$fao (&TimeGmtVmsStringFaoDsc, &Length, &TimeGmtVmsStringDsc,
            Hours, Minutes);
   TimeGmtVmsString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   TimeGmtVmsStringDsc.dsc$w_length = Length;

   if (VMSnok (status =
       sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary)))
      return (status);
   if (TimeGmtDeltaBinary[0] || TimeGmtDeltaBinary[1])
      return (status);
   /* time must have been zero, make it one, one-hundreth of a second */
   TimeGmtDeltaBinary[0] = -100000;
   TimeGmtDeltaBinary[1] = -1;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Translate the logical HTTPD$GMT (defined to be something like "+10:30" or "-
01:15") and convert it into a delta time structure and store in 
'TimeGmtDeltaBinary'.  Store whether it is in advance or behind GMT in boolean 
'TimeAheadOfGmt'.  Store the logical string in 'TimeGmtString'.
*/

int TimeSetHttpdGmt ()

{
   static unsigned short  Length;
   static $DESCRIPTOR (GmtLogicalNameDsc, "HTTPD$GMT");
   static $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   static struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   register char  *cptr, *sptr;

   int  status;
   $DESCRIPTOR (TimeGmtVmsStringDsc, TimeGmtVmsString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetHttpdGmt()\n");

   status = sys$trnlnm (0, &LnmFileDevDsc, &GmtLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);

   if (TimeGmtString[0] == '$') return (SS$_NORMAL);

   if (*(cptr = TimeGmtString) == '-')
      TimeAheadOfGmt = false;
   else
      TimeAheadOfGmt = true;
   if (*cptr == '+' || *cptr == '-') cptr++;
   sptr = TimeGmtVmsString;
   *sptr++ = '0';
   *sptr++ = ' ';
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   TimeGmtVmsStringDsc.dsc$w_length = sptr - TimeGmtVmsString;

   if (VMSnok (status =
       sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary)))
      return (status);
   if (TimeGmtDeltaBinary[0] || TimeGmtDeltaBinary[1])
      return (status);
   /* time must have been zero, make it one, one-hundreth of a second */
   TimeGmtDeltaBinary[0] = -100000;
   TimeGmtDeltaBinary[1] = -1;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
The GMT is generated by calculating using an offset using 
'TimeGmtDeltaOffset' and boolean 'TimeAheadOfGmt'.  Adjust either to or from 
GMT.
*/ 

int TimeAdjustGMT
(
boolean ToGmTime,
unsigned long *BinTimePtr
)
{
   int  status;
   unsigned long  AdjustedTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeAdjustGMT() ToGmTime: %d\n", ToGmTime);

   if ((ToGmTime && TimeAheadOfGmt) || (!ToGmTime && !TimeAheadOfGmt))
   {
      /* to GMT from local and ahead of GMT, or to local from GMT and behind */
      status = lib$sub_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   }
   else
   {
      /* to GMT from local and behind GMT, or to local from GMT and ahead */
      status = lib$add_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
   }

   if (Debug)
   {
      unsigned short  Length;
      char  String [64];
      $DESCRIPTOR (AdjustedTimeFaoDsc, "AdjustedTime: |!%D|\n");
      $DESCRIPTOR (StringDsc, String);

      sys$fao (&AdjustedTimeFaoDsc, &Length, &StringDsc, &AdjustedTime);
      String[Length] = '\0';
      fprintf (stdout, "%s", String);
   }

   BinTimePtr[0] = AdjustedTime[0];
   BinTimePtr[1] = AdjustedTime[1];

   return (status);
}

/*****************************************************************************/
/*
*/

HtmlPrintString
(
char *InString,
boolean AnchorUrls
)
{
#  define ELBOW_ROOM 128

   register char  *bptr, *cptr, *isptr, *sptr, *uptr, *zptr;

   boolean  WasMailAddress;
   char  OutString [512],
         Url [512];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessLine()\n");

   zptr = (sptr = OutString) + sizeof(OutString) - ELBOW_ROOM;

   isptr = InString;
   while (*isptr)
   {
      /** if (Debug) fprintf (stdout, "isptr |%s|\n", isptr); **/
      if (sptr >= zptr)
      {
         *sptr = '\0';
         fputs (OutString, stdout);
         zptr = (sptr = OutString) + sizeof(OutString) - ELBOW_ROOM;
      }

      if (AnchorUrls)
      {
         if (!memcmp (isptr, "http://", 7) ||
             !memcmp (isptr, "mailto:", 7) ||
             !memcmp (isptr, "ftp://", 6) ||
             !memcmp (isptr, "gopher://", 9) ||
             !memcmp (isptr, "news://", 7) ||
             !memcmp (isptr, "shttp://", 8) ||
             !memcmp (isptr, "wais://", 7))
         {
            /***********/
            /* web URL */
            /***********/

            *sptr = '\0';
            fputs (OutString, stdout);
            zptr = (sptr = OutString) + sizeof(OutString) - ELBOW_ROOM;

            uptr = Url;
            while (*isptr && !isspace(*isptr)) *uptr++ = *isptr++;
            /* if trailed by what is probably textual punctuation */
            if (uptr > Url && !isalnum(uptr[-1]) && uptr[-1] != '/')
            {
               uptr--;
               isptr--;
            }
            *uptr = '\0';

            fputs ("<A HREF=\"", stdout);
            HtmlPrintString (Url, false);
            fputs ("\">", stdout);
            HtmlPrintString (Url, false);
            fputs ("</A>", stdout);

            continue;
         }
   
         /*************************/
         /* detect a mail address */
         /*************************/

         cptr = isptr;
         /* look for the local part of an internet mail address */
         while (*cptr && *cptr != '@') cptr++;
         if (cptr > isptr && *cptr == '@')
         {
            WasMailAddress = false;
            /* use 'uptr' to buffer the location of one past the '@' */
            uptr = cptr + 1;

            /* the for() just allows us to break at an appropriate point */
            for (;;)
            {
               /* confirm it's a local part */
               if (*--cptr == '\"')
               {
                  /* quoted local part, anything goes! */
                  for (cptr--; cptr >= isptr && *cptr != '\"'; cptr--);
                  /* if not a mail address */
                  if (*cptr != '\"') break;
               }

               while (cptr >= isptr)
               {
                  if (isalnum(*cptr) || *cptr == '-' || *cptr == '_' ||
                      *cptr == '.')
                  {
                     cptr--;
                     continue;
                  }
                  /* not an allowed local part character */
                  cptr = NULL;
                  break;
               }
               /* if not a mail address */
               if (cptr == NULL) break;

               /* restore the buffered one past the '@' */
               cptr = uptr;

               /* confirm what may be the first domain part of an address */
               while (*cptr && *cptr != '.' &&
                      (isalnum(*cptr) || *cptr == '-' || *cptr == '_')) cptr++;
               /* if not a mail address */
               if (*cptr != '.') break;

               /* confirm what may be the rest of the domain part */
               bptr = cptr;
               while (*cptr &&
                      (isalnum(*cptr) || *cptr == '-'  ||
                       *cptr == '_' || *cptr == '.')) cptr++;
               /* if not a mail address */
               if (cptr <= bptr) break;

               /****************/
               /* mail address */
               /****************/

               WasMailAddress = true;

               *sptr = '\0';
               fputs (OutString, stdout);
               zptr = (sptr = OutString) + sizeof(OutString) - ELBOW_ROOM;

               uptr = Url;
               while (isptr < cptr) *uptr++ = *isptr++;
               /* if trailed by what is probably textual punctuation */
               if (uptr > Url && !isalnum(uptr[-1]) && uptr[-1] != '/')
               {
                  uptr--;
                  isptr--;
               }
               *uptr = '\0';

               fputs ("<A HREF=\"mailto:", stdout);
               HtmlPrintString (Url, false);
               fputs ("\">", stdout);
               HtmlPrintString (Url, false);
               fputs ("</A>", stdout);

               /* finished, leave the for(;;) loop */
               break;
            }

            if (WasMailAddress) continue;
         }

         /***************************/
         /* detect a HELP reference */
         /***************************/

         if (*isptr == '$')
         {
            cptr = isptr + 1;
            while (*cptr && isspace(*cptr)) cptr++;

            if (!memcmp (cptr, "HELP", 4) && isspace(cptr[4]))
            {
               *sptr++ = *isptr++; 
               while (*isptr && isspace(*isptr))
                  *sptr++ = *isptr++;

               *sptr = '\0';
               fputs (OutString, stdout);
               zptr = (sptr = OutString) + sizeof(OutString) - ELBOW_ROOM;

               /* don't include the "HELP" in the path */
               cptr = isptr + 4;
               uptr = Url;
               while (*cptr)
               {
                  if (isspace(*cptr))
                     while (isspace(*cptr)) cptr++;
                  else
                  {
                     *uptr++ = '/';
                     while (*cptr && !isspace(*cptr)) *uptr++ = *cptr++;
                  }
               }
               *uptr = '\0';

               fputs ("<A HREF=\"", stdout);
               HtmlPrintString (CgiScriptNamePtr, false);
               HtmlPrintString (Url, false);
               fputs ("\">", stdout);

               uptr = Url;
               while (*isptr) *uptr++ = *isptr++;
               if (uptr > Url)
               {
                  uptr--;
                  isptr--;
               }
               while (uptr > Url && isspace(*uptr))
               {
                  uptr--;
                  isptr--;
               }
               if (!isspace(*uptr))
               {
                  uptr++;
                  isptr++;
               }
               *uptr = '\0';

               HtmlPrintString (Url, false);
               fputs ("</A>", stdout);

               continue;
            }
         }
      }

      /********************/
      /* just a character */
      /********************/

      switch (*isptr)
      {
         case '<' :
            memcpy (sptr, "&lt;", 4); sptr += 4; isptr++; break;
         case '>' :
            memcpy (sptr, "&gt;", 4); sptr += 4; isptr++; break;
         case '&' :
            memcpy (sptr, "&amp;", 5); sptr += 5; isptr++; break;
         case '\"' :
            memcpy (sptr, "&quot;", 6); sptr += 6; isptr++; break;
         default :
            *sptr++ = *isptr++;
      }
   }

   if (sptr > OutString)
   {
      *sptr = '\0';
      fputs (OutString, stdout);
   }
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns number of characters 
copied into HTML string.
*/ 

int CopyIntoHtml
( 
register char *hptr,
register char *sptr,
register int ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = hptr;
   while (*sptr && ccnt--)
   {
      switch (*sptr)
      {
         case '<' : strcpy (hptr, "&lt;"); hptr += 4; sptr++; break;
         case '>' : strcpy (hptr, "&gt;"); hptr += 4; sptr++; break;
         case '&' : strcpy (hptr, "&amp;"); hptr += 5; sptr++; break;
         case '\"' : strcpy (hptr, "&quot;"); hptr += 6; sptr++; break;
         case '\t' : *hptr++ = *sptr++; break;
         case '\n' : *hptr++ = *sptr++; break;
         case '\r' : *hptr++ = *sptr++; break;
         default : if (isprint(*sptr)) *hptr++ = *sptr++; else sptr++;
      }
   }
   *hptr = '\0';
   return (hptr-FirstPtr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.
*/ 

int CopyIntoUri
( 
register char *uptr,
register char *sptr,
register int  ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = uptr;
   while (*sptr && ccnt--)
   {  
      switch (*sptr)
      {
         case '\t' : strcpy (uptr, "%09"); uptr += 3; sptr++; break;
         case ' ' : strcpy (uptr, "%20"); uptr += 3; sptr++; break;
         case '!' : strcpy (uptr, "%21"); uptr += 3; sptr++; break;
         case '\"' : strcpy (uptr, "%22"); uptr += 3; sptr++; break;
         case '#' : strcpy (uptr, "%23"); uptr += 3; sptr++; break;
         case '$' : strcpy (uptr, "%24"); uptr += 3; sptr++; break;
         case '%' : strcpy (uptr, "%25"); uptr += 3; sptr++; break;
         case '&' : strcpy (uptr, "%26"); uptr += 3; sptr++; break;
         case '\'' : strcpy (uptr, "%27"); uptr += 3; sptr++; break;
         case '(' : strcpy (uptr, "%28"); uptr += 3; sptr++; break;
         case ')' : strcpy (uptr, "%29"); uptr += 3; sptr++; break;
         case '*' : strcpy (uptr, "%2a"); uptr += 3; sptr++; break;
         case '+' : strcpy (uptr, "%2b"); uptr += 3; sptr++; break;
         case ',' : strcpy (uptr, "%2c"); uptr += 3; sptr++; break;
         case '-' : strcpy (uptr, "%2d"); uptr += 3; sptr++; break;
         case '.' : strcpy (uptr, "%2e"); uptr += 3; sptr++; break;
         case '/' : strcpy (uptr, "%2f"); uptr += 3; sptr++; break;
         case ':' : strcpy (uptr, "%3a"); uptr += 3; sptr++; break;
         case ';' : strcpy (uptr, "%3b"); uptr += 3; sptr++; break;
         case '<' : strcpy (uptr, "%3c"); uptr += 3; sptr++; break;
         case '=' : strcpy (uptr, "%3d"); uptr += 3; sptr++; break;
         case '>' : strcpy (uptr, "%3e"); uptr += 3; sptr++; break;
         case '?' : strcpy (uptr, "%3f"); uptr += 3; sptr++; break;
         case '[' : strcpy (uptr, "%5b"); uptr += 3; sptr++; break;
         case '\\' : strcpy (uptr, "%5c"); uptr += 3; sptr++; break;
         case ']' : strcpy (uptr, "%5d"); uptr += 3; sptr++; break;
         case '^' : strcpy (uptr, "%5e"); uptr += 3; sptr++; break;
         case '_' : strcpy (uptr, "%5f"); uptr += 3; sptr++; break;
         case '{' : strcpy (uptr, "%7b"); uptr += 3; sptr++; break;
         case '|' : strcpy (uptr, "%7c"); uptr += 3; sptr++; break;
         case '}' : strcpy (uptr, "%7d"); uptr += 3; sptr++; break;
         case '~' : strcpy (uptr, "%7e"); uptr += 3; sptr++; break;
         default : if (isprint(*sptr)) *uptr++ = *sptr++; else sptr++;
      }
   }
   *uptr = '\0';
   return (uptr-FirstPtr);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/
/*
This function provides a simple, uniform mechanism for a C Language script to
access CGI variable values transparently in both the standard CGI and the
CGIplus environments.  Requires the global storage 'CgiPlusEofPtr' to be
non-NULL to indicate a CGIplus environment.

It is completely self-contained. Simply call the function with the CGI variable
name value to be accessed and if that CGI variable exists a pointer to the
value will be returned.  Non-existant variables return CGIPLUS_NONE (see
below).  The values pointed to should NOT be modified in any way (copy the
value if this is required). 

Supplying an empty variable name (i.e. "") will cause the function to block
until the arrival of a request, upon which it just returns a CGIPLUS_NONE
pointer, allowing start-of-request synchronization.  This will also release
allocated memory and reset in preparation for reading the next request, an
alternative to calling with a NULL parameter at the end of request processing
(see below).

For the CGIplus environment supplying a wilcard variable name (i.e. "*")
returns each CGIplus variable string (line from CGIPLUSIN) in turn, with no
more being indicated by a CGIPLUS_NONE pointer.

At the end of a CGIplus request call the function with a NULL parameter to
release allocated memory and reset ready for processing the next request's
variables.  It is not necessary to do this if calling with an empty parameter
to synchronize on the start of a request (see above).

The CGIplus functionality works by initially reading all lines from
CGIPLUSIN until the terminating blank line is encountered. These lines are
stored as a series of contiguous null-terminated strings, terminated by an
empty string, in allocated memory. These strings are searched for the required
variable name and a pointer to its value returned if found.

The value CGIPLUS_NONE shold be set according to the requirements of the
application the function is included within.  If calling functions can handle
NULL pointers being returned then set the value to NULL.  If non-NULL pointers
are prefered then set the value to an empty string.

21-JUN-98  MGD  version 2 of this function attempts to increase efficiency by
                reducing the number of realloc()s required and introducing the
                length of each variable improving search behaviour
01-JUL-97  MGD  version 1
*/

char* CgiVar (char *VarName)

{
#ifndef CGIPLUS_DEBUG
#   define CGIPLUS_DEBUG 0
#endif
/** #  define CGIPLUS_NONE NULL **/
#define CGIPLUS_NONE ""
#define CGIPLUS_LINE_SIZE 1024
#define CGIPLUS_CHUNK 2048

   static int  VarPtrSize = 0;
   static char  *NextPtr = NULL,
                *VarPtr = NULL;

   register char  *cptr, *sptr, *zptr;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

#if CGIPLUS_DEBUG
   fprintf (stdout, "CgiVar() |%s|\n", VarName);
#endif

   if (VarName == NULL || !VarName[0])
   {
      /* end of request, release memory and reinitialize */
      if (VarPtr != NULL)
      {
         free (VarPtr);
         NextPtr = VarPtr = NULL;
      }
      if (VarName == NULL) return (CGIPLUS_NONE);
   }

   if (CgiPlusEofPtr == NULL)
   {
      /* standard CGI environment, get variable from environment */
      if (!VarName[0]) return (CGIPLUS_NONE);
      if ((cptr = getenv(VarName)) == NULL)
         return (CGIPLUS_NONE);
      else
         return (cptr);
   }

   if (VarPtr == NULL)
   {
      /* read lines containing CGIplus variables from <CGIPLUSIN> */
      int  CurrentVarLength;
      char  Line [CGIPLUS_LINE_SIZE];
      char  *LengthPtr;
      FILE  *CgiPlusIn;
                                                 
      CurrentVarLength = VarPtrSize = 0;
      zptr = NULL;
      if ((CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")) == NULL)
         exit (vaxc$errno);
      while (fgets (Line, sizeof(Line), CgiPlusIn) != NULL)
      {
         /* ignore any non-alphabetic line */
         if (Line[0] != '\n' && !isalpha(Line[0])) continue;
         /* point to the next name=value storage in the variable buffer */
         LengthPtr = VarPtr + CurrentVarLength;
         sptr = LengthPtr + sizeof(unsigned short);
         for (cptr = Line; *cptr; *sptr++ = *cptr++)
         {
            if (sptr < zptr) continue;
            /* out of storage (or first variable) get more (or some) memory */
            if ((VarPtr = realloc (VarPtr, VarPtrSize+CGIPLUS_CHUNK)) == NULL)
               exit (vaxc$errno);
            VarPtrSize += CGIPLUS_CHUNK;
            /* recalculate pointers */
            LengthPtr = VarPtr + CurrentVarLength;
            sptr = LengthPtr + sizeof(unsigned short) + (cptr - Line);
            zptr = VarPtr + VarPtrSize;
         }
         /* remove the trailing newline */
         *--sptr = '\0';
         /* insert the length of this name=value pair immediately before it */
         *((unsigned short*)LengthPtr) = Length = cptr - Line;
         /* adjust the value of the current variable storage space used */
         CurrentVarLength += Length + sizeof(unsigned short);
         /* first empty line signals the end of CGIplus variables */
         if (Line[0] == '\n') break;
      }
      fclose (CgiPlusIn);
      if (VarPtr == NULL) return (CGIPLUS_NONE);

#if CGIPLUS_DEBUG
      fprintf (stdout, "VarPtr: %d VarPtrSize: %d CurrentVarLength: %d\n",
               VarPtr, VarPtrSize, CurrentVarLength);
      sptr = VarPtr;
      for (sptr = VarPtr;
           Length = *((unsigned short*)sptr);
           sptr += sizeof(unsigned short) + Length)
         fprintf (stdout, "%3d |%s|\n", Length, sptr+sizeof(unsigned short));
#endif
   }

   /* if just waiting for next request return now it's arrived */
   if (!VarName[0]) return (CGIPLUS_NONE);

   if (VarName[0] == '*')
   {
      /* return each stored line one at a time */
      if (NextPtr == NULL) NextPtr = VarPtr;
      if ((Length = *((unsigned short*)(sptr = NextPtr))) == 0)
      {
         NextPtr = NULL;
#if CGIPLUS_DEBUG
         fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
         return (CGIPLUS_NONE);
      }
      sptr += sizeof(unsigned short);
      NextPtr = sptr + Length;
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr);
#endif
      return (sptr);
   }

   /* search CGIplus variable strings for required variable */
   for (sptr = VarPtr;
        Length = *((unsigned short*)sptr);
        sptr = zptr + Length)
   {
      /* simple comparison between supplied and in-string variable names */
      zptr = sptr = sptr + sizeof(unsigned short);
      cptr = VarName;
      while (*cptr && *sptr && *sptr != '=')
      {
         if (toupper(*cptr) != toupper(*sptr)) break;
         cptr++;
         sptr++;
      }
      if (*cptr || *sptr != '=') continue;
      /* found, return a pointer to the value */
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr+1);
#endif
      return (sptr+1);
   }
   /* not found */
#if CGIPLUS_DEBUG
   fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
   return (CGIPLUS_NONE);
}

/*****************************************************************************/

