/*****************************************************************************/
/*
                                Extract.c


CGI-compliant script to extract a specified range of lines from a plain text
file.

Page layout and colouration may be specified via the appropriate command-line
qualifiers (or corresponding logical/symbol name). Defaults apply for any not
specified.  See "Qualifiers" section below, and also about the logical name or
symbol "QUERY$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"

Don't like explicitly setting a browser's colours?  A colour may be disabled by
setting it to empty.  The following example disables all colours.

  /PBGCOLOR/PBBGCOLOR/PHBGCOLOR/PHTEXT/PLINK/PTEXT/PVLINK

The script can format a page in either of two layouts.

  1. Tables are used to create a coloured header and button bar (DEFAULT).
     Default colours are white page with grey heading and button outlines.
  2. Textual header, horizontal rules and a textual button bar.
     No default colours.

Select other than the default using the following:

  /PLAYOUT=2

Local information may be included in the header. For layout 1 this should be
integrated with the <TABLE> formatted header and to the right of the header
information. Text, an image logo, just about anything could be included. This
is a example of providing a textual form of a local logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><FONT SIZE=-1 COLOR=#ffffff>"-
  "<B>WASD HyperText Services</B></FONT></TD>"

This is an example of providing a local graphical logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><IMG SRC=/local/logo.gif></TD>"

Such local information with layout 2 is included immediately before the header
information at the top of the page.

Button labels are customizable (potentially to non-English language). They
comprise a label, equate symbol and URL-style path suitable for creating a
link. Multiple buttons are separated using the semicolon. Note that any such
button customization must provide escaped HTML-forbidden characters in the
button label and URI-forbidden characters in the path! The backslash
character, "\", escapes characters, including the button-delimitting "=" and
";". There are defaults, see DEFAULT_BUTTONS.

Here is an example of changing the button labels:

  /BUTTON="About=/extract/-/aboutextract.html"

Additional buttons may be created by adding "label=path;" elements to the
button string. In this way an additional information page could be referenced
as follows:

  /BUTTON="About=/extract/-/aboutextract.html;Other Information=/info/"

DIFFICULTY FITTING ALL THESE QUALIFIERS INTO A COMMAND LINE OR LOGICAL?
Use an existing, or create a new, DCL wrapper procedure for the script (same
name and directory) and build up a DCL symbol to provide the information. Up
to 1024 characters can be included in this way. For example:

  $ EXTRACT$PARAM = "/BUTTON=""About=/extract/-/aboutextract.html"""
  $ EXTRACT$PARAM = EXTRACT$PARAM + "/PBGCOLOR/PLINK/PVLINK"
  $ EXTRACT$PARAM = EXTRACT$PARAM + "/PHLOCAL=""<TD VALIGN=top>...</TD>"""
  $ RUN HT_EXE:EXTRACT


LOGICAL NAMES
-------------
EXTRACT$DBUG          turns on all "if (Debug)" statements
EXTRACT$PARAM         equivalent to (overrides) the command line
                      parameters/qualifiers (define as a system-wide logical)


QUALIFIERS
----------
/DBUG           turns on all "if (Debug)" statements
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PBBORDER=      width of button border
/PHBGCOLOR=     heading background color
/PHBORDER=      width of heading and button-bar border
/PHLOCAL=       local information to be included in header
/PHTEXT=        heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour
/TEXT=          list of comma separated TEXT file types (same as QUERY.C)


BUILD DETAILS
-------------
See BUILD_EXTRACT.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
24-JUL-98  MGD  v2.3.1, suppress table background colours if empty
20-MAY-98  MGD  v2.3.0, general maintenance,
                        cosmetic changes
19-SEP-95  MGD  v2.2.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient;
                        added 'previous' and 'next' number of lines link
24-MAY-95  MGD  v2.2.0, minor changes for AXP compatibility
27-MAR-95  MGD  v2.1.0, modifications to CGI interface
05-DEC-94  MGD  v2.0.0, major revision, URL mapping, CGI-like interface
10-JUN-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "EXTRACT AXP-2.3.1";
#else
   char SoftwareID [] = "EXTRACT VAX-2.3.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS-related header files */
#include <descrip.h>
#include <libdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

#define TEXT_FILE_RECORD_SIZE 1024 

/* the first comma delimits an empty file type (e.g. README.;) */
#define DEFAULT_TEXT_TYPES \
",ADA,ASCII,BAS,C,CNF,COB,CONF,COM,CPP,DIS,FOR,JAVA,H,LIS,LOG,MAR,\
PAS,PRO,RELEASE_NOTES,SDML,TEX,TEXT,TXT"

#define DEFAULT_BUTTONS ""

#define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

char  *PageScheme [16];

char  Utility [] = "EXTRACT";

char  Http200Header [] =
"HTTP/1.0 200 Success\n\
Content-Type: text/html\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error\n\
Content-Type: text/html\n\
\n";

boolean  Debug,
         HttpHasBeenOutput;

int  ExtractNumberOfRecords;

char  *ButtonPtr = DEFAULT_BUTTONS,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiRequestMethodPtr,
      *CgiScriptNamePtr,
      *CgiFormExactPtr,
      *CgiFormCasePtr,
      *CgiFormEndPtr,
      *CgiFormHighlightPtr,
      *CgiFormStartPtr,
      *TextFileTypesPtr = DEFAULT_TEXT_TYPES;

/* required prototypes */
char* CopyTextIntoHtml (char*, char*, int);
char* CopyTextIntoUri (char*, char*, int);
char* SearchTextString (char*, char*, boolean);

/*****************************************************************************/
/*
*/

main ()

{
   register char  *cptr;

   boolean  CaseSensitive = false,
            DocumentOnly = false;
   int  status,
        EndRecordNumber,
        StartRecordNumber;

   /*********/
   /* begin */
   /*********/

   if (getenv ("EXTRACT$DBUG") != NULL) Debug = true;

   GetParameters ();

   if (Debug)
      system ("show sym *");
   else
   {
      /* reopen output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
         exit (vaxc$errno);
#else
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "rfm=udf")) == NULL)
         exit (vaxc$errno);
#endif
   }

   SetPageScheme ();

   if (Debug)
      fprintf (stdout, "TextFileTypesPtr |%s|\n", TextFileTypesPtr);
   if (!*TextFileTypesPtr)
   {
      ErrorGeneral ("Text file extensions not configured.",
                    __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /*************************/
   /* get the CGI variables */
   /*************************/

   GetCgiVar (CgiRequestMethodPtr, "WWW_REQUEST_METHOD");
   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethodPtr != 0x00544547)
   {
      fprintf (stdout, "HTTP/1.0 501 Error\n\n");
      exit (SS$_NORMAL);
   }

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");
   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");

   GetCgiVar (CgiFormCasePtr, "WWW_FORM_CASE");
   GetCgiVar (CgiFormEndPtr, "WWW_FORM_END");
   GetCgiVar (CgiFormExactPtr, "WWW_FORM_EXACT");
   GetCgiVar (CgiFormHighlightPtr, "WWW_FORM_HIGHLIGHT");
   GetCgiVar (CgiFormStartPtr, "WWW_FORM_START");

   /***********/
   /* process */
   /***********/

   if (toupper(CgiFormCasePtr[0]) == 'Y')
      CaseSensitive = true;
   else
      CaseSensitive = false;

   CgiFormExactPtr[0] = toupper(CgiFormExactPtr[0]);

   StartRecordNumber = atoi (CgiFormStartPtr);
   EndRecordNumber = atoi (CgiFormEndPtr);

   for (cptr = CgiPathTranslatedPtr; *cptr; cptr++);
   while (*cptr != '.') cptr--;
   if (SameFileType (TextFileTypesPtr, cptr))
   {
      status = ExtractFile (CgiPathTranslatedPtr, CgiPathInfoPtr,
                            CgiFormHighlightPtr,
                            StartRecordNumber, EndRecordNumber,
                            CaseSensitive);

      if (VMSnok (status))
         ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr, __FILE__, __LINE__);
   }
   else
   {
      char  String [512];

      sprintf (String,
"<NOBR>Only plain-text files may be <I>extracted</I> from ...</NOBR> \
&quot;%s&quot;.",
               TextFileTypesPtr);
      ErrorGeneral (String,  __FILE__, __LINE__);
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if ((clptr = getenv ("EXTRACT$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (aptr, "/TEXT=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         TextFileTypesPtr = cptr;
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

SetPageScheme ()

{
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Modified ButtonBar() for extract.  Relies on calling code to set up button
enviroment.   This function just creates the buttons.  See other script for
information on how this function operates.
*/

ExtractButtonBar ()

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ExtractButtonBar()\n");

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';
         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }

      return;
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (PathPtr[0])
            fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
               PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
               PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
      }
   }
}

/*****************************************************************************/
/*
Extract the specified record (line) range from a plain-text file.  If the 
search string is supplied then highlight the first matched string in any line.
*/ 

int ExtractFile
(
char *FileName,
char *UriFileName,
char *HighlightString,
int StartRecordNumber,
int EndRecordNumber,
boolean CaseSensitive
)
{
   register char  *rptr, *sptr;
   register int  RecordNumber = 0;

   int  status,
        NumberOfRecords,
        HighlightStringLength = 0;
   char  ExpandedFileName [256],
         Record [TEXT_FILE_RECORD_SIZE+1],
         String [2048],
         UriHighlightString [256] = "";
   struct FAB  FileFab;
   struct RAB  FileRab;
   struct NAM  FileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ExtractFile() |%s|%s|\n", FileName, HighlightString);

   if (EndRecordNumber > StartRecordNumber)
      NumberOfRecords = EndRecordNumber - StartRecordNumber;
   else
      NumberOfRecords = 0;

   if (HighlightString[0])
   {
      HighlightStringLength = strlen(HighlightString);
      CopyTextIntoUri (UriHighlightString, HighlightString, -1);
   }

   FileFab = cc$rms_fab;
   FileFab.fab$b_fac = FAB$M_GET;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$b_shr = FAB$M_SHRGET;
   FileFab.fab$l_nam = &FileNam;

   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$open (&FileFab, 0, 0)))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      return (status);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   /* 2 buffers and read ahead performance option */
   FileRab.rab$b_mbf = 2;
   FileRab.rab$l_rop = RAB$M_RAH;
   FileRab.rab$l_ubf = Record;
   FileRab.rab$w_usz = sizeof(Record)-1;

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFab, 0, 0);
      return (status);
   }

   /*******************/
   /* extract records */
   /*******************/

   while (VMSok (status = sys$get (&FileRab, 0, 0)))
   {
      RecordNumber++;
      Record[FileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Record |%d|%s|\n", RecordNumber, Record);

      /* terminate at any carriage control that may be in the record */
      for (rptr = Record; *rptr && *rptr != '\r' && *rptr != '\n'; rptr++);
      *rptr = '\0';

      if (StartRecordNumber && RecordNumber < StartRecordNumber) continue;

      if (!HttpHasBeenOutput && !Record[0]) continue;

      if (!HttpHasBeenOutput)
      {
         /**************/
         /* begin page */
         /**************/

         fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"text-types\" CONTENT=\"%s\">\n\
<META NAME=\"file\" CONTENT=\"%s\">\n\
<TITLE>Extract ... %s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
            Http200Header,
            SoftwareID,
            TextFileTypesPtr,
            CgiPathTranslatedPtr,
            CgiPathInfoPtr,
            PageScheme[PS_BODYTAG]);

         if (PageScheme[PS_LAYOUT][0] == '2')
         {
            fprintf (stdout,
"%s<FONT SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>WASDextract</U>\n\
</FONT>\n",
               PageScheme[PS_HEADLOCAL],
               CgiPathInfoPtr);
         }
         else
         {
            fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>WASDextract</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
               PageScheme[PS_HEADBORDER],
               PageScheme[PS_HEADPADDING],
               PageScheme[PS_HEADBGCOLOR],
               PageScheme[PS_HEADTEXT],
               CgiPathInfoPtr,
               PageScheme[PS_HEADTEXT],
               PageScheme[PS_HEADLOCAL]);
         }

         /******************/
         /* top button bar */
         /******************/

         if (PageScheme[PS_LAYOUT][0] == '2')
         {
            fprintf (stdout, "<P>\n<FONT SIZE=-1><NOBR>[");
         }
         else
         {
            fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n\
<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n\
<TR>\n\
<TD ALIGN=center%s><FONT SIZE=-1><NOBR>&nbsp;&nbsp;",
               PageScheme[PS_HEADBORDER],
               PageScheme[PS_HEADBORDER],
               PageScheme[PS_HEADBGCOLOR],
               PageScheme[PS_BUTTONBORDER],
               PageScheme[PS_BUTTONBGCOLOR]);
         }

         if (StartRecordNumber)
         {
            if (EndRecordNumber && StartRecordNumber - NumberOfRecords > 0)
            {
               fprintf (stdout,
"begins at line %d, retrieve \
<A HREF=\"%s%s?highlight=%s&start=%d&end=%d\">previous %d lines</A> or \
<A HREF=\"%s%s?highlight=%s\">entire document</A> \
or <A HREF=\"%s\">file</A>",
                  StartRecordNumber, 
                  CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
                  StartRecordNumber-NumberOfRecords-1,
                  StartRecordNumber-1, NumberOfRecords+1,
                  CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
                  CgiPathInfoPtr);
            }
            else
            {
               fprintf (stdout,
"begins at line %d, retrieve \
<A HREF=\"%s%s?highlight=%s\">entire document</A> \
or <A HREF=\"%s\">file</A>",
                  RecordNumber,
                  CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
                  CgiPathInfoPtr);
            }
         }
         else
         {
            if (EndRecordNumber)
            {
               fprintf (stdout,
"begins at line %d (start of document), retrieve \
<A HREF=\"%s%s?highlight=%s\">entire document</A> \
or <A HREF=\"%s\">file</A>",
                  RecordNumber,
                  CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
                  CgiPathInfoPtr);
            }
            else
               fprintf (stdout,
"entire document, retrieve <A HREF=\"%s\">file</A>",
                  CgiPathInfoPtr);
         }

         if (PageScheme[PS_LAYOUT][0] == '2')
            fprintf (stdout, "]\n");
         else
            fprintf (stdout, "&nbsp;&nbsp;</NOBR></FONT></TD>\n");

         ExtractButtonBar ();

         if (PageScheme[PS_LAYOUT][0] == '2')
         {
            fprintf (stdout,
"</NOBR></FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=95%%>\n\
<PRE>");
         }
         else
         {
            fprintf (stdout,
"</TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
<PRE>");
         }
      }

      fflush (stdout);
      HttpHasBeenOutput = true;

      /******************/
      /* process record */
      /******************/

      if (EndRecordNumber && RecordNumber > EndRecordNumber)
      {
         if (CgiFormExactPtr[0] == 'T' || CgiFormExactPtr[0] == 'Y') break;
         /* if this is an empty (blank) line then end-of-section */
         if (!FileRab.rab$w_rsz) break;
         /* if there is only white-space in this line then end-of-section */
         for (rptr = Record; *rptr && isspace(*rptr); rptr++);
         if (!*rptr) break;
      }

      if (HighlightString[0])
         rptr = SearchTextString (Record, HighlightString, CaseSensitive);
      else
         rptr = "";
      if (*rptr)
      {
         /***********************************/
         /* hit! - Highlight matched string */
         /***********************************/

         /* copy the record up to the first character of the search string */
         sptr = CopyTextIntoHtml (String, Record, rptr-Record);
         /* emphasize the matched search string */
         strcpy (sptr, "<B>"); sptr += 3;
         sptr = CopyTextIntoHtml (sptr, rptr, HighlightStringLength);
         strcpy (sptr, "</B>"); sptr += 4;
         /* rest of record after the matched search string */
         sptr = CopyTextIntoHtml (sptr, rptr+HighlightStringLength, -1);
         *sptr++ = '\n'; *sptr = '\0';
      }
      else
      {
         /**********/
         /* no hit */
         /**********/

         sptr = CopyTextIntoHtml (String, Record, -1);
         *sptr++ = '\n'; *sptr = '\0';
      }

      if (Debug) fprintf (stdout, "String |%s|\n", String);
      fputs (String, stdout);
   }

   /***************/
   /* end of file */
   /***************/

   if (status == RMS$_EOF) status = SS$_NORMAL;
   sys$close (&FileFab, 0, 0);

   /*********************/
   /* bottom button bar */
   /*********************/

   fprintf (stdout, "</PRE>\n");

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"<HR ALIGN=left SIZE=2 WIDTH=95%%>\n\
<FONT SIZE=-1><NOBR>[");
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n\
<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n\
<TR>\n\
<TD ALIGN=center%s><FONT SIZE=-1><NOBR>&nbsp;&nbsp;",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_BUTTONBORDER],
         PageScheme[PS_BUTTONBGCOLOR]);
   }

   if (StartRecordNumber)
   {
      if (EndRecordNumber && RecordNumber > EndRecordNumber)
      {
         fprintf (stdout,
"ends at line %d, \
retrieve <A HREF=\"%s%s?Highlight=%s&start=%d&end=%d\">next %d lines</A> or \
<A HREF=\"%s%s?Highlight=%s&start=%d\">rest of document</A> \
or <A HREF=\"%s\">file</A>",
            RecordNumber-1, 
            CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
            RecordNumber, RecordNumber+NumberOfRecords, NumberOfRecords+1,
            CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString, RecordNumber,
            CgiPathInfoPtr);
      }
      else
      {
         fprintf (stdout,
"ends at line %d (end of document), \
retrieve <A HREF=\"%s%s?Highlight=%s\">entire document</A> \
or <A HREF=\"%s\">file</A>",
            RecordNumber, CgiScriptNamePtr, CgiPathInfoPtr, UriHighlightString,
            CgiPathInfoPtr);
      }
   }
   else
      fprintf (stdout, "entire document, retrieve <A HREF=\"%s\">file</A>",
               CgiPathInfoPtr);

   if (PageScheme[PS_LAYOUT][0] == '2')
      fprintf (stdout, "]\n");
   else
      fprintf (stdout, "&nbsp;&nbsp;</NOBR></FONT></TD>\n");

   ExtractButtonBar ();

   if (PageScheme[PS_LAYOUT][0] == '2')
      fprintf (stdout, "</NOBR></FONT>\n");
   else
   {
      fprintf (stdout,
"</TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");
   }

   fprintf (stdout, "</BODY>\n</HTML>\n");

   return (status);
}

/*****************************************************************************/
/*
Case sensistive or insensitive search of the string pointed to by 'sptr' for 
the string pointed to by 'SearchString'.
*/ 

char* SearchTextString
( 
register char *tptr,
register char *SearchString,
register boolean CaseSensitive
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   sptr = SearchString;
   if (CaseSensitive)
   {
      while (*tptr)
      {
         if (*tptr++ != *sptr) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (*cptr != *sptr) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   else
   {
      while (*tptr)
      {
         if (toupper(*tptr++) != toupper(*sptr)) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (toupper(*cptr) != toupper(*sptr)) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   return (tptr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.
*/ 

char* CopyTextIntoHtml
( 
register char *bptr,
register char *tptr,
register int  ccnt
)
{
    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '\t' : while ((int)(bptr-1) % 8) *bptr++ = ' '; tptr++; break;
          case '<' : strcpy (bptr, "&lt;"); bptr += 4; tptr++; break;
          case '>' : strcpy (bptr, "&gt;"); bptr += 4; tptr++; break;
          case '&' : strcpy (bptr, "&amp;"); bptr += 5; tptr++; break;
          case '\"' : strcpy (bptr, "&quot;"); bptr += 6; tptr++; break;
          default : if (isprint(*tptr)) *bptr++ = *tptr++; else tptr++;
       }
    }
    *bptr = '\0';
    return (bptr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.
*/ 

char* CopyTextIntoUri
( 
register char *bptr,
register char *tptr,
register int  ccnt
)
{
    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '\t' : strcpy (bptr, "%09"); bptr += 3; tptr++; break;
          case ' ' : strcpy (bptr, "%20"); bptr += 3; tptr++; break;
          case '#' : strcpy (bptr, "%23"); bptr += 3; tptr++; break;
          case '+' : strcpy (bptr, "%2b"); bptr += 3; tptr++; break;
          case '/' : strcpy (bptr, "%2f"); bptr += 3; tptr++; break;
          case '=' : strcpy (bptr, "%3d"); bptr += 3; tptr++; break;
          case '?' : strcpy (bptr, "%3f"); bptr += 3; tptr++; break;
          default : if (isprint(*tptr)) *bptr++ = *tptr++; else tptr++;
       }
    }
    *bptr = '\0';
    return (bptr);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber, Text);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber, Text);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      strcpy (Message, "&quot;sys$getmsg() failed&quot;");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/
/*
This function accepts a comma-separated list of file types (extensions, e.g.
"TXT,TEXT,COM,C,PAS,FOR") and a VMS file type (e.g. ".TXT;", ".TXT", "TXT").
It returns true if the file type is in the list, false if not.
*/

boolean SameFileType
(
char *TypeList,
char *FileType
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SameFileType() |%s|%s|\n", TypeList, FileType);

   if (*FileType == '.') FileType++;
   cptr = TypeList;
   while (*cptr)
   {
      sptr = FileType;
      while (*sptr && *sptr != ';' && *cptr && *cptr != ',' &&
             toupper(*sptr) == toupper(*cptr))
      {
         cptr++;
         sptr++;
      }
      if ((!*sptr || *sptr == ';') && (!*cptr || *cptr == ',')) return (true);
      while (*cptr && *cptr != ',') cptr++;
      if (*cptr) cptr++;
   }
   return (false);
}

/*****************************************************************************/

