/****************************************************************************/
/*
                                 HprintS.c

Hyper-PRINT-Server.

CGI-compliant script.

A simple HTTP script to allow point-and-click printing of files from within 
Hypertext documents.  This is limited to specified clients for obvious reasons!
Print queues available provided to the script via command line qualifiers or C
environment variables and  may be specified by assigning a symbol or defining a
logical name.  Three are required:

  1. HPRINTS_PS       list of available PostScript queues and characteristics
  2. HPRINTS_TEXT     ditto for plain text queues
  3. HPRINTS_ALLOWED  list of hosts that may print using the facility

The allowed hosts are specified as a comma-separated list of specifications.
Wildcard "*" and "%" specifications allow access to a range of hosts. Hosts may
be specified using either or both numeric addresses and host names. Hence with
the list "*.wasd.dsto.defence.gov.au,131.185.45.*" the hosts
"ws1.wasd.dsto.defence.gov.au" and "131.185.45.1", etc., can use the facility. 
Numeric addresses may be needed if DNS host name lookup is not enabled on the
server.

The list of queues is comma-separated. The "#" symbol indicates that this
print queue support the "NUMBER_UP=" ScriptPrinter parameter and is capable of
printing more than one output page per sheet. The "+" symbol indicates the
printer supports double-sided printing using the ScriptPrinter "SIDES="
parameter. Assigning symbols in a script DCL procedure:

  $ HPRINTS_TEXT = "LAS2#+,LAS3#,LAS4#,LAS5#,LAS6#,LP1,LP2,LP3"
  $ HPRINTS_PS = "PSLAS2#+,PSLAS3,PSLAS4#,PSLAS5,PSLAS6"
  $ HPRINTS_ALLOWED = "*.wasd.dsto.defence.gov.au,131.185.45.*"

or if logical names are prefered define in the process or system table:

  $ DEFINE /SYSTEM HPRINTS_TEXT "LAS2#+,LAS3#,LAS4#,LAS5#,LAS6#,LP1,LP2,LP3"
  $ DEFINE /SYSTEM HPRINTS_PS "PSLAS2#+,PSLAS3,PSLAS4#,PSLAS5,PSLAS6"
  $ DEFINE /SYSTEM HPRINTS_ALLOWED "*.wasd.dsto.defence.gov.au,131.185.45.*"

Page colouration may be specified via the appropriate command-line qualifiers
(or corresponding logical name). Defaults for any not specified.  Specifiy as
/WHATEVER="" to NOT specify the corresponding colour (i.e. leave it to the
browser). See "Qualifiers" section below, and also about the logical name
"HPRINTS$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"

The script can format a page in either of two layouts.

  1. Tables are used to create a coloured header and button bar (DEFAULT).
     Default colours are white page with grey heading and button outlines.
  2. Textual header, horizontal rules and a textual button bar.
     No default colours.


QUALIFIERS
----------
/ALLOWED=       list of allowed hosts/domains
/DBUG           turns on all "if (Debug)" statements
/POSTSCRIPT=    list of PostScript print queues
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PBBORDER=      width of button border
/PHBGCOLOR=     heading background color
/PHBORDER=      width of heading and button-bar border
/PHLOCAL=       local information to be included in header
/PHTEXT=        heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour
/TEXT=          list of text print queues


LOGICAL NAMES
-------------
HPRINTS$DBUG           turns on all "if (Debug)" statements
HPRINTS$PARAM          equivalent to (overrides) the command line
                       parameters/qualifiers (define as a system-wide logical)


BUILD DETAILS
-------------
See BUILD_HPRINTS.COM procedure.


VERSION HISTORY
---------------
24-JUL-98  MGD  v3.3.1, suppress table background colours if empty
29-APR-98  MGD  v3.3.0, some tidying up and cosmetic changes
09-SEP-97  MGD  v2.2.2, bugfix; to "job submitted" page
16-AUG-97  MGD  v2.2.1, bugfix; to SearchTextString()
28-JUN-97  MGD  v2.2.0, updated and made slightly more generic
19-SEP-95  MGD  v2.1.1, replace <CR><LF> carriage-control with single newline
24-MAY-95  MGD  v2.1.0, minor changes for AXP compatibility
29-MAR-95  MGD  v2.0.0, modified for CGI-compliance
05-DEC-94  MGD  v1.1.0, new URL mapping functions, minor revisions
12-SEP-94  MGD  v1.0.0, initial development
*/
/****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "HPRINTS AXP-3.3.1";
#else
   char SoftwareID [] = "HPRINTS VAX-3.3.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS-related header files */
#include <descrip.h>
#include <libdef.h>
#include <quidef.h>
#include <rmsdef.h>
#include <sjcdef.h>
#include <ssdef.h>
#include <stsdef.h>

#define JBC$_NOSUCHQUE 294970

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0

#define VMSok(x) (x) & STS$M_SUCCESS
#define VMSnok(x) !((x) & STS$M_SUCCESS)

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

#define DEFAULT_BUTTONS ""

/* this macro just plugs in some script-specific code into ButtonBar() */
#define SCRIPT_SPECIFIC_BUTTON_CODE /* none for this script */

#define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

char  *PageScheme [16];

char Http200Header [] =
"HTTP/1.0 200 Success\r\n\
Content-Type: text/html\r\n\
\r\n";

char Http404Header [] =
"HTTP/1.0 404 Error\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Utility [] = "HPRINTS";

boolean  Debug,
         HttpHasBeenOutput;

char  *AllowedListPtr,
      *ButtonPtr = DEFAULT_BUTTONS,
      *CgiFormQueuePtr,
      *CgiFormPagesPtr,
      *CgiFormSidesPtr,
      *CgiFormMenuPtr,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiRemoteAddrPtr,
      *CgiRemoteHostPtr,
      *CgiRequestMethodPtr,
      *CgiScriptNamePtr,
      *PostScriptQueueListPtr,
      *TextQueueListPtr;
      
/* required function prototypes */
char* SearchTextString (char*, char*, boolean, int*);

/****************************************************************************/
/*
*/

int main ()

{
   register int  acnt;
   register char  *cptr, *sptr;

   int  status;
   FILE  *PrintFile;

   /*********/
   /* begin */
   /*********/

   if (getenv ("HPRINTS$DBUG") != NULL) Debug = true;

   GetParameters ();

   if (Debug)
      system ("show sym *");
   else
   {
      /* reopen output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
         exit (vaxc$errno);
#else
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "rfm=udf")) == NULL)
         exit (vaxc$errno);
#endif
   }

   SetPageScheme ();

   if (PostScriptQueueListPtr == NULL)
   {
      if ((PostScriptQueueListPtr = getenv("HPRINTS_PS")) == NULL)
      {
         ErrorGeneral ("PostScript queue list not available.",
                       __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   if (TextQueueListPtr == NULL)
   {
      if ((TextQueueListPtr = getenv("HPRINTS_TEXT")) == NULL)
      {
         ErrorGeneral ("Plain text queue list not available.",
                       __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   if (AllowedListPtr == NULL)
   {
      if ((AllowedListPtr = getenv("HPRINTS_ALLOWED")) == NULL)
      {
         ErrorGeneral ("Allowed host list not available.",
                       __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   /*************************/
   /* get the CGI variables */
   /*************************/

   GetCgiVar (CgiRemoteAddrPtr, "WWW_REMOTE_ADDR");
   GetCgiVar (CgiRemoteHostPtr, "WWW_REMOTE_HOST");

   if (!AcceptClient ()) exit (SS$_NORMAL);

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");
   GetCgiVar (CgiRequestMethodPtr, "WWW_REQUEST_METHOD");
   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");

   GetCgiVar (CgiFormQueuePtr, "WWW_FORM_QUEUE");
   GetCgiVar (CgiFormPagesPtr, "WWW_FORM_PAGES");
   GetCgiVar (CgiFormSidesPtr, "WWW_FORM_SIDES");
   GetCgiVar (CgiFormMenuPtr, "WWW_FORM_MENU");

   /***********/
   /* process */
   /***********/

   if (!CgiPathInfoPtr[0] || (CgiPathInfoPtr[0] == '/' && !CgiPathInfoPtr[1]))
   {
      ErrorGeneral ("Please specify a document path.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /* check the file exists */
   if ((PrintFile = fopen (CgiPathTranslatedPtr, "r", "shr=get")) == NULL)
   {
      status = vaxc$errno;
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   fclose (PrintFile);

   /*
      If a print queue parameter was supplied then it is being submitted
      for printing.  If no print queue was supplied then it is requesting
      a print queue be selected.
   */
   if (CgiFormQueuePtr[0])
      SubmitPrintJob ();
   else
   {
      if (CgiFormMenuPtr[0])
      {
         ErrorGeneral ("Please select a print queue.", __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      SelectPrinter ();
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if ((clptr = getenv ("HPRINTS$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/ALLOWED=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         AllowedListPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (aptr, "/POSTSCRIPT=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         PostScriptQueueListPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/TEXT=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         TextQueueListPtr = cptr;
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

SetPageScheme ()

{
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Provides a divider for top and bottom of the content of the page. This can be
a coloured bar (using <TABLE>) or a horizontal rule depending on the page
layout. "Buttons" providing script-internal and/or additional user-specified
links ('ButtonPtr' string) can be placed with(in) this bar. All button labels
are derived from 'ButtonPtr', with script-internal buttons using link-paths
set up via 'ButtonInternal[]' array, or any user-specified path depending on
requirement. An empty path (i.e. PathPtr[0] == '\0') obviously does not have a
link created, it just displays the button label. For a button-bar at the top
of the document use 1, bottom of the document use 2, and for just a bar with
no buttons at all use 0.
*/

ButtonBar (int Top1Bottom2)

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ButtonBar() %d\n", Top1Bottom2);

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';
         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (Top1Bottom2)
   {
      /***********************************/
      /* set up script-specified buttons */
      /***********************************/

      SCRIPT_SPECIFIC_BUTTON_CODE
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      if (Top1Bottom2 == 2 || !Top1Bottom2)
      {
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=90%%>\n");
         if (!Top1Bottom2) return;
      }

      fprintf (stdout, "<FONT SIZE=-1><NOBR>\n");
      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (idx) fprintf (stdout, "&nbsp;");
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }
      fprintf (stdout, "</NOBR></FONT>\n");

      if (Top1Bottom2 == 1)
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=90%%>\n");
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR]);

      if (ButtonCount == 0 || !Top1Bottom2)
         fprintf (stdout, "&nbsp;\n");
      else
      {
         fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n",
            PageScheme[PS_BUTTONBORDER]);

         for (idx = 0; idx < ButtonCount; idx++)
         {
            if (ButtonInternal[idx] == NULL)
               PathPtr = ButtonPath[idx];
            else
               PathPtr = ButtonInternal[idx];
            if (PathPtr[0])
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
            else
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
         }

         fprintf (stdout, "</TR></TABLE>\n");
      }

      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n");
   }
}

/****************************************************************************/
/*
Check if this system belongs to any of the specified domains.  Return true if
it does.   Send an explanatory message to the client if it doesn't, and return
false.
*/

boolean AcceptClient ()

{
   register char  *cptr, *lptr, *sptr;

   char  Message [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AcceptClient()\n");

   /* return if anyone may use it (this could be dangerous!) */
   if (!*(lptr = AllowedListPtr)) return (true);

   while (*lptr)
   {
      sptr = lptr;
      while (*lptr && *lptr != ',') lptr++;
      if (*lptr) *lptr++ = '\0';
      /* match against host address or name */
      if (isdigit(*sptr))
         cptr = CgiRemoteAddrPtr;
      else
         cptr = CgiRemoteHostPtr;
      if (Debug) fprintf (stdout, "|%s|%s|\n", cptr, sptr);
      if (SearchTextString (cptr, sptr, false, NULL) != NULL) return (true);
   }

   sprintf (Message,
      "Sorry! &nbsp;This facility is not available to <TT>%s</TT>",
      CgiRemoteHostPtr);
   ErrorGeneral (Message, __FILE__, __LINE__);
   return (false);
}

/*****************************************************************************/
/*
String search allowing wildcard "*" (matching any multiple characters) and "%" 
(matching any single character).  Returns NULL if not found or a pointer to
start of matched string.
*/ 

char* SearchTextString
( 
register char *InThat,
register char *This,
register boolean CaseSensitive,
int *MatchedLengthPtr
)
{
/* wildcards implied at both ends of the search string */
#define IMPLIED_WILDCARDS 0

   register char  *cptr, *sptr, *inptr;
   char  *RestartCptr,
         *RestartInptr,
         *MatchPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchTextString()\n|%s|%s|\n", This, InThat);

   if (MatchedLengthPtr != NULL) *MatchedLengthPtr = 0;
   if (!*(cptr = This)) return (NULL);
   inptr = MatchPtr = InThat;

#if IMPLIED_WILDCARDS
   /* skip leading text up to first matching character (if any!) */
   if (*cptr != '*' && *cptr != '%')
   {
      if (CaseSensitive)
         while (*inptr && *inptr != *cptr) inptr++;
      else
         while (*inptr && toupper(*inptr) != toupper(*cptr)) inptr++;
      if (Debug && !*inptr) fprintf (stdout, "1. NOT matched!\n");
      if (!*inptr) return (NULL);
      cptr++;
      MatchPtr = inptr++;
   }
#endif /* IMPLIED_WILDCARDS */

   for (;;)
   {
      if (CaseSensitive)
      {
         while (*cptr && *inptr && *cptr == *inptr)
         {
            cptr++;
            inptr++;
         }
      }
      else
      {
         while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
         {
            cptr++;
            inptr++;
         }
      }

#if IMPLIED_WILDCARDS
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "1. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
#else
      if (!*cptr && !*inptr)
      {
         if (Debug) fprintf (stdout, "2. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
      else
      if (*cptr != '*' && *cptr != '%')
         return (NULL);
#endif /* IMPLIED_WILDCARDS */

      if (Debug && !*inptr) fprintf (stdout, "3. NOT matched!\n");
      if (*cptr != '*' && *cptr != '%' && !*inptr) return (NULL);

      if (*cptr != '*' && *cptr != '%')
      {
         cptr = This;
         MatchPtr = ++inptr;
         continue;
      }

      if (*cptr == '%')
      {
         /* single char wildcard processing */
         if (!*inptr) break;
         cptr++;
         inptr++;
         continue;
      }

      /* asterisk wildcard matching */
      while (*cptr == '*') cptr++;

      /* an asterisk wildcard at end matches all following */
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "4. matched!\n");
         while (*inptr) inptr++;
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }

      /* note the current position in the string (first after the wildcard) */
      RestartCptr = cptr;
      for (;;)
      {
         /* find first char in InThat matching char after wildcard */
         if (CaseSensitive)
            while (*inptr && *cptr != *inptr) inptr++;
         else
            while (*inptr && toupper(*cptr) != toupper(*inptr)) inptr++;
         /* if did not find matching char in InThat being searched */
         if (Debug && !*inptr) fprintf (stdout, "5. NOT matched!\n");
         if (!*inptr) return (NULL);
         /* note the current position in InThat being searched */
         RestartInptr = inptr;
         /* try to match the remainder of the string and InThat */
         if (CaseSensitive)
         {
            while (*cptr && *inptr && *cptr == *inptr)
            {
               cptr++;
               inptr++;
            }
         }
         else
         {
            while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
            {
               cptr++;
               inptr++;
            }
         }
         /* if reached the end of both string and InThat - match! */
#if IMPLIED_WILDCARDS
         if (!*cptr)
         {
            if (Debug) fprintf (stdout, "6. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#else
         if (!*cptr && !*inptr)
         {
            if (Debug) fprintf (stdout, "7. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#endif /* IMPLIED_WILDCARDS */
         /* break to the external loop if we encounter another wildcard */
         if (*cptr == '*' || *cptr == '%') break;
         /* lets have another go */
         cptr = RestartCptr;
         /* starting the character following the previous attempt */
         inptr = MatchPtr = RestartInptr + 1;
      }
   }
}

/****************************************************************************/
/*
Using the <FORM> tag provide the client with a list of print queues selectable 
for the print job.
*/ 

SelectPrinter ()

{
   register char  *cptr, *sptr, *zptr;

   boolean  NumberUpSupported = false,
            SidesTwoSupported = false;
   char  Comment [256],
         PrintQueueName [256];
   char  *QueueList;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SelectPrinter()\n");

   /* determine file extension and associated queue type */
   for (cptr = sptr = CgiPathInfoPtr; *cptr; cptr++)
      if (*cptr == '/') sptr = cptr;
   while (*sptr && *sptr != '.') sptr++;
   if (strsame (sptr, ".PS", -1))
      QueueList = PostScriptQueueListPtr;
   else
      QueueList = TextQueueListPtr;

   /**************************************/
   /* HTTP header and HTML start of form */
   /**************************************/

   fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<TITLE>Printing - %s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      Http200Header,
      SoftwareID,
      CgiPathInfoPtr,
      PageScheme[PS_BODYTAG]);
   HttpHasBeenOutput = true;

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s\
<FONT SIZE=+2>\n\
Select Print Queue\n\
</FONT>\n\
<BR>\
<FONT SIZE=+1>\n\
Printing %s\n\
</FONT>\n\
<BR>\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>hPRINTs</U>\n\
</FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=90%%>\n",
      PageScheme[PS_HEADLOCAL],
      CgiPathInfoPtr);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n\
<FONT COLOR=\"%s\" SIZE=+2>\n\
Select Print Queue\n\
</FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=+1>\n\
Printing %s\n\
</FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>hPRINTs</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
      PageScheme[PS_HEADPADDING],
      PageScheme[PS_HEADBGCOLOR],
      PageScheme[PS_HEADTEXT],
      PageScheme[PS_HEADTEXT],
      CgiPathInfoPtr,
      PageScheme[PS_HEADTEXT],
      PageScheme[PS_HEADLOCAL]);
   }

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"menu\" VALUE=\"yes\">\n",
      CgiScriptNamePtr, CgiPathInfoPtr);

   /****************************************************/
   /* check if any printer queues support fancy stuff! */
   /****************************************************/

   cptr = QueueList;
   while (*cptr)
   {
      sptr = PrintQueueName;
      while (*cptr && *cptr != ',' && *cptr != '#' && *cptr != '+')
          *sptr++ = *cptr++;
      *sptr = '\0';

      while (*cptr == '#' || *cptr == '+')
      {
         if (*cptr == '#') NumberUpSupported = true;
         if (*cptr == '+') SidesTwoSupported = true;
         cptr++;
      }

      /* step over the comma */
      if (*cptr) cptr++;
   }

   if (NumberUpSupported)
   {
      fprintf (stdout,
"<P>On supported printers ...\n\
<P><INPUT TYPE=checkbox NAME=\"pages\" VALUE=\"2\">\
 Check this box to <B>conserve paper</B> (2 document pages per sheet).\n");
   }

   if (NumberUpSupported && SidesTwoSupported)
      fputs ("<BR>", stdout);
   else
      fputs ("<P>On supported printers ...\n<P>", stdout);

   if (SidesTwoSupported)
   {
      fprintf (stdout,
"<INPUT TYPE=checkbox NAME=\"sides\" VALUE=\"2\">\
 Check this box to <B>print on both sides</B> of each sheet.\n");
   }

   fputs ("<P><HR ALIGN=left SIZE=2 WIDTH=45%%>\n", stdout);

   /******************************************/
   /* provide a radio box selector of queues */
   /******************************************/

   cptr = QueueList;
   while (*cptr)
   {
      sptr = PrintQueueName;
      while (*cptr && *cptr != ',' && *cptr != '#' && *cptr != '+')
          *sptr++ = *cptr++;
      *sptr = '\0';

      if (*cptr == '#' || *cptr == '+')
      {
         strcpy (zptr = Comment, " &nbsp;&nbsp;<FONT SIZE=-1>supports ");
         while (*zptr) zptr++;
         while (*cptr == '#' || *cptr == '+')
         {
            if (zptr > Comment+36) *zptr++ = ',';
            if (*cptr == '#')
            {
               strcpy (zptr, " multipage");
               while (*zptr) zptr++;
               cptr++;
               continue;
            }
            if (*cptr == '+')
            {
               strcpy (zptr, " double-sided");
               while (*zptr) zptr++;
               cptr++;
               continue;
            }
         }
         strcpy (zptr, "</FONT>");
      }
      else
         Comment[0] = '\0';

      fprintf (stdout,
      "<BR><INPUT TYPE=radio NAME=\"queue\" VALUE=\"%s\"> %s%s\n",
      PrintQueueName, PrintQueueName, Comment);

      /* step over the comma */
      if (*cptr) cptr++;
   }

   /***************************/
   /* rest of HTML form, etc. */
   /***************************/

   fprintf (stdout,
"<P><HR SIZE=2 ALIGN=left WIDTH=45%%>\n\
<P>\n\
<TABLE CELLPADDING=5 CELLSPACING=0 BORDER=0>\n\
<TR><TD>\n\
<INPUT TYPE=submit VALUE=\" Print \"> \
</TD><TD>\n\
<B>To cancel this print</B> use your browser's \
&quot;BACK&quot; button/keystroke to <B>navigate backwards</B>.\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
</BLOCKQUOTE>\n");

   ButtonBar (2);

   fprintf (stdout,
"</BODY>\n\
</HTML>\n");
}

/*****************************************************************************/
/*
Using system service $SNDJBCW, submit the job for printing.
*/

SubmitPrintJob ()

{
   unsigned short  Length; 
   int  status;
   char  JobStatusOutput [256];
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } items [4], NullItem = {0,0,0,0};
   int  icnt;
   struct VMSiosb {
      int  Status;
      int  DeviceInfo;
   } IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SubmitPrintJob() |%s|\n", CgiPathTranslatedPtr);

   icnt = 0;

   items[icnt].BufferLength = strlen(CgiFormQueuePtr);
   items[icnt].ItemCode = SJC$_QUEUE;
   items[icnt].BufferPtr = CgiFormQueuePtr;
   items[icnt++].LengthPtr = 0;

   if (CgiFormPagesPtr[0] == '2')
   {
      items[icnt].BufferLength = 11;
      items[icnt].ItemCode = SJC$_PARAMETER_1;
      items[icnt].BufferPtr = "NUMBER_UP=2";
      items[icnt++].LengthPtr = 0;
   }

   if (CgiFormSidesPtr[0] == '2')
   {
      items[icnt].BufferLength = 9;
      items[icnt].ItemCode = SJC$_PARAMETER_2;
      items[icnt].BufferPtr = "SIDES=TWO";
      items[icnt++].LengthPtr = 0;
   }

   items[icnt] = NullItem;

   status = sys$sndjbcw (0, SJC$_CREATE_JOB, 0, &items, &IOsb, 0, 0);

   if (VMSok (status)) status = IOsb.Status;
   if (VMSnok (status))
   {
      if (status == JBC$_NOSUCHQUE)
         ErrorVmsStatus (status, CgiFormQueuePtr, CgiPathTranslatedPtr,
                         __FILE__, __LINE__);
      else
         ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
                         __FILE__, __LINE__);
      return (status);
   }

   icnt = 0;
   items[icnt].BufferLength = strlen(CgiPathTranslatedPtr);
   items[icnt].ItemCode = SJC$_FILE_SPECIFICATION;
   items[icnt].BufferPtr = CgiPathTranslatedPtr;
   items[icnt++].LengthPtr = 0;
   items[icnt] = NullItem;

   status = sys$sndjbcw (0, SJC$_ADD_FILE, 0, &items, 0, 0, 0);

   if (VMSok (status)) status = IOsb.Status;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr, __FILE__, __LINE__);
      return (IOsb.Status);
   }

   icnt = 0;
   items[icnt].BufferLength = sizeof(JobStatusOutput)-1;
   items[icnt].ItemCode = SJC$_JOB_STATUS_OUTPUT;
   items[icnt].BufferPtr = JobStatusOutput;
   items[icnt++].LengthPtr = &Length;
   items[icnt] = NullItem;

   status = sys$sndjbcw (0, SJC$_CLOSE_JOB, 0, &items, 0, 0, 0);

   if (VMSok (status)) status = IOsb.Status;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr, __FILE__, __LINE__);
      return (status);
   }

   JobStatusOutput[Length] = '\0';

   /***************************************/
   /* advise the client of the job status */
   /***************************************/

   fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<TITLE>Printing - %s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      Http200Header,
      SoftwareID,
      CgiPathInfoPtr,
      PageScheme[PS_BODYTAG]);
   HttpHasBeenOutput = true;

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s\
<FONT SIZE=+2>\n\
Print Job Submitted\n\
</FONT>\n\
<BR>\
<FONT SIZE=+1>\n\
Printing %s to %s\n\
</FONT>\n\
<BR>\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>hPRINTs</U>\n\
</FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=90%%>\n",
      PageScheme[PS_HEADLOCAL],
      CgiPathInfoPtr, CgiFormQueuePtr);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n\
<FONT COLOR=\"%s\" SIZE=+2>\n\
Print Job Submitted\n\
</FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=+1>\n\
Printing %s to %s\n\
</FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>hPRINTs</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
      PageScheme[PS_HEADPADDING],
      PageScheme[PS_HEADBGCOLOR],
      PageScheme[PS_HEADTEXT],
      PageScheme[PS_HEADTEXT],
      CgiPathInfoPtr, CgiFormQueuePtr,
      PageScheme[PS_HEADTEXT],
      PageScheme[PS_HEADLOCAL]);
   }

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<P>\n\
<PRE>%s</PRE>\n\
<P>\n\
Do <B>not</B> use your browser's &quot;RELOAD&quot; button/keystroke, \n\
it will cause another print to be submitted!  Use your browser's\n\
&quot;BACK&quot; button/keystroke to <B>navigate backwards</B>.\n\
</BLOCKQUOTE>\n\
<P>\n",
      JobStatusOutput);

   ButtonBar (2);

   fprintf (stdout,
"</BODY>\n\
</HTML>\n");

   return (status);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=2>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber, Text);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber, Text);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      strcpy (Message, "&quot;sys$getmsg() failed&quot;");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=2>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/

