/*****************************************************************************/
/*
                                 Admin.c

Server administration support functions for HTTPd.  Many of the reports and
forms used for revision are produced by functions within the core-function
module.  The control function allow server restart, exit, abort, mapping file
reload, authorization path reload and authentication cache purging.


VERSION HISTORY
---------------
16-MAR-98  MGD  server abort changed to "exit NOW", added "restart NOW"
05-OCT-97  MGD  file cache
28-SEP-97  MGD  request durations
06-SEP-97  MGD  service list
09-AUG-97  MGD  message database
07-JUL-97  MGD  activity report and logging control functions,
                reworked the control function (it was getting untidy)
08-JUN-97  MGD  "Other" menu item, with "Clients" and "Scripting" reports,
                AdminRequestReport() and DclScriptingReport() functions
01-FEB-97  MGD  new for HTTPd version 4
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <prvdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "admin.h"
#include "dcl.h"
#include "error.h"
#include "HTAdmin.h"
#include "net.h"
#include "msg.h"
#include "support.h"

/******************/
/* global storage */
/******************/

char  ErrorLoggingNotEnabled [] = "Logging is not currently enabled.",
      ErrorLoggingAlreadyEnabled [] = "Logging is already enabled.";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  ControlExitRequested;
extern boolean  ControlRestartRequested;
extern boolean  LoggingEnabled;
extern boolean  MonitorEnabled;
extern boolean  ProtocolHttpsConfigured;
extern int  CurrentConnectCount;
extern int  ExitStatus;
extern int  RequestHistoryMax;
extern char  BuildInfo[];
extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  TimeGmtString[];
extern char  Utility[];
extern struct AccountingStruct Accounting;
extern struct AuthLoadStruct  ServerAuthLoad;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;
extern struct ServiceStruct  *ServiceListHead;

extern boolean  CacheEnabled;
extern int  CacheHits0,
            CacheHits10,
            CacheHits100,
            CacheHits100plus,
            CacheHitCount,
            CacheLoadCount,
            CacheMemoryInUse,
            CacheTotalKBytesMax;

/*****************************************************************************/
/*
*/ 

int AdminBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   register char  *cptr, *sptr, *qptr, *zptr;

   boolean  UseServerDatabase;
   int  status;
   unsigned short  Length;
   char  Buffer [1024],
         FieldName [128],
         FieldValue [256],
         FilePath [256],
         Path [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AdminBegin()\n");

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoServerAdminCount;

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /********************************************/
   /* modules that parse their own query field */
   /********************************************/

   if (strsame (rqptr->PathInfoPtr, HttpdInternalReviseAuth,
                sizeof(HttpdInternalReviseAuth)-1))
   {
      HTAdminBegin (rqptr, NextTaskFunction);
      return;
   }

   if (strsame (rqptr->PathInfoPtr, HttpdInternalReportActivity, -1))
   {
      GraphActivityReport (rqptr, NextTaskFunction);
      return;
   }

   if (strsame (rqptr->PathInfoPtr, HttpdInternalGraphicActivity, -1))
   {
      GraphActivityPlotBegin (rqptr, NextTaskFunction);
      return;
   }

   /**********************/
   /* parse query string */
   /**********************/

   UseServerDatabase = true;
   FilePath[0] = Path[0] = '\0';

   qptr = rqptr->QueryStringPtr;
   while (*qptr)
   {
      qptr = ParseQueryField (rqptr, qptr,
                              FieldName, sizeof(FieldName),
                              FieldValue, sizeof(FieldValue),
                              FI_LI);
      if (qptr == NULL)
      {
         /* error occured */
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      if (strsame (FieldName, "file", -1))
      {
         if (toupper(FieldValue[0]) == 'Y')
            UseServerDatabase = false;
         else
         if (toupper(FieldValue[0]) == 'N')
            UseServerDatabase = true;
      }
      else
      if (strsame (FieldName, "server", -1))
      {
         if (toupper(FieldValue[0]) == 'Y')
            UseServerDatabase = true;
         else
         if (toupper(FieldValue[0]) == 'N')
            UseServerDatabase = false;
      }
      else
      if (strsame (FieldName, "edit", -1))
         strcpy (FilePath, FieldValue);
      else
      if (strsame (FieldName, "path", -1))
         strcpy (Path, FieldValue);
      else
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, "Unknown query field.", FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
   }

   /***************/
   /* do function */
   /***************/

   if (strsame (rqptr->PathInfoPtr, HttpdInternalAdmin, -1) ||
       strsame (rqptr->PathInfoPtr, HttpdInternalReport, -1) ||
       strsame (rqptr->PathInfoPtr, HttpdInternalRevise, -1) ||
       strsame (rqptr->PathInfoPtr, HttpdInternalControl, -1))
   {
      /********/
      /* menu */
      /********/

      AdminMenu (rqptr, NextTaskFunction);
      return;
   }
   else
   if (strsame (rqptr->PathInfoPtr, HttpdInternalReport,
                sizeof(HttpdInternalReport)-1))
   {
      /***********/
      /* reports */
      /***********/

      cptr = rqptr->PathInfoPtr + sizeof(HttpdInternalReport)-1;

      if (strsame (cptr, "auth", -1) ||
          strsame (cptr, "auth?", 5))
      {
         AuthCacheTreeReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "cache", -1))
      {
         CacheReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "config", -1) ||
          strsame (cptr, "config?", 7))
      {
         ConfigReport (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "DCL", -1))
      {
         DclScriptingReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "DECnet", -1))
      {
         DECnetScriptingReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "memory", -1))
      {
         VmReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "messages", -1) ||
          strsame (cptr, "messages?", 9))
      {
         MsgReport (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "paths", -1) ||
          strsame (cptr, "paths?", 6))
      {
         AuthPathReport (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "request", -1))
      {
         RequestReport (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "stats", -1))
      {
         AdminReportServerStats (rqptr, NextTaskFunction);
         return;
      }
      else
      if (strsame (cptr, "rules", -1) ||
          strsame (cptr, "rules?", 6))
      {
         if (Path[0])
            MapUrl_ReportRuleCheck (rqptr, NextTaskFunction, Path,
                                    UseServerDatabase);
         else
            MapUrl_Report (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "SSL", -1))
      {
         SeSoLaReport (rqptr, NextTaskFunction);
         return;
      }
   }
   else
   if (strsame (rqptr->PathInfoPtr, HttpdInternalRevise,
                sizeof(HttpdInternalRevise)-1))
   {
      /**************************/
      /* configuration/revision */
      /**************************/

      cptr = rqptr->PathInfoPtr + sizeof(HttpdInternalRevise)-1;

      if (strsame (cptr, "config", -1) ||
          strsame (cptr, "config?", 7))
      {
         if (FilePath[0])
            UpdEditFileBegin (rqptr, NextTaskFunction);
         else
            ConfigRevise (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "paths", -1) ||
          strsame (cptr, "paths?", 6))
      {
         AuthPathReport (rqptr, NextTaskFunction, UseServerDatabase);
         return;
      }
      else
      if (strsame (cptr, "rules", -1) ||
          strsame (cptr, "rules?", 6))
      {
         MapUrl_ReportRuleCheck (rqptr, NextTaskFunction, Path,
                                 UseServerDatabase);
         return;
      }
   }
   else
   if (strsame (rqptr->PathInfoPtr, HttpdInternalControl,
                Length = sizeof(HttpdInternalControl)-1))
   {
      /******************/
      /* server control */
      /******************/

      AdminControl (rqptr, NextTaskFunction, rqptr->PathInfoPtr+Length);
      return;
   }

   /********************/
   /* unknown function */
   /********************/

   rqptr->ResponseStatusCode = 403;
   ErrorGeneral (rqptr, "Unknown function.", FI_LI);
   SysDclAst (NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Provide a menu of HTTPd server administration functions.
*/ 

AdminMenu
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Server Administration Menu</TITLE>\n\
</HEAD>\n\
<BODY LINK=\"#0000cc\" VLINK=\"#0000cc\">\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Server Administration Menu</H2>\n\
<P>\n\
\
<TABLE CELLPADDING=5 CELLSPACING=1 BORDER=1>\n\
\
<TR><TH COLSPAN=2>Configuration</TH><TR>\n\
\
<TD COLSPAN=2 VALIGN=top>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
\
<TR><TD></TD><TD>|</TD>\
<TH COLSPAN=2>Report</TH><TD>|</TD>\n\
<TH COLSPAN=3>Revise</TH><TD>|</TD>\n\
<TH>&nbsp;Action<SUP>*</SUP></TH><TD>|</TD><TR>\n\
\
<TR>\n\
<TH>Statistics</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZstats?\">Server</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZzero?\">Zero</A>]</TD>\n\
</TR>\n\
\
<TR>\n\
<TH>Configuration</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZconfig?server=yes\">Server</A>]</TD>\n\
<TD>[<A HREF=\"!AZconfig?file=yes\">File</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZconfig?server=yes\">Server</A>]</TD>\n\
<TD>[<A HREF=\"!AZconfig?file=yes\">File</A>]</TD>\n\
<TD>[<A HREF=\"!AZ!AZconfig\">Edit</A>]</TD>\n\
</TR>\n\
\
<TR>\n\
<TH>Messages</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZmessages?server=yes\">Server</A>]</TD>\n\
<TD>[<A HREF=\"!AZmessages?file=yes\">File</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZ!AZmessages\">Edit</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
</TR>\n\
\
<TR>\n\
<TH>Mapping Rules</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZrules?server=yes\">Server</A>]</TD>\n\
<TD>[<A HREF=\"!AZrules?file=yes\">File</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZ!AZrules\">Edit</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZrules\">Reload</A>]</TD>\n\
</TR>\n\
\
<TR>\n\
<TH>Path Authorization</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZpaths?server=yes\">Server</A>]</TD>\n\
<TD>[<A HREF=\"!AZpaths?file=yes\">File</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZ!AZpaths\">Edit</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZpaths\">Reload</A>]</TD>\n\
</TR>\n\
\
<TR>\n\
<TH>User Authentication</TH>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZauth\">Server</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZauth/\">File</A>]</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>&nbsp;</TD>\n\
<TD>[<A HREF=\"!AZpurge\">Purge</A>]</TD>\n\
</TR>\n\
\
<TR>\n\
<TH ROWSPAN=2>Other Reports</TH>\n\
<TD>&nbsp;</TD>\n\
<TD COLSPAN=7>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
\
<TR><TD>\n\
[<A HREF=\"!AZcache\">Cache</A>]\n\
[<A HREF=\"!AZDCL\">DCL</A>]\n\
[<A HREF=\"!AZDECnet\">DECnet</A>]\n\
[<A HREF=\"!AZmemory\">Memory</A>]\n\
[<A HREF=\"!AZrequest\">Request</A>]\n\
[!AZ!AZ!AZSSL!AZ]\n\
</TD></TR>\n\
\
<TR>\n\
<TD>\
[\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
<FONT SIZE=1> hours activity</FONT>]\n\
</TD></TR>\n\
\
</TABLE>\n\
\
</TD></TR>\n\
\
</TABLE>\n\
\
<TR>\n\
<TH><FONT COLOR=\"#ff0000\">Control</FONT> <SUP>*</SUP></TH>\n\
\
<TD ROWSPAN=2 ALIGN=CENTER>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH COLSPAN=4 ALIGN=center><FONT SIZE=-1><NOBR>!AZ</NOBR></FONT></TH></TR>\n\
<TR><TD COLSPAN=4 ALIGN=center><FONT SIZE=-2><NOBR>!AZ (!AZ)</NOBR></FONT></TD></TR>\n\
\
<TR><TH></TH></TR>\n\
\
<TR><TH ROWSPAN=2 ALIGN=left><FONT SIZE=-1>Times</FONT></TH>\n\
<TH ALIGN=right><FONT SIZE=-1>Up:</FONT></TH>\
<TD align=right><FONT SIZE=-1>!AZ</FONT></TD></TR>\n\
\
<TR><TH ALIGN=right><FONT SIZE=-1>CPU:</FONT></TH>\n\
<TD align=right><FONT SIZE=-1>!UL !2ZL:!2ZL:!2ZL.!2ZL</FONT></TD></TR>\n\
\
<TR><TH ROWSPAN=2 ALIGN=left><FONT SIZE=-1>Requests&nbsp;</FONT></TH>\n\
<TH ALIGN=right><FONT SIZE=-1>Read:</FONT></TH>\
<TD align=right><FONT SIZE=-1>!AZ</FONT></TD></TR>\n\
\
<TR><TH ALIGN=right><FONT SIZE=-1>Write:</FONT></TH>\
<TD align=right><FONT SIZE=-1>!AZ</FONT></TD></TR>\n\
\
<TR><TH ROWSPAN=2 ALIGN=left><FONT SIZE=-1>Bytes</FONT></TH>\n\
<TH ALIGN=right><FONT SIZE=-1>Rx:</FONT></TH>\
<TD align=right><FONT SIZE=-1>!AZ</FONT></TD></TR>\n\
\
<TR><TH ALIGN=right><FONT SIZE=-1>Tx:</FONT></TH>\
<TD align=right><FONT SIZE=-1>!AZ</FONT></TD></TR>\n\
\
</TABLE>\n\
\
</TD></TR>\n\
<TR><TD>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top ALIGN=left>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD><A HREF=\"!AZ\">RESTART</A></TD></TR>\n\
<TR><TD><A HREF=\"!AZ\"><FONT SIZE=-1>RESTART</FONT>NOW</A></TD></TR>\n\
<TR><TD><A HREF=\"!AZ\">EXIT</A></TD></TR>\n\
<TR><TD><A HREF=\"!AZ\"><FONT SIZE=-1>EXIT</FONT>NOW</A></TD></TR>\n\
</TABLE>\n\
\
</TD><TD>&nbsp;</TD><TD>&nbsp;</TD><TD>&nbsp;</TD><TD VALIGN=top>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD ALIGN=right>!AZ</TD><TD>!AZ!AZ!AZ!AZ!AZ</TD></TR>\n\
<TR><TD ALIGN=right>!AZ</TD><TD>!AZ!AZ!AZ!AZ!AZ</TD></TR>\n\
</TABLE>\n\
\
</TD></TR>\n\
\
<TR><TD COLSPAN=5 ALIGN=left>\n\
<SUP>*</SUP>\
<I><FONT COLOR=\"#ff0000\">Caution</FONT>, no confirmation!!</I>\n\
</TD></TR>\n\
\
</TABLE>\n\
\
</TD></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   static unsigned long  JpiCpuTime,
                         Pid;
   static unsigned long  ConnectTime [2],
                         CurrentTime [2],
                         JpiLoginTime [2];

   static char  UpTime [32];

   static $DESCRIPTOR (UpTimeFaoDsc, "!%D\0");
   static $DESCRIPTOR (UpTimeDsc, UpTime);

   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiCpuTime), JPI$_CPUTIM, &JpiCpuTime, 0 },
      { sizeof(JpiLoginTime), JPI$_LOGINTIM, &JpiLoginTime, 0 },
      {0,0,0,0}
   };

   register char  *cptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [128];
   char  Buffer [8192],
         RequestsRead [16],
         RequestsWrite [16],
         BytesRx [32],
         BytesTx [32];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AdminMenu()\n");

   if (VMSnok (status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0)))
   {
      rqptr->ErrorTextPtr = "sys$getjpi()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   sys$gettim (&CurrentTime);
   lib$sub_times (&CurrentTime, &JpiLoginTime, &ConnectTime);
   sys$fao (&UpTimeFaoDsc, 0, &UpTimeDsc, &ConnectTime);

   CommaNumber (32,
                Accounting.MethodGetCount +
                Accounting.MethodHeadCount,
                RequestsRead);

   CommaNumber (32,
                Accounting.MethodDeleteCount +
                Accounting.MethodPostCount +
                Accounting.MethodPutCount,
                RequestsWrite);

   CommaNumber (64, &Accounting.QuadBytesRx[0], BytesRx);
   CommaNumber (64, &Accounting.QuadBytesTx[0], BytesTx);

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   /* statistics */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalControl;

   /* configuration */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalRevise;
   *vecptr++ = HttpdInternalRevise;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = HttpdInternalRevise;

   /* messages */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = HttpdInternalRevise;

   /* mapping */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = HttpdInternalRevise;
   *vecptr++ = HttpdInternalControl;

   /* authorization */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = HttpdInternalRevise;
   *vecptr++ = HttpdInternalControl;

   /* authentication */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalRevise;
   *vecptr++ = HttpdInternalControl;

   /* other */
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   *vecptr++ = HttpdInternalReport;
   if (ProtocolHttpsConfigured)
   {
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReport;
      *vecptr++ = "SSL\">";
      *vecptr++ = "</A>";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   /* activity */
   if (Config.ActivityNumberOfDays)
   {
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?1\">1</A>,\n";
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?2\">2</A>,\n";
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?4\">4</A>,\n";
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?12\">12</A>,\n";
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?24\">24</A>,\n";
   }
   else
   {
      *vecptr++ = "1,\n";
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "2,\n";
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "4,\n";
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "12,\n";
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "24,\n";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   if (Config.ActivityNumberOfDays >= 3)
   {
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?72\">72</A>,\n";
   }
   else
   {
      *vecptr++ = "72,\n";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   if (Config.ActivityNumberOfDays >= 7)
   {
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?168\">168</A>,\n";
   }
   else
   {
      *vecptr++ = "168,\n";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   if (Config.ActivityNumberOfDays >= 28)
   {
      *vecptr++ = "<A HREF=\"";
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = "?672\">672</A>";
   }
   else
   {
      *vecptr++ = "672";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   /* informational accounting */
   *vecptr++ = DayDateTime (rqptr->BinaryTime, 20);
   *vecptr++ = rqptr->GmDateTime;
   *vecptr++ = TimeGmtString;
   for (cptr = UpTime; ISLWS(*cptr); cptr++);
   *vecptr++ = (unsigned long)cptr;
   *vecptr++ = JpiCpuTime / 8640000;             /* CPU day */
   *vecptr++ = (JpiCpuTime % 8640000) / 360000;  /* CPU hour */
   *vecptr++ = (JpiCpuTime % 360000) / 6000;     /* CPU minute */
   *vecptr++ = (JpiCpuTime % 6000 ) / 100;       /* CPU second */
   *vecptr++ = JpiCpuTime % 100;                 /* CPU 10 milliseconds */
   *vecptr++ = RequestsRead;
   *vecptr++ = RequestsWrite;
   *vecptr++ = BytesRx;
   *vecptr++ = BytesTx;

   /* server control menu items */
   *vecptr++ = HttpdInternalControlRestart;
   *vecptr++ = HttpdInternalControlRestartNow;
   *vecptr++ = HttpdInternalControlExit;
   *vecptr++ = HttpdInternalControlExitNow;

   /* logging control menu items */
   if (LoggingEnabled)
   {
      *vecptr++ = "<B>Log</B>";
      *vecptr++ = "<NOBR>[On] [<A HREF=\"";
      *vecptr++ = HttpdInternalControlLogOff;
      *vecptr++ = "\">Off</A>] [<A HREF=\"";
      *vecptr++ = HttpdInternalControlLogFlush;
      *vecptr++ = "\">Flush</A>]</NOBR>";
   }
   else
   {
      *vecptr++ = "<B>Log</B>";
      *vecptr++ = "<NOBR>[<A HREF=\"";
      *vecptr++ = HttpdInternalControlLogOn;
      *vecptr++ = "\">On</A>] [Off] [Flush]</NOBR>";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   /* cache control menu items */
   if (CacheEnabled)
   {
      *vecptr++ = "<B>Cache</B>";
      *vecptr++ = "<NOBR>[On] [<A HREF=\"";
      *vecptr++ = HttpdInternalControlCacheOff;
      *vecptr++ = "\">Off</A>] [<A HREF=\"";
      *vecptr++ = HttpdInternalControlCachePurge;
      *vecptr++ = "\">Purge</A>]</NOBR>";
   }
   else
   {
      *vecptr++ = "<B>Cache</B>";
      *vecptr++ = "<NOBR>[<A HREF=\"";
      *vecptr++ = HttpdInternalControlCacheOn;
      *vecptr++ = "\">On</A>] [Off] [Purge]</NOBR>";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, 0, Buffer, Length);

   SysDclAst (NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Return report on the HTTPd server's activities.  Uses blocking network write.
*/ 

AdminReportServerStats
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   /* ALTPRI and SETPRI synonyms? */
#  define IMAG_PRIV_0_ALLOWED \
      (~(PRV$M_ALTPRI | PRV$M_PRMMBX | PRV$M_PSWAPM | \
         PRV$M_SYSNAM | PRV$M_SYSPRV))

#  define IMAG_PRIV_1_ALLOWED (~(0x00000000))


/*
   VAX VMS does not seem to like "!@UQ" and must be made "!@UJ"!!!
   The VAX version not supporting this doesn't seem to be documented,
   but at run-time returns a %X00000014 status (%SYSTEM-F-BADPARAM).
   Implication: VAX version can only report a maximum of 2^32-1 bytes, or
   4,294,967,295 before overflow, AXP 2^64-1 (calculator gave an ERROR :^)
*/

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Server Statistics</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Server Statistics</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR>\
<TH COLSPAN=2>Software</TH></TD>\
</TR>\n\
<TR>\
<TH>Version</TH><TD>!AZ</TD>\
</TR>\n\
<TR>\
<TH>Build</TH><TD>!AZ</TD>\
</TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR>\
<TH COLSPAN=8>Server Process</TH></TD>\
</TR>\n\
<TR>\
<TH>Name</TH><TD>!AZ</TD>\
<TH></TH><TH>PID<TD>!8XL</TD>\
<TH></TH><TH>User<TD>!AZ</TD>\
</TR>\n\
<TR>\
<TH>AuthPriv</TH><TD>!AZ</TD>\
<TH></TH><TH>ImagPriv</TH><TD>!AZ</TD>\
<TH></TH><TH>CurPriv</TH><TD>!AZ</TD>\
</TR>\n\
<TR>\
<TH>Startup</TH><TD>!UL</TD>\
<TH></TH><TH>Exit Status</TH><TD>%X!8XL</TD>\
<TH></TH><TH>Zeroed</TH><TD>!UL</TD>\
</TR>\n\
<TR>\
<TH>Up-time</TH><TD>!AZ</TD>\
<TH></TH><TH>CPU-time</TH><TD>!UL !2ZL:!2ZL:!2ZL.!2ZL</TD>\n\
</TR>\n\
<TH>Pg.Faults</TH><TD>!UL</TD>\n\
<TH></TH><TH>Pg.Used</TH><TD>!UL%</TD>\
</TR>\n\
<TR>\
<TH>WsSize</TH><TD>!UL (!ULKB)</TD>\
<TH></TH><TH>WsPeak</TH><TD>!UL (!ULKB)</TD>\n\
<TH></TH><TH>PeakVirt</TH><TD>!UL (!ULKB)</TD>\
</TR>\n\
<TR>\
<TH>AST</TH><TD>!UL / !UL</TD>\
<TH></TH><TH>BIO</TH><TD>!UL / !UL</TD>\
<TH></TH><TH>BYT</TH><TD>!UL / !UL</TD>\
</TR>\n\
<TR>\
<TH>DIO</TH><TD>!UL / !UL</TD>\
<TH></TH><TH>ENQ</TH><TD>!UL / !UL</TD>\
<TH></TH><TH>FILE</TH><TD>!UL / !UL</TD>\
</TR>\n\
<TR>\
<TH>PRC</TH><TD>!UL / !UL</TD>\
<TH></TH><TH>TQ</TH><TD>!UL / !UL</TD>\n\
</TR>\n\
</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Services</TH></TR>\n\
<TR><TH>protocol://host:port</TH><TH>count</TH></TR>\n");

   static $DESCRIPTOR (ServiceFaoDsc,
"<TR><TD><TT>!AZ//!AZ</TT></TD><TD ALIGN=right>!UL</TD></TR>\n");

   static $DESCRIPTOR (StatisticsFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR>\
<TH COLSPAN=5>Connections</TH></TD>\
</TR>\n\
<TR>\
<TH>Total</TH><TD ALIGN=RIGHT>!UL</TD>\
<TD></TD><TH>Current</TH><TD ALIGN=RIGHT>!UL</TD>\
</TR>\n\
<TR>\
<TH>Accepted</TH><TD ALIGN=RIGHT>!UL</TD>\
<TD></TD><TH>Peak</TH><TD ALIGN=RIGHT>!UL</TD>\n\
</TR>\n\
<TR>\
<TH>Rejected</TH><TD ALIGN=RIGHT>!UL</TD>\
<TD></TD><TH>Too Busy</TH><TD ALIGN=RIGHT>!UL</TD>\
</TR>\n\
<TR>\
<TH>SSL (&quot;https:&quot;)</TH><TD ALIGN=RIGHT>!UL</TD>\
</TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR>\
<TH COLSPAN=2>Requests</TH></TD>\
</TR>\n\
<TR><TH>Total</TH><TD>!UL</TD></TR>\n\
<TR><TH>Parsed</TH><TD>!UL</TD></TR>\n\
<TR><TH>Error</TH><TD>!UL</TD></TR>\n\
<TR><TH>Forbidden</TH><TD>!UL</TD></TR>\n\
<TR><TH>Redirect remote/local</TH><TD>!UL / !UL</TD></TR>\n\
<TR><TH>Keep-Alive total/max</TH><TD>!UL / !UL</I></TD></TR>\n\
<TR><TH>Pragma: no-cache</TH><TD>!UL</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=7>Cache</TH></TD></TR>\n\
<TR><TH ALIGN=right>Memory</TH>\
<TD ALIGN=right>!UL / !UL</TD>\
<TD COLSPAN=5>kbytes <FONT SIZE=-1>(cur/max)</FONT></TD></TR>\n\
<TR><TH>Loads</TH><TH>Not Hit</TH><TH></TH>\
<TH>Hits</TH><TH>1-9</TH><TH>10-99</TH><TH>100+</TH></TR>\n\
<TR ALIGN=right><TD>!UL</TD><TD>!UL</TD><TD></TD>\
<TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD></TR>\n\
<TR ALIGN=right><TD></TD><TD>!UL%</TD><TD></TD>\
<TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Duration</TH></TD></TR>\n\
<TR><TH>Minimum</TH><TD ALIGN=right>!UL.!#ZL</TD></TR>\n\
<TR><TH>Maximum</TH><TD ALIGN=right>!UL.!#ZL</TD></TR>\n\
<TR><TH>Average</TH><TD ALIGN=right>!UL.!#ZL</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=5>HTTP Methods</TH></TD></TR>\n\
<TR>\
<TH>DELETE</TH><TH>GET</TH><TH>HEAD</TH><TH>POST</TH><TH>PUT</TH>\
</TR>\n\
<TR ALIGN=RIGHT>\
<TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD>\
</TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=6>Authorization</TH></TR>\n\
<TR><TH COLSPAN=3>Authentication</TH><TH></TH><TH COLSPAN=2>Paths</TH>\
<TR><TH COLSPAN=2>Failure</TH><TD ALIGN=RIGHT>!UL</TD><TH></TH>\
<TH>Yes</TH><TD ALIGN=RIGHT>!UL</TD></TR>\n\
<TR><TH ROWSPAN=3>From</TH>\
<TH>Cache</TH><TD ALIGN=RIGHT>!UL</TD><TH></TH>\
<TH>No</TH><TD ALIGN=RIGHT>!UL</TD></TR>\n\
<TR><TH><FONT SIZE=-1>HTA</FONT> Database</TH><TD ALIGN=RIGHT>!UL</TD></TR>\n\
<TR><TH>VMS <FONT SIZE=-1>SYSUAF</FONT></TH><TD ALIGN=RIGHT>!UL</TD></TR>\n\
<TR><TH ROWSPAN=2>Scheme</TH>\
<TH>Basic</TH><TD ALIGN=RIGHT>!UL</TD></TR>\
<TR><TH>Digest</TH><TD ALIGN=RIGHT>!UL</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Server Modules</TH></TD></TR>\n\
<TR><TH COLSPAN=2>Admin</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>Directory</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>File</TH><TD>!UL \
&nbsp;<I>(!UL not modified)</I></TD></TR>\n\
<TR><TH COLSPAN=2>IsMap</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>Menu</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>Put</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>SSI</TH><TD>!UL</TD></TR>\n\
<TR><TH COLSPAN=2>Update</TH><TD>!UL</TD></TR>\n\
<TR><TH ROWSPAN=5>DCL</TH>\
<TH>CGI <I>script</I></TH><TD>!UL</TD></TR>\n\
<TR><TH>CGIplus <I>script</I></TH>\
<TD>!UL &nbsp;<I>(!UL multiple)</I></TD></TR>\n\
<TR><TH>Auto-Scripted</TH><TD>!UL</TD></TR>\n\
<TR><TH>DCL command</TH><TD>!UL</TD></TR>\n\
<TR><TH>Spawned</TH><TD>!UL</TD></TR>\n\
<TR><TH ROWSPAN=3>DECnet</TH>\
<TH>Connect</TH><TD>!UL &nbsp;<I>(!UL failed)</I></TD></TR>\n\
<TR><TH>CGI <I>script</I></TH><TD>!UL</TD></TR>\n\
<TR><TH>OSU <I>script</I></TH><TD>!UL</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=7>Responses</TH></TD></TR>\n\
<TR>\
<TH>1nn</TH><TH>2nn</TH><TH>3nn</TH><TH>4nn</TH><TH>5nn</TH>\
<TH></TH><TH><I>Error</I></TH>\
</TR>\n\
<TR ALIGN=RIGHT>\
<TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD>\
<TH></TH><TD>!UL</TD>\
</TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=5>Bytes</TH></TD></TR>\n\
<TR><TH>Rx</TH><TD>!AZ</TD><TH></TH><TH>Tx</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Other</TH></TD></TR>\n\
<TR><TH>StreamLF document conversions</TH><TD>!UL</TD></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (PrivFaoDsc,
"<CENTER><FONT COLOR=\"#ff0000\">WARNING\
<BR><FONT SIZE=-1>!XL!XL</FONT></FONT></CENTER>\0");

   static char  JpiPrcNam [16],
                JpiUserName [13],
                UpTime [32];
   static unsigned long  JpiAstCnt,
                         JpiAstLm,
                         JpiBioCnt,
                         JpiBytLm,
                         JpiBytCnt,
                         JpiBioLm,
                         JpiCpuTime,
                         JpiDioCnt,
                         JpiDioLm,
                         JpiEnqCnt,
                         JpiEnqLm,
                         JpiFilCnt,
                         JpiFilLm,
                         JpiPageFlts,
                         JpiPagFilCnt,
                         JpiPgFlQuota,
                         JpiPid,
                         JpiPrcCnt,
                         JpiPrcLm,
                         JpiTqCnt,
                         JpiTqLm,
                         JpiVirtPeak,
                         JpiWsSize,
                         JpiWsPeak,
                         Pid;
   static unsigned long  ConnectTime [2],
                         CurrentTime [2],
                         JpiAuthPriv [2],
                         JpiCurPriv [2],
                         JpiImagPriv [2],
                         JpiLoginTime [2];
   static char  LastExitStatus [16] = "n/a";
   static $DESCRIPTOR (LastExitStatusDsc, LastExitStatus);
   static $DESCRIPTOR (LastExitStatusFaoDsc, "%X!XL");
   static $DESCRIPTOR (UpTimeFaoDsc, "!%D\0");
   static $DESCRIPTOR (UpTimeDsc, UpTime);
   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiAstCnt), JPI$_ASTCNT, &JpiAstCnt, 0 },
      { sizeof(JpiAstLm), JPI$_ASTLM, &JpiAstLm, 0 },
      { sizeof(JpiAuthPriv), JPI$_AUTHPRIV, &JpiAuthPriv, 0 },
      { sizeof(JpiBioCnt), JPI$_BIOCNT, &JpiBioCnt, 0 },
      { sizeof(JpiBioLm), JPI$_BIOLM, &JpiBioLm, 0 },
      { sizeof(JpiBytCnt), JPI$_BYTCNT, &JpiBytCnt, 0 },
      { sizeof(JpiBytLm), JPI$_BYTLM, &JpiBytLm, 0 },
      { sizeof(JpiCpuTime), JPI$_CPUTIM, &JpiCpuTime, 0 },
      { sizeof(JpiCurPriv), JPI$_CURPRIV, &JpiCurPriv, 0 },
      { sizeof(JpiDioCnt), JPI$_DIOCNT, &JpiDioCnt, 0 },
      { sizeof(JpiDioLm), JPI$_DIOLM, &JpiDioLm, 0 },
      { sizeof(JpiEnqCnt), JPI$_ENQCNT, &JpiEnqCnt, 0 },
      { sizeof(JpiEnqLm), JPI$_ENQLM, &JpiEnqLm, 0 },
      { sizeof(JpiFilCnt), JPI$_FILCNT, &JpiFilCnt, 0 },
      { sizeof(JpiFilLm), JPI$_FILLM, &JpiFilLm, 0 },
      { sizeof(JpiImagPriv), JPI$_IMAGPRIV, &JpiImagPriv, 0 },
      { sizeof(JpiLoginTime), JPI$_LOGINTIM, &JpiLoginTime, 0 },
      { sizeof(JpiPageFlts), JPI$_PAGEFLTS, &JpiPageFlts, 0 },
      { sizeof(JpiPagFilCnt), JPI$_PAGFILCNT, &JpiPagFilCnt, 0 },
      { sizeof(JpiPgFlQuota), JPI$_PGFLQUOTA, &JpiPgFlQuota, 0 },
      { sizeof(JpiPid), JPI$_PID, &JpiPid, 0 },
      { sizeof(JpiPrcCnt), JPI$_PRCCNT, &JpiPrcCnt, 0 },
      { sizeof(JpiPrcLm), JPI$_PRCLM, &JpiPrcLm, 0 },
      { sizeof(JpiPrcNam), JPI$_PRCNAM, &JpiPrcNam, 0 },
      { sizeof(JpiTqCnt), JPI$_TQCNT, &JpiTqCnt, 0 },
      { sizeof(JpiTqLm), JPI$_TQLM, &JpiTqLm, 0 },
      { sizeof(JpiUserName), JPI$_USERNAME, &JpiUserName, 0 },
      { sizeof(JpiVirtPeak), JPI$_VIRTPEAK, &JpiVirtPeak, 0 },
      { sizeof(JpiWsSize), JPI$_WSSIZE, &JpiWsSize, 0 },
      { sizeof(JpiWsPeak), JPI$_WSPEAK, &JpiWsPeak, 0 },
      {0,0,0,0}
   };

   register unsigned int  bit;
   register unsigned long  *vecptr;
   register char  *cptr;
   register struct ServiceStruct  *svptr;

   int  status;
   unsigned short  Length;
   unsigned long  ResponseDuration,
                  Remainder;
   unsigned long  FaoVector [128];
   char  AuthPrivString [128],
         CurPrivString [128],
         ImagPrivString [128],
         Buffer [8192],
         BytesRx [32],
         BytesTx [32];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (AuthPrivDsc, AuthPrivString);
   $DESCRIPTOR (ImagPrivDsc, ImagPrivString);
   $DESCRIPTOR (CurPrivDsc, CurPrivString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AdminReportServerStats()\n");

   if (VMSnok (status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0)))
   {
      rqptr->ErrorTextPtr = "sys$getjpi()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   JpiUserName[12] = '\0';
   for (cptr = JpiUserName; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiUserName |%s|\n", JpiUserName);

   JpiPrcNam[15] = '\0';
   for (cptr = JpiPrcNam; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiPrcNam |%s|\n", JpiPrcNam);

   sys$gettim (&CurrentTime);
   lib$sub_times (&CurrentTime, &JpiLoginTime, &ConnectTime);
   sys$fao (&UpTimeFaoDsc, 0, &UpTimeDsc, &ConnectTime);

   if (Accounting.StartupCount > 1)
   {
      sys$fao (&LastExitStatusFaoDsc, &Length, &LastExitStatusDsc, 
               Accounting.LastExitStatus);
      LastExitStatus[Length] = '\0';
   }

   CommaNumber (64, &Accounting.QuadBytesRx[0], BytesRx);
   CommaNumber (64, &Accounting.QuadBytesTx[0], BytesTx);

   if ((JpiAuthPriv[0] & ~(PRV$M_NETMBX | PRV$M_TMPMBX)) || JpiAuthPriv[1])
      sys$fao (&PrivFaoDsc, 0, &AuthPrivDsc, JpiAuthPriv[1],  JpiAuthPriv[0]);
   else
      strcpy (AuthPrivString, "as expected");

   if ((JpiImagPriv[0] & IMAG_PRIV_0_ALLOWED) ||
       (JpiImagPriv[1] & IMAG_PRIV_1_ALLOWED))
      sys$fao (&PrivFaoDsc, 0, &ImagPrivDsc, JpiImagPriv[1],  JpiImagPriv[0]);
   else
      strcpy (ImagPrivString, "as expected");

   if ((JpiCurPriv[0] & ~(PRV$M_NETMBX | PRV$M_TMPMBX)) || JpiCurPriv[1])
      sys$fao (&PrivFaoDsc, 0, &CurPrivDsc, JpiCurPriv[1],  JpiCurPriv[0]);
   else
      strcpy (CurPrivString, "as expected");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   /* META info */
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* page title, etc. */
   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   /* server */
   *vecptr++ = (unsigned long)SoftwareID;
   *vecptr++ = (unsigned long)BuildInfo;
   *vecptr++ = (unsigned long)JpiPrcNam;
   *vecptr++ = JpiPid;
   *vecptr++ = (unsigned long)JpiUserName;
   *vecptr++ = AuthPrivString;
   *vecptr++ = ImagPrivString;
   *vecptr++ = CurPrivString;
   *vecptr++ = Accounting.StartupCount;
   *vecptr++ = (unsigned long)LastExitStatus;
   *vecptr++ = Accounting.ZeroedCount;
   for (cptr = UpTime; ISLWS(*cptr); cptr++);
   *vecptr++ = (unsigned long)cptr;
   *vecptr++ = JpiCpuTime / 8640000;             /* CPU day */
   *vecptr++ = (JpiCpuTime % 8640000) / 360000;  /* CPU hour */
   *vecptr++ = (JpiCpuTime % 360000) / 6000;     /* CPU minute */
   *vecptr++ = (JpiCpuTime % 6000 ) / 100;       /* CPU second */
   *vecptr++ = JpiCpuTime % 100;                 /* CPU 10 milliseconds */
   *vecptr++ = JpiPageFlts;
   *vecptr++ = 100 - ((JpiPagFilCnt * 100) / JpiPgFlQuota);
   *vecptr++ = JpiWsSize;
   *vecptr++ = JpiWsSize / 2;
   *vecptr++ = JpiWsPeak;
   *vecptr++ = JpiWsPeak / 2;
   *vecptr++ = JpiVirtPeak;
   *vecptr++ = JpiVirtPeak / 2;
   *vecptr++ = JpiAstLm - JpiAstCnt;
   *vecptr++ = JpiAstLm;
   *vecptr++ = JpiBioLm - JpiBioCnt;
   *vecptr++ = JpiBioLm;
   *vecptr++ = JpiBytLm - JpiBytCnt;
   *vecptr++ = JpiBytLm;
   *vecptr++ = JpiDioLm - JpiDioCnt;
   *vecptr++ = JpiDioLm;
   *vecptr++ = JpiEnqLm - JpiEnqCnt;
   *vecptr++ = JpiEnqLm;
   *vecptr++ = JpiFilLm - JpiFilCnt;
   *vecptr++ = JpiFilLm;
   *vecptr++ = JpiPrcCnt;
   *vecptr++ = JpiPrcLm;
   *vecptr++ = JpiTqLm - JpiTqCnt;
   *vecptr++ = JpiTqLm;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /************/
   /* services */
   /************/

   for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
   {
      vecptr = FaoVector;
      /* the server and port in the service structure */
      *vecptr++ = svptr->RequestSchemeNamePtr;
      *vecptr++ = svptr->ServerHostPort;
      *vecptr++ = svptr->ConnectCount;

      status = sys$faol (&ServiceFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /**************/
   /* statistics */
   /**************/

   vecptr = FaoVector;

   /* connections */
   *vecptr++ = Accounting.ConnectCount;
   *vecptr++ = Accounting.ConnectCurrent;
   *vecptr++ = Accounting.ConnectAcceptedCount;
   *vecptr++ = Accounting.ConnectPeak;
   *vecptr++ = Accounting.ConnectRejectedCount;
   *vecptr++ = Accounting.ConnectTooBusyCount;
   *vecptr++ = Accounting.ConnectSslCount;

   /* requests */
   *vecptr++ = Accounting.RequestTotalCount;
   *vecptr++ = Accounting.RequestParseCount;
   *vecptr++ = Accounting.RequestErrorCount;
   *vecptr++ = Accounting.RequestForbiddenCount;
   *vecptr++ = Accounting.RedirectRemoteCount;
   *vecptr++ = Accounting.RedirectLocalCount;
   *vecptr++ = Accounting.RequestKeepAliveCount;
   *vecptr++ = Accounting.RequestKeepAliveMax;
   *vecptr++ = Accounting.RequestPragmaNoCacheCount;

   /* cache */
   *vecptr++ = CacheMemoryInUse >> 10;
   *vecptr++ = CacheTotalKBytesMax;
   *vecptr++ = CacheLoadCount;
   *vecptr++ = CacheHits0;
   *vecptr++ = CacheHitCount;
   *vecptr++ = CacheHits10;
   *vecptr++ = CacheHits100;
   *vecptr++ = CacheHits100plus;
   if (CacheLoadCount)
   {
      *vecptr++ = CacheHits0 * 100 / CacheLoadCount;
      *vecptr++ = CacheHitCount * 100 / CacheLoadCount;
   }
   else
   {
      *vecptr++ = 0;
      *vecptr++ = 0;
   }
   if (CacheHitCount)
   {
      *vecptr++ = CacheHits10 * 100 / CacheHitCount;
      *vecptr++ = CacheHits100 * 100 / CacheHitCount;
      *vecptr++ = CacheHits100plus * 100 / CacheHitCount;
   }
   else
   {
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = 0;
   }

   /* response duration */
   if (Accounting.ResponseDurationMin == -1)
   {
      *vecptr++ = 0;
      *vecptr++ = DURATION_DECIMAL_PLACES;
      *vecptr++ = 0;
   }
   else
   {
      *vecptr++ = Accounting.ResponseDurationMin / USECS_IN_A_SECOND;
      *vecptr++ = DURATION_DECIMAL_PLACES;
      *vecptr++ = (Accounting.ResponseDurationMin % USECS_IN_A_SECOND) /
                  DURATION_UNITS_USECS;
   }
   *vecptr++ = Accounting.ResponseDurationMax / USECS_IN_A_SECOND;
   *vecptr++ = DURATION_DECIMAL_PLACES;
   *vecptr++ = (Accounting.ResponseDurationMax % USECS_IN_A_SECOND) /
               DURATION_UNITS_USECS;
   if (Accounting.ResponseDurationCount)
   {
      status = lib$ediv (&Accounting.ResponseDurationCount,
                         &Accounting.QuadResponseDuration,
                         &ResponseDuration, &Remainder);
      if (Debug) fprintf (stdout, "lib$ediv() %%X%08.08X\n", status);
   }
   else
      ResponseDuration = 0;
   *vecptr++ = ResponseDuration / USECS_IN_A_SECOND;
   *vecptr++ = DURATION_DECIMAL_PLACES;
   *vecptr++ = (ResponseDuration % USECS_IN_A_SECOND) / DURATION_UNITS_USECS;

   /* HTTP methods */
   *vecptr++ = Accounting.MethodDeleteCount;
   *vecptr++ = Accounting.MethodGetCount;
   *vecptr++ = Accounting.MethodHeadCount;
   *vecptr++ = Accounting.MethodPostCount;
   *vecptr++ = Accounting.MethodPutCount;

   /* authorization */
   /*
      Note that the total number of paths being authorized may be
      greater than the total number of authentications.  This happens
      if a request contains both script and path authorizations,
      resulting in two authorizations for the one request!
   */
   *vecptr++ = Accounting.AuthNotAuthenticatedCount;
   *vecptr++ = Accounting.AuthAuthorizedCount;
   *vecptr++ = Accounting.AuthBasicCount +
               Accounting.AuthDigestCount -
               Accounting.AuthHtDatabaseCount -
               Accounting.AuthVmsCount;
   *vecptr++ = Accounting.AuthNotAuthorizedCount;
   *vecptr++ = Accounting.AuthHtDatabaseCount;
   *vecptr++ = Accounting.AuthVmsCount;
   *vecptr++ = Accounting.AuthBasicCount;
   *vecptr++ = Accounting.AuthDigestCount;

   /* modules */
   *vecptr++ = Accounting.DoServerAdminCount;
   *vecptr++ = Accounting.DoDirectoryCount;
   *vecptr++ = Accounting.DoFileCount;
   *vecptr++ = Accounting.DoFileNotModifiedCount;
   *vecptr++ = Accounting.DoIsMapCount;
   *vecptr++ = Accounting.DoMenuCount;
   *vecptr++ = Accounting.DoPutCount;
   *vecptr++ = Accounting.DoSsiCount;
   *vecptr++ = Accounting.DoUpdateCount;
   *vecptr++ = Accounting.DoScriptCount;
   *vecptr++ = Accounting.DoCgiPlusScriptCount;
   *vecptr++ = Accounting.DclCgiPlusReusedCount;
   *vecptr++ = Accounting.DoAutoScriptCount;
   *vecptr++ = Accounting.DoDclCommandCount;
   *vecptr++ = Accounting.DclSpawnCount;
   *vecptr++ = Accounting.DoDECnetCount;
   *vecptr++ = Accounting.DoDECnetCount -
               Accounting.DoDECnetCgiCount -
               Accounting.DoDECnetOsuCount;
   *vecptr++ = Accounting.DoDECnetCgiCount;
   *vecptr++ = Accounting.DoDECnetOsuCount;

   /* responses */
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[1];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[2];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[3];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[4];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[5];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[0];

   /* bytes rx & tx */
   *vecptr++ = BytesRx;
   *vecptr++ = BytesTx;

   /* other */
   *vecptr++ = Accounting.StreamLfConversionCount;

   status = sys$faol (&StatisticsFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NextTaskFunction, Buffer, Length);
}

/*****************************************************************************/
/*
Execute server administration menu control actions.
*/ 

int AdminControl
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *FunctionString
)
{
   int  status;
   char  String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AdminControl()\n");

   if (strsame (FunctionString, "DECnet_purge", -1))
   {
      /********************************/
      /* purge network scripting task */
      /********************************/

      DECnetDisconnect (rqptr, NextTaskFunction, false);
      return;
   }
   else
   if (strsame (FunctionString, "DECnet_force", -1))
   {
      /***********************************************/
      /* forcably disconnect network scripting tasks */
      /***********************************************/

      DECnetDisconnect (rqptr, NextTaskFunction, true);
      return;
   }
   else
   if (strsame (FunctionString, "exit", -1))
   {
      /***************/
      /* server exit */
      /***************/

      if (CurrentConnectCount > 1)
      {         
         sprintf (String,
           "exit when %d other, in-progress request(s) complete.</FONT>",
           CurrentConnectCount-1);
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">", String);

         /* stop the server from receiving incoming requests */
         NetShutdownServerSocket ();
         ControlExitRequested = true;
      }
      else
      {
         AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">",
                              "exit now!</FONT>");
         fprintf (stdout, "%%%s-I-ADMIN, %s, server exit\n",
                  Utility, DateTime(NULL,20));
         ExitStatus = SS$_NORMAL;
         HttpdExit ();
         sys$delprc (0, 0);
      }
   }
   else
   if (strsame (FunctionString, "exitNOW", -1))
   {
      /********************/
      /* server exit NOW! */
      /********************/

      rqptr->ResponseStatusCode = 200;
      AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">",
                                  "exit NOW!</FONT>");

      fprintf (stdout, "%%%s-I-ADMIN, %s, server exit NOW!\n",
               Utility, DateTime(NULL,20));
      ExitStatus = SS$_NORMAL;
      HttpdExit ();
      sys$delprc (0, 0);
   }
   else
   if (strsame (FunctionString, "restart", -1))
   {
      /******************/
      /* server restart */
      /******************/

      if (CurrentConnectCount > 1)
      {
         sprintf (String,
           "restart when %d other, in-progress request(s) complete.</FONT>",
           CurrentConnectCount-1);
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">", String);

         /* stop the server from receiving incoming requests */
         NetShutdownServerSocket ();
         ControlRestartRequested = true;
      }
      else
      {
         AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">",
                                     "restart now!</FONT>");
         fprintf (stdout, "%%%s-I-ADMIN, %s, server restart\n",
                  Utility, DateTime(NULL,20));
         exit (SS$_NORMAL);
      }
   }
   else
   if (strsame (FunctionString, "restartNOW", -1))
   {
      /***********************/
      /* server restart NOW! */
      /***********************/

      AdminControlSuccess (rqptr, "<FONT COLOR=\"#ff0000\">",
                                  "restart NOW!</FONT>");
      fprintf (stdout, "%%%s-I-ADMIN, %s, server restart NOW!\n",
               Utility, DateTime(NULL,20));
      exit (SS$_NORMAL);
   }
   else
   if (strsame (FunctionString, "subprocess_purge", -1))
   {
      /**************************/
      /* purge DCL subprocesses */
      /**************************/

      DclPurgeAllSubprocesses (rqptr, NextTaskFunction, false);
      return;
   }
   else
   if (strsame (FunctionString, "subprocess_force", -1))
   {
      /************************************/
      /* forcably delete DCL subprocesses */
      /************************************/

      DclPurgeAllSubprocesses (rqptr, NextTaskFunction, true);
      return;
   }
   else
   if (strsame (FunctionString, "rules", -1))
   {
      /************************/
      /* reload mapping rules */
      /************************/

      fprintf (stdout, "%%%s-I-ADMIN, %s, mapping rules reload\n",
               Utility, DateTime(NULL,20));

      MapUrl_ControlReload (rqptr, NextTaskFunction);
      return;
   }
   else
   if (strsame (FunctionString, "paths", -1))
   {
      /*****************************/
      /* authorization path reload */
      /*****************************/

      fprintf (stdout, "%%%s-I-ADMIN, %s, authorization paths reload\n",
               Utility, DateTime(NULL,20));

      AuthControlReload (rqptr, NextTaskFunction);
      return;
   }
   else
   if (strsame (FunctionString, "purge", -1))
   {
      /******************************/
      /* purge authentication cache */
      /******************************/

      fprintf (stdout, "%%%s-I-ADMIN, %s, purge authentication cache\n",
               Utility, DateTime(NULL,20));

      /* fake it :^) */
      rqptr->QueryStringPtr = "do=purge";
      HTAdminBegin (rqptr, NextTaskFunction);
      return;
   }
   else
   if (strsame (FunctionString, "zero", -1))
   {
      /*******************/
      /* zero accounting */
      /*******************/

      fprintf (stdout, "%%%s-I-ADMIN, %s, accounting zeroed\n",
               Utility, DateTime(NULL,20));

      ZeroAccounting ();
      rqptr->ResponseStatusCode = 200;
      AdminControlSuccess (rqptr, "", "accounting zeroed.");
   }
   else
   if (strsame (FunctionString, "cacheon", -1))
   {
      /****************/
      /* enable cache */
      /****************/

      if (CacheEnabled)
         ErrorGeneral (rqptr, "Cache already enabled.", FI_LI);
      else
      {
         CacheEnabled = true;
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", "cache enabled.");
      }
   }
   else
   if (strsame (FunctionString, "cacheoff", -1))
   {
      /*****************/
      /* disable cache */
      /*****************/

      if (CacheEnabled)
      {
         CacheEnabled = false;
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", "cache disabled.");
      }
      else
         ErrorGeneral (rqptr, "Cache already disabled", FI_LI);
   }
   else
   if (strsame (FunctionString, "cachepurge", -1))
   {
      /***************/
      /* purge cache */
      /***************/

      if (CacheEnabled)
      {
         int  PurgeCount, MarkedForPurgeCount;

         CachePurge (false, &PurgeCount, &MarkedForPurgeCount);
         sprintf (String, "cache %d purged, %d marked for purge.",
                  PurgeCount, MarkedForPurgeCount);
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", String);
      }
      else
         ErrorGeneral (rqptr, "Cache not enabled.", FI_LI);
   }
   else
   if (strsame (FunctionString, "logon", -1))
   {
      /******************/
      /* enable logging */
      /******************/

      if (LoggingEnabled)
         ErrorGeneral (rqptr, "Logging already enabled.", FI_LI);
      else
      if (VMSok (status = Logging (NULL, LOGGING_OPEN)))
      {
         fprintf (stdout, "%%%s-I-ADMIN, %s, logging enabled\n",
                  Utility, DateTime(NULL,20));
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", "logging enabled.");
      }
      else
      {
         rqptr->ErrorTextPtr = "Logging could NOT be enabled";
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
   }
   else
   if (strsame (FunctionString, "logoff", -1))
   {
      /*******************/
      /* disable logging */
      /*******************/

      if (!LoggingEnabled)
         ErrorGeneral (rqptr, "Logging not currently enabled.", FI_LI);
      else
      if (VMSok (status = Logging (NULL, LOGGING_CLOSE)))
      {
         fprintf (stdout, "%%%s-I-ADMIN, %s, logging disabled\n",
                  Utility, DateTime(NULL,20));
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", "logging disabled.");
      }
      else
      {
         rqptr->ErrorTextPtr = "Logging could NOT be disabled!";
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
   }
   else
   if (strsame (FunctionString, "logflush", -1))
   {
      /*************/
      /* flush log */
      /*************/

      if (!LoggingEnabled)
         ErrorGeneral (rqptr, ErrorLoggingNotEnabled, FI_LI);
      else
      if (VMSok (status = Logging (NULL, LOGGING_FLUSH)))
      {
         fprintf (stdout, "%%%s-I-ADMIN, %s, log flushed\n",
                  Utility, DateTime(NULL,20));
         rqptr->ResponseStatusCode = 200;
         AdminControlSuccess (rqptr, "", "log flushed.");
      }
      else
      {
         rqptr->ErrorTextPtr = "Log could NOT be flushed!";
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
   }
   else
   {
      /********************/
      /* unknown function */
      /********************/

      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, "Unknown control function.", FI_LI);
   }

   SysDclAst (NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Return a 200 success response to the client.
*/ 

int AdminControlSuccess
(
struct RequestStruct *rqptr,
char *StartText,
char *EndText
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>!AZServer !AZ !AZ\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;

   int  status;
   unsigned long  FaoVector [16];
   unsigned short  Length;
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AdminControlSuccess()\n");

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return (STS$K_ERROR);

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);
   *vecptr++ = StartText;
   *vecptr++ = ServerHostPort;
   *vecptr++ = EndText;
   sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   Buffer[Length] = '\0';

   NetWrite (rqptr, 0, Buffer, Length);

   return (SS$_NORMAL);
}

/****************************************************************************/

