/*****************************************************************************/
/*
                                  Auth.c


    THE GNU GENERAL PUBLIC LICENSE APPLIES DOUBLY TO ANYTHING TO DO WITH
                    AUTHENTICATION AND AUTHORIZATION!

    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License, or any later
    version.

>   This package is distributed in the hope that it will be useful,
>   but WITHOUT ANY WARRANTY; without even the implied warranty of
>   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
>   GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


Overview
--------

This is "path"-based authorization/authentication.  That is, access is
controlled according to the path specified in the request.  A path can have
capabilities, or access controls, allowing any combination of DELETE, GET,
HEAD, POST and PUT method capabilities.  The world can have capabilities
against a path. An authenticated user also can have capabilities.  The request
capability is the logical AND of the path and world/user capabilities, meaning
the request gets the minimum of the path and user's access.  There are no
access-control files in any data directory.  It is all under the control of the
HTTPd server manager.

AUTHORIZATION IS ALWAYS PERFORMED AGAINST THE REQUEST'S PATHINFO STRING (the
string parsed from the original request path), NOT AGAINST ANY RE-MAPPED PATH.
THERE SHOULD BE NO AMBIGUITY ABOUT WHAT IS BEING AUTHORIZED!

The authentication databases (password and capability files) are administered
using the server HTadmin.c module.

When a request contains a controlled path the authorization for access to that
path is made against a password database for the authentication "realm" or
"group" the path belongs to.  A realm or group is a 1 to 15 character string
describing the collection of one or more paths for purposes of authenticating
against a common password file.  A realm/group name should contain only alpha-
numeric, underscore and hyphen characters.  See below for a description of the
difference between realms and groups.

Three realm names are reserved, "EXTERNAL", "WORLD" and "VMS".

"EXTERNAL" refers to a script authenticating its own requests.
"VMS" refers to SYSUAF authentication.
"WORLD" refers to any request and is not authenticated in any way.

A path specification begins at the root and usually continues down to an
asterisk indicating a match against any path beginning with that string.  Paths
may overlap.  That is a parent directory may have subdirectory trees that
differ in access controls.  The same path specification may not belong to more
than one realm at a time :^)


Failure Reports
---------------

Authentication failures are reported to the process log.  Three messages can be
generated.  First, %HTTPD-W-AUTHFAIL, if the request is not authenticated due
to there being no such user or a password mismatch.  Isolated instances of this
are only of moderate interest.  Consecutive instances may indicate a user
thrashing about for the correct password, but they usually give up before a
dozen attempts.  The second message, %HTTPD-I-AUTHFAILOK (informational),
occurs if there has been at least one failed attempt to authenticate before a
successful attempt.  The third, %HTTPD-W-AUTHFAILIM, is of greater concern and
occurs if the total number of failed attempts exceeds the configuration limit
(such repeated attempts will always continue to fail for if this limit is
reached successive attempts are automatically failed).


Configuration
-------------

A collection of paths are controlled by a "realm" database, against which the
username/password is verified, and a "group" database, from which the
username's capabilities (what HTTP methods the username can apply against the
path).  Sometimes these databases are the same name (file), more commonly a
username/password database for identifying a client would be used in
combination with separate group databases for determining capabilities against
multiple paths or collections of paths.

A single file contains authorization configuration directives.  Realms (and
optionally group) are specified using the "realm" directive followed by a space
and then the realm name.  Optionally, the realm name database can administer
the password and a second, semicolon-separated group database control the
access methods allowed on a per-username basis.  Again optionally, method
keywords and IP address masks (see below) may be appended to specify default
access for any paths belonging to the realm.  Multiple "realm" directives for
the same realm name may be included, each with its own trailing paths.  These
multiple declarations just accumulate.

All paths specified after a "realm" directive, up until the next "realm"
directive will belong to that realm and use the relevant password database.
Paths are specified with a leading slash (i.e. from the root) down to either
the last character or a wildcard asterisk.

Method keywords, or the generic "READ", "R", "READ+WRITE", "R+W", "WRITE" and
"W", control access to the path (in combination with user access methods). The
method keywords are "DELETE", "GET" (implying "HEAD"), "HEAD" "PUT" and "POST".
These keywords are optionally supplied following a path.  If none are supplied
the realm defaults are applied.  The generic keyword "NONE" sets all access
permissions off, including relam defaults.  It is an error not to supply
either. Multiple method keywords may be separated by commas or spaces.  Method
keywords are applied in two groups.  First the group keywords.  Second, an
optional set for controlling world access.  These are delimited from the path
keywords by a semi-colon.

As part of the path access control a list of comma-separated elements may
optionally be supplied.  These elements may be numeric or alphabetic IP host
addresses (with or without asterisk wildcarding allowing wildcard matching of
domains), authenticated username(s) or the request scheme (protocol, "http:" or
"https:").  If supplied this list restricts access to those matching.  For
example:

   *.wasd.dsto.gov.au       would forbid any host outside the WASD subdomain
   131.185.250.*            similarly, but specifying using numeric notation
   *.wasd.dsto.gov.au,~daniel
   https:,*.131.185.250.*   limited to a subnet requesting via SSL service

The reserved word "localhost", or used as "#localhost", refers to the system
the server is executing on.  In this way various functions can be restricted to
browsers executing only on that same system.

Realm names, paths, IP addresses, usernames and access method keywords are all
case insensistive.

Comments may be included by prefixing the line with a "#" characters.  Blank
lines are ignored.  Directives may span multiple lines by ensuring the last
character on any line to be continued is a backslash ("\").

An example configuration file:

   |[WASD;CONFIDENTIAL]
   |/www/confidential/* get
   |/www/confidential/submissions/* https:,get,post;get
   |/www/authors/* get
   |
   |# realm-context access default
   |[WASD] get,head
   |/www/pub/*
   |/www/pub/submit/* ;get
   |
   |[WORLD]
   |/www/anyone-can-post-here/* post
   |/www/example/alpha/* *.wasd.dsto.gov.au,read
   |/www/example/numeric/* \
   |   131.185.250.*,read
   |
   |[EXTERNAL]
   |/cgi-bin/footypix*


Implementation
--------------

A linked-list, binary tree (using the the LIB$..._TREE routines) is used to 
store authentication records.  When a request with an "Authorization:" header 
line encounters a point where authentication is required this binary tree is 
checked for an existing record.  If one does not exist a new one is entered 
into the binary tree, with an empty password field.  The password in the 
authorization line is checked against the password in the record.  If it 
matches the request is authenticated.

If a binary tree record is not found the SYSUAF or REALM-based password is
checked.  If the hashed password matches, the plain password is copied into the
record and used for future authentications.  If not, the authentication fails.

Once a user name is authenticated from the password database, for a limited
period (currently 60 minutes), further requests are authenticated directly
from the binary tree.  This improves performance and reduces file system
references for authentication.  After that period expires the password and any
group access database is again explicitlty checked.  The internal, binary tree
database can be explicitly listed, flushed or re-loaded as required.

A request's authorization flags (->AuthRequestCan) governs what can and can't be
done using authorization.  These are arrived at by bit-wise ANDing of flags
from two sources ...

   o  'AuthGroupCan' flags associated with the authorization realm and path
   o  'AuthUserCan' flags associated with the authenticated user

... this ensuring that the minimum of actions permitted by either path or user
capabilities is reflected in the request capabilities.


SYSUAF-authentication by Identifier
-----------------------------------

The qualifier /SYSUAF=ID directs the server to allow SYSUAF authentication only
when the VMS account possesses a certain identifier.  No identifier, no
authentication, no matter how many correct passwords are provided!  An
excellent combination is /SYSUAF=(ID,SSL)!  In addition, other identifiers may
be used to further control access of VMS accounts.

  o  WASD_VMS_R .......... account has read access
  o  WASD_VMS_RW ......... account has read and write access
  o  WASD_VMS_HTTPS ...... account can only SYSUAF authenticate via SSL
  o  WASD_VMS_PWD ........ account can change it's SYSUAF password
  o  WASD_VMS__<group> ... account must possess identifier to access

The identifiers may be created using

  UAF> ADD /IDENT WASD_VMS_RW

and granted to desired accounts with

  UAF> GRANT /IDENT WASD_VMS_RW <account>


SYSUAF-authentication and VMS security profile
----------------------------------------------

The ability to be able to authenticate using the system's SYSUAF is controlled
by the server /SYSUAF qualifier.  By default it is disabled.  (This qualifier
was introduced in v4.4 as a result of creeping paranoia :^)

As of v4.4 it has become possible to control access to files and directories
based on the security profile of a SYSUAF-authenticated remote user. This
feature is controlled using the server /PROFILE qualifier.  By default it is
disabled.

A security-profile is created using sys$create_user_profile() and stored in
the authentication cache. This cached profile is the most efficient method of
implementing this as it obviously removes the need of creating a user profile
each time a resource is checked. If this profile exists in the cache it is
attached to each request structure authenticated via the cache.

When this profile is attached to a request all accesses to files and
directories are first assessed against that user profile using
sys$check_access(). If the user can access the resource (regardless of whether
that is because it is WORLD accessable, or because it is owned by, or
otherwise accessable to the user) SYSPRV is always enabled before accessing
the file/directory. This ensures that the authenticated user is always given
access to the resource (provided the SYSTEM permissions allow it!)

Of course, this functionality only provides access for the server, IT DOES NOT
PROPAGATE TO ANY SCRIPT ACCESS. If scripts must have a similar ability they
should implement their own sys$check_access() (which is not too difficult)
based on the WWW_AUTH_REALM which would be "VMS" indicating
SYSUAF-authentication, and the authenticated name in WWW_REMOTE_USER.

Note: the sys$assume_persona(), et.al., which might seem a more thorough
approach to this functionality, was not possible due to the "independently"
executing subprocesses associated with the server that might be adversely
affected by such an abrupt change in identity!


Controlling server write access
-------------------------------

Write access by the server into VMS directories is controlled using VMS ACLs.
This is in addition to the path authorization of the server itself of course!
The requirement to have an ACL on the directory prevents inadvertant
mapping/authorization path being able to be written to.  Two different ACLs
control two different grades of access.

1. If the ACL grants CONTROL access to the server account then only
VMS-authenticated usernames with security profiles can potentially write to
the directory, potentially, because a further check is made to assess whether
that VMS account has write access.

This example show a suitable ACL that stays only on the original directory:

  $ SET SECURITY directory.DIR -
    /ACL=(IDENT=HTTP$SERVER,ACCESS=READ+WRITE+EXECUTE+DELETE+CONTROL)

This example shows setting an ACL that will propagate to created
subdirectories:

  $ SET SECURITY directory.DIR -
    /ACL=((IDENT=HTTP$SERVER,OPTIONS=DEFAULT,ACCESS=READ+WRITE+EXECUTE+DELETE+CONTROL), -
          (IDENT=HTTP$SERVER,ACCESS=READ+WRITE+EXECUTE+DELETE+CONTROL))

2. If the ACL grants WRITE access then the directory can be written into by
any authenticated username for the authorized path.

This example show a suitable ACL that stays only on the original directory:

  $ SET SECURITY directory.DIR -
    /ACL=(IDENT=HTTP$SERVER,ACCESS=READ+WRITE+EXECUTE+DELETE)

This example shows setting an ACL that will propagate to created
subdirectories:

  $ SET SECURITY directory.DIR -
    /ACL=((IDENT=HTTP$SERVER,OPTIONS=DEFAULT,ACCESS=READ+WRITE+EXECUTE+DELETE), -
          (IDENT=HTTP$SERVER,ACCESS=READ+WRITE+EXECUTE+DELETE))


VERSION HISTORY
---------------
29-AUG-98  MGD  move authorization path processing into AuthPathLine(),
                report authentication failures in process log,
                change in behaviour: after cache minutes expires request
                revalidation by the user via browser dialog
16-JUL-98  MGD  /SYSUAF=ID, authentication with possession of an identifier,
                RESTRICTED_BY_OTHER indicates non-authentication forbidden
11-MAR-98  MGD  added local redirection kludge ('^'),
                configurable SYSUAF authentication of privileged accounts,
                bugfix; alpha-numeric hosts in access-restriction lists
                rejected as "unknown HTTP method"
08-FEB-98  MGD  provide SSL-only for SYSUAF and authorization in general,
                provide SSL-only for authorized paths (via "https:" or
                "http:" in path access restriction list),
                removed full method descriptions from user auth report
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile
16-JUL-97  MGD  fixed design flaw with WORLD realm and access restriction list
01-FEB-97  MGD  HTTPd version 4
01-JUL-96  MGD  path/realm-based authorization/authentication
15-MAR-96  MGD  bugfix; some ErrorGeneral() not passing the request pointer
01-DEC-95  MGD  HTTPd version 3
01-APR-95  MGD  initial development for local (SYSUAF) authentication
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <acldef.h>
#include <armdef.h>
#include <descrip.h>
#include <chpdef.h>
#include <iodef.h>
#include <libdef.h>
#include <prvdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

#include <uaidef.h>
/* not defined in VAX C 3.2 <uaidef.h> */
#define UAI$M_RESTRICTED 0x8
#define UAI$C_PURDY_S 3

/* application related header files */
#include "wasd.h"
#include "auth.h"
#include "basic.h"
#include "control.h"
#include "digest.h"
#include "error.h"
#include "httpd.h"
#include "mapurl.h"
#include "MD5.h"
#include "msg.h"
#include "support.h"

#if DBUG
#define FI_NOLI __FILE__, __LINE__
#else
/* in production let's keep the exact line to ourselves! */
#define FI_NOLI __FILE__, 0
#endif

/******************/
/* global storage */
/******************/

unsigned long  AuthDefaultCan = HTTP_METHOD_GET | HTTP_METHOD_HEAD;

boolean  AuthorizationEnabled,
         AuthSslOnly = false,
         AuthPromiscuous = false,
         AuthSysUafAllAccounts = false,
         AuthSysUafIdentifier = false,
         AuthSysUafEnabled = false,
         AuthSysUafPromiscuous = false,
         AuthSysUafSslOnly = false,
         AuthVmsUserSecProfileEnabled = false;

int  AuthDisplayRecordCount,
     AuthFreeRecordCount;

unsigned long  AuthCacheTreeBinTime [2];

char  *AuthPromiscuousPwdPtr;

struct AuthCacheRecordStruct  *AuthCacheTreeHead,
                              *AuthCacheTreeTailPtr;

struct AuthLoadStruct  ServerAuthLoad;

unsigned short  HttpdUserProfileLength;
unsigned char  *HttpdUserProfilePtr;
$DESCRIPTOR (HttpdUserProfileDsc, "");

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern unsigned short  ControlMbxChannel;
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  HttpdUserName[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
This function is pivotal to the HTTPd's authorization and authentication.

All requests should call this function before proceding.  This function should
allow or deny access to all requests.

Other functions called by this function are expected to return one of three
status codes.  SS$_NORMAL (or other success status), to indicate authentication
success, SS$_INVLOGIN to indicate authentication failure, or a VMS error status
(error messages must be generated where and when they occur).

If a non-success status is returned by this function the calling function
should immediately conclude processing.  It is expected that error will have
been reported (and thence a message generated) where and when they occur.
As usual, this would be reported during RequestEnd().
*/ 

int Authorize (struct RequestStruct *rqptr)

/*
{
int  status;

Debug = 1;
status = Authorize_ (rqptr);
Debug = 0;
return status;
}

int Authorize_ (struct RequestStruct *rqptr)
*/

{
   boolean  IsWorldRealm,
            RestrictedBy;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Authorize() |%s|\n", rqptr->PathInfoPtr);

   /* test/demo ... just in case an authenticated username is required */
   if (AuthPromiscuous)
   {
      rqptr->AuthRealmPtr = AUTH_PROMISCUOUS;
      if (VMSnok (status = AuthParseAuthorization (rqptr)))
         rqptr->RemoteUser[0] = rqptr->RemoteUserPassword[0] = '\0';
   }
   else
   {
      rqptr->AuthRealmPtr = "";
      rqptr->RemoteUser[0] = rqptr->RemoteUserPassword[0] = '\0';
   }

   if (!AuthorizationEnabled)
   {
      /********************************************/
      /* no authorization is mapped, use defaults */
      /********************************************/

      rqptr->AuthExternal = false;
      rqptr->AuthGroupPtr = "";
      rqptr->AuthGroupCan = rqptr->AuthRequestCan =
         rqptr->AuthUserCan = AuthDefaultCan;
      rqptr->AuthWorldCan = rqptr->AuthScheme = 0;
      return (SS$_NORMAL);
   }

   /****************************/
   /* authorization is enabled */
   /****************************/

   if (AuthSslOnly && rqptr->RequestScheme != SCHEME_HTTPS)
   {
      /* authorization is only allowed with "https:", and this is not! */
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
      return (STS$K_ERROR);
   }

   AuthPathCheck (rqptr, NULL);

   if (!rqptr->AuthRealmPtr[0])
   {
      /***************************************/
      /* not a controlled path, use defaults */
      /***************************************/

      /* do not reset the remote user! (may be second time through) */
      rqptr->AuthExternal = false;
      rqptr->AuthGroupPtr = "";
      rqptr->AuthGroupCan = rqptr->AuthRequestCan = 
         rqptr->AuthUserCan = AuthDefaultCan;
      rqptr->AuthWorldCan = rqptr->AuthScheme = 0;

      return (SS$_NORMAL);
   }

   /*********************************/
   /* controlled path, authorize it */
   /*********************************/

   if (Debug)
   {
      fprintf (stdout,
"|%s|\n\
GroupCan: %08.08X WorldCan: %08.08X\n\
GroupRestriction |%s|\n\
WorldRestriction |%s|\n",
      rqptr->AuthRealmPtr,
      rqptr->AuthGroupCan, rqptr->AuthWorldCan,
      rqptr->AuthRestrictListPtr,
      rqptr->AuthWorldRestrictListPtr);
   }

   rqptr->AuthExternal = false;

   if (rqptr->HttpMethod & rqptr->AuthWorldCan)
   {
      /**********************/
      /* world capabilities */
      /**********************/

      if (rqptr->AuthWorldRestrictListPtr[0])
      {
         /***********************************************/
         /* check access against world restriction list */
         /***********************************************/

         if (AuthRestrictList (rqptr, rqptr->AuthWorldRestrictListPtr))
         {
            rqptr->ResponseStatusCode = 403;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
            return (STS$K_ERROR);
         }
      }

      rqptr->AuthRequestCan = rqptr->AuthWorldCan;
      return (SS$_NORMAL);
   }

   if (!strcmp (rqptr->AuthRealmPtr, AUTH_EXTERNAL))
   {
      /**********************************/
      /* externally (script) authorized */
      /**********************************/

      rqptr->AuthExternal = true;
      rqptr->AuthRequestCan = rqptr->AuthGroupCan;
      return (SS$_NORMAL);
   }

   if (!strcmp (rqptr->AuthRealmPtr, AUTH_VMS))
   {
      if (!AuthSysUafEnabled ||
          (AuthSysUafSslOnly &&
           rqptr->RequestScheme != SCHEME_HTTPS))
      {
         /**********************/
         /* SYSUAF not allowed */
         /**********************/

         /* SYSUAF-authentication is disabled, or disabled for non-"https:" */ 
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
         return (STS$K_ERROR);
      }
   }

   if (rqptr->AuthRestrictListPtr[0])
   {
      /**********************************************/
      /* check HOST access against restriction list */
      /**********************************************/

      if (RestrictedBy = AuthRestrictList (rqptr, rqptr->AuthRestrictListPtr))
      {
         if (RestrictedBy == RESTRICTED_BY_HOSTNAME ||
             RestrictedBy == RESTRICTED_BY_PROTOCOL)
         {
            rqptr->ResponseStatusCode = 403;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
            return (STS$K_ERROR);
         }
      }
   }

   if (IsWorldRealm = (!strcmp (rqptr->AuthRealmPtr, AUTH_WORLD)))
   {
      /********************************************/
      /* the WORLD realm (i.e. no authentication) */
      /********************************************/

      rqptr->AuthRequestCan = rqptr->AuthUserCan = rqptr->AuthGroupCan;
      strcpy (rqptr->RemoteUser, AUTH_WORLD);
   }
   else
   {
      /*************************/
      /* authenticate the user */
      /*************************/

      if (VMSok (status = AuthParseAuthorization (rqptr)))
         status = AuthDoAuthorization (rqptr);

      if (VMSnok (status))
      {
         if (status == SS$_INVLOGIN)
         {
            /**************************/
            /* authentication failure */
            /**************************/

            Accounting.AuthNotAuthenticatedCount++;
            rqptr->ResponseStatusCode = 401;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_FAILED), FI_NOLI);
         }

         if (status == RESTRICTED_BY_OTHER)
         {
            /********************/
            /* access forbidden */
            /********************/

            rqptr->ResponseStatusCode = 403;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
            return (STS$K_ERROR);
         }

         return (status);
      }
   }

   if (rqptr->AuthRestrictListPtr[0])
   {
      /*****************************************/
      /* check access against restriction list */
      /*****************************************/

      if (RestrictedBy = AuthRestrictList (rqptr, rqptr->AuthRestrictListPtr))
      {
         if (IsWorldRealm ||
             RestrictedBy == RESTRICTED_BY_HOSTNAME ||
             RestrictedBy == RESTRICTED_BY_PROTOCOL)
            rqptr->ResponseStatusCode = 403;
         else
            rqptr->ResponseStatusCode = 401;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_NOLI);
         return (STS$K_ERROR);
      }
   }

   /**************************************/
   /* set and check request capabilities */
   /**************************************/

   /* bit-wise AND of capabilities ensures minimum apply */
   rqptr->AuthRequestCan = rqptr->AuthGroupCan & rqptr->AuthUserCan;

   if (Debug)
   {
      fprintf (stdout,
      "|%s|%s|%s|\nGroupCan: %08.08X UserCan: %08.08X RequestCan: %08.08X\n",
      rqptr->AuthRealmPtr, rqptr->RemoteUser, rqptr->RemoteUserPassword,
      rqptr->AuthGroupCan, rqptr->AuthUserCan, rqptr->AuthRequestCan);
   }

   if (rqptr->HttpMethod & rqptr->AuthRequestCan)
   {
      Accounting.AuthAuthorizedCount++;
      return (SS$_NORMAL);
   }

   /*
      If a realm controls access then prompt for authentication
      information by returning a 401 (authentication failure) status.
      If not realm-controlled return a 403 (forbidden) status.
   */

   if (rqptr->AuthRealmPtr[0] && !IsWorldRealm)
   {
      /******************/
      /* not authorized */
      /******************/

      static $DESCRIPTOR (StringFaoDsc, "Not authorized to !AZ <TT>!AZ</TT>");
      static $DESCRIPTOR (StringDsc, "");

      unsigned short  Length;
      char  String [256];

      Accounting.AuthNotAuthorizedCount++;

      StringDsc.dsc$a_pointer = String;
      StringDsc.dsc$w_length = sizeof(String)-1;

      sys$fao (&StringFaoDsc, &Length, &StringDsc,
               rqptr->HttpMethodName, rqptr->PathInfoPtr);
      String[Length] = '\0';

      ErrorGeneral (rqptr, String, FI_LI);
      return (STS$K_ERROR);
   }
   else
   {
      /*************/
      /* forbidden */
      /*************/

      static $DESCRIPTOR (StringFaoDsc, "Forbidden to !AZ <TT>!AZ</TT>");
      static $DESCRIPTOR (StringDsc, "");

      unsigned short  Length;
      char  String [256];

      Accounting.AuthNotAuthorizedCount++;

      StringDsc.dsc$a_pointer = String;
      StringDsc.dsc$w_length = sizeof(String)-1;

      sys$fao (&StringFaoDsc, &Length, &StringDsc,
               rqptr->HttpMethodName, rqptr->PathInfoPtr);
      String[Length] = '\0';

      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, String, FI_LI);
      return (STS$K_ERROR);
   }
}

/*****************************************************************************/
/*
Parse the "Authorization:" HTTP request header line.
*/ 

int AuthParseAuthorization (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr, *zptr;

   int  status;
   char  EncodedString [256],
         UserNamePassword [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthParseAuthorization()\n");

   if ((cptr = rqptr->HttpAuthorizationPtr) == NULL)
      return (SS$_INVLOGIN);

   /***********************************/
   /* get the HTTP authorization line */
   /***********************************/

   if (Debug) fprintf (stdout, "HttpAuthorizationPtr |%s|\n", cptr);

   zptr = (sptr = rqptr->AuthType) + sizeof(rqptr->AuthType);
   while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = toupper(*cptr++);
   if (sptr >= zptr)
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneralOverflow (rqptr, FI_LI);
      return (STS$K_ERROR);
   }
   *sptr = '\0';

   if (strsame (rqptr->AuthType, "BASIC"))
      return (BasicAuthorization (rqptr));
   else
   if (strsame (rqptr->AuthType, "DIGEST"))
      return (DigestAuthorization (rqptr));
   else
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_SCHEME), FI_LI);
      return (STS$K_ERROR);
   }
}

/*****************************************************************************/
/*
The list is a plain-text string of comma-separated elements, the presence of
which restricts the path to that element.  The element can be an IP host
address (numeric or alphabetic, with or without asterisk wildcard) or an
authenticated username.  IP addresses are recognised by either a leading digit
(for numeric host addresses), a leading asterisk (where hosts are wildcarded
across a domain), a leading "#" which forces an element to be compared to a
host name or address, "http:", "https:", or a leading "~" which indicates a
username.
*/ 

int AuthRestrictList
(
struct RequestStruct *rqptr,
char *RestrictList
)
{
   register char  *lptr, *cptr;

   boolean  CheckedHost,
            CheckedProtocol,
            CheckedUser,
            FoundHost,
            FoundProtocol,
            FoundUser,
            RestrictedOnUser,
            RestrictedOnProtocol,
            RestrictedOnHost;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthRestrictList() |%s|\n", RestrictList);

   RestrictedOnHost = RestrictedOnProtocol = RestrictedOnUser = true;
   FoundHost = FoundProtocol = FoundUser = false;
   lptr = RestrictList;
   while (*lptr)
   {
      CheckedUser = CheckedHost = false;
      while (*lptr && *lptr == ',') lptr++;
      if (isdigit(*lptr) || (lptr[0] == '#' && isdigit(lptr[1])))
      {
         /* numeric IP address */
         if (*lptr == '#') lptr++;
         cptr = rqptr->ClientInternetAddress;
         CheckedHost = FoundHost = true;
         /* drop through to the comparison routine */
      }
      else
      if (*lptr == '~')
      {
         /* authenticated user name */
         lptr++;
         cptr = rqptr->RemoteUser;
         CheckedUser = FoundUser = true;
         /* drop through to the comparison routine */
      }
      else
      if (tolower(*lptr) == 'h' && strsame (lptr, "https:", 6))
      {
         CheckedProtocol = FoundProtocol = true;
         if (rqptr->RequestScheme == SCHEME_HTTPS)
            RestrictedOnProtocol = false;
         while (*lptr && *lptr != ',') lptr++;
         continue;
      }
      else
      if (tolower(*lptr) == 'h' && strsame (lptr, "http:", 5))
      {
         CheckedProtocol = FoundProtocol = true;
         if (rqptr->RequestScheme == SCHEME_HTTP)
            RestrictedOnProtocol = false;
         while (*lptr && *lptr != ',') lptr++;
         continue;
      }
      else
      if (tolower(*lptr) == 'l' &&
          (strsame (lptr, "localhost", -1) ||
           strsame (lptr, "localhost,", 10)))
      {
         /* check reserved-word, server against client host's IP address */
         CheckedHost = FoundHost = true;
         if (strsame (rqptr->ServicePtr->ServerInternetAddress,
                      rqptr->ClientInternetAddress, -1))
            RestrictedOnHost = false;
         while (*lptr && *lptr != ',') lptr++;
         continue;
      }
      else
      {
         /* by default, alpha-numeric IP host name */
         if (*lptr == '#') lptr++;
         cptr = rqptr->ClientHostName;
         CheckedHost = FoundHost = true;
         /* drop through to the comparison routine */
      }
      if (Debug) fprintf (stdout, "|%s|%s|\n", lptr, cptr);

      while (*cptr && *lptr && *lptr != ',')
      {
         if (*lptr == '*')
         {
            while (*lptr == '*') lptr++;
            while (*cptr && tolower(*cptr) != tolower(*lptr)) cptr++;
         }
         while (tolower(*cptr) == tolower(*lptr) && *cptr && *lptr)
         {
            lptr++;
            cptr++;
         }
         if (*lptr != '*') break;
      }
      if (!*cptr && (!*lptr || *lptr == ','))
      {
         if (CheckedUser) RestrictedOnUser = false;
         if (CheckedHost) RestrictedOnHost = false;
      }
      while (*lptr && *lptr != ',') lptr++;
   }

   if (FoundProtocol && RestrictedOnProtocol) return (RESTRICTED_BY_PROTOCOL);
   if (FoundHost && RestrictedOnHost) return (RESTRICTED_BY_HOSTNAME);
   if (FoundUser && RestrictedOnUser) return (RESTRICTED_BY_USERNAME);
   return (0);
}

/*****************************************************************************/
/*
Look for an authentication record with the realm and username in the linked-
list, binary tree.  If one doesn't exists create a new one.  Check the supplied
password against the one in the record.  If it matches then authentication has
succeeded.  If not, the check the supplied password against the database.  If
it matches then copy the supplied password into the authentication record and
the authentication succeeds.  If it doesn't match then the authentication
fails.  Return SS$_INVLOGIN to indicate authentication failure, SS$_NORMAL
indicating authentication success, or any other error status indicating any
other error.
*/ 

int AuthDoAuthorization (struct RequestStruct *rqptr)

{
   static int  NoDuplicatesFlag = 0;

   register char  *cptr, *sptr, *zptr;

   boolean  Authenticated,
            SysUafAuthenticated;
   int  status,
        Minutes;
   struct AuthCacheRecordStruct  AuthCacheRecord;
   struct AuthCacheRecordStruct  *AuthCacheRecordPtr,
                                 *TreeNodePtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthDoAuthorization()\n");

   rqptr->AuthUserCan = 0;

   if (rqptr->AuthScheme == AUTH_SCHEME_DIGEST)
   {
      Accounting.AuthDigestCount++;
      if (!rqptr->RemoteUser[0] || !rqptr->AuthDigestResponsePtr[0])
         return (SS$_INVLOGIN);
   }
   else
   if (rqptr->AuthScheme == AUTH_SCHEME_BASIC)
   {
      Accounting.AuthBasicCount++;
      if (!rqptr->RemoteUser[0] || !rqptr->RemoteUserPassword[0])
         return (SS$_INVLOGIN);
   }
   else
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

   /*********************/
   /* lookup the record */
   /*********************/

   cptr = rqptr->AuthRealmPtr;
   zptr = (sptr = AuthCacheRecord.Realm) + sizeof(AuthCacheRecord.Realm);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      return (STS$K_ERROR); 
   }
   *sptr = '\0';
   AuthCacheRecord.RealmLength = sptr - AuthCacheRecord.Realm;

   cptr = rqptr->AuthGroupPtr;
   zptr = (sptr = AuthCacheRecord.Group) + sizeof(AuthCacheRecord.Group);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      return (STS$K_ERROR); 
   }
   *sptr = '\0';
   AuthCacheRecord.GroupLength = sptr - AuthCacheRecord.Group;

   cptr = rqptr->RemoteUser;
   zptr = (sptr = AuthCacheRecord.UserName) + sizeof(AuthCacheRecord.UserName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_USERNAME_LENGTH), FI_LI);
      return (STS$K_ERROR); 
   }
   *sptr = '\0';
   AuthCacheRecord.UserNameLength = sptr - AuthCacheRecord.UserName;

   cptr = rqptr->RemoteUserPassword;
   zptr = (sptr = AuthCacheRecord.Password) + sizeof(AuthCacheRecord.Password);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_PASSWORD_LENGTH), FI_LI);
      return (STS$K_ERROR); 
   }
   *sptr = '\0';

   status = lib$lookup_tree (&AuthCacheTreeHead,
                             &AuthCacheRecord,
                             &AuthCacheTreeCompareRecord,
                             &TreeNodePtr);
   if (Debug) fprintf (stdout, "lib$lookup_tree() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      /*******************/
      /* existing record */
      /*******************/

      if (Debug)
         fprintf (stdout,
"AuthCacheRecord |%s|%s|%s|%08.08X|\n\
cache: %d access: %d failure: %d\n\
VmsUserProfile: %d %d\n",
            TreeNodePtr->Realm, TreeNodePtr->Group, TreeNodePtr->UserName,
            TreeNodePtr->AuthUserCan, TreeNodePtr->CacheMinutes,
            TreeNodePtr->AccessCount, TreeNodePtr->FailureCount,
            TreeNodePtr->VmsUserProfileLength,
            TreeNodePtr->VmsUserProfilePtr);

      TreeNodePtr->LastAccessBinTime[0] = rqptr->BinaryTime[0];
      TreeNodePtr->LastAccessBinTime[1] = rqptr->BinaryTime[1];

      if (TreeNodePtr->FailureCount >= Config.AuthFailureLimit)
      {
         /*********************/
         /* TOO MANY FAILURES */
         /*********************/

         TreeNodePtr->FailureCount++;

         fprintf (stdout, "%%%s-W-AUTHFAILIM, %s, %d %s %s %s\n",
                  Utility, DateTime(&rqptr->BinaryTime, 20),
                  TreeNodePtr->FailureCount, TreeNodePtr->Realm,
                  TreeNodePtr->UserName, rqptr->ClientHostName);

         memset (TreeNodePtr->Password, 0, sizeof(TreeNodePtr->Password));
         memset (TreeNodePtr->DigestResponse, 0,
                 sizeof(TreeNodePtr->DigestResponse));
         return (SS$_INVLOGIN); 
      }

      if (Config.AuthRevalidateUserMinutes)
      {
         /********************/
         /* revalidate user? */
         /********************/

         /*
            User validation only lasts for a limited period (< 24 hours).
            Check how long ago was the last authorization attempt, if that is
            exceeded force the user to reenter via browser dialog.
         */

         Minutes = rqptr->NumericTime[3] * 60 + rqptr->NumericTime[4];
         if (Minutes < TreeNodePtr->RevalidateUserMinutes)
             TreeNodePtr->RevalidateUserMinutes =
                1440 - TreeNodePtr->RevalidateUserMinutes;
         if (Debug)
            fprintf (stdout, "Minutes %d + %d > %d\n",
               TreeNodePtr->RevalidateUserMinutes,
               Config.AuthRevalidateUserMinutes, Minutes);
         if (Minutes >= TreeNodePtr->RevalidateUserMinutes +
                        Config.AuthRevalidateUserMinutes)
         {
            TreeNodePtr->Password[0]  = '\0';
            TreeNodePtr->RevalidateUserMinutes = Minutes;
            return (SS$_INVLOGIN); 
         }
      }

      /*********************/
      /* revalidate cache? */
      /*********************/
      /*
         Cache authentication only lasts for a limited period (< 24 hours).
         Check how long ago was the last authorization attempt.
      */
      Minutes = rqptr->NumericTime[3] * 60 + rqptr->NumericTime[4];
      if (Minutes < TreeNodePtr->CacheMinutes)
          TreeNodePtr->CacheMinutes = 1440 - TreeNodePtr->CacheMinutes;
      if (Debug)
         fprintf (stdout, "Minutes %d + %d > %d\n",
            TreeNodePtr->CacheMinutes, Config.AuthCacheMinutes, Minutes);
      if (Minutes >= TreeNodePtr->CacheMinutes + Config.AuthCacheMinutes)
      {
         TreeNodePtr->Password[0]  = '\0';
         TreeNodePtr->CacheMinutes = Minutes;
         /* drop through to recheck authentication */
      }

      /************************/
      /* check authentication */
      /************************/

      if (rqptr->AuthScheme == AUTH_SCHEME_DIGEST)
      {
         if (TreeNodePtr->DigestResponse[0] &&
             !strcmp (rqptr->AuthDigestResponsePtr,
                      TreeNodePtr->DigestResponse))
         {
            Authenticated = true;
            TreeNodePtr->DigestCount++;
         }
         else
            Authenticated = false;
      }
      else
      if (rqptr->AuthScheme == AUTH_SCHEME_BASIC)
      {
         if (TreeNodePtr->Password[0] &&
             !strcmp (rqptr->RemoteUserPassword, TreeNodePtr->Password))
         {
            Authenticated = true;
            TreeNodePtr->BasicCount++;
         }
         else
            Authenticated = false;
      }

      if (Authenticated)
      {
         /*****************/
         /* authenticated */
         /*****************/

         TreeNodePtr->FailureCount = 0;
         TreeNodePtr->AccessCount++;
         rqptr->AuthUserCan = TreeNodePtr->AuthUserCan;
         rqptr->AuthVmsUserProfilePtr = TreeNodePtr->VmsUserProfilePtr;
         rqptr->AuthVmsUserProfileLength = TreeNodePtr->VmsUserProfileLength;
         rqptr->AuthCanChangeSysUafPwd = TreeNodePtr->CanChangeSysUafPwd;

         if (Debug) fprintf (stdout, "authenticated (in tree)!\n");
         return (SS$_NORMAL); 
      }

      /* any VMS profile is no longer valid */
      if (TreeNodePtr->VmsUserProfilePtr != NULL)
         VmFree (TreeNodePtr->VmsUserProfilePtr);
      TreeNodePtr->VmsUserProfileLength = 0;
      TreeNodePtr->VmsUserProfilePtr = NULL;
   }
   else
   if (status == LIB$_KEYNOTFOU)
   {
      /****************************************/
      /* record not found in tree, create one */
      /****************************************/

      AuthCacheRecordPtr = VmGet (sizeof(struct AuthCacheRecordStruct));
      
      /* we know these'll fit, sizes have been checked during authentication */
      strcpy (AuthCacheRecordPtr->Realm, AuthCacheRecord.Realm);
      AuthCacheRecordPtr->RealmLength = AuthCacheRecord.RealmLength;
      strcpy (AuthCacheRecordPtr->Group, AuthCacheRecord.Group);
      AuthCacheRecordPtr->GroupLength = AuthCacheRecord.GroupLength;
      strcpy (AuthCacheRecordPtr->UserName, AuthCacheRecord.UserName);
      AuthCacheRecordPtr->UserNameLength = AuthCacheRecord.UserNameLength;
      AuthCacheRecordPtr->CanChangeSysUafPwd =
         AuthCacheRecordPtr->HttpsOnly =
         AuthCacheRecordPtr->SysUafAuthenticated = false;
      AuthCacheRecordPtr->AccessCount = AuthCacheRecordPtr->BasicCount = 
         AuthCacheRecordPtr->DigestCount = AuthCacheRecordPtr->DataBaseCount =
         AuthCacheRecordPtr->FailureCount =
         AuthCacheRecordPtr->VmsUserProfileLength = 0;
      AuthCacheRecordPtr->VmsUserProfilePtr = NULL;
      Minutes = rqptr->NumericTime[3] * 60 + rqptr->NumericTime[4];
      AuthCacheRecordPtr->CacheMinutes =
         AuthCacheRecordPtr->RevalidateUserMinutes = Minutes;
      AuthCacheRecordPtr->LastAccessBinTime[0] = rqptr->BinaryTime[0];
      AuthCacheRecordPtr->LastAccessBinTime[1] = rqptr->BinaryTime[1];

      if (VMSnok (status =
          lib$insert_tree (&AuthCacheTreeHead, AuthCacheRecordPtr,
                           &NoDuplicatesFlag, &AuthCacheTreeCompareRecord,
                           &AuthCacheTreeAllocateRecord, &TreeNodePtr, 0)))
         VmFree (AuthCacheRecordPtr);
      if (Debug) fprintf (stdout, "lib$insert_tree() %%X%08.08X\n", status);
   }

   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /**********************************************/
   /* get authentication from appropriate source */
   /**********************************************/

   SysUafAuthenticated = false;
   TreeNodePtr->DataBaseCount++;

   if (AuthPromiscuous)
   {
      status = SS$_NORMAL;
      if (AuthPromiscuousPwdPtr != NULL)
         if (!strsame (rqptr->RemoteUserPassword, AuthPromiscuousPwdPtr, -1))
            status = SS$_INVLOGIN;

      if (VMSok (status))
      {
         /* allow the user to do anything */
         rqptr->AuthUserCan = HTTP_METHOD_DELETE |
                              HTTP_METHOD_GET |
                              HTTP_METHOD_HEAD |
                              HTTP_METHOD_POST |
                              HTTP_METHOD_PUT;
      }
   }
   else
   {
      /* password may be checked against VMS or HTA (HTTPd) database */
      if (!strcmp (rqptr->AuthRealmPtr, AUTH_VMS))
      {
         /* check against SYSUAF */
         Accounting.AuthVmsCount++;
         SysUafAuthenticated = true;
         status = AuthVerifyVmsPassword (rqptr);
         /*
            SYSUAF authentication can only be performed using the basic
            scheme (it needs a clear-text password to hash and then compare)
         */
         rqptr->AuthChallengeScheme = AUTH_SCHEME_BASIC;
      }
      else
      {
         /* check against REALM PASSWORD database */
         Accounting.AuthHtDatabaseCount++;
         status = AuthReadHtDatabase (rqptr, rqptr->AuthRealmPtr, true);

         /* check against HTA group database */
         if (VMSok (status) && rqptr->AuthGroupPtr[0])
         {
            /* check against GROUP capabilities database */
            status = AuthReadHtDatabase (rqptr, rqptr->AuthGroupPtr, false);
            /* if not in database then not in group! */
            if (status == SS$_INVLOGIN) status = RESTRICTED_BY_OTHER;
         }
      }
   }

   /* can only use this authentication with "https:" (SSL)? */
   if (rqptr->AuthHttpsOnly &&
       rqptr->RequestScheme != SCHEME_HTTPS)
      status = RESTRICTED_BY_PROTOCOL;

   if (VMSnok (status))
   {
      if (status != SS$_INVLOGIN) return (status);
      Authenticated = false;
   }
   else
      Authenticated = true;

   if (Debug)
      fprintf (stdout,
"Authenticated: %d\nAuthCacheRecord |%s|%s|%08.08X| access: %d failure: %d\n",
         Authenticated,
         TreeNodePtr->Realm, TreeNodePtr->UserName, TreeNodePtr->AuthUserCan,
         TreeNodePtr->AccessCount, TreeNodePtr->FailureCount);

   if (Authenticated)
   {
      /*****************/
      /* authenticated */
      /*****************/

      if (rqptr->AuthScheme == AUTH_SCHEME_BASIC)
      {
         strcpy (TreeNodePtr->Password, rqptr->RemoteUserPassword);
         TreeNodePtr->BasicCount++;
      }
      else
      if (rqptr->AuthScheme == AUTH_SCHEME_DIGEST)
      {
         strcpy (TreeNodePtr->DigestResponse, rqptr->AuthDigestResponsePtr);
         TreeNodePtr->DigestCount++;
      }

      if (TreeNodePtr->FailureCount)
      {
         /* just let the log know the access finally succeeded */
         fprintf (stdout, "%%%s-I-AUTHFAILOK, %s, %d %s %s %s\n",
                  Utility, DateTime(&rqptr->BinaryTime, 20),
                  TreeNodePtr->FailureCount, TreeNodePtr->Realm,
                  TreeNodePtr->UserName, rqptr->ClientHostName);
      }

      TreeNodePtr->FailureCount = 0;
      TreeNodePtr->AccessCount++;
      TreeNodePtr->AuthUserCan = rqptr->AuthUserCan;
      TreeNodePtr->HttpsOnly = rqptr->AuthHttpsOnly;
      TreeNodePtr->CanChangeSysUafPwd = rqptr->AuthCanChangeSysUafPwd;
      TreeNodePtr->VmsUserProfileLength = 0;
      TreeNodePtr->VmsUserProfilePtr = NULL;
      TreeNodePtr->SysUafAuthenticated = rqptr->AuthSysUafAuthenticated =
         SysUafAuthenticated;

      status = SS$_NORMAL;

      if (SysUafAuthenticated && AuthVmsUserSecProfileEnabled)
      {
         status = AuthCreateVmsUserProfile (rqptr);
         TreeNodePtr->VmsUserProfilePtr = rqptr->AuthVmsUserProfilePtr;
         TreeNodePtr->VmsUserProfileLength = rqptr->AuthVmsUserProfileLength;
      }

      if (Debug) fprintf (stdout, "authenticated!\n");

      return (status); 
   }
   else
   {
      /*********************/
      /* not authenticated */
      /*********************/

      TreeNodePtr->FailureCount++;

      fprintf (stdout, "%%%s-W-AUTHFAIL, %s, %d %s %s %s\n",
               Utility, DateTime(&rqptr->BinaryTime, 20),
               TreeNodePtr->FailureCount, TreeNodePtr->Realm,
               TreeNodePtr->UserName, rqptr->ClientHostName);

      return (SS$_INVLOGIN);
   }
}

/*****************************************************************************/
/*
Called by lib$insert_tree() and lib$lookup_tree() in a variety of functions.
Returns negative number if user node is lexicographically less than the tree
node, positive if greater, or zero if exactly the same.
*/ 

int AuthCacheTreeCompareRecord
(
struct AuthCacheRecordStruct *AuthCacheRecordPtr,
struct AuthCacheRecordStruct *TreeNodePtr,
unsigned long Unused
)
{
   register int  cmp;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCacheTreeCompareRecord()\n");

   if (cmp = AuthCacheRecordPtr->RealmLength - TreeNodePtr->RealmLength)
       return (cmp);
   if (cmp = AuthCacheRecordPtr->GroupLength - TreeNodePtr->GroupLength)
       return (cmp);
   if (cmp = AuthCacheRecordPtr->UserNameLength - TreeNodePtr->UserNameLength)
       return (cmp);

   if (cmp = strcmp (AuthCacheRecordPtr->Realm, TreeNodePtr->Realm))
      return (cmp);
   if (cmp = strcmp (AuthCacheRecordPtr->Group, TreeNodePtr->Group))
      return (cmp);
   return (strcmp (AuthCacheRecordPtr->UserName, TreeNodePtr->UserName));
}

/*****************************************************************************/
/*
Called by lib$insert_tree() in AuthDoAuthorization().  Insert the address
in 'UserNodePtr' into the location pointed at by 'NewTreeNodePtrAddress'
(inside the binary tree structure presumably!)
*/ 

AuthCacheTreeAllocateRecord
(
struct AuthCacheRecordStruct *UserNodePtr,
struct AuthCacheRecordStruct **NewTreeNodePtrAddress,
unsigned long Unused
)
{
   if (Debug) fprintf (stdout, "AuthCacheTreeAllocateRecord()\n");

   *NewTreeNodePtrAddress = UserNodePtr;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Clear all of the authentication records from the tree.  Will result in all
subsequent requests being re-authenticated from their respective on-disk
databases.  Called from the Admin.c and Control.c modules.
*/

int AuthCacheTreeFree ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCacheTreeFree()\n");

   sys$gettim (&AuthCacheTreeBinTime);
   if (!AuthCacheTreeHead) return (SS$_NORMAL);
   AuthFreeRecordCount = 0;

   status = lib$traverse_tree (&AuthCacheTreeHead,
                               &AuthCacheTreeFreeRecord, 0);
   if (Debug) fprintf (stdout, "lib$traverse_tree() %%X%08.08X\n", status);

   if (AuthCacheTreeTailPtr) VmFree (AuthCacheTreeTailPtr);
   AuthCacheTreeHead = AuthCacheTreeTailPtr = 0;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Free the dynamically allocated memory associated with the user tree record.
Called from AuthCacheTreeFree().
*/

AuthCacheTreeFreeRecord
(                      
struct AuthCacheRecordStruct *TreeNodePtr,
unsigned long  Unused
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCacheTreeFreeRecord()\n");

   if (Debug)
      fprintf (stdout, "AuthCacheRecord |%s|%s|\n",
               TreeNodePtr->Realm, TreeNodePtr->UserName);

   AuthFreeRecordCount++;

   if (TreeNodePtr->LeftLink)
   {
      /* any VMS profile is no longer valid either */
      if (TreeNodePtr->LeftLink->VmsUserProfilePtr != NULL)
         VmFree (TreeNodePtr->LeftLink->VmsUserProfilePtr);

      VmFree (TreeNodePtr->LeftLink);
   }
   else
      AuthCacheTreeTailPtr = TreeNodePtr;
}

/****************************************************************************/
/*
Verify the request username/password and/or read capabilities from the on-disk
HTA (HTTPd) authroization database.

Returns a success status if user password authenticated, SS$_INVLOGIN if not
authenticated, or other error status if a genuine VMS error occurs (which
should be reported where and when it occurs).

The digest is stored in both upper and lower case versions because clients
will have case-lock either on or off producing a digest of either upper or
lower case username and password.  Keep digests of both so that provided one
or other case is used exclusively the digest thereof can still be matched
with one or other stored here :^)
*/ 

int AuthReadHtDatabase
(
struct RequestStruct* rqptr,
char *DatabaseName,
boolean AuthenticatePassword
)
{
   static $DESCRIPTOR (PasswordDsc, "");
   static $DESCRIPTOR (UserNameDsc, "");

   boolean  PasswordAuthenticated,
            UserNameEnabled;
   int  status;
   unsigned long  HashedPwd [2];
   char  A1HexDigest [33],
         HexDigest [33];
   struct AuthHtRecordStruct AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthReadHtDatabase() |%s|%s|\n",
               DatabaseName, rqptr->RemoteUser);

   /* set the capabilities to zero before we do anything else! */
   rqptr->AuthUserCan = 0;

   /* look for the record, leave the database file open if found */
   status = AuthAccessHtDatabase (true, DatabaseName, rqptr->RemoteUser,
                                  &AuthHtRecord, NULL, NULL);
   if (status == SS$_INVLOGIN)
   {
      AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, NULL);
      return (status);
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_DATABASE);
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /****************/
   /* record found */
   /****************/

   PasswordAuthenticated = false;

   if (AuthHtRecord.Flags & AUTH_FLAG_ENABLED)
   {
      UserNameEnabled = true;
      if (AuthenticatePassword &&
          AuthHtRecord.HashedPwd[0] && AuthHtRecord.HashedPwd[1])
      {
         if (rqptr->AuthScheme == AUTH_SCHEME_BASIC)
         {
            if (Debug) fprintf (stdout, "BASIC\n");

            UserNameDsc.dsc$w_length = strlen(rqptr->RemoteUser);
            UserNameDsc.dsc$a_pointer = rqptr->RemoteUser;
            PasswordDsc.dsc$w_length = strlen(rqptr->RemoteUserPassword);
            PasswordDsc.dsc$a_pointer = rqptr->RemoteUserPassword;

            status = sys$hash_password (&PasswordDsc, UAI$C_PURDY_S, 0,
                                        &UserNameDsc, &HashedPwd);
            if (Debug)
               fprintf (stdout, "sys$hash_password() %%X%08.08X\n", status);
            if (VMSnok (status))
            {
               /* ensure the currently open database is closed */
               AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, NULL);
               rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
               ErrorVmsStatus (rqptr, status, FI_LI);
               return (status);
            }

            if (Debug)
               fprintf (stdout, "|%08.08X%08.08X|%08.08X%08.08X|\n",
                        HashedPwd[1], HashedPwd[0],
                        AuthHtRecord.HashedPwd[1], AuthHtRecord.HashedPwd[0]);

            if (HashedPwd[0] == AuthHtRecord.HashedPwd[0] &&
                HashedPwd[1] == AuthHtRecord.HashedPwd[1])
               PasswordAuthenticated = true;
         }
         else
         if (rqptr->AuthScheme == AUTH_SCHEME_DIGEST)
         {
            if (Debug)
               fprintf (stdout, "DIGEST |%s|\n",
                        rqptr->AuthDigestResponsePtr);

            DigestHexString (AuthHtRecord.A1DigestLoCase, 16, A1HexDigest);
            DigestResponse (A1HexDigest,
                            rqptr->AuthDigestNoncePtr,
                            rqptr->HttpMethodName,
                            rqptr->AuthDigestUriPtr,
                            HexDigest);
            if (!strcmp (rqptr->AuthDigestResponsePtr, HexDigest))
               PasswordAuthenticated = true;
            else
            {
               DigestHexString (AuthHtRecord.A1DigestUpCase, 16, A1HexDigest);
               DigestResponse (A1HexDigest,
                               rqptr->AuthDigestNoncePtr,
                               rqptr->HttpMethodName,
                               rqptr->AuthDigestUriPtr,
                               HexDigest);
               if (!strcmp (rqptr->AuthDigestResponsePtr, HexDigest))
                  PasswordAuthenticated = true;
            }
         }

         if (Debug)
            fprintf (stdout, "PasswordAuthenticated: %d\n",
                     PasswordAuthenticated);
      }

      if (!AuthenticatePassword || PasswordAuthenticated)
      {
         /* set the appropriate authorization flags */
         if (AuthHtRecord.Flags & AUTH_FLAG_DELETE)
            rqptr->AuthUserCan |= HTTP_METHOD_DELETE;
         if (AuthHtRecord.Flags & AUTH_FLAG_GET)
            rqptr->AuthUserCan |= HTTP_METHOD_GET | HTTP_METHOD_HEAD;
         if (AuthHtRecord.Flags & AUTH_FLAG_HEAD)
            rqptr->AuthUserCan |= HTTP_METHOD_HEAD;
         if (AuthHtRecord.Flags & AUTH_FLAG_POST)
            rqptr->AuthUserCan |= HTTP_METHOD_POST;
         if (AuthHtRecord.Flags & AUTH_FLAG_PUT)
            rqptr->AuthUserCan |= HTTP_METHOD_PUT;

         if (AuthHtRecord.Flags & AUTH_FLAG_HTTPS_ONLY)
            rqptr->AuthHttpsOnly = true;
         else
            rqptr->AuthHttpsOnly = false;
      }
   }

   if (UserNameEnabled && (!AuthenticatePassword || PasswordAuthenticated))
   {
      AuthHtRecord.AccessCount++;
      memcpy (&AuthHtRecord.LastAccessBinTime, &rqptr->BinaryTime, 8);
   }
   else
   {
      AuthHtRecord.FailureCount++;
      memcpy (&AuthHtRecord.LastFailureBinTime, &rqptr->BinaryTime, 8);
   }

   /* update the record, close the database file */
   status = AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, &AuthHtRecord);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_DATABASE);
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   if (!AuthenticatePassword || PasswordAuthenticated)
      return (SS$_NORMAL);
   else
      return (SS$_INVLOGIN);
}

/****************************************************************************/
/*
Locate a specified user record in a specified on-disk database.  Either return
that record if found or if an update record supplied, update it! Can return any
RMS or VMS error status code.  Returns SS$_NORMAL is operation (read, add or
update) completed successfully.  Returns SS$_INVLOGIN if the record could not
be found.  Returns SS$_INCOMPAT if the database file version is incorrect.

THE DATABASE FILE IS ALWAYS CLOSED ON AN ERROR CONDITION.

'LeaveFileOpen' will leave the database file open with the current record
context ready for update, or for locating another user record.  This function
is not reentrant and therefore this context is valid ONLY within the one AST.

Call with 'DatabaseName' and 'UserName' non-NULL and all other pointers
set to NULL to check whether the user name has a record in the database.
*/ 

int AuthAccessHtDatabase
(
boolean LeaveFileOpen,
char *DatabaseName,
char *UserName,
struct AuthHtRecordStruct *AuthHtRecordReadPtr,
struct AuthHtRecordStruct *AuthHtRecordAddPtr,
struct AuthHtRecordStruct *AuthHtRecordUpdatePtr
)
{
   static char  AuthFileName [64],
                PrevDatabaseName [64];
   static unsigned short  AuthFileNameLength;
   static $DESCRIPTOR (AuthFileNameFaoDsc, "!AZ!AZ!AZ");
   static $DESCRIPTOR (AuthFileNameDsc, AuthFileName);
   static struct FAB  AuthFileFab;
   static struct NAM  AuthFileNam;
   static struct RAB  AuthFileRab;

   register char  *cptr, *sptr;

   int  status,
        UserNameLength;
   struct AuthHtRecordStruct *AuthHtRecordPtr;
   struct AuthHtRecordStruct AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthAccessHtDatabase() %d |%s|%s| %d %d %d\n",
               LeaveFileOpen, DatabaseName, UserName,
               AuthHtRecordPtr, AuthHtRecordAddPtr, AuthHtRecordUpdatePtr);

   if (AuthHtRecordReadPtr == NULL &&
       AuthHtRecordAddPtr == NULL &&
       AuthHtRecordUpdatePtr == NULL)
   {
      /************************/
      /* force database close */
      /************************/

      sys$close (&AuthFileFab, 0, 0);
      return (SS$_NORMAL);
   }

   if (UserName == NULL && AuthHtRecordUpdatePtr != NULL)
   {
      /*********************************/
      /* update previously read record */
      /*********************************/

      AuthFileRab.rab$l_rbf = AuthHtRecordUpdatePtr;
      AuthFileRab.rab$w_rsz = sizeof(struct AuthHtRecordStruct);
      status = sys$update (&AuthFileRab, 0, 0);
      if (Debug) fprintf (stdout, "sys$update() %%X%08.08X\n", status);
      if (VMSok (status))
      {
         /* update completed successfully */
         if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
         return (SS$_NORMAL);
      }
      /* update error */
      sys$close (&AuthFileFab, 0, 0);
      return (status);
   }

   if (DatabaseName == NULL)
   {
      /*****************************************/
      /* locate within currently open database */
      /*****************************************/

      if (VMSnok (status = sys$rewind (&AuthFileRab, 0, 0)))
      {
         sys$close (&AuthFileFab, 0, 0);
         return (status);
      }
      if (Debug) fprintf (stdout, "sys$rewind() %%X%08.08X\n", status);
   }
   else
   {
      /**********************************/
      /* open currently closed database */
      /**********************************/

      if (strcmp (DatabaseName, PrevDatabaseName))
      {
         sys$fao (&AuthFileNameFaoDsc, &AuthFileNameLength, &AuthFileNameDsc,
                  HTA_DIRECTORY, DatabaseName, HTA_FILE_TYPE);
         AuthFileName[AuthFileNameLength] = '\0';
      }
      if (Debug) fprintf (stdout, "AuthFileName |%s|\n", AuthFileName);

      AuthFileFab = cc$rms_fab;
      /* if a read-only operation without file left open then read-only! */
      if (!LeaveFileOpen &&
          AuthHtRecordAddPtr == NULL &&
          AuthHtRecordUpdatePtr == NULL)
         AuthFileFab.fab$b_fac = FAB$M_GET;
      else
         AuthFileFab.fab$b_fac = FAB$M_GET | FAB$M_PUT | FAB$M_UPD;
      AuthFileFab.fab$l_fna = AuthFileName;  
      AuthFileFab.fab$b_fns = AuthFileNameLength;
      AuthFileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;

      /* turn on SYSPRV to allow access to authentication database file */
      EnableSysPrv ();
      status = sys$open (&AuthFileFab, 0, 0);
      DisableSysPrv ();

      /* status from sys$open() */
      if (VMSnok (status)) return (status);

      AuthFileRab = cc$rms_rab;
      AuthFileRab.rab$l_fab = &AuthFileFab;
      AuthFileRab.rab$b_mbf = 2;
      AuthFileRab.rab$l_rop = RAB$M_RAH;

      if (VMSnok (status = sys$connect (&AuthFileRab, 0, 0)))
      {
         sys$close (&AuthFileFab, 0, 0);
         return (status);
      }
   }

   /*****************/
   /* locate record */
   /*****************/

   /* for add and update read into the scratch record */
   if (AuthHtRecordReadPtr == NULL)
      AuthFileRab.rab$l_ubf = AuthHtRecordPtr = &AuthHtRecord;
   else
      AuthFileRab.rab$l_ubf = AuthHtRecordPtr = AuthHtRecordReadPtr;
   AuthFileRab.rab$w_usz = sizeof(struct AuthHtRecordStruct);

   if (UserName != NULL) UserNameLength = strlen(UserName);

   while (VMSok (status = sys$get (&AuthFileRab, 0, 0)))
   {
      /* check the version of the authorization database */
      if (AuthHtRecordPtr->DatabaseVersion &&
          AuthHtRecordPtr->DatabaseVersion != AuthCurrentDatabaseVersion)
      {
         status = SS$_INCOMPAT & 0xfffffffe;
         break;
      }

      /* if deleted record (all set to zeroes) continue */
      if (AuthHtRecordAddPtr == NULL && !AuthHtRecordPtr->UserNameLength)
         continue;

      /* if adding a record then use the first deleted one found */
      if (AuthHtRecordAddPtr != NULL && !AuthHtRecordPtr->UserNameLength)
         break;
/**
      if (Debug)
         fprintf (stdout, "AuthHtRecordPtr->UserName |%s|\n",
                  AuthHtRecordPtr->UserName);
**/
      if (UserNameLength != AuthHtRecordPtr->UserNameLength) continue;
      cptr = AuthHtRecordPtr->UserName;
      sptr = UserName;
      while (*cptr && *sptr && tolower(*cptr) == tolower(*sptr))
         { cptr++; sptr++; }
      if (*cptr || *sptr) continue;
      break;
   }
   if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status);

   if (AuthHtRecordUpdatePtr == NULL && AuthHtRecordAddPtr == NULL)
   {
      /***************/
      /* read record */
      /***************/

      if (VMSok (status))
      {
         /* record found */
         if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
         return (SS$_NORMAL);
      }

      if (status == RMS$_EOF)
      {
         /* user record not found */
         if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
         return (SS$_INVLOGIN);
      }
      /* RMS error */
      sys$close (&AuthFileFab);
      return (status);
   }

   if (AuthHtRecordUpdatePtr != NULL)
   {
      /*****************/
      /* update record */
      /*****************/

      if (VMSnok (status))
      {
         if (status == RMS$_EOF)
         {
            /* user record not found */
            if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
            return (SS$_INVLOGIN);
         }
         /* RMS error */
         sys$close (&AuthFileFab, 0, 0);
         return (status);
      }

      AuthFileRab.rab$l_rbf = AuthHtRecordUpdatePtr;
      AuthFileRab.rab$w_rsz = sizeof(struct AuthHtRecordStruct);

      status = sys$update (&AuthFileRab, 0, 0);
      if (Debug) fprintf (stdout, "sys$update() %%X%08.08X\n", status);
      if (VMSok (status))
      {
         /* update completed successfully */
         if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
         return (SS$_NORMAL);
      }
      /* RMS error */
      sys$close (&AuthFileFab, 0, 0);
      return (status);
   }

   if (AuthHtRecordAddPtr != NULL)
   {
      /****************/
      /* add a record */
      /****************/

      if (VMSnok (status) && status != RMS$_EOF)
      {
         /* RMS error */
         sys$close (&AuthFileFab, 0, 0);
         return (status);
      }

      AuthFileRab.rab$l_rbf = AuthHtRecordAddPtr;
      AuthFileRab.rab$w_rsz = sizeof(struct AuthHtRecordStruct);

      /* if reached end-of-file then add a record, else undate zeroed one */
      if (status == RMS$_EOF)
         status = sys$put (&AuthFileRab, 0, 0);
      else
         status = sys$update (&AuthFileRab, 0, 0);
      if (Debug) fprintf (stdout, "sys$put/update() %%X%08.08X\n", status);
      if (VMSok (status))
      {
         /* put or update completed successfully */
         if (!LeaveFileOpen) sys$close (&AuthFileFab, 0, 0);
         return (SS$_NORMAL);
      }
      /* RMS error */
      sys$close (&AuthFileFab, 0, 0);
      return (status);
   }

   /* sanity check error, should never get here */
   return (SS$_BUGCHECK);
}

/*****************************************************************************/
/*
Verify the request username/password from the on-disk SYSUAF database.

Get the specified user's flags, authorized privileges, quadword password, hash
salt and encryption algorithm from SYSUAF.  If any of specified bits in the
flags are set (e.g. "disusered") then fail the password authentication.
Using the salt and encryption algorithm hash the supplied password and compare
it to the UAF hashed password.  A SYSPRV privileged account is always failed.

Returns a success status if user password authenticated, SS$_INVLOGIN if not
authenticated, or other error status if a genuine VMS error occurs (which
should be reported where and when it occurs).
*/ 

int AuthVerifyVmsPassword (struct RequestStruct* rqptr)

{
   static unsigned long  DisallowFlags =
          UAI$M_DISACNT | UAI$M_PWD_EXPIRED | UAI$M_PWD2_EXPIRED |
          UAI$M_CAPTIVE | UAI$M_RESTRICTED;

   static unsigned long  Context = -1;

   int  status;
   unsigned long  UaiFlags,
                  UaiUic;
   unsigned long  UaiPriv [2],
                  HashedPwd [2],
                  UaiPwd [2];
   unsigned short  UaiSalt;
   unsigned char  UaiEncrypt;
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } items [] = 
   {
      { sizeof(UaiUic), UAI$_UIC, &UaiUic, 0 },
      { sizeof(UaiFlags), UAI$_FLAGS, &UaiFlags, 0 },
      { sizeof(UaiPriv), UAI$_PRIV, &UaiPriv, 0 },
      { sizeof(UaiPwd), UAI$_PWD, &UaiPwd, 0 },
      { sizeof(UaiEncrypt), UAI$_ENCRYPT, &UaiEncrypt, 0 },
      { sizeof(UaiSalt), UAI$_SALT, &UaiSalt, 0 },
      { 0, 0, 0, 0 }
   };
   $DESCRIPTOR (UserNameDsc, rqptr->RemoteUser);
   $DESCRIPTOR (PasswordDsc, rqptr->RemoteUserPassword);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthVerifyVmsPassword()\n");

   /* double-check! */
   if (!AuthSysUafEnabled) return (SS$_INVLOGIN);

   /* turn on SYSPRV to allow access to SYSUAF records */
   EnableSysPrv ();

   UserNameDsc.dsc$w_length = strlen(rqptr->RemoteUser);
   status = sys$getuai (0, &Context, &UserNameDsc, &items, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$getuai() %%X%08.08X\n", status);

   /* turn off SYSPRV */
   DisableSysPrv ();

   if (VMSnok (status)) 
   {
      if (status == RMS$_RNF) return (SS$_INVLOGIN);
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_DATABASE_VMS);
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /* automatically disallow if any of these flags are set! */
   if (UaiFlags & DisallowFlags)
   {
      if (Debug) fprintf (stdout, "failed on flags\n");
      return (SS$_INVLOGIN);
   }

   PasswordDsc.dsc$w_length = strlen(rqptr->RemoteUserPassword);
   status = sys$hash_password (&PasswordDsc, UaiEncrypt,
                               UaiSalt, &UserNameDsc, &HashedPwd);
   if (Debug) fprintf (stdout, "sys$hash_password() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   if (HashedPwd[0] != UaiPwd[0] || HashedPwd[1] != UaiPwd[1])
   {
      if (Debug) fprintf (stdout, "failed on password match\n");
      if (!AuthSysUafPromiscuous) return (SS$_INVLOGIN);
   }

   /* identifier checking overrides privilege checking! */
   if (AuthSysUafIdentifier)
   {
      /* check SYSUAF-authenticated account holds the correct identifier */
      return (AuthSysUafCheckIdentifier (rqptr, UaiUic));
   }
   else
   if (!AuthSysUafAllAccounts && (UaiPriv[0] & PRV$M_SYSPRV))
   {
      /* not allowing all accounts, exclude those with extended privileges */
      if (Debug) fprintf (stdout, "failed on privileges\n");
      return (SS$_INVLOGIN);
   }

   rqptr->AuthUserCan = HTTP_METHOD_DELETE |
                        HTTP_METHOD_GET |
                        HTTP_METHOD_HEAD |
                        HTTP_METHOD_POST |
                        HTTP_METHOD_PUT;
   if (Debug) fprintf (stdout, "SYSUAF verified!\n");

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
SYSUAF authentication is being controlled by possession of an identifier ...
got the identifier you can be SYSUAF-authenticated, haven't then you cant! See
description at beginning of module.  Returns normal status if account possesses
suitable identifier, RESTRICTED_BY_OTHER if it doesn't, or error status.
*/ 

AuthSysUafCheckIdentifier
(
struct RequestStruct *rqptr,
unsigned long  UaiUic
)
{
   static boolean  GetIdentifiers = true;
   static unsigned long  AuthVmsGroupId,
                         AuthVmsHttpsId,
                         AuthVmsPwdId,
                         AuthVmsReadId,
                         AuthVmsReadWriteId;
   static char  AuthVmsGroup [sizeof(AUTH_VMS__GROUP_ID) +
                              MaxAuthRealmGroupNameLength];
   static $DESCRIPTOR (AuthVmsHttpsIdDsc, AUTH_VMS_HTTPS_ID);
   static $DESCRIPTOR (AuthVmsPwdIdDsc, AUTH_VMS_PWD_ID);
   static $DESCRIPTOR (AuthVmsReadIdDsc, AUTH_VMS_R_ID);
   static $DESCRIPTOR (AuthVmsReadWriteIdDsc, AUTH_VMS_RW_ID);
   static $DESCRIPTOR (AuthVmsGroupDsc, AuthVmsGroup);
   static struct {
      unsigned long  Uic;
      unsigned long  Unused;
   } Holder = { 0, 0 };

   boolean  GroupOk;
   int  status;
   unsigned long  Ctx,
                  ThisId;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthSysUafCheckIdentifier()\n");

   if (GetIdentifiers)
   {
      /* we haven't got it, get the binary value of the identifier */
      rqptr->ErrorTextPtr = AUTH_VMS_HTTPS_ID;
      status = sys$asctoid (&AuthVmsHttpsIdDsc, &AuthVmsHttpsId, 0);
      if (VMSok (status))
      {
         rqptr->ErrorTextPtr = AUTH_VMS_PWD_ID;
         status = sys$asctoid (&AuthVmsPwdIdDsc, &AuthVmsPwdId, 0);
      }
      if (VMSok (status))
      {
         rqptr->ErrorTextPtr = AUTH_VMS_RW_ID;
         status = sys$asctoid (&AuthVmsReadWriteIdDsc, &AuthVmsReadWriteId, 0);
      }
      if (VMSok (status))
      {
         rqptr->ErrorTextPtr = AUTH_VMS_R_ID;
         status = sys$asctoid (&AuthVmsReadIdDsc, &AuthVmsReadId, 0);
      }
      if (VMSnok (status)) 
      {
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }
      rqptr->ErrorTextPtr = NULL;
      GetIdentifiers = false;
      strcpy (AuthVmsGroup, AUTH_VMS__GROUP_ID);
   }

   if (rqptr->AuthGroupPtr[0])
   {
      strcpy (AuthVmsGroup+sizeof(AUTH_VMS__GROUP_ID)-1, rqptr->AuthGroupPtr);
      AuthVmsGroupDsc.dsc$w_length = strlen(AuthVmsGroup);
      if (VMSnok (status = sys$asctoid (&AuthVmsGroupDsc, &AuthVmsGroupId, 0)))
      {
         rqptr->ErrorTextPtr = AuthVmsGroup;
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }
   }

   if (Debug)
      fprintf (stdout,
         "%%X%08.08X %%X%08.08X %%X%08.08X %%X%08.08X %%X%08.08X\n",
         AuthVmsHttpsId, AuthVmsPwdId, AuthVmsReadId, AuthVmsReadWriteId,
         AuthVmsGroupId);

   /* turn on SYSPRV to allow this to be done */
   EnableSysPrv ();

   GroupOk = false;
   Holder.Uic = UaiUic;
   Ctx = 0;
   while (VMSok (status = sys$find_held (&Holder, &ThisId, 0, &Ctx)))
   {
      if (Debug) fprintf (stdout, "%08.08X\n", ThisId);

      if (ThisId == AuthVmsHttpsId)
      {
         rqptr->AuthHttpsOnly = true;
         continue;
      }
      if (ThisId == AuthVmsPwdId)
      {
         rqptr->AuthCanChangeSysUafPwd = true;
         continue;
      }
      if (ThisId == AuthVmsReadId)
      {
         rqptr->AuthUserCan = HTTP_METHOD_GET |
                              HTTP_METHOD_HEAD;
         continue;
      }
      if (ThisId == AuthVmsReadWriteId)
      {                     
         rqptr->AuthUserCan = HTTP_METHOD_DELETE |
                              HTTP_METHOD_GET |
                              HTTP_METHOD_HEAD |
                              HTTP_METHOD_POST |
                              HTTP_METHOD_PUT;
         continue;
      }
      if (ThisId == AuthVmsGroupId)
      {
         GroupOk = true;
         continue;
      }
   }

   /* turn off SYSPRV */
   DisableSysPrv ();

   if (Debug) fprintf (stdout, "sys$find_held() %%X%08.08X\n", status);

   /* reset this per-request identifier */
   AuthVmsGroupId = 0;

   if (status == SS$_NOSUCHID) 
   {
      if (Debug)
         fprintf (stdout, "%08.08Xd %d %d\n",
            rqptr->AuthUserCan, rqptr->AuthGroupPtr[0], GroupOk);
      if (!rqptr->AuthUserCan) return (RESTRICTED_BY_OTHER);
      /* if does not have identifier then not in group! */
      if (rqptr->AuthGroupPtr[0] && !GroupOk) return (RESTRICTED_BY_OTHER);
      return (SS$_NORMAL);
   }

   sys$finish_rdb (&Ctx);
   rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_DATABASE_VMS);
   ErrorVmsStatus (rqptr, status, FI_LI);
   return (status);
}

/****************************************************************************/
/*
Place the VMS-hashed password into the pointed to quadword.  This CANNOT be
used to hash the UAF password as it does not use the UAF entry salt, etc.
*/ 

AuthGenerateHashPassword
(
char *UserName,
char *Password,
unsigned long *HashedPwdPtr
)
{
   static $DESCRIPTOR (PasswordDsc, "");
   static $DESCRIPTOR (UserNameDsc, "");

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthGenerateHashPassword() |%s|%s|\n",
               UserName, Password);

   UserNameDsc.dsc$w_length = strlen(UserName);
   UserNameDsc.dsc$a_pointer = UserName;

   PasswordDsc.dsc$w_length = strlen(Password);
   PasswordDsc.dsc$a_pointer = Password;

   status = sys$hash_password (&PasswordDsc, UAI$C_PURDY_S, 0,
                               &UserNameDsc, HashedPwdPtr);
   if (Debug)
      fprintf (stdout, "sys$hash_password() %%X%08.08X|%08.08X%08.08X|\n",
               status, HashedPwdPtr[1], HashedPwdPtr[0]);
   return (status);
}

/****************************************************************************/
/*
Place the MD5 Digest upper and lower case passwords into to pointed to
two 16-byte arrays.  This is the 128 bit, binary digest, NOT the hexadecimal
version.  See note on upper and lower case versions in program comments.
Assumes password has been checked to be 15 or less characters.
*/ 

AuthGenerateDigestPassword
(
char *DatabaseName,
char *UserName,
char *Password,
unsigned char *A1DigestLoCasePtr,
unsigned char *A1DigestUpCasePtr
)
{
   register char  *cptr, *sptr;

   int  status;
   char PasswordUpCase [16],
        PasswordLoCase [16],
        UserNameUpCase [16],
        UserNameLoCase [16];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthGenerateDigestPassword() |%s|%s|%s|\n",
               DatabaseName, UserName, Password);

   sptr = PasswordUpCase;
   for (cptr = Password; *cptr; *sptr++ = toupper(*cptr++));
   *sptr = '\0';
   sptr = PasswordLoCase;
   for (cptr = Password; *cptr; *sptr++ = tolower(*cptr++));
   *sptr = '\0';
   sptr = UserNameUpCase;
   for (cptr = UserName; *cptr; *sptr++ = toupper(*cptr++));
   *sptr = '\0';
   sptr = UserNameLoCase;
   for (cptr = UserName; *cptr; *sptr++ = tolower(*cptr++));
   *sptr = '\0';
   if (VMSok (status =
       DigestA1 (UserNameUpCase, DatabaseName, PasswordUpCase,
                 A1DigestUpCasePtr)))
       status = DigestA1 (UserNameLoCase, DatabaseName, PasswordLoCase,
                          A1DigestLoCasePtr);

   return (status);
}

/*****************************************************************************/
/*
Using sys$create_user_profile() create a permanent security profile of the
SYSUAF-authenticated user name.  This will be stored in the cache and attached
to the request structure each time the user is re-authenticated from it.  This
profile will be used to check file/directory (see AuthCheckVmsUserAccess())
access permission against the authenticated user not the server account.
*/

int AuthCreateVmsUserProfile (struct RequestStruct* rqptr)

{
   static unsigned long  Flags = 0;
   static unsigned long  Context = -1;

   int  status;
   unsigned short  Length;
   unsigned char  UserProfile [2048];
   $DESCRIPTOR (UserNameDsc, rqptr->RemoteUser);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthCreateVmsUserProfile() |%s|\n", rqptr->RemoteUser);

   rqptr->AuthVmsUserProfilePtr = NULL;
   rqptr->AuthVmsUserProfileLength = 0;

   /* turn on SYSPRV to allow this to be done */
   EnableSysPrv ();

   UserNameDsc.dsc$w_length = strlen(rqptr->RemoteUser);
   Length = sizeof(UserProfile);
   status = sys$create_user_profile (&UserNameDsc, 0, Flags,
                                     UserProfile, &Length, &Context);
   if (Debug)
      fprintf (stdout, "sys$create_user_profile() %%X%08.08X\n", status);

   /* turn off SYSPRV */
   DisableSysPrv ();

   if (VMSnok (status)) 
   {
      rqptr->ErrorTextPtr = "sys$create_user_profile()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   if (Debug) fprintf (stdout, "Length %d\n", Length);

   rqptr->AuthVmsUserProfilePtr =
      VmGet (rqptr->AuthVmsUserProfileLength = Length);
   if (Debug) fprintf (stdout, "ptr: %d\n", rqptr->AuthVmsUserProfilePtr);

   memcpy (rqptr->AuthVmsUserProfilePtr, UserProfile, Length);
      
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Using sys$create_user_profile() create a permanent security profile of the
HTTPd server account.  Returns no status, exits server if error encountered.
*/ 
 
AuthCreateHttpdProfile ()

{
   static unsigned long  Flags = 0;

   int  status;
   char  HttpdUserProfile [512];
   $DESCRIPTOR (HttpdUserNameDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCreateHttpdProfile()\n");

   if (HttpdUserProfileLength) return;

   HttpdUserNameDsc.dsc$a_pointer = HttpdUserName;
   HttpdUserNameDsc.dsc$w_length = strlen(HttpdUserName);

   status = sys$create_user_profile (&HttpdUserNameDsc, 0, Flags,
                                     HttpdUserProfile,
                                     &HttpdUserProfileLength, 0);
   if (Debug)
      fprintf (stdout, "sys$create_user_profile() %%X%08.08X\n", status);

   if (VMSnok (status))
      ErrorExitVmsStatus (status, "sys$create_user_profile()", FI_LI);

   HttpdUserProfilePtr = VmGet (HttpdUserProfileLength);
   if (Debug) fprintf (stdout, "ptr: %d\n", HttpdUserProfilePtr);

   memcpy (HttpdUserProfilePtr, HttpdUserProfile, HttpdUserProfileLength);
}

/*****************************************************************************/
/*
Using sys$check_access() check file read access permission for the supplied
file name against the SYSUAF-authenticated user's security profile (created by
AuthCreateVmsUserProfile()).  File or directory specifications can be supplied.
Returns SS$_NORMAL if access allowed, RMS$_PRV if denied, and other RMS status
codes. A VMS error other than from RMS (e.g. INSFMEM, which would disrupt the
system service results) will cause the server to exit.
*/

int AuthCheckVmsUserAccess
(
struct RequestStruct* rqptr,
char *FileName,
int FileNameLength
)
{
   static unsigned long  Flags = 0,
                         ArmReadAccess = ARM$M_READ;
   static $DESCRIPTOR (ClassNameDsc, "FILE");
   static $DESCRIPTOR (UserProfileDsc, "");
   static struct
   {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   }
   ReadItem [] = 
   {
      { sizeof(ArmReadAccess), CHP$_ACCESS, &ArmReadAccess, 0 },
      { 0, 0, 0, 0 }
   };

   int  status,
        DirFileNameLength;
   char  DirFileName [256];
   $DESCRIPTOR (FileNameDsc, FileName);

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      if (rqptr == NULL)
         fprintf (stdout, "AuthCheckVmsUserAccess() |%s|\n", HttpdUserName);
      else
         fprintf (stdout, "AuthCheckVmsUserAccess() |%s|%s| %d %d\n",
                  rqptr->RemoteUser, FileName,
                  rqptr->AuthVmsUserProfileLength,
                  rqptr->AuthVmsUserProfilePtr);
   }

   if (FileNameLength <= 0) FileNameLength = strlen(FileName);

   if (FileNameLength && FileName[FileNameLength-1] == ']')
   {
      /* a directory has been specified, get the directory file name */
      NameOfDirectoryFile (FileName, FileNameLength,
                           DirFileName, &DirFileNameLength);
      FileNameDsc.dsc$a_pointer = DirFileName;
      FileNameDsc.dsc$w_length = DirFileNameLength;
   }
   else
   {
      FileNameDsc.dsc$a_pointer = FileName;
      FileNameDsc.dsc$w_length = FileNameLength;
   }

   /****************/
   /* check access */
   /****************/

   if (rqptr == NULL)
   {
      /* against the HTTPd server account */
      if (!HttpdUserProfileLength) AuthCreateHttpdProfile ();
      UserProfileDsc.dsc$a_pointer = HttpdUserProfilePtr;
      UserProfileDsc.dsc$w_length = HttpdUserProfileLength;
   }
   else
   {
      /* against a SYSUAF-authenticated account */
      UserProfileDsc.dsc$a_pointer = rqptr->AuthVmsUserProfilePtr;
      UserProfileDsc.dsc$w_length = rqptr->AuthVmsUserProfileLength;
   }

   /* turn on SYSPRV to allow this to be done */
   EnableSysPrv ();

   status = sys$check_access (0, &FileNameDsc, 0, &ArmReadAccess,
                              0, &ClassNameDsc, 0, &UserProfileDsc);
   if (Debug) fprintf (stdout, "sys$check_access() %%X%08.08X\n", status);

   /* turn off SYSPRV */
   DisableSysPrv ();

   /* we are only dealing with files ... turn it into an RMS message */
   if (status == SS$_NOPRIV) return (RMS$_PRV);

   /* if access is permitted */
   if (VMSok (status)) return (status);

   /* if an RMS error return normal letting caller continue and report it */
   if (((status & STS$M_FAC_NO) >> STS$V_FAC_NO) == RMS$_FACILITY)
      return (SS$_NORMAL);

   /* cannot generate error message, do it at the calling routine */
   if (rqptr == NULL) return (status);

   rqptr->ErrorTextPtr = "sys$check_access()";
   rqptr->ErrorHiddenTextPtr = FileName;
   ErrorVmsStatus (rqptr, status, FI_LI);
   return (status);
}

/*****************************************************************************/
/*
Check that the HTTPd account has CONTROL or WRITE access to the parent
directory of the file or directory specification supplied in 'FileName'. If
the server has CONTROL access then there is no general access to the
directory, only an authenticated VMS user can write into it. If there is WRITE
access then any authenticated user can write into it. Returns SS$_NORMAL if
access allowed, RMS$_PRV if denied, and other status codes.  Error report
should be generated by the calling routine.
*/ 
 
int AuthCheckWriteAccess
(
struct RequestStruct *rqptr,
char *FileName,
int FileNameLength
)
{
   static unsigned long  Flags = 0;
   static unsigned long  ArmWriteAccess = ARM$M_WRITE,
                         ArmControlAccess = ARM$M_CONTROL;

   static unsigned short  HttpdUserProfileLength = 0;
   static char  HttpdUserProfile [512];

   static struct {
      unsigned short  BufferLength;
      unsigned short  ItemCode;
      unsigned long  BufferAddress;
      unsigned long  ReturnLengthAddress;
   } WriteAccessItem [] =
       { { sizeof(ArmWriteAccess), CHP$_ACCESS, &ArmWriteAccess, 0 },
         { 0, 0, 0, 0 } },
     ControlAccessItem [] =
       { { sizeof(ArmControlAccess), CHP$_ACCESS, &ArmControlAccess, 0 },
         { 0, 0, 0, 0 } };

   static $DESCRIPTOR (ClassNameDsc, "FILE");
   static $DESCRIPTOR (HttpdUserProfileDsc, HttpdUserProfile);
   static $DESCRIPTOR (ObjectNameDsc, "");
   static $DESCRIPTOR (UserProfileDsc, "");

   int  status,
        DirectoryFileLength;
   char  DirectoryFile [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCheckWriteAccess() |%s|\n", FileName);

   if (!HttpdUserProfileLength)
   {
      /*********************************************************/
      /* create a permanent user profile for the HTTPd account */
      /*********************************************************/

      $DESCRIPTOR (HttpdUserNameDsc, "");

      HttpdUserNameDsc.dsc$a_pointer = HttpdUserName;
      HttpdUserNameDsc.dsc$w_length = strlen(HttpdUserName);

      status = sys$create_user_profile (&HttpdUserNameDsc, 0, Flags,
                                        HttpdUserProfile,
                                        &HttpdUserProfileLength, 0);
      if (Debug)
         fprintf (stdout, "sys$create_user_profile() %%X%08.08X\n", status);
      if (VMSnok (status)) return (status);

      HttpdUserProfileDsc.dsc$w_length = HttpdUserProfileLength;
   }

   if (FileNameLength <= 0) FileNameLength = strlen(FileName);

   NameOfDirectoryFile (FileName, FileNameLength,
                        DirectoryFile, &DirectoryFileLength);

   ObjectNameDsc.dsc$a_pointer = DirectoryFile;
   ObjectNameDsc.dsc$w_length = DirectoryFileLength;

   /* turn on SYSPRV to allow this to be done */
   EnableSysPrv ();

   if (AuthVmsUserSecProfileEnabled)
   {
      /****************************/
      /* check for control access */
      /****************************/

      status = sys$check_access (0, &ObjectNameDsc, 0, &ControlAccessItem,
                                 0, &ClassNameDsc, 0, &HttpdUserProfileDsc);
      if (Debug) fprintf (stdout, "1 sys$check_access() %%X%08.08X\n", status);

      if (VMSok (status))
      {
         /* control access for HTTPd, must be authenticated VMS user! */
         if (!rqptr->AuthVmsUserProfileLength)
         {
            /* turn off SYSPRV! */
            DisableSysPrv ();
            /* not a SYSUAF-authenticated user therefore no privilege */
            return (SS$_NOPRIV);
         }

         /* now check if the authenticated VMS user has write access */
         UserProfileDsc.dsc$a_pointer = rqptr->AuthVmsUserProfilePtr;
         UserProfileDsc.dsc$w_length = rqptr->AuthVmsUserProfileLength;

         status = sys$check_access (0, &ObjectNameDsc, 0, &WriteAccessItem,
                                    0, &ClassNameDsc, 0, &UserProfileDsc);
         if (Debug)
            fprintf (stdout, "2 sys$check_access() %%X%08.08X\n", status);

         /* turn off SYSPRV */
         DisableSysPrv ();

         /* no privilege to write */
         if (status == SS$_NOPRIV) return (status);

         /* write access OK */
         if (VMSok (status)) return (status);

         /* if the directory file not exist then directory does not exist */
         if (status == RMS$_FNF) status = RMS$_DNF;

         return (status);
      }

      /* no control access, drop through to check for write access */
   }

   /**************************/
   /* check for write access */
   /**************************/

   status = sys$check_access (0, &ObjectNameDsc, 0, &WriteAccessItem,
                              0, &ClassNameDsc, 0, &HttpdUserProfileDsc);
   if (Debug) fprintf (stdout, "3 sys$check_access() %%X%08.08X\n", status);

   /* turn off SYSPRV */
   DisableSysPrv ();

   /* no privilege to write */
   if (status == SS$_NOPRIV) return (status);

   /* write access OK */
   if (VMSok (status)) return (status);

   /* if the directory file did not exist then directory does not exist */
   if (status == RMS$_FNF) status = RMS$_DNF;

   return (status);
}

/*****************************************************************************/
/*
Open the authorization configuration file defined by the logical name
HTTPD$AUTH.  Read the directives applying them to the authorization path tree. 
Close the file.  Can also be called from the HTTPd.c or Control.c modules.
*/

int AuthPathConfig (struct AuthLoadStruct *alptr)

{
   boolean  DebugBuffer;
   int  status;
   char  ExpandedFileName [256],
         Line [1024];
   struct FAB  AuthFileFab;
   struct NAM  AuthFileNam;
   struct RAB  AuthFileRab;
   struct XABDAT  AuthFileXabDat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathConfig()\n");

#ifdef DBUG
   /* not interested anymore in seeing debug information for path load! */
   DebugBuffer = Debug;
   Debug = false;
Debug = DebugBuffer;
#endif

   if (alptr == NULL) alptr = &ServerAuthLoad;

   if (AuthPromiscuous)
   {
      sprintf (Line, "%s authenticating any username/password!",
               AUTH_PROMISCUOUS);
      AuthPathConfigReportProblem (alptr, Line, 0);
   }

   if (AuthSysUafEnabled)
      AuthPathConfigReportProblem (alptr, "SYSUAF authentication enabled", 0);

   if (AuthVmsUserSecProfileEnabled)
      AuthPathConfigReportProblem (alptr, "SYSUAF security profile enabled", 0);

   if (AuthSysUafPromiscuous)
   {
      sprintf (Line, "SYSUAF %s (testing only)", AUTH_PROMISCUOUS);
      AuthPathConfigReportProblem (alptr, Line, 0);
   }

   /* if the server database being (re)loaded purge the authorization cache */
   if (alptr == &ServerAuthLoad) AuthCacheTreeFree ();

   AuthPathListFree (alptr);

   if (alptr->ProblemReportPtr != NULL) free (alptr->ProblemReportPtr);

   memset (alptr, 0, sizeof(struct AuthLoadStruct));

   sys$gettim (&alptr->LoadBinTime);

   /* initialize */
   AuthPathLine (alptr, NULL);
   alptr->LineNumber = alptr->ProblemCount = 0;

   /************************/
   /* open the config file */
   /************************/

   AuthFileFab = cc$rms_fab;
   AuthFileFab.fab$b_fac = FAB$M_GET;
   AuthFileFab.fab$l_fna = AuthConfigFileName;
   AuthFileFab.fab$b_fns = strlen(AuthConfigFileName);
   AuthFileFab.fab$l_nam = &AuthFileNam;
   AuthFileFab.fab$b_shr = FAB$M_SHRGET;
   AuthFileFab.fab$l_xab = &AuthFileXabDat;

   AuthFileNam = cc$rms_nam;
   AuthFileNam.nam$l_esa = ExpandedFileName;
   AuthFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   AuthFileNam.nam$l_rsa = alptr->LoadFileName;
   AuthFileNam.nam$b_rss = sizeof(alptr->LoadFileName)-1;

   /* initialize the date extended attribute block */
   AuthFileXabDat = cc$rms_xabdat;

   /* turn on SYSPRV to allow access to possibly protected file */
   EnableSysPrv ();
   status = sys$open (&AuthFileFab, 0, 0);
   DisableSysPrv ();

   /* status from sys$open() */
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      AuthPathConfigReportProblem (alptr, NULL, status);
      /* loading at startup, report any errors */
      if (alptr->ProblemCount && !alptr->RequestPtr)
         fputs (alptr->ProblemReportPtr, stdout);
#ifdef DBUG
      Debug = DebugBuffer;
#endif
      return (status);
   }

   alptr->LoadFileName[AuthFileNam.nam$b_rsl] = '\0';
   if (Debug)
      fprintf (stdout, "LoadFileName |%s|\n", alptr->LoadFileName);

   memcpy (alptr->RevBinTime, &AuthFileXabDat.xab$q_rdt, 8);

   /* record access block */
   AuthFileRab = cc$rms_rab;
   AuthFileRab.rab$l_fab = &AuthFileFab;
   /* 2 buffers */
   AuthFileRab.rab$b_mbf = 2;
   /* read ahead performance option */
   AuthFileRab.rab$l_rop = RAB$M_RAH;

   if (VMSnok (status = sys$connect (&AuthFileRab, 0, 0)))
   {
      sys$close (&AuthFileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      AuthPathConfigReportProblem (alptr, NULL, status);
      /* loading at startup, report any errors */
      if (alptr->ProblemCount && !alptr->RequestPtr)
         fputs (alptr->ProblemReportPtr, stdout);
#ifdef DBUG
      Debug = DebugBuffer;
#endif
      return (status);
   }

   /* guarantee access to the administration menu */
   if (AuthPromiscuous)
   {
      /* these may generate duplicate rule errors at startup! */
      AuthPathLine (alptr, "[PROMISCUOUS]");
      /* access to the server administration menu */
      AuthPathLine (alptr, "/httpd/-/admin/* r+w");
      /* access to write into the "normal" location for configuration files */
      AuthPathLine (alptr, "/ht_root/local/* r+w");
   }

   /************************/
   /* read the config file */
   /************************/

   AuthFileRab.rab$l_ubf = Line;
   AuthFileRab.rab$w_usz = sizeof(Line)-1;

   while (VMSok (status = sys$get (&AuthFileRab, 0, 0)))
   {
      alptr->LineNumber++;
      AuthFileRab.rab$l_ubf[AuthFileRab.rab$w_rsz] = '\0';
      if (Debug)
         fprintf (stdout, "line %d |%s|\n",
                  alptr->LineNumber, AuthFileRab.rab$l_ubf);

      if (AuthFileRab.rab$w_rsz)
      {
         if (AuthFileRab.rab$l_ubf[AuthFileRab.rab$w_rsz-1] == '\\')
         {
            /* directive is continued on next line */
            AuthFileRab.rab$l_ubf[AuthFileRab.rab$w_rsz-1] = ' ';
            AuthFileRab.rab$l_ubf += AuthFileRab.rab$w_rsz;
            AuthFileRab.rab$w_usz -= AuthFileRab.rab$w_rsz;
            continue;
         }
      }

      AuthPathLine (alptr, Line);

      AuthFileRab.rab$l_ubf = Line;
      AuthFileRab.rab$w_usz = sizeof(Line)-1;
   }

   if (status == RMS$_EOF) status = SS$_NORMAL;

   /******************/
   /* close the file */
   /******************/

   sys$close (&AuthFileFab, 0, 0); 

   if (Debug)
   {
      register struct AuthPathRecordStruct  *plptr;

      fprintf (stdout, "-\n");
      for (plptr = alptr->PathListHead;
           plptr != NULL;
           plptr = plptr->NextPtr)
         fprintf (stdout, "plptr |%s|%s|%s|\n",
                  plptr->PathPtr, plptr->RealmPtr, plptr->GroupPtr);
      fprintf (stdout, "-\n");
   }

   /* let this function do its initialization */
   AuthPathCheck (NULL, NULL);

   if (AuthorizationEnabled &&
       !(Config.AuthBasicEnabled || Config.AuthDigestEnabled))
   {
      AuthPathConfigReportProblem (alptr,
         "no authentication scheme configured", SS$_NORMAL);
   }

#ifdef DBUG
   Debug = DebugBuffer;
#endif
   return (status);
}

/*****************************************************************************/
/*
Process a line of authorization configuration.
*/

#define ACCESS_RESTRICTION_LIST_SIZE 256

int AuthPathLine
(
struct AuthLoadStruct *alptr,
char *Line
)
{
   static char  Group [16],
                Realm [16];

   register char  *cptr, *lptr, *sptr, *zptr;

   int  status;
   unsigned long  *CanFlagsPtr;
   unsigned long  GroupCanFlags,
                  WorldCanFlags;
   char  *RestrictListPtr;
   char  GroupRestrictList [ACCESS_RESTRICTION_LIST_SIZE],
         RealmCanString [256],
         WorldRestrictList [ACCESS_RESTRICTION_LIST_SIZE];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthPathLine() |%s|%s|%s|\n", Realm, Group, Line);

   if (Line == NULL)
   {
      /* initialize */
      Realm[0] = Group[0] = '\0';
      return;
   }

   lptr = Line;
   while (*lptr && ISLWS(*lptr)) lptr++;
   if (!*lptr || *lptr == '#') return;

   GroupCanFlags = WorldCanFlags = 0;
   GroupRestrictList[0] = WorldRestrictList[0] = '\0';
   CanFlagsPtr = &GroupCanFlags;
   RestrictListPtr = &GroupRestrictList;

   if (*lptr == '/')
   {
      /********/
      /* path */
      /********/

      /* unless in promiscuous mode completely ignore PROMISCUOUS realm! */
      if (!AuthPromiscuous && strsame (Realm, "PROMISCUOUS", -1))
         return;

      /* ignore any path where there is no realm specified */
      if (!Realm[0])
      {
         AuthPathConfigReportProblem (alptr,
            "no realm context (path ignored)", 0);
         return;
      }

      /* note start of path template */
      cptr = lptr;
      while (*lptr && !ISLWS(*lptr)) lptr++;
      /* terminate at the end of the path template */
      if (*lptr) *lptr++ = '\0';
      /* find start of access flags */
      while (*lptr && ISLWS(*lptr)) lptr++;
      if (!*lptr)
      {
         if (RealmCanString[0])
            lptr = RealmCanString;
         else
         {
            AuthPathConfigReportProblem (alptr,
               "no access specified (no realm defaults to apply)", 0);
            lptr = "";
         }
      }

      while (*lptr)
      {
         /********************************/
         /* set flags controlling access */
         /********************************/

         /* find the start of the next element */
         while (*lptr && (ISLWS(*lptr) || *lptr == ',' || *lptr == ';'))
         {
            if (*lptr++ != ';') continue;
            /* semicolon separates realm/group and optional world access */
            CanFlagsPtr = &WorldCanFlags;
            RestrictListPtr = &WorldRestrictList;
         }
         if (!*lptr) break;

         if (*lptr == '*' ||
             *lptr == '#' ||
             isdigit(*lptr) ||
             *lptr == '~' ||
             strsame (lptr, "http:", 5) ||
             strsame (lptr, "https:", 6) ||
             strsame (lptr, "localhost", 9))
         {
            /* access restriction list */
            for (sptr = RestrictListPtr; *sptr; sptr++);
            zptr = sptr + ACCESS_RESTRICTION_LIST_SIZE;
            if (sptr > RestrictListPtr && sptr < zptr) *sptr++ = ',';
            while (*lptr && !ISLWS(*lptr) &&
                   *lptr != ',' && *lptr != ';' &&
                   sptr < zptr)
               *sptr++ = *lptr++;
            if (sptr >= zptr)
            {
               AuthPathConfigReportProblem (alptr,
                  "restriction list too long (ignored)", 0);
               *RestrictListPtr = '\0';
            }
            else
               *sptr = '\0';
         }
         else
         if (strsame (lptr, "none", 4) && !isalpha(lptr[4]))
            *CanFlagsPtr = 0;
         else
         /* must come before "read" for obvious reasons! */
         if ((strsame (lptr, "READ+WRITE", 10) && !isalpha(lptr[10])) ||
             (strsame (lptr, "R+W", 3) && !isalpha(lptr[3])))
            *CanFlagsPtr |= (HTTP_METHOD_GET | HTTP_METHOD_HEAD |
                             HTTP_METHOD_DELETE | HTTP_METHOD_POST |
                             HTTP_METHOD_PUT);
         else
         if ((strsame (lptr, "READ", 4) && !isalpha(lptr[4])) ||
             (strsame (lptr, "R", 1) && !isalpha(lptr[1])))
            *CanFlagsPtr |= (HTTP_METHOD_GET | HTTP_METHOD_HEAD);
         else
         if ((strsame (lptr, "WRITE", 5) && !isalpha(lptr[5])) ||
             (strsame (lptr, "W", 1) && !isalpha(lptr[1])))
            *CanFlagsPtr |= (HTTP_METHOD_DELETE | HTTP_METHOD_POST |
                             HTTP_METHOD_PUT);
         else
         if (strsame (lptr, "DELETE", 6) && !isalpha(lptr[6]))
            *CanFlagsPtr |= HTTP_METHOD_DELETE;
         else
         if (strsame (lptr, "GET", 3) && !isalpha(lptr[3]))
            *CanFlagsPtr |= HTTP_METHOD_GET | HTTP_METHOD_HEAD;
         else
         if (strsame (lptr, "HEAD", 4) && !isalpha(lptr[4]))
            *CanFlagsPtr |= HTTP_METHOD_HEAD;
         else
         if (strsame (lptr, "POST", 4) && !isalpha(lptr[4]))
            *CanFlagsPtr |= HTTP_METHOD_POST;
         else
         if (strsame (lptr, "PUT", 3) && !isalpha(lptr[3]))
            *CanFlagsPtr |= HTTP_METHOD_PUT;
         else
         {
            /* assume it's an alpha-numeric host name */
            for (sptr = RestrictListPtr; *sptr; sptr++);
            zptr = sptr + ACCESS_RESTRICTION_LIST_SIZE;
            if (sptr > RestrictListPtr && sptr < zptr) *sptr++ = ',';
            while (*lptr && !ISLWS(*lptr) &&
                   *lptr != ',' && *lptr != ';' &&
                   sptr < zptr)
               *sptr++ = *lptr++;
            if (sptr >= zptr)
            {
               AuthPathConfigReportProblem (alptr,
                  "restriction list too long (ignored)", 0);
               *RestrictListPtr = '\0';
            }
            else
               *sptr = '\0';
         }
         while (*lptr && !ISLWS(*lptr) && *lptr != ',' && *lptr != ';')
            lptr++;
      }

      if (VMSnok (status =
          AuthPathAdd (alptr,
                       Realm, Group, cptr,
                       GroupRestrictList,
                       WorldRestrictList,
                       GroupCanFlags, WorldCanFlags)))
      if (status == RMS$_DUP)
         AuthPathConfigReportProblem (alptr,
               "duplicate path (ignored)", 0);
      else
         ErrorExitVmsStatus (status, "AuthPathAdd()", FI_LI);

      /* increment this boolean if authorization paths are loaded */
      AuthorizationEnabled++;

      return;
   }

   if (*lptr == '[' || strsame (lptr, "REALM", 5))
   {
      /*********/
      /* realm */
      /*********/

      if (*lptr == '[')
      {
         lptr++;
         while (*lptr && ISLWS(*lptr)) lptr++;
      }
      else
      {
         /* skip over keyword and find start of realm name */
         while (*lptr && !ISLWS(*lptr)) lptr++;
         while (*lptr && ISLWS(*lptr)) lptr++;
      }

      zptr = (sptr = Realm) + sizeof(Realm);
      while (*lptr &&
             *lptr != ';' &&
             *lptr != ']' &&
             !ISLWS(*lptr) &&
             sptr < zptr)
         *sptr++ = *lptr++;
      if (sptr >= zptr)
      {
         Realm[0] = '\0';
         AuthPathConfigReportProblem (alptr, "confused", 0);
      }
      else
      {
         *sptr = '\0';
         if (!Realm[0])
            AuthPathConfigReportProblem (alptr,
               "realm not specified (ignoring all context paths)", 0);
      }
      while (*lptr && ISLWS(*lptr)) lptr++;

      /* semicolon separating realm from optional group */
      Group[0] = '\0';
      if (*lptr == ';')
      {
         lptr++;
         while (*lptr && ISLWS(*lptr)) lptr++;
         zptr = (sptr = Group) + sizeof(Group);
         while (*lptr &&
                *lptr != ';' &&
                *lptr != ']' &&
                !ISLWS(*lptr) &&
                sptr < zptr)
            *sptr++ = *lptr++;
         if (sptr < zptr)
            *sptr = '\0';
         else
            Realm[0] = '\0';

         if (!Group[0])
            AuthPathConfigReportProblem (alptr,
               "group not specified (using only realm)", 0);
      }
      while (*lptr && ISLWS(*lptr)) lptr++;
      if (*lptr == ']') lptr++;

      /* find start of realm-context access flags */
      while (*lptr && ISLWS(*lptr)) lptr++;
      if (*lptr)
         strcpy (RealmCanString, lptr);
      else
         RealmCanString[0] = '\0';

      return;
   }

   AuthPathConfigReportProblem (alptr, "confused", 0);
}

/*****************************************************************************/
/*
This function formats an error report.  All lines are concatenated onto a
single string of dynamically allocated memory that (obviously) grows as
reports are added to it.  This string is then output if loading the server
configuration or is available for inclusion in an HTML page.
*/

int AuthPathConfigReportProblem
(
struct AuthLoadStruct *alptr,
char *Explanation,
int StatusValue
)
{
   static $DESCRIPTOR (ExplanationFaoDsc, "%HTTPD-W-AUTH, !AZ\n");
   static $DESCRIPTOR (ExplanationLineFaoDsc,
                       "%HTTPD-W-AUTH, !AZ at line !UL\n");

   int  status;
   unsigned short  Length;
   char  Buffer [256],
         HtmlBuffer [512];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathConfigReportProblem()\n");

   alptr->ProblemCount++;

   if (StatusValue)
   {
      if (VMSok (status = sys$getmsg (StatusValue, &Length, &BufferDsc, 0, 0)))
         Buffer[Length++] = '\n';
   }
   else
   if (alptr->LineNumber)
      status = sys$fao (&ExplanationLineFaoDsc, &Length, &BufferDsc,
                        Explanation, alptr->LineNumber);
   else
      status = sys$fao (&ExplanationFaoDsc, &Length, &BufferDsc,
                        Explanation);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
      ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
   else
      Buffer[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", Buffer);

   if (alptr->RequestPtr == NULL) fputs (Buffer, stdout);

   if ((Length = CopyToHtml (HtmlBuffer, sizeof(HtmlBuffer), Buffer, -1)) < 0)
      ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);

   alptr->ProblemReportPtr =
      VmRealloc (alptr->ProblemReportPtr, alptr->ProblemReportLength+Length+1);

   /* include the terminating null */
   memcpy (alptr->ProblemReportPtr+alptr->ProblemReportLength,
           HtmlBuffer, Length+1);
   alptr->ProblemReportLength += Length;
}

/*****************************************************************************/
/*
Scan through the path list until the first record GREATER than the new path is
encountered.  Insert the new path immediately BEFORE that record.  The data
structure has strings stored immediately following the formally declared
structure and pointed to by character pointers within the formal structure.
*/ 

int AuthPathAdd
(
struct AuthLoadStruct *alptr,
char *Realm,
char *Group,
char *Path,
char *GroupRestrictList,
char *WorldRestrictList,
unsigned long  GroupCanFlags,
unsigned long  WorldCanFlags
)
{
   register char  *cptr, *pptr, *sptr, *tptr;
   register struct AuthPathRecordStruct  *plptr, *nlptr;

   int  GroupLength,
        GroupRestrictListLength,
        PathLength,
        RealmLength,
        StringSpace,
        WorldRestrictListLength;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
         "AuthPathAdd() |%s|%s|%s|%s|%s|%08.08X|%08.08X|\n",
         Realm, Group, Path, GroupRestrictList,
         WorldRestrictList, GroupCanFlags, WorldCanFlags);

   if (alptr == NULL) alptr = &ServerAuthLoad;

   GroupLength = strlen(Group);
   PathLength = strlen(Path);
   RealmLength = strlen(Realm);
   GroupRestrictListLength = strlen(GroupRestrictList);
   WorldRestrictListLength = strlen(WorldRestrictList);
   cptr = Path;

   for (plptr = alptr->PathListHead; plptr != NULL; plptr = plptr->NextPtr)
   {
/*
      if (Debug)
         fprintf (stdout, "plptr |%s|%s|%08.08X|\n",
                  plptr->PathPtr, plptr->RealmPtr, plptr->AuthGroupCan);
*/

      pptr = cptr;
      tptr = plptr->PathPtr;
      while (*tptr && *pptr && tolower(*tptr) == tolower(*pptr))
      {
         tptr++;
         pptr++;
      }

      /* duplicate record */
      if (!*tptr && !*pptr) return (RMS$_DUP);

      /* got to end of template and new path longer (sub-path), goes first */
      if (*tptr == '*' && PathLength > plptr->PathLength) break;

      /* got to end of new path, must be a "parent"-path */
      if (*pptr == '*') continue;

      /* if not at end of template and new path is less then add now */
      if (*tptr && (tolower(*tptr) >= tolower(*pptr))) break;
   }

   StringSpace = GroupLength + 1 +
                 RealmLength + 1 +
                 PathLength + 1 +
                 GroupRestrictListLength + 1 +
                 WorldRestrictListLength + 1;

   nlptr = VmGet (sizeof(struct AuthPathRecordStruct) + StringSpace);
   StringSpace = sizeof(struct AuthPathRecordStruct);

   cptr = Path;
   nlptr->PathPtr = sptr = (char*)nlptr + StringSpace;
   while (*cptr) *sptr++ = tolower(*cptr++);      
   *sptr = '\0';
   StringSpace += (nlptr->PathLength = PathLength) + 1;

   cptr = Realm;
   nlptr->RealmPtr = sptr = (char*)nlptr + StringSpace;
   while (*cptr) *sptr++ = toupper(*cptr++);
   *sptr = '\0';
   StringSpace += (nlptr->RealmLength = RealmLength) + 1;

   cptr = Group;
   nlptr->GroupPtr = sptr = (char*)nlptr + StringSpace;
   while (*cptr) *sptr++ = toupper(*cptr++);
   *sptr = '\0';
   StringSpace += (nlptr->GroupLength = GroupLength) + 1;

   cptr = GroupRestrictList;
   nlptr->GroupRestrictListPtr = sptr = (char*)nlptr + StringSpace;
   while (*cptr) *sptr++ = tolower(*cptr++);      
   *sptr = '\0';
   StringSpace += (nlptr->GroupRestrictListLength =
                  GroupRestrictListLength) + 1;

   cptr = WorldRestrictList;
   nlptr->WorldRestrictListPtr = sptr = (char*)nlptr + StringSpace;
   while (*cptr) *sptr++ = tolower(*cptr++);      
   *sptr = '\0';

   nlptr->AuthGroupCan = GroupCanFlags;
   nlptr->AuthWorldCan = WorldCanFlags;

   if (plptr == NULL)
   {
      /* no entry greater than the new one */
      if (alptr->PathListHead == NULL)
      {
         /* empty list */
         alptr->PathListHead = alptr->PathListTail = nlptr;
         nlptr->PrevPtr = nlptr->NextPtr = NULL;
      }
      else
      {
         /* add to end of list */
         nlptr->PrevPtr = alptr->PathListTail;
         alptr->PathListTail->NextPtr = nlptr;
         nlptr->NextPtr = NULL;
         alptr->PathListTail = nlptr;
      }
   }
   else
   {
      /* insert before the entry found */
      nlptr->PrevPtr = plptr->PrevPtr;
      if (nlptr->PrevPtr != NULL) nlptr->PrevPtr->NextPtr = nlptr;
      nlptr->NextPtr = plptr;
      plptr->PrevPtr = nlptr;
      if (alptr->PathListHead == plptr)
          alptr->PathListHead = nlptr;
   }

   if (Debug)
      fprintf (stdout, "nlptr |%s|%s| GroupCan: %08.08X WorldCan: %08.08X\n",
               nlptr->PathPtr, nlptr->RealmPtr,
               nlptr->AuthGroupCan, nlptr->AuthWorldCan);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Remove the entry for the specified path from the path authorization list.
Return true if found and removed, false if not.
*/

boolean AuthPathRemove
(
struct AuthLoadStruct *alptr,
char *Path
)
{
   register int  EntryCount;
   register char  *cptr, *pptr, *tptr;
   register struct AuthPathRecordStruct  *plptr, *nlptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathRemove() |%s|\n", Path);

   if (alptr == NULL) alptr = &ServerAuthLoad;

   if (alptr->PathListHead == NULL) return (false);

   for (plptr = alptr->PathListHead; plptr != NULL; plptr = plptr->NextPtr)
   {
      if (Debug) fprintf (stdout, "plptr |%s|\n", plptr->PathPtr);

      pptr = cptr;
      tptr = plptr->PathPtr;
      while (*tptr && *pptr && tolower(*tptr) == tolower(*pptr))
      {
         tptr++;
         pptr++;
      }

      /* try next entry in list if not the same */
      if (*tptr && *pptr) continue;

      /* found entry for path */
      if (alptr->PathListHead == plptr && alptr->PathListTail == plptr)
      {
         /* only entry in list */
         alptr->PathListHead = alptr->PathListTail = NULL;
      }
      else
      if (alptr->PathListHead == plptr)
      {
         /* first entry in list */
         alptr->PathListHead = plptr->NextPtr;
         plptr->NextPtr->PrevPtr = NULL;
      }
      else
      if (alptr->PathListTail == plptr)
      {
         /* last entry in list */
         alptr->PathListTail = plptr->PrevPtr;
         plptr->PrevPtr->NextPtr = NULL;
      }
      else
      {
         /* somewhere in the middle! */
         plptr->NextPtr->PrevPtr = plptr->PrevPtr;
         plptr->PrevPtr->NextPtr = plptr->NextPtr;
      }

      VmFree (plptr);
      return (true);
   }

   return (false);
}

/*****************************************************************************/
/*
Frees all dynamic structures (memory) in the authorization path list.
*/

int AuthPathListFree (struct AuthLoadStruct *alptr)

{
   register int  EntryCount;
   register struct AuthPathRecordStruct  *plptr, *nlptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathListFree()\n");

   if (alptr == NULL) alptr = &ServerAuthLoad;

   if (alptr->PathListHead == NULL) return;

   for (plptr = alptr->PathListHead; plptr != NULL; plptr = nlptr)
   {
      if (Debug)
         fprintf (stdout, "plptr |%s|%s|%s|\n",
                  plptr->PathPtr, plptr->RealmPtr, plptr->GroupPtr);

      nlptr = plptr->NextPtr;
      VmFree (plptr);
   }

   alptr->PathListHead = alptr->PathListTail = NULL;
}

/*****************************************************************************/
/*
Look for a path template matching the request path in the path authorization
database.  If one can't be found return with '->AuthRealmPtr', 
->AuthGroupPtr', 'rqptr->AuthRestrictListPtr' and
'rqptr->AuthWorldRestrictListPtr' pointing to empty strings and the
path capabilities '->AuthGroupCan' and '->AuthWorldCan' set to zero.  If found
return with the realm name and path capabilities set to what's in the database. 
The 'QuickIndex' attempts to reduce search times in larger authorization
databases by allowing the search to begin in the vicinity of a likely hit. 
This function must be called with a parameter of NULL to initialize the quick
index after the paths are loaded.
*/ 

int AuthPathCheck
(
struct RequestStruct *rqptr,
struct AuthLoadStruct *alptr
)
{
   static struct AuthPathRecordStruct  *QuickIndex [26];

   register char  c;
   register char  *cptr, *pptr, *tptr;
   register struct AuthPathRecordStruct  *plptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathCheck()\n");

   if (alptr == NULL) alptr = &ServerAuthLoad;

   if (rqptr == NULL)
   {
      /***************************/
      /* function initialization */
      /***************************/

      /* note the first instance of a path beginning with 'A' thru to 'Z' */
      for (c = 0; c < 26; c++) QuickIndex[c] = NULL;
      for (plptr = alptr->PathListHead;
           plptr != NULL;
           plptr = plptr->NextPtr)
      {
         c = plptr->PathPtr[1];
         if (c < 'A' || c > 'Z') continue;
         c -= 'A';
         if (QuickIndex[c]) continue;
         QuickIndex[c] = plptr;
         if (Debug) fprintf (stdout, "%d |%s|\n", c, QuickIndex[c]->PathPtr);
      }
      return;
   }

   /**********************/
   /* function execution */
   /**********************/

   /** if (Debug) fprintf (stdout, "|%s|\n", rqptr->PathInfoPtr); **/

   rqptr->AuthGroupCan = rqptr->AuthWorldCan = 0;
   rqptr->AuthRealmPtr = rqptr->AuthGroupPtr =
      rqptr->AuthRestrictListPtr = rqptr->AuthWorldRestrictListPtr = "";

   if ((plptr = alptr->PathListHead) == NULL) return;

   /* no need to compare the leading slash */
   c = tolower(*(cptr = rqptr->PathInfoPtr + 1));

   /* if there is an entry for this character that start with that one */
   if (c >= 'A' && c <= 'Z')
   {
      c -= 'A';
      if (QuickIndex[c])
         plptr = QuickIndex[c];
      else
      {
         /* there can't be a controlled path beginning with that character */
         return;
      }
   }

   if (Debug) fprintf (stdout, "|%s|\n", plptr->PathPtr);

   for (/* from above */; plptr != NULL; plptr = plptr->NextPtr)
   {
/**
      if (Debug)
         fprintf (stdout,
"nlptr |%s|%s|%s|\n|%s|%s|\nGroupCan: %08.08X WorldCan: %08.08X\n",
            plptr->PathPtr, plptr->RealmPtr, plptr->GroupPtr,
            plptr->GroupRestrictListPtr,
            plptr->WorldRestrictListPtr,
            plptr->AuthGroupCan, plptr->AuthWorldCan);
**/

      pptr = cptr;
      tptr = plptr->PathPtr + 1;

      /* no need to continue on if past the lexicographic cutoff point */
      if (isalpha(*pptr) && isalpha(*tptr) && tolower(*pptr) < tolower(*tptr))
         return (SS$_NORMAL);

      while (*tptr && *pptr &&
             *tptr != '*' &&
             (tolower(*tptr) == tolower(*pptr) ||
             *pptr == '*' || *pptr == '%' || *pptr == '.'))
      {
         /** if (Debug) fprintf (stdout, "|%s|%s|\n", tptr, pptr); **/
         if (*pptr == '*' || *pptr == '%' || *pptr == '.')
            while (*pptr == '*' || *pptr == '%' || *pptr == '.') pptr++;
         else
         {
            tptr++;
            pptr++;
         }
      }

      /* path template ended in a wildcard or exact match, path matched! */
      if (*tptr == '*' || !(*tptr || *pptr)) break;
   }

   if (plptr == NULL) return;

   rqptr->AuthRealmPtr = plptr->RealmPtr;
   rqptr->AuthGroupPtr = plptr->GroupPtr;
   rqptr->AuthGroupCan = plptr->AuthGroupCan;
   rqptr->AuthWorldCan = plptr->AuthWorldCan;
   rqptr->AuthRestrictListPtr = plptr->GroupRestrictListPtr;
   rqptr->AuthWorldRestrictListPtr = plptr->WorldRestrictListPtr;
}

/*****************************************************************************/
/*
This function just wraps the reporting function, loading a temporary database
if necessary for reporting from the configuration file.
*/ 

AuthPathReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean  UseServerDatabase
)
{
   int  status;
   struct AuthLoadStruct  AuthLoad;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathReport()\n");

   if (UseServerDatabase)
   {
      memcpy (&AuthLoad, &ServerAuthLoad, sizeof(struct AuthLoadStruct)); 
      AuthPathReportNow (rqptr, &AuthLoad);
   }
   else
   {
      memset (&AuthLoad, 0, sizeof(struct AuthLoadStruct)); 
      /* indicate it's being used for a report */
      AuthLoad.RequestPtr = rqptr;
      AuthPathConfig (&AuthLoad);
      AuthPathReportNow (rqptr, &AuthLoad);
      AuthPathListFree (&AuthLoad);
      if (AuthLoad.ProblemReportPtr != NULL)
         VmFree (AuthLoad.ProblemReportPtr);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
Display all records in the authorization path linked-list.  Called from the
For HTTP report, uses blocking network write.
*/

AuthPathReportNow
(
struct RequestStruct *rqptr,
struct AuthLoadStruct *alptr
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Path Authorization</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Path Authorization</H2>\n\
!AZ\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (SourceFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR>\
<TH ALIGN=RIGHT>File:</TH>\
<TD ALIGN=LEFT>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH>\
</TR>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH>\
<TD ALIGN=LEFT>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR>\
<TH>path</TH><TH>realm</TH><TH>group</TH>\
<TH>group access</TH><TH>world access</TH>\
</TR>\n");

   static $DESCRIPTOR (PathFaoDsc,
"<TR>\
<TD><TT>!AZ</TT></TD><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD>\
</TR>\n\
!AZ!AZ!AZ!AZ!AZ\
!AZ!AZ!AZ");

   static $DESCRIPTOR (NoneFaoDsc,
"<TR ALIGN=CENTER><TD COLSPAN=5><I>(none)</I></TD></TR>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct AuthPathRecordStruct  *plptr;

   int  status,
        EntryCount;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  *CanStringPtr;
   char  Buffer [2048],
         GroupCanString [128],
         WorldCanString [128];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPathReportNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (alptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = alptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (alptr == &ServerAuthLoad)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr++ = alptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;

   if (alptr == &ServerAuthLoad)
      *vecptr++ = "Server";
   else
      *vecptr++ = "File";
   *vecptr++ = alptr->LoadFileName;
   *vecptr++ = MapVmsPath (alptr->LoadFileName);
   if (alptr == &ServerAuthLoad)
   {
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&alptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&alptr->RevBinTime, 20);
   }

   status = sys$faol (&SourceFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   EntryCount = 0;

   for (plptr = alptr->PathListHead; plptr != NULL; plptr = plptr->NextPtr)
   {
      EntryCount++;
/*
      if (Debug)
         fprintf (stdout, "plptr |%s|%s|%s|%08.08X|%08.08X|\n",
                  plptr->PathPtr, plptr->RealmPtr, plptr->GroupPtr,
                  plptr->AuthGroupCan, plptr->AuthWorldCan);
*/

      if ((CanStringPtr =
           AuthCanString (rqptr, plptr->AuthGroupCan)) == NULL)
         return;
      strcpy (GroupCanString, CanStringPtr);

      if ((CanStringPtr =
           AuthCanString (rqptr, plptr->AuthWorldCan)) == NULL)
         return;
      strcpy (WorldCanString, CanStringPtr);

      vecptr = FaoVector;

      *vecptr++ = plptr->PathPtr;
      *vecptr++ = plptr->RealmPtr;
      *vecptr++ = plptr->GroupPtr;
      *vecptr++ = GroupCanString;
      *vecptr++ = WorldCanString;

      if (plptr->GroupRestrictListPtr[0])
      {
         *vecptr++ = "<TR><TD></TD><TD COLSPAN=2><FONT SIZE=2><I>";
         if (plptr->GroupPtr[0])
            *vecptr++ = "group";
         else
            *vecptr++ = "realm";
         *vecptr++ = " restricted to</I></FONT></TD><TD COLSPAN=2><TT>";
         *vecptr++ = plptr->GroupRestrictListPtr;
         *vecptr++ = "</TT></TD></TR>\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (plptr->WorldRestrictListPtr[0])
      {
         *vecptr++ =
"<TR><TD></TD><TD COLSPAN=2>\
<FONT SIZE=2><I>world restricted to</I></FONT></TD><TD COLSPAN=2><TT>";
         *vecptr++ = plptr->WorldRestrictListPtr;
         *vecptr++ = "</TT></TD></TR>\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      status = sys$faol (&PathFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$fao()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   if (!EntryCount)
      NetWriteBuffered (rqptr, 0, NoneFaoDsc.dsc$a_pointer,
                                 NoneFaoDsc.dsc$w_length);

   NetWriteBuffered (rqptr, 0, EndPageFaoDsc.dsc$a_pointer,
                              EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
Display all records in the authentication linked-list, binary tree.  Called
by the Control.c or the Request.c module.  For HTTP report, uses blocking
network write.
*/

AuthCacheTreeReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... User Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>User Authentication</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Since</TH></TR>\n\
<TR><TH>Time</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\\
<TR>\
<TH>realm</TH><TH>group</TH><TH>user</TH><TH>access</TH>\
<TH COLSPAN=2>scheme</TH>\
<TH>most recent</TH>\
<TH COLSPAN=2>authenticated</TH>\
</TR>\n\
<TR>\
<TH COLSPAN=4></TH>\
<TH>basic</TH><TH>digest</TH>\
<TH></TH>\
<TH>yes<BR>(database)</TH><TH>NO</TH>\
</TR>\n");

   static $DESCRIPTOR (NoneFaoDsc,
"<TR ALIGN=CENTER><TD COLSPAN=9><I>(none)</I></TD></TR>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Buffer [2048],
         TimeCurrent [32];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCacheTreeReport()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = strcpy (TimeCurrent, DayDateTime (&rqptr->BinaryTime, 20));
   *vecptr++ = DayDateTime (&AuthCacheTreeBinTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   AuthDisplayRecordCount = 0;
   status = lib$traverse_tree (&AuthCacheTreeHead,
                               &AuthCacheTreeReportRecord,
                               rqptr);
   if (Debug) fprintf (stdout, "lib$traverse_tree() %%X%08.08X\n", status);


   if (!AuthDisplayRecordCount)
      NetWriteBuffered (rqptr, 0, NoneFaoDsc.dsc$a_pointer,
                                 NoneFaoDsc.dsc$w_length);

   NetWriteBuffered (rqptr, NextTaskFunction,
                    EndPageFaoDsc.dsc$a_pointer,
                    EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
Called by AuthCacheTreeReport().  Display the details of all records in the
linked-list binary tree.  Output the details via blocking network write.
*/

AuthCacheTreeReportRecord
(                      
struct AuthCacheRecordStruct *TreeNodePtr,
struct RequestStruct *rqptr
)
{
   static $DESCRIPTOR (ReportFaoDsc,
"<TR>\
<TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ!AZ</TD><TD>!UL</TD><TD>!UL</TD>\
<TD>!17%D</TD>\
<TD>!UL &nbsp;&nbsp;(!UL)</TD><TD>!UL</TD>\
</TR>\n");

   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  *UserCanStringPtr;
   char  Buffer [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCacheTreeReportRecord()\n");

   AuthDisplayRecordCount++;

   if ((UserCanStringPtr =
        AuthCanString (rqptr, TreeNodePtr->AuthUserCan)) == NULL)
      return (STS$K_ERROR);

   vecptr = FaoVector;

   *vecptr++ = TreeNodePtr->Realm;
   *vecptr++ = TreeNodePtr->Group;
   *vecptr++ = TreeNodePtr->UserName;
   *vecptr++ = UserCanStringPtr;
   if (TreeNodePtr->HttpsOnly)
      *vecptr++ = " (&quot;https:&quot;&nbsp;only)";
   else
      *vecptr++ = "";
   *vecptr++ = TreeNodePtr->BasicCount;
   *vecptr++ = TreeNodePtr->DigestCount;
   *vecptr++ = &TreeNodePtr->LastAccessBinTime;
   *vecptr++ = TreeNodePtr->AccessCount;
   *vecptr++ = TreeNodePtr->DataBaseCount;
   *vecptr++ = TreeNodePtr->FailureCount;

   status = sys$faol (&ReportFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Set string text according to capability vector in the in-memory authorization
information.  These may be different to the bits in the on-disk database,
reported by HTAdminCanString().
*/

char* AuthCanString
(
struct RequestStruct *rqptr,
unsigned long CanFlags
)
{
   static $DESCRIPTOR (CanBriefFaoDsc, "!AZ\0");
   static char  Buffer [128];
   static $DESCRIPTOR (BufferDsc, Buffer);

   register unsigned long  *vecptr;

   int  status;
   unsigned long  FaoVector [16];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthCanString()\n");

   vecptr = FaoVector;

   if ((CanFlags & HTTP_METHOD_DELETE ||
        CanFlags & HTTP_METHOD_POST ||
        CanFlags & HTTP_METHOD_PUT) &&
       CanFlags & HTTP_METHOD_GET)
      *vecptr++ = "read <B>+ write</B>";
   else
   if ((CanFlags & HTTP_METHOD_DELETE ||
        CanFlags & HTTP_METHOD_POST ||
        CanFlags & HTTP_METHOD_PUT))
      *vecptr++ = "write-only";
   else
   if (CanFlags & HTTP_METHOD_GET)
      *vecptr++ = "read-only";
   else
      *vecptr++ = "<I>none!</I>";

   status = sys$faol (&CanBriefFaoDsc, 0, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (NULL);
   }
   return (Buffer);
}

/*****************************************************************************/
/*
*/

AuthControlReload
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Server !AZ authorization paths reloaded.\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ \
Detected During Load</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct AuthPathRecordStruct  *plptr;

   int  status;
   unsigned long  FaoVector [16];
   unsigned short  Length;
   char  Buffer [2048];
   struct AuthLoadStruct  *alptr;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthControlReload()\n");

   AuthPathConfig (NULL);

   alptr = &ServerAuthLoad;

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (alptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = alptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      *vecptr++ = alptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);

      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   NetWriteBuffered (rqptr, NextTaskFunction,
                    EndPageFaoDsc.dsc$a_pointer,
                    EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/

