/*****************************************************************************/
/*
                                 Auth.h

Definitions, structures and function prototypes for authentication module.

*/
/*****************************************************************************/

#ifndef WASD_AUTH_H

#define WASD_AUTH_H 1

#ifndef boolean

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))

#endif

/* some storage maximums */
#define  MaxAuthDigestResponseLength   32
#define  MaxAuthPasswordLength         15
#define  MaxAuthRealmGroupNameLength   32
#define  MaxAuthUserNameLength         15
#define  MaxAuthFullNameLength         31
#define  MaxAuthEmailLength           127
#define  MaxAuthContactLength         127

/* constants used to identify authorization schemes */
#define  AUTH_SCHEME_BASIC   0x00000001
#define  AUTH_SCHEME_DIGEST  0x00000002

/* constants for bits for flags in HTA database */
#define  AUTH_FLAG_GET         0x00000001
#define  AUTH_FLAG_HEAD        0x00000002
#define  AUTH_FLAG_POST        0x00000004
#define  AUTH_FLAG_PUT         0x00000008
#define  AUTH_FLAG_DELETE      0x00000010
#define  AUTH_FLAG_TRACE       0x00000020
#define  AUTH_FLAG_HTTPS_ONLY  0x20000000
#define  AUTH_FLAG_MANAGER     0x40000000
#define  AUTH_FLAG_ENABLED     0x80000000

#define  AUTH_EXTERNAL     "EXTERNAL"
#define  AUTH_PROMISCUOUS  "-PROMISCUOUS-"
#define  AUTH_VMS          "VMS"
#define  AUTH_WORLD        "WORLD"
#define  HTA_DIRECTORY     "HT_AUTH:"
#define  HTA_FILE_TYPE     ".HTA"

/* bogus VMS status codes, these can be passed back and detected */
#define  RESTRICTED_NOT          0x00000001
#define  RESTRICTED_BY_HOSTNAME  0x0fffff1a
#define  RESTRICTED_BY_PROTOCOL  0x0fffff2a
#define  RESTRICTED_BY_USERNAME  0x0fffff3a
#define  RESTRICTED_BY_OTHER     0x0fffff4a

/* identifier names controlling SYSUAF authentication with /SYSUAF=ID */
#define  AUTH_VMS__GROUP_ID  "WASD_VMS__"
#define  AUTH_VMS_HTTPS_ID  "WASD_VMS_HTTPS"
#define  AUTH_VMS_PWD_ID    "WASD_VMS_PWD"
#define  AUTH_VMS_RW_ID     "WASD_VMS_RW"
#define  AUTH_VMS_R_ID      "WASD_VMS_R"

/**************/
/* structures */
/**************/

/*
   Store these structures naturally-aligned on AXP.
   Uses a bit more storage but makes access as efficient as possible.
*/

#ifdef __ALPHA
#   pragma member_alignment __save
#   pragma member_alignment
#endif

#define AuthCurrentDatabaseVersion 0x01

/* record structure of on-disk HTA database */
struct AuthHtRecordStruct
{
   char  UserName [MaxAuthUserNameLength+1];
   char  FullName [MaxAuthFullNameLength+1];
   char  Email [MaxAuthEmailLength+1];
   int  UserNameLength;
   unsigned long  Flags;
   unsigned long  HashedPwd [2];
   unsigned char  A1DigestLoCase [16];
   unsigned char  A1DigestUpCase [16];
   int  ChangeCount;
   unsigned long  LastChangeBinTime [2];
   int  AccessCount;
   unsigned long  LastAccessBinTime [2];
   int  FailureCount;
   unsigned long  LastFailureBinTime [2];
   unsigned long  AddedBinTime [2];
   unsigned char  Contact [MaxAuthContactLength+1];
   unsigned char  FillTo512Bytes [115];
   unsigned char  DatabaseVersion;
};

/* in-memory path authorization structure */
struct AuthPathRecordStruct
{
   struct AuthPathRecordStruct  *PrevPtr;
   struct AuthPathRecordStruct  *NextPtr;
   char  *RealmPtr;
   int  RealmLength;   
   char  *GroupPtr;
   int  GroupLength;   
   char  *PathPtr;
   int  PathLength;
   char  *GroupRestrictListPtr;
   int  GroupRestrictListLength;
   char  *WorldRestrictListPtr;
   int  WorldRestrictListLength;
   unsigned long  AuthGroupCan;
   unsigned long  AuthWorldCan;
   /* string data is stored here and pointed to by above pointers */
};

/* in-memory user authentication cache structure */
struct AuthCacheRecordStruct
{
   struct AuthCacheRecordStruct  *LeftLink;
   struct AuthCacheRecordStruct  *RightLink;
   unsigned short  Reserved;
   boolean  CanChangeSysUafPwd,
            HttpsOnly;
   int  CacheMinutes;
   int  RevalidateUserMinutes;
   int  DataBaseCount;
   char  Realm [MaxAuthRealmGroupNameLength+1];
   int  RealmLength;
   char  Group [MaxAuthRealmGroupNameLength+1];
   int  GroupLength;
   char  UserName [MaxAuthUserNameLength+1];
   int  UserNameLength;
   char  Password [MaxAuthPasswordLength+1];
   char  DigestResponse [MaxAuthDigestResponseLength+1];
   boolean  SysUafAuthenticated;
   unsigned char  *VmsUserProfilePtr;
   unsigned short  VmsUserProfileLength;
   unsigned long  BasicCount;
   unsigned long  DigestCount;
   unsigned long  AccessCount;
   unsigned long  FailureCount;
   unsigned long  AuthUserCan;
   unsigned long  LastAccessBinTime [2];
};

/* structure used to hold an authorization database loaded from file */
struct AuthLoadStruct
{
   int  LineNumber,
        ProblemCount,
        ProblemReportLength;
   unsigned long  LoadBinTime [2],
                  RevBinTime [2];
   char  *ProblemReportPtr;
   char  LoadFileName [256];
   struct AuthPathRecordStruct  *PathListHead,
                                *PathListTail;
   /* only used during HTTP request */
   struct RequestStruct  *RequestPtr;
};

#ifdef __ALPHA
#   pragma member_alignment __restore
#endif

/***********************/
/* function prototypes */
/***********************/

int Authorize (struct RequestStruct*);
int AuthParseAuthorization (struct RequestStruct*);
int AuthVerifyPassword (struct RequestStruct*);
int AuthVerifyVmsPassword (struct RequestStruct*);
int AuthReadHtDatabase (struct RequestStruct*, char*, boolean);

AuthUserResetRecord (char*, char*, char*);
AuthCacheTreeFree ();
int AuthCacheTreeAllocateRecord (struct AuthCacheRecordStruct*,
                                 struct AuthCacheRecordStruct**,
                                 unsigned long);
int AuthCacheTreeCompareRecord (struct AuthCacheRecordStruct*,
                                struct AuthCacheRecordStruct*,
                                unsigned long);
int AuthCacheTreeFreeRecord (struct AuthCacheRecordStruct*, unsigned long);
AuthCacheTreeReport (struct RequestStruct*, void*);
int AuthCacheTreeReportRecord (struct AuthCacheRecordStruct*,
                               struct RequestStruct*);
int AuthCacheTreeEntryReset (struct AuthCacheRecordStruct*,
                             struct AuthCacheRecordStruct*);

char* AuthCanString (struct RequestStruct*, unsigned long);

int AuthCheckVmsUserAccess (struct RequestStruct*, char*, int);
int AuthCheckWriteAccess (struct RequestStruct*, char*, int);
int AuthCreateVmsUserProfile (struct RequestStruct*);

int AuthControlReload (struct RequestStruct*, void*);
int AuthPathAdd (struct AuthLoadStruct*, char*, char*, char*, char*, char*,
                 unsigned long, unsigned long);
int AuthPathCheck (struct RequestStruct*, struct AuthLoadStruct*);
int AuthPathConfig (struct AuthLoadStruct*);
int AuthPathListFree (struct AuthLoadStruct*);
int AuthPathReport (struct RequestStruct*, void*, boolean);
int AuthSysUafCheckIdentifier (struct RequestStruct*, unsigned long);

#endif /* WASD_AUTH_H */

/*****************************************************************************/
