/*****************************************************************************/
/*
                                 Cache.c

This module implements a file data and revision time cache. It is different to
most other modules in that it doesn't have a "task" structure. The small
amount of storage required is integrated into the request structure. It is
designed to provided an efficient static document request (file transfer)
mechanism. Unlike the file module which may interleave it's activities within
the those of other modules (e.g. the directory module using it to provide
read-me information), it can only be used once, stand-alone per request.

Cache data is loaded by the file module while concurrently transfering that
data to the request client, using buffer space supplied by cache module, space
that can then be retained for reuse as cache. Hence the cache load adds no
significant overhead to the actual reading and initial transfer of the file.

The cache has limits imposed on it. A maximum number of files that can be
cached at any one time, and a maximum amount of memory that can be allocated
in total by the cache. Each of these two operates to limit the other. In
addition the maximum size of a file that can be loaded into the cache can be
specified.

Terminology
-----------

"hit" refers to a request path being found in cache. If the data is still
valid the request can be supplied from cache.

"load"ing the cache refers to reading the contents of a file into cache
memory.

"valid" means that the file from which the cached data was originally read has
not had it's revision date changed (the implication being is the file contents
have not changed.

Why Implement Caching?
----------------------

Caching, be definition, attempts to improve performance by keeping data in
storage that is faster to access than it's usual location.  The performance
improvement can be assessed in three basic ways.  Reduction in latency when
accessing the data, of processing involved, and in impact on the usual storage
location.

This cache is provided to address all three.  Where networks are particularly
responsive a reduction in request latency can often be noticable.  Where
servers are particularly busy or where disk systems particularly loaded a
reduction in the need to access the file system can significantly improve
performance.  My suggestion is though, that for most VMS sites high levels of
hits are not a great concern, and for these caching can easily be left
disabled.
            
Why take so long to implement caching? Well, WASD's intranet services are not
particularly busy sites. This coupled with powerful hardware has meant
performance has not been an overriding concern. However, this cache module
came about because I felt like creating it and it was an obvious lack of
functionality within the server, not because WASD (the organisation) needed
it.

Cache Suitability Considerations
--------------------------------

A cache is not always of benefit!  It's cost may outweigh it's return.

Any cache's efficiencies can only occur where subsets of data are consistently
being demanded. Although these subsets may change slowly over time a consistent
and rapidly changing aggregate of requests lose the benefit of more readily
accessable data to the overhead of cache management due to the constant and
continuous flushing and reloading of cache data. This server's cache is no
different, it will only improve performance if the site experiences some
consistency in the files requested. For sites that have only a small
percentage of files being repeatedly requested it is probably better that the
cache be disabled. The other major consideration is available system memory.
On a system where memory demand is high there is little value in having cache
memory sitting in page space, trading disk I/O and latency for paging I/O and
latency. On memory-challenged systems cache is probably best disabled.

To help assessment of the cache's efficiency for any given site monitor the
administration menu's cache report.

With "loads not hit", the count represents the cumulative number of files
loaded but never subsequently hit. If this percentage is high it means most
files loaded are never hit, indicating the site's request profile is possibly
unsuitable for caching.

The item "hits" respresents the cumulative, total number of hits against the
cumulative, total number of loads. The percentage here can range from zero to
many thousands of percent :^) with less than 100% indicating poor cache
performance, from 200% upwards better and good performance. The items "1-9",
"10-99" and "100+" show the count and percentage of total hits that occured
when a given entry had experienced hits within that range (e.g. if an entry
has had 8 previous hits, the ninth increments the "1-9" item whereas the tenth
and eleventh increments the "10-99" item, etc.)

Other considerations also apply when assessing the benefit of having a cache.
For example, a high number and percentage of hits can be generated while the
percentage of "loads not hit" could be in the also be very high. The
explanation for this would be one or two frequently requested files being hit
while most others are loaded, never hit, and flushed as other files request
cache space. In situations such as this it is difficult to judge whether cache
processing is improving performance or just adding overhead.

Again, my suggestion is, that for most VMS sites, high levels of access are not
a great concern, and for these caching can easily be left disabled.
            
Description
-----------

The path used to hit on the cache MUST be the mapped path, not the
client-supplied, request path. With conditional mapping, identical request
paths may be mapped to completely different virtual paths.

If a path contains a VMS version delimitter (';') then do not cache that path,
these are usually one-offs.

Space for a file's data is dynamically allocated and reallocated if necessary
as cache entries are reused.  It is allocated in user-specifiable chunks.  It
is expected this mechanism provides some efficiencies when reusing cache
entries.

A simple hash table is used to try and initially hit on a request path. A
collision list allows rapid subsequent searching. The search is based on three
factors. A simple, efficiently generated, case-insensitive hash value
providing a rapid but inconclusive index into the cached data. Second, the
length of the two paths. Finally a conclusive, case-insensitive string
comparison if the previous two tests were matches.

Cache entries are also maintained in a global linked list with the most recent
and most frequently hit entries towards the head of the list. The linked-list
organisation allows a simple implementation of a least-recently-used (LRU)
algorithm for selecting an entry when a new request demands an entry and space
for cache loading. The linked list is naturally ordered from most recently and
most frequently accessed at the head, to the least recently and least
frequently accessed at the tail. Hence an infrequently accessed entry is
selected from the tail end of the list, it's data invalidated and given to the
new request for cache load. Invalidated data cache entries are also
immediately placed at the tail of the list for reuse/reloading.

When a new entry is initially loaded it is placed at the top of the list. Hits
on other entries result in a check being made against the number of hits of
head entry in the list. If the entry being hit has a higher hit count it is
placed at the head of the list, pushing the previously head entry "down". If
not then it is again checked against the entry immediately before it in the
list. If higher then the two are swapped. This results in the most recently
loaded entries and the more frequently hit being nearest and migrating towards
the start of the search.

To help prevent the cache thrashing with floods of requests for not currently
loaded files, any entry that has a suitably high number of hits over the recent
past (suitably high ... how many is that, and recent past ... how long is
that?) are not reused until no hits have occured within that period.  Hopefully
this prevents lots of unnecessary loads of one-offs at the expense of genuinely
frequently accessed files.

To prevent multiple loads of the same path/file, for instance if a subsequent
request demands the same file as a previous request is still currently loading,
any subsequent request will merely transfer the file, not concurrently load it
into the cache.

Cache Content Validation
------------------------

The cache will automatically revalidate the file data after a specified number
of seconds by comparing the original file revision time to the current revision
time. If different the file contents have changed and the cache contents
declared invalid. If found invalid the file transfer then continues outside of
the cache with the new contents being concurrently reloaded into the cache.
Cache validation is also always performed if the request uses "Pragma:
no-cache" (i.e. as with the Netscape Navigator reload function). Hence there is
no need for any explicit flushing of the cache under normal operation. If a
document does not immediately reflect and changes made to it (i.e. validation
time has not been reached) validation (and consequent reload) can be "forced"
with a browser reload. The entire cache may be purged of cached data either
from the server administration menu or using command line server control.


VERSION HISTORY
---------------
14-MAY-98  MGD  request-specified content-type ("httpd=content&type=")
18-MAR-98  MGD  use file's VBN and first free byte to check size changes
                (allows variable record files to be cached more efficiently)
24-FEB-98  MGD  if CacheAcpInfo() reports a problem
                then let the file module handle/report it,
                add file size check to entry validation (allow for extend)
10-JAN-98  MGD  added a (much overdue) hash collision list,
                fixed problem with cache purge (it mostly didn't :^)
22-NOV-97  MGD  sigh, bugfix; need to proactively free memory when at capacity
05-OCT-97  MGD  initial development for v4.5
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <libdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "cache.h"
#include "httpd.h"
#include "error.h"
#include "file.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"

/******************/
/* global storage */
/******************/

boolean  CacheEnabled,
         CacheHashTableInitialised;

/* this is run-time storage */
int  CacheChunkInBytes,
     CacheEntryCount,
     CacheEntriesMax,
     CacheFileKBytesMax,
     CacheFrequentHits,
     CacheFrequentSeconds,
     CacheHashTableEntries,
     CacheHashTableMask,
     CacheMemoryInUse,
     CacheTotalKBytesMax,
     CacheValidateSeconds;

/* these counters may be zeroed when accounting is zeroed */
int  CacheHashTableCollsnCount,
     CacheHashTableCount,
     CacheHashTableHitCount,
     CacheHashTableMissCount,
     CacheHashTableCollsnHitCount,
     CacheHashTableCollsnMissCount,
     CacheHitCount,
     CacheHits0,
     CacheHits10,
     CacheHits100,
     CacheHits100plus,
     CacheListHitCount,
     CacheLoadCount,
     CacheNotHitCount,
     CacheNoHitsCount,
     CacheSearchCount;

struct ListHeadStruct  CacheList;

struct GenericCacheEntry  **CacheHashTable;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  CliCacheEnabled;
extern boolean  CliCacheDisabled;
extern int  OutputBufferSize;
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Initialize cache run-time parameters at startup (even though caching might be
initially disabled).  Also allocate the hash table when appropriate.
*/ 

CacheInit (boolean Startup)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheInit()\n");

   if (Startup)
   {
      if (!CliCacheDisabled && (CliCacheEnabled || Config.CacheEnabled))
         CacheEnabled = true;
      else
         CacheEnabled = false;

      CacheChunkInBytes = Config.CacheChunkKBytes << 10;
      CacheEntriesMax = Config.CacheEntriesMax;
      CacheFileKBytesMax = Config.CacheFileKBytesMax;
      CacheFrequentHits = Config.CacheFrequentHits;
      CacheFrequentSeconds = Config.CacheFrequentSeconds;
      CacheHashTableEntries = Config.CacheHashTableEntries;
      CacheTotalKBytesMax = Config.CacheTotalKBytesMax;
      CacheValidateSeconds = Config.CacheValidateSeconds;

      CacheEntryCount = CacheMemoryInUse = 0;

      CacheZeroCounters ();
   }

   if (CacheEntriesMax && CacheEnabled)
   {
      /* hash table is not created until caching is actually enabled */
      int  cnt;

      /* initialize cache virtual memory management */
      VmCacheInit (CacheTotalKBytesMax, CacheChunkInBytes >> 10);

      /* hash table needs to be a power of 2 (i.e. 64, 128, 256, 512 ...) */
      if (!CacheHashTableEntries) CacheHashTableEntries = CacheEntriesMax;
      for (cnt = 0; CacheHashTableEntries; cnt++)
         CacheHashTableEntries = CacheHashTableEntries >> 1;
      CacheHashTableEntries = 1;
      while (--cnt) CacheHashTableEntries = CacheHashTableEntries << 1;

      /* e.g. cache table size of 512 becomes a mask of 0x1ff */
      CacheHashTableMask = CacheHashTableEntries - 1;

      if (Debug)
         fprintf (stdout, "%d %d %d\n",
                  CacheHashTableEntries, CacheHashTableMask,
                  sizeof(struct CacheStruct*)*CacheHashTableEntries);

      CacheHashTable =
         VmGetCache (sizeof(struct CacheStruct*)*CacheHashTableEntries);

      CacheMemoryInUse += sizeof(struct CacheStruct*)*CacheHashTableEntries;

      for (cnt = 0; cnt < CacheHashTableEntries; cnt++)
         CacheHashTable[cnt] = NULL;

      CacheHashTableInitialised = true;
   }
}

/*****************************************************************************/
/*
Just zero the cache-associated counters :^)
*/ 

CacheZeroCounters ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheZeroCounters()\n");

   CacheHitCount = CacheEntryCount = CacheHashTableHitCount =
      CacheHashTableCollsnCount = CacheHashTableCollsnHitCount =
      CacheHashTableCollsnMissCount = CacheHashTableCount =
      CacheHashTableMissCount =
      CacheNoHitsCount = CacheHits0 = CacheHits10 = CacheHits100 =
      CacheHits100plus = CacheLoadCount = CacheSearchCount = 0;
}

/*****************************************************************************/
/*
Look through the cache list for a path that matches the request path.  Return
true if one found, with request cache pointer set to it.  Return false if not
found.

NOTE: as with CacheLoadBegin(), the path passed to this function MUST be the
mapped path, not the request path. With conditional mapping identical request
paths may be mapped to completely different virtual paths.
*/ 

CacheSearch
(
struct RequestStruct *rqptr,
char *Path
)
{
   register unsigned long  HashValue;
   register char  *cptr, *sptr;
   register struct CacheStruct  *captr;
   register struct ListEntryStruct  *leptr;

   int  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheSearch() |%s|\n", Path);

   if (!CacheHashTableInitialised) CacheInit (false);

   /* value of 1 represents cache was searched */
   rqptr->CacheInvolvement = 1;

   /* generate lower-case hash */
   HashValue = 1;
   for (cptr = Path; *cptr && *cptr != ';'; cptr++)
      HashValue = ((tolower(*cptr)*541)^(HashValue*3)) & CacheHashTableMask;
   if (*cptr)
   {
      /*
         Not interested in caching specific versions, only version-less paths.
         Zero indicates we are not at all interested in caching this file.
      */
      rqptr->CacheHashValue = 0;
      return (false);
   }

   Length = cptr - Path;
   if (Debug) fprintf (stdout, "%d %d\n", HashValue, Length);

   /* keep this in case we end up cache loading using the request */
   rqptr->CacheHashValue = HashValue;

   CacheHashTableCount++;

   /* check the hash table */
   if ((captr = CacheHashTable[HashValue]) == NULL)
   {
      /* nope, no pointer against that hash value */
      CacheHashTableMissCount++;
      rqptr->CachePtr = NULL;
      return (false);
   }

   /* now, ensure the hash hit is for the same path */
   if (captr->PathLength)
   {
      /* case insensitive string compare of request and cache paths */
      cptr = Path;
      sptr = captr->Path;
      while (*cptr && *sptr && tolower(*cptr) == tolower(*sptr))
      {
         cptr++;
         sptr++;
      }
      if (!*cptr && !*sptr)
      {
         /*************************************************/
         /* yup, the same ... hash table hit straight-up! */
         /*************************************************/

         CacheHashTableHitCount++;

         if (captr->CacheValid)
         {
            /* cache data can be used */
            rqptr->CachePtr = captr;
            return (true);
         }
         if (captr->CacheLoading)
         {
            /* data is currently being loaded and can't be used yet */
            rqptr->CacheLoading = true;
            rqptr->CachePtr = NULL;
            return (false);
         }

         /* data is not valid for some other reason, hmmm! */
         rqptr->CachePtr = NULL;
         return (false);
      }
   }

   /*************/
   /* collision */
   /*************/

   CacheHashTableCollsnCount++;

   /* process any collision list */
   for (captr = captr->HashCollisionNextPtr;
        captr != NULL;
        captr = captr->HashCollisionNextPtr)
   {
      if (Debug)
         fprintf (stdout, "%d %d |%s| %d %d\n",
                  captr->HashValue, captr->PathLength,
                  captr->Path, captr->CacheSize, captr->ContentLength);

      /* if any of these are not valid then just continue to the next entry */
      if (HashValue != captr->HashValue) continue;
      if (Length != captr->PathLength) continue;

      /*
         If the data is not valid but being loaded then consider it a hit.
         This will prevent the same path having concurrent, multiple loads.
      */
      if (!captr->CacheValid && !captr->CacheLoading) continue;

      /* case insensitive string compare of request and cache paths */
      cptr = Path;
      sptr = captr->Path;
      while (*cptr && *sptr && tolower(*cptr) == tolower(*sptr))
      {
         cptr++;
         sptr++;
      }
      /* if not the same then continue with next cache path */
      if (*cptr || *sptr) continue;

      /* hit in hash table but ultimately found in the collision list */
      CacheHashTableCollsnHitCount++;

      if (captr->CacheValid && !captr->CacheLoading)
      {
         /* cache data can be used */
         rqptr->CachePtr = captr;
         return (true);
      }
      if (captr->CacheLoading)
      {
         /* data is currently being loaded and can't be used yet */
         rqptr->CacheLoading = true;
         rqptr->CachePtr = NULL;
         return (false);
      }

      /* data is not valid for some other reason, hmmm! */
      rqptr->CachePtr = NULL;
      return (false);
   }

   CacheHashTableCollsnMissCount++;
   rqptr->CachePtr = NULL;
   return (false);
}

/*****************************************************************************/
/*
Use this cached entry as the contents for the client.  Check if modified, if
not then just reply with a 302 header.  If contents should be transfered to
the client then begin.  Returns true is all processing required has already
been done.  Returns false if the calling routine must continue to process the
request.
*/ 
 
boolean CacheBegin (struct RequestStruct *rqptr)

{
   static long  Subx2 = 2,
                EdivIntoSeconds = 10000000;

   register struct CacheStruct  *captr, *tcaptr;
   register struct ListEntryStruct  *leptr;

   int  status;
   unsigned long  Seconds,
                  Remainder;
   unsigned long  ResultTime [2];
   char  *ContentTypePtr,
         *KeepAlivePtr;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheBegin()\n");

   /* get the pointer to the cache entry */
   captr = rqptr->CachePtr;

   if (!rqptr->PragmaNoCache)
   {
      /* check if it needs validating */
      status = lib$subx (&rqptr->BinaryTime, &captr->ValidateBinaryTime,
                         &ResultTime, &Subx2);
      if (VMSok (status))
         status = lib$ediv (&EdivIntoSeconds, &ResultTime,
                            &Seconds, &Remainder);
      if (Debug) fprintf (stdout, "%%X%08.08X\n", status);
      if (VMSnok (status)) Seconds = 9999999;
      if (Debug) fprintf (stdout, "%d %d\n", Seconds, CacheValidateSeconds);
   }

   if (rqptr->PragmaNoCache || Seconds > CacheValidateSeconds)
   {
      /*****************************/
      /* recheck modification time */
      /*****************************/

      captr->ValidatedCount++;

      if (VMSnok (status = CacheAcpInfo (rqptr)))
      {
         /***********************/
         /* file access problem */
         /***********************/

         /* no longer valid data */
         CacheEntryNotValid (captr);

         /* move it to the end of the cache list ready for reuse */
         ListMoveTail (&CacheList, captr);

         /* this cache entry is no longer associated with this request */
         rqptr->CachePtr = NULL;

         /* let the file module handle/report it */
         return (false);
      }

      if (Debug)
      {
         fprintf (stdout,
"times: %08.08X%08.08X %08.08X%08.08X\nVBN: %d %d ffb: %d %d\n",
            captr->ModifiedBinaryTime[1],
            captr->ModifiedBinaryTime[0],
            captr->ValidateModifiedBinaryTime[1],
            captr->ValidateModifiedBinaryTime[0],
            captr->EndOfFileVbn,
            captr->ValidateEndOfFileVbn,
            captr->FirstFreeByte,
            captr->ValidateFirstFreeByte);
      }

      if (captr->ModifiedBinaryTime[0] !=
          captr->ValidateModifiedBinaryTime[0] ||
          captr->ModifiedBinaryTime[1] !=
          captr->ValidateModifiedBinaryTime[1] ||
          captr->EndOfFileVbn != captr->ValidateEndOfFileVbn ||
          captr->FirstFreeByte != captr->ValidateFirstFreeByte)
      {
         /************************/
         /* data no longer valid */
         /************************/

         /* no longer valid data */
         CacheEntryNotValid (captr);

         /* move it to the end of the cache list ready for reuse */
         ListMoveTail (&CacheList, captr);

         /* this cache entry is no longer associated with this request */
         rqptr->CachePtr = NULL;

         /* let the file module reload it */
         return (false);
      }

      /********************/
      /* data still valid */ 
      /********************/

      /* note the time the cached data was last validated */
      memcpy (&captr->ValidateBinaryTime, &rqptr->BinaryTime, 8);
   }

   /**************************************/
   /* request to be fulfilled from cache */
   /**************************************/

   /* value of 2 represents cache was used for request */
   rqptr->CacheInvolvement = 2;

   memcpy (&captr->HitBinaryTime, &rqptr->BinaryTime, 8);

   CacheHitCount++;
   if (!captr->HitCount) CacheHits0--;
   captr->HitCount++;
   if (captr->HitCount < 10)
      CacheHits10++;
   else
   if (captr->HitCount < 100)
      CacheHits100++;
   else
      CacheHits100plus++;

   /* if this entry has more hits move it to the head of the cache list */
   leptr = CacheList.HeadPtr;
   tcaptr = (struct CacheStruct*)leptr;
   if (captr->HitCount > tcaptr->HitCount)
      ListMoveHead (&CacheList, captr);
   else
   {
      /* if this entry has more hits than the one "above" it swap them */
      leptr = captr;
      if (leptr->PrevPtr != NULL)
      {
         leptr = leptr->PrevPtr;
         tcaptr = (struct CacheStruct*)leptr;
         if (captr->HitCount > tcaptr->HitCount)
         {
            ListRemove (&CacheList, captr);
            ListAddBefore (&CacheList, leptr, captr);
         }
      }
   }

   if (rqptr->HttpMethod == HTTP_METHOD_GET &&
       (rqptr->IfModifiedSinceBinaryTime[0] ||
        rqptr->IfModifiedSinceBinaryTime[1]) &&
        !rqptr->PragmaNoCache)
   {
      /*********************/
      /* if modified since */
      /*********************/

      if (VMSnok (status =
          HttpIfModifiedSince (rqptr, &captr->ModifiedBinaryTime,
                               captr->ContentLength)))
      {
         /* task status LIB$_NEGTIM if not modified/sent */
         rqptr->AccountingDone = ++Accounting.CacheHitCount;
         Accounting.CacheHitNotModifiedCount++;
         captr->HitNotModifiedCount++;

         /* this cache entry is no longer associated with this request */
         rqptr->CachePtr = NULL;

         RequestEnd (rqptr);
         return (true);
      }
   }

   rqptr->AccountingDone = ++Accounting.CacheHitCount;

   /****************************/
   /* generate response header */
   /****************************/

   if (Config.KeepAliveTimeoutSeconds &&
       rqptr->KeepAliveRequest &&
       captr->ContentLength > 0)
   {
      rqptr->KeepAliveResponse = true;
      rqptr->KeepAliveCount++;
      KeepAlivePtr = KeepAliveHttpHeader;
   }
   else
      KeepAlivePtr = "";

   if (rqptr->QueryStringPtr[0] &&
       tolower(rqptr->QueryStringPtr[0]) == 'h' &&
       strsame (rqptr->QueryStringPtr, "httpd=content", 13))
   {
      /* request-specified content-type (default to plain-text) */
      if (strsame (rqptr->QueryStringPtr+13, "&type=", 6))
         ContentTypePtr = rqptr->QueryStringPtr + 19;
      else
         ContentTypePtr = "text/plain";
   }
   else
   {
      /* whatever it was when the file was cached */
      ContentTypePtr = captr->ContentTypePtr;
   }

   rqptr->ResponseStatusCode = 200;
   if ((rqptr->ResponseHeaderPtr =
        HttpHeader (rqptr, 200, ContentTypePtr,
                    captr->ContentLength,
                    &captr->ModifiedBinaryTime, KeepAlivePtr))
       == NULL)
   {
      /* this cache entry is no longer associated with this request */
      rqptr->CachePtr = NULL;

      RequestEnd (rqptr);
      return (true);
   }

   /* quit here if the HTTP method was HEAD */
   if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
   {
      /* this cache entry is no longer associated with this request */
      rqptr->CachePtr = NULL;

      /* RequestEnd() is ok up to here */
      RequestEnd (rqptr);
      return (true);
   }

   /************************/
   /* transfer cached data */
   /************************/

   /* 'CacheInUse' keeps track of whether the entry is in use or not */
   captr->InUseCount++;

   rqptr->CacheContentPtr = captr->ContentPtr;
   rqptr->CacheContentCount = captr->ContentLength;

   /* network writes are checked for success, fudge the first one! */
   rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   /* begin the transfer */
   CacheNext (rqptr);

   return (true);
}

/*****************************************************************************/
/*
Write the next (or first) block of data from the cache buffer to the client.
*/ 

CacheNext (struct RequestStruct *rqptr)

{
   int  status,
        DataLength;
   unsigned char  *DataPtr;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheNext()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      CacheEnd (rqptr);
      return;
   }

   if (rqptr->CacheContentCount > OutputBufferSize)
   {
      DataPtr = rqptr->CacheContentPtr;
      DataLength = OutputBufferSize;
      rqptr->CacheContentPtr += DataLength;
      rqptr->CacheContentCount -= DataLength;

      if (Debug)
         fprintf (stdout, "%d %d %d\n",
                  DataPtr, DataLength, rqptr->CacheContentCount);

      NetWrite (rqptr, &CacheNext, DataPtr, DataLength);
   }
   else
   {
      DataPtr = rqptr->CacheContentPtr;
      DataLength = rqptr->CacheContentCount;
      rqptr->CacheContentCount = 0;

      if (Debug)
         fprintf (stdout, "%d %d %d\n",
                  DataPtr, DataLength, rqptr->CacheContentCount);

      NetWrite (rqptr, &CacheEnd, DataPtr, DataLength);
      return;
   }
}

/*****************************************************************************/
/*
End of transfer to client using cached contents.
*/

CacheEnd (struct RequestStruct *rqptr)

{
   register struct CacheStruct  *captr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheEnd()\n");

   captr = rqptr->CachePtr;

   /* this cache entry is no longer in use (if now zero) */
   if (captr->InUseCount) captr->InUseCount--;

   if (captr->Purge && !captr->InUseCount)
      CachePurgeEntry (captr, captr->PurgeCompletely);

   /* this cache entry is no longer associated with this request */
   rqptr->CachePtr = NULL;

   RequestEnd (rqptr);
} 

/*****************************************************************************/
/*
Get the file's revision date and it's current size.
Return these in 'rqptr->CachePtr->ValidateModifiedBinaryTime',
'rqptr->CachePtr->ValidateEndOfFileVbn' and
rqptr->CachePtr->ValidateFirstFreeByte'.

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's
Reference Manual", and is probably as fast as we can get for this type of file
system functionality!
*/ 

int CacheAcpInfo (struct RequestStruct *rqptr)

{
   static unsigned long  AtrRdt [2];
   static $DESCRIPTOR (DeviceDsc, "");
   static struct fibdef  SearchFib;

   static struct {
      unsigned char  RecordType;
      unsigned char  OfNoInterest1;
      unsigned short  RecordSize;
      unsigned char  OfNoInterest2 [4];
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned char   OfNoInterest3 [18];
   } AtrRecord;

   static struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { 0, 0, 0 }
   };

   register struct CacheStruct  *captr;

   int  status;
   unsigned short  AcpChannel;
   unsigned long  EndOfFileVbn;

   struct fibdef  FileFib;

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileFibAcpDsc,
     FileNameAcpDsc;
     
   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "CacheAcpInfo() |%s|\n", rqptr->CachePtr->FileName);

   captr = rqptr->CachePtr;

   /* initialize return values */
   memset (captr->ValidateModifiedBinaryTime, 0, 8);

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$a_pointer = captr->DeviceNamePtr;
   DeviceDsc.dsc$w_length = captr->DeviceNameLength;
   if (Debug)
      fprintf (stdout, "device |%*.*s|\n",
               DeviceDsc.dsc$w_length, DeviceDsc.dsc$w_length,
               DeviceDsc.dsc$a_pointer);

   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MapVmsPath (captr->FileName);
      rqptr->ErrorHiddenTextPtr = captr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;

   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   memcpy (&FileFib.fib$w_did, &captr->FileNam_did, 6);
#else
   memcpy (&FileFib.fib$r_did_overlay.fib$w_did, &captr->FileNam_did, 6);
#endif

   FileNameAcpDsc.Address = captr->NameTypePtr;
   FileNameAcpDsc.Length = captr->NameTypeLength;
   if (Debug)
      fprintf (stdout, "name |%*.*s|\n",
               FileNameAcpDsc.Length, FileNameAcpDsc.Length,
               FileNameAcpDsc.Address);

   /* read the file attributes */
   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
               status, AcpIOsb.Status);

   sys$dassgn (AcpChannel);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (status == SS$_NOSUCHFILE) return (RMS$_FNF);
   if (VMSnok (status)) return (status);

   memcpy (captr->ValidateModifiedBinaryTime, &AtrRdt, 8);

   captr->ValidateEndOfFileVbn = AtrRecord.EndOfFileVbnLo +
                                 (AtrRecord.EndOfFileVbnHi << 16);
   captr->ValidateFirstFreeByte = AtrRecord.FirstFreeByte;

   if (Debug)
   {
      char  GmtRdt [48],
            VmsRdt [32];
      $DESCRIPTOR (VmsRdtFaoDsc, "!%D\0");
      $DESCRIPTOR (VmsRdtDsc, VmsRdt);

      sys$fao (&VmsRdtFaoDsc, 0, &VmsRdtDsc, captr->ValidateModifiedBinaryTime);
      HttpGmTimeString (GmtRdt, captr->ValidateModifiedBinaryTime);
      fprintf (stdout, "|%s|%s| VBN: %d ffb: %d\n",
               VmsRdt, GmtRdt, 
               captr->ValidateEndOfFileVbn, captr->ValidateFirstFreeByte);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Find/create a cache structure ready to load a cache data buffer from a file. It
is entirely possible, given that the maximum number of cache entries has been
reached and all are currently in use (either for data transfer or being
loaded), that there will be no cache entry available for this request to use.
In this case the 'rqptr->CachePtr' will remain NULL indicating that.

NOTE: as with CacheSearch(), the path passed to this function MUST be the
mapped path, not the request path. With conditional mapping identical request
paths may be mapped to completely different virtual paths.

'SizeInBytes' is the APPROXIMATE and MAXIMUM size required for data storage.
The size of a VARIABLE record file can be used because the actual data content
is smaller (when the 16 bits for the record length is turned into 8 bits for a
new-line character!). Hence, with a VARIABLE record file the content length is
invariably a little less that the reported file size. The actual size of the
data in the cache entry is represented by 'ContentLength'. 
*/ 

CacheLoadBegin
(
struct RequestStruct *rqptr,
int SizeInBytes
)
{
   static unsigned long  Subx2 = 2,
                         EdivIntoSeconds = 10000000;

   register struct CacheStruct  *captr, *lcaptr;
   register struct ListEntryStruct  *leptr;

   int  status,
        PathLength;
   unsigned long  Duration,
                  Remainder,
                  Result,
                  Seconds;
   unsigned long  ResultTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "CacheLoadBegin() |%s| %d bytes\n",
               rqptr->CacheMappedPathPtr, SizeInBytes);

   if (!CacheHashTableInitialised) CacheInit (false);

   rqptr->CachePtr = captr = NULL;

   /* if it's larger than the maximum allowed then just forget it */
   if ((SizeInBytes >> 10) > CacheFileKBytesMax) return;

   /* if larger than the storage space available then also just forget it */
   if (rqptr->CacheMappedPathLength > sizeof(captr->Path)-1) return;

   /* calculate exactly how much data buffer we're going to use */
   if (SizeInBytes <= 512)
      SizeInBytes = 512;
   else
   {
      /* divide by whatever to get cache chunk */
      SizeInBytes = ((SizeInBytes / CacheChunkInBytes) + 1) * CacheChunkInBytes;
   }

   /*
      If we've run out of cache entries, or
      if we've run out of cache memory, or
      if there is an invalidated entry on the end of the list
         that is not otherwise unavailable.
   */
   if (CacheEntryCount >= CacheEntriesMax ||
       (CacheMemoryInUse + SizeInBytes) >> 10 > CacheTotalKBytesMax ||
       (CacheList.TailPtr != NULL &&
        !((struct CacheStruct*)CacheList.TailPtr)->CacheValid &&
        !((struct CacheStruct*)CacheList.TailPtr)->InUseCount &&
        !((struct CacheStruct*)CacheList.TailPtr)->CacheLoading))
   {
      /***********************/
      /* reuse a cache entry */
      /***********************/

      /* process the request list from least to most recent */
      for (leptr = CacheList.TailPtr; leptr != NULL; leptr = leptr->PrevPtr)
      {
         captr = (struct CacheStruct*)leptr;

         if (Debug)
            fprintf (stdout, "%d |%s| %d |%s| %d %d %d %d\n",
                     captr->PathLength, captr->Path,
                     captr->FileNameLength, captr->FileName,
                     captr->CacheSize, captr->ContentLength,
                     captr->CacheValid, captr->InUseCount);

         /* if it's in use then just continue */
         if (captr->InUseCount || captr->CacheLoading) continue;

         if (captr->CacheValid &&
             CacheFrequentHits &&
             captr->HitCount > CacheFrequentHits)
         {
            /* frequently hit, but has it been hit recently? */
            status = lib$subx (&rqptr->BinaryTime, &captr->HitBinaryTime,
                               &ResultTime, &Subx2);
            if (Debug) fprintf (stdout, "lib$subx() %%X%08.08X\n", status);
            status = lib$ediv (&EdivIntoSeconds, &ResultTime,
                               &Seconds, &Remainder);
            if (Debug) fprintf (stdout, "lib$ediv() %%X%08.08X\n", status);
            /* continue if it has been hit within the specified interval */
            if (Debug) fprintf (stdout, "%d\n", Seconds);
            if (Seconds <= CacheFrequentSeconds) continue;
         }

         /* note the number of entries loaded but never subsequently hit */
         if (captr->CacheValid && !captr->HitCount) CacheNoHitsCount++;

         /* no longer valid data */
         CacheEntryNotValid (captr);

         /* move it to the end of the cache list ready for reuse */
         ListMoveTail (&CacheList, captr);

         if (Debug) fprintf (stdout, "REUSE %d\n", captr);
         break;
      }

      /* if we got to the end of the list then we couldn't have found one! */
      if (leptr == NULL) captr = NULL;
   }

   if (captr == NULL &&
       CacheEntryCount < CacheEntriesMax)
   {
      /***************************************/
      /* allocate memory for new cache entry */
      /***************************************/

      captr = VmGetCache (sizeof(struct CacheStruct));

      CacheMemoryInUse += sizeof(struct CacheStruct);
      CacheEntryCount++;
      if (Debug)
         fprintf (stdout, "NEW %d (%d bytes)\n",
                  captr, sizeof(struct CacheStruct));

      /* add it to the list */
      ListAddTail (&CacheList, captr);
   }

   if (Debug) fprintf (stdout, "%d captr: %d\n", CacheEntryCount, captr);

   /**************************/
   /* cache entry available? */
   /**************************/

   /* couldn't reuse an entry, couldn't create a new one, hence no caching */
   if (captr == NULL) return;

   /************************/
   /* allocate data buffer */
   /************************/

   if (captr->ContentPtr != NULL &&
       captr->CacheSize != SizeInBytes)
   {
      if (Debug)
         fprintf (stdout, "free captr->ContentPtr: %d %d\n",
                  captr->ContentPtr, captr->CacheSize);

      VmFreeCache (captr->ContentPtr);
      CacheMemoryInUse -= captr->CacheSize;
      captr->ContentPtr = NULL;
      captr->CacheSize = captr->ContentBufferUsedCount = 0;
      captr->CacheScavenged = captr->CacheValid = false;
   }

   if (captr->ContentPtr == NULL)
   {
      /* if we haven't got sufficient cache memory then free some entries */
      if ((CacheMemoryInUse + SizeInBytes) >> 10 > CacheTotalKBytesMax)
      {
         /*******************/
         /* scavenge memory */
         /*******************/

         /* mark this one so it won't be touched */
         captr->CacheLoading = true;

         /* process the request list from least to most recent */
         for (leptr = CacheList.TailPtr; leptr != NULL; leptr = leptr->PrevPtr)
         {
            /* remember, use a separate pointer or we get very confused :^) */
            lcaptr = (struct CacheStruct*)leptr;

            if (Debug)
               fprintf (stdout, "%d |%s| %d |%s| %d %d %d %d\n",
                        lcaptr->PathLength, lcaptr->Path,
                        lcaptr->FileNameLength, lcaptr->FileName,
                        lcaptr->CacheSize, lcaptr->ContentLength,
                        lcaptr->CacheValid, lcaptr->InUseCount);

            /* if it's in use then just continue */
            if (lcaptr->InUseCount || lcaptr->CacheLoading) continue;

            /* note the number of entries loaded but never subsequently hit */
            if (lcaptr->CacheValid && !lcaptr->HitCount) CacheNoHitsCount++;

            /* no longer valid data */
            CacheEntryNotValid (lcaptr);
            lcaptr->CacheScavenged = true;

            if (lcaptr->ContentPtr != NULL)
            {
               if (Debug)
                  fprintf (stdout, "free lcaptr->ContentPtr: %d %d\n",
                           lcaptr->ContentPtr, lcaptr->CacheSize);

               VmFreeCache (lcaptr->ContentPtr);
               CacheMemoryInUse -= lcaptr->CacheSize;
               lcaptr->ContentPtr = NULL;
               lcaptr->CacheSize = lcaptr->ContentBufferUsedCount = 0;
            }

            /* if we've scavenged enough then break */
            if ((CacheMemoryInUse + SizeInBytes) >> 10 < CacheTotalKBytesMax)
               break;
         }

         /* if we still haven't got enough memory then forget it! */
         if ((CacheMemoryInUse + SizeInBytes) >> 10 > CacheTotalKBytesMax)
            ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
      }

      /* allocate memory for the cache data */
      captr->ContentPtr = VmGetCache (captr->CacheSize = SizeInBytes);

      CacheMemoryInUse += captr->CacheSize;
   }

   captr->ContentBufferUsedCount++;

   if (Debug)
      fprintf (stdout, "%d %d\n", captr->ContentPtr, SizeInBytes);

   /*********************/
   /* ready for loading */
   /*********************/

   /*
      By setting the path and hash table this entry can still be hit
      even though the data is still being loaded and is not yet valid.
      This will prevent the same path having concurrent, multiple loads.
   */

   captr->HashValue = rqptr->CacheHashValue;
   if (CacheHashTable[captr->HashValue] == NULL)
   {
      /* set hash table index */
      CacheHashTable[captr->HashValue] = captr;
      captr->HashCollisionPrevPtr = captr->HashCollisionNextPtr = NULL;
   }
   else
   {
      /* add to head of hash-collision list */
      lcaptr = CacheHashTable[captr->HashValue];
      lcaptr->HashCollisionPrevPtr = captr;
      captr->HashCollisionPrevPtr = NULL;
      captr->HashCollisionNextPtr = lcaptr;
      CacheHashTable[captr->HashValue] = captr;
   }
   memcpy (captr->Path, rqptr->CacheMappedPathPtr,
           (captr->PathLength = rqptr->CacheMappedPathLength)+1);
   captr->ContentLength = captr->HitCount = 0;
   captr->InUseCount++;
   captr->CacheScavenged = captr->CacheValid = captr->CacheLoadValid = false;
   captr->CacheLoading = true;

   /* point the request at this cache entry */
   rqptr->CachePtr = captr;
}

/*****************************************************************************/
/*
End loading a cache data buffer from a file.  Store the path and file name,
along with information used by CacheAcpInfo() for revalidating the file data.
*/ 

CacheLoadEnd
(
struct RequestStruct *rqptr,
struct FileTaskStruct *ftkptr
)
{
   register struct CacheStruct  *captr;

   int  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheLoadEnd() %d\n", rqptr);

   captr = rqptr->CachePtr;

   /* remember, we remove the version delimiter and number! */
   if (captr->CacheLoadValid &&
       ftkptr->FileNam.nam$b_esl <= sizeof(captr->FileName))
      captr->CacheValid = true;

   if (captr->CacheValid)
   {
      /* re-copy the expanded file name excluding any version */
      captr->FileNameLength = Length = ftkptr->FileNam.nam$b_esl -
                                       ftkptr->FileNam.nam$b_ver;
      memcpy (captr->FileName, ftkptr->FileNam.nam$l_esa, Length);
      captr->FileName[Length] = '\0';

      /* length and pointer "dev:" of file name (for CacheAcpInfo()) */
      captr->DeviceNamePtr = captr->FileName +
                             ftkptr->FileNam.nam$b_node;
      captr->DeviceNameLength = ftkptr->FileNam.nam$b_dev;

      /* length and pointer "name.type;n" (for CacheAcpInfo()) */
      captr->NameTypePtr = captr->FileName +
                           ftkptr->FileNam.nam$b_node +
                           ftkptr->FileNam.nam$b_dev +
                           ftkptr->FileNam.nam$b_dir;
      captr->NameTypeLength = ftkptr->FileNam.nam$b_name +
                              ftkptr->FileNam.nam$b_type;

      /* 6 byte directory id from NAM block, for use in CacheAcpInfo() */
      memcpy (&captr->FileNam_did, &ftkptr->FileNam.nam$w_did, 6);

      /* the file size data */
      captr->EndOfFileVbn = ftkptr->EndOfFileVbn;
      captr->FirstFreeByte = ftkptr->FirstFreeByte;

      /* quadword time file was loaded, last modified and validated */
      memcpy (&captr->LoadBinaryTime, &rqptr->BinaryTime, 8);
      memcpy (&captr->ModifiedBinaryTime, &ftkptr->ModifiedBinaryTime, 8);
      memcpy (&captr->ValidateBinaryTime, &rqptr->BinaryTime, 8);

      captr->ContentTypePtr = ftkptr->ContentTypePtr;
      captr->ContentLength = rqptr->CacheContentPtr - captr->ContentPtr;

      if (Debug)
         fprintf (stdout, "%d %d %d |%s|\n",
                  captr->HashValue, captr->ContentLength,
                  captr->FileNameLength, captr->FileName);

      Accounting.CacheLoadCount++;
      CacheLoadCount++;
      CacheHits0++;

      /* move entry to the head of the cache list */
      ListMoveHead (&CacheList, captr);
   }
   else
   {
      /* remove any vestiges of the failed load */
      CacheEntryNotValid (captr);

      /* move it to the end of the cache list ready for reuse */
      ListMoveTail (&CacheList, captr);
   }

   /* obviously no longer loading! */
   captr->CacheLoading = false;

   /* this cache entry is no longer in use (if now zero) */
   if (captr->InUseCount) captr->InUseCount--;

   /* this cache entry is no longer associated this request */
   rqptr->CachePtr = NULL;
}

/*****************************************************************************/
/*
Scan through the cache list.  If a cache entry is currently not in use then
free the data memory associated with it.  If purge completely then also remove
the entry from the list and free it's memory.  If the entry is currently in use
then mark it for purge and if necessary for complete removal.
*/

CachePurge
(
boolean Completely,
int *PurgeCountPtr,
int *MarkedForPurgeCountPtr
)
{
   register struct CacheStruct  *captr;
   register struct ListEntryStruct  *leptr;

   int  PurgeCount,
        MarkedForPurgeCount;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CachePurge()\n");

   CacheZeroCounters ();

   PurgeCount = MarkedForPurgeCount = 0;

   /* do it backwards! seeing they are pushed to the tail of the list */
   leptr = CacheList.TailPtr;
   while (leptr != NULL)
   {
      captr = (struct CacheStruct*)leptr;

      /* now, before stuffing around with the entry get the next one */
      leptr = leptr->PrevPtr;

      if (Debug)
         fprintf (stdout, "%d |%s| %d %d %d %d\n",
                  captr->FileNameLength, captr->FileName,
                  captr->CacheSize, captr->ContentLength,
                  captr->CacheValid, captr->InUseCount);

      if (captr->InUseCount)
      {
         if (Completely)
            captr->Purge = captr->PurgeCompletely = true;
         else
            captr->Purge = true;
         MarkedForPurgeCount++;
      }
      else
      {
         if (captr->CacheSize)
         {
            CachePurgeEntry (captr, Completely);
            PurgeCount++;
         }
      }
   }

   if (PurgeCountPtr != NULL) *PurgeCountPtr = PurgeCount;
   if (MarkedForPurgeCountPtr != NULL)
      *MarkedForPurgeCountPtr = MarkedForPurgeCount;
} 

/*****************************************************************************/
/*
Purge a cache entry, either just the data, or completely from the cache list.
*/

CachePurgeEntry
(
struct CacheStruct *captr,
boolean Completely
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CachePurgeEntry()\n");

   if (captr->InUseCount)
   {
      if (Completely)
         captr->Purge = captr->PurgeCompletely = true;
      else
         captr->Purge = true;
      return;
   }

   if (captr->ContentPtr != NULL)
   {
      VmFreeCache (captr->ContentPtr);
      captr->ContentPtr = NULL;
      CacheMemoryInUse -= captr->CacheSize;
   }

   if (Completely)
   {
      CacheEntryNotValid (captr);
      ListRemove (&CacheList, captr);
      VmFreeCache (captr);
      CacheMemoryInUse -= sizeof(struct CacheStruct);
   }
   else
   {
      CacheEntryNotValid (captr);
      /* move it to the end of the cache list ready for reuse */
      ListMoveTail (&CacheList, captr);
      captr->ContentLength = captr->CacheSize = 0;
      captr->CacheValid = captr->Purge = false;
   }
} 

/*****************************************************************************/
/*
Remove from hash table and any associated collision list, and move the end of
LRU list.
*/ 

#ifdef __DECC
#pragma inline(CacheEntryNotValid)
#endif

CacheEntryNotValid (struct CacheStruct *captr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheEntryNotValid() %d\n", captr);

   captr->CacheValid = false;

   if (CacheHashTable[captr->HashValue] == captr)
   {
      /* must be at the head of any collision list */
      CacheHashTable[captr->HashValue] = captr->HashCollisionNextPtr;
      if (captr->HashCollisionNextPtr != NULL)
         captr->HashCollisionNextPtr->HashCollisionPrevPtr =
            captr->HashCollisionPrevPtr;
   }
   else
   {
      /* if somewhere along the collision list */
      if (captr->HashCollisionPrevPtr != NULL)
         captr->HashCollisionPrevPtr->HashCollisionNextPtr =
            captr->HashCollisionNextPtr;
   }

   captr->HashValue = 0;
}

/*****************************************************************************/
/*
Return a report on cache usage.  This function blocks while executing.
*/ 

int CacheReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Cache Report</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Cache Report</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Caching</TH>\
<TD ALIGN=right>!AZ</TD></TR>\n\
<TR><TH>Cache Memory</TH>\
<TD ALIGN=right>!UL / !UL</TD>\
<TD><FONT SIZE=-1>kBytes (cur/max)</FONT></TD></TR>\n\
<TR><TH>Entries</TH>\
<TD ALIGN=right>!UL / !UL</TD>\
<TD><FONT SIZE=-1>(cur/max)</FONT></TD></TR>\n\
<TR><TH>Hash Table</TH>\
<TD ALIGN=right>!UL</TD>\
<TD><FONT SIZE=-1>entries</FONT></TD></TR>\n\
<TR><TH>Max File Size</TH>\
<TD ALIGN=right>!UL</TD><TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Chunk Size</TH>\
<TD ALIGN=right>!UL</TD><TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Validate</TH>\
<TD ALIGN=right>!UL</TD><TD><FONT SIZE=-1>seconds</FONT></TD></TR>\n\
<TR><TH>Frequent</TH>\
<TD COLSPAN=2>!UL <FONT SIZE=-1>hits, last within</FONT> \
!UL <FONT SIZE=-1>seconds</FONT></TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=6>Activity</TH></TR>\n\
<TR><TH ROWSPAN=3>Search</TH>\
<TH COLSPAN=5>Hash Table</TH></TR>\n\
<TR><TH ROWSPAN=2>Hit</TH><TH ROWSPAN=2>Miss</TH>\
<TH COLSPAN=3>Collision</TH></TR>\n\
<TR><TH>Total</TH><TH>Hit</TH><TH>Miss</TH></TR>\n\
<TR ALIGN=right><TD ROWSPAN=2>!UL</TD>\
<TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD><TD>!UL</TD></TR>\n\
<TR ALIGN=right>\
<TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=8>Files</TH></TR>\n\
<TR><TH>Loads</TH><TH>Not Hit</TH><TH></TH>\
<TH>Hits</TH><TH></TH>\
<TH>1-9</TH><TH>10-99</TH><TH>100+</TH></TR>\n\
<TR ALIGN=right><TD>!UL</TD><TD>!UL</TD><TD></TD>\
<TD>!UL</TD><TD></TD>\
<TD>!UL</TD><TD>!UL</TD><TD>!UL</TD></TR>\n\
<TR ALIGN=right><TD></TD><TD>!UL%</TD><TD></TD>\
<TD>!UL%</TD><TD></TD>\
<TD>!UL%</TD><TD>!UL%</TD><TD>!UL%</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=12>Entries</TH></TR>\
<TR><TH ROWSPAN=3></TH><TH COLSPAN=11>Path</TH></TR>\
<TR><TH COLSPAN=11>File Name</TH></TR>\
<TR>\
<TH>Size</TH><TH>Length</TH><TH>In-Use</TH><TH>Hash</TH><TH>Collsn</TH>\
<TH>Revised</TH><TH COLSPAN=2>Validated</TH>\
<TH>Loaded</TH><TH COLSPAN=2>Hit&nbsp;/&nbsp;304</TH>\
</TR>\n\
<TR></TR>\n");

   static $DESCRIPTOR (CacheFaoDsc,
"<TR><TD ROWSPAN=3 ALIGN=right>!UL</TD>\
<TD COLSPAN=11><TT><A HREF=\"!AZ\">!AZ</A></TT></TR>\n\
<TR><TD COLSPAN=11><TT>!AZ</TT></TR>\n\
<TR>\
<TD ALIGN=right>!UL</TD><TD ALIGN=right>!UL</TD><TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL</TD><TD ALIGN=right>!UL</TD>\
<TD><FONT SIZE=1>!AZ</FONT></TD>\
<TD>!AZ!AZ!AZ</TD><TD ALIGN=right>!UL</TD>\
<TD><FONT SIZE=1>!AZ</FONT></TD>\
<TD><FONT SIZE=1>!AZ</FONT></TD>\
<TD ALIGN=right>!UL&nbsp;/&nbsp;!UL</TD>\
</TR>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register int  cnt;
   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct CacheStruct  *captr;
   register struct ListEntryStruct  *leptr;

   int  status,
        Count,
        HashCollisionListLength;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char  Buffer [4096],
         HitString [48],
         LoadString [48],
         RevisedString [48],
         ValidatedString [48];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheReport()\n");

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   rqptr->ResponseStatusCode = 200;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   if (CacheEnabled)
      *vecptr++ = "[enabled]";
   else
      *vecptr++ = "<FONT COLOR=\"#ff000\">[disabled]</FONT>";

   *vecptr++ = CacheMemoryInUse >> 10;
   *vecptr++ = CacheTotalKBytesMax;
   *vecptr++ = CacheEntryCount;
   *vecptr++ = CacheEntriesMax;
   *vecptr++ = CacheHashTableEntries;
   *vecptr++ = CacheFileKBytesMax;
   *vecptr++ = CacheChunkInBytes >> 10;
   *vecptr++ = CacheValidateSeconds;
   *vecptr++ = CacheFrequentHits;
   *vecptr++ = CacheFrequentSeconds;

   *vecptr++ = CacheHashTableCount;
   *vecptr++ = CacheHashTableHitCount;
   *vecptr++ = CacheHashTableMissCount;
   *vecptr++ = CacheHashTableCollsnCount;
   *vecptr++ = CacheHashTableCollsnHitCount;
   *vecptr++ = CacheHashTableCollsnMissCount;
   if (CacheHashTableCount)
   {
      *vecptr++ = CacheHashTableHitCount * 100 / CacheHashTableCount;
      *vecptr++ = CacheHashTableMissCount * 100 / CacheHashTableCount;
      *vecptr++ = CacheHashTableCollsnCount * 100 / CacheHashTableCount;
      *vecptr++ = CacheHashTableCollsnHitCount * 100 / CacheHashTableCount;
      *vecptr++ = CacheHashTableCollsnMissCount * 100 / CacheHashTableCount;
   }
   else
   {
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = 0;
   }

   *vecptr++ = CacheLoadCount;
   *vecptr++ = CacheHits0;
   *vecptr++ = CacheHitCount;
   *vecptr++ = CacheHits10;
   *vecptr++ = CacheHits100;
   *vecptr++ = CacheHits100plus;
   if (CacheLoadCount)
   {
      *vecptr++ = CacheHits0 * 100 / CacheLoadCount;
      *vecptr++ = CacheHitCount * 100 / CacheLoadCount;
   }
   else
   {
      *vecptr++ = 0;
      *vecptr++ = 0;
   }
   if (CacheHitCount)
   {
      *vecptr++ = CacheHits10 * 100 / CacheHitCount;
      *vecptr++ = CacheHits100 * 100 / CacheHitCount;
      *vecptr++ = CacheHits100plus * 100 / CacheHitCount;
   }
   else
   {
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = 0;
   }

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /**************/
   /* cache list */
   /**************/

   Count = 0;

   if (Debug)
      fprintf (stdout, "Head %d\nTail %d\n",
               CacheList.HeadPtr, CacheList.TailPtr);

   /* process the request list from most to least recently hit */
   for (leptr = CacheList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      captr = (struct cacheStruct*)leptr;

      if (Debug)
         fprintf (stdout, "%d <- %d -> %d (%d)n",
                  leptr->PrevPtr, leptr, leptr->NextPtr, captr);

      HashCollisionListLength = 0;
      /* count further down the list */
      while (captr->HashCollisionNextPtr != NULL)
      {
         captr = captr->HashCollisionNextPtr;
         HashCollisionListLength++;
      }
      captr = (struct cacheStruct*)leptr;
      /* count further up the list */
      while (captr->HashCollisionPrevPtr != NULL)
      {
         captr = captr->HashCollisionPrevPtr;
         HashCollisionListLength++;
      }
      if (HashCollisionListLength) HashCollisionListLength++;
      captr = (struct cacheStruct*)leptr;

      strcpy (HitString, DayDateTime (&captr->HitBinaryTime, 23));
      strcpy (LoadString, DayDateTime (&captr->LoadBinaryTime, 23));
      strcpy (RevisedString, DayDateTime (&captr->ModifiedBinaryTime, 23));

      vecptr = FaoVector;
      *vecptr++ = ++Count;
      *vecptr++ = captr->Path;
      *vecptr++ = captr->Path;
      *vecptr++ = captr->FileName;
      *vecptr++ = captr->CacheSize;
      *vecptr++ = captr->ContentLength;
      *vecptr++ = captr->InUseCount;
      *vecptr++ = captr->HashValue;
      *vecptr++ = HashCollisionListLength;
      *vecptr++ = RevisedString;
      if (captr->CacheValid)
      {
         strcpy (ValidatedString, DayDateTime (&captr->ValidateBinaryTime, 23));
         *vecptr++ = "<FONT SIZE=1>";
         *vecptr++ = ValidatedString;
         *vecptr++ = "</FONT>";
      }
      else
      if (captr->CacheScavenged)
      {
         *vecptr++ = "<B>SCAVENGED</B>";
         *vecptr++ = "";
         *vecptr++ = "";
      }
      else
      if (captr->CacheLoading)
      {
         *vecptr++ = "<B>LOADING</B>";
         *vecptr++ = "";
         *vecptr++ = "";
      }
      else
      {
         *vecptr++ = "<B>INVALID</B>";
         *vecptr++ = "";
         *vecptr++ = "";
      }
      *vecptr++ = captr->ValidatedCount;
      *vecptr++ = LoadString;
      *vecptr++ = HitString;
      *vecptr++ = captr->HitCount;
      *vecptr++ = captr->HitNotModifiedCount;

      status = sys$faol (&CacheFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   NetWriteBuffered (rqptr, NextTaskFunction,
                     EndPageFaoDsc.dsc$a_pointer,
                     EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/

