/*****************************************************************************/
/*
                                   CGI.c

Provide generalized CGI scripting support.

Creates a buffer containing either, the DCL commands to create CGI variables at
the VMS command line (e.g. for standard CGI scripts), or a stream of
'name=value' pairs for the CGI variables (e.g. for CGIplus scripts). 

The buffer is dynamically allocated and contains a series of sequential records
comprising a word (16 bits) with the length of a following (varying)
null-terminated string (including the null character).  The end of the series
is indicated by a zero length record.

This buffer is scanned from from first to last and the strings passed to the
script 'input' stream as appropriate.


CGI VARIABLES
-------------
Most of these CGI variable names are those supported by the INTERNET-DRAFT
authored by D.Robinson (drtr@ast.cam.ac.uk), 8 January 1996, plus some
"convenience" variables, breaking the query string into its components (KEY_,
FORM_, etc.) The CGI symbols (CERN/VMS-HTTPd-like DCL symbols instead of Unix
environment variables) are created by the SYS$COMMAND stream of each
subprocess before the script DCL procedure is invoked. By default each
variable name is prefixed by "WWW_" (similar to CERN HTTPd), although this can
be modified at the command line when starting the server. CGI variable
(symbol) values are limited in size to approximately 1024 characters. 


  o  WWW_AUTH_GROUP ........... path authorization group (or empty)    [ext]
  o  WWW_AUTH_REALM ........... authentication realm (or empty)        [ext]
  o  WWW_AUTH_TYPE ............ authentication type (or empty)         [std]

  o  WWW_CONTENT_LENGTH ....... "Content-Length:" from header          [std]
  o  WWW_CONTENT_TYPE ......... "Content-Type:" from header            [std]

  o  WWW_FORM_field ........... query "&" separated form elements      [ext]

  o  WWW_GATEWAY_INTERFACE .... "CGI/1.1"                              [std]

  o  WWW_HTTP_ACCEPT .......... list of browser-accepted content types [std]
  o  WWW_HTTP_ACCEPT_CHARSET .. list of browser-accepted character set [std]
  o  WWW_HTTP_ACCEPT_LANGUAGE . list of browser-accepted languages     [std]
  o  WWW_HTTP_AUTHORIZATION ... allows a script to authenticate        [std]
  o  WWW_HTTP_COOKIE .......... any cookie sent by the client          [std]
  o  WWW_HTTP_FORWARDED ....... list of proxy/gateway hosts            [std]
  o  WWW_HTTP_HOST ............ destination host name/port             [std]
  o  WWW_HTTP_IF_NOT_MODIFIED . GMT time string                        [std]
  o  WWW_HTTP_PRAGMA .......... any pragma directive of header         [std]
  o  WWW_HTTP_REFERER ......... source document URL for this request   [std]
  o  WWW_HTTP_USER_AGENT ...... client/browser identification string   [std]

  o  WWW_KEY_n ................ query string "+" separated elements    [ext]
  o  WWW_KEY_COUNT ............ number of "+" separated elements       [ext]

  o  WWW_PATH_INFO ............ virtual path of data requested in URL  [std]
  o  WWW_PATH_TRANSLATED ...... VMS file path of data requested in URL [std]

  o  WWW_QUERY_STRING ......... string following "?" in URL            [std]

  o  WWW_REMOTE_ADDR .......... IP host address of HTTP client         [std]
  o  WWW_REMOTE_HOST .......... IP host name of HTTP client            [std]
  o  WWW_REMOTE_USER .......... authenticated username (or empty)      [std]
  o  WWW_REQUEST_METHOD ....... "GET", "PUT", etc.                     [std]
  o  WWW_REQUEST_SCHEME ....... "http:" or "https:"                    [ext]
  o  WWW_REQUEST_TIME_GMT ..... request GMT time                       [ext]
  o  WWW_REQUEST_TIME_LOCAL ... request local time                     [ext]

  o  WWW_SCRIPT_NAME .......... name of script (e.g. "/query")         [std]
  o  WWW_SERVER_GMT ........... offset from GMT time (e.g. "+09:30)    [ext]
  o  WWW_SERVER_NAME .......... IP host name of server system          [std]
  o  WWW_SERVER_PROTOCOL ...... HTTP protocol version ("HTTP/1.0")     [std]
  o  WWW_SERVER_PORT .......... IP port request was received on        [std]
  o  WWW_SERVER_SOFTWARE ...... software ID of the HTTPD daemon        [std]


VERSION HISTORY
---------------
28-APR-98  MGD  CGI variable memory allocation changed in support of SSI
03-APR-98  MGD  bugfix; extra quotes generated in form field value
19-MAR-98  MGD  suppress 'Authorization:' field unless "external" authorization
06-DEC-97  MGD  functionality unbundled from DCL.c, generalized for version 5
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */

#include <descrip.h>
#include <dvidef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "cgi.h"
#include "msg.h"
#include "support.h"
#include "vm.h"

/***********/
/* defines */
/***********/

#define DCL_LINE_SIZE_MAX 255
#define CGI_BUFFER_INITIAL 2048
#define CGI_BUFFER_INCREMENT 512

/******************/
/* global storage */
/******************/

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  ServerPort;
extern char  DclCgiVariablePrefix[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  TimeGmtString[];
extern struct ConfigStruct Config;
extern struct MsgStruct Msgs;

/*****************************************************************************/
/*
Creates a buffer containing a series of null-terminated strings, terminated by
a null-string (two successive null characters).  Each of the strings contains
either DCL command(s) to create a DCL symbol (CGI variable) using the CLI, or
an equate separated 'name=value' pair used for CGIplus variable streams, etc.
*/ 

CgiGenerateVariables
(
struct RequestStruct *rqptr,
int VarType
)
{
   static $DESCRIPTOR (NumberFaoDsc, "!UL\0");

   register char  c;
   register char  *cptr, *sptr, *zptr;

   int  status,
        Count;
   char  FormFieldName [256],
         LocalDateTime [48],
         String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiGenerateVariables()\n");

   if (rqptr->CgiBufferLength)
   {
      /* shouldn't happen! */
      if (rqptr->CgiBufferPtr != NULL)
         VmFreeFromHeap (rqptr, rqptr->CgiBufferPtr); 
      rqptr->CgiBufferLength = 0;
      rqptr->CgiBufferPtr = rqptr->CgiBufferCurrentPtr = NULL;
   }

   CgiVariableBufferMemory (rqptr);

   HttpLocalTimeString (LocalDateTime, &rqptr->BinaryTime);

   if (VMSnok (status =
       CgiVariable (rqptr, "AUTH_GROUP", rqptr->AuthGroupPtr, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "AUTH_REALM", rqptr->AuthRealmPtr, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "AUTH_TYPE", rqptr->AuthType, VarType)))
      return (status);

   sys$fao (&NumberFaoDsc, 0, &StringDsc, rqptr->ContentLength);
   if (VMSnok (status =
       CgiVariable (rqptr, "CONTENT_LENGTH", String, VarType)))
      return (status);

   if (rqptr->HttpContentTypePtr != NULL)
   {
      if (VMSnok (status =
          CgiVariable (rqptr, "CONTENT_TYPE",
                       rqptr->HttpContentTypePtr, VarType)))
         return (status);
   }
   else
   if (rqptr->ContentType.ContentTypePtr != NULL)
   {
      if (VMSnok (status =
          CgiVariable (rqptr, "CONTENT_TYPE",
                       rqptr->ContentType.ContentTypePtr, VarType)))
         return (status);
   }
   else
   {
      if (VMSnok (status =
          CgiVariable (rqptr, "CONTENT_TYPE", "", VarType)))
         return (status);
   }

   if (VMSnok (status =
       CgiVariable (rqptr, "GATEWAY_INTERFACE", "CGI/1.1", VarType)))
      return (status);

   if (rqptr->HttpAcceptPtr != NULL)
      if (VMSnok (status =
       CgiVariable (rqptr, "HTTP_ACCEPT", rqptr->HttpAcceptPtr, VarType)))
      return (status);

   if (rqptr->HttpAcceptCharsetPtr != NULL)
      if (VMSnok (status =
       CgiVariable (rqptr, "HTTP_ACCEPT_CHARSET",
                    rqptr->HttpAcceptCharsetPtr, VarType)))
      return (status);

   if (rqptr->HttpAcceptLangPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_ACCEPT_LANGUAGE",
                       rqptr->HttpAcceptLangPtr, VarType)))
         return (status);

   if (rqptr->AuthExternal && rqptr->HttpAuthorizationPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_AUTHORIZATION",
                       rqptr->HttpAuthorizationPtr, VarType)))
         return (status);

   if (rqptr->HttpCookiePtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_COOKIE", rqptr->HttpCookiePtr, VarType)))
         return (status);

   if (rqptr->HttpForwardedPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_FORWARDED", rqptr->HttpForwardedPtr, VarType)))
         return (status);

   if (rqptr->HttpHostPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_HOST", rqptr->HttpHostPtr, VarType)))
         return (status);

   if (rqptr->HttpIfModifiedSincePtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_IF_MODIFIED_SINCE",
                       rqptr->HttpIfModifiedSincePtr, VarType)))
         return (status);

   if (rqptr->HttpPragmaPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_PRAGMA", rqptr->HttpPragmaPtr, VarType)))
         return (status);

   if (rqptr->HttpRefererPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_REFERER", rqptr->HttpRefererPtr, VarType)))
         return (status);

   if (rqptr->HttpUserAgentPtr != NULL)
      if (VMSnok (status =
          CgiVariable (rqptr, "HTTP_USER_AGENT",
                       rqptr->HttpUserAgentPtr, VarType)))
         return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "PATH_INFO", rqptr->PathInfoPtr, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "PATH_TRANSLATED", rqptr->RequestFileName, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "QUERY_STRING", rqptr->QueryStringPtr, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REMOTE_ADDR", rqptr->ClientInternetAddress, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REMOTE_HOST", rqptr->ClientHostName, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REMOTE_USER", rqptr->RemoteUser, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REQUEST_METHOD", rqptr->HttpMethodName, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REQUEST_SCHEME", rqptr->RequestSchemeNamePtr, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REQUEST_TIME_GMT", rqptr->GmDateTime, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "REQUEST_TIME_LOCAL", LocalDateTime, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SCRIPT_NAME", rqptr->ScriptName, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SERVER_GMT", TimeGmtString, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SERVER_NAME", rqptr->ServicePtr->ServerHostName, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SERVER_PROTOCOL", HttpProtocol, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SERVER_PORT", rqptr->ServicePtr->ServerPortString, VarType)))
      return (status);

   if (VMSnok (status =
       CgiVariable (rqptr, "SERVER_SOFTWARE", SoftwareID, VarType)))
      return (status);

   /***************************/
   /* query string components */
   /***************************/

   if (rqptr->QueryStringPtr[0])
   {
      Count = 0;

      cptr = rqptr->QueryStringPtr;
      while (*cptr && *cptr != '=') cptr++;
      /* if an equal symbol was found then its a form not a keyword search */
      if (*cptr)
      {
         /***************/
         /* form fields */
         /***************/

         memcpy (FormFieldName, "FORM_", 5);
         cptr = rqptr->QueryStringPtr;
         while (*cptr)
         {
            sptr = FormFieldName + 5;
            zptr = FormFieldName + sizeof(FormFieldName);
            while (*cptr && *cptr != '=' && *cptr != '&' && sptr < zptr)
            {
               if (isalnum(*cptr))
                  *sptr++ = toupper(*cptr++);
               else
               {
                  *sptr++ = '_';
                  cptr++;
               }
            }
            if (sptr >= zptr)
            {
               ErrorVmsStatus (rqptr, SS$_BUFFEROVF, FI_LI);
               return (STS$K_ERROR);
            }
            *sptr = '\0';

            if (!FormFieldName[5] || *cptr != '=')
            {
               /* error; back-to-back '&' and/or '=', or no equate */
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_FORM), FI_LI);
               return (STS$K_ERROR);
            }

            /* must have encountered an '=' */
            if (*cptr) cptr++;
            zptr = (sptr = String) + sizeof(String);
            while (*cptr && *cptr != '&' && sptr < zptr)
            {
               if (*cptr == '+')
               {
                  *sptr++ = ' ';
                  cptr++;
               }
               else
               if (*cptr == '%')
               {
                  /* an escaped character ("%xx" where xx is a hex number) */
                  cptr++;
                  c = 0;
                  if (*cptr >= '0' && *cptr <= '9')
                     { c = (*cptr - (int)'0') << 4; cptr++; }
                  else
                  if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
                     { c = (tolower(*cptr) - (int)'a' + 10) << 4; cptr++; }
                  else
                  {
                     ErrorGeneral (rqptr,
                        MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                     return (STS$K_ERROR);
                  }
                  if (*cptr >= '0' && *cptr <= '9')
                     { c += (*cptr - (int)'0'); cptr++; }
                  else
                  if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
                     { c += (tolower(*cptr) - (int)'a' + 10); cptr++; }
                  else
                  {
                     ErrorGeneral (rqptr,
                        MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                     return (STS$K_ERROR);
                  }
                  if (sptr < zptr) *sptr++ = c;
               }
               else
                  *sptr++ = *cptr++;
            }
            if (sptr >= zptr)
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW), FI_LI);
               return (STS$K_ERROR);
            }
            *sptr = '\0';
            if (*cptr) cptr++;

            if (VMSnok (status =
                CgiVariable (rqptr, FormFieldName, String, VarType)))
               return (status);
         }

         /* no keywords if in form format */
         if (VMSnok (status =
             CgiVariable (rqptr, "KEY_COUNT", "0", VarType)))
            return (status);
      }
      else
      {
         /******************/
         /* query keywords */
         /******************/

         cptr = rqptr->QueryStringPtr;
         while (*cptr)
         {
            sys$fao (&NumberFaoDsc, 0, &StringDsc, ++Count);
            memcpy (FormFieldName, "KEY_", 4);
            strcpy (FormFieldName+4, String);

            zptr = (sptr = String) + sizeof(String);
            while (*cptr && *cptr != '+' && sptr < zptr)
            {
               if (*cptr == '%')
               {
                  /* an escaped character ("%xx" where xx is a hex number) */
                  cptr++;
                  c = 0;
                  if (*cptr >= '0' && *cptr <= '9')
                     { c = (*cptr - (int)'0') << 4; cptr++; }
                  else
                  if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
                     { c = (tolower(*cptr) - (int)'a' + 10) << 4; cptr++; }
                  else
                  {
                     ErrorGeneral (rqptr,
                        MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                     return (STS$K_ERROR);
                  }
                  if (*cptr >= '0' && *cptr <= '9')
                     { c += (*cptr - (int)'0'); cptr++; }
                  else
                  if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
                     { c += (tolower(*cptr) - (int)'a' + 10); cptr++; }
                  else
                  {
                     ErrorGeneral (rqptr,
                        MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                     return (STS$K_ERROR);
                  }
                  if (sptr < zptr) *sptr++ = c;
               }
               else
                  *sptr++ = *cptr++;
            }
            if (sptr >= zptr)
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW), FI_LI);
               return (STS$K_ERROR);
            }
            *sptr = '\0';
            if (*cptr) cptr++;

            if (VMSnok (status =
                CgiVariable (rqptr, FormFieldName, String, VarType)))
               return (status);
         }

         sys$fao (&NumberFaoDsc, 0, &StringDsc, Count);
         if (VMSnok (status =
             CgiVariable (rqptr, "KEY_COUNT", String, VarType)))
            return (status);
      }
   }
   else
   {
      /* no keywords if no query string! */
      if (VMSnok (status =
          CgiVariable (rqptr, "KEY_COUNT", "0", VarType)))
         return (status);
   }

   return (CgiVariable (rqptr, NULL, NULL, VarType));
}

/*****************************************************************************/
/*
Depending on whether the CGI variable is for creation at the DCL CLI or as part
of a CGI variable stream (e.g. CGIplus) create a null-terminated string
conatining either commands to create a DCL symbol, or a 'name=value' pair.

DCL symbol creation at the command line is limited by the CLI command line 
length (255 characters).  Symbol values however can be up to approximately 
1024 characters, probably enough for any CGI variable value.  If a CGI value 
is too large for for a single command-line assignment then build it up using 
multiple assignments, a symbol assignment kludge!
*/ 

int CgiVariable
(
struct RequestStruct *rqptr,
char *SymbolName,
char *SymbolValue,
int VarType
)
{
   static $DESCRIPTOR (StringDsc, "");
   static $DESCRIPTOR (StreamFaoDsc, "!AZ!AZ=!AZ");

   register char  *cptr, *sptr, *zptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "CgiVariable() |%s|%s|\n", SymbolName, SymbolValue);

   if (SymbolName == NULL)
   {
      /************************/
      /* end of CGI variables */
      /************************/

      if (rqptr->CgiBufferRemaining < sizeof(short))
         CgiVariableBufferMemory (rqptr);

      /* zero length record terminates generated CGI variables */
      *(short*)rqptr->CgiBufferCurrentPtr = 0;
      rqptr->CgiBufferRemaining -= sizeof(short);
      rqptr->CgiBufferCurrentPtr += sizeof(short);

      if (Debug)
         fprintf (stdout, "buffer used: %d\n",
                  rqptr->CgiBufferCurrentPtr - rqptr->CgiBufferPtr);

      return (SS$_NORMAL);
   }

   if (VarType == CGI_VARIABLE_STREAM)
   {
      /******************************/
      /* variables in record stream */
      /******************************/

      if (rqptr->CgiBufferRemaining <= 1024 + sizeof(short))
         CgiVariableBufferMemory (rqptr);
      StringDsc.dsc$w_length = 1023;
      StringDsc.dsc$a_pointer = rqptr->CgiBufferCurrentPtr + sizeof(short);
   
      status = sys$fao (&StreamFaoDsc, &Length, &StringDsc,
                        DclCgiVariablePrefix, SymbolName, SymbolValue);
      if (VMSok (status) && status != SS$_BUFFEROVF)
      {
         *(short*)rqptr->CgiBufferCurrentPtr = Length + 1;
         StringDsc.dsc$a_pointer[Length] = '\0';
         rqptr->CgiBufferRemaining -= Length + 1 + sizeof(short);
         rqptr->CgiBufferCurrentPtr += Length + 1 + sizeof(short);
         return (status);
      }
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status | STS$K_ERROR);
   }

   /********************************/
   /* variables created at DCL CLI */
   /********************************/

   if (rqptr->CgiBufferRemaining <= DCL_LINE_SIZE_MAX + 1 + sizeof(short))
      CgiVariableBufferMemory (rqptr);

   zptr = (sptr = rqptr->CgiBufferCurrentPtr + sizeof(short)) +
          DCL_LINE_SIZE_MAX;
   for (cptr = DclCgiVariablePrefix; *cptr; *sptr++ = *cptr++);
   for (cptr = SymbolName; *cptr && sptr < zptr; *sptr++ = *cptr++);
   if (sptr < zptr) *sptr++ = '=';
   if (sptr < zptr) *sptr++ = '=';
   if (sptr < zptr) *sptr++ = '\"';
   cptr = SymbolValue;
   while (*cptr && sptr < zptr)
   {
      /* escape quotes for DCL */
      if (*cptr == '\"')
      {
         /* only if there's room to double them up! */
         if (sptr < zptr-1)
            *sptr++ = '\"';
         else
            zptr = sptr;
      }
      if (sptr < zptr) *sptr++ = *cptr++;
   }
   if (sptr < zptr) *sptr++ = '\"';
   if (sptr < zptr)
   {
      *sptr++ = '\0';
      Length = sptr - (char*)rqptr->CgiBufferCurrentPtr - sizeof(short);
      *(short*)rqptr->CgiBufferCurrentPtr = Length;
      rqptr->CgiBufferRemaining -= Length + sizeof(short);
      rqptr->CgiBufferCurrentPtr += Length + sizeof(short);
      return (SS$_NORMAL);
   }

   /* buffer overflowed!  create it using multiple symbol assignments */
   if (rqptr->CgiBufferRemaining <= 5 + sizeof(short))
      CgiVariableBufferMemory (rqptr);
   memcpy (rqptr->CgiBufferCurrentPtr + sizeof(short), "X=\"\"\0", 5);
   *(short*)rqptr->CgiBufferCurrentPtr = 5;
   rqptr->CgiBufferRemaining -= 5 + sizeof(short);
   rqptr->CgiBufferCurrentPtr += 5 + sizeof(short);

   /* loop assigning maximum amount allowed by DCL until all assigned */
   cptr = SymbolValue;
   while (*cptr)
   {
      if (rqptr->CgiBufferRemaining <= DCL_LINE_SIZE_MAX + 1 + sizeof(short))
         CgiVariableBufferMemory (rqptr);

      if (Debug) fprintf (stdout, "|%s|\n", cptr);

      zptr = (sptr = rqptr->CgiBufferCurrentPtr + sizeof(short)) +
             DCL_LINE_SIZE_MAX-1;
      memcpy (sptr, "X=X+\"", 5);
      sptr += 5;
      while (*cptr && sptr < zptr)
      {
         /* escape quotes for DCL */
         if (*cptr == '\"')
         {
            /* only if there's room to double them up! */
            if (sptr < zptr-1)
               *sptr++ = '\"';
            else
               zptr = sptr;
         }
         if (sptr < zptr) *sptr++ = *cptr++;
      }
      *sptr++ = '\"';
      *sptr++ = '\0';
      Length = sptr - (char*)rqptr->CgiBufferCurrentPtr - sizeof(short);
      *(short*)rqptr->CgiBufferCurrentPtr = Length;
      rqptr->CgiBufferRemaining -= Length + sizeof(short);
      rqptr->CgiBufferCurrentPtr += Length + sizeof(short);
   }

   /* assign the temporary symbol value to the CGI symbol */
   if (rqptr->CgiBufferRemaining <= DCL_LINE_SIZE_MAX + 1 + sizeof(short))
      CgiVariableBufferMemory (rqptr);
   sptr = rqptr->CgiBufferCurrentPtr + sizeof(short);
   for (cptr = DclCgiVariablePrefix; *cptr; *sptr++ = *cptr++);
   for (cptr = SymbolName; *cptr; *sptr++ = *cptr++);
   for (cptr = "==X"; *cptr; *sptr++ = *cptr++);
   *sptr++ = '\0';
   Length = sptr - (char*)rqptr->CgiBufferCurrentPtr - sizeof(short);
   *(short*)rqptr->CgiBufferCurrentPtr = Length;
   rqptr->CgiBufferRemaining -= Length + sizeof(short);
   rqptr->CgiBufferCurrentPtr += Length + sizeof(short);

   /* not really necessary, but let's be tidy */
   if (rqptr->CgiBufferRemaining <= 16 + sizeof(short))
      CgiVariableBufferMemory (rqptr);
   memcpy (rqptr->CgiBufferCurrentPtr + sizeof(short), "DELETE/SYMBOL X\0", 16);
   *(short*)rqptr->CgiBufferCurrentPtr = 16;
   rqptr->CgiBufferRemaining -= 16 + sizeof(short);
   rqptr->CgiBufferCurrentPtr += 16 + sizeof(short);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Allocate, or reallocate on subsequent calls, memory for storing the CGI
variables in.  Adjusts the CGI variable buffer pointers, lengths, etc,
appropriately.
*/ 

int CgiVariableBufferMemory (struct RequestStruct *rqptr)

{
   int  CurrentLength,
        CurrentOffset,
        NewLength;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "CgiVariableBufferMemory() %d\n",
               rqptr->CgiBufferLength);

   if (!rqptr->CgiBufferLength)
   {
      /* initialize CGI variable buffer */
      rqptr->CgiBufferLength = rqptr->CgiBufferRemaining = CGI_BUFFER_INITIAL;
      rqptr->CgiBufferPtr = rqptr->CgiBufferCurrentPtr =
         VmGetHeap (rqptr, CGI_BUFFER_INITIAL);
      return;
   }

   CurrentLength = rqptr->CgiBufferLength;
   CurrentOffset = rqptr->CgiBufferCurrentPtr - rqptr->CgiBufferPtr;
   NewLength = CurrentLength + CGI_BUFFER_INCREMENT;

   rqptr->CgiBufferPtr = VmReallocHeap (rqptr, rqptr->CgiBufferPtr, NewLength);

   rqptr->CgiBufferLength = NewLength;
   rqptr->CgiBufferRemaining += CGI_BUFFER_INCREMENT;
   rqptr->CgiBufferCurrentPtr = rqptr->CgiBufferPtr + CurrentOffset;
}

/*****************************************************************************/
/*
Process output from a CGI script (or even plain DCL processing).  For CGI
scripts check the first output from each for CGI-relevant HTTP header lines. 
For all output check that carriage-control is as required.

Returns -1 to terminate processing, 0 to continue but absorb any output
(redirection), and a positive number, representing the number of bytes to
output (after possible carriage-control adjustment), to continue processing.
*/ 

int CgiOutput
(
struct RequestStruct *rqptr,
char *OutputPtr,
int OutputCount
)
{
   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiOutput() %d\n", OutputCount);

   if (rqptr->CgiEofLength)
   {
      if (OutputPtr[0] == rqptr->CgiEofPtr[0] &&
          OutputCount >= rqptr->CgiEofLength &&
          OutputCount <= rqptr->CgiEofLength+2 &&
          !memcmp (OutputPtr, rqptr->CgiEofPtr, rqptr->CgiEofLength))
      {
         /******************************/
         /* end of output from script! */
         /******************************/

         if (Debug) fprintf (stdout, "CGI EOF!\n");
         return (CGI_OUTPUT_TERMINATE);
      }
   }

   if (rqptr->LocationPtr != NULL)
   {
      /* absorb any output from script until the request processing ends */
      return (CGI_OUTPUT_ABSORB);
   }

   if (Debug)
      fprintf (stdout, "%d %d\n", rqptr->CgiScript, rqptr->CgiPlusScript);
   if (rqptr->CgiScript || rqptr->CgiPlusScript)
   {
      /********************/
      /* script execution */
      /********************/

      if (!rqptr->CgiCheckedHeader)
      {
         value = CgiScriptHeader (rqptr, OutputPtr, OutputCount);
         if (value == CGI_OUTPUT_TERMINATE ||
             value == CGI_OUTPUT_ABSORB)
            return (value);
      }
   }

   if (rqptr->CgiAddNewline)
   {
      /*************************/
      /* HTTP carriage-control */
      /*************************/

      if (Debug) fprintf (stdout, "rqptr->CgiAddNewline\n");

      if (OutputCount)
      {
         if (OutputPtr[OutputCount-1] != '\n')
            OutputPtr[OutputCount++] = '\n';
      }
      else
      {
         /* must be a blank line (empty record), add a newline */
         OutputPtr[OutputCount++] = '\n';

         /* first blank line is end of response header! */
         if (!rqptr->CgiContentTypeText) rqptr->CgiAddNewline = false;
      }
      if (Debug) OutputPtr[OutputCount] = '\0';
   }

   return (OutputCount);
}

/*****************************************************************************/
/*
For CGI-compliance the first line of the header should be a "Content-Type:", 
"Location:" or "Status:".  If a "Content-Type:" prepend a "200" header status 
line.  If a "Status:" use this to create a unique header status line.  If a 
"Location:" redirection header line call a function to establish the level of 
redirection, and do any associated processing.  For non-parse-header style CGI 
scripts check if the first line is a full HTTP status line.  If it is then do 
nothing, the script will supply the raw HTTP stream.  If none of the above 
then consider the script will supply nothing but plain text, none of the HTTP 
header, etc., and prepend a "200" status line and a "Content-Type: text/plain" 
line to create a complete HTTP header.  This is an extension to CGI behaviour, 
but makes simple, plain-text DCL scripts very easy!  Return true any action
required has been taken by this function, false if further processing is
required by the calling function.

Returns -1 to terminate processing, 0 to continue but absorb any output
(redirection), and 1 to continue processing.
*/ 

int CgiScriptHeader 
(
struct RequestStruct *rqptr,
char *OutputPtr,
int OutputCount
)
{
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiScriptHeader()\n|%s|\n", OutputPtr);

   rqptr->CgiCheckedHeader = true;
   cptr = OutputPtr;

   if (toupper(*cptr) == 'H' && strsame (cptr, "HTTP/", 5))
   {
      /*************************************************/
      /* script is supplying the full HTTP data stream */
      /*************************************************/

      if (Debug) fprintf (stdout, "(HTTP data stream)\n");
      rqptr->CgiAddNewline = false;

      /* get the response status code for logging purposes */
      while (*cptr && !ISLWS(*cptr)) cptr++;
      while (*cptr && !isdigit(*cptr)) cptr++;
      if (isdigit(*cptr)) rqptr->ResponseStatusCode = atoi(cptr);
      return (OutputCount);
   }
   else
   if (toupper(*cptr) == 'L' && strsame (cptr, "Location:", 9))
   {
      /*******************/
      /* redirection URL */
      /*******************/

      cptr += 9;
      /* locate the first character of the "Location:" URL */
      while (ISLWS(*cptr)) cptr++;
      sptr = cptr;
      while (*cptr && !ISLWS(*cptr) && *cptr != '\r' && *cptr != '\n') cptr++;
      rqptr->LocationPtr = VmGetHeap (rqptr, cptr-sptr+1);
      memcpy (rqptr->LocationPtr, sptr, cptr-sptr);
      rqptr->LocationPtr[cptr-sptr] = '\0';
      if (Debug) fprintf (stdout, "LocationPtr |%s|\n", rqptr->LocationPtr);
      return (CGI_OUTPUT_ABSORB);
   }
   else
   if (toupper(*cptr) == 'C' && strsame (cptr, "Content-Type:", 13))
   {

      /**************************/
      /* content-type specified */
      /**************************/

      /*
         For CGI content types of "text/..." add carriage control to each
         record needing it.  For non-text types this carriage-control is
         added to response header lines then turned off for the body.
      */

      cptr += 13;
      while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
      if (strsame (cptr, "text/", 5))
      {
         /* make sure each record received has HTTP carriage control */
         if (Debug) fprintf (stdout, "(RMS record stream)\n");
         rqptr->CgiAddNewline = rqptr->CgiContentTypeText = true;
      }
      else
         rqptr->CgiAddNewline = true;

      rqptr->ResponseStatusCode = 200;
      if ((rqptr->ResponseHeaderPtr =
           HttpHeader (rqptr, rqptr->ResponseStatusCode,
                       NULL, -1, NULL, NULL)) == NULL)
         return (CGI_OUTPUT_TERMINATE);
      /* remove generated empty header line to allow the script to supply it */
      if (rqptr->ResponseHeaderLength > 2)
      {
         rqptr->ResponseHeaderPtr[rqptr->ResponseHeaderLength-2] = '\0';
         rqptr->ResponseHeaderLength -= 2;
      }
      return (OutputCount);
   }
   else
   if (toupper(*cptr) == 'S' && strsame (cptr, "Status:", 7))
   {
      /********************/
      /* status specified */
      /********************/

      /* make sure each record received has HTTP carriage control */
      if (Debug) fprintf (stdout, "(RMS record stream)\n");
      rqptr->CgiAddNewline = true;

      /* create HTTP header status line using supplied status */
      cptr += 7;
      while (*cptr++ && !isdigit(*cptr) && *cptr != '\n') cptr++;
      /* get the response status code for header and logging purposes */
      if (isdigit(*cptr)) rqptr->ResponseStatusCode = atoi(cptr);

      if ((rqptr->ResponseHeaderPtr =
           HttpHeader (rqptr, rqptr->ResponseStatusCode,
                       "text/plain", -1, NULL, NULL)) == NULL)
         return (CGI_OUTPUT_TERMINATE);
      /* remove generated empty header line, allow the script to supply it */
      if (rqptr->ResponseHeaderLength > 2)
      {
         rqptr->ResponseHeaderPtr[rqptr->ResponseHeaderLength-2] = '\0';
         rqptr->ResponseHeaderLength -= 2;
      }
      return (OutputCount);
   }
   else
   {
      /******************/
      /* non-CGI output */
      /******************/

      /* make sure each record received has HTTP carriage control */
      if (Debug) fprintf (stdout, "(RMS record stream)\n");
      rqptr->CgiAddNewline = rqptr->CgiContentTypeText = true;
      rqptr->ResponseStatusCode = 200;

      if ((rqptr->ResponseHeaderPtr =
           HttpHeader (rqptr, rqptr->ResponseStatusCode,
                       "text/plain", -1, NULL, NULL)) == NULL)
         return (CGI_OUTPUT_TERMINATE);

      return (OutputCount);
   }
}

/*****************************************************************************/
/*
Generate a (hopefully) unique sequence of 280 bits (which is a fair bit :^) The
string is generated using the munged memory address (first !8XL) and the the
quadword binary time (subsequent !8XLs) each time it is called. This provides a
continuous, non-repeating series of unlikely bit combinations with a one in
2^280 chance (I think!) of presence in an output stream.
*/ 

int CgiEof
(
char *CgiEofPtr,
int *CgiEofLengthPtr
)
{
   $DESCRIPTOR (EofFaoDsc, "^z[!8XL?!8XL?!8XL]~EoF!!\0");
   $DESCRIPTOR (EofDsc, "");

   unsigned short  Length;
   unsigned long  BinTime[2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CgiEof()\n");

   sys$gettim(&BinTime);
   EofDsc.dsc$w_length = 36;
   EofDsc.dsc$a_pointer = CgiEofPtr;
   sys$fao (&EofFaoDsc, &Length, &EofDsc,
            ((long)CgiEofPtr ^ BinTime[1] ^ BinTime[0]),
            BinTime[1], BinTime[0]);
   *CgiEofLengthPtr = Length - 1;

   if (Debug) fprintf (stdout, "%d |%s|\n", *CgiEofLengthPtr, CgiEofPtr);
}

/*****************************************************************************/

