/*****************************************************************************
/*
                                   CLI.c

Module to process (interpret) the HTTPD command line.  Allows images activated
by a "foreign verb" to behave in a way that approximates the CLI$ (Command
Line Interpreter) utility calls.


VERSION HISTORY
---------------
07-AUG-98  MGD  /PROMISCUOUS[=password]
06-JUL-98  MGD  v5.1, /SYSUAF=ID
11-MAR-98  MGD  /SYSUAF=(SSL,ALL)
04-FEB-98  MGD  v5.0, /SSL, /SYSUAF=SSL
11-OCT-97  MGD  v4.5, /CACHE, /VERSION
06-SEP-97  MGD  v4.4, /FORMAT, /PROFILE, /SYSUAF, /SOFTWAREID
01-FEB-97  MGD  HTTPd version 4
01-DEC-95  MGD  HTTPd version 3
20-DEC-94  MGD  multi-threaded daemon
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS header files */
#include <descrip.h>
#include <stsdef.h>
#include <syidef.h>
#include <ssdef.h>
#include <stsdef.h>

#include "wasd.h"

/**********/
/* macros */
/**********/

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
/******************/
/* global storage */
/******************/

boolean  CliCacheDisabled = false,
         CliCacheEnabled = false,
         CliLoggingDisabled = false,
         CliLoggingEnabled = false,
         CliMonitorDisabled = false,
         CliMonitorEnabled = false,
         CliVersion = false;
int  CliServerPort = 0;
char  *CliAcceptHostsPtr = NULL,
      *CliRejectHostsPtr = NULL;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  AccountingZeroOnStartup;
extern boolean  AuthPromiscuous;
extern boolean  AuthSslOnly;
extern boolean  AuthSysUafAllAccounts;
extern boolean  AuthSysUafEnabled;
extern boolean  AuthSysUafIdentifier;
extern boolean  AuthSysUafPromiscuous;
extern boolean  AuthSysUafSslOnly;
extern boolean  AuthVmsUserSecProfileEnabled;
extern boolean  LoggingFormatCombined;
extern boolean  LoggingFormatCommon;
extern boolean  LoggingFormatCommonServer;
extern boolean  NoSwapOut;
extern int  DclSysOutputSize;
extern int  DclCgiVariablePrefixLength;
extern int  FileBufferSize;
extern int  MaxConcurrentConnections;
extern int  NetReadBufferSize;
extern int  OutputBufferSize;
extern int  ProcessPriority;
extern int  SeSoLaProtocolVersion;
extern char  *AuthPromiscuousPwdPtr;
extern char  CommandLine[];
extern char  ControlBuffer[];
extern char  DclCgiVariablePrefix[];
extern char  LoggingFileName[];
extern char  LoggingFormatUser[];
extern char  LoggingPeriodName[];
extern char  Services[];
extern char  SoftwareID[];
extern char  Utility[];
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { 255, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || ISLWS (*cptr) || !*cptr)
      {
         if (ISLWS (*cptr))
         {
            /* span the white space */
            while (*cptr && ISLWS (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && ISLWS (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   register char  *cptr;

   boolean  ok;
   int  Length;
   char  Scratch [256];

   if (Entity == NULL)
   {
      /* set up any argument defaults */
      return (true);
   }

   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/ACCEPT=", 4))
      {
         ok = ParseCommandString (Entity, Scratch, true, true, true);
         for (cptr = Scratch; *cptr && ISLWS(*cptr); cptr++);
         if (!*cptr) return (ok);
         CliAcceptHostsPtr = VmGet (Length = strlen(Scratch)+1);
         memcpy (CliAcceptHostsPtr, Scratch, Length);
         return (ok);
      }

      if (strsame (Entity, "/AUTHORIZATION=SSL", -1))
         return (AuthSslOnly = true);

      if (strsame (Entity, "/CACHE", 4))
      {
         CliCacheEnabled = true;
         CliCacheDisabled = false;
         return (true);
      }
      if (strsame (Entity, "/NOCACHE", 6))
      {
         CliCacheEnabled = false;
         CliCacheDisabled = true;
         return (true);
      }

      if (strsame (Entity, "/CGI_PREFIX=", 4))
      {
         ok = ParseCommandString (Entity, DclCgiVariablePrefix,
                                  true, true, true);
         DclCgiVariablePrefixLength = strlen(DclCgiVariablePrefix);
         return (ok);
      }

      if (strsame (Entity, "/DO=", 4))
         return (ParseCommandString (Entity, ControlBuffer, true, true, false));

#ifdef DBUG
      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);
#endif

      if (strsame (Entity, "/FILBUF=", 5))
         return (ParseCommandInteger (Entity, &FileBufferSize, 10, true));

      if (strsame (Entity, "/FORMAT=", 4))
         return (ParseCommandString (Entity, LoggingFormatUser,
                                     true, true, false));

      if (strsame (Entity, "/REJECT=", 4))
      {
         ok = ParseCommandString (Entity, Scratch, true, true, true);
         for (cptr = Scratch; *cptr && ISLWS(*cptr); cptr++);
         if (!*cptr) return (ok);
         CliRejectHostsPtr = VmGet (Length = strlen(Scratch)+1);
         memcpy (CliRejectHostsPtr, Scratch, Length);
         return (ok);
      }

      if (strsame (Entity, "/LOG=", 4))
      {
         CliLoggingEnabled = true;
         CliLoggingDisabled = false;
         if (Entity[4] == '=')
            return (ParseCommandString (Entity, LoggingFileName,
                                        true, true, true));
         else
            return (true);
      }
      if (strsame (Entity, "/NOLOG", 6))
      {
         CliLoggingEnabled = false;
         CliLoggingDisabled = true;
         return (true);
      }

      if (strsame (Entity, "/MONITOR", 4))
      {
         CliMonitorEnabled = true;
         CliMonitorDisabled = false;
         return (true);
      }
      if (strsame (Entity, "/NOMONITOR", 6))
      {
         CliMonitorEnabled = false;
         CliMonitorDisabled = true;
         return (true);
      }

      if (strsame (Entity, "/NETBUF=", 5))
         return (ParseCommandInteger (Entity, &NetReadBufferSize, 10, true));

      if (strsame (Entity, "/OUTBUF=", 5))
         return (ParseCommandInteger (Entity, &OutputBufferSize, 10, true));

      if (strsame (Entity, "/PERIOD=", 4))
         return (ParseCommandString (Entity, LoggingPeriodName,
                                      true, true, false));

      if (strsame (Entity, "/PORT=", 4))
         return (ParseCommandInteger (Entity, &CliServerPort, 10, true));

      if (strsame (Entity, "/PRIORITY=", 4))
         return (ParseCommandInteger (Entity, &ProcessPriority, 10, true));

      if (strsame (Entity, "/PROFILE", -1))
         return (AuthVmsUserSecProfileEnabled = true);
      if (strsame (Entity, "/NOPROFILE", -1))
      {
         AuthVmsUserSecProfileEnabled = false;
         return (true);
      }

      /* for testing only, also needs /SYSUAF=PROMSICUOUS is using SYSUAF! */
      if (strsame (Entity, "/PROMISCUOUS=", 4))
      {
         for (cptr = Entity; *cptr && *cptr != '='; cptr++);
         if (*cptr)
         {
            cptr++;
            AuthPromiscuousPwdPtr = VmGet (Length = strlen(cptr)+1);
            memcpy (AuthPromiscuousPwdPtr, cptr, Length);
         }
         return (AuthPromiscuous = true);
      }
      if (strsame (Entity, "/NOPROMISCUOUS", 6))
      {
         AuthPromiscuous = false;
         return (true);
      }

      if (strsame (Entity, "/SERVICES=", 4))
         return (ParseCommandString (Entity, Services, true, true, false));

      /* 
         This qualifier may come in handy if having to use VMS or WASD
         causes embarressment :^)  Seriously, if you consider providing
         the server identification a security issue use this qualifier
         to set it to something like "Mine", "Unknown", "?", etc.
      */
      if (strsame (Entity, "/SOFTWAREID=", 4))
         return (ParseCommandString (Entity, SoftwareID, true, true, false));

      if (strsame (Entity, "/SSL=", 4))
      {
         SeSoLaProtocolVersion = 0;
         ParseCommandInteger (Entity, &SeSoLaProtocolVersion, 10, false);
         return (true);
      }
      if (strsame (Entity, "/NOSSL", 6))
      {
         SeSoLaProtocolVersion = -1;
         return (true);
      }

      if (strsame (Entity, "/SUBBUF=", 5))
         return (ParseCommandInteger (Entity, &DclSysOutputSize, 10, true));

      if (strsame (Entity, "/SWAP", -1))
      {
         NoSwapOut = false;
         return (true);
      }
      if (strsame (Entity, "/NOSWAP", -1))
         return (NoSwapOut = true);

      if (strsame (Entity, "/SYSUAF", -1))
         return (AuthSysUafEnabled = true);
      if (strsame (Entity, "/NOSYSUAF", -1))
      {
         AuthSysUafEnabled = false;
         return (true);
      }
      if (strsame (Entity, "/SYSUAF=", 8))
      {
         if (strstr (Entity+8, "ALL") != NULL ||
             strstr (Entity+8, "all") != NULL)
            AuthSysUafAllAccounts = true;
         if (strstr (Entity+8, "ID") != NULL ||
             strstr (Entity+8, "id") != NULL)
            AuthSysUafIdentifier = true;
         if (strstr (Entity+8, "SSL") != NULL ||
             strstr (Entity+8, "ssl") != NULL)
            AuthSysUafSslOnly = true;
         return (AuthSysUafEnabled = true);
      }

      /* for testing SYSUAF-authenticated security profiles purposes only */
      if (strsame (Entity, "/SYSUAF=PROMISCUOUS", -1))
         return (AuthSysUafEnabled = AuthSysUafPromiscuous = true);

      if (strsame (Entity, "/VERSION", 4))
         return (CliVersion = true);

      if (strsame (Entity, "/ZERO", -1))
         return (AccountingZeroOnStartup = true);
      if (strsame (Entity, "/NOZERO", -1))
      {
         AccountingZeroOnStartup = false;
         return (true);
      }

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

