/*****************************************************************************/
/*
                                 Config.c

This module performs three functions.

  1.  It reads the HTTPD configuration file, converting the directives
      into internal representations.

  2.  Provides functions, accessed by other modules, returning the
      original directive information as appropriate.

  3.  Provides HTML form-based configuration and reports.

Configuration can only be performed at server startup.  To reset the 
configuration the server must be restarted.

WASD HTTPd Configuration is a relevant subset of the CERN configuration 
directives (sort-of, well it started that way anyhow!)  Although there are
some  differences (beware!) the CERN documentation can be consulted for a
general discussion of the purpose and functioning some of these directives. 

The file formats may be IN ONE OF TWO possible formats (not mixed).  The
older format contains one directive at the start of a line, followed by a
mandatory parameter.  The newer format (in part to support forms-based
configuration) has the directive contained within square brackets (e.g.
"[accept]") followed on the same line, or by one or more lines containing
comments, containing parameter information, or empty.  The directive applies
to all lines following up until another directive (square bracket in column
one) is encountered.

Directives are:

  o  Accept             <host/domain>
  o  ActivityDays       <integer>
  o  AddType            <suffix> <content-type> -|</script> [<description>]
  o  AddIcon            <URL> <alternative-text> <content-type-template>
  o  AddDirIcon         <URL> <alternative-text>
  o  AddBlankIcon       <URL> <alternative-text>
  o  AddUnknownIcon     <URL> <alternative-text>
  o  AddParentIcon      <URL> <alternative-text>
  o  AuthBasic          DISABLED | ENABLED
  o  AuthCacheMinutes   <integer>
  o  AuthDigest         DISABLED | ENABLED
  o  AuthDigestGetLife  <integer>
  o  AuthDigestPutLife  <integer>
  o  AuthFailureLimit   <integer>
  o  AuthRevalidateUserMinutes   <integer>
  o  Busy               <integer>
  o  Cache              DISABLED | ENABLED
  o  CacheChunkKBytes   <integer>
  o  CacheEntriesMax    <integer>
  o  CacheFileKBytesMax <integer>
  o  CacheFrequentHits      <integer>
  o  CacheFrequentSeconds   <integer>
  o  CacheHashTableEntries  <integer>
  o  CacheTotalKBytesMax    <integer>
  o  CacheValidateSeconds   <integer>
  o  CommentedInfo      DISABLED | ENABLED
  o  DclCgiPlusLifeTime <integer>
  o  DclHardLimit       <integer>
  o  DclScriptRunTime   <string>
  o  DclSoftLimit       <integer>
  o  DclSpawnAuthPriv   DISABLED | ENABLED
  o  DclFullRequest     DISABLED | ENABLED
  o  DclZombieLifeTime  <integer>
  o  DECnetReuseLifeTime    <integer>
  o  DECnetConnectListMax   <integer>
  o  DirAccess          DISABLED | ENABLED | SELECTIVE
  o  DirDescription     <integer>
  o  DirLayout          <string>
  o  DirOwner           DISABLED | ENABLED
  o  DirPreExpired      DISABLED | ENABLED
  o  DirReadMe          DISABLED | TOP | BOTTOM
  o  DirReadMeFile      <file.suffix>
  o  DirNoImpliedWildcard   DISABLED | ENABLED
  o  DirNoPrivIgnore    DISABLED | ENABLED
  o  DirWildcard        DISABLED | ENABLED
  o  DNSLookup          DISABLED | ENABLED
  o  ErrorRecommend     DISABLED | ENABLED
  o  ErrorSourceInfo    DISABLED | ENABLED
  o  InputTimeout       <integer>
  o  KeepAliveTimeout   <integer>
  o  Logging            DISABLED | ENABLED
  o  LogExcludeHosts    <string>
  o  LogFile            <string>
  o  LogFormat          <string>
  o  LogNaming          <string>
  o  LogPeriod          <string>
  o  LogPerService      DISABLED | ENABLED
  o  Monitor            DISABLED | ENABLED
  o  OutputTimeout      <integer>
  o  Port               <integer>
  o  PutMaxKbytes       <integer>
  o  Reject             <host/domain>
  o  RequestHistory     <integer>
  o  Search             <absolute-path-to-search-script>
  o  Service            <string>
  o  Ssi | Shtml                      DISABLED | ENABLED
  o  SsiAccesses | ShtmlAccesses      DISABLED | ENABLED
  o  SsiExec | ShtmlExec              DISABLED | ENABLED
  o  StreamLf           <integer>
  o  StreamLfPaths      <string>
  o  Welcome            <file.suffix>


VERSION HISTORY
---------------
27-AUG-98  MGD  generic ConfigSetCommaList(),
                add [AuthRevalidateUserMinutes], [LogExcludeHosts],
                    [DECnetReuseLifeTime], [DECnetConnectListMax],
                    [StreamLfPaths]
08-AUG-98  MGD  add [DirNoImpliedWildcard]
17-JUN-98  MGD  add [LogNaming], [LogPerService]
28-FEB-98  MGD  improved icon handling efficiency
07-FEB-98  MGD  v5.0 hash table for content-type suffix and icon searches,
                default content type (for unknowns) configurable,
                HTTP protocol name in [service] (for SSL),
                [DirDescription] changed from boolean to integer
05-OCT-97  MGD  cache parameters,
                DCL script run-time parameter,
                additional logging parameters
09-AUG-97  MGD  v4.4 retired "ErrorInfo", "ErrorSysAdmin" (now in messages)
                and AuthVMS (now /SYSUAF qualifier);  introduced
                "DirDescription", "DirNoPrivIgnore", "ErrorSourceInfo",
                bugfix; memory leak in file-based report and revise
01-AUG-97  MGD  v4.3 activity number of days, DCL supply full request,
                bugfix; annoying repeated instances of "unknown" content type
01-JUN-97  MGD  v4.2 required additional DCL/scripting parameters
01-FEB-97  MGD  HTTPd version 4;
                major changes for form-based config;
                added error message URLs
01-OCT-96  MGD  HTML configuration report
01-AUG-96  MGD  StreamLF conversion directive;
                AuthBasic and AuthDigest directives;
12-APR-96  MGD  file record/binary now determined by record format
                (has removed the need of the text/binary, etc., in "AddType")
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added auto-scripting funtionality
07-AUG-95  MGD  added support for VMS-style directory structure and file sizes
16-JUN-95  MGD  file contents description in 'AddType' (also see dir.c module)
20-DEC-94  MGD  developed for multi-threaded HTTP daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header */
#include "wasd.h"
#include "cache.h"
#include "config.h"
#include "error.h"
#include "httpd.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "vm.h"

/******************/
/* global storage */
/******************/

struct ConfigStruct  Config;

char  ConfigContentTypeBlank [] = "x-internal/blank",
      ConfigContentTypeDir [] = "x-internal/directory",
      ConfigContentTypeIsMap [] = "text/x-ismap",
      ConfigContentTypeMenu [] = "text/x-menu",
      ConfigContentTypeParent [] = "x-internal/parent",
      ConfigContentTypeSsi [] = "text/x-shtml",
      ConfigContentTypeUnknown [] = "x-internal/unknown",
      ConfigDefaultFileContentType [] = "application/octet-stream",
      ConfigStringOverflow [] = "*ERROR* string overflow";


char  *ConfigBlankIconPtr,
      *ConfigDirIconPtr,
      *ConfigParentIconPtr,
      *ConfigUnknownIconPtr;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  TimeGmtString[];
extern char  Utility[];
extern struct MsgStruct  Msgs;
extern struct ServiceStruct  *ServiceListHead;

/*****************************************************************************/
/*
Open the configuration file defined by the logical name HTTPD$CONFIG.  Read 
each line, passing to the appropriate interpretation function according to the 
directive on the line.  Close the file.  Function is either called before
AST-driven  processing or called at AST delivery level which makes processing
atomic.  Allow 32 extra characters on line buffer and parameter allocated
memory for concatenating additional string.
*/

int Configure (struct ConfigStruct *cfptr)

{
#  define PARAMETER_ELBOW_ROOM 32

   register char  *cptr, *sptr, *zptr;

   boolean  DebugBuffer,
            BracketDirectives,
            EquateDirectives;
   int  status,
        ParemeterLength,
        SetPrvStatus,
        LineLength;
   void  *RequestPtrBuffer;
   char  ExpandedFileName [256],
         Line [256 + PARAMETER_ELBOW_ROOM];
   struct FAB  ConfigFileFab;
   struct NAM  ConfigFileNam;
   struct RAB  ConfigFileRab;
   struct XABDAT  ConfigFileXabDat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Configure()\n");

#ifdef DBUG
   /* not interested anymore in seeing debug information for config load! */
   DebugBuffer = Debug;
   Debug = false;
#endif

   RequestPtrBuffer = cfptr->RequestPtr;
   memset (cfptr, 0, sizeof(struct ConfigStruct));
   cfptr->RequestPtr = RequestPtrBuffer;

   BracketDirectives = EquateDirectives = false;
   sys$gettim (&cfptr->LoadBinTime);

   /************************/
   /* open the config file */
   /************************/

   if (Debug) fprintf (stdout, "ConfigFileName |%s|\n", ConfigFileName);

   ConfigFileFab = cc$rms_fab;
   ConfigFileFab.fab$b_fac = FAB$M_GET;
   ConfigFileFab.fab$l_fna = ConfigFileName;
   ConfigFileFab.fab$b_fns = strlen(ConfigFileName);
   ConfigFileFab.fab$l_nam = &ConfigFileNam;
   ConfigFileFab.fab$b_shr = FAB$M_SHRGET;
   ConfigFileFab.fab$l_xab = &ConfigFileXabDat;

   ConfigFileNam = cc$rms_nam;
   ConfigFileNam.nam$l_esa = ExpandedFileName;
   ConfigFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   ConfigFileNam.nam$l_rsa = cfptr->LoadFileName;
   ConfigFileNam.nam$b_rss = sizeof(cfptr->LoadFileName)-1;

   /* initialize the date extended attribute block */
   ConfigFileXabDat = cc$rms_xabdat;

   /* turn on SYSPRV to allow access to possibly protected file */
   EnableSysPrv ();
   status = sys$open (&ConfigFileFab, 0, 0);
   DisableSysPrv ();

   /* status from sys$open() */
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (cfptr->RequestPtr)
      {
         ConfigReportProblem (cfptr, NULL, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, ConfigFileName, FI_LI);
   }

   cfptr->LoadFileName[ConfigFileNam.nam$b_rsl] = '\0';
   if (Debug)
      fprintf (stdout, "cfptr->LoadFileName |%s|\n", cfptr->LoadFileName);

   /* get the configuration file revision date and time */
   memcpy (&cfptr->RevBinTime,
           &ConfigFileXabDat.xab$q_rdt,
           sizeof(cfptr->RevBinTime));

   /* record access block */
   ConfigFileRab = cc$rms_rab;
   ConfigFileRab.rab$l_fab = &ConfigFileFab;
   /* try and optimise sequential read (next 3 lines) */
   ConfigFileRab.rab$b_mbc = 6;
   ConfigFileRab.rab$b_mbf = 2;
   ConfigFileRab.rab$l_rop = RAB$M_RAH;
   ConfigFileRab.rab$l_ubf = Line;
   ConfigFileRab.rab$w_usz = sizeof(Line)-PARAMETER_ELBOW_ROOM-1;

   if (VMSnok (status = sys$connect (&ConfigFileRab, 0, 0)))
   {
      sys$close (&ConfigFileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      if (cfptr->RequestPtr)
      {
         ConfigReportProblem (cfptr, NULL, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, ConfigFileName, FI_LI);
   }

   /************************/
   /* read the config file */
   /************************/

   cfptr->AuthCacheMinutes = cfptr->AuthRevalidateUserMinutes = 60;
   cfptr->AuthFailureLimit = 10;
   cfptr->DirPreExpired = PRE_EXPIRE_INDEX_OF;
   cfptr->DefaultPost = "/ht_root/cgi-bin/post";
   strcpy (cfptr->DefaultDirLayout, DEFAULT_DIR_LAYOUT);
   cfptr->AcceptHostsPtr = cfptr->RejectHostsPtr =
      cfptr->ContentTypeListHeadPtr = cfptr->IconListHeadPtr =
      cfptr->LogExcludeHostsPtr = cfptr->StreamLfPathsPtr = NULL;
   cfptr->ContentTypeStructOverhead = sizeof(struct ContentTypeStruct);
   cfptr->IconStructOverhead = sizeof(struct IconStruct);
   cfptr->ContentTypeDefaultPtr = "";

   while (VMSok (status = sys$get (&ConfigFileRab, 0, 0)))
   {
      cfptr->LineNumber++;
      Line[LineLength = ConfigFileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line %d |%s|\n", cfptr->LineNumber, Line);

      if (Line[0] == '#') continue;
      cptr = Line;
      while (*cptr && ISLWS(*cptr) && *cptr == '\n') cptr++;
      if (!*cptr) continue;

      if (Line[0] == '[')
      {
         if (EquateDirectives)
         {
            ConfigReportProblem (cfptr, "mixed directive formats", 0);
            break;
         }
         BracketDirectives = true;
         cptr++;
         zptr = (sptr = cfptr->DirectiveName) + sizeof(cfptr->DirectiveName);
         while (*cptr && *cptr != ']' && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ConfigReportProblem (cfptr, "missing closing bracket?", 0);
            continue;
         }
         *sptr = '\0';
         if (Debug)
            fprintf (stdout, "DirectiveName |%s|\n", cfptr->DirectiveName);
         if (*cptr) cptr++;
         while (*cptr && ISLWS(*cptr)) cptr++;
      }
      else
      if (!BracketDirectives)
      {
         EquateDirectives = true;
         zptr = (sptr = cfptr->DirectiveName) + sizeof(cfptr->DirectiveName);
         while (*cptr && !ISLWS(*cptr) && *cptr != '=' && sptr < zptr)
             *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ConfigReportProblem (cfptr, "missing parameter");
            continue;
         }
         *sptr = '\0';
         while (*cptr && ISLWS(*cptr)) cptr++;
         if (!*cptr)
         {
            ConfigReportProblem (cfptr, "missing parameter", 0);
            continue;
         }
      }

      if (BracketDirectives && !*cptr) continue;

      if (strsame (cfptr->DirectiveName, "Accept", -1))
         ConfigSetCommaList (cfptr, cptr, &cfptr->AcceptHostsPtr,
                                          &cfptr->AcceptHostsLength);
      else
      if (strsame (cfptr->DirectiveName, "ActivityDays", -1))
         cfptr->ActivityNumberOfDays = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AddType", -1))
         ConfigAddType (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AddIcon", -1))
         ConfigAddIcon (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AddBlankIcon", -1))
      {
         /* use up the PARAMETER_ELBOW_ROOM :^) */
         for (sptr = cptr; *sptr; sptr++);
         *sptr++ = ' ';
         strcpy (sptr, ConfigContentTypeBlank);
         ConfigAddIcon (cfptr, cptr);
      }
      else
      if (strsame (cfptr->DirectiveName, "AddDirIcon", -1))
      {
         /* use up the PARAMETER_ELBOW_ROOM :^) */
         for (sptr = cptr; *sptr; sptr++);
         *sptr++ = ' ';
         strcpy (sptr, ConfigContentTypeDir);
         ConfigAddIcon (cfptr, cptr);
      }
      else
      if (strsame (cfptr->DirectiveName, "AddParentIcon", -1))
      {
         /* use up the PARAMETER_ELBOW_ROOM :^) */
         for (sptr = cptr; *sptr; sptr++);
         *sptr++ = ' ';
         strcpy (sptr, ConfigContentTypeParent);
         ConfigAddIcon (cfptr, cptr);
      }
      else
      if (strsame (cfptr->DirectiveName, "AddUnknownIcon", -1))
      {
         /* use up the PARAMETER_ELBOW_ROOM :^) */
         for (sptr = cptr; *sptr; sptr++);
         *sptr++ = ' ';
         strcpy (sptr, ConfigContentTypeUnknown);
         ConfigAddIcon (cfptr, cptr);
      }
      else
      if (strsame (cfptr->DirectiveName, "AuthBasic", -1))
         cfptr->AuthBasicEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AuthCacheMinutes", -1))
         cfptr->AuthCacheMinutes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AuthDigestGetLife", -1))
         cfptr->AuthDigestNonceGetLifeTime = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AuthDigestPutLife", -1))
         cfptr->AuthDigestNoncePutLifeTime = ConfigSetInteger (cfptr, cptr);
      else
      /* "AuthDigest" MUST come after other AuthDigests for obvious reasons */
      if (strsame (cfptr->DirectiveName, "AuthDigest", -1))
         cfptr->AuthDigestEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AuthFailureLimit", -1))
         cfptr->AuthFailureLimit = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "AuthRevalidateUserMinutes", -1))
         cfptr->AuthRevalidateUserMinutes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Busy", -1))
         cfptr->Busy = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheChunkKBytes", -1))
         cfptr->CacheChunkKBytes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheEntriesMax", -1))
         cfptr->CacheEntriesMax = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheFileKBytesMax", -1))
         cfptr->CacheFileKBytesMax = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheFrequentHits", -1))
         cfptr->CacheFrequentHits = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheFrequentSeconds", -1))
         cfptr->CacheFrequentSeconds = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheHashTableEntries", -1))
         cfptr->CacheHashTableEntries = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheTotalKBytesMax", -1))
         cfptr->CacheTotalKBytesMax = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CacheValidateSeconds", -1))
         cfptr->CacheValidateSeconds = ConfigSetInteger (cfptr, cptr);
      else
      /* "cache" must follow all the others for the obvious reason */
      if (strsame (cfptr->DirectiveName, "Cache", -1))
         cfptr->CacheEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "CommentedInfo", -1))
         cfptr->IncludeCommentedInfo = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclHardLimit", -1))
         cfptr->DclSubprocessHardLimit = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclScriptRunTime", -1))
         ConfigSetScriptRunTime (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclSoftLimit", -1))
         cfptr->DclSubprocessSoftLimit = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclCgiPlusLifeTime", -1))
         cfptr->DclCgiPlusLifeTime = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclZombieLifeTime", -1))
         cfptr->DclZombieLifeTime = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclSpawnAuthPriv", -1))
         cfptr->DclSpawnAuthPriv = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DclFullRequest", -1))
         cfptr->DclFullRequest = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DECnetReuseLifeTime", -1))
         cfptr->DECnetReuseLifeTime = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DECnetConnectListMax", -1))
         cfptr->DECnetConnectListMax = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirAccess", -1))
         ConfigSetDirAccess (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirDescription", -1))
      {
         /*
            [DirDescription] boolean deprecated in v5.0, noow using
            integer [DirDescriptionLines], provide fallback behaviour.
         */
         if (ConfigSetBoolean (cfptr, cptr))
            cfptr->DirDescriptionLines = 30;
         else
            cfptr->DirDescriptionLines = 0;
      }
      else
      if (strsame (cfptr->DirectiveName, "DirDescriptionLines", -1))
         cfptr->DirDescriptionLines = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirLayout", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->DefaultDirLayout,
                          sizeof(cfptr->DefaultDirLayout));
      else
      if (strsame (cfptr->DirectiveName, "DirOwner", -1))
         cfptr->DirOwnerEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirPreExpired", -1))
         cfptr->DirPreExpired = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirNoImpliedWildcard", -1))
         cfptr->DirNoImpliedWildcard = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirNoPrivIgnore", -1))
         cfptr->DirNoPrivIgnore = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirReadMeFile", -1))
         ConfigSetDirReadMeFile (cfptr, cptr);
      else
      /* "ReadMe" MUST come after the other ReadMes for obvious reasons */
      if (strsame (cfptr->DirectiveName, "DirReadMe", -1))
         ConfigSetDirReadMe (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DirWildcard", -1))
         cfptr->DirWildcardEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "DNSLookup", -1))
         cfptr->DnsLookup = ConfigSetBoolean (cfptr, cptr);
      else
      /* "Recommend" for backward compatibility */
      if (strsame (cfptr->DirectiveName, "ErrorRecommend", -1) ||
          strsame (cfptr->DirectiveName, "Recommend", -1))
         cfptr->ErrorRecommend = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "ErrorSourceInfo", -1))
         cfptr->ErrorSourceInfo = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "InputTimeout", -1))
         cfptr->InputTimeoutMinutes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "KeepAliveTimeout", -1))
         cfptr->KeepAliveTimeoutSeconds = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Logging", -1))
         cfptr->LoggingEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "LogExcludeHosts", -1))
         ConfigSetCommaList (cfptr, cptr, &cfptr->LogExcludeHostsPtr,
                                          &cfptr->LogExcludeHostsLength);
      else
      if (strsame (cfptr->DirectiveName, "LogFile", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->LogFileName, sizeof(cfptr->LogFileName));
      else
      if (strsame (cfptr->DirectiveName, "LogFormat", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->LogFormat, sizeof(cfptr->LogFormat));
      else
      if (strsame (cfptr->DirectiveName, "LogNaming", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->LogNaming, sizeof(cfptr->LogNaming));
      else
      if (strsame (cfptr->DirectiveName, "LogPeriod", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->LogPeriod, sizeof(cfptr->LogPeriod));
      else
      if (strsame (cfptr->DirectiveName, "LogPerService", -1))
         cfptr->LogPerService = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Monitor", -1))
         cfptr->MonitorEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "OutputTimeout", -1))
         cfptr->OutputTimeoutMinutes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Port", -1))
         cfptr->ServerPort = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "PutMaxKbytes", -1))
         cfptr->PutMaxKbytes = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "PutVersionLimit", -1))
         cfptr->PutVersionLimit = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Reject", -1))
         ConfigSetCommaList (cfptr, cptr, &cfptr->RejectHostsPtr,
                                          &cfptr->RejectHostsLength);
      else
      if (strsame (cfptr->DirectiveName, "RequestHistory", -1))
         cfptr->RequestHistory = ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "Search", -1))
         ConfigSetString (cfptr, cptr,
                          cfptr->KeywordSearch, sizeof(cfptr->KeywordSearch));
      else
      if (strsame (cfptr->DirectiveName, "Service", -1))
         ConfigSetCommaList (cfptr, cptr, &cfptr->ServicePtr,
                                          &cfptr->ServiceLength);
      else
      /* "ShtmlAccesses" for backward compatibility */
      if (strsame (cfptr->DirectiveName, "SSIAccesses", -1) ||
          strsame (cfptr->DirectiveName, "ShtmlAccesses", -1))
         cfptr->SsiAccessesEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      /* "ShtmlExec" for backward compatibility */
      if (strsame (cfptr->DirectiveName, "SSIExec", -1) ||
          strsame (cfptr->DirectiveName, "ShtmlExec", -1))
         cfptr->SsiExecEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      /* "SSI" MUST come after the other SSIs for obvious reasons */
      /* "Shtml" for backward compatibility */
      if (strsame (cfptr->DirectiveName, "SSI", -1) ||
          strsame (cfptr->DirectiveName, "Shtml", -1))
         cfptr->SsiEnabled = ConfigSetBoolean (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "StreamLF", -1))
         cfptr->StreamLfConversionMaxKbytes =
            ConfigSetInteger (cfptr, cptr);
      else
      if (strsame (cfptr->DirectiveName, "StreamLfPaths", -1))
         ConfigSetCommaList (cfptr, cptr, &cfptr->StreamLfPathsPtr,
                                          &cfptr->StreamLfPathsLength);
      else
      if (strsame (cfptr->DirectiveName, "Welcome", -1))
         ConfigSetWelcome (cfptr, cptr);
      else
         ConfigReportProblem (cfptr, "?", 0);
   }

   if (status == RMS$_EOF) status = SS$_NORMAL;

   /*************************/
   /* close the config file */
   /*************************/

   sys$close (&ConfigFileFab, 0, 0); 

   if (!cfptr->PutMaxKbytes)
      cfptr->PutMaxKbytes = DefaultPutMaxKbytes;
   if (!cfptr->PutVersionLimit)
      cfptr->PutVersionLimit = DefaultPutVersionLimit;

   ConfigContentTypeIcon (cfptr);

   if (VMSnok (status))
   {
      if (cfptr->RequestPtr)
      {
         ConfigReportProblem (cfptr, NULL, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, ConfigFileName, FI_LI);
   }

#ifdef DBUG
   Debug = DebugBuffer;
#endif

   return (status);
}

/*****************************************************************************/
/*
Return true or false values equivalent to the string pointed to by 'pptr'.
*/

boolean ConfigSetBoolean
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   /*********/
   /* begin */
   /*********/

   if (strsame (pptr, "YES", 3) ||
       strsame (pptr, "ON", 2) ||
       strsame (pptr, "ENABLED", 7))
      return (true);

   if (strsame (pptr, "NO", 2) ||
       strsame (pptr, "OFF", 3) ||
       strsame (pptr, "DISABLED", 8))
      return (false);

   ConfigReportProblem (cfptr, NULL, 0);
   return (false);
}

/*****************************************************************************/
/*
Free all memory allocated for a configuration.  Only used after file-based
configuration report or revision.
*/

boolean ConfigFree (struct ConfigStruct *cfptr)

{
   int  Count;
   char  *cptr;
   struct ContentTypeStruct  *clptr;
   struct IconStruct  *ilptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigFree()\n");

   clptr = cfptr->ContentTypeListHeadPtr;
   while (clptr != NULL)
   {
      cptr = (char*)clptr;
      clptr = clptr->NextPtr;
      VmFree (cptr);
   }

   ilptr = cfptr->IconListHeadPtr;
   while (ilptr != NULL)
   {
      cptr = (char*)ilptr;
      ilptr = ilptr->NextPtr;
      VmFree (cptr);
   }

   for (Count = 0; Count < cfptr->HomePageCount; Count++)
      if (cfptr->HomePageArray[Count] != NULL)
         VmFree (cfptr->HomePageArray[Count]);

   for (Count = 0; Count < cfptr->ScriptRunTimeCount; Count++)
      if (cfptr->ScriptRunTimeArray[Count] != NULL)
         VmFree (cfptr->ScriptRunTimeArray[Count]);

   if (cfptr->AcceptHostsPtr != NULL) VmFree (cfptr->AcceptHostsPtr);
   if (cfptr->LogExcludeHostsPtr != NULL) VmFree (cfptr->LogExcludeHostsPtr);
   if (cfptr->ProblemReportPtr != NULL) VmFree (cfptr->ProblemReportPtr);
   if (cfptr->RejectHostsPtr != NULL) VmFree (cfptr->RejectHostsPtr);
   if (cfptr->StreamLfPathsPtr != NULL) VmFree (cfptr->StreamLfPathsPtr);
}

/*****************************************************************************/
/*
Return an integer equivalent to the numeric string pointed to by 'pptr'.
*/

int ConfigSetInteger
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   /*********/
   /* begin */
   /*********/

   if (isdigit(*pptr)) return (atoi(pptr));
   ConfigReportProblem (cfptr, NULL, 0);
   return (-1);
}

/*****************************************************************************/
/*
Copy a simple string configuration parameter trimming leadin and trailing
white-space.
*/

ConfigSetString
(
struct ConfigStruct *cfptr,
char *pptr,
char *String,
int SizeOfString
)
{
   register char  *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetString()\n");

   zptr = (sptr = String) + SizeOfString;
   while (*pptr && ISLWS(*pptr)) pptr++;
   while (*pptr && sptr < zptr) *sptr++ = *pptr++;
   if (sptr >= zptr)
   {
      *String = '\0';
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }
   *sptr = '\0';
   if (sptr > String)
   {
      sptr--;
      while (*sptr && ISLWS(*sptr) && sptr > String) sptr--;
      *++sptr = '\0';
   }
}

/*****************************************************************************/
/*
This function formats an error report.  All lines are concatenated onto a
single string of dynamically allocated memory that (obviously) grows as
reports are added to it.  This string is then output if loading the server
configuration or is available for inclusion in an HTML page.
*/

int ConfigReportProblem
(
struct ConfigStruct *cfptr,
char *Explanation,
int StatusValue
)
{
   static $DESCRIPTOR (DirectiveFaoDsc,
                       "%HTTPD-W-CONFIG, \"!AZ\" directive !AZ at line !UL\n");
   static $DESCRIPTOR (ExplanationFaoDsc,
                       "%HTTPD-W-CONFIG, \"!AZ\" !AZ at line !UL\n");

   int  status;
   unsigned short  Length;
   char  Buffer [256],
         HtmlBuffer [512];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReportProblem()\n");

   cfptr->ProblemCount++;

   if (StatusValue)
   {
      if (VMSok (status = sys$getmsg (StatusValue, &Length, &BufferDsc, 0, 0)))
         Buffer[Length++] = '\n';
   }
   if (Explanation == NULL)
      status = sys$fao (&DirectiveFaoDsc, &Length, &BufferDsc,
                        cfptr->DirectiveName, "problem", cfptr->LineNumber);
   else
   if (Explanation[0] == '?')
      status = sys$fao (&DirectiveFaoDsc, &Length, &BufferDsc,
                        cfptr->DirectiveName, "unknown", cfptr->LineNumber);
   else
      status = sys$fao (&ExplanationFaoDsc, &Length, &BufferDsc,
                        cfptr->DirectiveName, Explanation, cfptr->LineNumber);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
      ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
   else
      Buffer[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", Buffer);

   if (cfptr->RequestPtr == NULL) fputs (Buffer, stdout);

   if ((Length = CopyToHtml (HtmlBuffer, sizeof(HtmlBuffer), Buffer, -1)) < 0)
      ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);

   cfptr->ProblemReportPtr =
      VmRealloc (cfptr->ProblemReportPtr, cfptr->ProblemReportLength+Length+1);

   /* include the terminating null */
   memcpy (cfptr->ProblemReportPtr+cfptr->ProblemReportLength,
           HtmlBuffer, Length+1);
   cfptr->ProblemReportLength += Length;
}

/*****************************************************************************/
/*
*/

ConfigSetCommaList
(
struct ConfigStruct *cfptr,
char *pptr,
char **ListPtrPtr,
int *ListLengthPtr
)
{
   register char  *cptr, *sptr;

   int  Comma,
        NewLength,
        ListLength;
   char  *ListPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetCommaList()\n");

   if (!*pptr)
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }

   ListPtr = *ListPtrPtr;
   if (ListLength = *ListLengthPtr) ListLength--;
   if (ListPtr == NULL) Comma = 0; else Comma = 1;
   for (sptr = pptr; *sptr; sptr++);
   NewLength = ListLength + (sptr - pptr) + Comma + 1;
   ListPtr = VmRealloc (ListPtr, NewLength);
   sptr = ListPtr + ListLength;
   if (Comma) *sptr++ = ',';
   while (*pptr) if (ISLWS(*pptr)) pptr++; else *sptr++ = *pptr++;
   *sptr = '\0';
   *ListPtrPtr = ListPtr;
   *ListLengthPtr = NewLength;
   if (Debug)
      fprintf (stdout, "ListPtr %d |%s|\n", *ListLengthPtr, *ListPtrPtr);
}

/*****************************************************************************/
/*
Add a mime content type corresponding to the file suffix (extension, type).

Format: "AddType <suffix> <content-type> <-|/script> [<description>]"

Examples: "AddType .html text/html - HTML"
          "AddType .txt text/plain - plain text"
          "AddType .hlb text/x-script /conan VMS help library"
          "AddType .gif image/gif - GIF image"

The directive is stored as a linked list of structures of dynamically
allocated memory containing a reformatted version of the directive string.
Function ConfigContentType() searches this list to get pointers to the content
type, any auto-script, and description, into the client request data
structure. A hash table is used to more rapidy index into the list.
*/ 

ConfigAddType
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   register char  *cptr, *sptr;
   register unsigned long  HashValue;
   register struct ContentTypeStruct  *clptr, *hclptr;

   int  status,
        Count = 0;
   char  *DescriptionPtr = "",
         *ContentTypePtr = "",
         *AutoScriptNamePtr = "",
         *SuffixPtr = "";
   char  HtmlDescription [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigAddType() |%s|\n", pptr);

   /* suffix (file type/extension) */
   HashValue = 1;
   SuffixPtr = pptr;
   /* ensure suffix is in upper-case so we don't have to do it matching */
   for (/* above */; *pptr && !ISLWS(*pptr); pptr++) *pptr = toupper(*pptr);
   Count += pptr - SuffixPtr;
   if (*pptr) *pptr++ = '\0';

   /* "mime" content type (representation) */
   while (*pptr && ISLWS(*pptr)) pptr++;
   ContentTypePtr = pptr;
   while (*pptr && !ISLWS(*pptr)) pptr++;
   Count += pptr - ContentTypePtr;
   if (*pptr) *pptr++ = '\0';

   /* (optional) script activation */
   while (*pptr && ISLWS(*pptr)) pptr++;
   AutoScriptNamePtr = pptr;
   while (*pptr && !ISLWS(*pptr)) pptr++;
   Count += pptr - AutoScriptNamePtr;
   if (*pptr) *pptr++ = '\0';

   /* (optional) description, to end of line */
   while (*pptr && ISLWS(*pptr)) pptr++;
   DescriptionPtr = pptr;
   /* find the end-of-line */
   while (*pptr) pptr++;
   if (*pptr) *pptr++ = '\0';
   /* escape any HTML-forbidden characters */
   Count += CopyToHtml (HtmlDescription, sizeof(HtmlDescription),
                        DescriptionPtr, -1);

   if (!(*SuffixPtr && *ContentTypePtr))
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }

   /* plus structure overhead, plus 4 x null terminators */
   Count += cfptr->ContentTypeStructOverhead + 4;
   clptr = (struct ContentTypeStruct*)VmGet (Count);

   sptr = (char*)clptr + cfptr->ContentTypeStructOverhead;

   clptr->SuffixPtr = sptr;
   cptr = SuffixPtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr++ = '\0';

   clptr->ContentTypePtr = sptr;
   cptr = ContentTypePtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr++ = '\0';

   clptr->AutoScriptNamePtr = sptr;
   cptr = AutoScriptNamePtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr++ = '\0';

   clptr->ContentDescriptionPtr = sptr;
   cptr = HtmlDescription;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';

   /* add to list */
   if (cfptr->ContentTypeListHeadPtr == NULL)
      cfptr->ContentTypeListHeadPtr = clptr; 
   else
      cfptr->ContentTypeListTailPtr->NextPtr = clptr; 
   cfptr->ContentTypeListTailPtr = clptr;
   clptr->NextPtr = NULL;

   /* generates an upper-case hash */
   for (cptr = SuffixPtr; *cptr && *cptr != ';'; cptr++)
      HashValue = (((*cptr)*541)^(HashValue*3)) &
                  CONTENT_TYPE_HASH_TABLE_MASK;
   if (Debug)
      fprintf (stdout, "HashValue: %d %d\n",
               HashValue, cfptr->ContentTypeHashTable[HashValue]);
   if (cfptr->ContentTypeHashTable[HashValue])
   {
      /* add to the hash-collision list */
      for (hclptr = cfptr->ContentTypeHashTable[HashValue];
           hclptr->HashCollisionPtr != NULL;
           hclptr = hclptr->HashCollisionPtr);
      hclptr->HashCollisionPtr = clptr;
   }
   else
      cfptr->ContentTypeHashTable[HashValue] = clptr;

   /* special case, default content-type for unknown suffixes */
   if (*(unsigned short*)clptr->SuffixPtr == '*\0')
   {
      cfptr->ContentTypeDefaultPtr = VmGet (strlen(DescriptionPtr+1));
      strcpy (cfptr->ContentTypeDefaultPtr, DescriptionPtr);
      clptr->ContentTypeUnknown = true;
      /* cancel type description */
      *clptr->ContentDescriptionPtr = '\0';
   }
   else
      clptr->ContentTypeUnknown = false;
}

/*****************************************************************************/
/*
After loading all content-types (i.e. after configuration) this function sets
the corresponding icon for each type.
*/ 

ConfigContentTypeIcon (struct ConfigStruct *cfptr)

{
   register struct ContentTypeStruct  *clptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigContentTypeIcon()\n");

   for (clptr = cfptr->ContentTypeListHeadPtr;
        clptr != NULL;
        clptr = clptr->NextPtr)
   {
      if (Debug) fprintf (stdout, "clptr: %d\n", clptr);
      if (clptr->ContentTypeUnknown)
         clptr->IconPtr = ConfigIconFor (ConfigContentTypeUnknown);
      else
         clptr->IconPtr = ConfigIconFor (clptr->ContentTypePtr);
   }

   ConfigBlankIconPtr = ConfigIconFor (ConfigContentTypeBlank);
   ConfigDirIconPtr = ConfigIconFor (ConfigContentTypeDir);
   ConfigParentIconPtr = ConfigIconFor (ConfigContentTypeParent);
   ConfigUnknownIconPtr = ConfigIconFor (ConfigContentTypeUnknown);
}

/*****************************************************************************/
/*
Using the hash table this function searches the list of file suffixes (types,
extensions) and content-types, returning a pointer to the content-type string
and if supplied with a content-type structure fills out the appropriate
fields.
*/ 

char* ConfigContentType
(
struct ContentTypeStruct *ctptr,
char *Suffix
)
{
   register char  *cptr, *sptr;
   register unsigned long  HashValue;
   register struct ContentTypeStruct  *clptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigContentType() |%s|\n", Suffix);

   /* generate an upper-case hash */
   HashValue = 1;
   for (cptr = Suffix; *cptr && *cptr != ';'; cptr++)
      HashValue = ((toupper(*cptr)*541)^(HashValue*3)) &
                  CONTENT_TYPE_HASH_TABLE_MASK;
   if (Debug)
      fprintf (stdout, "HashValue: %d %d\n",
               HashValue, Config.ContentTypeHashTable[HashValue]);

   clptr = Config.ContentTypeHashTable[HashValue];
   while (clptr != NULL)
   {
      if (Debug) fprintf (stdout, "clptr: %d\n", clptr);

      /* 'clptr->SuffixPtr' has already been converted to upper case! */
      cptr = clptr->SuffixPtr;
      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      sptr = Suffix;
      while (*cptr && *sptr && *sptr != ';' && *cptr == toupper(*sptr))
         { cptr++; sptr++; }

      if (*cptr || (*sptr && *sptr != ';'))
      {
         /* didn't match, traversing the hash-collision list */
         clptr = clptr->HashCollisionPtr;
         continue;
      }

      /**********/
      /* match! */
      /**********/

      /* if just getting a pointer to a content type then just return that */
      if (ctptr == NULL) return (clptr->ContentTypePtr);

      ctptr->ContentTypeUnknown = false;
      ctptr->IconPtr = clptr->IconPtr;
      ctptr->ContentTypePtr = clptr->ContentTypePtr;
      ctptr->AutoScriptNamePtr = clptr->AutoScriptNamePtr;
      ctptr->ContentDescriptionPtr = clptr->ContentDescriptionPtr;

      if (Debug)
         fprintf (stdout, "|%s|%s|%s|\n",
                  ctptr->ContentTypePtr, ctptr->AutoScriptNamePtr,
                  ctptr->ContentDescriptionPtr);

      return (clptr->ContentTypePtr);
   }

   /*************/
   /* no match! */
   /*************/

   /* if just getting a pointer to a content type then just return that */
   if (ctptr == NULL) return (ConfigContentTypeUnknown);

   ctptr->ContentTypeUnknown = true;
   ctptr->IconPtr = ConfigUnknownIconPtr;
   ctptr->ContentTypePtr = ConfigContentTypeUnknown;
   ctptr->AutoScriptNamePtr = ctptr->ContentDescriptionPtr = "";
   return (ConfigContentTypeUnknown);
}

/*****************************************************************************/
/*
Add a URL for an icon corresponding to the mime content type.

Format:   "AddIcon <URL> <alternative-text> <content-type-template>"

Examples: "AddIcon /icon/httpd/text.xbm [TXT] text/plain"
          "AddIcon /icon/httpd/doc.xbm [HTM] text/html"
          "AddIcon /icon/httpd/image.xbm [IMG] image/jpeg"

The template can contain a wildcard in place of a string in either side of the 
template slash.  In this case the wildcard matches any string on that side.

The directive is stored as a linked-list of structures of dynamically
allocated memory containing a reformatted version of the directive string,
including a complete HTML anchor for the URL. Function ConfigIconFor()
searches this list to return a pointer to the anchor. A hash table is used to
more rapidy index into the list.
*/ 

ConfigAddIcon
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   register unsigned long  HashValue;
   register char  *cptr;
   register struct IconStruct  *ilptr, *hilptr;

   int  Count = 0;
   char  *AltTextPtr,
         *IconUrlPtr,
         *ContentTypePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigAddIcon()\n");

   IconUrlPtr = pptr;
   while (*pptr && !ISLWS(*pptr)) pptr++;
   Count = pptr - IconUrlPtr;
   if (*pptr) *pptr++ = '\0';

   /* alternative text */
   while (*pptr && ISLWS(*pptr)) pptr++;
   AltTextPtr = pptr;
   while (*pptr && !ISLWS(*pptr))
   {
      /* underscores represent spaces! (especially for the "blank" icon) */
      if (*pptr == '_') *pptr = ' ';
      pptr++;
   }
   Count += pptr - AltTextPtr;
   if (*pptr) *pptr++ = '\0';

   /* template */
   while (*pptr && ISLWS(*pptr)) pptr++;
   ContentTypePtr = pptr;
   /* force to lower case to save a tolower() in ConfigIconFor() */
   for (/* above */; *pptr && !ISLWS(*pptr); pptr++)
      *pptr = tolower(*pptr);
   Count += pptr - ContentTypePtr;
   if (*pptr) *pptr++ = '\0';

   if (!(*IconUrlPtr && *AltTextPtr && *ContentTypePtr))
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }

   /* plus structure overhead, plus the HTML IMG element used below */
   Count += cfptr->IconStructOverhead + 32;
   ilptr = (struct IconStruct*)VmGet (Count);

   sprintf ((char*)ilptr + cfptr->IconStructOverhead,
            "%s <IMG ALIGN=top SRC=\"%s\" ALT=\"%s\">",
            ContentTypePtr, IconUrlPtr, AltTextPtr);
   if (Debug)
      fprintf (stdout, "|%s|%s|%s|%s|\n",
               ContentTypePtr, IconUrlPtr, AltTextPtr,
               (char*)ilptr + cfptr->IconStructOverhead);

   /* add to list */
   if (cfptr->IconListHeadPtr == NULL)
      cfptr->IconListHeadPtr = ilptr; 
   else
      cfptr->IconListTailPtr->NextPtr = ilptr; 
   cfptr->IconListTailPtr = ilptr;
   ilptr->NextPtr = NULL;

   /* generates a lower-case hash */
   HashValue = 1;
   for (cptr = ContentTypePtr; *cptr; cptr++)
      HashValue = (((*cptr)*541)^(HashValue*3)) &
                  ICON_HASH_TABLE_MASK;
   if (Debug)
      fprintf (stdout, "HashValue: %d %d\n",
               HashValue, cfptr->ContentTypeHashTable[HashValue]);
   if (cfptr->IconHashTable[HashValue])
   {
      /* add to the hash-collision list */
      for (hilptr = cfptr->IconHashTable[HashValue];
           hilptr->HashCollisionPtr != NULL;
           hilptr = hilptr->HashCollisionPtr);
      hilptr->HashCollisionPtr = ilptr;
   }
   else
      cfptr->IconHashTable[HashValue] = ilptr;
}

/*****************************************************************************/
/*
Using the hash table this function searches the icons for one corresponding to
the supplied mime content type. The directive content type and associated HTML
anchor (containing the original configuration-supplied URL) is stored as an
list of structures dynamically allocated memory containing a string.
*/

char* ConfigIconFor (char *ContentType)

{
   register char  *cptr, *sptr;
   register unsigned long  HashValue;
   register struct IconStruct  *ilptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigIconFor() |%s|\n", ContentType);

   /* generates a lower-case hash */
   HashValue = 1;
   for (cptr = ContentType; *cptr; cptr++)
      HashValue = ((tolower(*cptr)*541)^(HashValue*3)) &
                  ICON_HASH_TABLE_MASK;
   if (Debug)
      fprintf (stdout, "HashValue: %d %d\n",
               HashValue, Config.IconHashTable[HashValue]);

   ilptr = Config.IconHashTable[HashValue];
   while (ilptr != NULL)
   {
      if (Debug) fprintf (stdout, "ilptr: %d\n", ilptr);

      cptr = (char*)ilptr + Config.IconStructOverhead;
      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      sptr = ContentType;
      while (*cptr && *cptr != '*' && *sptr && *cptr == tolower(*sptr))
         { cptr++; sptr++; }
      if (*cptr == ' ' && !*sptr)
      {
         /* return a pointer to the HTML IMG element for the icon */
         if (*cptr) cptr++;
         return (cptr);
      }
      if (*cptr == '*')
      {
         /* template contained a wildcard on the left side of the slash */
         while (*cptr && *cptr != '/' && !ISLWS(*cptr)) cptr++;
         if (*cptr) cptr++;
         /* find the right side of the content type */
         while (*sptr && *sptr != '/') sptr++;
         if (*sptr) sptr++;
         /* compare the right side of the slash */
         while (*cptr && *cptr != '*' && *sptr && *cptr == tolower(*sptr))
            { cptr++; sptr++; }
         if (*cptr == '*')
         {
            while (*cptr == '*') cptr++;
            while (*sptr) sptr++;
         }
         if (*cptr == ' ' && !*sptr)
         {
            /* return a pointer to the HTML IMG element for the icon */
            if (*cptr) cptr++;
            return (cptr);
         }
      }

      /* didn't match, traversing the hash-collision list */
      ilptr = ilptr->HashCollisionPtr;
   }

   /* nothing found, essentially an error! */
   return ("<FONT COLOR=\"#ff0000\"><B>[?]</B></FONT>");
}

/*****************************************************************************/
/*
Accept or reject the client connection.  Lists of hosts must be a comma-
separated string.  A wildcard '*' in a domain specification matches any string 
that occurs between the dots.
Example: "host1.network1.domain1,*.network2.domain1,*.*.domain2". 
*/ 

boolean ConfigAcceptClientHostName
(
char *ClientInternetAddress,
char *ClientHostName
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ConfigAcceptConnection() |%s|\n", ClientHostName);

   if (Config.AcceptHostsPtr == NULL && Config.RejectHostsPtr == NULL)
      return (true);

   if (!ClientHostName[0] || ClientHostName[0] == '?') return (false);

   /******************/
   /* hosts accepted */
   /******************/

   if (Config.AcceptHostsPtr != NULL)
   {
      sptr = Config.AcceptHostsPtr;
      while (*sptr)
      {
         /** if (Debug) fprintf (stdout, "sptr |%s|\n", sptr); **/
         if (isdigit(*sptr))
            cptr = ClientInternetAddress;
         else
            cptr = ClientHostName;
         while (*cptr)
         {
            if (*sptr == ',') break;
            if (*sptr == '*')
            {
               while (*sptr && *sptr == '*' && *sptr != ',') sptr++;
               while (*cptr && *cptr != *sptr) cptr++;
            }
            if (tolower(*cptr) != tolower(*sptr)) break;
            if (*cptr) cptr++;
            if (*sptr) sptr++;
         }
         if (!*cptr && (!*sptr || *sptr == ',')) break;
         while (*sptr && *sptr != ',') sptr++;
         if (*sptr) sptr++;
      }
      /* if it was NOT found then reject the connection request */
      if (!(!*cptr && (!*sptr || *sptr == ',')))
         return (false);
   }

   /******************/
   /* hosts rejected */
   /******************/

   if (Config.RejectHostsPtr != NULL)
   {
      sptr = Config.RejectHostsPtr;
      while (*sptr)
      {
         /** if (Debug) fprintf (stdout, "sptr |%s|\n", sptr); **/
         cptr = ClientHostName;
         while (*cptr)
         {
            if (*sptr == ',') break;
            if (*sptr == '*')
            {
               while (*sptr && *sptr == '*' && *sptr != ',') sptr++;
               while (*cptr && *cptr != *sptr) cptr++;
            }
            if (tolower(*cptr) != tolower(*sptr)) break;
            if (*cptr) cptr++;
            if (*sptr) sptr++;
         }
         if (!*cptr && (!*sptr || *sptr == ',')) break;
         while (*sptr && *sptr != ',') sptr++;
         if (*sptr) sptr++;
      }
      /* if it WAS found then reject the connection request */
      if (!*cptr && (!*sptr || *sptr == ','))
         return (false);
   }

   /* accept the connection */
   return (true);
}

/*****************************************************************************/
/*
Set directory access booleans.  Allowed values OFF, ON, SELECTIVE.
*/

ConfigSetDirAccess
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirAccess()\n");

   if (strsame (pptr, "YES", 3) ||
       strsame (pptr, "ON", 2) ||
       strsame (pptr, "ENABLED", 7))
   {
      cfptr->DirAccess = true;
      cfptr->DirAccessSelective = false;
   }
   else
   if (strsame (pptr, "NO", 2) ||
       strsame (pptr, "OFF", 3) ||
       strsame (pptr, "DISABLED", 8))
      cfptr->DirAccess = cfptr->DirAccessSelective = false;
   else
   if (strsame (pptr, "SELECTIVE", 9))
      cfptr->DirAccess = cfptr->DirAccessSelective = true;
   else
      ConfigReportProblem (cfptr, NULL, 0);
}

/*****************************************************************************/
/*
Set directory README file booleans.  Allowed values OFF, TOP, BOTTOM.
*/

ConfigSetDirReadMe
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   if (Debug) fprintf (stdout, "ConfigSetDirReadMe()\n");

   if (strsame (pptr, "NO", 2) ||
       strsame (pptr, "OFF", 3) ||
       strsame (pptr, "DISABLED", 8))
      cfptr->DirReadMeBottom = cfptr->DirReadMeTop = false;
   else
   if (strsame (pptr, "BOTTOM", 6))
   {
      cfptr->DirReadMeTop = false;
      cfptr->DirReadMeBottom = true;
   }
   else
   if (strsame (pptr, "TOP", 3))
   {
      cfptr->DirReadMeTop = true;
      cfptr->DirReadMeBottom = false;
   }
   else
      ConfigReportProblem (cfptr, NULL, 0);
}

/*****************************************************************************/
/*
Add a "read-me" file name.

Format:   "ReadMeFile file.suffix"

Examples: "ReadMeFile README.HTML"
          "ReadMeFile README.TXT"
          "ReadMeFile README."

The directive is stored as an array of pointers to dynamically allocated 
memory containing the "read-me" file name.  Function ConfigReadMeFile() gets 
these by index number.
*/

ConfigSetDirReadMeFile
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   int  status,
        Count;
   char  *NamePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirReadMeFile()\n");

   if (cfptr->ReadMeFileCount >= MaxConfigReadMeFiles)
   {
      ConfigReportProblem (cfptr, "maximum read-me files", 0);
      return;
   }

   NamePtr = pptr;
   while (*pptr && !ISLWS(*pptr))
   {
      *pptr = toupper(*pptr);
      pptr++;
   }
   Count = pptr - NamePtr;
   *pptr = '\0';

   if (!*NamePtr)
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }

   if (Debug) fprintf (stdout, "VmGet()ing %d bytes\n", Count+1);
   cfptr->ReadMeFileArray[cfptr->ReadMeFileCount] = VmGet(Count+1);

   strcpy (cfptr->ReadMeFileArray[cfptr->ReadMeFileCount], NamePtr);
   if (Debug)
      fprintf (stdout, "|%s|\n",
               cfptr->ReadMeFileArray[cfptr->ReadMeFileCount]);
   cfptr->ReadMeFileCount++;
}

/*****************************************************************************/
/*
Return a pointer to the "read-me" file name stored against the index number of 
the array.  If none exists return a null string.  The calling function would 
use this by stepping through the index numbers 0 to n, until the null string 
was returned indicating the possible "read-me" file names were exhausted.
*/

char* ConfigReadMeFile (int Number)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReadMeFile |%d|\n", Number);

   if (Number >= 0 && Number < Config.ReadMeFileCount)
      return (Config.ReadMeFileArray[Number]);
   else
      return ("");
}

/*****************************************************************************/
/*
Add a script type.

Format:   "file-type verb"

Examples: ".PL PERL"
          ".PL $PERL_EXE:PERL.EXE"

The directive is stored as an array of pointers to dynamically allocated 
memory containing definition.  Function DclFindScript() uses this information.
The array entries comprise a reconstructed string from the directive.  The
verb part of the directive can have a leading '@' or '$' indicating that the
verb should be create before invoking the script.  If not having these leading
characters the verb is considered to be globally available.  The leading '@'
and '$' verbs are reconstructed from "$perl_exe:perl.exe" into
"PERL=\"$perl_exe:perl.exe\"".  From this the DCL module creates an
excecutable DCL command sequence.
*/ 

ConfigSetScriptRunTime
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   register char  *cptr, *sptr;

   int  status,
        Count;
   char  Scratch [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetScriptRunTime()\n");

   if (cfptr->ScriptRunTimeCount >= MaxConfigScriptRunTimes)
   {
      ConfigReportProblem (cfptr, "maximum script types", 0);
      return;
   }

   sptr = Scratch;
   if (*pptr != '.') *sptr++ = '.';
   while (*pptr && !ISLWS(*pptr)) *sptr++ = toupper(*pptr++);
   if (sptr > Scratch && *(sptr-1) != ';') *sptr++ = ';';
   cfptr->ScriptRunTimeLengthArray[cfptr->ScriptRunTimeCount] = sptr - Scratch;
   *sptr++ = ' ';
   while (*pptr && ISLWS(*pptr)) pptr++;
   if (!*pptr)
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }
   if (*pptr == '@' || *pptr == '$')
   {
      *sptr++ = *(cptr = pptr);
      while (*cptr != ':') cptr++;
      if (*cptr) cptr++;
      if (*cptr == '[')
      {
         while (*cptr && *cptr != ']') cptr++;
         if (*cptr) cptr++;
      }
      while (*cptr && *cptr != '.' && !ISLWS(*cptr))
         *sptr++ = toupper(*cptr++);
      *sptr++ = '=';
      *sptr++ = '\"';
      while (*pptr)
      {
         /* escape quotes for DCL */
         if (*pptr == '\"') *sptr++ = '\"';
         *sptr++ = *pptr++;
      }
      *sptr++ = '\"';
      *sptr++ = '\0';
      Count = sptr - Scratch;
   }
   else
   {
      while (*pptr) *sptr++ = *pptr++;
      *sptr++ = '\0';
      Count = sptr - Scratch;
   }

   if (Debug) fprintf (stdout, "VmGet()ing %d bytes\n", Count);
   cfptr->ScriptRunTimeArray[cfptr->ScriptRunTimeCount] = VmGet (Count);

   strcpy (cfptr->ScriptRunTimeArray[cfptr->ScriptRunTimeCount], Scratch);
   if (Debug)
      fprintf (stdout, "|%s|\n",
               cfptr->ScriptRunTimeArray[cfptr->ScriptRunTimeCount]);
   cfptr->ScriptRunTimeCount++;
}

/*****************************************************************************/
/*
Add a welcome (home) page file name.

Format:   "Welcome file.suffix"

Examples: "Welcome HOME.HTML"
          "Welcome HOME.HTM"
          "Welcome HOME.MENU"

The directive is stored as an array of pointers to dynamically allocated 
memory containing the home page file name.  Function ConfigHomePage() gets 
these by index number.
*/ 

ConfigSetWelcome
(
struct ConfigStruct *cfptr,
char *pptr
)
{
   int  status,
        Count;
   char  *WelcomePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetWelcome()\n");

   if (cfptr->HomePageCount >= MaxConfigHomePages)
   {
      ConfigReportProblem (cfptr, "maximum welcome files", 0);
      return;
   }

   WelcomePtr = pptr;
   while (*pptr && !ISLWS(*pptr))
   {
      *pptr = toupper(*pptr);
      pptr++;
   }
   Count = pptr - WelcomePtr;
   *pptr = '\0';

   if (!*WelcomePtr)
   {
      ConfigReportProblem (cfptr, NULL, 0);
      return;
   }

   if (Debug) fprintf (stdout, "VmGet()ing %d bytes\n", Count+1);
   cfptr->HomePageArray[cfptr->HomePageCount] = VmGet (Count+1);

   strcpy (cfptr->HomePageArray[cfptr->HomePageCount], WelcomePtr);
   if (Debug)
      fprintf (stdout, "|%s|\n", cfptr->HomePageArray[cfptr->HomePageCount]);
   cfptr->HomePageCount++;
}

/*****************************************************************************/
/*
Return a pointer to the home page (welcome) file name stored against the index 
number of the array.  If none exists return a null string.  The calling 
function would use this by stepping through the index numbers 0 to n, until 
the null string was returned indicating the possible home page file names were 
exhausted.
*/

char* ConfigHomePage (int Number)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigHomePage |%d|\n", Number);

   if (Number >= 0 && Number < Config.HomePageCount)
      return (Config.HomePageArray[Number]);
   else
      return ("");
}

/*****************************************************************************/
/*
A server administration report on the server's configuration. This function
just wraps the reporting function, loading a temporary database if necessary
for reporting from the configuration file.
*/ 

ConfigReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean  UseServerDatabase
)
{
   int  status;
   struct ConfigStruct  LocalConfig;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReport()\n");

   if (UseServerDatabase)
      ConfigReportNow (rqptr, &Config);
   else
   {
      memset (&LocalConfig, 0, sizeof(struct ConfigStruct));
      /* indicate it's being used for a report */
      Config.RequestPtr = rqptr;
      Configure (&LocalConfig);
      ConfigReportNow (rqptr, &LocalConfig);
      ConfigFree (&LocalConfig);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
Return a report on the HTTPd server's configuration ... now!  This function
blocks while executing.
*/ 

ConfigReportNow
(
struct RequestStruct *rqptr,
struct ConfigStruct *cfptr
)
{
/* macro to push booleans onto the sys$faol() parameter list */
#define REPBOOL(b) \
if (b) *vecptr++ = "[enabled]"; else *vecptr++ = "[disabled]";

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Server Configuration</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Server Configuration</H2>\n\
!AZ\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (ReportSourceFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>File:</TH>\
<TD ALIGN=left>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH>\
<TH>[<A HREF=\"!AZ!AZ\">Edit</A>]</TH>\
</TR>\n\
<TR><TH ALIGN=right>!AZ</TH>\
<TD ALIGN=left>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Service</TH></TR>\n");

   static $DESCRIPTOR (ServiceFaoDsc, "<TR><TD>!AZ//!AZ</TD></TR>\n");
   static $DESCRIPTOR (ServiceListFaoDsc, "<TR><TD>!AZ</TD></TR>\n");

   static char  NotNone [] = "<FONT SIZE=-1><I>(none)</I></FONT>\n",
                ServiceNone [] =
                   "<TR><TD><FONT SIZE=-1><I>(none)</I></FONT></TD></TR>\n";

   static $DESCRIPTOR (ReportGeneralFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>GMT</TH></TR>\n\
<TR><TH>Difference</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Current</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>General</TH></TR>\n\
<TR><TH>Busy</TH><TD>!UL</TD><TD><FONT SIZE=-1>connections</FONT></TR>\
<TR><TH>Request History</TH><TD>!UL</TD></TR>\n\
<TR><TH>Activity History</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>days</FONT></TD></TR>\n\
<TR><TH>Commments</TH><TD>!AZ</TD></TR>\n\
<TR><TH>DNS Lookup</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Error Recommend</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Error Source Information</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Monitor</TH><TD>!AZ</TD></TR>\n\
<TR><TH>IP Port</TH><TD>!UL</TD></TR>\n\
<TR><TH>StreamLF</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>StreamLF Paths</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH>Search Script</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH><FONT SIZE=-1>PUT/POST</FONT> Max.Length</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH><FONT SIZE=-1>PUT/POST</FONT> Files</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>version limit</FONT></TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Log</TH></TR>\n\
<TR><TH>Logging</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Format</TH><TD>!AZ</TD>\
<TR><TH>Naming</TH><TD>!AZ</TD>\
<TR><TH>Period</TH><TD>!AZ</TD>\
<TR><TH>Per-Service</TH><TD>!AZ</TD>\
<TR><TH>File</TH><TD>!AZ</TD>\
<TR><TH>Exclude Hosts</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Cache</TH></TR>\n\
<TR><TH>Caching</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Hash Table</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>entries</FONT></TD></TR>\n\
<TR><TH>Max Entries</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>files</FONT></TD></TR>\n\
<TR><TH>Max Cache Size</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Max File Size</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Chunk Size</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>Bytes</FONT></TD></TR>\n\
<TR><TH>Validate</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>seconds</FONT></TD></TR>\n\
<TR><TH>Frequent</TH><TD COLSPAN=2>\
!UL <FONT SIZE=-1>hits, last within </FONT> \
!UL <FONT SIZE=-1>seconds</FONT></TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Timeouts</TH></TR>\n\
<TR><TH>Input</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Output</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Keep-Alive</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>seconds</FONT></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (ReportWelcomeFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Welcome (Home) Pages</TH></TR>\n");

   static $DESCRIPTOR (ReportHostsFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Hosts</TH></TR>\n\
<TR><TH>Accept</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Reject</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=5>Authentication</TH></TR>\n\
<TR><TH COLSPAN=2>Cache</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH COLSPAN=2>Revalidate User</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH COLSPAN=2>Failure Limit</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>attempts</FONT></TD></TR>\n\
<TR><TH COLSPAN=2>Basic</TH><TD>!AZ</TD></TR>\n\
<TR><TH ROWSPAN=3>Digest</TH><TH>Use</TH><TD>!AZ</TD></TR>\n\
<TR><TH>GET lifetime</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>PUT lifetime</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Scripting &nbsp;(DCL & DECnet)</TH></TR>\n\
<TR><TH>Spawn Authorized Privileges</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Supply Header and Body</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Soft Limit</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>subprocesses</FONT></TD></TR>\n\
<TR><TH>Hard Limit</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>subprocesses</FONT></TD></TR>\n\
<TR><TH>Zombie Life-Time</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>CGIplus Life-Time</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>DECnet Reuse Life-Time</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>DECnet Connect List Max</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>connections</FONT></TD></TR>\n\
<TR><TH ROWSPAN=!UL>Script Run-Time</TH>\
<TH>File Type</TH><TH>Verb</TH></TR>\n");

   static $DESCRIPTOR (ReportSsiFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Server-Side Includes</TH></TR>\n\
<TR><TH>SSI</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Exec</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Accesses</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Directory</TH></TR>\n\
<TR><TH>Access</TH><TD>!AZ</TD></TR>\n\
<TR><TH>HTML Description</TH><TD>!UL</TD>\
<TD><FONT SIZE=-1>lines</FONT></TD></TR>\n\
<TR><TH>Layout</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Owner</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Pre-Expired</TH><TD>!AZ</TD></TR>\n\
<TR><TH>No Implied Wildcard</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Protection Violations Ignored</TH><TD>!AZ</TD></TR>\n\
<TR><TH>Readme</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n\
<TR><TH>Wildcard</TH><TD>!AZ</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Readme Files</TH></TR>\n");

   static $DESCRIPTOR (ReportIconsFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=4>Directory Icons</TH></TR>\n\
<TR><TH>content-type</TH><TH>icon</TH><TH>path</TH>\
<TH>alternate text</TH></TR>\n");

   static $DESCRIPTOR (ReportContentTypeFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=4>Content-Type</TH></TR>\n\
<TR><TH>suffix</TH><TH>content-type</TH><TH>icon</TH>\
<TH>auto-script</TH><TH>description</TH></TR>\n");

   static $DESCRIPTOR (ContentTypeFaoDsc,
"<TR><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD><TD>!AZ</TD></TR>\n");

   static char  EndPage [] =
"</BODY>\n\
</HTML>\n";

   register int  idx;
   register unsigned long  *vecptr;
   register char  *cptr, *sptr;
   register struct ContentTypeStruct  *clptr;
   register struct IconStruct  *ilptr;
   register struct ServiceStruct  *svptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [128];
   char  Buffer [8192],
         HtmlIconAltText [256],
         IconAltText [256],
         IconPath [256];
   char  *AutoScriptNamePtr,
         *ContentDescriptionPtr;
   struct ContentTypeStruct  ContentType;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReportNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (cfptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = cfptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (cfptr == &Config)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr++ = cfptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   cptr = MapVmsPath (cfptr->LoadFileName);

   vecptr = FaoVector;

   /* source file */
   if (cfptr == &Config)
      *vecptr++ = "Server";
   else
      *vecptr++ = "File";
   *vecptr++ = cfptr->LoadFileName;
   *vecptr++ = cptr;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = cptr;
   if (cfptr == &Config)
   {
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&cfptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&cfptr->RevBinTime, 20);
   }

   status = sys$faol (&ReportSourceFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /************/
   /* services */
   /************/

   if (cfptr->ServicePtr == NULL)
      NetWriteBuffered (rqptr, 0, ServiceNone, sizeof(ServiceNone)-1);
   else
   if (cfptr != &Config)
   {
      for (cptr = cfptr->ServicePtr; *cptr; cptr++)
      {
         sptr = cptr;
         while (*cptr && *cptr != ',') cptr++;
         if (*cptr)
         {
            *cptr = '\0';
            status = sys$fao (&ServiceListFaoDsc, &Length, &BufferDsc, sptr);
            *cptr = ',';
         }
         else
            status = sys$fao (&ServiceListFaoDsc, &Length, &BufferDsc, sptr);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            rqptr->ErrorTextPtr = "sys$fao()";
            ErrorVmsStatus (rqptr, status, FI_LI);
            return;
         }
         Buffer[Length] = '\0';
         NetWriteBuffered (rqptr, 0, Buffer, Length);
      }
   }
   else
   {
      for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
      {
         vecptr = FaoVector;
         /* the service protocol, host and port from the service structure */
         *vecptr++ = svptr->RequestSchemeNamePtr;
         *vecptr++ = svptr->ServerHostPort;

         status = sys$faol (&ServiceFaoDsc, &Length, &BufferDsc, &FaoVector);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            rqptr->ErrorTextPtr = "sys$faol()";
            ErrorVmsStatus (rqptr, status, FI_LI);
            return;
         }
         Buffer[Length] = '\0';
         NetWriteBuffered (rqptr, 0, Buffer, Length);
      }
   }

   /***********/
   /* general */
   /***********/

   vecptr = FaoVector;

   /* GMT */
   *vecptr++ = TimeGmtString;
   *vecptr++ = rqptr->GmDateTime;

   /* general */
   *vecptr++ = cfptr->Busy;
   *vecptr++ = cfptr->RequestHistory;
   *vecptr++ = cfptr->ActivityNumberOfDays;
   REPBOOL (cfptr->IncludeCommentedInfo)
   REPBOOL (cfptr->DnsLookup)
   REPBOOL (cfptr->ErrorRecommend)
   REPBOOL (cfptr->ErrorSourceInfo)
   REPBOOL (cfptr->MonitorEnabled)
   *vecptr++ = cfptr->ServerPort;

   *vecptr++ = cfptr->StreamLfConversionMaxKbytes;
   if (cfptr->StreamLfPathsPtr == NULL)
      *vecptr++ = NotNone;
   else
      *vecptr++ = ConfigCommaList (rqptr, cfptr->StreamLfPathsPtr, ',');

   *vecptr++ = cfptr->KeywordSearch;
   *vecptr++ = cfptr->PutMaxKbytes;
   *vecptr++ = cfptr->PutVersionLimit;

   /* logging */
   REPBOOL (cfptr->LoggingEnabled)
   if (cfptr->LogFormat[0])
      *vecptr++ = cfptr->LogFormat;
   else
      *vecptr++ = NotNone;
   if (cfptr->LogNaming[0])
      *vecptr++ = cfptr->LogNaming;
   else
      *vecptr++ = NotNone;
   if (cfptr->LogPeriod[0])
      *vecptr++ = cfptr->LogPeriod;
   else
      *vecptr++ = NotNone;
   REPBOOL (cfptr->LogPerService)
   if (cfptr->LogFileName[0])
      *vecptr++ = cfptr->LogFileName;
   else
      *vecptr++ = NotNone;

   if (cfptr->LogExcludeHostsPtr == NULL)
      *vecptr++ = NotNone;
   else
      *vecptr++ = ConfigCommaList (rqptr, cfptr->LogExcludeHostsPtr, ',');

   /* cache */
   REPBOOL (cfptr->CacheEnabled)
   *vecptr++ = cfptr->CacheHashTableEntries;
   *vecptr++ = cfptr->CacheEntriesMax;
   *vecptr++ = cfptr->CacheTotalKBytesMax;
   *vecptr++ = cfptr->CacheFileKBytesMax;
   *vecptr++ = cfptr->CacheChunkKBytes;
   *vecptr++ = cfptr->CacheValidateSeconds;
   *vecptr++ = cfptr->CacheFrequentHits;
   *vecptr++ = cfptr->CacheFrequentSeconds;

   /* timeouts */
   *vecptr++ = cfptr->InputTimeoutMinutes;
   *vecptr++ = cfptr->OutputTimeoutMinutes;
   *vecptr++ = cfptr->KeepAliveTimeoutSeconds;

   status = sys$faol (&ReportGeneralFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /*****************************/
   /* welcome (home page) files */
   /*****************************/

   NetWriteBuffered (rqptr, 0, ReportWelcomeFaoDsc.dsc$a_pointer,
                              ReportWelcomeFaoDsc.dsc$w_length);

   for (idx = 0; idx < cfptr->HomePageCount; idx++)
   {
      memcpy (sptr = Buffer, "<TR><TD>", 8);
      sptr += 8;
      for (cptr = cfptr->HomePageArray[idx]; *cptr; *sptr++ = *cptr++);
      memcpy (sptr, "</TD></TR>\n", 12);  /* includes terminating null */
      sptr += 11;
      NetWriteBuffered (rqptr, 0, Buffer, sptr-Buffer);
   }

   NetWriteBuffered (rqptr, 0, "</TABLE>\n", 9);

   /*********************/
   /* host access, etc. */
   /*********************/

   vecptr = FaoVector;

   if (cfptr->AcceptHostsPtr == NULL)
      *vecptr++ = NotNone;
   else
      *vecptr++ = ConfigCommaList (rqptr, cfptr->AcceptHostsPtr, ',');

   if (cfptr->RejectHostsPtr == NULL)
      *vecptr++ = NotNone;
   else
      *vecptr++ = ConfigCommaList (rqptr, cfptr->RejectHostsPtr, ',');

   /* authentication */
   *vecptr++ = cfptr->AuthCacheMinutes;
   *vecptr++ = cfptr->AuthRevalidateUserMinutes;
   *vecptr++ = cfptr->AuthFailureLimit;
   REPBOOL (cfptr->AuthBasicEnabled)
   REPBOOL (cfptr->AuthDigestEnabled)
   *vecptr++ = cfptr->AuthDigestNonceGetLifeTime;
   *vecptr++ = cfptr->AuthDigestNonceGetLifeTime;

   /* DCL/Scripting */
   REPBOOL (cfptr->DclSpawnAuthPriv)
   REPBOOL (cfptr->DclFullRequest)
   *vecptr++ = cfptr->DclSubprocessSoftLimit;
   *vecptr++ = cfptr->DclSubprocessHardLimit;
   *vecptr++ = cfptr->DclZombieLifeTime;
   *vecptr++ = cfptr->DclCgiPlusLifeTime;
   *vecptr++ = cfptr->DECnetReuseLifeTime;
   *vecptr++ = cfptr->DECnetConnectListMax;
   if (cfptr->ScriptRunTimeCount)
      *vecptr++ = cfptr->ScriptRunTimeCount+1;
   else
      *vecptr++ = 2;

   status = sys$faol (&ReportHostsFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /****************/
   /* script types */
   /****************/

   if (cfptr->ScriptRunTimeCount)
   {
      for (idx = 0; idx < cfptr->ScriptRunTimeCount; idx++)
      {
         sptr = Buffer;
         memcpy (sptr, "<TR><TD>", 8);
         sptr += 8;
         for (cptr = cfptr->ScriptRunTimeArray[idx];
              *cptr && *cptr != ';';
              *sptr++ = *cptr++);
         memcpy (sptr, "</TD><TD>", 9);
         sptr += 9;
         if (*cptr) cptr++;
         if (*cptr) cptr++;
         if (*cptr == '@' || *cptr == '$')
         {
            while (*cptr && *cptr != '\"') cptr++;
            if (*cptr) cptr++;
            while (*cptr)
            {
               /* unescape any quotes */
               if (cptr[0] == '\"' && cptr[1] == '\"')
                  cptr++;
               else
               if (*cptr == '\"')
                  break;
               *sptr++ = *cptr++;
            }
         }
         else
            while (*cptr) *sptr++ = *cptr++;
         memcpy (sptr, "</TD></TR>\n", 12);
         sptr += 11;
         NetWriteBuffered (rqptr, 0, Buffer, sptr-Buffer);
      }
   }
   else
   {
      static char  ScriptRunTimeNone [] =
"<TR><TD COLSPAN=2><FONT SIZE=-1><I>(none)</I></FONT></TD></TR>\n";

      NetWriteBuffered (rqptr, 0, ScriptRunTimeNone,
                                 sizeof(ScriptRunTimeNone)-1);
   }

   /*************/
   /* SSI, etc. */
   /*************/

   vecptr = FaoVector;

   /* SSI (.shtml) */
   REPBOOL (cfptr->SsiEnabled)
   REPBOOL (cfptr->SsiExecEnabled)
   REPBOOL (cfptr->SsiAccessesEnabled)

   REPBOOL (cfptr->DirAccess)
   *vecptr++ = cfptr->DirDescriptionLines;
   *vecptr++ = cfptr->DefaultDirLayout;
   REPBOOL (cfptr->DirOwnerEnabled)
   REPBOOL (cfptr->DirPreExpired)
   REPBOOL (cfptr->DirNoImpliedWildcard)
   REPBOOL (cfptr->DirNoPrivIgnore)
   REPBOOL (cfptr->DirReadMeTop | cfptr->DirReadMeBottom)
   if (cfptr->DirReadMeTop)
      *vecptr++ = "top";
   else
   if (cfptr->DirReadMeBottom)
      *vecptr++ = "bottom";
   else
      *vecptr++ = "";
   REPBOOL (cfptr->DirWildcardEnabled)

   status = sys$faol (&ReportSsiFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /*********************/
   /* directory (contd) */
   /*********************/

   for (idx = 0; idx < cfptr->ReadMeFileCount; idx++)
   {
      memcpy (sptr = Buffer, "<TR><TD>", 8);
      sptr += 8;
      for (cptr = cfptr->ReadMeFileArray[idx]; *cptr; *sptr++ = *cptr++);
      memcpy (sptr, "</TD></TR>\n", 12);  /* includes terminating null */
      sptr += 11;
      NetWriteBuffered (rqptr, 0, Buffer, sptr - Buffer);
   }

   NetWriteBuffered (rqptr, 0, "</TABLE>\n", 9);

   NetWriteBuffered (rqptr, 0, ReportIconsFaoDsc.dsc$a_pointer,
                              ReportIconsFaoDsc.dsc$w_length);

   /*********/
   /* icons */
   /*********/

   ilptr = cfptr->IconListHeadPtr;
   while (ilptr != NULL)
   {
      /* find the ALT="" text from the icon IMG */ 
      IconAltText[0] = IconPath[0] = '\0';
      cptr = (char*)ilptr + cfptr->IconStructOverhead;
      while (*cptr && !ISLWS(*cptr)) cptr++;
      if (*cptr) cptr++;
      while (*cptr)
      {
         if (*cptr == 'S' && !memcmp (cptr, "SRC=\"", 5))
         {
            cptr += 5;
            sptr = IconPath;
            while (*cptr && *cptr != '\"') *sptr++ = *cptr++;
            *sptr = '\0';
         }
         else
         if (*cptr == 'A' && !memcmp (cptr, "ALT=\"", 5))
         {
            cptr += 5;
            sptr = IconAltText;
            while (*cptr && *cptr != '\"') *sptr++ = *cptr++;
            *sptr = '\0';
         }
         else
            cptr++;
      }
      if (!IconAltText[0])
         strcpy (IconAltText, NotNone);
      CopyToHtml (HtmlIconAltText, sizeof(HtmlIconAltText), IconAltText, -1);
      if (!IconPath[0])
         strcpy (IconPath, NotNone);

      memcpy (sptr = Buffer, "<TR><TD>", 8);
      sptr += 8;
      cptr = (char*)ilptr + cfptr->IconStructOverhead;
      while (*cptr && *cptr != ' ') *sptr++ = tolower(*cptr++);
      if (*cptr) cptr++;
      memcpy (sptr, "</TD><TD ALIGN=center>", 22);
      sptr += 22;
      while (*cptr) *sptr++ = *cptr++;
      memcpy (sptr, "</TD><TD>", 9);
      sptr += 9;
      for (cptr = IconPath; *cptr; *sptr++ = *cptr++);
      memcpy (sptr, "</TD><TD>", 9);
      sptr += 9;
      for (cptr = HtmlIconAltText; *cptr; *sptr++ = *cptr++);
      memcpy (sptr, "</TD></TR>\n", 12);  /* includes null */
      sptr += 11;

      NetWriteBuffered (rqptr, 0, Buffer, sptr - Buffer);

      ilptr = ilptr->NextPtr;
   }

   NetWriteBuffered (rqptr, 0, "</TABLE>\n", 9);

   /*****************/
   /* content-types */
   /*****************/

   NetWriteBuffered (rqptr, 0, ReportContentTypeFaoDsc.dsc$a_pointer,
                              ReportContentTypeFaoDsc.dsc$w_length);

   clptr = cfptr->ContentTypeListHeadPtr;
   for (;;)
   {
      if (clptr == NULL) break;

      if (clptr->AutoScriptNamePtr[0] == '/')
         AutoScriptNamePtr = clptr->AutoScriptNamePtr;
      else
         AutoScriptNamePtr = "";

      if (clptr->ContentTypeUnknown)
         ContentDescriptionPtr = cfptr->ContentTypeDefaultPtr;
      else
         ContentDescriptionPtr = clptr->ContentDescriptionPtr;

      status = sys$fao (&ContentTypeFaoDsc, &Length, &BufferDsc,
                        clptr->SuffixPtr,
                        clptr->ContentTypePtr,
                        clptr->IconPtr,
                        AutoScriptNamePtr,
                        ContentDescriptionPtr);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$fao()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);

      clptr = clptr->NextPtr;
   }

   NetWriteBuffered (rqptr, 0, "</TABLE>\n", 9);

   /**************/
   /* end report */
   /**************/

   NetWriteBuffered (rqptr, 0, EndPage, sizeof(EndPage)-1);
}

/*****************************************************************************/
/*
This function just wraps the revision function, loading a temporary database
if necessary for reporting from the configuration file.
*/ 

ConfigRevise
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean  UseServerDatabase
)
{
   int  status;
   struct ConfigStruct  LocalConfig;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigRevise()\n");

   if (UseServerDatabase)
      ConfigReviseNow (rqptr, &Config);
   else
   {
      memset (&LocalConfig, 0, sizeof(struct ConfigStruct));
      /* indicate it's being used for a report */
      Config.RequestPtr = rqptr;
      Configure (&LocalConfig);
      ConfigReviseNow (rqptr, &LocalConfig);
      ConfigFree (&LocalConfig);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
Return a form for the HTTPd server's configuration.  POSTing the form to the
server results in the URL-decoded form as a configuration file.  This function
blocks while executing.
*/ 

ConfigReviseNow
(
struct RequestStruct *rqptr,
struct ConfigStruct *cfptr
)
{
#define SIZEOF_SPECIAL_ICON 256

#define REPBOOL(b)\
if (b)\
{\
   *vecptr++ = RadioButtonChecked;\
   *vecptr++ = RadioButtonUnchecked;\
}\
else\
{\
   *vecptr++ = RadioButtonUnchecked;\
   *vecptr++ = RadioButtonChecked;\
}

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Configuration</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Revise Configuration</H2>\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (ConfigGeneralFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR>\
<TH ALIGN=right>File:</TH>\
<TD ALIGN=left>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH>\
<TH>[<A HREF=\"!AZ!AZ\">Edit</A>]</TH>\
</TR>\n\
<TR><TH ALIGN=right>!AZ</TH>\
<TD ALIGN=left>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
\
<INPUT TYPE=hidden NAME=x VALUE=\"\
\# Configuration:  !AZ&#10;\
\#                 !AZ&#10;\
\# Last Modified:  !AZ&#10;\
\#                 !AZ@!AZ&#10;\
\">\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Service</TH></TR>\n\
<TR><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Service]&#10;\">\n\
<TEXTAREA NAME=Service ROWS=3 COLS=40>\
!AZ\
</TEXTAREA>\n\
</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>General</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Busy</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Busy]  \">\n\
<INPUT TYPE=text SIZE=2 NAME=Busy VALUE=!UL>\n\
<FONT SIZE=-1>connections</FONT></TD></TR>\n\
<TR><TH>Request History</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[RequestHistory]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=RequestHistory VALUE=!UL>\n\
</TD></TR>\n\
<TR><TH>Activity History</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[ActivityDays]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=ActivityDays VALUE=!UL>\n\
<FONT SIZE=-1>days</FONT></TD></TR>\n\
<TR><TH>Comments</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CommentedInfo]  \">\n\
<INPUT TYPE=radio NAME=CommentedInfo VALUE=enabled!AZ>yes\n\
<INPUT TYPE=radio NAME=CommentedInfo VALUE=disabled!AZ>no\n\
</TD></TR>\n\
<TR><TH>DNS Lookup</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DNSLookup]  \">\n\
<INPUT TYPE=radio NAME=DNSLookup VALUE=enabled!AZ>yes\n\
<INPUT TYPE=radio NAME=DNSLookup VALUE=disabled!AZ>no\n\
</TD></TR>\n\
<TR><TH>Error Recommend</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[ErrorRecommend]  \">\n\
<INPUT TYPE=radio NAME=ErrorRecommend VALUE=enabled!AZ>yes\n\
<INPUT TYPE=radio NAME=ErrorRecommend VALUE=disabled!AZ>no\n\
</TD></TR>\n\
<TR><TH>Error Source Information</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[ErrorSourceInfo]  \">\n\
<INPUT TYPE=radio NAME=ErrorSourceInfo VALUE=enabled!AZ>yes\n\
<INPUT TYPE=radio NAME=ErrorSourceInfo VALUE=disabled!AZ>no\n\
</TD></TR>\n\
<TR><TH>Monitor</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Monitor]  \">\n\
<INPUT TYPE=radio NAME=Monitor VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=Monitor VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>IP Port</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Port]  \">\n\
<INPUT TYPE=text SIZE=5 NAME=Port VALUE=!UL>\n\
<FONT SIZE=-1>(overridden by <I>service</I> directive)</FONT></TD></TR>\n\
<TR><TH>StreamLF</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[StreamLF]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=StreamLF VALUE=!UL>\n\
<FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>StreamLF Paths</TH></TR>\n\
<TR><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[StreamLFpaths]&#10;\">\n\
<TEXTAREA NAME=StreamLFpaths ROWS=3 COLS=40>\
!AZ\
</TEXTAREA>\n\
</TD></TR>\n\
<TR><TH>Search</TH><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Search]  \">\n\
<INPUT TYPE=text size=40 NAME=Search VALUE=\"!AZ\">\n\
</TD></TR>\n\
<TR><TH>Put/Post Max.Length</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[PutMaxKbytes]  \">\n\
<INPUT TYPE=text SIZE=5 NAME=PutMaxKbytes VALUE=!UL>\n\
<FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Put/Post Files</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[PutVersionLimit]  \">\n\
<INPUT TYPE=text SIZE=5 NAME=PutVersionLimit VALUE=!UL>\n\
<FONT SIZE=-1>version limit</FONT></TD></TR>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Log</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Logging</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Logging]  \">\n\
<INPUT TYPE=radio NAME=Logging VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=Logging VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Format</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogFormat]  \">\n\
<INPUT TYPE=text SIZE=10 NAME=LogFormat VALUE=!AZ></TD></TR>\n\
<TR><TH>Naming</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogNaming]  \">\n\
<INPUT TYPE=text SIZE=10 NAME=LogNaming VALUE=!AZ></TD></TR>\n\
</TD></TR>\n\
<TR><TH>Period</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogPeriod]  \">\n\
<INPUT TYPE=text SIZE=10 NAME=LogPeriod VALUE=!AZ></TD></TR>\n\
</TD></TR>\n\
<TR><TH>Per-Service</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogPerService]  \">\n\
<INPUT TYPE=radio NAME=LogPerService VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=LogPerService VALUE=disabled!AZ>disabled\n\
<TR><TH>File</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogFile]  \">\n\
<INPUT TYPE=text SIZE=40 NAME=LogFile VALUE=!AZ></TD></TR>\n\
<TR><TH>Exclude Hosts</TH></TR>\n\
<TR><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[LogExcludeHosts]&#10;\">\n\
<TEXTAREA NAME=LogExcludeHosts ROWS=3 COLS=40>\
!AZ\
</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Cache</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Caching</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Cache]  \">\n\
<INPUT TYPE=radio NAME=CacheEnabled VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=CacheEnabled VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Hash Table</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheHashTableEntries]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=CacheHashTableEntries VALUE=!UL>\n\
<FONT SIZE=-1>entries</FONT></TD></TR>\n\
<TR><TH>Max Entries</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheEntriesMax]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=CacheEntriesMax VALUE=!UL>\n\
<FONT SIZE=-1>files</FONT></TD></TR>\n\
<TR><TH>Max Cache Size</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheTotalKBytesMax]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=CacheTotalKBytesMax VALUE=!UL>\n\
<FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Max File Size</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheFileKBytesMax]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=CacheFileKBytesMax VALUE=!UL>\n\
<FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Chunk Size</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheChunkKBytes]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=CacheChunkKBytes VALUE=!UL>\n\
<FONT SIZE=-1>kBytes</FONT></TD></TR>\n\
<TR><TH>Validate</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheValidateSeconds]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=CacheValidateSeconds VALUE=!UL>\n\
<FONT SIZE=-1>seconds</FONT></TD></TR>\n\
<TR><TH>Frequent After</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheFrequentHits]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=CacheFrequentHits VALUE=!UL> \
<FONT SIZE=-1>hits, last within</FONT>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[CacheFrequentSeconds]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=CacheFrequentSeconds VALUE=!UL>\n\
<FONT SIZE=-1>seconds</FONT></TD></TR>\n\
<TR><TH></TH><TD>\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Timeouts</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Input</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[InputTimeout]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=InputTimeout VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Output</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[OutputTimeout]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=OutputTimeout VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Keep-Alive</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[KeepAliveTimeout]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=KeepAliveTimeout VALUE=!UL>\n\
<FONT SIZE=-1>seconds</FONT></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Welcome (index/home pages)</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>Format: \"file-name.file-type\" \
(e.g. HOME.HTML)</FONT></TD></TR>\n\
<TR><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Welcome]&#10;\">\n\
<TEXTAREA NAME=Welcome ROWS=3 COLS=40>");

   static $DESCRIPTOR (ConfigAcceptFaoDsc,
"</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Host Access Control</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Accept</TH></TR>\n\
<TR><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Accept]&#10;\">\n\
<TEXTAREA NAME=Accept ROWS=3 COLS=40>\
!AZ\
</TEXTAREA>\n\
</TD></TR>\n\
<TR><TH>Reject</TH></TR>\n\
<TR><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[Reject]&#10;\">\n\
<TEXTAREA NAME=Reject ROWS=3 COLS=40>\
!AZ\
</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Authorization</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Cache</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthCacheMinutes]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=AuthCacheMinutes VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Revalidate User</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthRevalidateUserMinutes]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=AuthRevalidateUserMinutes VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>Failure Limit</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthFailureLimit]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=AuthFailureLimit VALUE=!UL>\n\
<FONT SIZE=-1>attempts</FONT></TD></TR>\n\
<TR><TH>Basic Authorization</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthBasic]  \">\n\
<INPUT TYPE=radio NAME=AuthBasic VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=AuthBasic VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Digest Authorization</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthDigest]  \">\n\
<INPUT TYPE=radio NAME=AuthDigest VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=AuthDigest VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TH>GET nonce lifetime</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthDigestGetLife]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=AuthDigestGetLife VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD>\n\
</TD></TR>\n\
<TR><TH>PUT nonce lifetime</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AuthDigestPutLife]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=AuthDigestPutLife VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH ALIGN=right>Note ...</A></TH>\
<TD>paths must also be configured!!</TD></TR>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Scripting &nbsp;(DCL & DECnet)</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Spawn Authorized Privileges</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclSpawnAuthPriv]  \">\n\
<INPUT TYPE=radio NAME=DclSpawnAuthPriv VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DclSpawnAuthPriv VALUE=disabled!AZ>disabled\n\
</TD><TD></TD></TR>\n\
<TR><TH>Supply Header and Body</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclFullRequest]  \">\n\
<INPUT TYPE=radio NAME=DclFullRequest VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DclFullRequest VALUE=disabled!AZ>disabled\n\
</TD><TD></TD></TR>\n\
<TR><TH>Soft Limit</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclSoftLimit]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=DclSoftLimit VALUE=!UL>\n\
<FONT SIZE=-1>subprocesses</FONT></TD></TR>\n\
<TR><TH>Hard Limit</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclHardLimit]  \">\n\
<INPUT TYPE=text SIZE=3 NAME=DclHardLimit VALUE=!UL>\n\
<FONT SIZE=-1>subprocesses</FONT></TD></TR>\n\
<TR><TH>Zombie Life-Time</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclZombieLifeTime]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=DclZombieLifeTime VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>CGIplus Life-Time</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclCgiPlusLifeTime]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=DclCgiPlusLifeTime VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>DECnet Reuse Life-Time</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DECnetReuseLifeTime]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=DECnetReuseLifeTime VALUE=!UL>\n\
<FONT SIZE=-1>minutes</FONT></TD></TR>\n\
<TR><TH>DECnet Connect List Limit</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DECnetConnectListMax]  \">\n\
<INPUT TYPE=text SIZE=4 NAME=DECnetConnectListMax VALUE=!UL>\n\
<FONT SIZE=-1>connections</FONT></TD></TR>\n\
<TR><TH>Script Run-Time</TH></TR>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>\
Format: &quot;file-type [SPACE] verb&quot;<BR>\
(e.g. &quot;.PL [SPACE] PERL&quot; or &quot;.PL [SPACE] $PERL_EXE:PERL&quot;)\
</FONT>\
</TD></TR>\n\
<TR><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DclScriptRunTime]&#10;\">\n\
<TEXTAREA NAME=DclScriptRunTime ROWS=3 COLS=60>");

   static $DESCRIPTOR (ConfigSsiFaoDsc,
"</TEXTAREA>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Server-Side Includes</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>SSI</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[SSI]  \">\n\
<INPUT TYPE=radio NAME=SSI VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=SSI VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Exec</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[SSIexec]  \">\n\
<INPUT TYPE=radio NAME=SSIexec VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=SSIexec VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Access Counting</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[SSIaccesses]  \">\n\
<INPUT TYPE=radio NAME=SSIaccesses VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=SSIaccesses VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Directory (Index of...)</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH>Access</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirAccess]  \">\n\
<INPUT TYPE=radio NAME=DirAccess VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirAccess VALUE=selective!AZ>selective\n\
<INPUT TYPE=radio NAME=DirAccess VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>HTML Description</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirDescriptionLines]  \">\n\
<INPUT TYPE=text SIZE=6 NAME=DirDescriptionLines VALUE=!UL>\n\
<FONT SIZE=-1>lines</FONT></TD></TR>\n\
</TD></TR>\n\
<TR><TH>Layout</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirLayout]  \">\n\
<INPUT TYPE=text size=40 NAME=DirLayout VALUE=\"!AZ\">\n\
</TD></TR>\n\
<TR><TH>Owner</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirOwner]  \">\n\
<INPUT TYPE=radio NAME=DirOwner VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirOwner VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Pre-Expired</TH><TD>\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirPreExpired]  \">\n\
<INPUT TYPE=radio NAME=DirPreExpired VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirPreExpired VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>No Implied Wildcard</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirNoImpliedWildcard]  \">\n\
<INPUT TYPE=radio NAME=DirNoImpliedWildcard VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirNoImpliedWildcard VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Protection Violations Ignored</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirNoPrivIgnore]  \">\n\
<INPUT TYPE=radio NAME=DirNoPrivIgnore VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirNoPrivIgnore VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Wildcard</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirWildcard]  \">\n\
<INPUT TYPE=radio NAME=DirWildcard VALUE=enabled!AZ>enabled\n\
<INPUT TYPE=radio NAME=DirWildcard VALUE=disabled!AZ>disabled\n\
</TD></TR>\n\
<TR><TH>Readme</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirReadme]  \">\n\
<INPUT TYPE=radio NAME=DirReadme VALUE=top!AZ>top\n\
<INPUT TYPE=radio NAME=DirReadme VALUE=bottom!AZ>bottom\n\
<INPUT TYPE=radio NAME=DirReadme VALUE=disabled!AZ>none\n\
</TD></TR>\n\
<TR><TH>Readme Files</TH></TR>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>Format: &quot;file-name.file-type&quot; \
(e.g. README.HTML)</FONT></TD></TR>\n\
<TR><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[DirReadmeFile]&#10;\">\n\
<TEXTAREA NAME=DirReadmeFile ROWS=3 COLS=40>");

   static $DESCRIPTOR (ConfigIconsFaoDsc,
"</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Icons</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH COLSPAN=2>Icons</TH></TR>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>Format: &quot;icon-path [SPACE] \
alternative-text [SPACE] content-type&quot;</FONT></TD></TR>\n\
<TR><TD COLSPAN=2>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddIcon]&#10;\">\n\
<TEXTAREA NAME=AddIcon ROWS=5 COLS=60>");

   static $DESCRIPTOR (ConfigSpecialIconsFaoDsc,
"</TEXTAREA>\n\
</TD></TR>\n\
<TR><TH COLSPAN=2>Special Icons</TH></TR>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>Format: &quot;icon-path [SPACE] \
alternative-text&quot;</FONT></TD></TR>\n\
<TR><TH>Blank</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddBlankIcon]  \">\n\
<INPUT TYPE=text SIZE=40 NAME=AddBlankIcon VALUE=\"!AZ\">\n\
</TD></TR>\n\
<TR><TH>Directory</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddDirIcon]  \">\n\
<INPUT TYPE=text SIZE=40 NAME=AddDirIcon VALUE=\"!AZ\">\n\
</TD></TR>\n\
<TR><TH>Parent Directory</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddParentIcon]  \">\n\
<INPUT TYPE=text SIZE=40 NAME=AddParentIcon VALUE=\"!AZ\">\n\
</TD></TR>\n\
<TR><TH>Unknown</TH><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddUnknownIcon]  \">\n\
<INPUT TYPE=text SIZE=40 NAME=AddUnknownIcon VALUE=\"!AZ\">\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (ConfigContentTypesFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Content Types</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD COLSPAN=2><FONT SIZE=-1>Format: &quot;file-suffix [SPACE] \
content-type [SPACE] auto-script-path <B>or</B> a hyphen [SPACE] \
description (to-end-of-line)&quot;</FONT></TD></TR>\n\
<TR><TD>\n\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;[AddType]&#10;\">\n\
<TEXTAREA NAME=AddType ROWS=5 COLS=60>");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<INPUT TYPE=hidden NAME=x VALUE=\"&#10;# End!&#10;\">\n\
<P><INPUT TYPE=submit VALUE=\" Commit Changes \">\n\
<INPUT TYPE=reset VALUE=\" Reset \">\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   static char  RadioButtonChecked [] = " CHECKED",
                RadioButtonUnchecked [] = "";

   register int  idx;
   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct ContentTypeStruct  *clptr;
   register struct IconStruct  *ilptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [128];
   char  *AutoScriptNamePtr,
         *ContentDescriptionPtr,
         *IconAltPtr,
         *IconAltEndPositionPtr,
         *IconAltStartPositionPtr,
         *IconUrlPtr,
         *IconTypePtr,
         *IconTypePositionPtr,
         *SpecialIconPtr;
   char  Buffer [8192],
         HtmlIconUrl [256],
         Scratch [256],
         SpecialIconBlank [SIZEOF_SPECIAL_ICON],
         SpecialIconDir [SIZEOF_SPECIAL_ICON],
         SpecialIconParent [SIZEOF_SPECIAL_ICON],
         SpecialIconUnknown [SIZEOF_SPECIAL_ICON],
         TimeCurrent [32];
   struct ContentTypeStruct  ContentType;
   struct ConfigStruct  LocalConfig;
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (ScratchDsc, Scratch);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReviseNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (cfptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = cfptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (cfptr == &Config)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr++ = cfptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   strcpy (TimeCurrent, DayDateTime (&rqptr->BinaryTime, 20));
   cptr = MapVmsPath (cfptr->LoadFileName);

   vecptr = FaoVector;

   /* source information (i.e. from executing server or static file) */
   if (cfptr == &Config)
      *vecptr++ = "Server";
   else
      *vecptr++ = "File";
   *vecptr++ = cfptr->LoadFileName;
   *vecptr++ = cptr;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = cptr;

   if (cfptr == &Config)
   {
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&cfptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&cfptr->RevBinTime, 20);
   }

   /* form action */
   *vecptr++ = cptr;

   /* comments */
   *vecptr++ = ServerHostPort;
   *vecptr++ = SoftwareID;
   *vecptr++ = TimeCurrent;
   *vecptr++ = rqptr->RemoteUser;
   *vecptr++ = rqptr->ClientHostName;

   /* services */
   *vecptr++ = ConfigCommaList (rqptr, cfptr->ServicePtr, '\n');

   /* general */
   if (cfptr->Busy)
      *vecptr++ = cfptr->Busy;
   else
      *vecptr++ = DefaultMaxConcurrentConnections;
   *vecptr++ = cfptr->RequestHistory;
   *vecptr++ = cfptr->ActivityNumberOfDays;
   REPBOOL (cfptr->IncludeCommentedInfo)
   REPBOOL (cfptr->DnsLookup)
   REPBOOL (cfptr->ErrorRecommend)
   REPBOOL (cfptr->ErrorSourceInfo)
   REPBOOL (cfptr->MonitorEnabled)
   *vecptr++ = cfptr->ServerPort;
   *vecptr++ = cfptr->StreamLfConversionMaxKbytes;
   *vecptr++ = ConfigCommaList (rqptr, cfptr->StreamLfPathsPtr, '\n');
   *vecptr++ = cfptr->KeywordSearch;
   *vecptr++ = cfptr->PutMaxKbytes;
   *vecptr++ = cfptr->PutVersionLimit;

   /* logging */
   REPBOOL (cfptr->LoggingEnabled)
   *vecptr++ = cfptr->LogFormat;
   *vecptr++ = cfptr->LogNaming;
   *vecptr++ = cfptr->LogPeriod;
   REPBOOL (cfptr->LogPerService)
   *vecptr++ = cfptr->LogFileName;
   *vecptr++ = ConfigCommaList (rqptr, cfptr->LogExcludeHostsPtr, '\n');

   /* cache */
   REPBOOL (cfptr->CacheEnabled)
   *vecptr++ = cfptr->CacheHashTableEntries;
   *vecptr++ = cfptr->CacheEntriesMax;
   *vecptr++ = cfptr->CacheTotalKBytesMax;
   *vecptr++ = cfptr->CacheFileKBytesMax;
   *vecptr++ = cfptr->CacheChunkKBytes;
   *vecptr++ = cfptr->CacheValidateSeconds;
   *vecptr++ = cfptr->CacheFrequentHits;
   *vecptr++ = cfptr->CacheFrequentSeconds;

   /* timeouts */
   *vecptr++ = cfptr->InputTimeoutMinutes;
   *vecptr++ = cfptr->OutputTimeoutMinutes;
   *vecptr++ = cfptr->KeepAliveTimeoutSeconds;

   status = sys$faol (&ConfigGeneralFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /*****************************/
   /* welcome (home page) files */
   /*****************************/

   for (idx = 0; idx < cfptr->HomePageCount; idx++)
   {
      zptr = (sptr = Scratch) + sizeof(Scratch);
      if (!idx) *sptr++ = '\n';
      for (cptr = cfptr->HomePageArray[idx];
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = '\n';
      if (sptr >= zptr)
         strcpy (Scratch, ConfigStringOverflow);
      else
         *sptr = '\0';

      Length = CopyToHtml (Buffer, sizeof(Buffer), Scratch, -1);
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /********/
   /* more */
   /********/

   vecptr = FaoVector;

   /* accept and reject hosts */
   *vecptr++ = ConfigCommaList (rqptr, cfptr->AcceptHostsPtr, '\n');
   *vecptr++ = ConfigCommaList (rqptr, cfptr->RejectHostsPtr, '\n');

   /* authentication */
   *vecptr++ = cfptr->AuthCacheMinutes;
   *vecptr++ = cfptr->AuthRevalidateUserMinutes;
   *vecptr++ = cfptr->AuthFailureLimit;
   REPBOOL (cfptr->AuthBasicEnabled)
   REPBOOL (cfptr->AuthDigestEnabled)
   *vecptr++ = cfptr->AuthDigestNonceGetLifeTime;
   *vecptr++ = cfptr->AuthDigestNonceGetLifeTime;

   /* DCL/scripting */
   REPBOOL (cfptr->DclSpawnAuthPriv)
   REPBOOL (cfptr->DclFullRequest)
   *vecptr++ = cfptr->DclSubprocessSoftLimit;
   *vecptr++ = cfptr->DclSubprocessHardLimit;
   *vecptr++ = cfptr->DclZombieLifeTime;
   *vecptr++ = cfptr->DclCgiPlusLifeTime;
   *vecptr++ = cfptr->DECnetReuseLifeTime;
   *vecptr++ = cfptr->DECnetConnectListMax;

   status = sys$faol (&ConfigAcceptFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /****************/
   /* script types */
   /****************/

   for (idx = 0; idx < cfptr->ScriptRunTimeCount; idx++)
   {
      sptr = Buffer;
      for (cptr = cfptr->ScriptRunTimeArray[idx];
           *cptr && *cptr != ';';
           *sptr++ = *cptr++);
      *sptr++ = ' ';
      if (*cptr) cptr++;
      if (*cptr) cptr++;
      if (*cptr == '@' || *cptr == '$')
      {
         while (*cptr && *cptr != '\"') cptr++;
         if (*cptr) cptr++;
         while (*cptr)
         {
            /* unescape any quotes */
            if (cptr[0] == '\"' && cptr[1] == '\"')
               cptr++;
            else
            if (*cptr == '\"')
               break;
            *sptr++ = *cptr++;
         }
      }
      else
         while (*cptr) *sptr++ = *cptr++;
      *sptr++ = '\n';
      *sptr = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, sptr-Buffer);
   }

   /********/
   /* more */
   /********/

   vecptr = FaoVector;

   /* SSI (.shtml) */
   REPBOOL (cfptr->SsiEnabled)
   REPBOOL (cfptr->SsiExecEnabled)
   REPBOOL (cfptr->SsiAccessesEnabled)

   /* directory */
   if (cfptr->DirAccess)
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   else
   if (cfptr->DirAccessSelective)
   {
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
   }
   *vecptr++ = cfptr->DirDescriptionLines;
   *vecptr++ = cfptr->DefaultDirLayout;
   REPBOOL (cfptr->DirOwnerEnabled)
   REPBOOL (cfptr->DirPreExpired)
   REPBOOL (cfptr->DirNoImpliedWildcard)
   REPBOOL (cfptr->DirNoPrivIgnore)
   REPBOOL (cfptr->DirWildcardEnabled)
   if (cfptr->DirReadMeTop)
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   else
   if (cfptr->DirReadMeBottom)
   {
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
   }
   REPBOOL (cfptr->DirWildcardEnabled)

   status = sys$faol (&ConfigSsiFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /****************/
   /* readme files */
   /****************/

   for (idx = 0; idx < cfptr->ReadMeFileCount; idx++)
   {
      if (!idx) NetWriteBuffered (rqptr, 0, "\n", 1);
      Length = CopyToHtml (Buffer, sizeof(Buffer)-1,
                           cfptr->ReadMeFileArray[idx], -1);
      Buffer[Length++] = '\n';
      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /********/
   /* more */
   /********/

   NetWriteBuffered (rqptr, 0, ConfigIconsFaoDsc.dsc$a_pointer,
                              ConfigIconsFaoDsc.dsc$w_length);

   /*********/
   /* icons */
   /*********/

   SpecialIconBlank[0] = SpecialIconDir[0] = SpecialIconParent[0] =
      SpecialIconUnknown[0] = '\n';
   SpecialIconBlank[1] = SpecialIconDir[1] = SpecialIconParent[1] =
      SpecialIconUnknown[1] = '\0';

   ilptr = cfptr->IconListHeadPtr;
   while (ilptr != NULL)
   {
      /*
         Convert from 1 back to 2.

         1.  sprintf ((char*)ilptr + cfptr->IconStructOverhead,
                      "%s <IMG ALIGN=top SRC=\"%s\" ALT=\"%s\">",
                      ContentTypePtr, IconUrlPtr, AltTextPtr);

         2.  <URL>[SPACE]<alternative-text>[SPACE]<content-type>
      */

      IconTypePtr = cptr = (char*)ilptr + cfptr->IconStructOverhead;
      while (*cptr && !ISLWS(*cptr)) cptr++;
      if (*cptr) cptr++;
      while (*cptr)
      {
         if (*cptr == 'S' && !memcmp (cptr, "SRC=\"", 5))
         {
            cptr += 5;
            IconUrlPtr = cptr;
            while (*cptr && *cptr != '\"') cptr++;
            if (*cptr) cptr++;
         }
         else
         if (*cptr == 'A' && !memcmp (cptr, "ALT=\"", 5))
         {
            cptr += 5;
            IconAltPtr = cptr;
            while (*cptr && *cptr != '\"') cptr++;
            if (*cptr) cptr++;
         }
         else
            cptr++;
      }
      zptr = (sptr = Scratch) + sizeof(Scratch);
      if (ilptr == cfptr->IconListHeadPtr) *sptr++ = '\n';
      /* icon URL */
      for (cptr = IconUrlPtr;
           *cptr && *cptr != '\"' && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = ' ';
      if (sptr < zptr) *sptr++ = ' ';
      /* note the start of alternative text */
      IconAltStartPositionPtr = sptr;
      /* alternative text */
      for (cptr = IconAltPtr; *cptr && *cptr != '\"' && sptr < zptr; cptr++)
      {
         if (*cptr == ' ')
            *sptr++ = '_';
         else
            *sptr++ = *cptr;
      }
      IconAltEndPositionPtr = sptr;
      if (sptr < zptr) *sptr++ = ' ';
      if (sptr < zptr) *sptr++ = ' ';
      /* note position of null-terminated content type in the scratch string */
      IconTypePositionPtr = sptr;
      /* get the content type */
      for (cptr = IconTypePtr;
           *cptr && *cptr != ' ' && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      if (sptr >= zptr)
         Scratch[0] = '\0';
      else
         *sptr = '\0';

      SpecialIconPtr = NULL;
      if (strsame (IconTypePositionPtr, ConfigContentTypeBlank, -1))
      {
         /* replace the blanks with equivalent underscores */
         while (IconAltStartPositionPtr < IconAltEndPositionPtr)
            *IconAltStartPositionPtr++ = '_';
         SpecialIconPtr = SpecialIconBlank;
      }
      else
      if (strsame (IconTypePositionPtr, ConfigContentTypeDir, -1))
         SpecialIconPtr = SpecialIconDir;
      else
      if (strsame (IconTypePositionPtr, ConfigContentTypeParent, -1))
         SpecialIconPtr = SpecialIconParent;
      else
      if (strsame (IconTypePositionPtr, ConfigContentTypeUnknown, -1))
         SpecialIconPtr = SpecialIconUnknown;
      /* do not include content-type in special icon strings */
      if (SpecialIconPtr != NULL) *IconAltEndPositionPtr = '\0';

      if (sptr < zptr) *sptr++ = '\n';
      if (sptr >= zptr)
         strcpy (Scratch, ConfigStringOverflow);
      else
         *sptr = '\0';

      Length = CopyToHtml (Buffer, sizeof(Buffer), Scratch, -1);

      if (SpecialIconPtr == NULL)
         NetWriteBuffered (rqptr, 0, Buffer, Length);
      else
      {
         zptr = (sptr = SpecialIconPtr) + SIZEOF_SPECIAL_ICON;
         for (cptr = Buffer; *cptr && sptr < zptr; *sptr++ = *cptr++);
         if (sptr >= zptr)
            strcpy (SpecialIconPtr, ConfigStringOverflow);
         else
            *sptr = '\0';
      }

      ilptr = ilptr->NextPtr;
   }

   vecptr = FaoVector;

   *vecptr++ = SpecialIconBlank;
   *vecptr++ = SpecialIconDir;
   *vecptr++ = SpecialIconParent;
   *vecptr++ = SpecialIconUnknown;

   status = sys$faol (&ConfigSpecialIconsFaoDsc, &Length, &BufferDsc,
                      &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /*****************/
   /* content types */
   /*****************/

   NetWriteBuffered (rqptr, 0, ConfigContentTypesFaoDsc.dsc$a_pointer,
                              ConfigContentTypesFaoDsc.dsc$w_length);

   clptr = cfptr->ContentTypeListHeadPtr;
   for (;;)
   {
      if (clptr == NULL) break;

      if (clptr->AutoScriptNamePtr[0] == '/')
         AutoScriptNamePtr = clptr->AutoScriptNamePtr;
      else
         AutoScriptNamePtr = "-";

      if (clptr->ContentTypeUnknown)
         ContentDescriptionPtr = cfptr->ContentTypeDefaultPtr;
      else
         ContentDescriptionPtr = clptr->ContentDescriptionPtr;

      zptr = (sptr = Scratch) + sizeof(Scratch);
      if (clptr == cfptr->ContentTypeListHeadPtr) *sptr++ = '\n';
      for (cptr = clptr->SuffixPtr;
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = ' ';
      if (sptr < zptr) *sptr++ = ' ';
      for (cptr = clptr->ContentTypePtr;
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = ' ';
      if (sptr < zptr) *sptr++ = ' ';
      for (cptr = AutoScriptNamePtr;
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = ' ';
      if (sptr < zptr) *sptr++ = ' ';
      for (cptr = ContentDescriptionPtr;
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = '\n';
      if (sptr >= zptr)
         strcpy (Scratch, ConfigStringOverflow);
      else
         *sptr = '\0';

      Length = CopyToHtml (Buffer, sizeof(Buffer), Scratch, -1);
      NetWriteBuffered (rqptr, 0, Buffer, Length);

      clptr = clptr->NextPtr;
   }

   /************/
   /* end form */
   /************/

   NetWriteBuffered (rqptr, 0, EndPageFaoDsc.dsc$a_pointer,
                              EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
Takes a comma-separated list of elements and returns a pointer to a dynamically
allocated string with that list reformated for inclusion in HTML.  The
separator is intended to be either a newline or comma character.  If a comma an
extra space is added after each (so that when displayed in HTML it will be
wrapped).
*/

char* ConfigCommaList
(
struct RequestStruct *rqptr,
char *ListPtr,
char Separator
)
{
   register char  *cptr, *sptr, *zptr;

   int  Length;
   char  ListBuffer [1024],
         HtmlListBuffer [1024];

   /*********/
   /* begin */
   /*********/
   
   if (Debug) fprintf (stdout, "ConfigCommaList() |%s|\n", ListPtr);

   if (ListPtr == NULL) return ("");
   zptr = (sptr = ListBuffer) + sizeof(ListBuffer);
   for (cptr = ListPtr; *cptr && sptr < zptr; cptr++)
   {
      if (*cptr == ',')
          *sptr++ = Separator;
      else
          *sptr++ = *cptr;
      if (Separator == ',' && *cptr == ',' && sptr < zptr) *sptr++ = ' ';         
   }
   if (sptr >= zptr) return (ConfigStringOverflow);
   *sptr = '\0';
   Length = CopyToHtml (HtmlListBuffer, sizeof(HtmlListBuffer), ListBuffer, -1);
   if (Length == -1) return (ConfigStringOverflow);
   sptr = VmGetHeap (rqptr, Length);
   memcpy (sptr, HtmlListBuffer, Length+1);
   if (Debug) fprintf (stdout, "|%s|\n", sptr);
   return (sptr);
}

/*****************************************************************************/
