/*****************************************************************************/
/*
                                 Control.c

This module implements the HTTPd command-line control functionality.

Command-line server control commands:

  o  AUTH=LOAD   reload authorization file
  o  AUTH=PURGE  purge authentication records from cache
  o  CACHE=ON    turn file caching on
  o  CACHE=OFF   turn file caching off
  o  CACHE=PURGE         purge all data cached
  o  DECNET=PURGE        disconnect idle DECnet script tasks
  o  DECNET=DISCONNECT   forceably disconnect all DECnet script tasks
  o  DCL=DELETE  unconditionally delete all DCL subprocesses, busy or not
  o  DCL=PURGE   delete idle subprocesses, mark those busy for later deletion
  o  EXIT        exit after all client activity complete (nothing new started)
  o  EXIT=NOW    exit right now regardless of connections
  o  LOG=OPEN    open the log file
  o  LOG=OPEN=name       open the log file using the specified name
  o  LOG=REOPEN          closes then reopens the log (to change format perhaps)
  o  LOG=REOPEN=name     log is closed and then reopened using specified name
  o  LOG=CLOSE   close the log file
  o  LOG=FLUSH   flush the log file
  o  LOG=FORMAT=string   set the log format (requires subsequent open/reopen)
  o  LOG=PERIOD=string   set the log period (requires subsequent open/reopen)
  o  MAP          reload mapping rule file
  o  RESTART      restart the image, effectively exit and re-execute
  o  RESTART=NOW  restart the image right now regardless of connections
  o  ZERO         zero accounting

These commands are entered at the DCL command line (interpreted in the CLI.c
module) in the following manner:

  $ HTTPD /DO=command

For example:

  $ HTTPD /DO=ABORT          !the server exists immediately
  $ HTTPD /DO=EXIT           !the server exits if when clients connected
  $ HTTPD /DO=RESTART        !server exits and the restarts
  $ HTTPD /DO=DCL=PURGE      !delete all persistent DCL subprocesses
  $ HTTPD /DO=LOG=CLOSE      !close the log file


VERSION HISTORY
---------------
15-AUG-98  MGD  decnet=purge, decnet=disconnect
16-MAR-98  MGD  restart=now
05-OCT-97  MGD  cache and logging period control
28-SEP-97  MGD  reinstated some convenience commands (AUTH, DCL, and ZERO)
01-FEB-97  MGD  HTTPd version 4 (removed much of its previous functionality)
01-OCT-96  MGD  minor changes to authorization do commands
01-JUL-96  MGD  controls for path/realm-based authorization/authentication
01-DEC-95  MGD  HTTPd version 3
12-APR-95  MGD  support logging
03-APR-95  MGD  support authorization
20-DEC-94  MGD  initial development for multi-threaded version of HTTPd
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <prvdef.h>
#include <psldef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "auth.h"
#include "cache.h"
#include "control.h"
#include "dcl.h"
#include "error.h"
#include "httpd.h"
#include "logging.h"
#include "mapurl.h"
#include "support.h"

/******************/
/* global storage */
/******************/

/* 5 seconds */
$DESCRIPTOR (ControlCommandTimeoutDsc, "0 00:00:05.00");

boolean  ControlExitRequested,
         ControlRestartRequested;

unsigned short  ControlMbxChannel;

char  ControlBuffer [256],
      ControlMbxName [32];
      
struct AnIOsb  ControlMbxReadIOsb,
               ControlMbxWriteIOsb;

$DESCRIPTOR (ControlMbxNameDsc, ControlMbxName);
$DESCRIPTOR (ControlMbxNameFaoDsc, "HTTPD!UL$CONTROL");

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  CacheEnabled;
extern boolean  MonitorEnabled;
extern int  CurrentConnectCount;
extern int  ExitStatus;
extern int  ServerPort;
extern char  LoggingFileName[];
extern char  LoggingFormatUser[];
extern char  LoggingPeriodName[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct Accounting;

/*****************************************************************************/
/*
This, and the other supporting functions, are used by the serving HTTPd 
process.  Create a system-permanent mailbox for the receipt of control 
messages sent by another process.
*/ 

int ControlHttpd ()

{
   /* no world or group access, full owner and system access */
#  define ControlMbxProtectionMask 0xff00

   /*
       PRMMBX to allow a permanent mailbox to be created.
       SYSNAM to create its logical name in the system table.
       SYSPRV just in case it was originally created by a privileged,
              non-HTTPd account (as has happened to me when developing!)
   */
   static unsigned long  CreMbxPrivMask [2] =
          { PRV$M_PRMMBX | PRV$M_SYSNAM | PRV$M_SYSPRV, 0 };

   int  status,
        SetPrvStatus;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ControlHttpd()\n");

   if (VMSnok (status =
       sys$fao (&ControlMbxNameFaoDsc, &Length, &ControlMbxNameDsc,
                ServerPort)))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      return (status);
   }
   ControlMbxName[ControlMbxNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ControlMbxName |%s|\n", ControlMbxName);

   /* turn on privileges to allow access to the permanent mailbox */
   if (VMSnok (status = sys$setprv (1, &CreMbxPrivMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
      return (status);
   }

   status = sys$crembx (1,
                        &ControlMbxChannel,
                        sizeof(ControlBuffer),
                        sizeof(ControlBuffer),
                        ControlMbxProtectionMask,
                        PSL$C_USER,
                        &ControlMbxNameDsc,
                        0);
   if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);

   if (VMSok (status))
   {
     /* pro-actively mark the mailbox for deletion */
      status = sys$delmbx (ControlMbxChannel);
      if (Debug) fprintf (stdout, "sys$delmbx() %%X%08.08X\n", status);
   }

   if (VMSnok (SetPrvStatus = sys$setprv (0, &CreMbxPrivMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning them off! */
      ErrorExitVmsStatus (SetPrvStatus, "sys$setprv()", FI_LI);
   }

   if (VMSnok (status)) return (status);

   /* queue up the first asynchronous read from the control mailbox */
   return (sys$qio (0, ControlMbxChannel, IO$_READVBLK, &ControlMbxReadIOsb,
                    &ControlMbxReadAST, 0,
                    ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0));
}

/*****************************************************************************/
/*
An asynchronous read from the system-permanent control mailbox has completed.  
Act on the command string read into the buffer.
*/ 

ControlMbxReadAST ()

{
   register char  *cptr, *sptr;

   int  status;
   char  Scratch [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "ControlMbxReadAST() IO %d Status %%X%08.08X\n",
               ControlMbxReadIOsb.Count, ControlMbxReadIOsb.Status);

   if (VMSnok (ControlMbxReadIOsb.Status) &&
       ControlMbxReadIOsb.Status != SS$_ABORT &&
       ControlMbxReadIOsb.Status != SS$_CANCEL)
   {
      fprintf (stdout, "%%%s-E-CONTROLMBX, read failed\n-%s\n",
               Utility, SysGetMsg(ControlMbxReadIOsb.Status)+1);
      exit (ControlMbxReadIOsb.Status | STS$M_INHIB_MSG);
   }

   ControlBuffer[ControlMbxReadIOsb.Count] = '\0';
   if (Debug) fprintf (stdout, "ControlBuffer |%s|\n", ControlBuffer);

   if (strsame (ControlBuffer, "AUTH", 3))
   {
      /*****************/
      /* authorization */
      /*****************/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "LOAD", 3))
         {
            if (VMSok (AuthPathConfig (NULL)))
            {
               ControlMbxWrite ("authorization loaded");
               ControlMessage ("authorization loaded");
            }
            else
            {
               ControlMbxWrite ("!loading authorization rules");
               ControlMessage ("!loading authorization rules");
            }
            return;
         }
         if (strsame (cptr, "PURGE", 3))
         {
            AuthCacheTreeFree ();
            ControlMbxWrite ("authentication cache purged");
            ControlMessage ("authentication cache purged");
            return;
         }
         ControlMbxWrite ("!syntax error");
         return;
      }
   }
   else
   if (strsame (ControlBuffer, "CACHE=", 3))
   {
      /***********/
      /* logging */
      /***********/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "ON", -1))
         {
            if (CacheEnabled)
               ControlMbxWrite ("!cache already enabled");
            else
            {
               CacheEnabled = true;
               ControlMbxWrite ("cache enabled");
            }
            return;
         }
         if (strsame (cptr, "OFF", -1))
         {
            if (CacheEnabled)
            {
               CacheEnabled = false;
               ControlMbxWrite ("cache disabled");
            }
            else
               ControlMbxWrite ("!cache already disabled");
            return;
         }
         if (strsame (cptr, "PURGE", 3))
         {
            int  PurgeCount, MarkedForPurgeCount;

            CachePurge (false, &PurgeCount, &MarkedForPurgeCount);
            sprintf (Scratch, "%d purged, %d marked for purge",
                     PurgeCount, MarkedForPurgeCount);
            ControlMbxWrite (Scratch);

            return;
         }
         ControlMbxWrite ("!syntax error");
         return;
      }
   }
   else
   if (strsame (ControlBuffer, "DCL", 3))
   {
      /*****************/
      /* DCL/scripting */
      /*****************/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "DELETE", 3))
         {
            cptr = DclControlPurgeAllSubprocesses (true);
            ControlMbxWrite (cptr);
            ControlMessage ("DCL subprocess delete");
            return;
         }
         if (strsame (cptr, "PURGE", 3))
         {
            cptr = DclControlPurgeAllSubprocesses (false);
            ControlMbxWrite (cptr);
            ControlMessage ("DCL subprocess purge");
            return;
         }
         ControlMbxWrite ("!syntax error");
         return;
      }
   }
   else
   if (strsame (ControlBuffer, "DECnet", 3))
   {
      /****************************/
      /* DECnet CGI/OSU scripting */
      /****************************/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "DISCONNECT", 3))
         {
            cptr = DECnetControlDisconnect (true);
            ControlMbxWrite (cptr);
            ControlMessage ("DECnet link disconnect");
            return;
         }
         if (strsame (cptr, "PURGE", 3))
         {
            cptr = DECnetControlDisconnect (false);
            ControlMbxWrite (cptr);
            ControlMessage ("DECnet link purge");
            return;
         }
         ControlMbxWrite ("!syntax error");
         return;
      }
   }
   else
   if (strsame (ControlBuffer, "EXIT", -1))
   {
      /*******************/
      /* server shutdown */
      /*******************/

      ControlMessage ("server shutdown");

      if (CurrentConnectCount)
      {
         /* stop the server from receiving incoming requests */
         NetShutdownServerSocket ();
         ControlExitRequested = true;
         ControlMbxWrite ("exiting when all clients disconnected");
         return;
      }
      else
      {
         ControlMbxWrite ("exiting now");
         ExitStatus = SS$_NORMAL;
         HttpdExit ();
         sys$delprc (0, 0);
      }
   }
   else
   if (strsame (ControlBuffer, "EXIT=NOW", -1) ||
       strsame (ControlBuffer, "ABORT", -1))
   {
      /*********************************/
      /* unconditional server shutdown */
      /*********************************/

      ControlMessage ("server exit NOW!");
      ControlMbxWrite ("server exit NOW!");
      ExitStatus = SS$_NORMAL;
      HttpdExit ();
      sys$delprc (0, 0);
   }
   else
   if (strsame (ControlBuffer, "LOG=", 3))
   {
      /***********/
      /* logging */
      /***********/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "OPEN=", 3))
         {
            strcpy (Scratch, "log opened");
            while (*cptr && *cptr != '=') cptr++;
            if (*cptr)
            {
               cptr++;
               sptr = LoggingFileName;
               while (*cptr) *sptr++ = toupper(*cptr++);
               *sptr = '\0';
               sprintf (Scratch+10, "\n \\%s\\", LoggingFileName);
            }
            if (VMSok (Logging (NULL, LOGGING_OPEN)))
            {
               ControlMbxWrite ("log opened");
               ControlMessage (Scratch);
            }
            else
            {
               ControlMbxWrite ("!log not opened");
               ControlMessage ("!log not opened");
            }
            return;
         }
         if (strsame (cptr, "CLOSE", 3))
         {
            if (VMSok (Logging (NULL, LOGGING_CLOSE)))
            {
               ControlMbxWrite ("log closed");
               ControlMessage ("log closed");
            }
            else
            {
               ControlMbxWrite ("!log not closed");
               ControlMessage ("!log not closed");
            }
            return;
         }
         if (strsame (cptr, "FLUSH", 3))
         {
            if (VMSok (Logging (NULL, LOGGING_FLUSH)))
            {
               ControlMbxWrite ("log flushed");
               ControlMessage ("log flushed");
            }
            else
            {
               ControlMbxWrite ("!log not flushed");
               ControlMessage ("!log not flushed");
            }
            return;
         }
         if (strsame (cptr, "FORMAT=", 3))
         {
            while (*cptr && *cptr != '=') cptr++;
            if (*cptr)
            {
               strcpy (LoggingFormatUser, cptr+1);
               ControlMbxWrite ("!log format changed");
               sprintf (Scratch, "log format changed\n \\%s\\",
                        LoggingFormatUser);
               ControlMessage (Scratch);
               return;
            }
            ControlMbxWrite ("!syntax error");
            return;
         }
         if (strsame (cptr, "PERIOD=", 3))
         {
            while (*cptr && *cptr != '=') cptr++;
            if (*cptr)
            {
               strcpy (LoggingPeriodName, cptr+1);
               ControlMbxWrite ("!log period changed");
               sprintf (Scratch, "log period changed\n \\%s\\",
                        LoggingFormatUser);
               ControlMessage (Scratch);
               return;
            }
            ControlMbxWrite ("!syntax error");
            return;
         }
         if (strsame (cptr, "REOPEN=", 3))
         {
            while (*cptr && *cptr != '=') cptr++;
            if (*cptr)
            {
               cptr++;
               sptr = LoggingFileName;
               while (*cptr) *sptr++ = toupper(*cptr++);
               *sptr = '\0';
               sprintf (Scratch, "log reopened\n \\%s\\", LoggingFileName);
            }
            else
               strcpy (Scratch, "log reopened");
            if (VMSnok (Logging (NULL, LOGGING_CLOSE)))
            {
               ControlMbxWrite ("!log not closed");
               ControlMessage ("!log not closed");
               return;
            }
            if (VMSok (Logging (NULL, LOGGING_OPEN)))
            {
               ControlMbxWrite ("log reopened");
               ControlMessage (Scratch);
            }
            else
            {
               ControlMbxWrite ("!log not reopened");
               ControlMessage ("!log not reopened");
            }
            return;
         }
         ControlMbxWrite ("!syntax error");
         return;
      }
   }
   else
   if (strsame (ControlBuffer, "MAP", 3))
   {
      /*********************************/
      /* re-read the mapping rule file */
      /*********************************/

      cptr = MapUrl_Map (NULL, NULL, NULL, NULL, NULL);
      if (!cptr[0] && cptr[1])
      {
         /* error */
         Scratch[0] = '!';
         strcpy (Scratch+1, cptr+1);
         ControlMbxWrite (Scratch);
         ControlMessage ("!loading mapping rules");
      }
      else
      {
         ControlMbxWrite ("mapping rules loaded");
         ControlMessage ("mapping rules loaded");
      }
      return;
   }
   else
   if (strsame (ControlBuffer, "RESTART", -1))
   {
      /******************/
      /* server restart */
      /******************/

      ControlMessage ("server restart");

      /* relies on the supporting DCL procedure looping immediately */
      if (CurrentConnectCount)
      {
         /* stop the server from receiving incoming requests */
         NetShutdownServerSocket ();
         ControlRestartRequested = true;
         ControlMbxWrite ("restarting when all clients disconnected");
         return;
      }
      else
      {
         ControlMbxWrite ("restarting now");
         sleep (1);
         exit (SS$_NORMAL);
      }
   }
   else
   if (strsame (ControlBuffer, "RESTART=NOW", -1))
   {
      /***********************/
      /* server restart NOW! */
      /***********************/

      ControlMessage ("server restart NOW!");

      /* relies on the supporting DCL procedure looping immediately */
      ControlMbxWrite ("restarting NOW!");
      sleep (1);
      exit (SS$_NORMAL);
   }
   else
   if (strsame (ControlBuffer, "ZERO", 3))
   {
      /*******************/
      /* zero accounting */
      /*******************/

      ZeroAccounting ();
      ControlMbxWrite ("accounting zeroed");
      ControlMessage ("accounting zeroed");
      return;
   }

   /*********/
   /* what? */
   /*********/

   /* all handled situations must return/exit before here */
   ControlMbxWrite ("!what?");
}

/*****************************************************************************/
/*
This is the function used by the controlling process.  Synchronously write the 
command to the HTTPd control mailbox, then synchronously read the response.  
Set a timer to limit the wait for the command to be read, and then response to 
be written, by the HTTPd process.
*/ 

int ControlCommand (char *Command)

{
   int  status;
   unsigned short  Length;
   unsigned long  CommandTimeout[2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ControlCommand() |%s|\n", Command);

   if (VMSnok (status =
       sys$fao (&ControlMbxNameFaoDsc, &Length, &ControlMbxNameDsc,
                ServerPort)))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      return (status);
   }
   ControlMbxName[ControlMbxNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ControlMbxName |%s|\n", ControlMbxName);

   if (VMSnok (status =
       sys$assign (&ControlMbxNameDsc, &ControlMbxChannel, 0, 0, 0)))
   {
      if (status == SS$_NOSUCHDEV)
      {
         fprintf (stdout,
         "%%%s-E-NOSUCHDEV, server (control mailbox) does not exist\n",
         Utility);
         return (status | STS$M_INHIB_MSG);
      }
      return (status);
   }

   /* initialize a delta-time seconds structure used in sys$setimr() */
   if (VMSnok (status =
       sys$bintim (&ControlCommandTimeoutDsc, &CommandTimeout)))
   {
      if (Debug) fprintf (stdout, "sys$bintim() %%X%08.08X\n", status);
      exit (status);
   }

   status = sys$setimr (0, &CommandTimeout, &ControlCommandTimeoutAST, 0, 0);
   if (Debug) fprintf (stdout, "sys$setimr() %%X%08.08X\n", status);

   /* synchronously write the command to the HTTPd process */
   status = sys$qiow (0, ControlMbxChannel, IO$_WRITEVBLK,
                      &ControlMbxReadIOsb, 0, 0,
                      Command, strlen(Command), 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status: %%X%08.08X\n",
               status, ControlMbxReadIOsb.Status);
   if (VMSok (status) && VMSnok (ControlMbxReadIOsb.Status))
      status = ControlMbxReadIOsb.Status;
   if (status == SS$_ABORT)
   {
      /* timer has expired */
      fprintf (stdout,
      "%%%s-E-TIMEOUT, server failed to respond within 5 seconds\n",
      Utility);
      return (SS$_TIMEOUT | STS$M_INHIB_MSG);
   }
   if (VMSnok (status)) return (status);

   /* read the acknowledgement from the HTTPd process */
   status = sys$qiow (0, ControlMbxChannel, IO$_READVBLK,
                      &ControlMbxReadIOsb, 0, 0,
                      ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status: %%X%08.08X\n",
               status, ControlMbxReadIOsb.Status);
   if (VMSok (status) && VMSnok (ControlMbxReadIOsb.Status))
      status = ControlMbxReadIOsb.Status;
   if (VMSok (status))
   {
      ControlBuffer[ControlMbxReadIOsb.Count] = '\0';
      if (Debug) fprintf (stdout, "%s\n", ControlBuffer);
   }

   if (status == SS$_ABORT)
   {
      /* timer has expired */
      fprintf (stdout,
         "%%%s-E-TIMEOUT, server failed to respond within 5 seconds\n",
         Utility);
      return (SS$_TIMEOUT | STS$M_INHIB_MSG);
   }
   if (VMSnok (status)) return (status);

   /* cancel the timer */
   sys$cantim (0, 0);

   /* present the response to the user (leading '!' indicates an error) */
   if (ControlBuffer[0] == '!')
      fprintf (stdout, "%%%s-E-RESPONSE, %s\n", Utility, ControlBuffer+1);
   else
      fprintf (stdout, "%%%s-I-RESPONSE, %s\n", Utility, ControlBuffer);
   return (status);
}

/*****************************************************************************/
/*
Queue an I/O to the mailbox with AST.  When the I/O completes the AST routine
queues another read from the control mailbox, setting the situation for another
control command to be sent to the HTTPd.  This function should always be called
as the final I/O of control output.
*/

ControlMbxWrite (char *String)

{
   int  status;

   if (Debug) fprintf (stdout, "ControlMbxWrite()\n");

   status = sys$qio (0, ControlMbxChannel, IO$_WRITEVBLK,
                     &ControlMbxWriteIOsb, &ControlMbxWriteAST, 0,
                     String, strlen(String), 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
This function is called after the HTTPd process has asynchronously written a 
response back to the controlling process, and that I/O has been read.  It 
merely queues another asynchronous read from the control mailbox.
*/ 

ControlMbxWriteAST ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "ControlMbxWriteAST() IO %d Status %%X%08.08X\n",
               ControlMbxWriteIOsb.Count, ControlMbxWriteIOsb.Status);

   /* queue up another read from the control mailbox */
   sys$qio (0, ControlMbxChannel, IO$_READVBLK, &ControlMbxReadIOsb,
            &ControlMbxReadAST, 0,
            ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0);
}

/*****************************************************************************/
/*
When the timer expires this function is called as an AST routine.  It merely 
cancels the outstanding I/O on the control mailbox channel.  This is detected 
by an I/O return code of SS$_CANCEL.
*/

ControlCommandTimeoutAST ()

{
   if (Debug) fprintf (stdout, "ControlCommandTimeoutAST()\n");

   sys$cancel (ControlMbxChannel);
}

/*****************************************************************************/
/*
Output information or error status message to SYS$OUTPUT.
*/ 

int ControlMessage (char *Message)

{
   if (Debug) fprintf (stdout, "ControlMessage()\n");

   if (Message[0] == '!')
      fprintf (stdout, "%%%s-E-CONTROL, %s, %s\n",
               Utility, DateTime(NULL,20), Message+1);
   else
      fprintf (stdout, "%%%s-I-CONTROL, %s, %s\n",
               Utility, DateTime(NULL,20), Message);
}

/*****************************************************************************/

