/*****************************************************************************/
/*
                                  DCL.c

*************
** CAUTION **
*************

THIS MODULE IS TASK-ORIENTED, NOT REQUEST-ORIENTED.

That is, most of the functions take a pointer to DCL task rather than a
pointer to request as do other modules. The reason is simple. In this module
the more-permanent data-structures are those associated with the DCL
subprocesses, which persist for multiple requests. With the requests transient
the DCL subprocesses must be managed in their absence. Hence requests are
associated with DCL subprocesses not vice-versa.


OVERVIEW
--------
Provides multi-threaded, concurrent HTTPd subprocesses executing DCL.

This module never returns a valid status and ALWAYS calls the supplied next
task (AST) function.  This should check for a generated error message to
determine is there were any problems.

The DCL can either be in the form of a command, or a procedure or executable
image file specification. Both should not be supplied, but file specifications
have precedence. If a file specfication is supplied the module verifies its
existance, and if not qualified with an extension, looks for a procedure first
(".COM"), then an executable image (".EXE"), then through any user-defined
list of file types (extensions) and associated scripting executables (e.g.
".PL" associated with the verb "PERL"). Furthermore, the DCL can be executed
either standalone or as a CGI script (indicated by the presence of a script
name in the request data structure). If a CGI script, then the output stream
is parsed for header information, if not a script then the stream is just
checked for correct carriage control (terminated by a newline). CGI variables
are created for both standalone DCL and scripts, although some (e.g.
WWW_SCRIPT_NAME) will be empty, and meaningless, for standalone DCL. 

The AST-driven nature makes the code a little more difficult to follow, but 
creates a powerful, event-driven, multi-threaded server.  All of the 
necessary functions implementing this module are designed to be non-blocking. 

All of these functions are designed to be, and should be, called at AST 
delivery level, which means they cannot be interrupted at the same level (in 
this case USER mode), and so their actions are essentially atomic at that 
level, meaning no additional synchronization is required for such activities 
as thread disposal, etc.

HTTPD can maintain a number of subprocesses limited only by its process 
quotas, memory is dynamically allocated and there are no fixed data structures 
related to subprocess management. 

The use of byte-streams (effectively "pipes") allows even DCL procedures to
output as HTTP servers, without the need for explicit network I/O. 

Four mailboxes are created for each subprocess' IPC:

  1.  A mailbox connected to its SYS$COMMAND.  This is used to pass DCL
      commands and/or other data to the subprocess.  It effectively allows
      the HTTPd to control the activities of the subprocess this way.

  2.  A mailbox connected to its SYS$OUTPUT.  This recieves records from
      the subprocess, if required appends the HTTP-required carriage-control
      (single <LF>), then sends the record to the client via the network.
      This allows even DCL procedures to supply a correct output stream to
      the client (see next paragraph).

      If the first line from a script is an HTTP status line (e.g.
      "HTTP/1.0 200 ok") then HTTPD assumes the script will be supplying
      a complete HTTP data stream, including full header and required
      carriage control (single <LF> terminating each line).  If the first
      line is not a HTTP status line it assumes CGI script output compliance
      and also ensures each record (line) received has correct HTTP
      carriage-control.

      This stream also attempts to maintain compliance with CGI scripting.
      If the first line output by a script is not an HTTP status line it
      creates and sends one to the client before sending the first line.

  3.  A mailbox defined for the subprocess by the name HTTP$INPUT.
      This may be used to read the request data steam sent by the client.
      Note that from v4.2 this is also the SYS$INPUT <stdin> stream as well,
      which appears to be the more common CGI implementation. As of version
      4.3 the default behaviour is to supply only the request body to the
      script (CGI standard and the more common implmentation).  Using the 
      configuration parameter [DclFullRequest] it is possible to revert
      to the previous behaviour of supplying the header then the body.
      (It's an easy enough modification for most scripts to skip the header by
      reading until the header-separating empty (blank) line is encountered.)

  4.  A mailbox defined by the name CGIPLUSIN.  This allows CGIplus scripts
      to read a stream of CGI variable information.  Each request begins with
      a comment line "!", which can be used for request start synchronisation
      and may always be discarded, and ends with an empty record (blank line),
      with a variable number of records in between.

The script subprocesses can use the basic CGI variables (VMS CERN-like) and 
behave very much like a CGI script.

That is, if a script wants to be CGI-compliant it provides as the first line a 
"Status:", "Content-type:" or a "Location:" then a blank line.   If the first 
line output by a script is a "Content-Type:" header line an HTTP "200" status 
line is prepended.  If the first line output by a script is a "Location:" 
redirection header line the redirection is processed to ensure CERN HTTPD/CGI 
behaviour.  An HTTP "302" status line is prepended if not a local redirection. 
If none of these, HTTPD creates a complete HTTP header comprising status line, 
"Content-Type: text/plain" and blank line (this is an extension of CERN HTTPD 
behaviour). 

If the first characters are "HTTP/1.0 ..." the script will be considered to be 
supplying the raw HTTP stream and record boundaries, carriage-control, etc., 
are of no further concern to the module.  This is the equivalent of a
"no-parse-header" script.  If CGI-compliant each record should represent a 
line of output.   That is lines should not be buffered together and sent as a 
block unless the script is supplying a raw HTTP data stream. 


CGI VARIABLES
-------------
See CGI.c module.


CGI-PLUS
--------
CGI plus lower latency, plus greater efficiency, plus less system impact!

CGIplus attempts to eliminate the overhead associated with creating the
subprocess and then executing the image of a CGI script.  It does this by
allowing the subprocess and associated image to continue executing in-between
uses, eliminating the startup overhead.  This both reduces the load on the
system and the request latency where one of these scripts is involved.
In this sense these advantages parallel those offered by commercial HTTP
server-integration APIs, such as Netscape NSAPI and Microsoft ISAPI.

The script interface is still largely CGI, which means a new API does not need
to be learned and that existing CGI scripts are simple to modify. The script
must read the CGI variables from CGIPLUSIN. They are supplied as a series of
records (lines) containing the CGI variable name, an equate symbol and then
the variable value. This format may be easily parsed and as the value contains
no encoded characters may be directly used. An empty record (blank line)
indicates the end of the request information. The request may be processed at
that stage.

After processing the CGIplus script can loop, waiting to read the details of
the next request from CGIPLUSIN. The first line read may ALWAYS be discarded
allowing a read to be used as a simple synchronization mechanism.

HTTP output (to the client) is written to SYS$OUTPUT (stdout). End of output
MUST be indicated by writing a special EOF string to the output stream. This
is a KLUDGE, and the least elegant part of CGIplus design, BUT it is also the
simplest implementation and should work in all but the most exceptional
circumstances. The special EOF string has been chosen to be most unlikely to
occur in normal output (a hopefully unique 280 bit sequence), but there is
still a very, very small chance! The CGIplus EOF string is obtained from the
logical name CGIPLUSEOF, defined in the subprocess's process table, using the
language's equivalent of F$TRNLNM(), SYS$TRNLNM(), or a getenv() call in the C
Language. This string will always contain less than 64 characters and comprise
only common, printable characters. It must be written at the conclusion of a
request's output to the output stream as a single record (line) but may also
contain a <CR><LF> or just <LF> trailing carriage-control (to allow for
programming language constraints). See examples in HT_ROOT:[SRC.CGIPLUS]

HTTP input (raw request stream, header and any body) is still available to the
CGIplus script.

Multiple CGIplus scripts may be executing in subprocesses at any one time.
This includes multiple instances of any particular script.  It is the server's
task to track these, distributing appropriate requests to idle subprocesses,
monitoring those currently processing requests, creating new instances if and
when necessary, and deleting the least-used, idle CGIplus subprocesses when
configurable thresholds are reached.

A CGIplus script can be terminated by the server at any time (the subprocess
SYS$DELPRC()ed) so resources should be largely quiescent when not actually
processing. CGIplus subprocesses may also be terminated from the command line
using STOP/ID=. The server administration menu provides a simple mechansim to
purge (stop) all CGIplus processes, allowing the server to be flushed of all
subprocesses. This can be useful if some new compilation of a CGIplus script
executable needs to made available.

CGIplus scripts are differentiated from "normal" CGI scripts in the mapping
rule configuration file using the "script+" and "exec+" directives.  Of course
it would be possible to design a script to simply modify it's behaviour so it
was possible to execute in both environments.  Simply detecting the presence
or absence of one of the "normal" CGI variables (i.e. DCL symbols,
e.g. WWW_PATH_INFO, WWW_CGI_GATEWAY_INTERFACE, etc.) would be sufficient
indication.

See examples in HT_ROOT:[SRC.CGIPLUS]

April 1998 Note:  It has been observed that under rare circumstances a
persistant subprocess script-serviced request can die unexpectedly.  This has
been isolated to the following scenario.  A request underway results in a
CGIplus script terminating and the subprocess exiting.  Any output in the
script's C RTL buffers is flushed during exit, including possibly the CGI
output EOF indicator, generating an I/O AST which is serviced indicating the
request is complete and the CGIplus script is ready for reuse.  In fact it
isn't because the process is running-down.  Before the subprocess termination
AST can be processed another AST containing a request for that same CGIplus
script is serviced.  The DCL task structure is allocated to the new request but
shortly, possibly immediately after, that DCL task receives the termination AST
and is run-down.  The request receives no output and Netscape Navigator for
instance reports "Document contains no data".  This has been addressed by
checking whether the script has begun processing (by reading from the COMMAND
or CGIPLUSIN variable stream.  If this stream has not been read from it is
considered the above has happened and the script request is resubmitted to
DclBegin().  A limit is placed on the number of times this may happen in
succession, to prevent an errant script from producing a runaway condition in
the server.


ZOMBIES
-------
The reuse of DCL subprocesses for CGI scripting provides very significant
performance gains with very little _real_ possibility of undesirable
interaction between uses (where scripts are "well-behaved", which should be
all enviornments). When a non-zero zombie lifetime is specified DCL
subprocesses implicitly persist between uses for standard CGI and DCL (SSI)
commands as well as explicitly with CGIplus scripts. When not being used to
process a request these subprocesses are termed "zombies" ;^) they are neither
"alive" (executing a script and processing a request) nor are they "dead"
(subprocess deleted and task structure free). Instead the subprocess is in a
LEF state waiting for more input via CGIPLUSIN. A great deal of care is
taken to ensure there is no interaction between uses (all symbols are deleted,
output mailbox is emptied), but there can be no "iron-clad" guarantee. Use of
zombies (persistant DCL processes) is disabled by setting the appropriate
configuration parameter to zero. The same DCL EOF mechansism is used to signal
end-of-output in all process-persistant environments.


SCRIPT DEFINITIONS
------------------
File types (extensions) and associated scripting languages can be defined in
the configuration file.  The syntax is "type foreign-verb".  For example:

  .PL  $PERL_EXE:PERL.EXE
  .CGI PERL

Two are predefined, ".COM" for DCL procedures, and ".EXE" for executables.


LOGICAL NAMES
-------------
HTTPD$VERIFY    if defined, turns on script/DCL-level verify


VERSION HISTORY
---------------
15-AUG-98  MGD  replace per-subprocess timers with DclSupervisor()
27-MAY-98  MGD  only generate CGI variables once
02-APR-98  MGD  see "April 1998 Note" above,
                report status 500/501 if script returns no output
28-MAR-98  MGD  ensure script output is null-terminated (for CGI.c processing)
16-DEC-97  MGD  generalized CGI processing unbundled into CGI.c module
06-DEC-97  MGD  resolving a suspected inconsistent AST delivery situation
                by requiring all $QIO()s with an AST routine to ensure any
                queueing errors, etc. are reported via the AST routine
19-OCT-97  MGD  extensible script run-time environment,
                HTTP_ACCEPT_CHARSET, HTTP_FORWARDED and HTTP_HOST variables,
                change in behaviour: CGI "Content-Type:" response bodies now
                only have carriage-control checked/adjusted if type "text/..."
10-SEP-97  MGD  add "!'F$VERIFY(0)" to start of DCL in case verify on account
09-AUG-97  MGD  ACCEPT_LANGUAGE variable
01-AUG-97  MGD  allow supplying request header AS WELL AS body, or only body,
                added AUTH_REALM, AUTH_GROUP, HTTP_ACCEPT, and 
                REQUEST_TIME_GMT/LOCAL CGI variables
01-JUN-97  MGD  Persistant DCL processes, CGIplus, new for HTTPd v4.2,
                DCL.C completely re-designed!
26-APR-97  MGD  bugfix; serious flaw POST content handling since modifications
                to REQUEST.C, PUT.C in version 4.0 (rewrite of HTTP$INPUT)
01-FEB-97  MGD  HTTPd version 4
14-NOV-96  MGD  bugfix; no status was being returned after "DELETE/SYMBOL X"
                in DclSysCommandStringSymbol()
06-APR-96  MGD  miscellaneous refinements
26-MAR-96  MGD  added WWW_HTTP_AUTHENTICATION, allowing scripts to authenticate
01-DEC-95  MGD  HTTPd version 3
19-SEP-95  MGD  changed carriage-control on records from <CR><LF> (the strict
                HTTP requirement) to single newline (<LF>, de facto standard)
                This will be slightly more efficient, and "more compliant"!
21-APR-95  MGD  bugfix; DclSysOutputAst()
03-APR-95  MGD  added remote user authentication and CGI symbol
20-MAR-95  MGD  bugfix; DclQioHttpInput()
20-DEC-94  MGD  initial development as a module for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */

/* cmbdef.h is not defined for VAXC 3.n */
#define CMB$M_READONLY 0x01
#define CMB$M_WRITEONLY 0x02

#include <descrip.h>
#include <dvidef.h>

#include <iodef.h>
/* these should be, but are not defined for VAXC 3.n */
#define IO$M_READERCHECK 0x100
#define IO$M_WRITERCHECK 0x200
#define IO$M_READERWAIT 0x400
#define IO$M_WRITERWAIT 0x800

#include <jpidef.h>
#include <prvdef.h>
#include <rms.h>

#include <ssdef.h>
/* these should be, but are not defined for VAXC 3.n */
#define SS$_NOREADER 9412
#define SS$_NOWRITER 9420

#include <stsdef.h>
#include <syidef.h>

/* application header files */
#include "wasd.h"
#include "cgi.h"
#include "dcl.h"
#include "error.h"
#include "httpd.h"
#include "msg.h"
#include "net.h"
#include "support.h"

/******************/
/* global storage */
/******************/

boolean  DclUseZombies;

int  DclCgiPlusLifeTimePurgeCount,
     DclHitHardLimitCount,
     DclPurgeCount,
     DclPurgeAllSubprocessesCount,
     DclSoftLimitPurgeCount,
     DclSubprocessCount,
     DclSubprocessHardLimit,
     DclSubprocessSoftLimit,
     DclZombieLifeTimePurgeCount;

struct ListHeadStruct  DclTaskList;

#define DEFAULT_CGI_VARIABLE_PREFIX "WWW_"
char DclCgiVariablePrefix [32] = DEFAULT_CGI_VARIABLE_PREFIX;
int DclCgiVariablePrefixLength = sizeof(DEFAULT_CGI_VARIABLE_PREFIX)-1;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  DclSysOutputSize;
extern int  NetReadBufferSize;
extern int  ServerPort;
extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern struct AccountingStruct Accounting;
extern struct ConfigStruct Config;
extern struct MsgStruct Msgs;

/*****************************************************************************/
/*
Set and ensure limits are reasonable at server startup.
*/ 

DclInit ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclInit()\n");

   DclSubprocessSoftLimit = Config.DclSubprocessSoftLimit;
   if (DclSubprocessSoftLimit <= 0 || DclSubprocessSoftLimit > 99)
      DclSubprocessSoftLimit = Config.Busy;

   DclSubprocessHardLimit = Config.DclSubprocessHardLimit;
   if (DclSubprocessHardLimit <= DclSubprocessSoftLimit ||
       DclSubprocessHardLimit > 99)
      DclSubprocessHardLimit = DclSubprocessSoftLimit +
                               (DclSubprocessSoftLimit / 4);

   if (Config.DclZombieLifeTime) DclUseZombies = true;
}

/*****************************************************************************/
/*
This function does not return a status value.  If an error occurs the
'NextTaskFunction()' is executed.  The calling routine may assume that this
module will always execute the 'NextTaskFunction()' at some stage.
*/ 

DclBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *DclCommand,
char *CgiFileName
)
{
   register struct DclTaskStruct  *tkptr;

   int  status,
        Count,
        Length;
   char  *ContentPtr,
         *RunTimePtr,
         *ScriptFileTypePtr;

   /*********/
   /* begin */
   /*********/

   if (CgiFileName == NULL) CgiFileName = "";
   if (DclCommand == NULL) DclCommand = "";

   if (Debug)
   {
      fprintf (stdout,
"DclBegin()\n\
DclCommand     |%s|\n\
CgiFileName    |%s|\n\
ScriptName     |%s|%d|\n\
PathInfoPtr    |%s|\n\
QueryStringPtr |%s|\n",
      DclCommand, CgiFileName, rqptr->ScriptName, rqptr->IsCgiPlusScript,
      rqptr->PathInfoPtr, rqptr->QueryStringPtr);
   }

   /*****************/
   /* begin request */
   /*****************/

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /************************************/
   /* check DCL procedure/image exists */
   /************************************/

   if (CgiFileName[0])
   {
      if (VMSnok (status =
          DclFindScript (rqptr, CgiFileName,
                         &RunTimePtr, &ScriptFileTypePtr)))
      {
         if (status == RMS$_FNF)
         {
            rqptr->ResponseStatusCode = 404;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_SCRIPT_NOT_FOUND), FI_LI);
         }
         /* else assumes error message generated by DclFindScript() */
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
   }

   /***************************/
   /* allocate task structure */
   /***************************/

   if (VMSnok (DclAllocateTask (rqptr)))
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /*********************************/
   /* initialize the task structure */
   /*********************************/

   /* get a local pointer to the newly allocated DCL task structure */
   tkptr = rqptr->DclTaskPtr;

   tkptr->NextTaskFunction = NextTaskFunction;

   rqptr->CgiScript = tkptr->CgiScript;
   rqptr->CgiPlusScript = tkptr->CgiPlusScript;

   if (rqptr->ScriptName[0])
      strcpy (tkptr->ScriptName, rqptr->ScriptName);
   else
      tkptr->ScriptName[0] = '\0';

   if (DclCommand[0])
   {
      strncpy (tkptr->DclCommand, DclCommand, sizeof(tkptr->DclCommand));
      tkptr->DclCommand[sizeof(tkptr->DclCommand)-1] = '\0';
   }
   else
      tkptr->DclCommand[0] = '\0';

   if (CgiFileName[0])
      strcpy (tkptr->CgiFileName, CgiFileName);
   else
      tkptr->CgiFileName[0] = '\0';

   memcpy (&tkptr->LastUsedBinaryTime, &rqptr->BinaryTime, 8);

   if (tkptr->CgiScript || tkptr->CgiPlusScript)
      rqptr->CgiAddNewline = rqptr->CgiBufferRecords =
         rqptr->CgiContentTypeText = false;
   else
      rqptr->CgiAddNewline = rqptr->CgiBufferRecords =
         rqptr->CgiContentTypeText = true;

   if (tkptr->IsDclCommand)
      Accounting.DoDclCommandCount++;

   if (!rqptr->AccountingDone)
   {
      rqptr->AccountingDone = true;
      if (rqptr->CgiScript)
         Accounting.DoScriptCount++;
      else
      if (tkptr->CgiPlusScript)
      {
         Accounting.DoCgiPlusScriptCount++;
         if (tkptr->CgiPlusUsageCount > 1)
            Accounting.DclCgiPlusReusedCount++;
      }
   }

   /*********************************************/
   /* provide data to subprocess' input streams */
   /*********************************************/

   if (Debug)
      fprintf (stdout, "CgiScript: %d CgiPlusScript: %d DclCommand |%s|\n",
               tkptr->CgiScript, tkptr->CgiPlusScript, tkptr->DclCommand);
   if (tkptr->CgiScript || tkptr->CgiPlusScript)
   {
      /* provide request stream (HTTP$INPUT) to subprocess */
      if (Debug) 
         fprintf (stdout, "|%d|%s|\n",
                  rqptr->ContentLength, rqptr->ContentBufferPtr);

      /* initialize content information (POST, PUT, etc., only) */
      tkptr->ContentPtr = rqptr->ContentBufferPtr;
      tkptr->ContentLength = rqptr->ContentLength;

      if (Config.DclFullRequest)
      {
         /* supply request header and body (pre-4.3 behaviour) */
         if (VMSnok (status =
             sys$qio (0, tkptr->HttpInputChannel,
                      IO$_WRITELBLK, &tkptr->HttpInputIOsb,
                      &DclHttpInputAst, tkptr,
                      rqptr->RequestHeaderPtr, rqptr->RequestHeaderLength,
                      0, 0, 0, 0)))
         {
            /* report this error via the AST */
            tkptr->HttpInputIOsb.Status = status;
            SysDclAst (&DclHttpInputAst, tkptr);
         }
      }
      else
      if (tkptr->ContentLength)
      {
         /* supply only the body of the request (standard CGI behaviour) */
         if (tkptr->ContentLength > NetReadBufferSize)
            Length = NetReadBufferSize;
         else
            Length = tkptr->ContentLength;
         ContentPtr = tkptr->ContentPtr;
         tkptr->ContentPtr += Length;
         tkptr->ContentLength -= Length;

         if (VMSnok (status =
             sys$qio (0, tkptr->HttpInputChannel,
                      IO$_WRITELBLK, &tkptr->HttpInputIOsb,
                      &DclHttpInputAst, tkptr,
                      ContentPtr, Length,
                      0, 0, 0, 0)))
         {
            /* report this error via the AST */
            tkptr->HttpInputIOsb.Status = status;
            SysDclAst (&DclHttpInputAst, tkptr);
         }
      }
      else
      {
         /* the request has no body */
         if (VMSnok (status =
             sys$qio (0, tkptr->HttpInputChannel,
                      IO$_WRITEOF, &tkptr->HttpInputIOsb,
                      &DclHttpInputAst, tkptr,
                      0, 0, 0, 0, 0, 0)))
         {
            /* report this error via the AST */
            tkptr->HttpInputIOsb.Status = status;
            SysDclAst (&DclHttpInputAst, tkptr);
         }
      }
      tkptr->QueuedHttpInput++;
   }

   if (tkptr->CgiScript)
   {
      tkptr->QueuedSysCommandAllowed = 0;
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_SCRIPT_DCL_ENVIRONMENT);
      DclCgiScriptSysCommand (tkptr, "", CgiFileName,
                              RunTimePtr, ScriptFileTypePtr);
   }
   else
   if (tkptr->CgiPlusScript)
   {
      /* allow for the outstanding queued "STOP/ID=0" and EOF */
      tkptr->QueuedSysCommandAllowed = 2;
      if (tkptr->QueuedSysCommand < tkptr->QueuedSysCommandAllowed)
         DclCgiPlusScriptSysCommand (tkptr,
            CgiFileName, RunTimePtr, ScriptFileTypePtr);
      /* here comes the CGI variable stream */
      DclCgiPlusScriptCgiPlusIn (tkptr);
   }
   else
   {
      tkptr->QueuedSysCommandAllowed = 0;
      DclCgiScriptSysCommand (tkptr, DclCommand, "", "", "");
   }

   /* remove default error message */
   rqptr->ErrorTextPtr = NULL;
}

/*****************************************************************************/
/*
Allocate a DCL task structure to the request. All task structures are linked
together in a single list, function and state indicated by the various flags
and counters associated with each. If a CGIplus task script is to be executed
then check for an existing, idle CGIplus task structure executing that
particular script. If none found, or CGIplus script not required, and zombies
in use (persistant-subprocesses) then look through the list for an idle zombie
subprocess. If no zombie available (or not enabled) then check if we have
reached the subprocess creation hard-limit. If not reached look through the
list for an existing but no-subprocess-executing DCL task structure. If none
found create an additional DCL task structure and add it to the list.
Initialize the task structure (no matter from which scan it originated) and if
necessary create a subprocess for it. (An obvious improvement to processing
would be to have multiple lists, but that will have to wait for another time
:^) If an error is encountered an error message is generated and the error
status returned. It is up to the calling routine to abort the processing.
*/

int DclAllocateTask (struct RequestStruct *rqptr)

{
   register struct DclTaskStruct  *tkptr;
   register struct ListEntryStruct  *leptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclAllocateTask()\n");

   tkptr = NULL;

   if (rqptr->IsCgiPlusScript)
   {
      /******************************************************/
      /* look for an unused instance of this CGIplus script */
      /******************************************************/

      for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
      {
         tkptr = (struct DclTaskStruct*)leptr;

         if (Debug) DclTaskItemDebug (leptr, tkptr);
 
         if (!tkptr->SubprocessPid ||
             !tkptr->CgiPlusScript ||
             tkptr->IsMarkedForDelete ||
             tkptr->QueuedSysCommand > tkptr->QueuedSysCommandAllowed ||
             tkptr->QueuedSysOutput ||
             tkptr->QueuedCgiPlusIn ||
             tkptr->QueuedHttpInput ||
             tkptr->rqptr ||
             !strsame (rqptr->ScriptName, tkptr->ScriptName, -1))
         {
            tkptr = NULL;
            continue;
         }

         break;
      }

      if (Debug) fprintf (stdout, "CGIplus tkptr: %d\n", tkptr);
   }

   if (tkptr == NULL && DclUseZombies)
   {
      /************************/
      /* look for idle zombie */
      /************************/

      for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
      {
         tkptr = (struct DclTaskStruct*)leptr;

         if (Debug) DclTaskItemDebug (leptr, tkptr);

         if (!tkptr->SubprocessPid ||
             tkptr->CgiPlusScript ||
             tkptr->QueuedSysCommand ||
             tkptr->QueuedSysOutput ||
             tkptr->QueuedCgiPlusIn ||
             tkptr->QueuedHttpInput ||
             tkptr->rqptr != NULL)
         {
            tkptr = NULL;
            continue;
         }
         break;
      }

      if (Debug) fprintf (stdout, "idle zombie tkptr: %d\n", tkptr);
   }

   if (tkptr == NULL && DclSubprocessCount >= DclSubprocessHardLimit)
   {
      /* let's see if we can do something about it! */
      DclSubprocessPurge ();

      DclHitHardLimitCount++;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_SCRIPT_HARD_LIMIT), FI_LI);
      return (STS$K_ERROR);
   }

   if (tkptr == NULL)
   {
      /********************************/
      /* look for free task structure */
      /********************************/

      for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
      {
         tkptr = (struct DclTaskStruct*)leptr;

         if (Debug) DclTaskItemDebug (leptr, tkptr);

         if (tkptr->SubprocessPid ||
             tkptr->QueuedSysCommand ||
             tkptr->QueuedSysOutput ||
             tkptr->QueuedCgiPlusIn ||
             tkptr->QueuedHttpInput ||
             tkptr->rqptr != NULL)
         {
            tkptr = NULL;
            continue;
         }
         break;
      }

      if (Debug) fprintf (stdout, "free tkptr: %d\n", tkptr);
   }

   if (tkptr == NULL)
   {
      /* if we're getting short of subprocesses then start purging */
      if (DclSubprocessCount >= DclSubprocessSoftLimit) DclSubprocessPurge ();
   }

   if (tkptr == NULL)
   {
      /*********************/
      /* create a new task */
      /*********************/

      tkptr = VmGet (sizeof(struct DclTaskStruct));
      if (Debug) fprintf (stdout, "new tkptr: %d\n", tkptr);

      /*
         Allocate memory in the DCL task for SYS$OUTPUT buffer.
         Allow two bytes for carriage control and terminating null.
      */
      tkptr->SysOutputPtr = VmGet (DclSysOutputSize+2);
      tkptr->SysOutputSize = DclSysOutputSize;

      if (VMSnok (status = DclCreateMailboxes (tkptr)))
      {
         VmFree (tkptr->SysOutputPtr);
         VmFree (tkptr);
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_SCRIPT_IPC);
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }

      ListAddTail (&DclTaskList, tkptr);
   }

   /*******************/
   /* initialize task */
   /*******************/

   /* associate the DCL task and the request */
   rqptr->DclTaskPtr = tkptr;
   tkptr->rqptr = rqptr;          

   tkptr->TotalUsageCount++;
   tkptr->IsMarkedForDelete = false;

   if (rqptr->ScriptName[0])
   {
      /* a script is being executed */
      if (rqptr->IsCgiPlusScript)
      {
         if (Debug) fprintf (stdout, "CGIplus script\n");
         tkptr->CgiPlusScript = true;
         tkptr->IsDclCommand = tkptr->CgiScript = false;
         tkptr->CgiPlusUsageCount++;

         /* give it three-score years and ten if life-time is specified */
         if (Config.DclCgiPlusLifeTime)
         {
            /* plus one allows at least that period */
            tkptr->LifeTimeCount = Config.DclCgiPlusLifeTime+1;
            DclSupervisor (false);
         }

         /* reset the zombie count, CGIplus scripts never become zombies */
         tkptr->ZombieCount = 0;

         /* CGIplus MUST retain the original EOF until subprocess deletion */
         rqptr->CgiEofPtr = tkptr->DclEof;
         rqptr->CgiEofLength = tkptr->DclEofLength;
      }
      else
      {
         if (Debug) fprintf (stdout, "CGI script\n");
         tkptr->CgiScript = true;
         tkptr->IsDclCommand = tkptr->CgiPlusScript = false;
         tkptr->CgiPlusUsageCount = 0;

         /* limited life in the twilight zone */
         if (DclUseZombies && Config.DclZombieLifeTime)
         {
            /* plus one allows at least that period */
            tkptr->LifeTimeCount = Config.DclZombieLifeTime+1;
            DclSupervisor (false);
         }

         if (DclUseZombies)
         {
            /* always generate a new EOF string for standard CGI scripts */
            CgiEof (tkptr->DclEof, &tkptr->DclEofLength);
            rqptr->CgiEofPtr = tkptr->DclEof;
            rqptr->CgiEofLength = tkptr->DclEofLength;
         }
      }

      tkptr->CgiScriptActivated = tkptr->CgiScriptResponded = false;
      rqptr->CgiScriptRetryCount = 0;
   }
   else
   {
      if (Debug) fprintf (stdout, "DCL command\n");
      tkptr->IsDclCommand = true;
      tkptr->CgiScript = tkptr->CgiPlusScript = false;
      tkptr->CgiPlusUsageCount = 0;

      /* limited life in the twilight zone */
      if (DclUseZombies && Config.DclZombieLifeTime)
      {
         /* plus one allows at least that period */
         tkptr->LifeTimeCount = Config.DclZombieLifeTime+1;
         DclSupervisor (false);
      }

      if (DclUseZombies)
      {
         /* always generate a new EOF string for DCL commands */
         CgiEof (tkptr->DclEof, &tkptr->DclEofLength);
         rqptr->CgiEofPtr = tkptr->DclEof;
         rqptr->CgiEofLength = tkptr->DclEofLength;
      }
   }

   if (tkptr->SubprocessPid)
   {
      /*********************/
      /* subprocess exists */
      /*********************/

      /* queue the initial read of the subprocess' SYS$OUTPUT */
      DclQioSysOutput (tkptr);

      return (SS$_NORMAL);
   }

   /*********************/
   /* create subprocess */
   /*********************/

   if (VMSok (status = DclCreateSubprocess (tkptr)))
   {
      /* if a CGIplus script and a life-time is specified then set it */
      if (tkptr->CgiPlusScript && Config.DclCgiPlusLifeTime)
      {
         /* plus one allows at least that period */
         tkptr->LifeTimeCount = Config.DclCgiPlusLifeTime+1;
         DclSupervisor (false);
      }

      tkptr->ZombieCount = 0;
      DclSubprocessCount++;
      return (status);
   }
   else
   {
      /* disassociate the DCL task and request structures */
      rqptr->DclTaskPtr = tkptr->rqptr = NULL;
      return (status);
   }
}

/*****************************************************************************/
/*
Called if the HTTPd server is being restarted (image run down and then
restarted).  Does not need to be done if process is exiting.  This function
should be included in the image's onexit() function.
*/

DclExit ()

{
   register struct DclTaskStruct  *tkptr;
   register struct ListEntryStruct  *leptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclExit()\n");

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;
      if (!tkptr->SubprocessPid) continue;
      status = sys$delprc (&tkptr->SubprocessPid, 0);
      if (Debug)
         fprintf (stdout, "sys$delprc() %d %08.08X %%X%08.08X\n",
                  tkptr, tkptr->SubprocessPid, status);
   }
}

/*****************************************************************************/
/*
The maximum number of concurrent subprocesses has been reached. Look through
the DCL task structure for a CGIplus subprocess not currently in use. Find the
least used subprocess and delete it.
*/

DclSubprocessPurge ()

{
   register struct DclTaskStruct  *tkptr;
   register struct ListEntryStruct  *leptr;

   int  status,
        MinUsageCount;
   struct DclTaskStruct  *MinTkPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DclSubprocessPurge() %d %d\n",
               DclSubprocessCount, DclSubprocessSoftLimit);

   DclPurgeCount++;
   MinUsageCount = 999999999;
   MinTkPtr = NULL;

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;

      if (Debug) DclTaskItemDebug (leptr, tkptr);

      if (!tkptr->SubprocessPid ||
          !tkptr->CgiPlusScript ||
          tkptr->QueuedSysOutput ||
          tkptr->QueuedCgiPlusIn ||
          tkptr->QueuedHttpInput ||
          tkptr->rqptr != NULL ||
          tkptr->IsMarkedForDelete)
         continue;

      if (tkptr->CgiPlusUsageCount < MinUsageCount)
         MinUsageCount = (MinTkPtr = tkptr)->CgiPlusUsageCount;
   }

   if (MinTkPtr == NULL) return;

   MinTkPtr->IsMarkedForDelete = true;
   DclConcludeTask (MinTkPtr, true);
   DclSoftLimitPurgeCount++;
}

/*****************************************************************************/
/*
This function can be called at any stage to rundown or abort a DCL task.
If there is still outstanding IO this is cancelled as appropriate to task
rundown or abort.  If no outstanding IO then if there is an associated request
that request's next task function is called.
*/

DclConcludeTask
(
struct DclTaskStruct *tkptr,
boolean AbortTask
)
{
   int  status;
   char  CgiFileName [256];
   void  *NextTaskFunction;
   struct RequestStruct  *RequestPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "DclConcludeTask(%d) %d %d %d %d %d %d %08.08X\n",
         AbortTask, tkptr, tkptr->rqptr, tkptr->QueuedSysCommand,
         tkptr->QueuedSysOutput, tkptr->QueuedCgiPlusIn,
         tkptr->QueuedHttpInput, tkptr->SubprocessPid);
   }

   if (AbortTask && tkptr->SubprocessPid)
   {
      tkptr->IsMarkedForDelete = true;
      tkptr->QueuedSysCommandAllowed = 0;
   }

   if (tkptr->QueuedSysCommand > tkptr->QueuedSysCommandAllowed ||
       tkptr->QueuedSysOutput ||
       tkptr->QueuedCgiPlusIn ||
       tkptr->QueuedHttpInput)
   {
      /****************/
      /* rundown task */
      /****************/

      if (AbortTask)
      {
         /* cancel everything everywhere! */
         sys$cancel (tkptr->SysCommandChannel);
         sys$cancel (tkptr->SysOutputChannel);
         sys$cancel (tkptr->CgiPlusInChannel);
         sys$cancel (tkptr->HttpInputChannel);
      }
      else
      {
         /* cancel any outstanding I/O on input channels */
         if (tkptr->QueuedCgiPlusIn) sys$cancel (tkptr->CgiPlusInChannel);
         if (tkptr->QueuedHttpInput) sys$cancel (tkptr->HttpInputChannel);
         if (tkptr->QueuedSysCommand > tkptr->QueuedSysCommandAllowed)
            sys$cancel (tkptr->SysCommandChannel);
      }
   }
   else
   {
      /*********************/
      /* task has finished */
      /*********************/

      if (tkptr->rqptr != NULL)
      {
         /* check if the script exited before actually starting! */
         if (!tkptr->CgiScriptActivated &&
             tkptr->rqptr->CgiScriptRetryCount <= DclSubprocessSoftLimit)
         {
            /* yup, definitely suspect, get rid of it */
            tkptr->IsMarkedForDelete = true;
            tkptr->rqptr->CgiScriptRetryCount++;

            /* retrieve necessary information from the current DCL task */
            RequestPtr = tkptr->rqptr;
            NextTaskFunction = tkptr->NextTaskFunction;
            strcpy (CgiFileName, tkptr->CgiFileName);

            /* disassociate the DCL task and request structures */
            tkptr->rqptr->DclTaskPtr = NULL;
            tkptr->rqptr = tkptr->NextTaskFunction = NULL;

            /* restart using a new DCL task */
            DclBegin (RequestPtr, NextTaskFunction, NULL, CgiFileName);
         }
         else
         {
            if (!tkptr->CgiScriptResponded)
            {
               /* hmmm, script has not provided any output! */
               if (tkptr->rqptr->HttpMethod == HTTP_METHOD_GET)
               {
                  /* take the blame for general GET method failures */
                  tkptr->rqptr->ResponseStatusCode = 500;
                  ErrorGeneral (tkptr->rqptr,
                     MsgFor(tkptr->rqptr,MSG_GENERAL_INTERNAL), FI_LI);
               }
               else
               {
                  /* other methods are probably not implemented by the script */
                  tkptr->rqptr->ResponseStatusCode = 501;
                  ErrorGeneral (tkptr->rqptr,
                     MsgFor(tkptr->rqptr,MSG_REQUEST_METHOD), FI_LI);
               }
            }

            /* still has an associated request, declare the next task */
            SysDclAst (tkptr->NextTaskFunction, tkptr->rqptr);

            /* disassociate the DCL task and request structures */
            tkptr->rqptr->DclTaskPtr = NULL;
            tkptr->rqptr = tkptr->NextTaskFunction = NULL;
         }
      }

      if (DclUseZombies || tkptr->CgiPlusScript)
      {
         /* empty any remains in sys$output! */
         DclEmptySysOutput (tkptr);
      }

      /* if marked for process deletion */
      if (tkptr->IsMarkedForDelete)
      {
         tkptr->IsMarkedForDelete = false;
         if (tkptr->SubprocessPid)
         {
            status = sys$delprc (&tkptr->SubprocessPid, 0);
            if (Debug)
               fprintf (stdout, "sys$delprc() %08.08X %%X%08.08X\n",
                        tkptr->SubprocessPid, status);
         }
      }

      if (DclUseZombies && !tkptr->CgiPlusScript) tkptr->ZombieCount++;
   }
}

/*****************************************************************************/
/*
If an error is encountered an error message is generated and the error status
returned.  It is up to the calling routine to abort the processing.  Queue a
writer-wait I/O to the SYS$OUTPUT channel to stall I/O until the subprocess
has started.
*/ 

DclCreateSubprocess (struct DclTaskStruct *tkptr)

{
   /* spawn DCL subprocess with account's authorized privileges */
   static unsigned long  /* authpriv,trusted,nowait,noclisym,nolognam,nokey */
                         AuthPrivSpawnFlags = 0xcf;

   /* spawn DCL subprocess with current privileges (none at all!) */
   static unsigned long  /* nowait,noclisym,nolognam,nokey */
                         CurrPrivSpawnFlags = 0x0f;

   static int  SubprocessNumber;
   static char  SubprocessName [16];
   static $DESCRIPTOR (SubprocessNameFaoDsc, "!15<HTTPd:!UL-!UL!>");
   static $DESCRIPTOR (SubprocessNameDsc, SubprocessName);

   register struct RequestStruct  *rqptr;

   int  status,
        Count,
        DclAstStatus;
   unsigned short  Length;
   unsigned long  SpawnFlags;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclCreateSubprocess()\n");

   /* get a pointer to the request structure */
   rqptr = tkptr->rqptr;

   /*
      Queue up an asynchronous wait for a writer channel to be assigned
      to the mailbox as the subprocess' SYS$OUTPUT.  When the channel is
      assigned the AST function will queue the first read from the mailbox.
   */
   if (VMSnok (status =
       sys$qio (0, tkptr->SysOutputChannel,
                IO$_SETMODE | IO$M_WRITERWAIT,
                &tkptr->SysOutputIOsb,
                &DclSysOutputWaitAst, tkptr,
                0, 0, 0, 0, 0, 0)))
   {
      /* report this error via the AST */
      tkptr->SysOutputIOsb.Status = status;
      SysDclAst (&DclSysOutputWaitAst, tkptr);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$qio() WRITERWAIT %%X%08.08X\n", status);
   tkptr->QueuedSysOutput++;

   if (Config.DclSpawnAuthPriv)
      SpawnFlags = AuthPrivSpawnFlags;
   else
      SpawnFlags = CurrPrivSpawnFlags;

   for (Count = 100; Count; Count--)
   {
      /* subprocesses are consecutively numbered from 1..999 */
      SubprocessNameDsc.dsc$w_length = sizeof(SubprocessName)-1;
      sys$fao (&SubprocessNameFaoDsc, &Length, &SubprocessNameDsc, 
               ServerPort, SubprocessNumber++ % 1000 + 1);
      SubprocessName[SubprocessNameDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "SubprocessName |%s|\n", SubprocessName);

      status = lib$spawn (0,
                          &tkptr->SysCommandDevNameDsc,
                          &tkptr->SysOutputDevNameDsc,
                          &SpawnFlags,
                          &SubprocessNameDsc,
                          &tkptr->SubprocessPid,
                          &tkptr->SubprocessCompletionStatus,
                          0,
                          &DclSubprocessCompletionAST, tkptr,
                          0, 0, 0);
      if (Debug)
         fprintf (stdout, "lib$spawn() %%X%08.08X PID: %08.08X\n",
                  status, tkptr->SubprocessPid);

      if (status != SS$_DUPLNAM) break;
   }

   if (VMSok (status))
   {
      Accounting.DclSpawnCount++;
      return (status);
   }

   /* cancel the writer-wait IO */
   sys$cancel (tkptr->SysOutputChannel);

   rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_SCRIPT_SPAWN);
   ErrorVmsStatus (rqptr, status, FI_LI);
   return (status);
}

/*****************************************************************************/
/*
This AST is called when the subprocesses exits.
*/

DclSubprocessCompletionAST (struct DclTaskStruct *tkptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclSubprocessCompletionAST() %08.08X %%X%08.08X\n",
         tkptr->SubprocessPid, tkptr->SubprocessCompletionStatus);
   }

   if (tkptr->CgiPlusScript)
   {
      /* ensure SYS$COMMAND gets emptied! */
      tkptr->QueuedSysCommandAllowed = 0;

      /* may be called due to life-time expiry, but this won't hurt anyway! */
      if (Config.DclCgiPlusLifeTime) sys$cantim (tkptr, 0);
   }
   else
   if (DclUseZombies)
   {
      /* may be called due to life-time expiry, but this won't hurt anyway! */
      sys$cantim (tkptr, 0);
   }

   /* set the PID and lifetime to zero */
   tkptr->SubprocessPid = tkptr->LifeTimeCount = 0;

   /* keep track of how many subprocesses are executing */
   if (DclSubprocessCount) DclSubprocessCount--;

   /* ensure any old EOF string is not reused */
   tkptr->DclEof[0] = '\0';
   tkptr->DclEofLength = 0;

   DclConcludeTask (tkptr, false);
}

/*****************************************************************************/
/*
If a script application writes any output before it was/is terminated that
output will still be laying unread in the mailbox.  Clear any such noise lest
it be read by the next subprocess to use the mailbox.  Does this synchronously
(i.e. with QIO waits).
*/

DclEmptySysOutput (struct DclTaskStruct *tkptr)

{
   struct MbxSenseIOsb  SenseIOsb;
   int  LastMessageCount;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclEmptySysOutput()\n");

   LastMessageCount = 999999999;
   SenseIOsb.MessageCount = 0;

   for (;;)
   {
      sys$qiow (0, tkptr->SysOutputChannel,
                IO$_SENSEMODE, &SenseIOsb,
                0, 0, 0, 0, 0, 0, 0, 0);

      if (VMSnok (SenseIOsb.Status))
         ErrorExitVmsStatus (SenseIOsb.Status, "sys$qiow()", FI_LI);

      if (Debug)
         fprintf (stdout, "sys$qiow() IO$_SENSEMODE %%X%08.08X %d %d\n", 
                  SenseIOsb.Status, SenseIOsb.MessageCount,
                  SenseIOsb.MessageBytes);

      if (!SenseIOsb.MessageCount) break;
      /* potential infinite loop, check message count is decreasing! */
      if (LastMessageCount <= SenseIOsb.MessageCount) break;
      LastMessageCount = SenseIOsb.MessageCount;

      sys$qiow (0, tkptr->SysOutputChannel,
                IO$_READLBLK | IO$M_WRITERCHECK,
                &tkptr->SysOutputIOsb, 0, 0,
                tkptr->SysOutputPtr, tkptr->SysOutputSize, 0, 0, 0, 0);

      if (Debug)
      {
         tkptr->SysOutputPtr[tkptr->SysOutputIOsb.Count] = '\0';
         fprintf (stdout, "sys$qiow() IO$_READLBLK %%X%08.08X %d\n|%s|\n", 
                  tkptr->SysOutputIOsb.Status, tkptr->SysOutputIOsb.Count,
                  tkptr->SysOutputPtr);
      }
   }
}

/*****************************************************************************/
/*
If an error is encountered an error message is generated and the error status
returned.  It is up to the calling routine to abort the processing.  Create
four mailboxes that will be associated with the subprocess I/O streams.  If
an error occurs any mailbox created up to that point is deleted and the
channel set back to zero.
*/ 

DclCreateMailboxes (struct DclTaskStruct *tkptr)

{
   static unsigned long  DevNamItem = DVI$_DEVNAM;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclCreateMailboxes()\n");

   /**********************************/
   /* create the SYS$COMMAND mailbox */
   /**********************************/

   if (VMSnok (status =
       sys$crembx (0,
                   &tkptr->SysCommandChannel,
                   DclSysOutputSize/2, DclSysOutputSize/2,
                   SubprocessMbxProtectionMask,
                   0, 0, CMB$M_WRITEONLY)))
   {
      if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
      return (status);
   }

   tkptr->SysCommandDevNameDsc.dsc$w_length = sizeof(tkptr->SysCommandDevName);
   tkptr->SysCommandDevNameDsc.dsc$a_pointer = tkptr->SysCommandDevName;
   tkptr->SysCommandDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
   tkptr->SysCommandDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

   if (VMSnok (status =
       lib$getdvi (&DevNamItem, &tkptr->SysCommandChannel,
                   0, 0, &tkptr->SysCommandDevNameDsc, &Length)))
   {
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      return (status);
   }
   tkptr->SysCommandDevName
      [tkptr->SysCommandDevNameDsc.dsc$w_length = Length] = '\0';
   if (Debug)
      fprintf (stdout, "SysCommandDevName |%s|\n", tkptr->SysCommandDevName);

   /*********************************/
   /* create the SYS$OUTPUT mailbox */
   /*********************************/

   if (VMSnok (status =
       sys$crembx (0,
                   &tkptr->SysOutputChannel,
                   DclSysOutputSize, DclSysOutputSize,
                   SubprocessMbxProtectionMask,
                   0, 0, CMB$M_READONLY)))
   {
      if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      return (status);
   }

   tkptr->SysOutputDevNameDsc.dsc$w_length = sizeof(tkptr->SysOutputDevName);
   tkptr->SysOutputDevNameDsc.dsc$a_pointer = tkptr->SysOutputDevName;
   tkptr->SysOutputDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
   tkptr->SysOutputDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

   if (VMSnok (status =
       lib$getdvi (&DevNamItem, &tkptr->SysOutputChannel,
                   0, 0, &tkptr->SysOutputDevNameDsc, &Length)))
   {
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      sys$dassgn (tkptr->SysOutputChannel);
      tkptr->SysOutputChannel = 0;
      tkptr->HttpInputChannel = tkptr->SysCommandChannel = 0;
      return (status);
   }
   tkptr->SysOutputDevName
      [tkptr->SysOutputDevNameDsc.dsc$w_length = Length] = '\0';
   if (Debug)
      fprintf (stdout, "SysOutputDevName |%s|\n", tkptr->SysOutputDevName);

   /********************************/
   /* create the CGIPLUSIN mailbox */
   /********************************/

   if (VMSnok (status =
       sys$crembx (0,
                   &tkptr->CgiPlusInChannel,
                   DclSysOutputSize/2, DclSysOutputSize/2,
                   SubprocessMbxProtectionMask,
                   0, 0, CMB$M_WRITEONLY)))
   {
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      sys$dassgn (tkptr->SysOutputChannel);
      tkptr->SysOutputChannel = 0;
      if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
      return (status);
   }

   tkptr->CgiPlusInDevNameDsc.dsc$w_length = sizeof(tkptr->CgiPlusInDevName);
   tkptr->CgiPlusInDevNameDsc.dsc$a_pointer = tkptr->CgiPlusInDevName;
   tkptr->CgiPlusInDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
   tkptr->CgiPlusInDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

   if (VMSnok (status =
       lib$getdvi (&DevNamItem, &tkptr->CgiPlusInChannel,
                   0, 0, &tkptr->CgiPlusInDevNameDsc, &Length)))
   {
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      sys$dassgn (tkptr->SysOutputChannel);
      tkptr->SysOutputChannel = 0;
      sys$dassgn (tkptr->CgiPlusInChannel);
      tkptr->CgiPlusInChannel = 0;
      return (status);
   }
   tkptr->CgiPlusInDevName
     [tkptr->CgiPlusInDevNameDsc.dsc$w_length = Length] = '\0';
   if (Debug)
      fprintf (stdout, "CgiPlusInDevName |%s|\n", tkptr->CgiPlusInDevName);

   /*********************************/
   /* create the HTTP$INPUT mailbox */
   /*********************************/

   if (VMSnok (status =
       sys$crembx (0,
                   &tkptr->HttpInputChannel,
                   NetReadBufferSize, NetReadBufferSize,
                   SubprocessMbxProtectionMask,
                   0, 0, CMB$M_WRITEONLY)))
   {
      if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      sys$dassgn (tkptr->SysOutputChannel);
      tkptr->SysOutputChannel = 0;
      sys$dassgn (tkptr->CgiPlusInChannel);
      tkptr->CgiPlusInChannel = 0;
      return (status);
   }

   tkptr->HttpInputDevNameDsc.dsc$w_length = sizeof(tkptr->HttpInputDevName);
   tkptr->HttpInputDevNameDsc.dsc$a_pointer = tkptr->HttpInputDevName;
   tkptr->HttpInputDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
   tkptr->HttpInputDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

   if (VMSnok (status =
       lib$getdvi (&DevNamItem, &tkptr->HttpInputChannel,
                   0, 0, &tkptr->HttpInputDevNameDsc, &Length)))
   {
      sys$dassgn (tkptr->SysCommandChannel);
      tkptr->SysCommandChannel = 0;
      sys$dassgn (tkptr->SysOutputChannel);
      tkptr->SysOutputChannel = 0;
      sys$dassgn (tkptr->CgiPlusInChannel);
      tkptr->CgiPlusInChannel = 0;
      sys$dassgn (tkptr->HttpInputChannel);
      tkptr->HttpInputChannel = 0;
      return (status);
   }
   tkptr->HttpInputDevName
     [tkptr->HttpInputDevNameDsc.dsc$w_length = Length] = '\0';
   if (Debug)
      fprintf (stdout, "HttpInputDevName |%s|\n", tkptr->HttpInputDevName);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
A channel has been assigned to the subprocess' SYS$OUTPUT mailbox.  If it is
still associated with an actual subprocess (PID is not empty) queue the first
asynchronous read of the subprocess' output.
*/ 

DclSysOutputWaitAst (struct DclTaskStruct *tkptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclSysOutputWaitAst() %d %d %d %d %d %d %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->QueuedSysCommand,
         tkptr->QueuedSysOutput, tkptr->QueuedCgiPlusIn,
         tkptr->QueuedHttpInput, tkptr->SysOutputIOsb.Status);
   }

   if (tkptr->QueuedSysOutput) tkptr->QueuedSysOutput--;

   /* abort if a request or subprocess is not associated with the task */
   if (tkptr->rqptr == NULL || !tkptr->SubprocessPid)
   {
      DclConcludeTask (tkptr, true);
      return;
   }

   /* if the subprocess is in the process or aborting/being deleted */
   if (tkptr->IsMarkedForDelete)
   {
      DclConcludeTask (tkptr, false);
      return;
   }

   if (VMSnok (tkptr->SysOutputIOsb.Status))
   {
      if (tkptr->SysOutputIOsb.Status == SS$_ABORT ||
          tkptr->SysOutputIOsb.Status == SS$_CANCEL)
      {
         /* looks like the I/O has been cancelled before it started! */
         DclConcludeTask (tkptr, true);
         return;
      }

      tkptr->rqptr->ErrorTextPtr = MsgFor(tkptr->rqptr,MSG_SCRIPT_IPC);
      ErrorVmsStatus (tkptr->rqptr, tkptr->SysOutputIOsb.Status, FI_LI);
      DclConcludeTask (tkptr, true);
      return;
   }

   /* queue the initial read of the subprocess' SYS$OUTPUT */
   DclQioSysOutput (tkptr);
}

/*****************************************************************************/
/*
Queue up a read from the subprocess "SYS$OUTPUT" mailbox.  When the read 
completes call function DclSysOutputAst(), do any post-processing 
required and write the data to the client over the network.  The next read 
from the subprocess via the mailbox will be queued by the network write 
completion AST function.
*/ 

DclQioSysOutput (struct DclTaskStruct *tkptr)

{
   int  status,
        DclAstStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSysOutput() %d %d\n", tkptr, tkptr->rqptr);

   if (VMSnok (status =
       sys$qio (0, tkptr->SysOutputChannel,
                IO$_READLBLK | IO$M_WRITERCHECK,
                &tkptr->SysOutputIOsb,
                &DclSysOutputAst, tkptr,
                tkptr->SysOutputPtr, tkptr->SysOutputSize, 0, 0, 0, 0)))
   {
      /* report this error via the AST */
      tkptr->SysOutputIOsb.Status = status;
      SysDclAst (&DclSysOutputAst, tkptr);
   }
   tkptr->QueuedSysOutput++;
}

/*****************************************************************************/
/*
A queued asynchronous read from the subprocess "SYS$OUTPUT" mailbox has 
completed.  If this record is part of the HTTP header do some specific 
processing.  If required append carriage-control (newline) to this record to 
make it HTTP compliant.  Queue a write of this data to the client over the 
network.  The next read from the subprocess via the mailbox will be queued by 
the network write completion AST function, DclQioSysOutput().
*/ 

DclSysOutputAst (struct DclTaskStruct *tkptr)

{
   int  status,
        value;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclSysOutputAst() %d %d %08.08X %d %d %d %d %d %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->SubprocessPid,
         tkptr->QueuedSysCommand, tkptr->QueuedSysOutput,
         tkptr->QueuedCgiPlusIn, tkptr->QueuedHttpInput,
         tkptr->SysOutputIOsb.Count, tkptr->SysOutputIOsb.Status);

/*
      tkptr->SysOutputPtr[tkptr->SysOutputIOsb.Count] = '\0';
      fprintf (stdout, "|%s|\n", tkptr->SysOutputPtr);
*/
   }

   if (tkptr->QueuedSysOutput) tkptr->QueuedSysOutput--;

   if (tkptr->SysOutputIOsb.Status == SS$_ENDOFFILE)
   {
      /* 
          If a script spawns multiple subprocesses each will
          terminate by queueing an end-of-file.  Ignore these.
          Queue the next read of the subprocess' SYS$OUTPUT, a
          SS$_NOWRITER or DCL EOF signals completion of output!
      */
      DclQioSysOutput (tkptr);
      return;
   }

   if (VMSnok (tkptr->SysOutputIOsb.Status))
   {
      if (tkptr->SysOutputIOsb.Status == SS$_NOWRITER)
      {
         /* subprocess has terminated */
         DclConcludeTask (tkptr, false);
         return;
      }

      if (tkptr->SysOutputIOsb.Status == SS$_ABORT ||
          tkptr->SysOutputIOsb.Status == SS$_CANCEL)
      {
         DclConcludeTask (tkptr, true);
         return;
      }

      tkptr->rqptr->ErrorTextPtr = MsgFor(tkptr->rqptr,MSG_SCRIPT_IPC);
      ErrorVmsStatus (tkptr->rqptr, tkptr->SysOutputIOsb.Status, FI_LI);
      DclConcludeTask (tkptr, true);
      return;
   }

   tkptr->SysOutputPtr[tkptr->SysOutputIOsb.Count] = '\0';

   value = CgiOutput (tkptr->rqptr,
                      tkptr->SysOutputPtr,
                      tkptr->SysOutputIOsb.Count);
   if (value == CGI_OUTPUT_TERMINATE)
   {
      /* terminate processing */
      DclConcludeTask (tkptr, false);
      return;
   }

   /* finally it's true */
   tkptr->CgiScriptResponded = true;

   if (value == CGI_OUTPUT_ABSORB)
   {
      /* absorb output, queue the next read of the subprocess' SYS$OUTPUT */
      DclQioSysOutput (tkptr);
      return;
   }

   /* otherwise it's the count of bytes in the output buffer */
   tkptr->SysOutputIOsb.Count = value;

   if (tkptr->rqptr->CgiBufferRecords)
   {
      /* buffer the record */
      NetWriteBuffered (tkptr->rqptr, &DclSysOutputToClientAst,
                        tkptr->SysOutputPtr, tkptr->SysOutputIOsb.Count);
      tkptr->QueuedSysOutput++;
      return;
   }

   /********************/
   /* write the record */
   /********************/

   NetWrite (tkptr->rqptr, &DclSysOutputToClientAst,
             tkptr->SysOutputPtr, tkptr->SysOutputIOsb.Count);
   tkptr->QueuedSysOutput++;
}

/*****************************************************************************/
/*
************
*** NOTE ***  This function takes a pointer to a request!!!
************  Due to is being an AST from a general network write function.

A queued asynchronous write of subprocess SYS$OUTPUT (mailbox) to the client
over the network has completed.
*/ 

DclSysOutputToClientAst (struct RequestStruct *rqptr)

{
   register struct DclTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   /* get a pointer to the DCL task from the request structure */
   tkptr = rqptr->DclTaskPtr;

   if (Debug)
   {
      fprintf (stdout,
         "DclSysOutputToClientAst() %d %d %d %d %d %d %%X%08.08X %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->QueuedSysCommand,
         tkptr->QueuedSysOutput, tkptr->QueuedCgiPlusIn,
         tkptr->QueuedHttpInput, tkptr->SysOutputIOsb.Status,
         rqptr->NetWriteIOsb.Status);
   }

   if (tkptr->QueuedSysOutput) tkptr->QueuedSysOutput--;

   /* if I/O cancelled then just return */
   if (rqptr->NetWriteIOsb.Status == SS$_ABORT ||
       rqptr->NetWriteIOsb.Status == SS$_CANCEL)
      return;

   /* abort task if NETWORK ERROR when writing TO CLIENT */
   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      DclConcludeTask (tkptr, true);
      return;
   }

   /* queue the next read of the subprocess' SYS$OUTPUT */
   DclQioSysOutput (tkptr);
}

/*****************************************************************************/
/*
Queue up a write of data to the subprocess "SYS$COMMAND" mailbox.  This is the 
subprocesses' "SYS$COMMAND", supplying the DCL commands to execute, CGI
information, etc.
*/ 

int DclQioSysCommand 
(
struct DclTaskStruct *tkptr,
char *DataPtr,
int DataLength
)
{
   int  status,
        DclAstStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DclQioSysCommand() %d %d |%s|\n",
               tkptr, tkptr->rqptr, DataPtr);

   if (DataPtr == NULL)
   {
      /* NULL pointer means write an end-of-file to the channel */
      status = sys$qio (0, tkptr->SysCommandChannel,
                        IO$_WRITEOF, &tkptr->SysCommandIOsb,
                        &DclSysCommandAst, tkptr,
                        0, 0, 0, 0, 0, 0);
   }
   else
   {
      status = sys$qio (0, tkptr->SysCommandChannel,
                        IO$_WRITELBLK, &tkptr->SysCommandIOsb,
                        &DclSysCommandAst, tkptr,
                        DataPtr, DataLength, 0, 0, 0, 0);
   }

   if (VMSnok (status))
   {
      /* report this error via the AST */
      tkptr->SysCommandIOsb.Status = status;
      SysDclAst (&DclSysCommandAst, tkptr);
   }
   tkptr->QueuedSysCommand++;
}

/*****************************************************************************/
/*
A queued write to the subprocess "SYS$COMMAND" mailbox has completed.  This is 
the subprocesses' "SYS$COMMAND", supplying the DCL commands to execute.
*/ 

DclSysCommandAst (struct DclTaskStruct *tkptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclSysCommandAst() %d %d %08.08X %d %d %d %d %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->SubprocessPid,
         tkptr->QueuedSysCommand, tkptr->QueuedSysOutput,
         tkptr->QueuedCgiPlusIn, tkptr->QueuedHttpInput,
         tkptr->SysCommandIOsb.Status);
   }

   if (tkptr->QueuedSysCommand) tkptr->QueuedSysCommand--;

   /* if (effectively) no outstanding I/O then conclude the DCL task */
   if (tkptr->QueuedSysCommand <= tkptr->QueuedSysCommandAllowed &&
       !tkptr->QueuedSysOutput &&
       !tkptr->QueuedCgiPlusIn &&
       !tkptr->QueuedHttpInput)
   {
      DclConcludeTask (tkptr, false);
      return;
   }

   /* if I/O cancelled then just return */
   if (tkptr->SysCommandIOsb.Status == SS$_ABORT ||
       tkptr->SysCommandIOsb.Status == SS$_CANCEL)
      return;

   /* don't know why, first AST is delivered %x00000000, make %X00000001 */
   if (!tkptr->SysCommandIOsb.Status) tkptr->SysCommandIOsb.Status = 1;

   /* abort if an error writing SYS$COMMAND stream to subprocess */
   if (VMSnok (tkptr->SysCommandIOsb.Status))
   {
      tkptr->rqptr->ErrorTextPtr = MsgFor(tkptr->rqptr,MSG_SCRIPT_IPC);
      ErrorVmsStatus (tkptr->rqptr, tkptr->SysCommandIOsb.Status, FI_LI);
      DclConcludeTask (tkptr, true);
      return;
   }

   /* at least one DCL command was read indicating script activation */
   tkptr->CgiScriptActivated = true;
}

/*****************************************************************************/
/*
Queue up a write of data to the subprocess "CGIPLUSIN" mailbox.  This is a
CGIplus subprocesses' CGI variable stream mailbox.             
*/ 

int DclQioCgiPlusIn 
(
struct DclTaskStruct *tkptr,
char *DataPtr,
int DataLength
)
{
   int  status,
        DclAstStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DclQioCgiPlusIn() %d %d |%s|\n",
               tkptr, tkptr->rqptr, DataPtr);

   if (DataPtr == NULL)
   {
      /* NULL pointer means write an end-of-file to the channel */
      status = sys$qio (0, tkptr->CgiPlusInChannel,
                        IO$_WRITEOF, &tkptr->CgiPlusInIOsb,
                        &DclCgiPlusInAst, tkptr,
                        0, 0, 0, 0, 0, 0);
   }
   else
   {
      status = sys$qio (0, tkptr->CgiPlusInChannel,
                        IO$_WRITELBLK, &tkptr->CgiPlusInIOsb,
                        &DclCgiPlusInAst, tkptr,
                        DataPtr, DataLength, 0, 0, 0, 0);
   }

   if (VMSnok (status))
   {
      /* report this error via the AST */
      tkptr->CgiPlusInIOsb.Status = status;
      SysDclAst (&DclCgiPlusInAst, tkptr);
   }
   tkptr->QueuedCgiPlusIn++;
}

/*****************************************************************************/
/*
A queued write to the subprocess "CGIPLUSIN" mailbox has completed.  This is a
CGIplus subprocesses' CGI variable stream mailbox.             
*/ 

DclCgiPlusInAst (struct DclTaskStruct *tkptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclCgiPlusInAst() %d %d %08.08X %d %d %d %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->SubprocessPid, tkptr->QueuedCgiPlusIn,
         tkptr->QueuedSysOutput, tkptr->QueuedHttpInput,
         tkptr->CgiPlusInIOsb.Status);
   }

   if (tkptr->QueuedCgiPlusIn) tkptr->QueuedCgiPlusIn--;

   /* if (effectively) no outstanding I/O then conclude the DCL task */
   if (tkptr->QueuedSysCommand <= tkptr->QueuedSysCommandAllowed &&
       !tkptr->QueuedSysOutput &&
       !tkptr->QueuedCgiPlusIn &&
       !tkptr->QueuedHttpInput)
   {
      DclConcludeTask (tkptr, false);
      return;
   }

   /* if I/O cancelled then just return */
   if (tkptr->CgiPlusInIOsb.Status == SS$_ABORT ||
       tkptr->CgiPlusInIOsb.Status == SS$_CANCEL)
      return;

   /* don't know why, first AST is delivered %x00000000, make %X00000001 */
   if (!tkptr->CgiPlusInIOsb.Status) tkptr->CgiPlusInIOsb.Status = 1;

   /* abort if an error writing CGIPLUSIN stream to subprocess */
   if (VMSnok (tkptr->CgiPlusInIOsb.Status))
   {
      tkptr->rqptr->ErrorTextPtr = MsgFor(tkptr->rqptr,MSG_SCRIPT_IPC);
      ErrorVmsStatus (tkptr->rqptr, tkptr->CgiPlusInIOsb.Status, FI_LI);
      DclConcludeTask (tkptr, true);
      return;
   }

   /* at least one CGIPLUSIN variable was read indicating script activation */
   tkptr->CgiScriptActivated = true;
}

/*****************************************************************************/
/*
A queued write to the subprocess "HTTP$INPUT" mailbox has completed.
Provide more (possibly first) of the request body, or EOF.
*/ 

DclHttpInputAst (struct DclTaskStruct *tkptr)

{
   int  status,
        DclAstStatus,
        Length;
   char  *ContentPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DclHttpInputAst() %d %d %08.08X %d %d %d %d %d %%X%08.08X\n",
         tkptr, tkptr->rqptr, tkptr->SubprocessPid,
         tkptr->QueuedSysCommand, tkptr->QueuedSysOutput,
         tkptr->QueuedCgiPlusIn, tkptr->QueuedHttpInput,
         tkptr->HttpInputIOsb.Count, tkptr->HttpInputIOsb.Status);
   }

   if (tkptr->QueuedHttpInput) tkptr->QueuedHttpInput--;

   /* if (effectively) no outstanding I/O then conclude the DCL task */
   if (tkptr->QueuedSysCommand <= tkptr->QueuedSysCommandAllowed &&
       !tkptr->QueuedSysOutput &&
       !tkptr->QueuedCgiPlusIn &&
       !tkptr->QueuedHttpInput)
   {
      DclConcludeTask (tkptr, false);
      return;
   }

   /* if I/O cancelled then just return */
   if (tkptr->HttpInputIOsb.Status == SS$_ABORT ||
       tkptr->HttpInputIOsb.Status == SS$_CANCEL)
      return;

   /* abort if an error writing HTTP stream to subprocess */
   if (VMSnok (tkptr->HttpInputIOsb.Status))
   {
      tkptr->rqptr->ErrorTextPtr = MsgFor(tkptr->rqptr,MSG_SCRIPT_IPC);
      ErrorVmsStatus (tkptr->rqptr, tkptr->HttpInputIOsb.Status, FI_LI);
      DclConcludeTask (tkptr, true);
      return;
   }

   if (Debug)
      fprintf (stdout, "%d %d\n", tkptr->ContentLength, tkptr->ContentPtr);

   if (tkptr->ContentLength)
   {
      if (tkptr->ContentLength > NetReadBufferSize)
         Length = NetReadBufferSize;
      else
         Length = tkptr->ContentLength;
      ContentPtr = tkptr->ContentPtr;
      tkptr->ContentPtr += Length;
      tkptr->ContentLength -= Length;

      status = sys$qio (0, tkptr->HttpInputChannel,
                        IO$_WRITELBLK, &tkptr->HttpInputIOsb,
                        &DclHttpInputAst, tkptr,
                        ContentPtr, Length, 0, 0, 0, 0);
   }
   else
   {
      status = sys$qio (0, tkptr->HttpInputChannel,
                        IO$_WRITEOF, &tkptr->HttpInputIOsb,
                        &DclHttpInputAst, tkptr,
                        0, 0, 0, 0, 0, 0);
   }

   if (VMSnok (status))
   {
      /* report this error via the AST */
      tkptr->HttpInputIOsb.Status = status;
      SysDclAst (&DclHttpInputAst, tkptr);
   }
   tkptr->QueuedHttpInput++;
}

/*****************************************************************************/
/*
Check if the script procedure or executable file exists.  Set the pointer to
the "invocation" string and the file type (e.g. ".PL;").  Refer to any list of
user-defined script file types if necessary.
*/ 

int DclFindScript
(
struct RequestStruct *rqptr,
char *FileName,
char **InvokePtr,
char **FileTypePtr
)
{
   register int  idx;
   register char  *cptr;

   int  status;
   char  ExpandedFileName [256];
   char  *TypePtr;
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclFindScript() |%s|\n", FileName);

   TypePtr = "";

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_fna = FileName;
   SearchFab.fab$b_fns = strlen(FileName);
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   /* try with any script specified file type first */
   SearchFab.fab$b_dns = 0;
   if (VMSok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if (VMSok (status = sys$search (&SearchFab, 0, 0)))
      {
         /* script specification supplied the file type and found it */
         if (!memcmp (SearchNam.nam$l_type, ".COM;", 5))
            cptr = "@";
         else
         if (!memcmp (SearchNam.nam$l_type, ".EXE;", 5))
            cptr = "$";
         else
         {
            /* look through the list of user-definable script file types */
            for (idx = 0; idx < Config.ScriptRunTimeCount; idx++)
            {
               if (!memcmp (SearchNam.nam$l_type,
                            Config.ScriptRunTimeArray[idx],
                            Config.ScriptRunTimeLengthArray[idx]))
                  break;
            }
            if (idx < Config.ScriptRunTimeCount)
            {
               /* found the file type (file extension) */
               for (cptr = Config.ScriptRunTimeArray[idx];
                    *cptr && *cptr != ' ';
                    cptr++);
               if (*cptr) cptr++;
               /* 'cptr' now points at the script invocation string */
            }
            else
            {
               static $DESCRIPTOR (ErrorScriptRunTimeFaoDsc,
"Execution of &nbsp;<TT>!AZ</TT>&nbsp; script types not configured.\0");
               char  String [256];
               $DESCRIPTOR (StringDsc, String);

               *SearchNam.nam$l_ver = '\0';
               sys$fao (&ErrorScriptRunTimeFaoDsc, 0, &StringDsc, 
                        SearchNam.nam$l_type);
               rqptr->ResponseStatusCode = 500;
               ErrorGeneral (rqptr, String, FI_LI);
               status = STS$K_ERROR;
            }
         }
      }  
   }

   /* if script file not found and the type (file extension) not specified */
   if (status == RMS$_FNF && SearchNam.nam$b_type == 1)
   {
      /* first look for a DCL procedure */
      SearchFab.fab$l_dna = ".COM;";
      SearchFab.fab$b_dns = 5;
      if (VMSok (status = sys$parse (&SearchFab, 0, 0)))
         if (VMSok (status = sys$search (&SearchFab, 0, 0)))
            cptr = "@";

      if (status == RMS$_FNF)
      {
         /* next, look for an executable */
         SearchFab.fab$l_dna = ".EXE;";
         SearchFab.fab$b_dns = 5;
         if (VMSok (status = sys$parse (&SearchFab, 0, 0)))
            if (VMSok (status = sys$search (&SearchFab, 0, 0)))
               cptr = "$";
      }

      if (status == RMS$_FNF)
      {
         /* then look through any list of user-defined script file types */
         for (idx = 0; idx < Config.ScriptRunTimeCount; idx++)
         {
            if (Debug)
               fprintf (stdout, "|%s|\n", Config.ScriptRunTimeArray[idx]);
            SearchFab.fab$l_dna = Config.ScriptRunTimeArray[idx];
            SearchFab.fab$b_dns = Config.ScriptRunTimeLengthArray[idx];
            if (VMSok (status = sys$parse (&SearchFab, 0, 0)))
               if (VMSok (status = sys$search (&SearchFab, 0, 0)))
                  break;
         }
         if (idx < Config.ScriptRunTimeCount)
         {
            /* found the file type (file extension) */
            for (cptr = TypePtr = Config.ScriptRunTimeArray[idx];
                 *cptr && *cptr != ';';
                 cptr++);
            if (*cptr) cptr++;
            /*
               'cptr' now points at the script invocation string.
               'TempPtr' points at the file extension (terminated by a ';')
            */
         }
      }
   }

   /* release parse and search internal data structures */
   SearchFab.fab$l_fna = "a:[b]c.d;";
   SearchFab.fab$b_fns = 9;
   SearchFab.fab$b_dns = 0;
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);

   /* might be NULL (script not found), might point to "$", "@", etc. */
   *InvokePtr = cptr;

   /* if file extension was not supplied this will point to one */
   *FileTypePtr = TypePtr;

   return (status);
}

/*****************************************************************************/
/*
*/ 

DclCgiScriptSysCommand
(
struct DclTaskStruct *tkptr,
char *DclCommand,
char *CgiFileName,
char *RunTimePtr,
char *ScriptFileTypePtr
)
{
   static char  DclHttpdVerify [] = "!'F$VERIFY(F$TRNLNM(\"HTTPD$VERIFY\"))";
   static char  DeleteIsDelete [] = "DELETE=\"DELETE\"";
   static char  DeleteAllLocalSymbol[] = "DELETE/SYMBOL/LOCAL/ALL";
   static char  DeleteAllGlobalSymbol[] = "DELETE/SYMBOL/GLOBAL/ALL";
   static char  NoVerify [] = "!'F$VERIFY(0)";
   static char  SetNoOn[] = "SET NOON";
   static char  StopId [] = "STOP/ID=0";
   static char  WriteIsWrite [] = "WRITE=\"WRITE\"";

   static $DESCRIPTOR (WriteDclEofFaoDsc, "WRITE SYS$OUTPUT \"!AZ\"");

   static $DESCRIPTOR (DefineHttpInputFaoDsc,
                       "DEFINE/NOLOG/SUPER HTTP$INPUT !AZ");
   static $DESCRIPTOR (DefineSysInputFaoDsc,
                       "DEFINE/NOLOG/SUPER SYS$INPUT !AZ");

   register struct RequestStruct *rqptr;
   register char  c;
   register char  *cptr, *sptr, *zptr;

   int  status,
        Count;
   unsigned short  Length;
   char  FormFieldName [256],
         Scratch [1024],
         DclLine [256];
   $DESCRIPTOR (DclLineDsc, DclLine);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DclCgiScriptSysCommand() |%s|%s|%s|%s|\n",
               DclCommand, CgiFileName, RunTimePtr, ScriptFileTypePtr);

   /* get the pointer to the request structure */
   rqptr = tkptr->rqptr;

   DclQioSysCommand (tkptr, NoVerify, sizeof(NoVerify)-1);
   DclQioSysCommand (tkptr, SetNoOn, sizeof(SetNoOn)-1);
   DclQioSysCommand (tkptr, DeleteIsDelete, sizeof(DeleteIsDelete)-1);
   DclQioSysCommand (tkptr, DeleteAllLocalSymbol,
                     sizeof(DeleteAllLocalSymbol )-1);
   DclQioSysCommand (tkptr, DeleteAllGlobalSymbol,
                     sizeof(DeleteAllGlobalSymbol )-1);
   DclQioSysCommand (tkptr, DclHttpdVerify, sizeof(DclHttpdVerify)-1);

   if (tkptr->CgiScript)
   {
      sys$fao (&DefineSysInputFaoDsc, &Length, &DclLineDsc,
               tkptr->HttpInputDevName);
      DclLine[Length] = '\0';
      DclQioSysCommand (tkptr, DclLine, Length);

      /* for backward compatibility */
      sys$fao (&DefineHttpInputFaoDsc, &Length, &DclLineDsc,
               tkptr->HttpInputDevName);
      DclLine[Length] = '\0';
      DclQioSysCommand (tkptr, DclLine, Length);
   }

   /*****************/
   /* CGI variables */
   /*****************/

   /*
      CGI script requests can only call this once.
      DCL commands from SSI documents can call this multiple times.
      Once the CGI information is generated is is essentially static
      for the life of the request ... so only generate it the once!
   */
   if (rqptr->CgiBufferPtr == NULL)
   {
      if (VMSnok (status = CgiGenerateVariables (rqptr, CGI_VARIABLE_DCL)))
      {
         DclSysCommandError (tkptr, status);
         return;
      }
   }

   cptr = rqptr->CgiBufferPtr;
   for (;;)
   {
      if (!(Length = *(short*)cptr)) break;
      DclQioSysCommand (tkptr, cptr+sizeof(short), Length-1);
      cptr += Length + sizeof(short);
   }

   /*******************************/
   /* DCL command/procedure/image */
   /*******************************/

   if (CgiFileName[0])
   {
      sptr = DclLine;
      cptr = RunTimePtr;
      if (*(unsigned short*)cptr == '@\0')
      {
         /* DCL procedure */
         while (*cptr) *sptr++ = *cptr++;
      }
      else
      if (*(unsigned short*)cptr == '$\0')
      {
         /* execute an image */
         cptr = "RUN ";
         while (*cptr) *sptr++ = *cptr++;
      }
      else
      if (*cptr == '@' || *cptr == '$')
      {
         /* foreign-verb DCL procedure or executable, create the verb */
         cptr++;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         DclQioSysCommand (tkptr, DclLine, sptr - DclLine);

         /* now place it as the verb before the script file */
         sptr = DclLine;
         cptr = RunTimePtr + 1;
         while (*cptr && *cptr != '=') *sptr++ = *cptr++;
         *sptr++ = ' ';
      }
      else
      {
         /* verb must already exist on site, place before the script file */
         while (*cptr) *sptr++ = *cptr++;
         *sptr++ = ' ';
      }
      cptr = CgiFileName;
      while (*cptr) *sptr++ = *cptr++;
      /* this will add the file type (extension) if one was not supplied */
      cptr = ScriptFileTypePtr;
      while (*cptr && *cptr != ';') *sptr++ = *cptr++;
      *sptr = '\0';
      Length = sptr - (cptr = DclLine);
   }
   else
   {
      for (cptr = sptr = DclCommand; *sptr; sptr++);
      Length = sptr - cptr;
   }
   DclQioSysCommand (tkptr, cptr, Length);

   /*********************/
   /* after script runs */
   /*********************/

   if (DclUseZombies)
   {
      DclQioSysCommand (tkptr, NoVerify, sizeof(NoVerify)-1);
      DclQioSysCommand (tkptr, WriteIsWrite, sizeof(WriteIsWrite)-1);

      sys$fao (&WriteDclEofFaoDsc, &Length, &DclLineDsc, tkptr->DclEof);
      DclLine[Length] = '\0';
      DclQioSysCommand (tkptr, DclLine, Length);

      /* do not send an end-of-file! */
      return (SS$_NORMAL);
   }
   else
   {
      /* ensure subprocess terminates! */
      DclQioSysCommand (tkptr, StopId, sizeof(StopId));

      /* send end-of-file */
      DclQioSysCommand (tkptr, NULL, 0);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Queue an AST to act on the error status.
*/ 

DclSysCommandError
(
struct DclTaskStruct *tkptr,
int StatusValue
)
{
   int  status,
        DclAstStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSysCommandError()\n");

   if (StatusValue)
      tkptr->SysCommandIOsb.Status = StatusValue;
   else
      tkptr->SysCommandIOsb.Status = STS$K_ERROR;
   SysDclAst (&DclSysCommandAst, tkptr);
}

/*****************************************************************************/
/*
Send DCL commands to the CGIplus subprocess' SYS$COMMAND. This sets up the DCL
environment (defines logical names, assigns symbols) executes the procedure or
image.
*/ 

DclCgiPlusScriptSysCommand
(
struct DclTaskStruct *tkptr,
char *CgiFileName,
char *RunTimePtr,
char *ScriptFileTypePtr
)
{
   static char  DclHttpdVerify [] = "!'F$VERIFY(F$TRNLNM(\"HTTPD$VERIFY\"))";
   static char  DeleteIsDelete [] = "DELETE=\"DELETE\"";
   static char  DeleteAllLocalSymbol[] = "DELETE/SYMBOL/LOCAL/ALL";
   static char  DeleteAllGlobalSymbol[] = "DELETE/SYMBOL/GLOBAL/ALL";
   static char  NoVerify [] = "!'F$VERIFY(0)";
   static char  SetNoOn [] = "SET NOON";
   static char  StopId [] = "STOP/ID=0";

   static $DESCRIPTOR (DefineCgiPlusInFaoDsc,
                       "DEFINE/NOLOG/SUPER CGIPLUSIN !AZ");
   static $DESCRIPTOR (DefineCgiPlusEofFaoDsc,
                       "DEFINE/NOLOG/SUPER CGIPLUSEOF \"!AZ\"");
   static $DESCRIPTOR (DefineHttpInputFaoDsc,
                       "DEFINE/NOLOG/SUPER HTTP$INPUT !AZ");
   static $DESCRIPTOR (DefineSysInputFaoDsc,
                       "DEFINE/NOLOG/SUPER SYS$INPUT !AZ");

   register char  *cptr, *sptr;
   register struct RequestStruct *rqptr;

   int  status;
   unsigned short  Length;
   char  DclLine [256];
   $DESCRIPTOR (DclLineDsc, DclLine);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclCgiPlusScriptSysCommand()\n");

   /* get the pointer to the request structure */
   rqptr = tkptr->rqptr;

   /* multiple requests to the same subprocess use the same EOF string! */
   if (tkptr->DclEofLength)
   {
      /* reuse existing EOF string */
      rqptr->CgiEofPtr = tkptr->DclEof;
      rqptr->CgiEofLength = tkptr->DclEofLength;
   }
   else
   {
      /* only generate if a new one is necessary */
      CgiEof (tkptr->DclEof, &tkptr->DclEofLength);
      rqptr->CgiEofPtr = tkptr->DclEof;
      rqptr->CgiEofLength = tkptr->DclEofLength;
   }

   DclQioSysCommand (tkptr, NoVerify, sizeof(NoVerify)-1);
   DclQioSysCommand (tkptr, SetNoOn, sizeof(SetNoOn)-1);
   DclQioSysCommand (tkptr, DeleteIsDelete, sizeof(DeleteIsDelete)-1);
   DclQioSysCommand (tkptr, DeleteAllLocalSymbol,
                     sizeof(DeleteAllLocalSymbol )-1);
   DclQioSysCommand (tkptr, DeleteAllGlobalSymbol,
                     sizeof(DeleteAllGlobalSymbol )-1);
   DclQioSysCommand (tkptr, DclHttpdVerify, sizeof(DclHttpdVerify)-1);

   sys$fao (&DefineSysInputFaoDsc, &Length, &DclLineDsc,
            tkptr->HttpInputDevName);
   DclLine[Length] = '\0';
   DclQioSysCommand (tkptr, DclLine, Length);

   /* for backward compatibility */
   sys$fao (&DefineHttpInputFaoDsc, &Length, &DclLineDsc,
            tkptr->HttpInputDevName);
   DclLine[Length] = '\0';
   DclQioSysCommand (tkptr, DclLine, Length);

   sys$fao (&DefineCgiPlusInFaoDsc, &Length, &DclLineDsc,
            tkptr->CgiPlusInDevName);
   DclLine[Length] = '\0';
   status = DclQioSysCommand (tkptr, DclLine, Length);

   sys$fao (&DefineCgiPlusEofFaoDsc, &Length, &DclLineDsc, tkptr->DclEof);
   DclLine[Length] = '\0';
   status = DclQioSysCommand (tkptr, DclLine, Length);

   /***********************/
   /* DCL procedure/image */
   /***********************/

   sptr = DclLine;
   cptr = RunTimePtr;
   if (*(unsigned short*)cptr == '@\0')
   {
      /* DCL procedure */
      while (*cptr) *sptr++ = *cptr++;
   }
   else
   if (*(unsigned short*)cptr == '$\0')
   {
      /* execute an image */
      cptr = "RUN ";
      while (*cptr) *sptr++ = *cptr++;
   }
   else
   if (*cptr == '@' || *cptr == '$')
   {
      /* foreign-verb DCL procedure or executable, create the verb */
      cptr++;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DclQioSysCommand (tkptr, DclLine, sptr - DclLine);

      /* now place it as the verb before the script file */
      sptr = DclLine;
      cptr = RunTimePtr + 1;
      while (*cptr && *cptr != '=') *sptr++ = *cptr++;
      *sptr++ = ' ';
      *sptr = '\0';
   }
   else
   {
      /* verb must already exist on site, place before the script file */
      while (*cptr) *sptr++ = *cptr++;
      *sptr++ = ' ';
      *sptr = '\0';
   }
   cptr = CgiFileName;
   while (*cptr) *sptr++ = *cptr++;
   /* this will add the file type (extension) if one was not supplied */
   cptr = ScriptFileTypePtr;
   while (*cptr && *cptr != ';') *sptr++ = *cptr++;
   *sptr = '\0';
   Length = sptr - (cptr = DclLine);
   DclQioSysCommand (tkptr, cptr, Length);

   /* ensure subprocess terminates! */
   DclQioSysCommand (tkptr, StopId, sizeof(StopId));

   /* send end-of-file */
   DclQioSysCommand (tkptr, NULL, 0);
}

/*****************************************************************************/
/*
Send CGI variables to the subprocess' CGIPLUSIN input stream.
*/ 

DclCgiPlusScriptCgiPlusIn (struct DclTaskStruct *tkptr)

{
   register char  *cptr;
   register struct RequestStruct *rqptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclCgiPlusScriptCgiPlusIn()\n");

   /* get the pointer to the request structure */
   rqptr = tkptr->rqptr;

   /* just a line for "start-of-request" that can always be discarded */
   DclQioCgiPlusIn (tkptr, "!", 1);

   if (VMSnok (status = CgiGenerateVariables (rqptr, CGI_VARIABLE_STREAM)))
   {
      DclCgiPlusInError (tkptr, status);
      return;
   }

   cptr = rqptr->CgiBufferPtr;
   for (;;)
   {
      if (!(Length = *(short*)cptr)) break;
      DclQioCgiPlusIn (tkptr, cptr+sizeof(short), Length-1);
      cptr += Length + sizeof(short);
   }

   /* empty line terminates CGI variables */
   DclQioCgiPlusIn (tkptr, "", 0);
}

/*****************************************************************************/
/*
Queue an AST to act on the error status.
*/ 

DclCgiPlusInError
(
struct DclTaskStruct *tkptr,
int StatusValue
)
{
   int  status,
        DclAstStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclCgiPlusInError() |%s|\n");

   if (StatusValue)
      tkptr->CgiPlusInIOsb.Status = StatusValue;
   else
      tkptr->CgiPlusInIOsb.Status = STS$K_ERROR;
   SysDclAst (&DclCgiPlusInAst, tkptr);
}

/*****************************************************************************/
/*
Every minute scan the list of DCL subprocesses looking for those whose
lifetimes have expired. Run those subprocesses down! As this is a one minute
tick set the lifetime counters to configuration plus one to ensure at least
that number of minutes before expiry.
*/

DclSupervisor (boolean TimerExpired)

{
   static boolean  TimerSet = false;
   static unsigned long  OneMinuteDelta [2] = { -600000000, -1 };

   register struct ListEntryStruct  *leptr;
   register struct DclTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSupervisor() %d\n", TimerExpired);

   if (!TimerExpired)
   {
      /* new request, if the supervisor is already running just return */
      if (TimerSet) return;

      /* kick the supervisor timer into life, and then return */
      if (VMSnok (status =
          sys$setimr (0, &OneMinuteDelta, &DclSupervisor, true, 0)))
         ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);
      TimerSet = true;
      return;
   }

   /* must have got here via the supervisor's timer expiring */
   TimerSet = false;

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;
      if (Debug) DclTaskItemDebug (leptr, tkptr);
 
      /* let the output timer take care of those in use, ignore marked */
      if (tkptr->rqptr != NULL || tkptr->IsMarkedForDelete) continue;

      if (tkptr->LifeTimeCount > 0)
      {
         TimerSet = true;
         if (--tkptr->LifeTimeCount) continue;
      }

      /* process timer has expired, exterminate ... exterminate */
      tkptr->IsMarkedForDelete = true;
      DclConcludeTask (tkptr, true);
      if (tkptr->CgiPlusScript)
         DclCgiPlusLifeTimePurgeCount++;
      else
         DclZombieLifeTimePurgeCount++;
   }

   if (TimerSet)
   {
      /* at least one item in the connect list is still counting down */
      if (VMSnok (status =
          sys$setimr (0, &OneMinuteDelta, &DclSupervisor, true, 0)))
         ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);
   }
}

/*****************************************************************************/
/*
************
*** NOTE ***  This function takes a pointer to a request!!!
************  Due to it being a general report processing function.

Return a report on the DCL task structure.  This function blocks while
executing.
*/ 

DclScriptingReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... CGI/DCL Scripting</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>CGI/DCL Scripting</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Statistics</TH></TR>\n\
<TH ALIGN=left>CGI <I>script</I></TH><TD>!UL</TD></TR>\n\
<TR ALIGN=left><TH>CGIplus <I>script</I></TH>\
<TD>!UL &nbsp;<I>(!UL multiple)</I></TD></TR>\n\
<TR ALIGN=left><TH>DCL command</TH><TD>!UL</TD></TR>\n\
<TR ALIGN=left><TH>Spawned</TH><TD>!UL</TD></TR>\n\
</TABLE>\n\
\
</TD><TD>&nbsp;&nbsp;&nbsp;&nbsp;</TD><TD VALIGN=top>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3>Subprocesses</TH></TR>\n\
<TR><TH COLSPAN=2 ALIGN=right>Current</TH><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH ROWSPAN=2>Soft-Limit</TH>\
<TH ALIGN=right>Value</TH><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TD ALIGN=right>Purged At</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH ROWSPAN=2>Hard-Limit</TH>\
<TH ALIGN=right>Value</TH><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TD ALIGN=right>Hit Against</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH ROWSPAN=2>Purge</TH>\
<TD ALIGN=right>At Soft-Limit</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TD ALIGN=right>Explicit</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH ROWSPAN=2>Zombie</TH>\
<TH ALIGN=right>Life-Time</TH><TD ALIGN=right>!AZ</TD></TR>\n\
<TR><TD ALIGN=right>Purged at Life-Time</TD>\
<TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH ROWSPAN=3>CGIplus</TH>\
<TH ALIGN=right>Life-Time</TH><TD ALIGN=right>!AZ</TD></TR>\n\
<TR><TD ALIGN=right>Purged at Life-Time</TD><TD ALIGN=right>!UL</TD></TR>\n\
</TABLE>\n\
\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH></TH><TH>Script/DCL</TH><TH>PID</TH><TH>Lifetime</TH>\
<TH>Zombie</TH><TH>CGIplus</TH><TH>Total</TH><TH>Last</TH></TR>\n\
<TR><TH></TH><TH COLSPAN=1>Client</TH><TH COLSPAN=6>Request</TH></TR>\n\
<TR><TH COLSPAN=7></TH></TR>\n");

   static $DESCRIPTOR (TaskFaoDsc,
"<TR><TD ALIGN=right>!UL</TD>\
<TD ALIGN=left>!AZ</TD><TD>!AZ</TD><TD ALIGN=right>!AZ!UL!AZ</TD>\
<TD ALIGN=right>!AZ</TD><TD ALIGN=right>!AZ</TD><TD ALIGN=right>!UL</TD>\
<TD><FONT SIZE=-2>!AZ</FONT></TD></TR>\n\
<TR><TD></TD><TD ALIGN=left COLSPAN=1>!AZ!AZ!AZ</TD>\
<TD ALIGN=left COLSPAN=6><TT>!AZ</TT></TD></TR>\n");

   static $DESCRIPTOR (ButtonsFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=submit VALUE=\" Purge \">\n\
</FORM>\n\
</TD><TD>&nbsp;</TD><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=submit VALUE=\" Force Delete \">\n\
</FORM>\n\
</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (PidFaoDsc, "<TT>!8XL</TT>\0");
   static $DESCRIPTOR (CgiPlusCountFaoDsc, "!UL\0");
   static $DESCRIPTOR (LifeTimeFaoDsc, "!UL\0");
   static $DESCRIPTOR (ZombieCountFaoDsc, "!UL\0");

   register unsigned long  *vecptr;
   register char  *sptr;
   register struct DclTaskStruct  *tkptr;
   register struct ListEntryStruct  *leptr;

   int  status,
        Count;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [4096],
         CgiPlusCount [32],
         CgiPlusLifeTime [32],
         Path [512],
         Pid [32],
         ZombieCount [32],
         ZombieLifeTime [32];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (CgiPlusCountDsc, CgiPlusCount);
   $DESCRIPTOR (CgiPlusLifeTimeDsc, CgiPlusLifeTime);
   $DESCRIPTOR (PidDsc, Pid);
   $DESCRIPTOR (ZombieCountDsc, ZombieCount);
   $DESCRIPTOR (ZombieLifeTimeDsc, ZombieLifeTime);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclScriptingReport()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   *vecptr++ = Accounting.DoScriptCount;
   *vecptr++ = Accounting.DoCgiPlusScriptCount;
   *vecptr++ = Accounting.DclCgiPlusReusedCount;
   *vecptr++ = Accounting.DoDclCommandCount;
   *vecptr++ = Accounting.DclSpawnCount;

   *vecptr++ = DclSubprocessCount;
   *vecptr++ = DclSubprocessSoftLimit;
   *vecptr++ = DclSoftLimitPurgeCount;
   *vecptr++ = DclSubprocessHardLimit;
   *vecptr++ = DclHitHardLimitCount;
   *vecptr++ = DclPurgeCount;
   *vecptr++ = DclPurgeAllSubprocessesCount;
   if (DclUseZombies && Config.DclZombieLifeTime)
   {
      sys$fao (&LifeTimeFaoDsc, 0, &ZombieLifeTimeDsc,
               Config.DclZombieLifeTime);
      *vecptr++ = ZombieLifeTime;
   }
   else
      *vecptr++ = "<I>disabled</I>";
   *vecptr++ = DclZombieLifeTimePurgeCount;
   if (Config.DclCgiPlusLifeTime)
   {
      sys$fao (&LifeTimeFaoDsc, 0, &CgiPlusLifeTimeDsc,
               Config.DclCgiPlusLifeTime);
      *vecptr++ = CgiPlusLifeTime;
   }
   else
      *vecptr++ = "<I>none</I>";
   *vecptr++ = DclCgiPlusLifeTimePurgeCount;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   Count = 0;

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;

      vecptr = FaoVector;

      *vecptr++ = ++Count;

      if (tkptr->IsDclCommand)
         *vecptr++ = tkptr->DclCommand;
      else
         *vecptr++ = tkptr->ScriptName;

      if (tkptr->SubprocessPid)
      {
         sys$fao (&PidFaoDsc, 0, &PidDsc, tkptr->SubprocessPid);
         *vecptr++ = Pid;
      }
      else
         *vecptr++ = "";

      if (tkptr->LifeTimeCount)
         *vecptr++ = "<B>";
      else
         *vecptr++ = "";
      /* fudge allows for correct reporting of lifetime */
      if ((tkptr->CgiPlusScript &&
           tkptr->LifeTimeCount == Config.DclCgiPlusLifeTime+1) ||
          (!tkptr->CgiPlusScript &&
           tkptr->LifeTimeCount == Config.DclZombieLifeTime+1))
         *vecptr++ = tkptr->LifeTimeCount-1;
      else
         *vecptr++ = tkptr->LifeTimeCount;
      if (tkptr->LifeTimeCount)
         *vecptr++ = "</B>";
      else
         *vecptr++ = "";

      if (tkptr->ZombieCount)
      {
         sys$fao (&ZombieCountFaoDsc, 0, &ZombieCountDsc,
                  tkptr->ZombieCount);
         *vecptr++ = ZombieCount;
      }
      else
         *vecptr++ = "";

      if (tkptr->CgiPlusScript)
      {
         sys$fao (&CgiPlusCountFaoDsc, 0, &CgiPlusCountDsc,
                  tkptr->CgiPlusUsageCount);
         *vecptr++ = CgiPlusCount;
      }
      else
         *vecptr++ = "";

      *vecptr++ = tkptr->TotalUsageCount;
      *vecptr++ = DayDateTime (&tkptr->LastUsedBinaryTime, 23);

      if (tkptr->rqptr != NULL)
      {
         if (tkptr->rqptr->RemoteUser[0])
         {
            *vecptr++ = tkptr->rqptr->RemoteUser;
            *vecptr++ = "@";
         }
         else
         {
            *vecptr++ = "";
            *vecptr++ = "";
         }
         *vecptr++ = tkptr->rqptr->ClientHostName;

         sptr = Path;
         sptr += CopyToHtml (sptr, sizeof(Path)-(sptr-Path),
                             tkptr->rqptr->ScriptName, -1);
         CopyToHtml (sptr, sizeof(Path)-(sptr-Path),
                     tkptr->rqptr->PathInfoPtr, -1);
         *vecptr++ = Path;
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      status = sys$faol (&TaskFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;
   *vecptr++ = HttpdInternalControl;
   *vecptr++ = "subprocess_purge";
   *vecptr++ = HttpdInternalControl;
   *vecptr++ = "subprocess_force";

   status = sys$faol (&ButtonsFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NextTaskFunction, Buffer, Length);
}

/*****************************************************************************/
/*
************
*** NOTE ***  This function takes a pointer to a request!!!
************  Due to it being a general control processing function.

Control function to delete and mark-for-delete all subprocesses. This is a
quick and effective means for purging all zombie and CGIplus subprocesses
forcing the next activation to reload the (possibly new) image.  The
'WithExtremePrejudice' option deletes subprocesses unconditionally, processing
or not.  It can be used to clear errant scripts without restarting the server. 
Called from Admin.c module.
*/

DclPurgeAllSubprocesses
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean WithExtremePrejudice
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Server !AZ had !UL subprocess!AZ deleted, \
!UL marked for deletion.\n\
</BODY>\n\
</HTML>\n");

   register struct  ListEntryStruct  *leptr;
   register struct DclTaskStruct  *tkptr;
   register unsigned long  *vecptr;

   int  status,
        DeletedCount,
        MarkedCount;
   unsigned long  FaoVector [16];
   unsigned short  Length;
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclPurgeAllSubprocesses()\n");

   DclPurgeAllSubprocessesCount++;

   DeletedCount = MarkedCount = 0;

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;

      if (!tkptr->SubprocessPid) continue;

      tkptr->IsMarkedForDelete = true;

      if (WithExtremePrejudice ||
          (tkptr->QueuedSysCommand <= tkptr->QueuedSysCommandAllowed &&
           !tkptr->QueuedSysOutput &&
           !tkptr->QueuedCgiPlusIn &&
           !tkptr->QueuedHttpInput &&
           tkptr->rqptr == NULL))
      {
         /* forced delete or subprocess not currently active, abort task */
         DclConcludeTask (tkptr, true);
         DeletedCount++;
      }
      else
      {
         /* subprocess is currently active, just leave marked for delete */
         MarkedCount++;
      }
   }

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);
   *vecptr++ = ServerHostPort;
   *vecptr++ = DeletedCount;
   if (DeletedCount == 1)
      *vecptr++ = "";
   else
      *vecptr++ = "es";
   *vecptr++ = MarkedCount;

   sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NextTaskFunction, Buffer, Length);
}

/*****************************************************************************/
/*
Same as for DclPurgeAllSubprocesses() except called from control module from
command line!
*/

char* DclControlPurgeAllSubprocesses (boolean WithExtremePrejudice)

{
   static char  Response [64];
   static $DESCRIPTOR (ResponseFaoDsc, "!UL deleted, !UL marked for delete");
   static $DESCRIPTOR (ResponseDsc, Response);

   register struct  ListEntryStruct  *leptr;
   register struct DclTaskStruct  *tkptr;

   int  status,
        DeletedCount,
        MarkedCount;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclControlPurgeAllSubprocesses()\n");

   DclPurgeAllSubprocessesCount++;

   DeletedCount = MarkedCount = 0;

   for (leptr = DclTaskList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      tkptr = (struct DclTaskStruct*)leptr;

      if (!tkptr->SubprocessPid) continue;

      tkptr->IsMarkedForDelete = true;

      if (WithExtremePrejudice ||
          (tkptr->QueuedSysCommand <= tkptr->QueuedSysCommandAllowed &&
           !tkptr->QueuedSysOutput &&
           !tkptr->QueuedCgiPlusIn &&
           !tkptr->QueuedHttpInput &&
           tkptr->rqptr == NULL))
      {
         /* forced delete or subprocess not currently active, abort task */
         DclConcludeTask (tkptr, true);
         DeletedCount++;
      }
      else
      {
         /* subprocess is currently active, just leave marked for delete */
         MarkedCount++;
      }
   }

   sys$fao (&ResponseFaoDsc, &Length, &ResponseDsc,
            DeletedCount, MarkedCount);
   Response[Length] = '\0';

   return (Response);
}

/*****************************************************************************/
/*
*/

DclTaskItemDebug
(
struct ListEntryStruct *leptr,
struct DclTaskStruct *tkptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclTaskItemDebug()\n");

#ifdef DBUG

   fprintf (stdout, "leptr %d %d %d (%d)\n",
            leptr->PrevPtr, leptr, leptr->NextPtr, tkptr);

   fprintf (stdout, "%08.08X [%d] %d %d %d %d %d %d |%s| %d\n",
            tkptr->SubprocessPid, tkptr->rqptr,
            tkptr->CgiPlusUsageCount, tkptr->TotalUsageCount,
            tkptr->QueuedSysCommand, tkptr->QueuedSysOutput,
            tkptr->QueuedCgiPlusIn, tkptr->QueuedHttpInput,
            tkptr->ScriptName, tkptr->LifeTimeCount);

#endif /* DBUG */

}

/*****************************************************************************/

