/*****************************************************************************/
/*
                                 DECnet.c

This module implements a full multi-threaded, AST-driven, script execution via
DECnet.  Primarily this is to provide a substantially OSU-compatible scripting
enviroment, but does support WASD-style CGI scripting transparently over
DECnet. This code is functional but does not pretend to be highly optimized.

Two modes are supported.


WASD CGI MODE
-------------
These scripts behave in exactly the same manner as subprocess-based, standard
WASD CGI scripts. A stream of DCL commands create the CGI variable environment
and execute the script. The output stream can be CGI-compliant or return a
full HTTP response stream.

This is the default mode. If a mapped script contains a DECnet node component
but without a "TASK=" string the CGI mode is used.  It is also the mode for
any "TASK=" string containing "CGIWASD".


OSU MODE
--------
This mode emulates the OSU DECnet scripting environment.  It is based in large
part on reverse engineering the OSU 'script_execute.c' module, and lots of
trial-and-error!

This is the mode used when the mapped script contains a DECnet component and a
"TASK=" string that does not contain "WASD" as the final four characters.  In
general the string will be "TASK=WWWEXEC", the basic OSU script executive.
Any other task name will also be assumed to be OSU compliant.


MAPPING
-------
DECnet scripting is enabled for individual scripts by including a DECnet
component in the result of a "script" or "exec" mapping rule.

Examples ...

  exec /FRODO/* /FRODO::/cgi-bin/*
  exec /frodo/* /frodo::"task=cgiwasd"/cgi-bin/*

executes any WASD CGI script specified on node FRODO
(both forms result in the same script execution)

  exec /bilbo/osu/* /BILBO::"TASK=WWWEXEC"/*

executes any OSU script specified on node BILBO


CONNECTION REUSE
----------------
As of v3.3 OSU provided the capability to resuse existing connections to the
WWWEXEC.COM task to provide multiple, consecutive requests for a single link.
An efficiency advantage gained through avoiding the overhead of connection
establishment with each script activation.  WASD provides similar functionality
for both OSU and CGI tasked scripts.

A list of connection structures is maintained. Some of these may have channels
assigned to established network tasks, others may not, just depending on
previous activity. The channel non-zero is used when scanning the list for an
appropriate connected task to reuse. Each established connection has a finite
lifetime after which the channel is deassigned effectively disconnecting the
link. Could have done the process control with an intermediate mailbox but
this will work as well most of the time almost as efficiently.

Related configuration parameters:

  [DECnetReuseLifeTime] .... minutes the connection to the task is maintained,
                             non-zero enables connection reuse by the server
  [DECnetConnectListMax] ... number of concurrent reuse connections the server
                             will maintain before reporting an error


VERSION HISTORY
---------------
15-AUG-98  MGD  reuse network task-connections,
                report status 500/501 if script returns no output
12-AUG-98  MGD  report unknown OSU dialog tags as an error,
                ignore <DNETREUSE> dialog tag,
                return translated path in Unix-style syntax
16-DEC-97  MGD  initial development for v5.0
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <libdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "auth.h"
#include "CGI.h"
#include "DECnet.h"
#include "error.h"
#include "httpd.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"

/**********/
/* macros */
/**********/

#define DECNET_CONTENT_MAX 1024

/* space for adding file types (e.g. ".COM") onto script specifications */
#define FIND_SCRIPT_OVERHEAD 48

#define DECNET_TASK_CGI "CGIWASD"
#define DECNET_TASK_OSU "WWWEXEC"

#define SCRIPT_CGI      1
#define SCRIPT_OSU      2

#define CGI_BEGIN       1
#define CGI_REUSE       2
#define CGI_DCL         3
#define CGI_OUTPUT      4

#define OSU_BEGIN       1
#define OSU_REUSE       2
#define OSU_DIALOG      3
#define OSU_DNET_HDR    4
#define OSU_DNET_INPUT  5
#define OSU_DNET_XLATE  6
#define OSU_OUTPUT_RAW  7
#define OSU_OUTPUT_REC  8
#define OSU_DNETTEXT    9

/******************/
/* global storage */
/******************/

char  ErrorDECnetReuseListExhausted [] = "DECnet reuse list exhausted.",
      ErrorOsuImplementation [] = "Possible OSU implementation problem!",
      ErrorOsuNoInvCache [] =
         "OSU &quot;invalidate cache&quot; dialog not implemented.",
      ErrorOsuNoManage [] = "OSU &quot;manage&quot; dialog not implemented.";

$DESCRIPTOR (ErrorOsuUnknownDsc,
      "Unknown OSU dialog ... &quot;<TT>!AZ</TT>&quot;\0");

struct ListHeadStruct  DECnetConnectList;

int  DECnetConnectListCount,
     DECnetPurgeAllConnectCount;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  NetReadBufferSize;
extern int  OutputBufferSize;
extern char  DclCgiVariablePrefix[];
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Returns true if there is a DECnet specific problem, else returns false.
*/ 
 
#ifdef __DECC
#pragma inline(DECnetNotOk)
#endif

boolean DECnetNotOk (int StatusValue)
{
   if (VMSok (StatusValue)) return (false);
   if (StatusValue == SS$_ABORT ||
       StatusValue == SS$_CANCEL ||
       StatusValue == SS$_LINKABORT ||
       StatusValue == SS$_LINKDISCON ||
       StatusValue == SS$_LINKEXIT)
      return (true);
   else
      return (false);
}

/*****************************************************************************/
/*
Initiate a connection to a DECnet node using the connection details specified
in 'ConnectString'.
*/ 
 
int DECnetBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *MappedScript
)
{
   register char  *cptr, *sptr, *zptr;
   register struct DECnetTaskStruct  *tkptr;

   int  status;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetBegin() |%s|\n", MappedScript);

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoDECnetCount;

   /* set up the task structure */
   if (rqptr->DECnetTaskPtr == NULL)
   {
      rqptr->DECnetTaskPtr = tkptr = (struct DECnetTaskStruct*)
         VmGetHeap (rqptr, sizeof(struct DECnetTaskStruct));
   }
   else
   {
      tkptr = rqptr->DECnetTaskPtr;
      memset (tkptr, 0, sizeof(struct DECnetTaskStruct));
   }
   tkptr->rqptr = rqptr;
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = MappedScript;
   zptr = (sptr = tkptr->MappedScript) + sizeof(tkptr->MappedScript);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DECnetEnd (rqptr);
      return;
   }
   *sptr = '\0';

   /*******************************/
   /* generate the connect string */
   /*******************************/

   zptr = (sptr = tkptr->ConnectString) + sizeof(tkptr->ConnectString);
   for (cptr = tkptr->MappedScript;
        *cptr && *cptr != ':' && sptr < zptr;
        *sptr++ = *cptr++);
   if (cptr[0] == ':' && cptr[1] == ':')
   {
      /* isolate the DECnet component from the script file specification */
      if (sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = *cptr++;
      if (*cptr == '\"')
      {
         /* mapping rule is supplying the DECnet task information */
         if (sptr < zptr) *sptr++ = *cptr++;
         while (*cptr && *cptr != '\"' && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = *cptr++;
         *sptr = '\0';
         tkptr->ScriptPtr = cptr + 1;
         DECnetSetDialog (tkptr);
      }
      else
      {
         /* supply the default WASD CGI DECnet task */
         tkptr->ScriptPtr = cptr;
         cptr = "\"TASK=";
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         cptr = DECNET_TASK_CGI;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = '\"';
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            DECnetEnd (rqptr);
            return;
         }
         *sptr = '\0';
         tkptr->ScriptType = SCRIPT_CGI;
         rqptr->CgiScript = true;
      }

      tkptr->ConnectStringLength = sptr - tkptr->ConnectString;

      if (Debug)
         fprintf (stdout, "|%s|%s|\n", tkptr->ConnectString, tkptr->ScriptPtr);
   }
   else
   {
      /* shouldn't be in this function if no DECnet in specification! */
      ErrorGeneral (rqptr, ErrorSanityCheck, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /**********************/
   /* access DECnet task */
   /**********************/

   if (VMSnok (status = DECnetConnectBegin(tkptr)))
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
}

/*****************************************************************************/
/*
End of request's DECnet task.  If outstanding I/O cancel and wait for that to
complete.  Disassociate (disconnect) the request and the connection structure.
Queue an AST for the next task.
*/ 
 
int DECnetEnd (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetEnd() %d %d\n",
               rqptr, rqptr->DECnetTaskPtr->QueuedDECnetIO);

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->QueuedDECnetIO)
   {
      sys$cancel (tkptr->DECnetChannel);
      return;
   }

   DECnetConnectEnd (tkptr);

   if (!tkptr->ScriptResponded)
   {
      /* hmmm, script has not provided any output! */
      if (tkptr->rqptr->HttpMethod == HTTP_METHOD_GET)
      {
         /* take the blame for general GET method failures */
         tkptr->rqptr->ResponseStatusCode = 500;
         ErrorGeneral (tkptr->rqptr,
            MsgFor(tkptr->rqptr,MSG_GENERAL_INTERNAL), FI_LI);
      }
      else
      {
         /* other methods are probably not implemented by the script */
         tkptr->rqptr->ResponseStatusCode = 501;
         ErrorGeneral (tkptr->rqptr,
            MsgFor(tkptr->rqptr,MSG_REQUEST_METHOD), FI_LI);
      }
   }

   /* declare the next task */
   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Examine the connect string to determine the scripting dialog required (CGI or
OSU). Set a flag in the task structure to indicate the type of dialog.
*/ 
 
int DECnetSetDialog (struct DECnetTaskStruct *tkptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetSetDialog() |%s|\n", tkptr->ConnectString);

   cptr = tkptr->ConnectString;
   while (*cptr && *cptr != '\"')
   {
      if (!(strsame (cptr, "\"TASK=", 6) || strsame (cptr, "\"0=", 3)))
      {
         cptr++;
         continue;
      }
      if (cptr[1] == '0')
         cptr += 3;
      else
         cptr += 6;

      if (strsame (cptr, DECNET_TASK_CGI, sizeof(DECNET_TASK_CGI)-1) &&
          cptr[sizeof(DECNET_TASK_CGI)] == '\"')
      {
         tkptr->ScriptType = SCRIPT_CGI;
         tkptr->rqptr->CgiScript = true;
         if (Debug) fprintf (stdout, "dialog: %d\n", tkptr->ScriptType);
         return;
      }
   }

   /* any task name not otherwise recognized becomes an OSU-based script */
   tkptr->ScriptType = SCRIPT_OSU;
   if (Debug) fprintf (stdout, "dialog: %d\n", tkptr->ScriptType);
}

/*****************************************************************************/
/*
Set up a connection between the request and the DECnet task involved. Scan
through the list of already created connection items, looking for an item,
idle but network-connected task-string of the required flavour. If none is
found scan again, this time looking for any unused item (idle, not connected).
If none is found create a new one and place it in the list. If already
network-connected to the task just increment a counter and queue a "fake" AST
to the connection-established function. If not already network-connected
assign a channel and queue an I/O to connect to the desired network task, a
real AST being deliver this time to the connection-established function.
*/ 
 
int DECnetConnectBegin (struct DECnetTaskStruct *tkptr)

{
   static $DESCRIPTOR (NetDeviceDsc, "_NET:");
   static $DESCRIPTOR (NcbDsc, "");

   register struct DECnetConnectStruct  *cnptr;
   register struct ListEntryStruct  *leptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetConnectBegin()\n");

   cnptr = NULL;

   /*****************************************/
   /* look for available AND connected item */
   /*****************************************/

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;
      if (Debug) DECnetConnectItemDebug (leptr, cnptr);

      if (!cnptr->DECnetChannel ||
          !cnptr->ReuseConnection ||
          cnptr->rqptr ||
          !strsame (tkptr->ConnectString, cnptr->ConnectString, -1))
      {
         cnptr = NULL;
         continue;
      }

      break;
   }

   if (Debug) fprintf (stdout, "cnptr: %d\n", cnptr);

   if (cnptr == NULL)
   {
      /****************************************/
      /* look for available disconnected item */
      /****************************************/

      for (leptr = DECnetConnectList.HeadPtr;
           leptr != NULL;
           leptr = leptr->NextPtr)
      {
         cnptr = (struct DECnetConnectStruct*)leptr;
         if (Debug) DECnetConnectItemDebug (leptr, cnptr);

         if (cnptr->DECnetChannel)
         {
            cnptr = NULL;
            continue;
         }

         break;
      }
   }

   if (Debug) fprintf (stdout, "cnptr: %d\n", cnptr);

   if (cnptr == NULL)
   {
      /********************************/
      /* create a new connection item */
      /********************************/

      if (Config.DECnetConnectListMax &&
          DECnetConnectListCount >= Config.DECnetConnectListMax)
      {
         tkptr->rqptr->ResponseStatusCode = 500;
         ErrorGeneral (tkptr->rqptr, ErrorDECnetReuseListExhausted, FI_LI);
         DECnetEnd (tkptr->rqptr);
         return;
      }
      DECnetConnectListCount++;

      cnptr = VmGet (sizeof(struct DECnetConnectStruct));
      if (Debug) fprintf (stdout, "NEW cnptr: %d\n", cnptr);
      ListAddTail (&DECnetConnectList, cnptr);
   }

   /*******************/
   /* initialize item */
   /*******************/

   if (cnptr->DECnetChannel)
   {
      /* (probably) still connected */
      cnptr->ReUsageCount++;
   }
   else
   {
      /* assign channel for new connection */
      status = sys$assign (&NetDeviceDsc, &cnptr->DECnetChannel, 0, 0);
      if (Debug) fprintf (stdout, "lib$assign() %%X%08.08X\n", status);
      if (VMSnok (status))
      {
         cnptr->DECnetChannel = 0;
         tkptr->rqptr->ErrorTextPtr = tkptr->rqptr->ScriptName;
         ErrorVmsStatus (tkptr->rqptr, status, FI_LI);
         return (status);
      }

      cnptr->ReUsageCount = 0;
      cnptr->ReuseConnection = false;
      strcpy (cnptr->ConnectString, tkptr->ConnectString);
   }

   /* associate the request, task and connect instance */
   tkptr->cnptr = cnptr;
   cnptr->rqptr = tkptr->rqptr;
   tkptr->DECnetChannel = cnptr->DECnetChannel;
   cnptr->UsageCount++;
   memcpy (&cnptr->LastUsedBinaryTime, &tkptr->rqptr->BinaryTime, 8);


   if (Debug) DECnetConnectListDebug ();

   /* for the first call to DECnetRead() ensure status check is OK */
   tkptr->rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   if (cnptr->ReuseConnection)
   {
      /******************************/
      /* (probably) still connected */
      /******************************/

      /* restart the connected task lifetime */
      tkptr->cnptr->LifeTimeCount = Config.DECnetReuseLifeTime+1;

      switch (tkptr->ScriptType)
      {
         case SCRIPT_CGI :

            /* reusing, check the connection, ASTs to DECnetCgiDialog() */
            tkptr->CgiDialogState = CGI_REUSE;
            DECnetConnectProbeCgi (tkptr);
            return;

         case SCRIPT_OSU :

            /* reusing, check the connection, ASTs to DECnetOsuDialog() */
            tkptr->OsuDialogState = OSU_REUSE;
            DECnetConnectProbeOsu (tkptr);
            return;

         default :

            ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
      }
   }

   /**************************/
   /* connect to DECnet task */
   /**************************/

   /* apply that to the network Connect Block descriptor */
   NcbDsc.dsc$w_length = tkptr->ConnectStringLength;
   NcbDsc.dsc$a_pointer = tkptr->ConnectString;

   if (Debug)
      fprintf (stdout, "|%*.*s|\n",
               NcbDsc.dsc$w_length, NcbDsc.dsc$w_length, NcbDsc.dsc$a_pointer);

   status = sys$qio (0, tkptr->DECnetChannel, IO$_ACCESS,
                     &tkptr->DECnetConnectIOsb, &DECnetConnectAst, tkptr->rqptr,
                     0, &NcbDsc, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return (status);

   /* an error occured when queueing the access */
   tkptr->DECnetConnectIOsb.Status = status;
   SysDclAst (&DECnetConnectAst, tkptr->rqptr);
   /* return normal, the connect AST will handle the error reporting */
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
The connection request to the specified DECnet task has completed (real or
"faked").  Check the status and report any error.  If no error then initialize
the required script dialog and call the appropriate function to begin it. 
Connection that may be reused begin with the ..._REUSE dialog, which probes the
connection ensuring it's still valid.  Non-reused connections begin with the
appropriate ..._BEGIN dialog.
*/ 
 
int DECnetConnectAst (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetConnectAst() %d %%X%08.08X\n",
               rqptr, rqptr->DECnetTaskPtr->DECnetConnectIOsb);

   tkptr = rqptr->DECnetTaskPtr;

   if (VMSnok (tkptr->DECnetConnectIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      rqptr->ErrorTextPtr = rqptr->ScriptName;
      ErrorVmsStatus (rqptr, tkptr->DECnetConnectIOsb.Status, FI_LI);
      DECnetEnd (rqptr);
      return;
   }

   if (Config.DECnetReuseLifeTime)
   {
      /* reuse enabled, plus one allows at least that period */
      tkptr->cnptr->LifeTimeCount = Config.DECnetReuseLifeTime+1;
      /* start the supervisor ticking for connected task lifetimes */
      DECnetSupervisor (false);
   }

   switch (tkptr->ScriptType)
   {
      case SCRIPT_CGI :

         /* just begin the CGI dialog phase */
         tkptr->CgiDialogState = CGI_BEGIN;
         DECnetCgiDialog (rqptr);
         return;

      case SCRIPT_OSU :

         /* just begin the OSU dialog phase */
         tkptr->OsuDialogState = OSU_BEGIN;
         DECnetOsuDialog (rqptr);
         return;

      default :

         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }
}

/*****************************************************************************/
/*
Write just a DCL comment character to the script network process (will be
ignored by DCL) to check that the connection still exists.  Delivers an AST to
DECnetCGIDialog() which checks the success of the probe.
*/ 
 
int DECnetConnectProbeCgi (struct DECnetTaskStruct *tkptr)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetConnectProbeCgi() %d\n", tkptr);

   tkptr->QueuedDECnetIO++;
   status = sys$qio (0, tkptr->DECnetChannel, IO$_WRITEVBLK,
                     &tkptr->DECnetWriteIOsb, &DECnetCgiDialog, tkptr->rqptr,
                     "!", 1, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return;

   /* report error via AST */
   tkptr->DECnetWriteIOsb.Status = status;
   SysDclAst (&DECnetCgiDialog, tkptr->rqptr);
}

/*****************************************************************************/
/*
Write a <DNETREUSE> dialog tag to the script (will be ignored by the task
procedure WWWEXEC.COM) to check that the connection still exists.  Delivers an
AST to DECnetOsuDialog() which checks the success of the probe.
*/ 
 
int DECnetConnectProbeOsu (struct DECnetTaskStruct *tkptr)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetConnectProbeOsu() %d\n", tkptr);

   tkptr->QueuedDECnetIO++;
   status = sys$qio (0, tkptr->DECnetChannel, IO$_WRITEVBLK,
                     &tkptr->DECnetWriteIOsb, &DECnetOsuDialog, tkptr->rqptr,
                     "<DNETREUSE>", 11, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return;

   /* report error via AST */
   tkptr->DECnetWriteIOsb.Status = status;
   SysDclAst (&DECnetOsuDialog, tkptr->rqptr);
}

/*****************************************************************************/
/*
Conclude a request's use of a DECnet connect item.  If the connection can
reused then leave the channel assigned, otherwise deassign it disconnecting
from the network task.
*/ 
 
int DECnetConnectEnd (struct DECnetTaskStruct *tkptr)

{
   register struct DECnetConnectStruct  *cnptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetConnectEnd() %d %d %d\n",
               tkptr, tkptr->rqptr, tkptr->cnptr->ReuseConnection);

   cnptr = tkptr->cnptr;

   if (!cnptr->ReuseConnection ||
       cnptr->IsMarkedForDisconnect)
   {
      status = sys$dassgn (cnptr->DECnetChannel);
      if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
      cnptr->DECnetChannel = cnptr->LifeTimeCount = 0;
   }

   /* disassociate the request, task and connect instance */
   tkptr->cnptr = cnptr->rqptr = NULL;

   if (Debug) DECnetConnectListDebug ();
}

/*****************************************************************************/
/*
Write a record to the remote DECnet task.  AST function will process any
errors at all.
*/ 
 
int DECnetWrite
(
struct RequestStruct *rqptr,
char *DataPtr,
int DataLength
)
{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      int  len;
      if (DataLength >= 0) len = DataLength; else len = strlen(DataPtr);
      fprintf (stdout, "DECnetWrite()\n");
      fprintf (stdout, "%d |%*.*s|\n", DataLength, len, len, DataPtr);
   }

   tkptr = rqptr->DECnetTaskPtr;

   if (DataLength < 0) DataLength = strlen (DataPtr);

   tkptr->QueuedDECnetIO++;
   status = sys$qio (0, tkptr->DECnetChannel, IO$_WRITEVBLK,
                     &tkptr->DECnetWriteIOsb, &DECnetWriteAst, rqptr,
                     DataPtr, DataLength, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return;

   tkptr->DECnetWriteIOsb.Status = status;
   SysDclAst (&DECnetWriteAst, rqptr);
}

/*****************************************************************************/
/*
*/ 
 
int DECnetWriteAst (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetWriteAst() %d %d %%X%08.08X\n",
               rqptr, rqptr->DECnetTaskPtr->QueuedDECnetIO,
               rqptr->DECnetTaskPtr->DECnetWriteIOsb.Status);

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->QueuedDECnetIO) tkptr->QueuedDECnetIO--;

   if (DECnetNotOk (tkptr->DECnetWriteIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      if (!tkptr->QueuedDECnetIO ||
          tkptr->DECnetReadIOsb.Status != SS$_CANCEL)
         DECnetEnd (rqptr);
      return;
   }

   /* reusing the IOsb, expect it to occasionally have been initialized! */
   if (!tkptr->DECnetWriteIOsb.Status)
      tkptr->DECnetWriteIOsb.Status = SS$_NORMAL;
   if (VMSnok (tkptr->DECnetWriteIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      rqptr->ErrorTextPtr = rqptr->ScriptName;
      ErrorVmsStatus (rqptr, tkptr->DECnetWriteIOsb.Status, FI_LI);
      DECnetEnd (rqptr);
      return;
   }
}

/*****************************************************************************/
/*
Write a chunk of the request header or body to the script. When the content
data is exhausted write an empty record as and end-of-data marker. This must
be checked for and detected by the script as there is no end-of-file
associated with this stream!
*/ 
 
int DECnetWriteRequest (struct RequestStruct *rqptr)

{
   register char  *cptr;
   register struct DECnetTaskStruct  *tkptr;

   int  status,
        DataLength;
   char  *DataPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetWriteRequest() %d %d\n",
               rqptr->DECnetTaskPtr->ContentPtr,
               rqptr->DECnetTaskPtr->ContentCount);

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->ContentPtr == NULL)
   {
      /*********************************/
      /* initalize the data to be sent */
      /*********************************/

      switch (tkptr->ScriptType)
      {
         case SCRIPT_CGI :

            /* request body */
            if (rqptr->ContentBufferPtr == NULL)
            {
               tkptr->ContentPtr = "";
               tkptr->ContentCount = 0;
            }
            else
            {
               tkptr->ContentPtr = rqptr->ContentBufferPtr;
               tkptr->ContentCount = rqptr->ContentLength;
            }
            break;

         case SCRIPT_OSU :

            switch (tkptr->OsuDialogState)
            {
               case OSU_DNET_HDR :

                  /* request header (excluding first line) */
                  cptr = rqptr->RequestHeaderPtr;
                  while (NOTEOL(*cptr)) cptr++;
                  if (*cptr == '\r') cptr++;
                  if (*cptr == '\n') cptr++;
                  tkptr->ContentPtr = cptr;
                  tkptr->ContentCount = -1;
                  break;

               case OSU_DNET_INPUT :

                  /* request body */
                  if (rqptr->ContentBufferPtr == NULL)
                  {
                     tkptr->ContentPtr = "";
                     tkptr->ContentCount = 0;
                  }
                  else
                  {
                     tkptr->ContentPtr = rqptr->ContentBufferPtr;
                     tkptr->ContentCount = rqptr->ContentLength;
                  }
                  break;

               default :
                  ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
            }
            break;

         default :
            ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

      }
      tkptr->ContentCountZero = 0;
   }

   /*************/
   /* send data */
   /*************/

   if (tkptr->ScriptType == SCRIPT_OSU &&
       tkptr->OsuDialogState == OSU_DNET_HDR)
   {
      /* OSU gets the header line-by-line (excluding carriage-control) */
      for (cptr = DataPtr = tkptr->ContentPtr; NOTEOL(*cptr); cptr++);
      DataLength = cptr - DataPtr;
      if (*cptr == '\r') cptr++;
      if (*cptr == '\n') cptr++;
      /* if end-of-header blank line, indicate this */
      if (*DataPtr == '\r' || *DataPtr == '\n')
         tkptr->ContentPtr = NULL;
      else
         tkptr->ContentPtr = cptr;
   }
   else
   if (!tkptr->ContentCount)
   {
      /* write end-of-output empty record (blank line) */
      DataPtr = "";
      DataLength = 0;
      /* indicate we're all done */
      tkptr->ContentPtr = NULL;
   }
   else
   {
      /* request body is sent in "chunks" */
      DataPtr = tkptr->ContentPtr;
      if (tkptr->ContentCount > DECNET_CONTENT_MAX)
         DataLength = DECNET_CONTENT_MAX;
      else
         DataLength = tkptr->ContentCount;
      tkptr->ContentCount -= DataLength;
      tkptr->ContentPtr += DataLength;
   }
   if (Debug) fprintf (stdout, "%d %d\n", DataPtr, DataLength);

   status = sys$qio (0, tkptr->DECnetChannel, IO$_WRITEVBLK,
                     &tkptr->DECnetWriteIOsb, &DECnetWriteRequestAst, rqptr,
                     DataPtr, DataLength, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return;

   /* report the sys$qio() problem via the AST routine */
   tkptr->DECnetWriteIOsb.Status = status;
   SysDclAst (&DECnetWriteRequestAst, rqptr);
}

/*****************************************************************************/
/*
A content write has completed.  Check the IO status block for errors.  If no
error then check for more content to be written, if so the write it!
*/ 
 
int DECnetWriteRequestAst (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetWriteRequestAst() %d %d %%X%08.08X %d\n",
               rqptr, rqptr->DECnetTaskPtr->QueuedDECnetIO,
               rqptr->DECnetTaskPtr->DECnetWriteIOsb.Status,
               rqptr->DECnetTaskPtr->DECnetWriteIOsb.Count);

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->QueuedDECnetIO) tkptr->QueuedDECnetIO--;

   if (DECnetNotOk (tkptr->DECnetWriteIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      if (!tkptr->QueuedDECnetIO ||
          tkptr->DECnetReadIOsb.Status != SS$_CANCEL)
         DECnetEnd (rqptr);
      return;
   }

   /* reusing the one IOsb, expect it occasionally to be initialized! */
   if (!tkptr->DECnetWriteIOsb.Status)
      tkptr->DECnetWriteIOsb.Status = SS$_NORMAL;
   if (VMSnok (tkptr->DECnetWriteIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      rqptr->ErrorTextPtr = rqptr->ScriptName;
      ErrorVmsStatus (rqptr, tkptr->DECnetWriteIOsb.Status, FI_LI);
      DECnetEnd (rqptr);
      return;
   }

   if (tkptr->ContentPtr == NULL)
   {
      /* end-of-output empty record has been written ... finished! */
      if (tkptr->ScriptType == SCRIPT_OSU)
      {
         /* back from <DNETHDR> or <DNETINPUT> into dialog */
         tkptr->OsuDialogState = OSU_DIALOG;
         DECnetRead (rqptr);
      }
   }
   else
   {
      /* write more data */
      DECnetWriteRequest (rqptr);
   }
}

/*****************************************************************************/
/*
Queue a read from the remote DECnet task. AST function will process any errors
at all.
*/ 
 
int DECnetRead (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetRead() %d %%X%08.08X\n",
               rqptr, rqptr->NetWriteIOsb.Status);

   tkptr = rqptr->DECnetTaskPtr;

   /* abort task if NETWORK ERROR when writing TO CLIENT */
   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* best way to clean out network buffers, etc., is to break link */
      tkptr->cnptr->ReuseConnection = false;
      DECnetEnd (rqptr);
      return;
   }

   if (rqptr->OutputBufferPtr == NULL)
   {
      /* initialize output buffer */
      NetWriteBuffered (rqptr, 0, NULL, 0);
   }

   tkptr->QueuedDECnetIO++;
   status = sys$qio (0, tkptr->DECnetChannel, IO$_READVBLK | IO$M_MULTIPLE,
                     &tkptr->DECnetReadIOsb, &DECnetReadAst, rqptr,
                     rqptr->OutputBufferPtr, OutputBufferSize,
                     0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSok (status)) return;

   tkptr->DECnetReadIOsb.Status = status;
   SysDclAst (&DECnetReadAst, rqptr);
}

/*****************************************************************************/
/*
The queued read from the remote DECnet task has completed.  Check for any
errors reported in the IO status block.  If no errors call the function
appropriate for processing the dialog being used by this script.
*/ 
 
int DECnetReadAst (struct RequestStruct *rqptr)

{
   register struct DECnetTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DECnetReadAst() %d %d %%X%08.08X %d bytes\n",
               rqptr, rqptr->DECnetTaskPtr->QueuedDECnetIO,
               rqptr->DECnetTaskPtr->DECnetReadIOsb.Status,
               rqptr->DECnetTaskPtr->DECnetReadIOsb.Count);

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->QueuedDECnetIO) tkptr->QueuedDECnetIO--;

   if (DECnetNotOk (tkptr->DECnetReadIOsb.Status))
   {
      tkptr->cnptr->ReuseConnection = false;
      if (!tkptr->QueuedDECnetIO ||
          tkptr->DECnetReadIOsb.Status != SS$_CANCEL)
         DECnetEnd (rqptr);
      return;
   }

   if (VMSnok (tkptr->DECnetReadIOsb.Status))
   {
      rqptr->ErrorTextPtr = rqptr->ScriptName;
      ErrorVmsStatus (rqptr, tkptr->DECnetReadIOsb.Status, FI_LI);
      DECnetEnd (rqptr);
      return;
   }

   rqptr->OutputBufferPtr[tkptr->DECnetReadIOsb.Count] = '\0';

   switch (tkptr->ScriptType)
   {
      case SCRIPT_CGI :
         DECnetCgiDialog (rqptr);
         return;

      case SCRIPT_OSU :
         DECnetOsuDialog (rqptr);
         return;

      default :
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }
}

/*****************************************************************************/
/*
Dialog for a WASD standard CGI DECnet-based script. These scripts behave
identically to subprocess-based standard CGI scripts. The dialog comprises two
phases. The first sends a stream of records to the CGI DECnet task. These
records comprise DCL commands, used to set up the CGI environment and to
execute the script. The second phase uses the generic CGI modules functions to
process CGI script output, writing it to the client.
*/ 
 
DECnetCgiDialog (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (DefineEofFaoDsc, "DEFINE/PROCESS CGIEOF \"!AZ\"");
   static $DESCRIPTOR (DclCommandDsc, "");

   register char  *cptr;
   register struct DECnetTaskStruct  *tkptr;

   int  status,
        value;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetCgiDialog()\n");

   tkptr = rqptr->DECnetTaskPtr;

   if (Debug) fprintf (stdout, "state: %d\n", tkptr->CgiDialogState);

   if (tkptr->CgiDialogState == CGI_REUSE)
   {
      /**********************/
      /* probing connection */
      /**********************/

      if (tkptr->QueuedDECnetIO) tkptr->QueuedDECnetIO--;

      /* reuse needs to be reset with each use of the DECnet task */
      tkptr->cnptr->ReuseConnection = false;

      if (Debug)
         fprintf (stdout, "DECnetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->DECnetTaskPtr->DECnetWriteIOsb.Status);

      if (VMSnok (tkptr->DECnetWriteIOsb.Status))
      {
         /* problem writing to established connection, restart */
         DECnetConnectEnd (tkptr);
         DECnetConnectBegin (tkptr);
         return;
      }

      Accounting.DoDECnetReuseCount++;
      Accounting.DoDECnetCgiReuseCount++;
      tkptr->CgiDialogState = CGI_BEGIN;

      /* drop through to continue with first phase of dialog */
   }

   if (tkptr->CgiDialogState == CGI_BEGIN)
   {
      /***********************/
      /* look for the script */
      /***********************/

      status = DECnetFindCgiScript (rqptr);
      if (Debug) fprintf (stdout, "find script %%X%08.08X\n", status);

      /* retry indicates the search is still underway */
      if (status == SS$_RETRY) return;

      Accounting.DoDECnetCgiCount++;

      if (VMSnok (status))
      {
         /* fudge the status so the write AST reports the problem */
         tkptr->DECnetWriteIOsb.Status = status;
         SysDclAst (&DECnetWriteAst, rqptr);
         return;
      }

      /* must have found the script, set up the DCL/CGI environment */
      tkptr->CgiDialogState = CGI_DCL;
   }

   if (tkptr->CgiDialogState == CGI_DCL)
   {
      /*****************/
      /* define CGIEOF */
      /*****************/

      /* always generate a new EOF string for standard CGI scripts */
      CgiEof (tkptr->DECnetEof, &tkptr->DECnetEofLength);
      rqptr->CgiEofPtr = tkptr->DECnetEof;
      rqptr->CgiEofLength = tkptr->DECnetEofLength;

      DclCommandDsc.dsc$a_pointer = tkptr->DclCommand;
      DclCommandDsc.dsc$w_length = sizeof(tkptr->DclCommand)-1;
      sys$fao (&DefineEofFaoDsc, &Length, &DclCommandDsc, tkptr->DECnetEof);
      tkptr->DclCommand[Length] = '\0';
      DECnetWrite (rqptr, tkptr->DclCommand, Length);

      /*******************/
      /* define CGIREUSE */
      /*******************/

      if (Config.DECnetReuseLifeTime)
      {
         /* reuse enabled, do not drop DECnet link, reuse it */
         tkptr->cnptr->ReuseConnection = true;
         /* indicate this to the task procedure */
         DECnetWrite (rqptr, "DEFINE/PROCESS CGIREUSE 1", 25);
      }

      /*****************/
      /* CGI variables */
      /*****************/

      if (VMSnok (status = CgiGenerateVariables (rqptr, CGI_VARIABLE_DCL)))
      {
         /* fudge the status so the write AST reports the problem */
         tkptr->DECnetWriteIOsb.Status = status;
         SysDclAst (&DECnetWriteAst, rqptr);
         return;
      }

      cptr = rqptr->CgiBufferPtr;
      for (;;)
      {
         if (!(Length = *(short*)cptr)) break;
         DECnetWrite (rqptr, cptr+sizeof(short), Length-1);
         cptr += Length + sizeof(short);
      }

      /*******************************/
      /* DCL command/procedure/image */
      /*******************************/

      if (tkptr->CgiScriptForeignVerb[0])
         DECnetWrite (rqptr, tkptr->CgiScriptForeignVerb, -1);

      DECnetWrite (rqptr, "GOTO DOIT", 9);

      DECnetWrite (rqptr, tkptr->CgiScriptDcl, -1);

      if (rqptr->HttpMethod == HTTP_METHOD_POST ||
          rqptr->HttpMethod == HTTP_METHOD_PUT)
      {
         /* begin to write the request content (body) */
         DECnetWriteRequest (rqptr);
      }

      /*******************************/
      /* begin to read script output */
      /*******************************/

      /* now into dialog state */
      tkptr->CgiDialogState = CGI_OUTPUT;

      DECnetRead (rqptr);
      return;
   }
   else
   /* CGI_OUTPUT */
   {
      /*************************/
      /* process script output */
      /*************************/

      value = CgiOutput (rqptr,
                         rqptr->OutputBufferPtr,
                         tkptr->DECnetReadIOsb.Count);

      if (value == CGI_OUTPUT_TERMINATE)
      {
         /* terminate processing */
         DECnetEnd (rqptr);
         return;
      }

      if (value == CGI_OUTPUT_ABSORB)
      {
         /* absorb output, queue the next read from the script */
         DECnetRead (rqptr);
         return;
      }

      /* note that the script has generated some output */
      tkptr->ScriptResponded += value;

      /* otherwise it's the count of bytes in the output buffer */
      NetWrite (rqptr, &DECnetRead, rqptr->OutputBufferPtr, value);
      return;
   }
}

/*****************************************************************************/
/*
Use the DECnet CGI task to search for the script file. A DCL command is
constructed with the F$SEARCH() lexical and written to the DECnet task which
then executes it, writing the result as an output record which is read by
DECnetRead(), passed to DECnetCgiDialog() and passed back to this same
function for result assessment. If not found and if no explicit file type,
step through each of ".COM", ".EXE" and then any configuration specified file
types and run-time environments, repeating this until found or until all file
types exhausted, at which time a "script not found" report is generated. If
found the DCL required to execute the script is generated in
'tkptr->CgiScriptForeignVerb' and 'tkptr->CgiScriptDcl'.
*/ 

int DECnetFindCgiScript (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (SearchFaoDsc, "WRITE NET$LINK F$SEARCH(\"!AZ\")");
   static $DESCRIPTOR (DclCommandDsc, "");

   register int  idx;
   register char  *cptr, *sptr, *zptr;
   register struct DECnetTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetFindCgiScript()\n");

   tkptr = rqptr->DECnetTaskPtr;

   if (tkptr->FindScriptCount && tkptr->DECnetReadIOsb.Count)
   {
      rqptr->OutputBufferPtr[tkptr->DECnetReadIOsb.Count] = '\0';
      if (Debug)
      {
         fprintf (stdout, "f$search() %d |%s| %d |%s|\n",
                  tkptr->DECnetReadIOsb.Count, rqptr->OutputBufferPtr,
                  tkptr->FindScriptTotalLength, tkptr->FindScript);
      }

      if (strsame (rqptr->OutputBufferPtr,
                   tkptr->FindScript,
                   tkptr->FindScriptTotalLength))
      {
         /*************/
         /* found it! */
         /*************/

         if (Debug) fprintf (stdout, "FOUND!\n");
         tkptr->CgiScriptDcl[0] = tkptr->CgiScriptForeignVerb[0] = '\0';

         cptr = tkptr->RunTimePtr;
         if (*(unsigned short*)cptr == '@\0')
         {
            /* DCL procedure */
            sptr = tkptr->CgiScriptDcl;
            *sptr++ = '@';
         }
         else
         if (*(unsigned short*)cptr == '$\0')
         {
            /* execute an image */
            sptr = tkptr->CgiScriptDcl;
            for (cptr = "RUN "; *cptr; *sptr++ = *cptr++);
         }
         else
         {
            if (*cptr == '@' || *cptr == '$')
            {
               /* foreign-verb DCL procedure or executable, create the verb */
               sptr = tkptr->CgiScriptForeignVerb;
               cptr++;
               while (*cptr) *sptr++ = *cptr++;
               *sptr = '\0';

               /* now place it as the verb before the script file */
               sptr = tkptr->CgiScriptDcl;
               while (*cptr && *cptr != '=') *sptr++ = *cptr++;
               *sptr++ = ' ';
               *sptr = '\0';
            }
            else
            {
               /* verb already exists, place before the script file */
               sptr = tkptr->CgiScriptDcl;
               while (*cptr) *sptr++ = *cptr++;
               *sptr++ = ' ';
               *sptr = '\0';
            }
         }

         cptr = tkptr->FindScript;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';

         if (Debug)
            fprintf (stdout, "|%s|%s|\n",
                     tkptr->CgiScriptForeignVerb, tkptr->CgiScriptDcl);
         return (SS$_NORMAL);
      }

      /* not our dialog, force not found report */
      tkptr->FindScriptCount = 999999;
   }

   /*******************/
   /* not found (yet) */
   /*******************/

   if (tkptr->FindScriptCount == 1) 
   {
      /* explicitly supplied file type was not found */
      rqptr->ResponseStatusCode = 404;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_SCRIPT_NOT_FOUND), FI_LI);
      return (STS$K_ERROR);
   }

   /* if started searching then search for the next one */
   if (tkptr->FindScriptCount > 1) tkptr->FindScriptCount++;

   if (!tkptr->FindScriptCount)
   {
      /****************/
      /* initial call */
      /****************/

      /* get the script file name (minus all DECnet-related stuff) */
      zptr = (sptr = tkptr->FindScript) +
             sizeof(tkptr->FindScript) - FIND_SCRIPT_OVERHEAD;
      for (cptr = tkptr->MappedScript; *cptr && *cptr != ':'; cptr++);
      if (*cptr == ':') cptr++;
      if (*cptr == ':') cptr++;
      if (*cptr == '\"')
      {
         /* task specification string */
         cptr++;
         while (*cptr != '\"') cptr++;
         if (*cptr == '\"') cptr++;
      }
      /* here we go with the file name */
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         return (STS$K_ERROR);
      }
      *sptr = '\0';
      tkptr->FindScriptLength = sptr - tkptr->FindScript;
      if (Debug)
         fprintf (stdout, "%d |%s|\n",
                  tkptr->FindScriptLength, tkptr->FindScript);

      /* look backwards to see if a file type has been supplied */
      cptr = tkptr->FindScript;
      while (*sptr != '.' && *sptr != ']' && sptr > cptr) sptr--;
      if (*sptr != '.')
      {
         /* indicate no file type using 2 */
         tkptr->FindScriptCount = 2;
      }
      else
      {
         /**********************/
         /* file type supplied */
         /**********************/

         /* indicate file type supplied using 1 */
         tkptr->FindScriptCount = 1;
         tkptr->FindScriptTypePtr = sptr;
         tkptr->FindScriptTotalLength = tkptr->FindScriptLength;

         if (strsame (sptr, ".COM", -1))
            tkptr->RunTimePtr = "@";
         else
         if (strsame (sptr, ".EXE", -1))
            tkptr->RunTimePtr = "$";
         else
         {
            /* let's look for it in the run-time environment information */
            for (idx = 0; idx < Config.ScriptRunTimeCount; idx++)
            {
               if (Debug)
                  fprintf (stdout, "|%s|\n",
                           Config.ScriptRunTimeArray[idx]);
               sptr = tkptr->FindScriptTypePtr;
               cptr = Config.ScriptRunTimeArray[idx];
               while (*cptr && *cptr != ';' && *sptr)
               {
                  if (toupper(*cptr) != toupper(*sptr)) break;
                  cptr++;
                  sptr++;
               }
               if (*cptr == ';' && !*sptr) break;
            }

            if (idx >= Config.ScriptRunTimeCount)
            {
               /* nope, not found */
               static $DESCRIPTOR (ErrorScriptRunTimeFaoDsc,
"Execution of &nbsp;<TT>!AZ</TT>&nbsp; script types not configured.\0");
               char  String [256];
               $DESCRIPTOR (StringDsc, String);

               sys$fao (&ErrorScriptRunTimeFaoDsc, 0, &StringDsc, 
                        tkptr->FindScriptTypePtr);
               rqptr->ResponseStatusCode = 500;
               ErrorGeneral (rqptr, String, FI_LI);
               return (STS$K_ERROR);
            }

            /* found the file type, point to the run-time stuff */
            if (*cptr) cptr++;
            while (ISLWS(*cptr)) cptr++;
            tkptr->RunTimePtr = cptr;
            if (Debug) fprintf (stdout, "|%s|\n", tkptr->RunTimePtr);
         }
      }
   }

   if (tkptr->FindScriptCount == 2)
   {
      /*************************/
      /* first, DCL procedures */
      /*************************/

      memcpy (sptr = tkptr->FindScript+tkptr->FindScriptLength, ".COM", 5);
      tkptr->FindScriptTypePtr = sptr;
      tkptr->FindScriptTotalLength = tkptr->FindScriptLength + 4;
      tkptr->RunTimePtr = "@";
   }
   else
   if (tkptr->FindScriptCount == 3)
   {
      /***********************/
      /* second, executables */
      /***********************/

      memcpy (sptr = tkptr->FindScript+tkptr->FindScriptLength, ".EXE", 5);
      tkptr->FindScriptTypePtr = sptr;
      tkptr->FindScriptTotalLength = tkptr->FindScriptLength + 4;
      tkptr->RunTimePtr = "$";
   }
   else
   if (tkptr->FindScriptCount != 1 &&
       tkptr->FindScriptCount-4 < Config.ScriptRunTimeCount)
   {
      /*************************************/
      /* subsequent, run-time environments */
      /*************************************/

      tkptr->FindScriptTypePtr = sptr =
         tkptr->FindScript + tkptr->FindScriptLength;
      if (Debug)
         fprintf (stdout, "|%s|\n",
                  Config.ScriptRunTimeArray[tkptr->FindScriptCount-4]);
      for (cptr = Config.ScriptRunTimeArray[tkptr->FindScriptCount-4];
           *cptr && *cptr != ';';
           *sptr++ = *cptr++);
      *sptr = '\0';
      tkptr->FindScriptTotalLength = sptr - tkptr->FindScript;
      if (*cptr) cptr++;
      while (*cptr && ISLWS(*cptr)) cptr++;
      tkptr->RunTimePtr = cptr;
   }
   else
   if (tkptr->FindScriptCount != 1)
   {
      /**************/
      /* not found! */
      /**************/

      rqptr->ResponseStatusCode = 404;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_SCRIPT_NOT_FOUND), FI_LI);
      return (STS$K_ERROR);
   }
   /* else 'tkptr->FindScriptCount' can be -1, when the type is supplied! */

   /********************/
   /* write search DCL */
   /********************/

   DclCommandDsc.dsc$a_pointer = tkptr->DclCommand;
   DclCommandDsc.dsc$w_length = sizeof(tkptr->DclCommand)-1;
   sys$fao (&SearchFaoDsc, &Length, &DclCommandDsc, tkptr->FindScript);
   if (Debug)
   {
      tkptr->DclCommand[Length] = '\0';      
      fprintf (stdout, "|%s|\n", tkptr->DclCommand);
   }

   DECnetWrite (rqptr, tkptr->DclCommand, Length);
   DECnetRead (rqptr);   
   return (SS$_RETRY);
}

/*****************************************************************************/
/*
Implement (most) OSU server scripting dialogs.
Behaviour determined from the OSU 'script_execute.c' module.
*/ 
 
DECnetOsuDialog (struct RequestStruct *rqptr)

{
   register int  cnt;
   register char  *cptr, *sptr, *zptr;
   register struct DECnetTaskStruct  *tkptr;

   int  value;
   char  Buffer [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetOsuDialog()\n");

   tkptr = rqptr->DECnetTaskPtr;

   if (Debug) fprintf (stdout, "state: %d\n", tkptr->OsuDialogState);

   if (tkptr->OsuDialogState == OSU_REUSE)
   {
      /**********************/
      /* probing connection */
      /**********************/

      if (tkptr->QueuedDECnetIO) tkptr->QueuedDECnetIO--;

      /* reuse needs to be reset with each use of the DECnet task */
      tkptr->cnptr->ReuseConnection = false;

      if (Debug)
         fprintf (stdout, "DECnetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->DECnetTaskPtr->DECnetWriteIOsb.Status);

      if (VMSnok (tkptr->DECnetWriteIOsb.Status))
      {
         /* problem writing to established connection, restart */
         DECnetConnectEnd (tkptr);
         DECnetConnectBegin (tkptr);
         return;
      }

      Accounting.DoDECnetReuseCount++;
      Accounting.DoDECnetOsuReuseCount++;
      tkptr->OsuDialogState = OSU_BEGIN;

      /* drop through to continue with first phase of dialog */
   }

   if (tkptr->OsuDialogState == OSU_BEGIN)
   {
      /******************/
      /* initial dialog */
      /******************/

      /*
         OSU task executive expects four records as initial and unsolicited
         data from the server.  These are: script type, HTTP method name,
         HTTP protocol version, request path (excluding any query string).
      */

      Accounting.DoDECnetOsuCount++;
      tkptr->OsuDnetRecMode = tkptr->OsuDnetCgi = false;

      DECnetWrite (rqptr, "HTBIN", 5);
      DECnetWrite (rqptr, rqptr->HttpMethodName, -1);
      DECnetWrite (rqptr, HttpProtocol, -1);

      /* 
         URL (everything before any query string in original request)
         OSU requires the mapped script portion, which we have have to
         construct from the script name and the path information.
         "<DNETRQURL>" provides the unmapped (original) request path.
      */
      zptr = (sptr = Buffer) + sizeof(Buffer);
      cptr = rqptr->ScriptName;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      cptr = rqptr->PathInfoPtr;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         DECnetEnd (rqptr);
         return;
      }
      *sptr = '\0';
      /* it's an asynchronous write so we need some persistant storage */
      cptr = VmGetHeap (rqptr, sptr-Buffer+1);
      memcpy (cptr, Buffer, sptr-Buffer+1);
      DECnetWrite (rqptr, cptr, sptr-Buffer);

      /* now into dialog state */
      tkptr->OsuDialogState = OSU_DIALOG;

      DECnetRead (rqptr);
      return;
   }

   /**********************/
   /* dialog established */
   /**********************/

   cptr = rqptr->OutputBufferPtr;
   cnt = tkptr->DECnetReadIOsb.Count;
   cptr[cnt] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", cptr);

   if (tkptr->OsuDialogState == OSU_DIALOG && *cptr != '<')
   {
      /************************/
      /* start of data stream */
      /************************/

      if (tkptr->OsuDnetCgi)
      {
         /* CGI script, check the initial response */
         value = CgiScriptHeader (rqptr, cptr, cnt);
         if (value == CGI_OUTPUT_TERMINATE)
         {
            /* terminate processing */
            DECnetEnd (rqptr);
            return;
         }
      }

      /* now set the carriage-control appropriately */
      if (tkptr->OsuDnetRecMode)
         tkptr->OsuDialogState = OSU_OUTPUT_REC;
      else
         tkptr->OsuDialogState = OSU_OUTPUT_RAW;

      /* drop through so the line can be processed by "_REC" or "_RAW" */
   }

   switch (tkptr->OsuDialogState)
   {
      case OSU_DIALOG :

         /********************/
         /* stream massaging */
         /********************/

         if (!memcmp (cptr, "<DNETRECMODE>", 13))
         {
            /* forces text mode */
            tkptr->OsuDnetRecMode = true;
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETCGI>", 9))
         {
            /* similar to raw mode except response header is checked */
            tkptr->OsuDnetCgi = true;
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETRAW>", 9))
         {
            /* script handles all response formatting, return byte-for-byte */
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETTEXT>", 10))
         {
            /* script sends status line then plain text */
            tkptr->OsuDialogState = OSU_DNETTEXT;
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "</DNETCGI>", 10) ||
             !memcmp (cptr, "</DNETRAW>", 10) ||
             !memcmp (cptr, "</DNETTEXT>", 11))
         {
            /* end of script output */
            if (tkptr->cnptr->ReuseConnection)
               DECnetWrite (rqptr, "<DNETREUSE>", 11);

            DECnetEnd (rqptr);
            return;
         }

         /**********************/
         /* information dialog */
         /**********************/

         if (!memcmp (cptr, "<DNETARG>", 9))
         {
            /* full search argument (query string) */
            cptr = VmGetHeap (rqptr, rqptr->QueryStringLength+2);
            /* OSU supplies a leading '?' query separator */
            *cptr = '?';
            memcpy (cptr+1, rqptr->QueryStringPtr, rqptr->QueryStringLength);
            DECnetWrite (rqptr, cptr, rqptr->QueryStringLength+1);
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETARG2>", 10))
         {
            /* truncated search argument (query string) */
            cptr = VmGetHeap (rqptr, 256);
            /* OSU supplies a leading '?' query separator */
            *cptr = '?';
            if (rqptr->QueryStringLength > 254)
            {
               memcpy (cptr+1, rqptr->QueryStringPtr, 254);
               cptr[255] = '\0';
               DECnetWrite (rqptr, cptr, 255);
            }
            else
            {
               memcpy (cptr+1, rqptr->QueryStringPtr, rqptr->QueryStringLength);
               DECnetWrite (rqptr, cptr, rqptr->QueryStringLength+1);
            }
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETPATH>", 10))
         {
            /* script path (but excluding script name itself) */
            register char  *cptr;
            for (cptr = rqptr->ScriptName; *cptr; cptr++);
            while (cptr > rqptr->ScriptName && *cptr != '/') cptr--;
            DECnetWrite (rqptr, rqptr->ScriptName,
                         cptr-(char*)rqptr->ScriptName+1);
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETHDR>", 9))
         {
            /* send the full request header */
            tkptr->OsuDialogState = OSU_DNET_HDR;
            DECnetWriteRequest (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETINPUT>", 11))
         {
            /* begin to write the request body */
            tkptr->OsuDialogState = OSU_DNET_INPUT;
            DECnetWriteRequest (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETBINDIR>", 12))
         {
            /*
               OSU script and binaries VMS directory specification.
               Isolate the 'HT_ROOT:[dir]' from the example
               'node::"0=wwwexec"HT_ROOT:[dir]script.ext'
            */
            for (cptr = tkptr->MappedScript; *cptr && *cptr != ':'; cptr++);
            if (*cptr == ':') cptr++;
            if (*cptr == ':') cptr++;
            if (*cptr == '\"')
            {
               cptr++;
               while (*cptr && *cptr != '\"') cptr++;
               if (*cptr) cptr++;
            }
            sptr = cptr;
            while (*cptr) cptr++;
            while (cptr > tkptr->MappedScript && *cptr != ']') cptr--;
            DECnetWrite (rqptr, sptr, cptr - sptr + 1);
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETRQURL>", 11))
         {
            /* full request URL prior to any mapping */
            DECnetWrite (rqptr, rqptr->ResourcePtr, -1);
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETINVCACHE>", 14))
         {
            /* invalidate cache */
            rqptr->ResponseStatusCode = 501;
            ErrorGeneral (rqptr, ErrorOsuNoInvCache, FI_LI);
            DECnetEnd (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETHOST>", 10))
         {
            /* server host name */
            DECnetWrite (rqptr, rqptr->ServicePtr->ServerHostName, -1);
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETID>", 8))
         {
            /* composite line of request data */
            DECnetOsuDnetId (rqptr, tkptr, false);
            return;
         }

         if (!memcmp (cptr, "<DNETID2>", 9))
         {
            /* composite line of request data (plus a bit!) */
            DECnetOsuDnetId (rqptr, tkptr, true);
            return;
         }

         if (!memcmp (cptr, "<DNETMANAGE>", 12))
         {
            /* server management */
            rqptr->ResponseStatusCode = 500;
            ErrorGeneral (rqptr, ErrorOsuNoManage, FI_LI);
            DECnetEnd (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETREUSE>", 11))
         {
            if (Config.DECnetReuseLifeTime)
            {
               /* reuse enabled, do not drop DECnet link */
               tkptr->cnptr->ReuseConnection = true;
            }
            DECnetRead (rqptr);
            return;
         }

         if (!memcmp (cptr, "<DNETXLATE>", 11) ||
             !memcmp (cptr, "<DNETXLATEV>", 12))
         {
            /* into translate state */
            tkptr->OsuDialogState = OSU_DNET_XLATE;
            /* read the path to be translated */
            DECnetRead (rqptr);
            return;
         }

         /* hmmm, unknown dialog tag */
         {
            char  Tag [256];
            $DESCRIPTOR (BufferDsc, Buffer);

            CopyToHtml (Tag, sizeof(Tag), cptr, -1);
            rqptr->ResponseStatusCode = 500;
            sys$fao (&ErrorOsuUnknownDsc, 0, &BufferDsc, Tag);
            ErrorGeneral (rqptr, Buffer, FI_LI);
            DECnetEnd (rqptr);
            return;
         }


      /****************/
      /* other states */
      /****************/

      case OSU_DNETTEXT :

         /*
            This output record contains a status line something like "200 ok".
            Get the status code, ignore the rest, generate an HTTP header.
            Then move into a state to accept record-oriented script output.
         */
         while (*cptr && !isdigit(*cptr) && *cptr != '\n') cptr++;
         if (isdigit(*cptr)) rqptr->ResponseStatusCode = atoi(cptr);
         if ((rqptr->ResponseHeaderPtr =
              HttpHeader (rqptr, rqptr->ResponseStatusCode,
                          "text/plain", -1, NULL, NULL)) == NULL)
         {
            DECnetEnd (rqptr);
            return;
         }

         /* note that the script has generated some output */
         tkptr->ScriptResponded += cnt;

         /* now move into record output state */
         tkptr->OsuDialogState = OSU_OUTPUT_REC;
         DECnetRead (rqptr);

         return;


      case OSU_OUTPUT_RAW :

         /*
            This script output record is "binary" data.  Write it to
            the client via the network without any further processing.
         */
         if (!memcmp (cptr, "</DNETCGI>", 10) ||
             !memcmp (cptr, "</DNETRAW>", 10) ||
             !memcmp (cptr, "</DNETTEXT>", 11))
         {
            /* end of script output */
            DECnetEnd (rqptr);
            return;
         }

         /* note that the script has generated some output */
         tkptr->ScriptResponded += cnt;

         NetWrite (rqptr, &DECnetRead, cptr, cnt);
         return;


      case OSU_OUTPUT_REC :

         /*
            This record of script output is essentially a line of "text".
            Esure it has correct HTTP carriage-control (trailing newline).
            Write this (possibly) massaged record to the client (network).
         */
         if (!memcmp (cptr, "</DNETCGI>", 10) ||
             !memcmp (cptr, "</DNETRAW>", 10) ||
             !memcmp (cptr, "</DNETTEXT>", 11))
         {
            /* end of script output */
            DECnetEnd (rqptr);
            return;
         }

         /* adjust carriage-control if necessary */
         if (cnt)
            if (cptr[cnt-1] != '\n')
               cptr[cnt++] = '\n';
            else;
         else
            cptr[cnt++] = '\n';
         if (Debug) cptr[cnt] = '\0';

         /* note that the script has generated some output */
         tkptr->ScriptResponded += cnt;

         NetWrite (rqptr, &DECnetRead, cptr, cnt);
         return;


      case OSU_DNET_XLATE :

         /* record contains a URL-style path to be translated into RMS */
         DECnetOsuDnetXlate (rqptr, tkptr, cptr);
         return;


      default :

         /* Hmmm, problem? */
         rqptr->ResponseStatusCode = 500;
         ErrorGeneral (rqptr, ErrorOsuImplementation, FI_LI);
         DECnetEnd (rqptr);
         return;
   }
}

/*****************************************************************************/
/*
Write a single record comprising a series of white-space separated request
data to the OSU task.
*/ 
 
DECnetOsuDnetId
(
struct RequestStruct *rqptr,
struct DECnetTaskStruct *tkptr,
boolean Id2
)
{
   static $DESCRIPTOR (SignedLongFaoDsc, "!SL\0");
   register char  *cptr, *sptr, *zptr;

   char Buffer [1024],
        String [256];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetOsuDnetId()\n");

   cptr = rqptr->OutputBufferPtr;
   zptr = (sptr = Buffer) + sizeof(Buffer);

   /* server identification (underscores substituted for spaces) */
   cptr = SoftwareID;
   while (*cptr && sptr < zptr)
   {
      if (*cptr == ' ')
      {
         *sptr++ = '_';
         cptr++;
      }
      else
         *sptr++ = *cptr++;
   }

   /* local host */
   if (sptr < zptr) *sptr++ = ' ';
   for (cptr = rqptr->ServicePtr->ServerHostName;
        *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* local port */
   if (sptr < zptr) *sptr++ = ' ';
   for (cptr = rqptr->ServicePtr->ServerPortString;
        *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* remote port (not available for WASD) */
   if (sptr < zptr) *sptr++ = ' ';
   for (cptr = "65535"; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* remote host address */
   if (sptr < zptr) *sptr++ = ' ';
   sys$fao (&SignedLongFaoDsc, 0, &StringDsc,
            rqptr->ClientInternetAddress32bit);
   for (cptr = String; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* remote user */
   if (rqptr->RemoteUser[0])
   {
      if (sptr < zptr) *sptr++ = ' ';
      for (cptr = rqptr->RemoteUser;
           *cptr && sptr < zptr; *sptr++ = *cptr++);
   }
   else
   if (Id2)
      if (sptr < zptr) *sptr++ = ' ';

   if (Id2 && Config.DnsLookup)
   {
      /* remote host name */
      if (sptr < zptr) *sptr++ = ' ';
      for (cptr = rqptr->ClientHostName;
           *cptr && sptr < zptr; *sptr++ = *cptr++);
   }

   *sptr = '\0';

   /* it's an asynchronous write so we need some persistant storage */
   cptr = VmGetHeap (rqptr, sptr-Buffer+1);
   memcpy (cptr, Buffer, sptr-Buffer+1);

   DECnetWrite (rqptr, cptr, sptr-Buffer);
   DECnetRead (rqptr);
}

/*****************************************************************************/
/*
Map the supplied path (either absolute or relative to the request path) into a
VMS file system specification and write that to the OSU task.
*/ 
 
DECnetOsuDnetXlate
(
struct RequestStruct *rqptr,
struct DECnetTaskStruct *tkptr,
char *Path
)
{
   register char  *cptr;

   int  TransPathLength;
   char AbsPath [256],
        TransPath [256],
        VmsPath [256+7];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetOsuDnetXlate() |%s|\n", Path);

   /* WASD always ignores any method */
   if (*(cptr = Path) == '(')
   {
      while (*cptr && *cptr != ')') cptr++;
      if (*cptr) cptr++;
   }

   TransPath[0] = '\0';
   TransPathLength = 0;
   if (*cptr == '.')
   {
      MapUrl_VirtualPath (rqptr->PathInfoPtr, cptr, AbsPath, sizeof(AbsPath));
      cptr = AbsPath;
   }
   if (*cptr)
   {
      MapUrl_Map (cptr, VmsPath, NULL, NULL, rqptr);
      TransPathLength = MapUrl_VmsToUrl (TransPath, VmsPath, false);
   }

   DECnetWrite (rqptr, TransPath, TransPathLength);
   DECnetRead (rqptr);

   /* back from translate into dialog state */
   tkptr->OsuDialogState = OSU_DIALOG;
}

/*****************************************************************************/
/*
Every minute scan the list of network connections looking for those whose
lifetimes have expired. Break the connection of those who have. As this is a
one minute tick set the lifetime counters to configuration plus one to ensure
at least that number of minutes before expiry.
*/

DECnetSupervisor (boolean TimerExpired)

{
   static boolean  TimerSet = false;
   static unsigned long  OneMinuteDelta [2] = { -600000000, -1 };

   register struct ListEntryStruct  *leptr;
   register struct DECnetConnectStruct  *cnptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetSupervisor() %d\n", TimerExpired);

   if (!TimerExpired)
   {
      /* new request, if the supervisor is already running just return */
      if (TimerSet) return;

      /* kick the supervisor timer into life, and then return */
      if (VMSnok (status =
          sys$setimr (0, &OneMinuteDelta, &DECnetSupervisor, true, 0)))
         ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);
      TimerSet = true;
      return;
   }

   /* must have got here via the supervisor's timer expiring */
   TimerSet = false;

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;
      if (Debug) DECnetConnectItemDebug (leptr, cnptr);

      /* only interested in those which are currently connected */
      if (!cnptr->DECnetChannel) continue;

      if (cnptr->LifeTimeCount > 0)
      {
         TimerSet = true;
         if (--cnptr->LifeTimeCount) continue;
      }

      status = sys$dassgn (cnptr->DECnetChannel);
      if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
      cnptr->DECnetChannel = cnptr->LifeTimeCount = 0;
   }

   if (TimerSet)
   {
      /* at least one item in the connect list is still counting down */
      if (VMSnok (status =
          sys$setimr (0, &OneMinuteDelta, &DECnetSupervisor, true, 0)))
         ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);
   }
}

/*****************************************************************************/
/*
Return a report on DECnet scripting, in particular displaying each item in the
connection list.
*/ 

int DECnetScriptingReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... DECnet Scripting Report</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>DECnet Scripting Report</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=2>Statistics</TH></TR>\n\
<TH ALIGN=left>Total</TH>\
<TD>!UL &nbsp;<I>(!UL reuse)</I></TD></TR>\n\
<TR ALIGN=left><TH>CGI <I>script</I></TH>\
<TD>!UL &nbsp;<I>(!UL reuse)</I></TD></TR>\n\
<TR ALIGN=left><TH>OSU <I>script</I></TH>\
<TD>!UL &nbsp;<I>(!UL reuse)</I></TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH></TH><TH>Connect</TH><TH>Lifetime</TH>\
<TH>Reused</TH><TH>Total</TH><TH>Last</TH></TR>\n\
<TR><TH></TH><TH>Client</TH><TH COLSPAN=4>Request</TH></TR>\n\
<TR><TH COLSPAN=6></TH></TR>\n");

   static $DESCRIPTOR (ItemFaoDsc,
"<TR><TD ALIGN=right>!UL</TD>\
<TD>!AZ</TD><TD ALIGN=right>!AZ!UL!AZ</TD><TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL</TD><TD><FONT SIZE=-2>!AZ</FONT></TD><TR>\n\
<TR><TD></TD><TD>!AZ</TD><TD COLSPAN=4><TT>!AZ</TT></TD></TR>\n");

   static $DESCRIPTOR (ButtonsFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=submit VALUE=\" Purge \">\n\
</FORM>\n\
</TD><TD>&nbsp;</TD><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=submit VALUE=\" Force Disconnect \">\n\
</FORM>\n\
</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;
   register struct DECnetConnectStruct  *cnptr;
   register struct ListEntryStruct  *leptr;

   int  status,
        Count;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         Client [256],
         Path [512];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetScriptingReport()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   *vecptr++ = Accounting.DoDECnetCount;
   *vecptr++ = Accounting.DoDECnetReuseCount;
   *vecptr++ = Accounting.DoDECnetCgiCount;
   *vecptr++ = Accounting.DoDECnetCgiReuseCount;
   *vecptr++ = Accounting.DoDECnetOsuCount;
   *vecptr++ = Accounting.DoDECnetOsuReuseCount;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   Count = 0;

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;
      if (Debug) DECnetConnectItemDebug (leptr, cnptr);

      vecptr = FaoVector;

      *vecptr++ = ++Count;
      if (cnptr->DECnetChannel)
         *vecptr++ = cnptr->ConnectString;
      else
         *vecptr++ = cnptr->ConnectString;
      if (cnptr->LifeTimeCount)
         *vecptr++ = "<B>";
      else
         *vecptr++ = "";
      /* fudge allows for correct reporting of lifetime */
      if (cnptr->LifeTimeCount == Config.DECnetReuseLifeTime+1)
         *vecptr++ = cnptr->LifeTimeCount-1;
      else
         *vecptr++ = cnptr->LifeTimeCount;
      if (cnptr->LifeTimeCount)
         *vecptr++ = "</B>";
      else
         *vecptr++ = "";
      *vecptr++ = cnptr->ReUsageCount;
      *vecptr++ = cnptr->UsageCount;
      *vecptr++ = DayDateTime (&cnptr->LastUsedBinaryTime, 23);

      if (cnptr->rqptr != NULL)
      {
         sptr = Client;
         if (cnptr->rqptr->RemoteUser[0])
         {
            cptr = cnptr->rqptr->RemoteUser;
            while (*cptr) *sptr++ = *cptr++;
            *sptr++ = '@';
         }
         cptr = cnptr->rqptr->ClientHostName;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         *vecptr++ = Client;

         sptr = Path;
         sptr += CopyToHtml (sptr, sizeof(Path)-(sptr-Path),
                             cnptr->rqptr->ScriptName, -1);
         CopyToHtml (sptr,  sizeof(Path)-(sptr-Path),
                     cnptr->rqptr->PathInfoPtr, -1);
         *vecptr++ = Path;
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
      }

      status = sys$faol (&ItemFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;
   *vecptr++ = HttpdInternalControl;
   *vecptr++ = "DECnet_purge";
   *vecptr++ = HttpdInternalControl;
   *vecptr++ = "DECnet_force";

   status = sys$faol (&ButtonsFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NextTaskFunction, Buffer, Length);
}

/*****************************************************************************/
/*
Control function to disconnect all network connections. The
'WithExtremePrejudice' option disconnects unconditionally, processing or not.
Called from Admin.c module.
*/

DECnetDisconnect
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean WithExtremePrejudice
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Server !AZ had !UL link!AZ disconnected, \
!UL marked for disconnection.\n\
</BODY>\n\
</HTML>\n");

   register struct  ListEntryStruct  *leptr;
   register struct DECnetConnectStruct  *cnptr;
   register unsigned long  *vecptr;

   int  status,
        DisconnectedCount,
        MarkedCount;
   unsigned long  FaoVector [16];
   unsigned short  Length;
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetPurgeAllConnect()\n");

   DECnetPurgeAllConnectCount++;
   DisconnectedCount = MarkedCount = 0;

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;
      if (Debug) DECnetConnectItemDebug (leptr, cnptr);

      /* only interested in those which are currently connected */
      if (!cnptr->DECnetChannel) continue;

      if (WithExtremePrejudice ||
          cnptr->rqptr == NULL)
      {
         /* connection not active, or we don't care, disconnect */
         status = sys$dassgn (cnptr->DECnetChannel);
         if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
         cnptr->DECnetChannel = cnptr->LifeTimeCount = 0;
         DisconnectedCount++;
         continue;
      }
      else
      {
         /* connection is currently active, just leave marked for delete */
         cnptr->IsMarkedForDisconnect = true;
         MarkedCount++;
      }
   }

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);
   *vecptr++ = ServerHostPort;
   *vecptr++ = DisconnectedCount;
   if (DisconnectedCount == 1)
      *vecptr++ = "";
   else
      *vecptr++ = "s";
   *vecptr++ = MarkedCount;

   sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NextTaskFunction, Buffer, Length);
}

/*****************************************************************************/
/*
Same as for DECnetPurgeAllConnect() except called from control module from
command line!
*/

char* DECnetControlDisconnect (boolean WithExtremePrejudice)

{
   static char  Response [64];
   static $DESCRIPTOR (ResponseFaoDsc,
          "!UL disconnected, !UL marked for disconnection");
   static $DESCRIPTOR (ResponseDsc, Response);

   register struct  ListEntryStruct  *leptr;
   register struct DECnetConnectStruct  *cnptr;

   int  status,
        DisconnectedCount,
        MarkedCount;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetControlPurgeAllConnect()\n");

   DECnetPurgeAllConnectCount++;
   DisconnectedCount = MarkedCount = 0;

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;
      if (Debug) DECnetConnectItemDebug (leptr, cnptr);

      /* only interested in those which are currently connected */
      if (!cnptr->DECnetChannel) continue;

      if (WithExtremePrejudice ||
          cnptr->rqptr == NULL)
      {
         /* connection not active, or we don't care, disconnect */
         status = sys$dassgn (cnptr->DECnetChannel);
         if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
         cnptr->DECnetChannel = cnptr->LifeTimeCount = 0;
         DisconnectedCount++;
         continue;
      }
      else
      {
         /* connection is currently active, disconnect when finished */
         cnptr->IsMarkedForDisconnect = true;
         MarkedCount++;
      }
   }

   sys$fao (&ResponseFaoDsc, &Length, &ResponseDsc,
            DisconnectedCount, MarkedCount);
   Response[Length] = '\0';

   return (Response);
}

/*****************************************************************************/
/*
*/ 
 
int DECnetConnectItemDebug
(
struct ListEntryStruct *leptr,
struct DECnetConnectStruct *cnptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetConnectItemDebug()\n");

#ifdef DBUG

   if (leptr != NULL)
      fprintf (stdout,
"leptr %d %d %d (%d)\n",
         leptr->PrevPtr, leptr, leptr->NextPtr, cnptr);

   if (cnptr != NULL)
      fprintf (stdout,
"%d |%s| %d %d %d [%d]\n",
         cnptr->DECnetChannel, cnptr->ConnectString, cnptr->LifeTimeCount,
         cnptr->UsageCount, cnptr->ReUsageCount, cnptr->rqptr);

#endif /* DBUG */

}

int DECnetConnectListDebug (struct DECnetTaskStruct *tkptr)

{
   register struct DECnetConnectStruct  *cnptr;
   register struct ListEntryStruct  *leptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DECnetConnectListDebug()\n");

#ifdef DBUG

   for (leptr = DECnetConnectList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      cnptr = (struct DECnetConnectStruct*)leptr;

      fprintf (stdout, "leptr %d %d %d (%d)\n",
               leptr->PrevPtr, leptr, leptr->NextPtr, cnptr);

      fprintf (stdout,
"%d |%s| %d %d %d [%d]\n",
         cnptr->DECnetChannel, cnptr->ConnectString, cnptr->LifeTimeCount,
         cnptr->UsageCount, cnptr->ReUsageCount, cnptr->rqptr);
   }

#endif /* DBUG */

}

/*****************************************************************************/

