/*****************************************************************************/
/*
                                  Descr.c

This module implements a full multi-threaded, AST-driven, asynchronous
description generator for a file.

This is not a full task in the sense of a file transfer.  This module merely
determines a file description.  For an HTML file this is by looking for the
contents of a <TITLE></TITLE> or <H1></H1> tag pair.  For a plain text file,
the first non-blank line.  If no internal description can be determined the
buffer is made an empty string.

This module never returns a valid status and ALWAYS calls the supplied next
task (AST) function.  This should check for a generated error message to
determine is there were any problems.


VERSION HISTORY
---------------
25-OCT-97  MGD  DescriptionEscape() now includes only printable characters
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile
01-FEB-97  MGD  HTTPd version 4
23-MAY-96  MGD  functionality generalized, moved from directory module
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "descr.h"
#include "error.h"
#include "httpd.h"
#include "msg.h"
#include "support.h"
#include "vm.h"

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  FileBufferSize;
extern char  SoftwareID[];
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
*/

Description
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *FilePath,
char *FileName,
char *DescriptionBuffer,
int SizeOfDescriptionBuffer,
int ContentTypes
)
{
   register char  *cptr, *sptr, *zptr;
   register struct DescrTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Description() %d\n", rqptr->ErrorMessagePtr);

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (possibly multiple serially) */
   if (rqptr->DescrTaskPtr == NULL)
   {
      rqptr->DescrTaskPtr = tkptr = (struct DescrTaskStruct*)
         VmGetHeap (rqptr, sizeof(struct DescrTaskStruct));
   }
   else
   {
      tkptr = rqptr->DescrTaskPtr;
      memset (tkptr, 0, sizeof(struct DescrTaskStruct));
   }
   tkptr->NextTaskFunction = NextTaskFunction;

   *(tkptr->DescriptionBufferPtr = DescriptionBuffer) = '\0';
   tkptr->SizeOfDescriptionBuffer = SizeOfDescriptionBuffer;

   cptr = FilePath;
   zptr = (sptr = tkptr->FilePath) + sizeof(tkptr->FilePath);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DescriptionEnd (rqptr);
      return;
   }
   *sptr = '\0';

   cptr = FileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DescriptionEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error, declare the next task */
         SysDclAst (tkptr->NextTaskFunction, rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /***************/
   /* description */
   /***************/

   /* using the pointer from above find the file type */
   while (sptr > tkptr->FileName && *sptr != '.') sptr--;
   cptr = ConfigContentType (NULL, sptr);

   if (tolower(cptr[0]) == 't' && strsame (cptr, "text/", 5))
   {
      if ((ContentTypes & DESCRIPTION_TEXT_HTML ||
           ContentTypes & DESCRIPTION_ALL) &&
          strsame (cptr+5, "html", -1))
         status = DescriptionHtml (rqptr);

      if ((ContentTypes & DESCRIPTION_TEXT_PLAIN ||
           ContentTypes & DESCRIPTION_ALL) &&
          strsame (cptr+5, "plain", -1))
         status = DescriptionPlain (rqptr);

      if (VMSnok (status))
      {
         /* declare the next task */
         SysDclAst (tkptr->NextTaskFunction, rqptr);
      }
      return;
   }
   else
   {
      /* indicate no description could be generated */
      tkptr->Description[0] = DESCRIPTION_IMPOSSIBLE;

      /* declare the next task */
      SysDclAst (tkptr->NextTaskFunction, rqptr);
   }
}

/*****************************************************************************/
/*
*/ 

DescriptionEnd (struct RequestStruct *rqptr)

{
   register struct DescrTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DescriptionEnd()\n");

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   if (tkptr->FileFab.fab$w_ifi) sys$close (&tkptr->FileFab, 0, 0);

   /* declare the next task */
   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
*/

int DescriptionPlain (struct RequestStruct *rqptr)

{
   register struct DescrTaskStruct  *tkptr;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DescriptionPlain() |%s|\n",
               rqptr->DescrTaskPtr->FileName);

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      DescriptionError (rqptr, status, __LINE__);
      return (status);
   }

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   /* 2 buffers and read ahead performance option */
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH;
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-1;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&tkptr->FileFab, 0, 0);
      DescriptionError (rqptr, status, __LINE__);
      return (status);
   }

   *(tkptr->DescriptionPtr = tkptr->DescriptionBufferPtr) = '\0';
   tkptr->DescriptionRetrieved = false;
   tkptr->DescriptionLineCount = 0;

   /* set the RAB user context storage to the client thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;

   /* queue up a read of the first record in the HTML file */
   sys$get (&tkptr->FileRab, &DescriptionPlainRecord, &DescriptionPlainRecord);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
This is an AST completion routine called each time sys$get() completes.  It 
returns with the next record, end-of-file, or some other error.
*/

DescriptionPlainRecord (struct RAB *RabPtr)

{
   register char  *dptr, *rptr, *zptr;
   register struct RequestStruct  *rqptr;
   register struct DescrTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DescriptionPlainRecord() sts: %%X%08.08X stv: %%X%08.08X\n",
         RabPtr->rab$l_sts, RabPtr->rab$l_stv);
   }

   /* get the pointer to the thread from the RAB user context storage */
   rqptr = RabPtr->rab$l_ctx;

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end-of-file */
         /***************/

         sys$close (&tkptr->FileFab, 0, 0);
         DescriptionEnd (rqptr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      {
         DescriptionError (rqptr, tkptr->FileRab.rab$l_sts, __LINE__);
         DescriptionEnd (rqptr);
         return;
      }
   }

   tkptr->DescriptionLineCount++;
   tkptr->FileRab.rab$l_ubf[tkptr->FileRab.rab$w_rsz] = '\0';
   if (Debug) fprintf (stdout,"|%s|\n", tkptr->FileRab.rab$l_ubf);

   /************************************/
   /* look for first plain-text record */
   /************************************/

   for (rptr = tkptr->FileRab.rab$l_ubf; *rptr && !isalnum(*rptr); rptr++);
   if (!isalnum(*rptr))
   {
      if (tkptr->DescriptionLineCount <= Config.DirDescriptionLines)
      {
         /* queue another read, completion AST back to this function again */
         sys$get (&tkptr->FileRab,
                  &DescriptionPlainRecord, &DescriptionPlainRecord);
         return;
      }

      /********************/
      /* out-of-patience! */
      /********************/

      sys$close (&tkptr->FileFab, 0, 0);

      DescriptionEnd (rqptr);
      return;
   }

   /*******************/
   /* get description */
   /*******************/

   dptr = tkptr->DescriptionBufferPtr;
   zptr = tkptr->DescriptionBufferPtr + tkptr->SizeOfDescriptionBuffer - 1;
   while (*rptr && dptr < zptr) *dptr++ = *rptr++;
   *dptr = '\0';

   DescriptionEscape (rqptr);

   sys$close (&tkptr->FileFab, 0, 0);

   if (Debug)
      fprintf (stdout, "tkptr->DescriptionBufferPtr |%s|\n",
               tkptr->DescriptionBufferPtr);

   DescriptionEnd (rqptr);
}

/*****************************************************************************/
/*
Initiate an asynchronous read of enough records of an HTML file to locate the 
<TITLE> element.  This is used as the file "description".  When the first read 
is queued return immediately to the calling routine.
*/

int DescriptionHtml (struct RequestStruct *rqptr)

{
   register struct DescrTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DescriptionHtml() |%s|\n",
               rqptr->DescrTaskPtr->FileName);

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      DescriptionError (rqptr, status, __LINE__);
      return (status);
   }

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   /* 2 buffers and read ahead performance option */
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH;
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-1;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&tkptr->FileFab, 0, 0);
      DescriptionError (rqptr, status, __LINE__);
      return (status);
   }

   *(tkptr->DescriptionPtr = tkptr->DescriptionBufferPtr) = '\0';
   tkptr->DescriptionInside = tkptr->DescriptionRetrieved = false;
   tkptr->DescriptionOpeningTagCount = tkptr->DescriptionClosingTagCount =
      tkptr->DescriptionLineCount = 0;

   /* set the RAB user context storage to the client thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;

   /* queue up a read of the first record in the HTML file */
   sys$get (&tkptr->FileRab, &DescriptionHtmlRecord, &DescriptionHtmlRecord);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
This is an AST completion routine called each time sys$get() completes.  It 
returns with the next record, end-of-file, or some other error.  Examine the 
record looking for text comprising the first of either the <TITLE></TITLE> 
element or a heading element (e.g. <H1></H1>), retrieving this and using it 
as the file description.
*/

DescriptionHtmlRecord (struct RAB *RabPtr)

{
   register char  *dptr, *rptr, *zptr;
   register struct RequestStruct  *rqptr;
   register struct DescrTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DescriptionHtmlRecord() sts: %%X%08.08X stv: %%X%08.08X\n|%s|\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv,
         ((struct RequestStruct*)(RabPtr->rab$l_ctx))->DescrTaskPtr->DescriptionBufferPtr);
   }

   /* get the pointer to the thread from the RAB user context storage */
   rqptr = RabPtr->rab$l_ctx;

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end-of-file */
         /***************/

         sys$close (&tkptr->FileFab, 0, 0);
         DescriptionEnd (rqptr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      {
         sys$close (&tkptr->FileFab, 0, 0);
         DescriptionError (rqptr, tkptr->FileRab.rab$l_sts, __LINE__);
         DescriptionEnd (rqptr);
         return;
      }
   }

   tkptr->DescriptionLineCount++;
   tkptr->FileRab.rab$l_ubf[tkptr->FileRab.rab$w_rsz] = '\0';

   /****************************************/
   /* look for the <TITLE> or <Hn> element */
   /****************************************/

   /* 
      'zptr' points at the last character of the description storage.
      It prevents non-closed <TITLE> or <Hn> structures from running off
      the end of the storage area.
   */

   dptr = tkptr->DescriptionPtr;
   zptr = tkptr->DescriptionBufferPtr + tkptr->SizeOfDescriptionBuffer - 1;

   /*
      Add a space for any record boundary occuring in the title,
      except if it's a leading space (i.e. first after the leading quote).
   */
   if (tkptr->DescriptionInside &&
       dptr > tkptr->DescriptionBufferPtr+1 &&
       dptr < zptr)
      *dptr++ = ' ';

   for (rptr = tkptr->FileRab.rab$l_ubf; *rptr; rptr++)
   {
      if (*rptr == '<')
      {
         tkptr->DescriptionOpeningTagCount++;
         if (toupper(rptr[1]) == 'T' && strsame (rptr+2, "ITLE>", 5))
         {
            tkptr->DescriptionInside = true;
            rptr += 6;
            tkptr->DescriptionClosingTagCount++;
            continue;
         }
         if (toupper(rptr[1]) == 'H' && isdigit(rptr[2]) && rptr[3] == '>')
         {
            tkptr->DescriptionInside = true;
            rptr += 3;
            tkptr->DescriptionClosingTagCount++;
            continue;
         }
         if (rptr[1] == '/')
         {
            if (toupper(rptr[2]) == 'T' && strsame (rptr+3, "ITLE>", 5))
            {
               tkptr->DescriptionInside = false;
               tkptr->DescriptionRetrieved = true;
               rptr += 7;
               tkptr->DescriptionClosingTagCount++;
               /* suppress any trailing white-space */
               if (dptr > tkptr->DescriptionBufferPtr+1) dptr--;
               while (dptr > tkptr->DescriptionBufferPtr+1 &&
                      ISLWS(*dptr)) dptr--;
               *++dptr = '\0';
               continue;
            }
            if (toupper(rptr[2]) == 'H' && isdigit(rptr[3]) && rptr[4] == '>')
            {
               tkptr->DescriptionInside = false;
               tkptr->DescriptionRetrieved = true;
               rptr += 4;
               tkptr->DescriptionClosingTagCount++;
               /* suppress any trailing white-space */
               if (dptr > tkptr->DescriptionBufferPtr+1) dptr--;
               while (dptr > tkptr->DescriptionBufferPtr+1 &&
                      ISLWS(*dptr)) dptr--;
               *++dptr = '\0';
               continue;
            }
         }
      }
      else
      if (*rptr == '>')
      {
         tkptr->DescriptionClosingTagCount++;
         continue;
      }

      /* don't scan the line further if we've found what we're looking for */
      if (tkptr->DescriptionRetrieved) break;

      /* if we're currently inside a tag name (between "<" and ">") */
      if (tkptr->DescriptionOpeningTagCount > tkptr->DescriptionClosingTagCount)
         continue;

      /* suppress leading white-space */
      if (tkptr->DescriptionInside &&
          !(dptr == tkptr->DescriptionBufferPtr && ISLWS(*rptr)) &&
          isprint (*rptr) &&
          dptr < zptr)
         *dptr++ = *rptr;
   }

   /* terminate the title/description */
   if (dptr > tkptr->DescriptionBufferPtr) *dptr = '\0';
   tkptr->DescriptionPtr = dptr;

   if (tkptr->DescriptionRetrieved ||
       tkptr->DescriptionLineCount > Config.DirDescriptionLines)
   {
      /************************************/
      /* title found, or out-of-patience! */
      /************************************/

      sys$close (&tkptr->FileFab, 0, 0);

      if (Debug)
         fprintf (stdout, "tkptr->DescriptionBufferPtr |%s|\n",
                  tkptr->DescriptionBufferPtr);

      DescriptionEnd (rqptr);
      return;
   }
   else
   {
      /*****************/
      /* still looking */
      /*****************/

      /* queue another read, completion AST back to this function again */
      sys$get (&tkptr->FileRab,
               &DescriptionHtmlRecord, &DescriptionHtmlRecord);
   }
}

/*****************************************************************************/
/*
Make sure we only meaningful and HTML-acceptable characters into the
description.
*/

DescriptionEscape (struct RequestStruct *rqptr)

{
   register boolean  Changed;
   register char  *cptr, *sptr, *zptr;
   register struct DescrTaskStruct  *tkptr;

   char  Buffer [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DescriptionEscape()\n");

   /* retrieve the pointer to the task structure */
   tkptr = rqptr->DescrTaskPtr;

   if (Debug) fprintf (stdout, "|%s|\n", tkptr->DescriptionBufferPtr);

   Changed = false;
   if (tkptr->SizeOfDescriptionBuffer > sizeof(Buffer))
      zptr = (sptr = Buffer) + tkptr->SizeOfDescriptionBuffer - 1;
   else
      zptr = (sptr = Buffer) + sizeof(Buffer)-1;
   for (cptr = tkptr->DescriptionBufferPtr; *cptr && sptr+6 < zptr; cptr++)
   {
      switch (*cptr)
      {
         case '<' :
            memcpy (sptr, "&lt;", 4); sptr += 4; Changed = true; break;
         case '>' :
            memcpy (sptr, "&gt;", 4); sptr += 4; Changed = true; break;
         case '&' :
            memcpy (sptr, "&amp;", 5); sptr += 5; Changed = true; break;
         case '\"' :
            *sptr++ = '\''; Changed = true; break; 
         default:
            if (isprint(*cptr)) *sptr++ = *cptr;
      } 
   }
   *sptr++ = '\0';
   if (Changed) memcpy (tkptr->DescriptionBufferPtr, Buffer, sptr - Buffer);
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->DescriptionBufferPtr);
}

/*****************************************************************************/
/*
Put the hexadecimal VMS status value into the description buffer as an error
indication.
*/

DescriptionError
(
struct RequestStruct *rqptr,
int StatusValue,
int SourceLineNumber
)
{
   static  $DESCRIPTOR (ErrorFaoDsc,
           "<FONT COLOR=\"#ff0000\">[Error %X!8XL]</FONT><!!-- !UL -->\0");
   static  $DESCRIPTOR (BufferDsc, "");

   register struct DescrTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DescriptionError() %%X%08.08X\n", StatusValue);

   tkptr = rqptr->DescrTaskPtr;

   BufferDsc.dsc$a_pointer = tkptr->DescriptionBufferPtr;
   BufferDsc.dsc$w_length = tkptr->SizeOfDescriptionBuffer;
   sys$fao (&ErrorFaoDsc, 0, &BufferDsc, StatusValue, SourceLineNumber);
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->DescriptionBufferPtr);
}

/*****************************************************************************/

