/*****************************************************************************/
/*
                                  Digest.c

This module implements the DIGEST authentication functionality.  The actual 
RSA MD5 digest code is contained in its own module MD5.c, and is almost 
exclusively the example implementation code presented in RFC1321.  This module
implements the basic digest access authentication as proposed in the HTTP 
Working Group's, IETF draft, dated June 6th 1996.  Functions herein are called
from the Auth.c module.

The nonce is generated using three 8 digit, zero-filled, hexadecimal numbers.
The client's 32 bit IP address, and each of the least and most significant 
longwords of the request's binary time.  The nonce therefore comprises a 
string of 24 hexadecimal digits.  It does not rely on any non-public value 
or format for its operation.  

This format allows the returned nonce to have the IP address and timestamp 
easily extracted and checked.  If the client's IP address and the nonce's 
IP address do not match then it is rejected.  If the timestamp indicates 
an unacceptable period between generation and use of the nonce has occured 
then it is rejected with a 'stale nonce' flag.  The period is varied for
different methods.  Very much shorter for PUT and POST methods, etc., longer 
for GET, etc.

WARNING!  To date (09-JUL-96) this functionality has only been tested with
NCSA X Mosaic 2.7-4 .. This version Mosaic seems a bit flakey with digest
authentication.

NCSA X Mosaic 2.7-4 seems to require the 'opaque' header field to generate a
correct digest response.  I thought it was optional, but I'll put in a dummy
anyway.

30-OCT-97  MGD  a little maintenance
01-JUL-96  MGD  initial development of "Digest" authentication
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "config.h"
#include "digest.h"
#include "error.h"
#include "httpd.h"
#include "msg.h"
#include "vm.h"

#define DefaultNonceLifeTimeSeconds 60

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char SoftwareID[];
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/**********************/
/* external functions */
/**********************/

Md5String (char*, int, unsigned char*);
Md5HexString (char*, int, unsigned char*);

/*****************************************************************************/
/*
*/

DigestHexString
(
register unsigned char *cptr,
register int cnt,
register unsigned char *sptr
)
{
   static char  HexDigits [] = "0123456789abcdef";

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DigestHexString()\n");

   while (cnt--)
   {
      *sptr++ = HexDigits[(*cptr & 0xf0) >> 4];
      *sptr++ = HexDigits[*cptr++ & 0x0f];
   }
   *sptr = '\0';
}

/*****************************************************************************/
/*
Create an MD5 16 byte digest (RFC 1321 and RFC????) of the username, realm and
password.
*/

int DigestA1
(
char *UserName,
char *Realm,
char *Password,
unsigned char *A1Digest
)
{
   static $DESCRIPTOR (A1FaoDsc, "!AZ:!AZ:!AZ");
   static $DESCRIPTOR (StringDsc, "");

   int  status;
   char  A1String [128];
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DigestA1() |%s|%s|%s|\n", UserName, Realm, Password);

   StringDsc.dsc$a_pointer = A1String;
   StringDsc.dsc$w_length = sizeof(A1String)-1;
   if (VMSnok (status =
       sys$fao (&A1FaoDsc, &Length, &StringDsc,
                UserName, Realm, Password)))
   {
      memset (A1Digest, 0, 16);
      return (status);
   }
   Md5String (A1String, Length, A1Digest);
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Create an MD5 hex digest (RFC 1321 and RFC????) of the username, realm and
password for use in HTTP Digest authentication.
*/

int DigestA1Hex
(
char *UserName,
char *Realm,
char *Password,
char *A1HexDigest
)
{
   static $DESCRIPTOR (A1FaoDsc, "!AZ:!AZ:!AZ");
   static $DESCRIPTOR (StringDsc, "");

   int  status;
   char  A1String [128];
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DigestA1() |%s|%s|%s|\n", UserName, Realm, Password);

   StringDsc.dsc$a_pointer = A1String;
   StringDsc.dsc$w_length = sizeof(A1String)-1;
   if (VMSnok (status =
       sys$fao (&A1FaoDsc, &Length, &StringDsc,
                UserName, Realm, Password)))
   {
      A1HexDigest[0] = '\0';
      return (status);
   }
   Md5HexString (A1String, Length, A1HexDigest);
   if (Debug) fprintf (stdout, "A1HexDigest |%s|\n", A1HexDigest);
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

int DigestResponse
(
unsigned char *A1HexDigest,
char *Nonce,
char *Method,
char *Uri,
char *ResponseHexDigest
)
{
   static $DESCRIPTOR (A2FaoDsc, "!AZ:!AZ");
   static $DESCRIPTOR (ResponseFaoDsc, "!AZ:!AZ:!AZ");
   static $DESCRIPTOR (StringDsc, "");

   int  status;
   char  A2HexDigest [33],
         A2String [256],
         ResponseString [256];
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DigestResponse() |%s|%s|%s|%s|\n",
               A1HexDigest, Nonce, Method, Uri);

   ResponseHexDigest[0] = '\0';

   StringDsc.dsc$a_pointer = A2String;
   StringDsc.dsc$w_length = sizeof(A2String)-1;
   if (VMSnok (status =
       sys$fao (&A2FaoDsc, &Length, &StringDsc,
                Method, Uri)))
      return (status);
   Md5HexString (A2String, Length, A2HexDigest);

   StringDsc.dsc$a_pointer = ResponseString;
   StringDsc.dsc$w_length = sizeof(ResponseString)-1;
   if (VMSnok (status =
       sys$fao (&ResponseFaoDsc, &Length, &StringDsc,
                A1HexDigest, Nonce, A2HexDigest)))
      return (status);

   Md5HexString (ResponseString, Length, ResponseHexDigest);

   if (Debug) fprintf (stdout, "ResponseHexDigest |%s|\n", ResponseHexDigest);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Generate a basic DIGEST challenge string, comprising the authentication realm
and the nonce.  Store this null-terminated string in dynamically allocated 
memory pointed to by 'rqptr->AuthDigestChallengePtr'.  The 'Other' allows
other digest fields to be included, for example ", stale=true".

NCSA Mosaic 2.7-4 seems to require the 'opaque' header field to generate a
correct digest response.  I thought it was optional, but I'll put in a dummy
anyway.
*/ 
 
int DigestChallenge
(
struct RequestStruct *rqptr,
char *Other
)
{
   static $DESCRIPTOR (StringDsc, "");
   static $DESCRIPTOR (DigestChallengeFaoDsc,
          "WWW-Authenticate: Digest realm=\"!AZ\", nonce=\"!AZ\"!AZ!AZ\r\n");
   static $DESCRIPTOR (NonceFaoDsc, "!8XL!8XL!8XL");

   int  status;
   unsigned short  Length;
   char  Nonce [25],
         String [1024];
   char  *OpaquePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DigestChallenge()\n");

   /* NCSA Mosaic 2.7-4 seems to "require" an opaque parameter */
   if (rqptr->HttpUserAgentPtr != NULL &&
       strstr (rqptr->HttpUserAgentPtr, "Mosaic") != NULL)
      OpaquePtr = ", opaque=\"*\"";
   else
      OpaquePtr = "";

   StringDsc.dsc$a_pointer = Nonce;
   StringDsc.dsc$w_length = sizeof(Nonce)-1;

   sys$fao (&NonceFaoDsc, &Length, &StringDsc,
            rqptr->ClientInternetAddress32bit,
            rqptr->BinaryTime[0], rqptr->BinaryTime[1]);
   Nonce[Length] = '\0';
   if (Debug) fprintf (stdout, "Nonce |%s|\n", Nonce);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   sys$fao (&DigestChallengeFaoDsc, &Length, &StringDsc,
            rqptr->AuthRealmPtr, Nonce, OpaquePtr, Other);
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory */
   rqptr->AuthDigestChallengePtr = VmGetHeap (rqptr, Length+1);
   memcpy (rqptr->AuthDigestChallengePtr, String, Length+1);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Parse the "Digest" authorization HTTP header field.  Extract and store the
digest response, nonce, realm, username, and digest URI.
*/ 
 
int DigestAuthorization (struct RequestStruct *rqptr)

{
   register char  *cptr, *lptr, *sptr, *zptr;

   int  status,
        AlgorithmSize,
        NonceSize,
        OpaqueSize,
        RealmSize,
        ResponseSize,
        UriSize,
        UserNameSize;
   int  *SizePtr;      
   char  Algorithm [16],
         Nonce [25],
         Opaque [33],
         Realm [16],
         Response [33],
         Uri [256],
         UserName [16];
   char  *ResourcePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) 
      fprintf (stdout, "DigestAuthorization() |%s|\n",
               rqptr->HttpAuthorizationPtr);

   rqptr->AuthScheme = rqptr->AuthChallengeScheme = AUTH_SCHEME_DIGEST;

   cptr = rqptr->HttpAuthorizationPtr;
   /* skip across the authorization scheme name ("digest") */
   while (*cptr && !ISLWS(*cptr)) cptr++;
   /* skip over white-space between scheme and digest parameters */
   while (*cptr && ISLWS(*cptr)) cptr++;

   /****************/
   /* parse values */
   /****************/

   Algorithm[0] = Nonce[0] = Opaque[0] = Realm[0] =
      Response[0] = Uri[0] = UserName[0] = '\0';

   while (*cptr)
   {
      while (ISLWS(*cptr) || *cptr == ',') cptr++;
      /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
      if (toupper(*cptr) == 'A' && strsame (cptr, "algorithm=", 10))
      {   
         cptr += 10;
         zptr = (lptr = sptr = Algorithm) + sizeof(Algorithm);
         SizePtr = &AlgorithmSize;
      }
      else   
      if (toupper(*cptr) == 'N' && strsame (cptr, "nonce=", 6))
      {   
         cptr += 6;
         zptr = (lptr = sptr = Nonce) + sizeof(Nonce);
         SizePtr = &NonceSize;
      }
      else   
      if (toupper(*cptr) == 'O' && strsame (cptr, "opaque=", 7))
      {   
         cptr += 7;
         zptr = (lptr = sptr = Opaque) + sizeof(Opaque);
         SizePtr = &OpaqueSize;
      }
      else   
      if (toupper(*cptr) == 'R' && strsame (cptr, "realm=", 6))
      {   
         cptr += 6;
         zptr = (lptr = sptr = Realm) + sizeof(Realm);
         SizePtr = &RealmSize;
      }
      else   
      if (toupper(*cptr) == 'R' && strsame (cptr, "response=", 9))
      {   
         cptr += 9;
         zptr = (lptr = sptr = Response) + sizeof(Response);
         SizePtr = &ResponseSize;
      }
      else   
      if (toupper(*cptr) == 'U' && strsame (cptr, "uri=", 4))
      {   
         cptr += 4;
         zptr = (lptr = sptr = Uri) + sizeof(Uri);
         SizePtr = &UriSize;
      }
      else   
      if (toupper(*cptr) == 'U' && strsame (cptr, "username=", 9))
      {   
         cptr += 9;
         zptr = (lptr = sptr = UserName) + sizeof(UserName);
         SizePtr = &UserNameSize;
      }
      else   
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_FORMAT), FI_LI);
         return (STS$K_ERROR);
      }
      
      if (*cptr == '\"')
      {   
         cptr++;
         while (*cptr && *cptr != '\"' && sptr < zptr) *sptr++ = *cptr++;
      }
      if (sptr >= zptr || !*cptr)
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_FORMAT), FI_LI);
         return (STS$K_ERROR);
      }
      *sptr++ = '\0';
      /* note that the size INCLUDES the terminating null */
      *SizePtr = sptr - lptr;
      cptr++;
   }

   if (Debug)
      fprintf (stdout,
"nonce |%s|\nopaque |%s|\nrealm |%s|\nresponse |%s|\n\
username |%s|\nuri |%s|\n",
      Nonce, Opaque, Realm, Response, UserName, Uri);

   /****************/
   /* check values */
   /****************/

   /* NCSA Mosaic 2.7-4 seems to lop off the leading slash */
   if (Uri[0] == '/')
      ResourcePtr = rqptr->ResourcePtr;
   else
   if (rqptr->ResourcePtr[0])
      ResourcePtr = rqptr->ResourcePtr+1;
   else
      ResourcePtr = rqptr->ResourcePtr;

   if ((Algorithm[0] && !strsame (Algorithm, "MD5", -1)) ||
       !Nonce[0] ||
       !Realm[0] ||
       !Response[0] ||
       !Uri[0] ||
       !UserName[0] ||
       strcmp (ResourcePtr, Uri) ||
       strcmp (rqptr->AuthRealmPtr, Realm) ||
       strlen(Nonce) != 24)
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_FAILED), FI_LI);
      return (STS$K_ERROR);
   }

   if (VMSnok (status = DigestCheckNonce (rqptr, Nonce)))
      return (status);

   /****************/
   /* store values */
   /****************/

   rqptr->AuthDigestNoncePtr = VmGetHeap (rqptr, NonceSize);
   memcpy (rqptr->AuthDigestNoncePtr, Nonce, NonceSize);

   rqptr->AuthRealmPtr = VmGetHeap (rqptr, RealmSize);
   memcpy (rqptr->AuthRealmPtr, Realm, RealmSize);

   rqptr->AuthDigestResponsePtr = VmGetHeap (rqptr, ResponseSize);
   memcpy (rqptr->AuthDigestResponsePtr, Response, ResponseSize);

   rqptr->AuthDigestUriPtr = VmGetHeap (rqptr, UriSize);
   memcpy (rqptr->AuthDigestUriPtr, Uri, UriSize);

   if (UserNameSize > sizeof(rqptr->RemoteUser))
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneralOverflow (rqptr, FI_LI);
      return (STS$K_ERROR);
   }
   /* user names are always upper case for us! */
   sptr = rqptr->RemoteUser;
   for (cptr = UserName; *cptr; *sptr++ = toupper(*cptr++));
   *sptr = '\0';

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
The nonce comprises three 8 digit, zero-filled, hexadecimal numbers. The
client's 32 bit IP address, and each of the least and most significant 
longwords of the request's binary time.  The nonce is therefore a string of 24
hexadecimal digits.

This format allows the returned nonce to have the IP address and timestamp 
easily extracted and checked.  If the client's IP address and the nonce's 
IP address do not match then it is rejected.  If the timestamp indicates 
an unacceptable period between generation and use of the nonce has occured 
then it is rejected with a 'stale nonce' flag.
*/ 
 
int DigestCheckNonce
(
struct RequestStruct *rqptr,
char *Nonce
)
{
   static unsigned long  LibSecondOfYear = LIB$K_SECOND_OF_YEAR;

   int  status;
   unsigned long  ClientInternetAddress32bit,
                  NonceLifeTimeSeconds,
                  NonceSecondOfYear,
                  SecondOfYear;
   unsigned long  NonceBinaryTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DigestCheckNonce()\n");

   /* lib$cvt_htb() returns a 1 for OK or a 0 for conversion error */
   if (status = lib$cvt_htb (8, Nonce, &ClientInternetAddress32bit))
      if (status = lib$cvt_htb (8, Nonce+8, &NonceBinaryTime[0]))
         status = lib$cvt_htb (8, Nonce+16, &NonceBinaryTime[1]);

   if (status != 1)
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_FAILED), FI_LI);
      return (STS$K_ERROR);
   }

   if (Debug)
   {
      char  NonceTime [32];
      $DESCRIPTOR (NonceTimeFaoDsc, "!%D\0");
      $DESCRIPTOR (NonceTimeDsc, NonceTime);

      sys$fao (&NonceTimeFaoDsc, 0, &NonceTimeDsc, &NonceBinaryTime);
      fprintf (stdout, "32bit %08.08X (%08.08X)\nnonce time |%s|\n",
               ClientInternetAddress32bit,
               rqptr->ClientInternetAddress32bit,
               NonceTime);
   }

   if (ClientInternetAddress32bit != rqptr->ClientInternetAddress32bit)
   {
      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_FAILED), FI_LI);
      return (STS$K_ERROR);
   }

   lib$cvt_from_internal_time (&LibSecondOfYear, &SecondOfYear,
                               &rqptr->BinaryTime);

   lib$cvt_from_internal_time (&LibSecondOfYear, &NonceSecondOfYear,
                               &NonceBinaryTime);

   if ((rqptr->HttpMethod & HTTP_METHOD_GET) ||
       (rqptr->HttpMethod & HTTP_METHOD_HEAD))
      NonceLifeTimeSeconds = Config.AuthDigestNonceGetLifeTime;
   else
      NonceLifeTimeSeconds = Config.AuthDigestNoncePutLifeTime;
   if (!NonceLifeTimeSeconds)
      NonceLifeTimeSeconds = DefaultNonceLifeTimeSeconds;

   if (Debug)
      fprintf (stdout, "SecondOfYear: %d Nonce: %d LifeTime: %d Stale: %d\n",
               SecondOfYear, NonceSecondOfYear, NonceLifeTimeSeconds,
               (NonceSecondOfYear+NonceLifeTimeSeconds<SecondOfYear));

   if (NonceSecondOfYear + NonceLifeTimeSeconds < SecondOfYear)
   {
      /*
          Nonce is stale.
          Now I know this test will break once a year :^(
          (midnight, New Year's Eve) but too bad, it's simple :^)
      */

      /* generate a special "stale nonce" challenge */
      rqptr->AuthScheme = rqptr->AuthChallengeScheme = AUTH_SCHEME_DIGEST;

      DigestChallenge (rqptr, ", stale=true");

      rqptr->ResponseStatusCode = 401;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_FAILED), FI_LI);
      return (STS$K_ERROR);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/

