/*****************************************************************************/
/*
                                  Dir.c

This module implements a full multi-threaded, AST-driven, asynchronous 
directory listing.  The AST-driven nature makes the code a little more 
difficult to follow, but creates a powerful, event-driven, multi-threaded 
server.  All of the necessary functions implementing this module are designed 
to be non-blocking. 

This module never returns a valid status and ALWAYS calls the supplied next
task (AST) function.  This should check for a generated error message to
determine is there were any problems.

This module uses the NetWriteBuffered() function to buffer any output it can
into larger chunks before sending it to the client.

A directory listing is recognised by the server whenever a wildcard is present 
in a specification and there is no query string directing another activity 
(e.g. a query/search).  Compliant with other HTTPD implementations, a 
directory listing is also generated if a URL specifies a directory only and 
the directory contains no home page. 

The directory listing is designed to look very much like the default layout of 
the CERN and NCSA HTTPD implementations, with the exception that all 
directories are grouped at the top, not dispersed throughout the file names.  
This looks and functions better than the above (at least in the experience of 
the author). 

Access Control
--------------

If the server configuration specifies "DirAccessSelective" then the directory
must contain the file ".WWW_BROWSABLE" to be listed by this module.

If a directory contains the file ".WWW_HIDDEN" it cannot be listed with this
module, but individual files are accessable for server access.  If a
subdirectory contains this file it does not appear in a listing.

If a directory contains the file ".WWW_NOWILD" a wildcard directory
specification will not work even if allowed by server configuration.

If a directory contains the file ".WWW_NOP" it does not show a parent.
If a directory contains the file ".WWW_NOS" it does not show subdirectories.
If a directory contains the file ".WWW_NOPS" both the above apply.

Listing Layout
--------------

The layout of the listing can be controlled from the configuration file or via
a server directive.  A decimal number may optionally, immediately precede any
directive, this becomes the width of that particular field.  If a width is not
specified an appropriate default is used.  Information wider than the field
width is truncated, generally without indication.  The following layout
directives provide:

  _     (underscore) each underscore provides one space between fields
  C     creation date
  D     content-type description (BEST be the last field specified)
  D:L   content-type description as a link
  I     icon
  L     file anchor (link, including hot-spot name)
  N     file name (without anchor, why I can't imagine :^)
  O     file owner (only when SYSUAF-authenticated and profile accessed)
  P     file protection (only when SYSUAF-authenticated and profile accessed)
  R     revision date
  S     file size
  S:B   file size to be in comma-formatted bytes
  S:D   express file sizes in 1000 (decimal) kilos not 1024 (binary) kilos
  S:F   file size Mb and Kb to be expressed to one significant place
  S:K   file size to be in K-bytes
  S:M   file size to be in M-bytes
  U     file/directory name in upper case (MUST be the FIRST directive)

These may placed in any order ("D" is best last because it does not use a 
field width) and with any (reasonable) field-width.  For example see 
'DefaultDirLayout' below. 

Query String Directives
-----------------------

  autoscript=   yes (default), no, true, false, 1, 0
  delimit=      header, footer, both (default), none
  expired=      yes, no, true, false, 1, 0 (listing pre-expired)
  layout=       see "listing layout" immediately above
  nop=          yes, no (default), true, false, 1, 0 (as if .WWW_NOP found)
  nops=         yes, no (default), true, false, 1, 0 (as if .WWW_NOPS found)
  nos=          yes, no (default), true, false, 1, 0 (as if .WWW_NOS found)
  readme=       yes, no (default), true, false, 1, 0


VERSION HISTORY
---------------
08-AUG-98  MGD  configurable implied wildcards
19-JUL-98  MGD  bugfix; MapUrl_Map() pass 'rqptr' for conditionals to work
14-MAY-98  MGD  request-specified content-type ("httpd=index&type=")
16-MAR-98  MGD  bugfix; file bytes incorrect when 'FirstFreeByte' zero
28-FEB-98  MGD  improved icon handling efficiency
19-FEB-98  MGD  size layout allows ":B", ":D", ":F", ":K", ":M" modifiers,
                description layout allows ":L" to make a link out of it,
                description may now have an associated field width,
                allow for directory paths like "/dir1/dir2"
02-NOV-97  MGD  "delimit=", "nop=", "nops=", "nos=" and "readme=" directives
                changed file sizes back from 1000 to 1024 kilos, megas, etc.
17-AUG-97  MGD  message database, internationalized file date/times,
                file/directory names moved to lower-case (see 'U' above),
                SYSUAF-authenticated users security-profile
20-APR-97  MGD  changed file sizes from 1024 to 1000 kilos, megas, etc.
01-FEB-97  MGD  HTTPd version 4
24-APR-96  MGD  chagrin ... just realized I don't need full URLs in HREFs
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  major revision of file and directory anchor generation
                (far fewer MapUrl()s yielding greater efficiency);
                added query string capability;
                added automatic script capability
07-AUG-95  MGD  include VMS-style directory layout and file sizes
16-JUN-95  MGD  file contents description for non-HTML files (see config.c)
25-MAR-95  MGD  bugfix; error handling with DirReadMe() and DirFileExists()
20-DEC-94  MGD  multi-threaded daemon (it suddenly got very complex!)
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <libdtdef.h>
#include <libdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header file */
#include "wasd.h"
#include "config.h"
#include "descr.h"
#include "dir.h"
#include "error.h"
#include "file.h"
#include "httpd.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "vm.h"

/**********/
/* macros */
/**********/

#define DEFAULT_FIELD_WIDTH_INTERFIELD 1
#define DEFAULT_FIELD_WIDTH_CDT 15
#define DEFAULT_FIELD_WIDTH_NAME 25
#define DEFAULT_FIELD_WIDTH_OWNER 20
#define DEFAULT_FIELD_WIDTH_PROTECTION 19
#define DEFAULT_FIELD_WIDTH_RDT 15
#define DEFAULT_FIELD_WIDTH_SIZE 6
#define DEFAULT_FIELD_WIDTH_DECIMAL 11

#define DELIMIT_BOTH    1
#define DELIMIT_HEADER  2
#define DELIMIT_FOOTER  3
#define DELIMIT_NONE    4

#define LAYOUT_PARAMETER ':'

/******************/
/* global storage */
/******************/

char  DirBrowsableFileName [] = ".WWW_BROWSABLE",
      DirHiddenFileName [] = ".WWW_HIDDEN",
      DirNoWildFileName [] = ".WWW_NOWILD",
      DirNopFileName [] = ".WWW_NOP",
      DirNosFileName [] = ".WWW_NOS",
      DirNopsFileName [] = ".WWW_NOPS";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  NaturalLanguageEnglish;
extern int  FileBufferSize;
char  *DirBlankIconPtr,
      *DirDirIconPtr,
      *DirParentIconPtr;

extern char  *ConfigBlankIconPtr,
             *ConfigDirIconPtr,
             *ConfigParentIconPtr;
extern char  ConfigContentTypeSsi[];
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
This function never returns a valid status and ALWAYS calls the supplied next
task (AST) function.  This should check for a generated error message to
determine is there were any problems.
*/ 
 
DirBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *DirSpec,
char *DirQueryString,
char *RelativePath
)
{
   static $DESCRIPTOR (IndexOfFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<META NAME=\"kilo\" CONTENT=\"!UL\">\n\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>!AZ <TT>!AZ</TT></NOBR></H1>\n");

   register char  *cptr, *sptr, *zptr;
   register unsigned long  *vecptr;
   register struct DirTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [8];
   char  Scratch [256],
         String [1024];
   void  *AstFunctionPtr;
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirBegin() %d |%s|%s|%s|\n",
               rqptr->ErrorMessagePtr, DirSpec, DirQueryString, RelativePath);

   /* network writes are checked for success, fudge the first one! */
   rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoDirectoryCount;

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (!Config.DirAccess)
   {
      Accounting.RequestForbiddenCount++;
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_DISABLED), FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (possibly multiple concurrent) */
   if (rqptr->DirTaskPtr != NULL &&
       LIST_HAS_NEXT (rqptr->DirTaskPtr))
   {
      rqptr->DirTaskPtr = tkptr = LIST_NEXT (rqptr->DirTaskPtr);
      memset (LIST_DATA(tkptr), 0,
              sizeof(struct DirTaskStruct) - sizeof(struct ListEntryStruct));
   }
   else
   {
      rqptr->DirTaskPtr = tkptr =
         VmGetHeap (rqptr, sizeof(struct DirTaskStruct));
      ListAddTail (&rqptr->DirTaskList, tkptr);
   }
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = MapVmsPath (DirSpec);
   if (!cptr[0])
   {
      ErrorGeneral (rqptr, cptr+1, FI_LI);
      DirEnd (rqptr);
      return;
   }
   zptr = (sptr = tkptr->DirPath) + sizeof(tkptr->DirPath);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   *sptr = '\0';

   cptr = DirSpec;
   zptr = (sptr = tkptr->DirSpec) + sizeof(tkptr->DirSpec);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->DirSpecLength = sptr - tkptr->DirSpec;

   /* only when specified as "/dir1/dir2" this contains "./dir2/" */
   cptr = RelativePath;
   zptr = (sptr = tkptr->RelativePath) + sizeof(tkptr->RelativePath);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   *sptr = '\0';

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   cptr = DirSpec;

   /* first get just the directory part */
   zptr = (sptr = tkptr->DirSpec) + sizeof(tkptr->DirSpec);
   while (*cptr && *cptr != ']' && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   if (*cptr == ']' && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';
   tkptr->DirSpecLength = sptr - tkptr->DirSpec;

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->DirSpec,
                                       tkptr->DirSpecLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         DirEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /* get any file component */
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->DirSpecLength = sptr - tkptr->DirSpec;

   /********************************************/
   /* check if there are any server directives */
   /********************************************/

   cptr = DirQueryString;
   zptr = (sptr = tkptr->QueryString) + sizeof(tkptr->QueryString);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      DirEnd (rqptr);
      return;
   }
   *sptr = '\0';

   if (Debug) fprintf (stdout, "tkptr->QueryString |%s|\n", tkptr->QueryString);

   rqptr->ResponsePreExpired = Config.DirPreExpired;
   tkptr->AsIfNopFound = tkptr->AsIfNosFound = false;
   tkptr->AutoScriptEnabled = tkptr->IncludeAnyReadme = true;
   tkptr->Delimit = DELIMIT_BOTH;

   if (tkptr->QueryString[0])
   {
      cptr = tkptr->QueryString;
      while (*cptr)
      {
         if (Debug) fprintf (stdout, "cptr |%s|\n", cptr);

         if (tolower(*cptr) == 'a' &&
             strsame (cptr, "autoscript=", 11))
         {
            cptr += 11;
            /* if "false", "no" or "0" then turn off auto-scripting */
            if (tolower(*cptr) == 'f' || tolower(*cptr) == 'n' || *cptr == '0')
               tkptr->AutoScriptEnabled = false;
            else
               tkptr->AutoScriptEnabled = true;
         }
         else
         if (tolower(*cptr) == 'd' &&
             strsame (cptr, "delimit=", 8))
         {
            cptr += 8;
            switch (tolower(*cptr))
            {
               case 'h' :
                  tkptr->Delimit = DELIMIT_HEADER;
                  break;
               case 'f' :
                  tkptr->Delimit = DELIMIT_FOOTER;
                  break;
               case 'n' :
                  tkptr->Delimit = DELIMIT_NONE;
                  break;
               default :
                  tkptr->Delimit = DELIMIT_BOTH;
            }
         }
         else
         /* experience shows both make it easier! */
         if (tolower(*cptr) == 'e' &&
             (strsame (cptr, "expire=", 7) ||
              strsame (cptr, "expired=", 8)))
         {
            if (cptr[7] == '=')
               cptr += 8;
            else
               cptr += 7;
            /* "true", "yes", "1", "false", "no" or "0" */
            if (tolower(*cptr) == 't' || tolower(*cptr) == 'y' || *cptr == '1')
               rqptr->ResponsePreExpired = true;
            else
            if (tolower(*cptr) == 'f' || tolower(*cptr) == 'n' || *cptr == '0')
               rqptr->ResponsePreExpired = false;
         }
         else
         if (tolower(*cptr) == 'l' &&
             strsame (cptr, "layout=", 7))
         {
            cptr += 7;
            for (sptr = cptr; *sptr && *sptr != '&'; sptr++);
            tkptr->LayoutPtr = VmGetHeap (rqptr, sptr-cptr+1);
            memcpy (tkptr->LayoutPtr, cptr, sptr-cptr);
            tkptr->LayoutPtr[sptr-cptr] = '\0';
         }
         else
         if (tolower(*cptr) == 'n' &&
             strsame (cptr, "nop=", 4))
         {
            cptr += 4;
            /* if "true", "yes" or "1" then do not display parent directory */
            if (tolower(*cptr) == 't' || tolower(*cptr) == 'y' || *cptr == '1')
               tkptr->AsIfNopFound = true;
            else
               tkptr->AsIfNopFound = false;
         }
         else
         if (tolower(*cptr) == 'n' &&
             strsame (cptr, "nops=", 5))
         {
            cptr += 5;
            /* if "true", "yes" or "1" don't display parent or sub directory */
            if (tolower(*cptr) == 't' || tolower(*cptr) == 'y' || *cptr == '1')
               tkptr->AsIfNopFound = tkptr->AsIfNosFound = true;
            else
               tkptr->AsIfNopFound = tkptr->AsIfNosFound = false;
         }
         else
         if (tolower(*cptr) == 'n' &&
             strsame (cptr, "nos=", 4))
         {
            cptr += 4;
            /* if "true", "yes" or "1" then do not display sub directory */
            if (tolower(*cptr) == 't' || tolower(*cptr) == 'y' || *cptr == '1')
               tkptr->AsIfNosFound = true;
            else
               tkptr->AsIfNosFound = false;
         }
         else
         if (tolower(*cptr) == 'r' &&
             strsame (cptr, "readme=", 7))
         {
            cptr += 7;
            /* if "false", "no" or "0" then do not display any readme */
            if (tolower(*cptr) == 'f' || tolower(*cptr) == 'n' || *cptr == '0')
               tkptr->IncludeAnyReadme = false;
            else
               tkptr->IncludeAnyReadme = true;
         }
         else
         if (tolower(*cptr) == 's' &&
             strsame (cptr, "script=", 7))
         {
            /* local storage */
            register char  *zptr;

            cptr += 7;
            zptr = (sptr = tkptr->ScriptName) + sizeof(tkptr->ScriptName);
            if (*cptr != '/') *sptr++ = '/';
            while (*cptr && *cptr != '&' && sptr < zptr) *sptr++ = *cptr++;
            if (sptr >= zptr)
            {
               ErrorGeneralOverflow (rqptr, FI_LI);
               DirEnd (rqptr);
               return;
            }
            *sptr = '\0';
         }
         else
         if (tolower(*cptr) == 't' &&
             strsame (cptr, "type=", 5))
         {
            cptr += 5;
            for (sptr = cptr; *sptr && *sptr != '&'; sptr++);
            tkptr->QueryContentTypePtr = VmGetHeap (rqptr, sptr-cptr+1);
            memcpy (tkptr->QueryContentTypePtr, cptr, sptr-cptr);
            tkptr->QueryContentTypePtr[sptr-cptr] = '\0';
         }

         while (*cptr && *cptr != '&') cptr++;
         if (*cptr) cptr++;
      }
   }

   /*************************************/
   /* parse and get required components */
   /*************************************/

   tkptr->ParseInUse = true;

   tkptr->SearchFab = cc$rms_fab;
   tkptr->SearchFab.fab$l_fna = tkptr->DirSpec;
   tkptr->SearchFab.fab$b_fns = tkptr->DirSpecLength;
   tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

   tkptr->SearchNam = cc$rms_nam;
   tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&tkptr->SearchFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = tkptr->DirPath;
      rqptr->ErrorHiddenTextPtr = tkptr->DirSpec;
      ErrorVmsStatus (rqptr, status, FI_LI);
      DirEnd (rqptr);
      return;
   }

   if (tkptr->SearchNam.nam$l_fnb & NAM$M_WILD_DIR)
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_NO_WILDCARD), FI_LI);
      DirEnd (rqptr);
      return;
   }

   memcpy (tkptr->DirectoryPart,
           tkptr->SearchNam.nam$l_dev,
           Length = tkptr->SearchNam.nam$l_name - tkptr->SearchNam.nam$l_dev);
   tkptr->DirectoryPart[Length] = '\0';

   Scratch[0] = '\0';
   sptr = MapUrl_Map (Scratch, tkptr->DirectoryPart, NULL, NULL, rqptr);
   if (!sptr[0] && sptr[1])
   {
      /* mapping report */
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, sptr+1, FI_LI);
      DirEnd (rqptr);
      return;
   }
   Length = strlen(Scratch) + 1;
   tkptr->DirectoryPathPtr = VmGetHeap (rqptr, Length);
   memcpy (tkptr->DirectoryPathPtr, Scratch, Length);

   if (Debug)
      fprintf (stdout,
      "DirectoryPart |%s|\nFilePart |%s|\nDirDirectoryUrl |%s|\n",
      tkptr->DirectoryPart, tkptr->FilePart,
      tkptr->DirectoryPathPtr);

   /************************/
   /* directory access ok? */
   /************************/

   /* ensure we can read these directory listing "control" files */
   EnableSysPrv ();

   if (Config.DirAccessSelective)
      status = FileExists (tkptr->DirectoryPart, DirBrowsableFileName);
   else
   {
      if (VMSok (status = FileExists (tkptr->DirectoryPart, DirHiddenFileName)))
         status = RMS$_FNF;
      else
      if (VMSok (status = FileExists (tkptr->DirectoryPart, DirNoWildFileName)))
      {
         if (tkptr->SearchNam.nam$l_fnb & NAM$M_WILD_NAME ||
             tkptr->SearchNam.nam$l_fnb & NAM$M_WILD_TYPE ||
             tkptr->SearchNam.nam$l_fnb & NAM$M_WILD_VER)
            status = SS$_ABORT;
         else
            status = SS$_NORMAL;
      }
      else
         status = SS$_NORMAL;
   }

   DisableSysPrv ();

   if (VMSnok (status))
   {
      if (status == SS$_ABORT)
      {
         /* abort here means the facility is disabled */
         Accounting.RequestForbiddenCount++;
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_DISABLED), FI_LI);
      }
      else
      {
         /* give some "disinformation"  ;^)  */
         if (status == RMS$_FNF) status = RMS$_DNF;
         rqptr->ErrorTextPtr = tkptr->DirPath;
         rqptr->ErrorHiddenTextPtr = tkptr->DirSpec;
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
      DirEnd (rqptr);
      return;
   }

   /*********************************/
   /* further process specification */
   /*********************************/

   tkptr->DirSpecIncludedFilePart = tkptr->SearchNam.nam$b_name > 0 ||
                                    tkptr->SearchNam.nam$b_type > 1 ||
                                    tkptr->SearchNam.nam$b_ver > 1;

   if (!tkptr->DirSpecIncludedFilePart)
   {     
      /* no name or type was supplied with the request, wildcard it */
      tkptr->SearchFab.fab$l_dna = "*.*";
      tkptr->SearchFab.fab$b_dns = 3;

      if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

      status = sys$parse (&tkptr->SearchFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);

      if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = tkptr->DirPath;
         rqptr->ErrorHiddenTextPtr = tkptr->DirSpec;
         ErrorVmsStatus (rqptr, status, FI_LI);
         DirEnd (rqptr);
         return;
      }
   }

   if (tkptr->SearchNam.nam$l_fnb & NAM$M_WILD_VER ||
       tkptr->SearchNam.nam$l_fnb & NAM$M_EXP_VER)
   {
      /* wildcard or explicit version number supplied ... VMS format */
      memcpy (tkptr->FilePart,
              tkptr->SearchNam.nam$l_name,
              Length = tkptr->SearchNam.nam$l_ver -
                       tkptr->SearchNam.nam$l_name +
                       tkptr->SearchNam.nam$b_ver);
      tkptr->FormatLikeVms = true;
   }
   else
   {
      /* name and/or type or implied wildcards */
      memcpy (tkptr->FilePart,
              tkptr->SearchNam.nam$l_name,
              Length = tkptr->SearchNam.nam$l_ver -
                       tkptr->SearchNam.nam$l_name);
   }
   tkptr->FilePart[Length] = '\0';

   /********************/
   /* begin processing */
   /********************/

   tkptr->FileCount = tkptr->DirectoryCount = 0;

   if (VMSnok (DirFormatLayout (rqptr)))
   {
      DirEnd (rqptr);
      return;
   }

   if (!(tkptr->ResponseHeaderSent = (rqptr->ResponseHeaderPtr != NULL)))
   {
      /* "index of"s can have pre-expiry controlled from the query string */
      rqptr->ResponseStatusCode = 200;
      if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      {
         DirEnd (rqptr);
         return;
      }

      if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
      {
         DirEnd (rqptr);
         return;
      }
   }

   if (tkptr->Delimit == DELIMIT_BOTH || tkptr->Delimit == DELIMIT_HEADER)
   {
      vecptr = FaoVector;

      *vecptr++ = HtmlSgmlDoctype;
      *vecptr++ = HtmlMetaInfo (rqptr, tkptr->DirSpec);
      *vecptr++ = tkptr->SizeKilo;

      cptr = MsgFor(rqptr,MSG_DIR_INDEX_OF);
      if (tkptr->FormatLikeVms)
      {
         *vecptr++ = cptr;
         *vecptr++ = tkptr->DirSpec;
         *vecptr++ = cptr;
         *vecptr++ = tkptr->DirSpec;
      }
      else
      {
         *vecptr++ = cptr;
         *vecptr++ = tkptr->DirPath;
         *vecptr++ = cptr;
         *vecptr++ = tkptr->DirPath;
      }

      status = sys$faol (&IndexOfFaoDsc, &Length, &StringDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         DirEnd (rqptr);
         return;
      }
      String[Length] = '\0';
   }
   else
      Length = 0;

   if (Config.DirReadMeTop && tkptr->IncludeAnyReadme)
      AstFunctionPtr = &DirReadMeTop;
   else
      AstFunctionPtr = &DirHeading;

   if (Length)
   {
      NetWriteBuffered (rqptr, AstFunctionPtr, String, Length);
      return;
   }
   else
   {
      SysDclAst (AstFunctionPtr, rqptr);
      return;
   }
}

/*****************************************************************************/
/*
Release any dynamic memory allocated for parse structures.
*/ 

DirEnd (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirEnd()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (tkptr->ParseInUse)
   {
      /* ensure parse internal data structures are released */
      tkptr->SearchFab.fab$l_fna = "a:[b]c.d;";
      tkptr->SearchFab.fab$b_fns = 9;
      tkptr->SearchFab.fab$b_dns = 0;
      tkptr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tkptr->SearchFab, 0, 0);
   }

   /* restore previous directory task (if any) */
   rqptr->DirTaskPtr = LIST_PREV (tkptr);

   /* declare the next task */
   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
AST-driven output of column headings (these have already generated by 
DirFormatLayout()).
*/ 

DirHeading (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirHeading()\n");

   /* error in readme file? */
   if (rqptr->ErrorMessagePtr != NULL)
   {
      DirEnd (rqptr);
      return;
   }

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (tkptr->Delimit == DELIMIT_BOTH ||
       tkptr->Delimit == DELIMIT_HEADER)
   {
      /* header and horizontal line at top of page */
      NetWriteBuffered (rqptr, &DirBeginDirectories,
                        tkptr->LayoutHeadingPtr, tkptr->LayoutHeadingLength);
      return;
   }

   /* just start the preformtted text */
   NetWriteBuffered (rqptr, &DirBeginDirectories, "\n<PRE>", 6);
}

/*****************************************************************************/
/*
Begin a listing with directories if not expressly forbidden by configuration 
or local directory-specific configuration files.  If required check if there 
is an accessable parent directory and if so generate a formatted entry for it. 
*/ 
 
DirBeginDirectories (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr;
   register struct DirTaskStruct  *tkptr;

   boolean  ParentDirectory,
            SubDirectories;
   int  status;
   char  ParentDirPath [256],
         ParentDirVms [256];
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirBeginDirectories()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   /*
      Determine if any parent directory or subdirectories will be
      displayed by checking for the presence of directory listing
      control files (and a kludge for user directory mapping).
   */

   /* ensure we can read these directory listing "control" files */
   EnableSysPrv ();

   ParentDirectory = SubDirectories = true;
   cptr = tkptr->DirectoryPart;

   if (tkptr->AsIfNopFound && tkptr->AsIfNosFound)
      ParentDirectory = SubDirectories = false;
   else
   if (FileExists (cptr, DirNopsFileName) != RMS$_FNF)
      ParentDirectory = SubDirectories = false;
   else
   {
      if (tkptr->AsIfNosFound)
         SubDirectories = false;
      else
      if (FileExists (cptr, DirNosFileName) != RMS$_FNF)
         SubDirectories = false;

      if (tkptr->AsIfNopFound)
         ParentDirectory = false;
      else
      if (FileExists (cptr, DirNopFileName) != RMS$_FNF)
         ParentDirectory = false;
      else
      if (*(unsigned short*)(cptr = tkptr->DirPath) == 0x7e2f)
      {
         /*
            This is a kludge to better support user directories ("/~username/").
            It prevents a "top-level" user directory from displaying a parent
            directory even if it is not a "top-level" VMS directory.  It does
            this by looking to see if there is any subdirectory to the "top-
            level" user directory (e.g. "/~username/subdirectory/").
         */

         /* step over the leading "/~" */
         cptr += 2;
         /* scan to the end of the username */
         while (*cptr && *cptr != '/') cptr++;
         if (*cptr)
         {
            /* check to see if there is a directory beyond the username */
            cptr++;
            while (*cptr && *cptr != '/') cptr++;
         }
         if (!*cptr) ParentDirectory = false;
      }
   }

   DisableSysPrv ();

   if (SubDirectories)
   {
      /* next function to be executed will be the search for directories */
      AstFunctionPtr = &DirSearchDirectories;

      tkptr->ParseInUse = true;

      /* set up ready to search for all files ending in ".DIR" (directories) */
      tkptr->SearchFab = cc$rms_fab;
      /* set the FAB user context to the client thread pointer */
      tkptr->SearchFab.fab$l_ctx = rqptr;
      tkptr->SearchFab.fab$l_dna = "*.DIR;";
      tkptr->SearchFab.fab$b_dns = 6;
      tkptr->SearchFab.fab$l_fna = tkptr->DirectoryPart;
      tkptr->SearchFab.fab$b_fns = strlen(tkptr->DirectoryPart);
      tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
      tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

      tkptr->SearchNam = cc$rms_nam;
      tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
      tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
      tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
      tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

      if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

      status = sys$parse (&tkptr->SearchFab, 0, 0);

      if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

      if (VMSnok (status))
      {
         if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
         rqptr->ErrorTextPtr = tkptr->DirPath;
         rqptr->ErrorHiddenTextPtr = tkptr->DirectoryPart;
         ErrorVmsStatus (rqptr, status, FI_LI);
         DirEnd (rqptr);
         return;
      }
   }
   else
   {
      /* next function to be executed will be to start the search for files */
      AstFunctionPtr = &DirBeginFiles;
   }

   if (ParentDirectory)
   {
      cptr = tkptr->DirectoryPart;
      sptr = ParentDirVms;
      while (*cptr && *cptr != '[') *sptr++ = *cptr++;
      if (!memcmp (cptr, "[000000]", 8))
      {
         /* in Master-File-Directory, therefore no parent */
         ParentDirectory = false;
      }
      else
      {
         /*
            Not in a Master-File-Directory, create a parent directory
            (e.g. "DEVICE:[DIR1.DIR2]" from "DEVICE:[DIR1.DIR2.DIR3]",
            and "DEVICE:[000000]" from "DEVICE:[DIR1]", etc.)
         */
         if (!memcmp (cptr, "[000000.", 8))
         {
            /* skip over the Master-File-Directory in the specification */
            *sptr++ = *cptr++;
            cptr += 7;
         }
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         while (*sptr != ']') sptr--;
         while (*sptr != '.' && *sptr != '[') sptr--;
         if (*sptr == '.')
         {
            *sptr++ = ']';
            *sptr = '\0';
         }
         else
            memcpy (sptr, "[000000]", 9);
         if (Debug) fprintf (stdout, "ParentDirVms |%s|\n", ParentDirVms);

         /*
            If access is allowed display the details.
            Otherwise, its as if the directory just didn't exist!
         */

         /* ensure we can read these directory listing "control" files */
         EnableSysPrv ();

         if ((Config.DirAccessSelective &&
              VMSnok (FileExists (ParentDirVms, DirBrowsableFileName))) ||
             FileExists (ParentDirVms, DirHiddenFileName) != RMS$_FNF)
            ParentDirectory = false;

         DisableSysPrv ();
      }

      if (ParentDirectory)
      {
         ParentDirPath[0] = '\0';
         sptr = MapUrl_Map (ParentDirPath, ParentDirVms, NULL, NULL, rqptr);
         if (sptr[0])
         {
            DirFormat (rqptr, AstFunctionPtr, ParentDirPath, false);
            return;
         }
         /*
            Mapping error, most probably access forbidden, we'll assume that.
            This can happen where a subdirectory is mapable but the parent is
            not (e.g. "pass /parent/subd/* /device/parent/subd/*"), making it
            forbidden to request "/parent/file" or "/parent/*.*".
            Hence, for directory listings, it's as if parent does not exist!
         */
         ParentDirectory = false;
      }
   }

   if (!ParentDirectory)
   {
      /* parent directory was not output, explicitly drive the search */
      SysDclAst (AstFunctionPtr, rqptr);
   }
}

/*****************************************************************************/
/*
AST function to invoke another sys$search() call when listing directories.
*/ 

DirSearchDirectories (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirSearchDirectories()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      DirEnd (rqptr);
      return;
   }

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirDirectories() */
   sys$search (&tkptr->SearchFab, &DirDirectories, &DirDirectories);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another file name found or have "no more files found" status 
(or an error!).  If a file (directory) has been found check if its a "hidden" 
directory and if not format its details for the listing.
*/ 

DirDirectories (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct DirTaskStruct  *tkptr;

   int  status;
   char  LocalDirectory [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "DirDirectories() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (VMSnok (status = tkptr->SearchFab.fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
           status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /***************************/
         /* end of directory search */
         /***************************/

         tkptr->ParseInUse = false;
         DirBeginFiles (rqptr);
         return;
      }

      /**********************/
      /* sys$search() error */
      /**********************/

      rqptr->ErrorTextPtr = tkptr->DirPath;
      rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      DirEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "Directory |%s|\n", tkptr->ResultantFileName);

   /************************/
   /* directory access ok? */
   /************************/

   /*
      Create a directory specification from a directory file name.
      (e.g. "DEVICE:[DIR1.DIR2]" from "DEVICE:[DIR1]DIR2.DIR")
   */
   sptr = LocalDirectory;
   cptr = tkptr->SearchNam.nam$l_dev;
   *tkptr->SearchNam.nam$l_type = '\0';
   while (*cptr) *sptr++ = *cptr++;
   *tkptr->SearchNam.nam$l_type = '.';
   *sptr++ = ']';
   *sptr = '\0';
   sptr -= 2;
   while (*sptr != ']') sptr--;
   *sptr = '.';

   /*
      If access is allowed display the details.
      Otherwise, its as if the directory just didn't exist!
      Access may also be denied if file protections prevent checking!
   */

   if (Config.DirAccessSelective)
      status = FileExists (LocalDirectory, DirBrowsableFileName);
   else
   {
      if (VMSok (status = FileExists (LocalDirectory, DirHiddenFileName)))
         status = RMS$_FNF;
      else
         status = SS$_NORMAL;
   }

   if (VMSnok (status))
   {
      /* directory shouldn't/couldn't be accessed */
      DirSearchDirectories (rqptr);
      return;
   }

   tkptr->Description[0] = '\0';
   ConfigContentType (&tkptr->ContentType, tkptr->SearchNam.nam$l_type);
   DirFormat (rqptr, &DirSearchDirectories, "", false);
}

/*****************************************************************************/
/*
Initiate the listing of non-directory files.
*/ 

DirBeginFiles (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirBeginFiles()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   tkptr->ParseInUse = true;

   tkptr->SearchFab = cc$rms_fab;
   /* set the FAB user context to the client thread pointer */
   tkptr->SearchFab.fab$l_ctx = rqptr;
   tkptr->SearchFab.fab$l_dna = tkptr->DirectoryPart;
   tkptr->SearchFab.fab$b_dns = strlen(tkptr->DirectoryPart);
   tkptr->SearchFab.fab$l_fna = tkptr->FilePart;
   tkptr->SearchFab.fab$b_fns = strlen(tkptr->FilePart);
   tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;
   tkptr->SearchNam = cc$rms_nam;
   tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
   tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
   tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&tkptr->SearchFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = tkptr->DirPath;
      rqptr->ErrorHiddenTextPtr = tkptr->DirectoryPart;
      ErrorVmsStatus (rqptr, status, FI_LI);
      DirEnd (rqptr);
      return;
   }

   DirSearchFiles (rqptr);
}

/*****************************************************************************/
/*
AST function to invoke another sys$search() call when listing files.
*/ 

DirSearchFiles (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirSearchFiles()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      DirEnd (rqptr);
      return;
   }

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirFiles() */
   sys$search (&tkptr->SearchFab, &DirFiles, &DirFiles);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
This is an AST completion routine called each time sys$search() completes.  It 
will either point to another file name found or have "no more files found" 
status (or an error!).  If a file call a function to determine if the file 
will contain a "description".  When that function returns just return to the 
AST interrupted routine.
*/ 

DirFiles (struct FAB *FabPtr)

{
   register struct RequestStruct  *rqptr;
   register struct DirTaskStruct  *tkptr;

   int  status;
   void *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "DirFiles() sts: %%X%08.08X stv: %%X%08.08X\n",
               FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (VMSnok (status = tkptr->SearchFab.fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /**********************/
         /* end of file search */
         /**********************/

         tkptr->ParseInUse = false;
         if (tkptr->FormatLikeVms)
            DirFormat (rqptr, &DirEndOutput, "", true);
         else
            DirEndOutput (rqptr);

         return;
      }

      /**********************/
      /* sys$search() error */
      /**********************/

      rqptr->ErrorTextPtr = tkptr->DirPath;
      rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      DirEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';

   /* we've already output the directories, ignore them */
   if (!memcmp (tkptr->SearchNam.nam$l_type, ".DIR;", 5))
   {
      DirSearchFiles (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "File |%s|\n", tkptr->SearchNam.nam$l_dev);

   /* in true Unix-style "hidden" files do not appear (those beginning ".") */
   if (*(char*)tkptr->SearchNam.nam$l_name == '.')
   {
      DirSearchFiles (rqptr);
      return;
   }

   ConfigContentType (&tkptr->ContentType, tkptr->SearchNam.nam$l_type);

   if (Config.DirDescriptionLines &&
       strsame (tkptr->ContentType.ContentTypePtr, "text/html", -1))
   {
      /* generate an HTML description */
      if (VMSnok (status =
          Description (rqptr,
                       &DirEndDescription, 
                       tkptr->SearchNam.nam$l_rsa,
                       tkptr->SearchNam.nam$l_rsa,
                       tkptr->Description,
                       sizeof(tkptr->Description),
                       DESCRIPTION_TEXT_HTML)))
         DirFormat (rqptr, &DirSearchFiles, "", false);
      return;
   }
   else
   {
      /* description is the content-type */
      tkptr->Description[0] = '\0';
      DirFormat (rqptr, &DirSearchFiles, "", false);
      return;
   }
}

/*****************************************************************************/
/*
*/ 

boolean DirEndDescription (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirEndDescription()\n");

   DirFormat (rqptr, &DirSearchFiles, "", false);
}

/*****************************************************************************/
/*
Directories have been listed, files have been listed.  If appropriate provide 
a readme at the bottom, then conclude the HTML and the directory listing.
*/ 

boolean DirEndOutput (struct RequestStruct *rqptr)

{
   register struct DirTaskStruct  *tkptr;

   int  status;
   void *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirEndOutput()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (Config.DirReadMeBottom && tkptr->IncludeAnyReadme)
      AstFunctionPtr = &DirReadMeBottom;
   else
      AstFunctionPtr = &DirEnd;
 
   if (tkptr->Delimit == DELIMIT_BOTH ||
       tkptr->Delimit == DELIMIT_FOOTER)
   {
      if (!tkptr->ResponseHeaderSent)
      {
         NetWriteBuffered (rqptr, AstFunctionPtr,
                           "</PRE>\n<HR SIZE=2>\n</BODY>\n</HTML>\n", 36);
         return;
      }
      else
      {
         NetWriteBuffered (rqptr, AstFunctionPtr,
                           "</PRE>\n<HR SIZE=2>\n", 19);
         return;
      }
   }

   /* footer not required, just tie-off the preformatted text */
   NetWriteBuffered (rqptr, AstFunctionPtr, "</PRE>\n", 7);
}

/*****************************************************************************/
/*
Using the directive string pointed to by 'tkptr->LayoutPtr' format the 
directory listing column headings and sys$fao() directive string used to format
the information for each directory/file.  Return normal or error status.  This
function buffers the details of the default generic and VMS directory layout
the first time each is required.  There-after the buffered layout information
is used, reducing overhead.  Only if a user-supplied layout is required does
any format processing occur again.
*/ 
 
int DirFormatLayout (struct RequestStruct *rqptr)

{
#  define LAYOUT_SIZE 256

   static int  FieldWidthCdt,
               FieldWidthDescription,
               FieldWidthName,
               FieldWidthOwner,
               FieldWidthRdt,
               FieldWidthSize,
               HeadingCreatedLength,
               HeadingDescriptionLength,
               HeadingNameLength,
               HeadingOwnerLength,
               HeadingProtectionLength,
               HeadingRevisedLength,
               HeadingSizeLength,
               LayoutFaoLength,
               LayoutHeadingLength,
               LayoutSizeKilo,
               VmsFieldWidthCdt,
               VmsFieldWidthDescription,
               VmsFieldWidthName,
               VmsFieldWidthOwner,
               VmsFieldWidthRdt,
               VmsFieldWidthSize,
               VmsLayoutFaoLength,
               VmsLayoutHeadingLength,
               VmsLayoutSizeKilo;

   static char  LayoutFao [LAYOUT_SIZE/2],
                LayoutHeading [LAYOUT_SIZE],
                VmsLayoutFao [LAYOUT_SIZE/2],
                VmsLayoutHeading [LAYOUT_SIZE];

   static char  *HeadingCreatedPtr,
                *HeadingDescriptionPtr,
                *HeadingNamePtr,
                *HeadingOwnerPtr,
                *HeadingProtectionPtr,
                *HeadingRevisedPtr,
                *HeadingSizePtr,
                *ColumnHeadingsPtr;

   register int  cnt;
   register char  *cptr, *lptr, *sptr, *wptr, *zptr;
   register struct DirTaskStruct  *tkptr;

   boolean  ForbiddenLayout;
   int  size;
   char  String [LAYOUT_SIZE];
   char  *LayoutBufferPtr,
         *LayoutFaoPtr,
         *LayoutHeadingPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirFormatLayout()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   tkptr->MakeDescriptionLink = false;

   if (ColumnHeadingsPtr == NULL)
   {
      /******************************/
      /* initialize column headings */
      /******************************/

      sptr = MsgFor(rqptr,MSG_DIR_COLUMN_HEADINGS);
      cptr = ColumnHeadingsPtr = VmGet (strlen(sptr));
      strcpy (cptr, sptr);

      HeadingCreatedPtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingCreatedLength = strlen(HeadingCreatedPtr);
      HeadingDescriptionPtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingDescriptionLength = strlen(HeadingDescriptionPtr);
      HeadingNamePtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingNameLength = strlen(HeadingNamePtr);
      HeadingOwnerPtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingOwnerLength = strlen(HeadingOwnerPtr);
      HeadingProtectionPtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingProtectionLength = strlen(HeadingProtectionPtr);
      HeadingRevisedPtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingRevisedLength = strlen(HeadingRevisedPtr);
      HeadingSizePtr = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      HeadingSizeLength = strlen(HeadingSizePtr);
   }

   if (tkptr->LayoutPtr == NULL)
   {
      /******************/
      /* default layout */
      /******************/

      if (Config.DefaultDirLayout[0])
         tkptr->LayoutPtr = Config.DefaultDirLayout;
      else
         tkptr->LayoutPtr = DEFAULT_DIR_LAYOUT;

      if (!tkptr->FormatLikeVms && LayoutFaoLength)
      {
         /* default generic heading and fao have already been generated */
         tkptr->LayoutFaoPtr = LayoutFao;
         tkptr->LayoutFaoLength = LayoutFaoLength;
         tkptr->LayoutHeadingPtr = LayoutHeading;
         tkptr->LayoutHeadingLength = LayoutHeadingLength;
         tkptr->FieldWidthCdt = FieldWidthCdt;
         tkptr->FieldWidthDescription = FieldWidthDescription;
         tkptr->FieldWidthName = FieldWidthName;
         tkptr->FieldWidthOwner = FieldWidthOwner;
         tkptr->FieldWidthRdt = FieldWidthRdt;
         tkptr->FieldWidthSize = FieldWidthSize;
         tkptr->SizeKilo = LayoutSizeKilo;

         if (!Debug) return (SS$_NORMAL);

         fprintf (stdout,"|%d|%s|\n|%d|%s|\n",
                  LayoutHeadingLength, LayoutHeading,
                  LayoutFaoLength, LayoutFao);
         return (SS$_NORMAL);
      }

      if (tkptr->FormatLikeVms && VmsLayoutFaoLength)
      {
         /* default VMS heading and fao have already been generated */
         tkptr->LayoutFaoPtr = VmsLayoutFao;
         tkptr->LayoutFaoLength = VmsLayoutFaoLength;
         tkptr->LayoutHeadingPtr = VmsLayoutHeading;
         tkptr->LayoutHeadingLength = VmsLayoutHeadingLength;
         tkptr->FieldWidthCdt = VmsFieldWidthCdt;
         tkptr->FieldWidthDescription = VmsFieldWidthDescription;
         tkptr->FieldWidthName = VmsFieldWidthName;
         tkptr->FieldWidthOwner = VmsFieldWidthOwner;
         tkptr->FieldWidthRdt = VmsFieldWidthRdt;
         tkptr->FieldWidthSize = VmsFieldWidthSize;
         tkptr->SizeKilo = VmsLayoutSizeKilo;

         if (!Debug) return (SS$_NORMAL);

         fprintf (stdout,"|%d|%s|\n|%d|%s|\n",
                  VmsLayoutHeadingLength, VmsLayoutHeading,
                  VmsLayoutFaoLength, VmsLayoutFao);
         return (SS$_NORMAL);
      }

      /* default headings and layouts have not been generated ... do it! */
      if (tkptr->FormatLikeVms)
      {
         /* use the function-internal VMS buffers */
         LayoutHeadingPtr = VmsLayoutHeading;
         LayoutFaoPtr = VmsLayoutFao;
      }
      else
      {
         /* use the function-internal generic buffers */
         LayoutHeadingPtr = LayoutHeading;
         LayoutFaoPtr = LayoutFao;
      }
   }
   else
   {
      /**********************/
      /* non-default layout */
      /**********************/

      /* use scratch space to generate the non-default heading and fao */
      LayoutHeadingPtr = LayoutFaoPtr = String;
      tkptr->SizeKilo = 1024;
   }

   /********************************/
   /* assess layout for forbiddens */
   /********************************/

   if (!Config.DirOwnerEnabled && !rqptr->AuthVmsUserProfileLength)
   {
      /* remove OWNER and PROTECTION directives if found */
      ForbiddenLayout = false;
      cptr = tkptr->LayoutPtr;
      while (*cptr)
      {
         if (*cptr == ':')
         {
            cptr++;
            if (*cptr) cptr++;
            continue;
         }
         if (toupper(*cptr) == 'P' || toupper(*cptr) == 'O')
         {
            ForbiddenLayout = true;
            cptr++;
            continue;
         }
         cptr++;
      }
      if (ForbiddenLayout)
      {
         /* remove forbidden directives creating new layout dynamic buffer */
         sptr = LayoutBufferPtr = VmGetHeap (rqptr, LAYOUT_SIZE);
         zptr = sptr + LAYOUT_SIZE;
         cptr = tkptr->LayoutPtr;
         while (*cptr)
         {
            if (sptr >= zptr) break;
            if (*cptr == ':')
            {
               if (sptr < zptr) *sptr++ = *cptr++;
               if (*cptr && sptr < zptr) *sptr++ = *cptr++;
               continue;
            }
            if (toupper(*cptr) == 'P' || toupper(*cptr) == 'O')
            {
               cptr++;
               while (*cptr == '_') cptr++;
               continue;
            }
            if (sptr < zptr) *sptr++ = *cptr++;
         }
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            DirEnd (rqptr);
            return;
         }
         *sptr = '\0';
         tkptr->LayoutPtr = LayoutBufferPtr;
         if (Debug)
            fprintf (stdout, "tkptr->LayoutPtr |%s|\n", tkptr->LayoutPtr);
      }
   }

   /******************************/
   /* first, the column headings */
   /******************************/

   if (Debug) fprintf (stdout, "tkptr->LayoutPtr |%s|\n", tkptr->LayoutPtr);

   zptr = (sptr = LayoutHeadingPtr) + LAYOUT_SIZE;
   lptr = tkptr->LayoutPtr;
   wptr = "";

   cptr = "<PRE>";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   while (*lptr && *lptr != '&' && sptr < zptr)
   {
      /** if (Debug) fprintf (stdout, "lptr |%s|\n", lptr); **/

      if (isdigit(*lptr))
      {
         wptr = lptr;
         while (*lptr && isdigit(*lptr)) lptr++;
         continue;
      }

      if (wptr[0])
      {
         cnt = atoi(wptr);
         if (cnt < 0) cnt = 0;
         wptr = "";
      }
      else
         cnt = 0;

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_INTERFIELD;
            break;

         case 'C' :
            cptr = HeadingCreatedPtr;
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_CDT;
            if (cnt < (size = HeadingCreatedLength)) cnt = size;
            if (tkptr->FormatLikeVms) cnt += 5;
            tkptr->FieldWidthCdt = cnt;
            break;

         case 'D' :
            /* flush left */
            cptr = HeadingDescriptionPtr;
            if (cnt)
            {
               if (cnt > 255) cnt = 255;
               if (cnt < (size = HeadingDescriptionLength)) cnt = size;
               tkptr->FieldWidthDescription = cnt;
            }
            else
            {
               cnt = size = HeadingDescriptionLength;
               tkptr->FieldWidthDescription = 0;
            }
            if (lptr[1] == LAYOUT_PARAMETER)
            {
               switch (toupper(lptr[2]))
               {
                  case 'L' :
                     tkptr->MakeDescriptionLink = true;
                     break;
                  default :
                     /* unknown layout directive character */
                     rqptr->ResponseStatusCode = 501;
                     ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
                     return (STS$K_ERROR);
               }
            }
            break;

         case 'I' :  /* icon */
            break;

         case 'L' :
         case 'N' :
            cptr = HeadingNamePtr;
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_NAME;
            if (cnt < (size = HeadingNameLength)) cnt = size;
            tkptr->FieldWidthName = cnt;
            break;

         case 'O' :
            cptr = HeadingOwnerPtr;
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_OWNER;
            if (cnt < (size = HeadingOwnerLength)) cnt = size;
            tkptr->FieldWidthOwner = cnt;
            break;

         case 'P' :
            cptr = HeadingProtectionPtr;
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_PROTECTION;
            if (cnt < (size = HeadingProtectionLength)) cnt = size;
            tkptr->FieldWidthProtection = cnt;
            break;

         case 'R' :
            cptr = HeadingRevisedPtr;
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_RDT;
            if (cnt < (size = HeadingRevisedLength)) cnt = size;
            if (tkptr->FormatLikeVms) cnt += 5;
            tkptr->FieldWidthRdt = cnt;
            break;

         case 'S' :
            if (tkptr->FormatLikeVms)
            {
               if (!cnt) cnt = DEFAULT_FIELD_WIDTH_SIZE;
               if (cnt < (size = DEFAULT_FIELD_WIDTH_SIZE - 2)) cnt = size;
               /* expand size field width by 7 for blocks allocated/used */
               cnt += 7;
            }
            else
            if (lptr[1] == LAYOUT_PARAMETER)
            {
               switch (toupper(lptr[2]))
               {
                  case 'D' :
                  case 'F' :
                     if (!cnt) cnt = DEFAULT_FIELD_WIDTH_SIZE;
                     if (cnt < (size = DEFAULT_FIELD_WIDTH_SIZE - 2))
                        cnt = size;
                     break;
                  case 'B' :
                  case 'K' :
                  case 'M' :
                     if (!cnt) cnt = DEFAULT_FIELD_WIDTH_DECIMAL;
                     if (cnt < (size = DEFAULT_FIELD_WIDTH_DECIMAL - 7))
                        cnt = size;
                     break;
                  default :
                     /* unknown layout directive character */
                     rqptr->ResponseStatusCode = 501;
                     ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
                     return (STS$K_ERROR);
               }
               tkptr->SizeKilo = 1000;
            }
            else
            {
               if (!cnt) cnt = DEFAULT_FIELD_WIDTH_SIZE;
               if (cnt < (size = DEFAULT_FIELD_WIDTH_SIZE - 2)) cnt = size;
               /* expand size field width by 7 for blocks allocated/used */
               if (tkptr->FormatLikeVms) cnt += 7;
            }
            cptr = HeadingSizePtr;
            tkptr->FieldWidthSize = cnt;
            break;

         case 'U' :
            /* force to upper case, just ignore, MUST be first directive */
            break;

         default :
            /* unknown layout directive character */
            rqptr->ResponseStatusCode = 501;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
            return (STS$K_ERROR);
      }

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'C' :
         case 'R' :
         case 'S' :
            /* flush the column heading right using spaces */
            cnt -= size;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'D' :
         case 'L' :
         case 'N' :
         case 'O' :
         case 'P' :
            /* flush the column heading left using spaces */
            cnt -= size;
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'I' :
            cptr = ConfigBlankIconPtr;
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;
      }
      if (lptr[1] == LAYOUT_PARAMETER) lptr += 2;
      lptr++;
   }

   cptr = "\n<HR SIZE=2>\n";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   if (sptr >= zptr)
   {
      /* should be enough for all but a deliberate formatting hack! */
      rqptr->ResponseStatusCode = 501;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   cnt = sptr - LayoutHeadingPtr;

   if (Debug)
      fprintf (stdout, "LayoutHeadingPtr |%d|%s|\n", cnt, LayoutHeadingPtr);

   if (LayoutHeadingPtr == LayoutHeading)
   {
      /* first time a generic layout has been required, buffer details */
      tkptr->LayoutHeadingPtr = LayoutHeading;
      tkptr->LayoutHeadingLength = LayoutHeadingLength = cnt;
      FieldWidthCdt = tkptr->FieldWidthCdt;
      FieldWidthDescription = tkptr->FieldWidthDescription;
      FieldWidthName = tkptr->FieldWidthName;
      FieldWidthOwner = tkptr->FieldWidthOwner;
      FieldWidthRdt = tkptr->FieldWidthRdt;
      FieldWidthSize = tkptr->FieldWidthSize;
   }
   else
   if (LayoutHeadingPtr == VmsLayoutHeading)
   {
      /* first time a VMS layout has been required, buffer details */
      tkptr->LayoutHeadingPtr = VmsLayoutHeading;
      tkptr->LayoutHeadingLength = VmsLayoutHeadingLength = cnt;
      VmsFieldWidthCdt = tkptr->FieldWidthCdt;
      VmsFieldWidthDescription = tkptr->FieldWidthDescription;
      VmsFieldWidthName = tkptr->FieldWidthName;
      VmsFieldWidthOwner = tkptr->FieldWidthOwner;
      VmsFieldWidthRdt = tkptr->FieldWidthRdt;
      VmsFieldWidthSize = tkptr->FieldWidthSize;
   }
   else
   {
      /* this is a user-supplied layout, place it in the thread's heap */
      tkptr->LayoutHeadingPtr = lptr = VmGetHeap (rqptr, cnt+1);
      memcpy (lptr, LayoutHeadingPtr, cnt+1);
      tkptr->LayoutHeadingLength = cnt;
   }

   /******************************************/
   /* second, the sys$fao() directive string */
   /******************************************/

   zptr = (sptr = LayoutFaoPtr) + LAYOUT_SIZE / 2;
   lptr = tkptr->LayoutPtr;
   wptr = "";

   /* accomodate an optional leading new-line (for creating a blank line) */
   cptr = "!AZ";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   while (*lptr && *lptr != '&' && sptr < zptr)
   {
      /** if (Debug) fprintf (stdout, "lptr |%s|\n", lptr); **/

      if (isdigit(*lptr))
      {
         wptr = lptr;
         while (*lptr && isdigit(*lptr)) lptr++;
         continue;
      }

      if (wptr[0])
      {
         cnt = atoi(wptr);
         if (cnt < 0) cnt = 0;
         wptr = "";
      }
      else
         cnt = 0;

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            if (!cnt) cnt = DEFAULT_FIELD_WIDTH_INTERFIELD;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'C' :
         case 'R' :
         case 'S' :
            /* right justify, then value */
            cptr = "!#<!#AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'D' :
            /* nothing fancy (allow for leading/trailing quotes and link) */
            if (tkptr->MakeDescriptionLink)
               if (tkptr->FieldWidthDescription)
                  cptr = "!AZ!AZ!AZ!AZ!#<!#AZ!AZ!AZ!AZ!>";
               else
                  cptr = "!AZ!AZ!AZ!AZ!AZ!AZ!AZ";
            else
               if (tkptr->FieldWidthDescription)
                  cptr = "!AZ!#<!#AZ!AZ!AZ!>";
               else
                  cptr = "!AZ!AZ!AZ";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'I' :
            /* nothing fancy */
            cptr = "!AZ";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'L' :
            /* left justify, then link, name, and name overflow */
            cptr = "!AZ!AZ!AZ!#<!AZ!AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'N' :
            /* left justify, string, plus overflow */
            cptr = "!#<!AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'O' :
         case 'P' :
            /* '!#AZ' will left justify value with spaces */
            cptr = "!#AZ";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;
      }
      if (lptr[1] == LAYOUT_PARAMETER) lptr += 2;
      lptr++;
   }
   if (sptr < zptr) *sptr++ = '\n';

   if (sptr >= zptr)
   {
      /* should be enough for all but a deliberate formatting hack! */
      rqptr->ResponseStatusCode = 501;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   cnt = sptr - LayoutFaoPtr;

   if (Debug) fprintf (stdout, "LayoutFaoPtr |%d|%s|\n", cnt, LayoutFaoPtr);

   if (LayoutFaoPtr == LayoutFao)
   {
      /* first time a generic layout has been required, buffer details */
      tkptr->LayoutFaoPtr = LayoutFao;
      tkptr->LayoutFaoLength = LayoutFaoLength = cnt;
      LayoutSizeKilo = tkptr->SizeKilo;
   }
   else
   if (LayoutFaoPtr == VmsLayoutFao)
   {
      /* first time a VMS layout has been required, buffer details */
      tkptr->LayoutFaoPtr = VmsLayoutFao;
      tkptr->LayoutFaoLength = VmsLayoutFaoLength = cnt;
      VmsLayoutSizeKilo = 1024;
   }
   else
   {
      /* this is a user-supplied layout, place it in the thread's heap */
      tkptr->LayoutFaoPtr = lptr = VmGetHeap (rqptr, cnt+1);
      memcpy (lptr, LayoutFaoPtr, cnt+1);
      tkptr->LayoutFaoLength = cnt;
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
This overly-long (sorry) function formats and outputs a line of information 
for a parent directory, directory file, non-directory file, or for the final 
line giving used/allocated totals for a VMS format listing. 

For directory and ordinary files retrieve and display the specified file's 
statistics (name, size, modification  time, etc.)  This function completes 
synchronously, that is it waits for the single QIO on the file details to 
complete before assembling the details for the listing, because of this the 
function introduces slight granularity.  It is a candidate for future 
improvement. 

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int DirFormat
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *ParentDirPath,
boolean BlockTotals
)
{
   static unsigned long  LibDateTimeContext = 0,
                         LibDateTimeVmsContext = 0,
                         LibOutputFormat = LIB$K_OUTPUT_FORMAT;

   static $DESCRIPTOR (BlocksFaoDsc, "!UL/!UL");
   static $DESCRIPTOR (BytesFaoDsc, "!UL");
   static $DESCRIPTOR (CdtDsc, "");
   static $DESCRIPTOR (DeviceDsc, "");
   static $DESCRIPTOR (LayoutFaoDsc, "");
   static $DESCRIPTOR (KBytesFaoDsc, "!ULK");
   static $DESCRIPTOR (KBytesFractFaoDsc, "!UL.!ULK");
   static $DESCRIPTOR (KBytesFullFractFaoDsc, "!UL.!3ZLK");
   static $DESCRIPTOR (MBytesFaoDsc, "!ULM");
   static $DESCRIPTOR (MBytesFractFaoDsc, "!UL.!ULM");
   static $DESCRIPTOR (MBytesFullFractFaoDsc, "!UL.!6ZLM");
   static $DESCRIPTOR (OutputFormatDsc, "|!DB-!MAAC-!Y2|!H04:!M0|");
   static $DESCRIPTOR (OutputFormatVmsDsc, "|!DB-!MAAU-!Y4|!H04:!M0:!S0|");
   static $DESCRIPTOR (OwnerFaoDsc, "!%I");
   static $DESCRIPTOR (OwnerDsc, "");
   static $DESCRIPTOR (RdtDsc, "");
   static $DESCRIPTOR (SizeDsc, "");
   static $DESCRIPTOR (SizeFaoDsc, "!#* !AZ");
   static $DESCRIPTOR (StringDsc, "");
   static $DESCRIPTOR (TotalFilesDsc, "");
   static $DESCRIPTOR (TotalFilesFaoDsc, "!UL files");


   register int  idx, MaxIdx;
   register char  *cptr, *sptr, *zptr;
   register struct DirTaskStruct  *tkptr;

   boolean  ThisIsADirectory,
            VmsUserHasAccess;
   int  status;
   unsigned short  AcpChannel,
                   Length;
   unsigned short  NumTime [7];
   unsigned long  AllocatedVbn,
                  Bytes,
                  Count,
                  DateLength,
                  EndOfFileVbn;
   unsigned long  FaoVector [64];
   char  *IconPtr,
         *AnchorNameOverflowPtr;
   char  AnchorLink [1024],
         AnchorName [128],
         Cdt [32],
         Description [256],
         Owner [64],
         Protection [64],
         Rdt [32],
         Scratch [256],
         Size [32],
         String [2048],
         TotalFiles [32];

   unsigned short  AtrFpro;
   unsigned long  AtrUic;
   unsigned long  AtrCdt [2],
                  AtrRdt [2];
   unsigned long  *ctxptr;
   struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   struct fibdef  SearchFib;
   struct atrdef  SearchAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrCdt), ATR$C_CREDATE, &AtrCdt },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { sizeof(AtrUic), ATR$C_UIC, &AtrUic },
      { sizeof(AtrFpro), ATR$C_FPRO, &AtrFpro },
      { 0, 0, 0 }
   };

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     SearchFibAcpDsc,
     SearchAtrAcpDsc;

   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirFormat() |%s|\n",
               rqptr->DirTaskPtr->SearchNam.nam$l_dev);

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   if (!LibDateTimeContext)
   {
      /* initialize the date/time formats */
      status = lib$init_date_time_context (&LibDateTimeContext,
                                           &LibOutputFormat,
                                           &OutputFormatDsc);
      if (VMSok (status))
         status = lib$init_date_time_context (&LibDateTimeVmsContext,
                                              &LibOutputFormat,
                                              &OutputFormatVmsDsc);

      if (VMSnok (status))
         ErrorExitVmsStatus (status, "lib$init_date_time_context()", FI_LI);
   }

   if (!(ParentDirPath[0] || BlockTotals))
   {
      /*******************************************/
      /* queue an ACP I/O to get file attributes */
      /*******************************************/

      /* assign a channel to the disk device containing the file */
      DeviceDsc.dsc$w_length = tkptr->SearchNam.nam$b_dev;
      DeviceDsc.dsc$a_pointer = tkptr->SearchNam.nam$l_dev;
      status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = tkptr->DirPath;
         rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
         ErrorVmsStatus (rqptr, status, FI_LI);
         DirEnd (rqptr);
         return;
      }

      /* set up the File Information Block for the ACP interface */
      memset (&SearchFib, 0, sizeof(struct fibdef));
      SearchFibAcpDsc.Length = sizeof(SearchFib);
      SearchFibAcpDsc.Address = &SearchFib;
      /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
      memcpy (&SearchFib.fib$w_did, &tkptr->SearchNam.nam$w_did, 6);
#else
      memcpy (&SearchFib.fib$r_did_overlay.fib$w_did,
              &tkptr->SearchNam.nam$w_did, 6);
#endif

      FileNameAcpDsc.Length =
         tkptr->SearchNam.nam$b_name +
         tkptr->SearchNam.nam$b_type +
         tkptr->SearchNam.nam$b_ver;
      FileNameAcpDsc.Address = tkptr->SearchNam.nam$l_name;

      /* specially check permissions for each file! */
      if (rqptr->AuthVmsUserProfileLength)
      {
         /*
            SYSUAF-authenticated user.
            Interestingly the sys$qio() could still return valid information
            if the directory had group permissions allowing it, even though
            sys$check_access() indicated no privilege!
         */
         status = AuthCheckVmsUserAccess (rqptr, tkptr->SearchNam.nam$l_rsa,
                                                 tkptr->SearchNam.nam$b_rsl);
         if (VMSok (status))
         {
            EnableSysPrv ();
            status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                               &SearchFibAcpDsc, &FileNameAcpDsc, 0, 0,
                               &SearchAtr, 0);
            DisableSysPrv ();
         }
      }
      else
      {
         /* bog-standard access */
         status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                            &SearchFibAcpDsc, &FileNameAcpDsc, 0, 0,
                            &SearchAtr, 0);
      }

      /* immediately deassign the channel in case we return on an error */
      sys$dassgn (AcpChannel);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, AcpIOsb.Status);

      if (VMSok (status)) status = AcpIOsb.Status;
      if (VMSnok (status)) 
      {
         /*************************/
         /* protection violation? */
         /*************************/

         if (status == SS$_NOPRIV && Config.DirNoPrivIgnore)
         {
            /* can't return without queueing the next bit of processing! */
            SysDclAst (AstFunctionPtr, rqptr);
            return;
         }

         /*******************/
         /* something else! */
         /*******************/

         rqptr->ErrorTextPtr = tkptr->DirPath;
         rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
         ErrorVmsStatus (rqptr, status, FI_LI);
         DirEnd (rqptr);
         return;
      }

      /* test for numeric equivalent of "DIR;" */
      if (ThisIsADirectory = !memcmp(tkptr->SearchNam.nam$l_type, ".DIR;", 5))
      {
         /* no need to display the MFD */
         if (tkptr->SearchNam.nam$l_name[0] == '0' &&
             !memcmp (tkptr->SearchNam.nam$l_name, "000000.DIR;", 11))
         {
            /* can't return without queueing the next bit of processing! */
            SysDclAst (AstFunctionPtr, rqptr);
            return;
         }
      }
   }

   /********************************/
   /* process file name components */
   /********************************/

   MaxIdx = sizeof(FaoVector);
   idx = 0;

   if (BlockTotals)
   {
      /****************/
      /* block totals */
      /****************/

      /* must be block totals, which requires a leading newline */
      if (idx < MaxIdx) FaoVector[idx++] = "\n";
   }
   else
   if (ParentDirPath[0])
   {
      /********************/
      /* parent directory */
      /********************/

      /* no leading newline required for the parent directory */
      if (idx < MaxIdx) FaoVector[idx++] = "";
      tkptr->DirectoryCount = -1;

      zptr = (sptr = AnchorLink) + sizeof(AnchorLink);
      if (tkptr->RelativePath[0])
      {
         /* copy in the directory portion of the URL */
         cptr = ParentDirPath;
      }
      else
      {
         /* copy in the parent directory shorthand */
         cptr = "../";
      }
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      /* append any file name/type/version from the original spec */
      if (tkptr->DirSpecIncludedFilePart ||
          !Config.DirNoImpliedWildcard)
      {
         cptr = tkptr->FilePart;
         while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      }
      /* propagate any query string */
      if (tkptr->QueryString[0])
      {
         if (sptr < zptr) *sptr++ = '?';
         for (cptr = tkptr->QueryString;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         DirEnd (rqptr);
         return;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      if (tkptr->FormatLikeVms)
         memcpy (AnchorName, "[-]", 4);
      else
         memcpy (AnchorName, "../", 4);
      sptr = AnchorName + 4;
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/
   }
   else
   if (ThisIsADirectory)
   {
      /***************/
      /* a directory */
      /***************/

      /* no leading newline required for directories */
      if (idx < MaxIdx) FaoVector[idx++] = "";
      if (tkptr->DirectoryCount <= 0)
         tkptr->DirectoryCount = 1;
      else
         tkptr->DirectoryCount++;

      /* terminate to remove the file type (".DIR") */
      *tkptr->SearchNam.nam$l_type = '\0';

      zptr = (sptr = AnchorLink) + sizeof(AnchorLink);

      /* only when specified as "/dir1/dir2" this contains "./dir2/" */
      cptr = tkptr->RelativePath;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      /* copy in the directory (file) name */
      cptr = tkptr->SearchNam.nam$l_name;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* end directory */
      if (sptr < zptr) *sptr++ = '/';
      /* append any file name/type/version from the original spec */
      if (tkptr->DirSpecIncludedFilePart ||
          !Config.DirNoImpliedWildcard)
       {
         cptr = tkptr->FilePart;
         while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      }
      /* propagate any query string */
      if (tkptr->QueryString[0])
      {
         if (sptr < zptr) *sptr++ = '?';
         for (cptr = tkptr->QueryString;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         DirEnd (rqptr);
         return;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      sptr = AnchorName;
      if (tkptr->FormatLikeVms)
      {
         *sptr++ = '[';
         *sptr++ = '.';
      }
      cptr = tkptr->SearchNam.nam$l_name;
      if (toupper(tkptr->LayoutPtr[0]) == 'U' || tkptr->FormatLikeVms)
         while (*cptr) *sptr++ = *cptr++;
      else
         while (*cptr) *sptr++ = tolower(*cptr++);
      if (tkptr->FormatLikeVms)
         *sptr++ = ']';
      else
         *sptr++ = '/';
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/

      /* restore the type delimitter */
      *tkptr->SearchNam.nam$l_type = '.';
   }
   else
   {
      /**********/
      /* a file */
      /**********/

      /* leading newline required for first file after directories */
      if (tkptr->DirectoryCount && !tkptr->FileCount)
         { if (idx < MaxIdx) FaoVector[idx++] = "\n"; }
      else
         if (idx < MaxIdx) FaoVector[idx++] = "";
      tkptr->FileCount++;

      /* if appropriate, terminate to remove the version number */
      if (!tkptr->FormatLikeVms) *tkptr->SearchNam.nam$l_ver = '\0';

      zptr = (sptr = AnchorLink) + sizeof(AnchorLink);

      if (tkptr->ScriptName[0])
      {
         /* prepend the (optional) script component of the path */
         cptr = tkptr->ScriptName;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;

         /* need the directory portion of the URL for scripting */
         cptr = tkptr->DirectoryPathPtr;
         while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      }
      else
      {
         /* only when specified as "/dir1/dir2" this contains "./dir2/" */
         cptr = tkptr->RelativePath;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      }
      /* copy in the file name and type */
      cptr = tkptr->SearchNam.nam$l_name;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* disable auto-scripting by appending a (most-recent) version number */
      if (!tkptr->FormatLikeVms && !tkptr->AutoScriptEnabled)
      {
         if (sptr < zptr) *sptr++ = ';';
         if (sptr < zptr) *sptr++ = '0';
      }

      /* propagate any request-specified content-type as a query string */
      if (tkptr->QueryContentTypePtr != NULL &&
          tkptr->QueryContentTypePtr[0])
      {
         for (cptr = "?httpd=content&type=";
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
         for (cptr = tkptr->QueryContentTypePtr;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }

      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         DirEnd (rqptr);
         return;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      sptr = AnchorName;
      cptr = tkptr->SearchNam.nam$l_name;
      if (toupper(tkptr->LayoutPtr[0]) == 'U' || tkptr->FormatLikeVms)
         while (*cptr) *sptr++ = *cptr++;
      else
         while (*cptr) *sptr++ = tolower(*cptr++);
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/

      /* if appropriate, restore the version delimitter */
      if (!tkptr->FormatLikeVms) *tkptr->SearchNam.nam$l_ver = ';';
   }

   if (!BlockTotals)
   {
      /********************************/
      /* check if file name overflows */
      /********************************/

      /* assumes that the end of 'AnchorName' is still pointed to by 'sptr' */
      if (sptr - AnchorName > tkptr->FieldWidthName)
      {
         AnchorNameOverflowPtr = "+";
         AnchorName[tkptr->FieldWidthName-1] = '\0';
      }
      else
         AnchorNameOverflowPtr = "";
   }

   /***********************************/
   /* build the sys$faol() paremeters */
   /***********************************/

   cptr = tkptr->LayoutPtr;
   while (*cptr)
   {
      if (!isalpha(*cptr))
      {
         cptr++;
         continue;
      }
      /** if (Debug) fprintf (stdout, "idx: %d cptr |%s|\n", idx, cptr); **/

      switch (toupper(*cptr))
      {
         case 'C' :

            /*****************/
            /* creation date */
            /*****************/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthCdt;
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthCdt;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            /* format the creation date/time */
            CdtDsc.dsc$a_pointer = Cdt;
            CdtDsc.dsc$w_length = sizeof(Cdt)-1;
            if (tkptr->FormatLikeVms)
               ctxptr = &LibDateTimeVmsContext;
            else
               ctxptr = &LibDateTimeContext;
            if (VMSnok (status =
                lib$format_date_time (&CdtDsc, &AtrCdt, ctxptr,
                                      &DateLength, 0)))
            {
               rqptr->ErrorTextPtr = "lib$format_date_time()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               DirEnd (rqptr);
               return;
            }
            else
               Cdt[DateLength] = '\0';
            /** if (Debug) fprintf (stdout, "Cdt |%s|\n", Cdt); **/
            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthCdt;
            /* next parameter right justifies the field */
            if (tkptr->FieldWidthCdt - DateLength > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = tkptr->FieldWidthCdt - DateLength;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Cdt;
            break;

         case 'D' :

            /***************/
            /* description */
            /***************/

            if (ParentDirPath[0])
            {
               /********************/
               /* parent directory */
               /********************/

               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (tkptr->MakeDescriptionLink)
               {
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
               }
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
               }
               if (idx < MaxIdx) FaoVector[idx++] = 
                  MsgFor(rqptr,MSG_DIR_PARENT);
               if (idx < MaxIdx && tkptr->MakeDescriptionLink)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
            }
            else
            if (BlockTotals)
            {
               /***************************/
               /* VMS format block totals */
               /***************************/

               if (tkptr->DirectoryCount < 0)
                  tkptr->DirectoryCount = 0;
               TotalFilesDsc.dsc$a_pointer = TotalFiles;
               TotalFilesDsc.dsc$w_length = sizeof(TotalFiles)-1;
               sys$fao (&TotalFilesFaoDsc, &Length, &TotalFilesDsc,
                        tkptr->DirectoryCount + tkptr->FileCount);
               TotalFiles[Length] = '\0';
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (tkptr->MakeDescriptionLink)
               {
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
               }
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
               }
               if (idx < MaxIdx) FaoVector[idx++] = TotalFiles;
               if (idx < MaxIdx && tkptr->MakeDescriptionLink)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
            }
            else
            if (ThisIsADirectory)
            {
               /*************/
               /* directory */
               /*************/

               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (tkptr->MakeDescriptionLink)
               {
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
                  if (idx < MaxIdx) FaoVector[idx++] = "";
               }
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
                  FaoVector[idx++] = tkptr->FieldWidthDescription;
               }
               if (idx < MaxIdx)
               {
                  if (tkptr->MsgSubDirPtr == NULL)
                     FaoVector[idx++] = tkptr->MsgSubDirPtr =
                        MsgFor(rqptr,MSG_DIR_SUB);
                  else
                     FaoVector[idx++] = tkptr->MsgSubDirPtr;
               }
               if (idx < MaxIdx && tkptr->MakeDescriptionLink)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
                  FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
            }
            else
            if (tkptr->Description[0] &&
                tkptr->Description[0] != DESCRIPTION_IMPOSSIBLE)
            {
               /******************************/
               /* file with HTML description */
               /******************************/

               if (idx < MaxIdx) FaoVector[idx++] = "\"";
               if (tkptr->MakeDescriptionLink)
               {
                  if (idx < MaxIdx) FaoVector[idx++] = "<A HREF=\"";
                  if (idx < MaxIdx) FaoVector[idx++] = AnchorLink;
                  if (idx < MaxIdx) FaoVector[idx++] = "\">";
               }
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  /* the 4 - 1 allows for the leading quote */
                  if (tkptr->MakeDescriptionLink)
                     FaoVector[idx++] = tkptr->FieldWidthDescription + 3;
                  else
                     FaoVector[idx++] = tkptr->FieldWidthDescription - 1;
                  if ((Count = strlen(tkptr->Description)) >
                      tkptr->FieldWidthDescription - 3)
                     FaoVector[idx++] = tkptr->FieldWidthDescription - 3;
                  else
                     FaoVector[idx++] = Count;
               }
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->Description;
               if (idx < MaxIdx && tkptr->MakeDescriptionLink)
                  FaoVector[idx++] = "</A>";
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  if (Count > tkptr->FieldWidthDescription - 3)
                     FaoVector[idx++] = "+";
                  else
                     FaoVector[idx++] = "";
               }
               if (idx < MaxIdx) FaoVector[idx++] = "\"";
            }
            else
            {
               /*********************************/
               /* file with content description */
               /*********************************/

               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (tkptr->MakeDescriptionLink)
               {
                  if (idx < MaxIdx) FaoVector[idx++] = "<A HREF=\"";
                  if (idx < MaxIdx) FaoVector[idx++] = AnchorLink;
                  if (idx < MaxIdx) FaoVector[idx++] = "\">";
               }
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  if (tkptr->MakeDescriptionLink)
                     FaoVector[idx++] = tkptr->FieldWidthDescription + 4;
                  else
                     FaoVector[idx++] = tkptr->FieldWidthDescription;
                  if ((Count =
                      strlen(tkptr->ContentType.ContentDescriptionPtr)) >
                      tkptr->FieldWidthDescription - 1)
                     FaoVector[idx++] = tkptr->FieldWidthDescription - 1;
                  else
                     FaoVector[idx++] = Count;
               }
               if (idx < MaxIdx)
                  FaoVector[idx++] = tkptr->ContentType.ContentDescriptionPtr;
               if (idx < MaxIdx && tkptr->MakeDescriptionLink)
                  FaoVector[idx++] = "</A>";
               if (idx < MaxIdx && tkptr->FieldWidthDescription)
               {
                  if (Count > tkptr->FieldWidthDescription - 3)
                     FaoVector[idx++] = "+";
                  else
                     FaoVector[idx++] = "";
               }
               if (idx < MaxIdx) FaoVector[idx++] = "";
            }
            break;

         case 'I' :

            /********/
            /* icon */
            /********/

            if (ThisIsADirectory)
               IconPtr = ConfigDirIconPtr;
            else
            if (ParentDirPath[0])
               IconPtr = ConfigParentIconPtr;
            else
            if (BlockTotals)
               IconPtr = ConfigBlankIconPtr;
            else
               IconPtr = tkptr->ContentType.IconPtr;
            if (idx < MaxIdx) FaoVector[idx++] = IconPtr;
            break;

         case 'L' :

            /***************/
            /* link/anchor */
            /***************/

            if (BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthName;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            if (idx < MaxIdx) FaoVector[idx++] = "<A HREF=\"";
            if (idx < MaxIdx) FaoVector[idx++] = AnchorLink;
            if (idx < MaxIdx) FaoVector[idx++] = "\">";
            if (idx < MaxIdx)
               FaoVector[idx++] = tkptr->FieldWidthName + sizeof("</A>")-1;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorName;
            if (idx < MaxIdx) FaoVector[idx++] = "</A>";
            if (idx < MaxIdx) FaoVector[idx++] = AnchorNameOverflowPtr;
            break;

         case 'N' :

            /********/
            /* name */
            /********/

            if (BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthName;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthName;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorName;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorNameOverflowPtr;
            break;

         case 'O' :

            /*********/
            /* owner */
            /*********/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthOwner;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            OwnerDsc.dsc$a_pointer = Owner;
            OwnerDsc.dsc$w_length = sizeof(Owner)-1;
            sys$fao (&OwnerFaoDsc, &Length, &OwnerDsc, AtrUic);
            Owner[Length] = '\0';
            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthOwner;
            if (idx < MaxIdx) FaoVector[idx++] = Owner;
            break;

         case 'P' :

            /**************/
            /* protection */
            /**************/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthProtection;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            FormatProtection (AtrFpro, Protection);
            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthProtection;
            if (idx < MaxIdx) FaoVector[idx++] = Protection;
            break;

         case 'R' :

            /*****************/
            /* revision date */
            /*****************/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthRdt;
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthRdt;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            /* format the revision date/time */
            RdtDsc.dsc$a_pointer = Rdt;
            RdtDsc.dsc$w_length = sizeof(Rdt)-1;
            if (tkptr->FormatLikeVms)
               ctxptr = &LibDateTimeVmsContext;
            else
               ctxptr = &LibDateTimeContext;
            if (VMSnok (status =
                lib$format_date_time (&RdtDsc, &AtrRdt, ctxptr,
                                      &DateLength, 0)))
            {
               rqptr->ErrorTextPtr = "lib$format_date_time()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               DirEnd (rqptr);
               return;
            }
            else
               Rdt[DateLength] = '\0';
            /** if (Debug) fprintf (stdout, "Rdt |%s|\n", Rdt); **/
            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthRdt;
            /* next parameter right justifies the field */
            if (tkptr->FieldWidthRdt - DateLength > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = tkptr->FieldWidthRdt - DateLength;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Rdt;
            break;

         case 'S' :

            /********/
            /* size */
            /********/

            if (ParentDirPath[0])
            {                                   
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthSize;
               if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthSize;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            SizeDsc.dsc$a_pointer = Size;
            SizeDsc.dsc$w_length = sizeof(Size)-1;

            if (BlockTotals)
            {
               sys$fao (&BlocksFaoDsc, &Length, &SizeDsc,
                        tkptr->TotalUsedBlocks,
                        tkptr->TotalAllocatedBlocks);
            }
            else
            {
               AllocatedVbn = AtrRecord.AllocatedVbnLo +
                              (AtrRecord.AllocatedVbnHi << 16);
               EndOfFileVbn = AtrRecord.EndOfFileVbnLo +
                              (AtrRecord.EndOfFileVbnHi << 16);

               if (Debug)
                  fprintf (stdout,
                  "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
                  AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);

               if (tkptr->FormatLikeVms)
               {
                  sys$fao (&BlocksFaoDsc, &Length, &SizeDsc,
                           EndOfFileVbn, AllocatedVbn);

                  tkptr->TotalAllocatedBlocks += AllocatedVbn;
                  tkptr->TotalUsedBlocks += EndOfFileVbn;
               }
               else
               {
                  if (EndOfFileVbn <= 1)
                     Bytes = AtrRecord.FirstFreeByte;
                  else
                     Bytes = ((EndOfFileVbn-1) << 9) + AtrRecord.FirstFreeByte;
                  if (Debug) fprintf (stdout, "Bytes %d\n", Bytes);

                  if (cptr[1] == LAYOUT_PARAMETER)
                  {
                     switch (toupper(cptr[2]))
                     {
                        case 'D' :
                           if (Bytes >= 1000000)
                              sys$fao (&MBytesFaoDsc, &Length, &SizeDsc,
                                       Bytes / 1000000);
                           else
                           if (Bytes >= 1000)
                              sys$fao (&KBytesFaoDsc, &Length, &SizeDsc,
                                       Bytes / 1000);
                           else
                              sys$fao (&BytesFaoDsc, &Length, &SizeDsc, Bytes);
                           break;
      
                        case 'F' :
                           if (Bytes >= 1000000)
                              sys$fao (&MBytesFractFaoDsc, &Length, &SizeDsc,
                                       Bytes / 1000000,
                                       (Bytes % 1000000) / 100000);
                           else
                           if (Bytes >= 1000)
                              sys$fao (&KBytesFractFaoDsc, &Length, &SizeDsc,
                                       Bytes / 1000, (Bytes % 1000) / 100);
                           else
                              sys$fao (&BytesFaoDsc, &Length, &SizeDsc, Bytes);
                           break;
      
                        case 'B' :
                           Length = CommaNumber (32, Bytes, Size);
                           break;

                        case 'K' :
                           sys$fao (&KBytesFullFractFaoDsc, &Length, &SizeDsc,
                                    Bytes / 1000, Bytes % 1000);
                           break;

                        case 'M' :
                           sys$fao (&MBytesFullFractFaoDsc, &Length, &SizeDsc,
                                    Bytes / 1000000, Bytes % 1000000);
                           break;
                     }
                  }
                  else
                  {
                     if (Bytes >= 1048576)
                        sys$fao (&MBytesFaoDsc, &Length, &SizeDsc,
                                 Bytes >> 20);
                     else
                     if (Bytes >= 1024)
                        sys$fao (&KBytesFaoDsc, &Length, &SizeDsc,
                                 Bytes >> 10);
                     else
                        sys$fao (&BytesFaoDsc, &Length, &SizeDsc, Bytes);
                  }
               }
            }

            Size[Length] = '\0';
            /** if (Debug) fprintf (stdout, "Size |%s|\n", Size); **/

            if (idx < MaxIdx) FaoVector[idx++] = tkptr->FieldWidthSize;
            /* next parameter right justifies the field */
            if (tkptr->FieldWidthSize - Length > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = tkptr->FieldWidthSize - Length;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Size;
            break;
      }

      if (cptr[1] == LAYOUT_PARAMETER) cptr += 2;
      cptr++;

      if (idx >= MaxIdx)
      {
         /* should be enough for all but a deliberate formatting hack! */
         rqptr->ResponseStatusCode = 501;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_DIR_LAYOUT), FI_LI);
         DirEnd (rqptr);
         return;
      }
   }

   /** if (Debug) fprintf (stdout, "idx: %d\n", idx); **/

   /*********************/
   /* format and output */
   /*********************/

/**
   if (Debug)
      fprintf (stdout, "LayoutFaoPtr |%s|\n", tkptr->LayoutFaoPtr);
**/
   LayoutFaoDsc.dsc$a_pointer = tkptr->LayoutFaoPtr;
   LayoutFaoDsc.dsc$w_length = tkptr->LayoutFaoLength;
   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   status = sys$faol (&LayoutFaoDsc, &Length, &StringDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = tkptr->DirPath;
      rqptr->ErrorHiddenTextPtr = tkptr->LayoutFaoPtr;
      ErrorVmsStatus (rqptr, status, FI_LI);
      DirEnd (rqptr);
      return;
   }
   String[Length] = '\0';
   /** if (Debug) fprintf (stdout, "String: %d |%s|\n", Length, String); **/

   /* empty any file-internal description previously generated */
   tkptr->Description[0] = '\0';

   NetWriteBuffered (rqptr, AstFunctionPtr, String, Length);
}

/*****************************************************************************/
/*
These comments apply equally to DirReadMeBottom() below.

Attempt to send one of possible multiple files ("README.", et. al.) in the 
specified directory.  Then, provided there was not serious error with the 
send, execute the function specified by the address in 'NextTaskFunction'.  
The file can be HTML or plain-text.  With plain-text the contents will be 
encapsulated.
*/ 
 
int DirReadMeTop (struct RequestStruct *rqptr)

{
   register char  *cptr, *rptr, *sptr;
   register struct DirTaskStruct  *tkptr;

   boolean  PreformatFile;
   int  status;
   char  *TypePtr;
   char  FileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirReadMeTop()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   while (*(rptr = ConfigReadMeFile(tkptr->ReadMeFileIndex++)))
   {
      sptr = FileName;
      for (cptr = tkptr->DirectoryPart; *cptr; *sptr++ = *cptr++);
      TypePtr = NULL;
      for (/* 'rptr' initialized above! */; *rptr; *sptr++ = *rptr++)
         if (*rptr == '.') TypePtr = rptr;
      *sptr = '\0';
      if (TypePtr == NULL) TypePtr = rptr;

      cptr = ConfigContentType (&tkptr->ContentType, TypePtr);

      /* ignore any non-text types! */
      if (!strsame (cptr, "text/", 5)) continue;

      if (strsame (cptr, ConfigContentTypeSsi, -1))
         SsiBegin (rqptr, &DirHeading, &DirReadMeTop, FileName);
      else
      {
         if (strsame (cptr, "text/plain", -1))
            PreformatFile = true;
         else
            PreformatFile = false;

         FileBegin (rqptr, &DirHeading, &DirReadMeTop,
                    FileName, cptr, false, PreformatFile, false);
      }
      return;
   }

   DirHeading (rqptr);
}

/*****************************************************************************/
/*
See comment in DirReadMeTop() above.
*/ 
 
int DirReadMeBottom (struct RequestStruct *rqptr)

{
   register char  *cptr, *rptr, *sptr;
   register struct DirTaskStruct  *tkptr;

   boolean  PreformatFile;
   int  status;
   char  *TypePtr;
   char  FileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirReadMeBottom()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->DirTaskPtr;

   while (*(rptr = ConfigReadMeFile(tkptr->ReadMeFileIndex++)))
   {
      sptr = FileName;
      for (cptr = tkptr->DirectoryPart; *cptr; *sptr++ = *cptr++);
      TypePtr = NULL;
      for (/* 'rptr' initialized above! */; *rptr; *sptr++ = *rptr++)
         if (*rptr == '.') TypePtr = rptr;
      *sptr = '\0';
      if (TypePtr == NULL) TypePtr = rptr;

      cptr = ConfigContentType (&tkptr->ContentType, TypePtr);

      /* ignore any non-text types! */
      if (!strsame (cptr, "text/", 5)) continue;

      if (strsame (cptr, ConfigContentTypeSsi, -1))
         SsiBegin (rqptr, &DirHeading, &DirReadMeBottom, FileName);
      else
      {
         if (strsame (cptr, "text/plain", -1))
            PreformatFile = true;
         else
            PreformatFile = false;

         FileBegin (rqptr, &DirHeading, &DirReadMeBottom,
                    FileName, cptr, false, PreformatFile, false);
      }
      return;
   }

   DirEnd (rqptr);
}

/*****************************************************************************/
