/*****************************************************************************/
/*
                                 Error.c

Error reporting functions.

All errors are reported using HTML formatting.  The assumptions being: 

  o  That most errors will occur before a document begins being transfered
     (e.g. file not found), and so its content type has not been set, and 
     the error can sent with a "text/html" type.

  o  HTML documents are probably more extensive than plain-text documents.
     Hence if an error occurs during transfer (e.g. record too big for user's
     buffer) it can be reported within the document with correct formatting 
     (document context notwithstanding).

  o  If an error occurs while transfering a plain-text document the HTML
     message is still readable, albeit a little cluttered.  Additional,
     obvious text is included to try and highlight the message.

  o  In non-"text/..." transfers the message text is not included anyway.

After a request thread is created and an error is generated the storage
'rqptr->ErrorMessagePtr' becomes non-NULL.  This mechanism can be used 
to detect whether an error has occured.  Most common errors create an error 
message in heap memory and point at this using 'rqptr->ErrorMessagePtr'. 
This message can be output at some time convenient to the task underway.  
After an error message is sent to the client the 'rqptr->ErrorMessagePtr' 
is returned to a NULL value to indicate no outstanding error (but usually the 
thread is disposed of after an error occurs).  Notification of other, more 
critical errors are sent directly to the client at the time they occur (for 
instance, failed heap memory allocation).

If ErrorGeneral() is called without first setting rqptr->ResponseStatusCode
it will default to 500, or "internal server error".  Therefore string
overflows and the like can be called without setting this code.  Query string
errors should set to 403 or 404 indication request cannot be processed.

If ErrorVmsStatus() is called without first setting rqptr->ResponseStatusCode
the function will attempt to determine the most appropriate HTTP status code
from the VMS status value.  It will then default to 500, or "internal server
error", indicating some unsual (e.g. non file system) condition.

The server configuration can supply additional information included whenever
an error report is generated.  This includes a URL for contacting the system
administrator (note: URL, not text, etc.) when an implementation,
configuration, process quota, etc., error is reported.   It also can provide
text to be included against any other error report.  This text can contain
HTML markup, carriage control, etc., and is designed for a brief message
containing a link to an explanation page.


VERSION HISTORY
---------------
18-OCT-97  MGD  401 without authentication schemes enabled into a 403 too!
30-AUG-97  MGD  401 without a realm is now converted into a 403
09-AUG-97  MGD  message database, with considerable rewrite!
01-FEB-97  MGD  HTTPd version 4
06-JUN-96  MGD  added authentication failure
01-DEC-95  MGD  HTTPd version 3
25-MAR-95  MGD  minor functional and cosmetic changes
20-DEC-94  MGD  multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "error.h"
#include "httpd.h"
#include "net.h"
#include "msg.h"
#include "support.h"
#include "vm.h"

/******************/
/* global storage */
/******************/

char  ErrorSanityCheck [] = "Sanity check failure!";

char  *ErrorRecommendNotifyPtr;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Exit the application after output of the module and line location reporting 
the error, a brief explanation, and then exiting generating a DCL-reported 
message.
*/

ErrorExitVmsStatus
(
int VmsStatus,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorExitVmsStatus()\n");

   /* 
      The source file format provided by the "__FILE__" macro will
      be "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   fprintf (stdout,
"%%%s-E-SOFTWAREID, %s\n\
-%s-E-WHERE, module: %s line: %d\n\
-%s-E-WHAT, %s\n",
            Utility, SoftwareID,
            Utility, cptr, SourceLineNumber,
            Utility, Explanation);

   if (!VmsStatus) VmsStatus = STS$K_ERROR | STS$M_INHIB_MSG;
   exit (VmsStatus);
}

/****************************************************************************/
/*
It is a fatal error to call this function without an error message.  If the
HTTP method is HEAD then generate an appropriate response header and send it.
If not HEAD and a text document, generate a reponse header and include the
error message as the body of the response.  Generates "Content-Length:" and
"Keep-Alive:" fields as required.   Allows error reports of any length.
*/

ErrorSendToClient
(
struct RequestStruct *rqptr,
void *AstFunctionPtr
)
{
   /* contains HTML, also enough to make it obvious in plain text */
   static $DESCRIPTOR (DividerFaoDsc,
"<FONT COLOR=\"#ff0000\"><B><HR>\n\
\n\
<!!-- -- --   !AZ   -- -- -->\n\
\n");

   static $DESCRIPTOR (HttpErrorHeaderFaoDsc,
"!AZ !UL Error\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Content-Type: text/html\r\n\
Content-Length: !UL\r\n\
!AZ\
\r\n");

   static $DESCRIPTOR (HttpErrorAuthorizationFaoDsc,
"!AZ 401 Error\r\n\
!AZ\
!AZ\
Server: !AZ\r\n\
Date: !AZ\r\n\
ContentType: text/html\r\n\
Content-Length: !UL\r\n\
!AZ\
\r\n");

   static $DESCRIPTOR (BufferDsc, "");

   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [8];
   char  *KeepAlivePtr;
   char  Buffer [2048];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorSendToClient()\n");

   if (rqptr->ErrorMessagePtr == NULL || !rqptr->ErrorMessageLength)
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

   if (rqptr->ResponseHeaderPtr != NULL &&
       rqptr->OutputBufferPtr != NULL)
   {
      /*******************/
      /* provide divider */
      /*******************/

      BufferDsc.dsc$a_pointer = Buffer;
      BufferDsc.dsc$w_length = sizeof(Buffer)-1;

      status = sys$fao (&DividerFaoDsc, &Length, &BufferDsc,
                        MsgFor(rqptr,MSG_STATUS_ERROR));
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         ErrorInternal (rqptr, status, "sys$fao()", FI_LI);
         return (status);
      }

      /* buffer the original AST completion address */
      rqptr->ErrorMessageAstAddress = AstFunctionPtr;
      NetWrite (rqptr, &ErrorMessageDividerAst, Buffer, Length);
      return (SS$_NORMAL);
   }
   else
   {
      /***********************/
      /* provide HTTP header */
      /***********************/

      BufferDsc.dsc$a_pointer = Buffer;
      BufferDsc.dsc$w_length = sizeof(Buffer)-1;

      /*
         Mosaic 2.4-7 seems to spit on authentication if a persistent
         connection times-out (09-JUL-96)
      */
      if (rqptr->HttpUserAgentPtr != NULL &&
          strstr (rqptr->HttpUserAgentPtr, "Mosaic") != NULL)
         KeepAlivePtr = "";
      else
      if (Config.KeepAliveTimeoutSeconds &&
          rqptr->KeepAliveRequest &&
          (rqptr->HttpMethod == HTTP_METHOD_GET ||
           rqptr->HttpMethod == HTTP_METHOD_HEAD))
      {
         rqptr->KeepAliveCount++;
         rqptr->KeepAliveResponse = true;
         KeepAlivePtr = KeepAliveHttpHeader;
      }
      else
         KeepAlivePtr = "";

      /* without a realm for authentication convert it to forbidden */
      if (rqptr->ResponseStatusCode == 401 && !rqptr->AuthRealmPtr[0])
         rqptr->ResponseStatusCode = 403;

      if (rqptr->ResponseStatusCode == 401 && !rqptr->AuthChallengeScheme)
      {
         if (Config.AuthDigestEnabled)
            rqptr->AuthChallengeScheme |= AUTH_SCHEME_DIGEST;
         if (Config.AuthBasicEnabled)
            rqptr->AuthChallengeScheme |= AUTH_SCHEME_BASIC;

         /* if neither scheme enabled don't challenge */
         if (!rqptr->AuthChallengeScheme) rqptr->ResponseStatusCode = 403;
      }

      if (rqptr->ResponseStatusCode == 401)
      {
         /*************************/
         /* authorization failure */
         /*************************/

         if ((rqptr->AuthChallengeScheme & AUTH_SCHEME_DIGEST) &&
             rqptr->AuthDigestChallengePtr == NULL)
            DigestChallenge (rqptr, "");

         if ((rqptr->AuthChallengeScheme & AUTH_SCHEME_BASIC) &&
             rqptr->AuthBasicChallengePtr == NULL)
            BasicChallenge (rqptr);

         if (rqptr->AuthDigestChallengePtr == NULL)
            rqptr->AuthDigestChallengePtr = "";
         if (rqptr->AuthBasicChallengePtr == NULL)
            rqptr->AuthBasicChallengePtr = "";

         vecptr = FaoVector;
         *vecptr++ = HttpProtocol;
         *vecptr++ = rqptr->AuthDigestChallengePtr;
         *vecptr++ = rqptr->AuthBasicChallengePtr;
         *vecptr++ = SoftwareID;
         *vecptr++ = rqptr->GmDateTime;
         *vecptr++ = rqptr->ErrorMessageLength;
         *vecptr++ = KeepAlivePtr;

         status = sys$faol (&HttpErrorAuthorizationFaoDsc, &Length, &BufferDsc,
                            &FaoVector);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            ErrorInternal (rqptr, status, "sys$faol()", FI_LI);
            return (status);
         }
      }
      else
      {
         /*****************/
         /* general error */
         /*****************/

         vecptr = FaoVector;
         *vecptr++ = HttpProtocol;
         *vecptr++ = rqptr->ResponseStatusCode;
         *vecptr++ = SoftwareID;
         *vecptr++ = rqptr->GmDateTime;
         *vecptr++ = rqptr->ErrorMessageLength;
         *vecptr++ = KeepAlivePtr;

         status = sys$faol (&HttpErrorHeaderFaoDsc, &Length, &BufferDsc,
                            &FaoVector);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            ErrorInternal (rqptr, status, "sys$faol()", FI_LI);
            return (status);
         }
      }

      Buffer[BufferDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "%d |%s|\n", Length, Buffer);

      rqptr->ResponseHeaderPtr = VmGetHeap (rqptr, Length+1);
      memcpy (rqptr->ResponseHeaderPtr, Buffer, Length+1);
   }

   /**********************/
   /* send error message */
   /**********************/

   NetWrite (rqptr, AstFunctionPtr,
             rqptr->ErrorMessagePtr, rqptr->ErrorMessageLength);

   /* indicate the error message has been sent */
   rqptr->ErrorMessagePtr = NULL;
   rqptr->ErrorMessageLength = 0;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorMessageDividerAst (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorMessageDividerAst()\n");

   NetWrite (rqptr, rqptr->ErrorMessageAstAddress,
             rqptr->ErrorMessagePtr, rqptr->ErrorMessageLength);

   /* indicate the error message has been sent */
   rqptr->ErrorMessagePtr = NULL;
   rqptr->ErrorMessageLength = 0;
}

/*****************************************************************************/
/*
Generate an error message about a VMS status problem for subsequent reporting
to the client.  Generally most errors are reported as documents in their own
right, making the full HTML document desirable.  If reported part-way through
some action the redundant tags should be ignored anyway.
*/

ErrorVmsStatus
(
struct RequestStruct *rqptr,
int StatusValue,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !UL</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>!AZ ... <TT>!AZ</TT>\n\
<!!-- sts: %X!XL \"!AZ\" -->\n\
!AZ!AZ\
!AZ\
</BODY>\n\
</HTML>\n");

   register char  *cptr, *sptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  *ContentTypePtr,
         *MessagePtr,
         *RecommendLfPtr,
         *RecommendPtr;
   char  Message [256],
         Buffer [4096];
   $DESCRIPTOR (MessageDsc, Message);
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ErrorVmsStatus() %%X%08.08X %s %d\n",
               StatusValue, SourceFileName, SourceLineNumber);
                                 
   /* don't overwrite any existing error message */
   if (rqptr->ErrorMessagePtr != NULL) return;

   if (rqptr->ContentType.ContentTypePtr == NULL)
      ContentTypePtr = "";
   else
      ContentTypePtr = rqptr->ContentType.ContentTypePtr;

   if (StatusValue == RMS$_FNF)
   {
      if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 404;
      if (strsame (ContentTypePtr, "text/", 5))
         MessagePtr = MsgFor(rqptr,MSG_STATUS_DOC_NOT_FOUND);
      else
         MessagePtr = MsgFor(rqptr,MSG_STATUS_FILE_NOT_FOUND);
   }
   else
   if (StatusValue == RMS$_PRV)
   {
      if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 403;
      if (strsame (ContentTypePtr, "text/", 5))
         MessagePtr = MsgFor(rqptr,MSG_STATUS_DOC_PROTECTION);
      else
         MessagePtr = MsgFor(rqptr,MSG_STATUS_FILE_PROTECTION);
   }
   else
   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0)))
   {
      if (!rqptr->ResponseStatusCode)
      {
         if (StatusValue == RMS$_DNF ||
             StatusValue == RMS$_SYN ||
             StatusValue == RMS$_DEV ||
             StatusValue == RMS$_FNM ||
             StatusValue == RMS$_TYP ||
             StatusValue == RMS$_DIR ||
             StatusValue == SS$_NOSUCHFILE ||
             StatusValue == SS$_DIRNOTEMPTY ||
             StatusValue == SS$_EXDISKQUOTA)
            rqptr->ResponseStatusCode = 400;
         else
         if (StatusValue == SS$_NOPRIV)
            rqptr->ResponseStatusCode = 403;
         else
            rqptr->ResponseStatusCode = 500;
      }
      Message[Length] = '\0';
      cptr = sptr = MessagePtr = Message;
      *cptr = toupper(*cptr);
      /* improve the look by removing any embedded sys$fao() formatting */
      while (*cptr)
      {
         if (*cptr == '!')
         {
            cptr++;
            *sptr++ = '?';
            /* step over any field width digits */
            while (isdigit(*cptr)) cptr++;
            /* usually two formatting characters */
            if (isalpha(*cptr)) cptr++;
            if (isalpha(*cptr)) cptr++;
         }
         else
            *sptr++ = *cptr++;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "Message |%s|\n", Message); **/
   }
   else
   {
      ErrorInternal (rqptr, status, "sys$getmsg()", FI_LI);
      return;
   }

   if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 500;

   if (rqptr->ErrorTextPtr == NULL)
   {
      if (rqptr->PathInfoPtr == NULL)
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_STATUS_NO_INFORMATION);
      else
         rqptr->ErrorTextPtr = rqptr->PathInfoPtr;
   }

   if (rqptr->ErrorHiddenTextPtr == NULL || !Config.IncludeCommentedInfo)
      rqptr->ErrorHiddenTextPtr = MsgFor(rqptr,MSG_STATUS_NO_INFORMATION);

   RecommendPtr = NULL;
   if (!Config.ErrorRecommend || rqptr->ResponseStatusCode == 401)
      RecommendPtr = "";
   else
   if (StatusValue == RMS$_FNF ||
       StatusValue == RMS$_DNF ||
       StatusValue == SS$_NOSUCHFILE)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_NOSUCHFILE);
   else
   if (StatusValue == RMS$_PRV)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_PRV);
   else
   if (StatusValue == SS$_NOPRIV)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_NOPRIV);
   else
   if (StatusValue == RMS$_SYN ||
       StatusValue == RMS$_DEV ||
       StatusValue == RMS$_DIR ||
       StatusValue == RMS$_FNM ||
       StatusValue == RMS$_TYP)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_SYNTAX);
   else
   if (StatusValue == RMS$_FLK)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_FLK);
   else
   if (StatusValue == SS$_DIRNOTEMPTY ||
       StatusValue == SS$_EXDISKQUOTA)
      RecommendPtr = MsgFor(rqptr,MSG_STATUS_ADVISE_CORRECT);

   if (RecommendPtr == NULL)
      RecommendPtr = RecommendLfPtr = "";
   else
      RecommendLfPtr = "\n";

   vecptr = FaoVector;
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = ErrorSourceInfo (SourceFileName, SourceLineNumber);
   *vecptr++ = cptr = MsgFor(rqptr,MSG_STATUS_ERROR);
   *vecptr++ = rqptr->ResponseStatusCode;
   *vecptr++ = cptr;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);
   *vecptr++ = MessagePtr;
   *vecptr++ = rqptr->ErrorTextPtr;
   *vecptr++ = StatusValue;
   *vecptr++ = rqptr->ErrorHiddenTextPtr;
   *vecptr++ = RecommendPtr;
   *vecptr++ = RecommendLfPtr;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_INFO);

   status = sys$faol (&ErrorFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      ErrorInternal (rqptr, status, "sys$faol()", FI_LI);
      return;
   }

   Buffer[rqptr->ErrorMessageLength = Length] = '\0';
   rqptr->ErrorMessagePtr = VmGetHeap (rqptr, Length+1);
   memcpy (rqptr->ErrorMessagePtr, Buffer, Length);
   rqptr->ErrorMessagePtr[rqptr->ErrorMessageLength = Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", rqptr->ErrorMessagePtr);
}

/*****************************************************************************/
/*
Generate an error message about a general (non-VMS status) problem for
subsequent reporting to the client.
*/
 
ErrorGeneral
(
struct RequestStruct *rqptr,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !UL</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>!AZ\n\
!AZ\
</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [4096];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorGeneral()\n");

   /* don't overwrite any existing error message */
   if (rqptr->ErrorMessagePtr != NULL) return;

   if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 500;

   vecptr = FaoVector;
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = ErrorSourceInfo (SourceFileName, SourceLineNumber);
   *vecptr++ = cptr = MsgFor(rqptr,MSG_STATUS_ERROR);
   *vecptr++ = rqptr->ResponseStatusCode;
   *vecptr++ = cptr;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);
   *vecptr++ = Explanation;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_INFO);

   status = sys$faol (&ErrorFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      ErrorInternal (rqptr, status, "sys$fao()", FI_LI);
      return;
   }

   Buffer[rqptr->ErrorMessageLength = Length] = '\0';
   rqptr->ErrorMessagePtr = VmGetHeap (rqptr, Length+1);
   memcpy (rqptr->ErrorMessagePtr, Buffer, Length);
   rqptr->ErrorMessagePtr[rqptr->ErrorMessageLength = Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", rqptr->ErrorMessagePtr);
}

/*****************************************************************************/
/*
Report general string overflow.
*/

ErrorGeneralOverflow
(
struct RequestStruct *rqptr,
char *SourceFileName,
int SourceLineNumber
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorGeneralOverflow()\n");

   if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 500;
   ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW),
                 SourceFileName, SourceLineNumber);
}

/*****************************************************************************/
/*
*/
 
ErrorInternal
(
struct RequestStruct *rqptr,
int StatusValue,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<HEAD>\n\
<TITLE>!AZ 500</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P><FONT COLOR=\"#ff0000\">INTERNAL ERROR ... !AZ</FONT><!!-- %X!XL -->\n\
!AZ\
!AZ\
</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [4096];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorInternal()\n");

   rqptr->ResponseStatusCode = 500;

   vecptr = FaoVector;
   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = ErrorSourceInfo (SourceFileName, SourceLineNumber);
   *vecptr++ = cptr = MsgFor(rqptr,MSG_STATUS_ERROR);
   *vecptr++ = cptr;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);
   *vecptr++ = Explanation;
   *vecptr++ = StatusValue;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORT);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_INFO);

   status = sys$faol (&ErrorFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
      ErrorExitVmsStatus (status, ErrorSanityCheck, FI_LI);

   Buffer[rqptr->ErrorMessageLength = Length] = '\0';
   rqptr->ErrorMessagePtr = VmGetHeap (rqptr, Length+1);
   memcpy (rqptr->ErrorMessagePtr, Buffer, Length+1);
}

/*****************************************************************************/
/*
If the server is configured for it return a pointer to some META information
containing software ID, source code module and line in which the error occured,
otherwise pointer to an empty string.  This function is not used reentrantly
and so provided the contents of the static buffer are used before it is
recalled it will continue to work.
*/
 
char* ErrorSourceInfo
(
char *SourceFileName,
int SourceLineNumber
) 
{
   static $DESCRIPTOR (SourceInfoFaoDsc,
"<META NAME=\"generator\" CONTENT=\"!AZ\">\n\
<META NAME=\"module\" CONTENT=\"!AZ\">\n\
<META NAME=\"line\" CONTENT=\"!UL\">\n\0");

   static char  SourceInfo [256];
   static $DESCRIPTOR (SourceInfoDsc, SourceInfo);

   register char  *sptr, *cptr;
   int  status;
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorSourceInfo()\n");

   if (!Config.ErrorSourceInfo) return ("");

   /* 
      The source file format provided by the "__FILE__" macro will
      be "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      sptr = cptr;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   status = sys$fao (&SourceInfoFaoDsc, 0, &SourceInfoDsc,
                     SoftwareID, cptr, SourceLineNumber);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
      return ("sys$faol() ERRROR");

   return (SourceInfo);
}
 
/*****************************************************************************/
/*
*/
 
char* SysGetMsg (int StatusValue)
 
{
   static char  Message [256];
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);
 
   sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0);
   Message[Length] = '\0';
   if (Debug) fprintf (stdout, "SysGetMsg() |%s|\n", Message);
   return (Message);
}
 
/****************************************************************************/

